/***************************************************************************/
/*   graphplace:                                                           */
/*   Copyright     Jos van Eijndhoven                                      */
/*                 Eindhoven Univ of Tech, Dept. of Elec. Eng.             */
/*                 5600 MB Eindhoven, The Netherlands                      */
/*                 J.T.J.v.Eijndhoven@ele.tue.nl                           */
/*   Distribution November 1, 1994                                         */
/***************************************************************************/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include "defines.h"
#include "files.h"
#include "mem_alloc.h"
#include "scale.h"
#include "place.h"

/****** imported ****/
extern char *optarg;                   /* silently set by getopt() */
extern int optind, opterr, optopt;     /* silently set by getopt() */

/****** exported *******/
export int debuglevel;     /* value 0-9, controls the amount of debug output */
export char *prologname;   /* filename argument for prolog to include */
export char *boxarg;	    /* specified bounding box argument */
export int postscript; /* file is postscript */
export char *nodeword;	    /* keyword to look for in infile */
export char *edgeword;	    /* keyword to look for in infile */
export char *myname;       /* name of this program for error messages */
export Boolean flexscale;  /* flexible scaling of x and y indep., not square */
export int rotation;	   /* rotation of final graph, * 90 degrees */
export int margin;	   /* free margin in % of bbox size */
export Boolean writeunits; /* write a postscript statement on unit distance*/
export Boolean showpage;   /* write a postscript 'showpage' statement on end*/
export Boolean curvedges;   /* draw postscript edges as curves */
export Boolean arrowedges;   /* draw postscript edges with arrow heads */
export Boolean routedges;   /* route long edges along intermediate points */
export Boolean mergeedges;  /* collapse the routing of long edges */
export char splitchar;

/****** forward declarations ***/
static void process_opts( int argc, char *argv[]);
static void usage();
static char *infile;       /* name of input file */

/****** exported program code *****/
int main( int argc, char *argv[])
{
	process_opts( argc, argv);
	read_files( infile);
	place_graph();
	set_scale();
	write_data();
	if (debuglevel >= 1) print_resources();
	exit(0);
}

export void error( char *fmt, ...)
{
	va_list args;

	va_start( args, fmt);
	fprintf( stderr, "%s: ", myname);
	vfprintf( stderr, fmt, args);
	va_end( args);

	exit(1);
}

export void warning( char *fmt, ...)
{
	va_list args;

	va_start( args, fmt);
	fprintf( stderr, "%s: Warning: ", myname);
	vfprintf( stderr, fmt, args);
	va_end( args);
}

export void verbose( char *fmt, ...)
{
	va_list args;

	va_start( args, fmt);
	vfprintf( stderr, fmt, args);
	va_end( args);
}

/****** local program code *****/
static void process_opts( int argc, char *argv[])
{
	int opt;

	myname = argv[0];
	debuglevel = 0;
	postscript = False;
	nodeword = "node";
	edgeword = "edge";
	rotation = 0;
	flexscale = False;
	margin = 10;
	writeunits = True;/* former option, now follows `postscript' value */
	showpage = True;  /* former option, now follows `postscript' value */
	curvedges = False;
	arrowedges = False;
	routedges = False;
	mergeedges = False;
	splitchar = '\0';

	/***** process options *****/
	while ((opt = getopt( argc, argv, "vd:i:pb:n:e:fr:m:uscag:yz")) != EOF)
	{	switch(opt)
		{ case 'v': if (debuglevel < 1) debuglevel = 1; break;
		  case 'd': debuglevel = atoi( optarg); break;
		  case 'i': prologname = optarg; break;
		  case 'p': postscript = True; break;
		  case 'b': boxarg = optarg; break;
		  case 'n': nodeword = optarg; break;
		  case 'e': edgeword = optarg; break;
		  case 'f': flexscale = True; break;
		  case 'g': splitchar = optarg[1] ? atoi(optarg)
					: *optarg; break;
		  case 'r': rotation = atoi( optarg); break;
		  case 'm': margin = atoi( optarg); break;
		  case 'u': writeunits = True; break;/*left for compatibility*/
		  case 's': showpage = True; break;  /*left for compatibility*/
		  case 'c': curvedges = True; break;
		  case 'a': arrowedges = True; break;
		  case 'y': mergeedges = True; break;
		  case 'z': routedges = True; break;
		  default : usage(); break;
		}
	}
	/**** check correctness of command line options ***/
	if (debuglevel < 0 || debuglevel > 9)
		error( "Illegal debug level %d! (use 0-9)\n", debuglevel);

	if (rotation >= 4 || rotation <= -4)
		warning( "Requested rotation is along %d * 90 degrees?\n",
			rotation);

	if (margin < 0 || margin >= 50)
		error( "Free margin along all sides of %d%% is impossible!\n",
			margin);

	/**** dependencies between options ****/
	if (postscript && !prologname) prologname = "-";
	if (mergeedges) routedges = True;

	/**** take first non-option argument as module name ****/
	if (optind < argc)
		infile = argv[ optind++];
	else 	infile = "-";

	if (debuglevel>=2) verbose( "infile set to '%s'\n", infile);

	while (optind < argc)
		warning( "Argument '%s' not used, skipped!\n", argv[optind++]);
}

static void usage()
{
	fprintf( stderr, "Usage: %s [options] [infile] > outfile\n\n", myname);
	fprintf( stderr,
		"Options:\n"
		"-a		Draw edges with arrow-points (ps mode only)\n"
		"-b <box>	Specification of a bounding box\n"
		"		Xll,Yll,Xur,Yur  -or-\n"
		"		Xur,Yur          with Xll,Yll implicitly 0,0\n"
		"-c             Draw edges slightly curved (ps mode only)\n"
		"-d <digit>     Set debug level (>2 for development use only)\n"
		"-e <word>      Set `edge' keyword\n"
		"-f             Flexible scaling, x- and y- axis independent\n"
		"-g <c>		Vertical grouping of nodes\n"
		"		<c> is a single char or decimal ascii value\n"
		"-i <file>	Include file, to be copied to the output\n"
		"		If not starting with `/', the <file> is looked\n"
		"		for in a sequence of directories.\n"
		"		The name `-' is shorthand for `graphpro.ps'\n"
		"		(`-i-' is default in ps mode)\n"
		"-m <int>	Margin in %% to be kept free from the bounding box\n"
		"		Default is 10 (%%)\n"
		"-n <word>	The `node' keyword\n"
		"-p		Set postscript (ps) mode\n"
		"		Default if input or -i file is recognised as such\n"
		"-r <digit>	Rotate picture clockwise in units of 90 degrees\n"
		"-v		Be verbose, same as `-d1'\n"
		"-y		Merge the origin of edges crossing multiple levels\n"
		"		This implies `-z'\n"
		"-z		Rout edges which cross vertical levels between nodes\n"
		);

	exit(1);
}
