/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/


#include <strings.h>
#include <stdio.h>
#include <math.h>
 
#include "kask.h"
#include "kasktri.h"
 
static void InnerEdges(), DeleteTr(), DeleteEdg(), RelGreen(),
       DelTE();
static int DoRefTr(), RefineTr(), Mark2(), DoDelTr(), MkDelGreen(),
       NeighborRefined(), Refined2();

static int noRefTr, noDelTr;

/* ******************************************************************
   OpenRef     prepares the triangulation modul to accept requests
               to refine a triangle. The green closure of the actual
               triangulation is removed.
   --------
   return int  true (or false)
***************************************************************** */
int OpenRef()
  {
    if (actTriang==nil)
      { ZIBStdOut("Tri: no triangulation (OpenRef)\n"); return false; }
    RelGreen();
    noRefTr = 0;
    (actTriang->refLevel)++;
	actTriang->iluAvailable = false;
    return true;
  }
 
/* *******************************************************************
   RefTr(t)     request to refine the triangle. The actual refinement
                is postphoned to the call of CloseRef
   ----------
   TR *t        Adress of the triangle
   return int   true (or false)
****************************************************************** */
int RefTr(t)
  TR *t;
  {
    t->mark = true;
    return true;
  }
 
/* *****************************************************************
   CloseRef    refines the marked triangles and completes the
               triangulation with the neccessary green ones
   ----------
   return int  true or false
***************************************************************** */
int CloseRef(verboseP)
  int verboseP;
  {    
    int rc = true, countP = actTriang->noOfPoints,
		countE = actTriang->noOfEdges,
		countT = actTriang->noOfTriangles;

    if (ApplyT(DoRefTr,all)==false)
      {
        ZIBStdOut("Tri: failed refining triangles (CloseRef)\n");
        return false;
      }
    if (verboseP)
	  {
	    sprintf(globBuf, "Tri: %4d points, %4d edges, %4d triangles generated\n",
		    (actTriang->noOfPoints)-countP,
			(actTriang->noOfEdges)-countE,
			(actTriang->noOfTriangles)-countT);
		ZIBStdOut(globBuf);
	  }

	countE = actTriang->noOfEdges;
	countT = actTriang->noOfTriangles;

    if (ApplyT(MkGreen,all)==false)
      {
        ZIBStdOut("Tri: failed making green triangles (CloseRef)\n");
        return false;
      }
    if (verboseP)
	  {
	    sprintf(globBuf, "Tri:             %4d edges, %4d triangles for green closure\n",
				(actTriang->noOfEdges)-countE,
				(actTriang->noOfTriangles)-countT);
		ZIBStdOut(globBuf);
	  }
    return rc;
  }
 
/* ***************************************************************
   DoRefTr     internal routine to refine a triangle, lotrues for
               neighbors do be refined too.
   --------
   TR *t       Adress of Triangle
   return int  true or false
*************************************************************** */
static int DoRefTr(t)
  TR *t;
  {
    TR *neighbor, *bigNeighbor;
    EDG *ed, *edFather, **eds;
	int k;

    if (t->mark) t->mark = false;         /* return if not marked */
    else return true;
    if ((t->firstSon)!=nil) return true;  /* return if already marked */

    if (RefineTr(t)!=true) return false;  /* refine it */
    noRefTr++;
/*
    Algorithm: For all three edges:
          (1)  look if neighbor triangle has 2 refined neighbors
               and refine it this case,
          (2)  if no neighbor triangle exists, "father neighbor"
               triangle needs to be refined (green refined at
			   the last level),
    This routine uses recursion.
*/
	eds = &(t->e1);
	for (k = 0; k<3; k++)
	  {
		ed=eds[k];
		if ((ed->boundP)==INTERIOR)
		  {
			neighbor = NEIGH_TR(ed,t);
			if (neighbor!=nil)
			  {
				if (Mark2(neighbor)==true)                      /* (1) */
				  {
				    neighbor->mark=true;
					if (!DoRefTr(neighbor)) return false;
				  }
			  }
			else											  /* (2) */
			  {
				edFather=ed->father;
				if (edFather==nil) { continue; }
				if ((edFather->t1)==nil) { continue; }
				bigNeighbor=(((edFather->t1)->firstSon)==nil)?
										edFather->t1:edFather->t2;
				if (bigNeighbor==nil) return true;
				bigNeighbor->mark=true;
				if (!DoRefTr(bigNeighbor))
				  { ZIBStdOut("Huch\n"); return false; }
			  }
  	    }
	  }
    return true;
  }
 
/* *******************************************************************
   Mark2(t)     tests if triangle has two refined neighbors
   ----------
   TR *t        Adress of the triangle
   return int   true or false
****************************************************************** */
static int Mark2(t)
  TR *t;
  {
    int k, anz=0;
    EDG **eds = &(t->e1);

    if (t->mark) return true;
	for ( k=0; k<3; k++) if ((eds[k]->firstSon)!=nil) anz++;
    return (anz>1)?true:false;
  }
 
/* *******************************************************************
   RefineTr(t)  does the actual refinement
   ----------
   TR *t        Adress of the triangle
   return int   true or false
****************************************************************** */
static int RefineTr(t)
  TR *t;
  {
    TR  *t1, *t2, *t3, *t4;
    EDG *e1 = t->e1,  *e2 = t->e2,  *e3 = t->e3, *ereg,
        *ie1,*ie2,*ie3, *e11, *e12, *e21, *e22, *e31, *e32;
/*
    Get 4 new triangles, 3 new edges, return false if not enough space
*/
    t1 = New4T(); t2 = &(t1[1]); t3 = &(t1[2]); t4 = &(t1[3]);
	(actTriang->noOfTriangles) += 3;
    ie1 = NewE(); ie2 = NewE(); ie3 = NewE();
	(actTriang->noOfEdges) += 3;
    if ((t1==nil)||(ie3==nil)) return false;
/*
   Set the "son" edges eik, i=1,2,3; k=1,2;
   if neeccessary refine the edges
*/
    if ((e1->firstSon)==nil)         /* Edge not refined */
      {
        e11 = RefineEdg(e1);
        if (e11==nil) return false; /* No space for Refinement */
      }
    else e11 = e1->firstSon;         /* Take the existing "son" edge */
    e12 = e11->next;
	if ((e11->p1)==(t->p3))
	  { ereg = e11; e11 = e12; e12 = ereg; }
    if ((e2->firstSon)==nil)
      {
        e21 = RefineEdg(e2);
        if (e21==nil) return false;
      }
    else e21 = e2->firstSon;
    e22 = e21->next;
	if ((e21->p1)==(t->p1))
	  { ereg = e21; e21 = e22; e22 = ereg; }
    if ((e3->firstSon)==nil)
      {
        e31 = RefineEdg(e3);
        if (e31==nil) return false;
      }
    else e31 = e3->firstSon;
    e32 = e31->next;
	if ((e31->p1)==(t->p2))
	  { ereg = e31; e31 = e32; e32 = ereg; }
/*
   Build the new triangles.
*/
	InnerEdges(t,e1,e2,e3,ie1,ie2,ie3);
    t1->e1 = e32; t1->e2 = e11; t1->e3 = ie1;
    t2->e1 = e12; t2->e2 = e21; t2->e3 = ie2;
    t3->e1 = e22; t3->e2 = e31; t3->e3 = ie3;
    t4->e3 = ie1; t4->e2 = ie3; t4->e1 = ie2;
    SetTP(t1,t); SetTE(t1);
    SetTP(t2,t); SetTE(t2);
    SetTP(t3,t); SetTE(t3);
    SetTP(t4,t); SetTE(t4);
    t->firstSon = t1;
	t->type = T_RED;
/*
   The four new trinagles substitute the old triangle in the
   triangulation list.
*/
    if ((t->last)==nil) actTriang->firstTriangle = t1;
    else (t->last)->next = t1;
    if ((t->next)==nil) actTriang->lastTriangle = t4;
    else (t->next)->last = t4;
    t1->last = t->last;
    t4->next = t->next;
    return true;
  }
 
/* *******************************************************************
   RefineEdg(ed)  does the actual refinement of an edge
   ----------
   EDG *ed        Adress of the edge
   return int     true or false
****************************************************************** */
EDG* RefineEdg(ed)
  EDG *ed;
  {
    EDG *e1, *e2;
    PT *p;
/*
   Get two new edges; return false if not enough space
*/
    e1 = NewE(); e2 = NewE(); (actTriang->noOfEdges) += 2;
    if (e2==nil) return false;
/*
   Call the midpoint generating routine and set the new fields
*/
    if ((ed->pm)==nil)
      {
		p = (ed->MidPoint)(ed,nil);
		ed->pm = p;
		p->boundP = ed->boundP;
	  }
	else
	  {
	    p = ed->pm;
		(actTriang->lastPoint)->next = p;
		p->last = actTriang->lastPoint;
		actTriang->lastPoint = p;
		p->next = nil;
		p->indexP = (actTriang->noOfPoints)++;
	  }
    e1->p1 = ed->p1; e1->p2 = p; e1->MidPoint = ed->MidPoint;
    e1->bound = ed->bound;
    e1->boundP = ed->boundP; e1->father = ed; e1->type = ed->type;
    e2->p1 = p; e2->p2 = ed->p2; e2->MidPoint = ed->MidPoint;
    e2->bound = ed->bound;
    e2->boundP = ed->boundP; e2->father = ed; e2->type = ed->type;
    e1->class = e2->class = ed->class;
    p->class = ed->class;
    ed->firstSon = e1;
/*
	ReturnElem(EDGVEC,(PTR)(ed->vec));
	ed->vec = nil;
*/
    return e1;
  }
 
/* *******************************************************************
   SetTP(t,f)   set the points of a triangle, computed from the
                edges
   ----------
   TR *t        Adress of the triangle
   TR *f        Father of t
****************************************************************** */
void SetTP(t,f)
  TR *t,*f;
  {
    PT *P11 = (t->e1)->p1, *P12 = (t->e1)->p2,
       *P21 = (t->e2)->p1, *P22 = (t->e2)->p2,
       *P31 = (t->e3)->p1, *P32 = (t->e3)->p2;
    t->p1 = ((P31==P11)||(P31==P12))?P32:P31;
    t->p2 = ((P11==P21)||(P11==P22))?P12:P11;
    t->p3=((P21==P31)||(P21==P32))?P22:P21;
    t->father = f;
	if (f!=nil)
          { t->depth = (f->depth)+1; t->class = f->class; }
	if ((t->depth)>(actTriang->maxDepth))
	  actTriang->maxDepth = t->depth;
    return;
  }
 
/* *******************************************************************
   SetTE(t)     set the triangles of an edge, computed from a new
                triangle, makes checks for programming errors
   ----------
   TR *t        Adress of the triangle
****************************************************************** */
void SetTE(t)
  TR *t;
  {
    EDG *ed;
    int k = 0;
 
    ed = t->e1;
    if ((ed->t1)==nil)      { ed->t1 = t; k++; }
    else if ((ed->t2)==nil) { ed->t2 = t; k++; }
    ed = t->e2;
    if ((ed->t1)==nil)      { ed->t1 = t; k++; }
    else if ((ed->t2)==nil) { ed->t2 = t; k++; }
    ed = t->e3;
    if ((ed->t1)==nil)      { ed->t1 = t; k++; }
    else if ((ed->t2)==nil) { ed->t2 = t; k++; }
    if (k!=3)
	  {
		sprintf(globBuf, "Tri: impossible state (SetTE), k=%d)\n", k);
		ZIBStdOut(globBuf);
		PrintTr("t",t);
		PrintTr("t->e1->t1",(t->e1)->t1);
		PrintTr("t->e1->t2",(t->e1)->t2);
		PrintTr("t->e2->t1",(t->e2)->t1);
		PrintTr("t->e2->t2",(t->e2)->t2);
		PrintTr("t->e3->t1",(t->e3)->t1);
		PrintTr("t->e3->t2",(t->e3)->t2);
	  }
    return;
  }
 
/* *******************************************************************
   InnerEdges(..)      sets the points of the inner edges
   ----------
   EDG *e1,*e2,*e3     Edges of the triangle to be refined
   EDG *ie1,*ie2,*ie3  Inner edges
****************************************************************** */
static void InnerEdges(t,e1,e2,e3,ie1,ie2,ie3)
  TR *t;
  EDG *e1,*e2,*e3,*ie1,*ie2,*ie3;
  {
    ie1->p1 = e1->pm; ie1->p2 = e3->pm; ie1->type = T_RED; ie1->class = t->class;
    ie2->p1 = e2->pm; ie2->p2 = e1->pm; ie2->type = T_RED; ie2->class = t->class;
    ie3->p1 = e3->pm; ie3->p2 = e2->pm; ie3->type = T_RED; ie3->class = t->class;
    return;
  }

/* *******************************************************************
   MkGreen(t)   generates a two green triangles
   ----------
   TR *t        Adress of the triangle
   return int   true or false
****************************************************************** */
int MkGreen(t)
  TR *t;
  {
    PT *p1, *p2;
    EDG *e1 = t->e1,*e2 = t->e2,*e3 = t->e3,*eNew;
    TR *t1New,*t2New;
/*
   Testing if one neighbor triangle is refined and setting pi, ei
   corresponding to this neighbor
*/    
    if ((e1->firstSon)!=nil)
      {
        p1 = t->p1; p2 = t->p2;
        e1 = t->e1; e2 = t->e2; e3 = t->e3;
      }
    else
      {
        if ((e2->firstSon)!=nil)
          {
            p1 = t->p2; p2 = t->p3;
            e1 = t->e2; e2 = t->e3; e3 = t->e1;
          }
        else
          {
            if ((e3->firstSon)!=nil)
              {
                p1 = t->p3; p2 = t->p1;
                e1 = t->e3; e2 = t->e1; e3 = t->e2;
              }
            else return true;                /* nothing to refine */
          }
      }
/*
   Get one new edge and two new triangles, if not enough space
   return false
*/
    eNew = NewE(); (actTriang->noOfEdges)++;
    if (eNew==nil) return false;
    t1New = New4T();
    if (t1New==nil) return false;
	t2New = &(t1New[1]); (actTriang->noOfTriangles) += 1;
/*
   Set the fields for the new objects
*/
    eNew->p1 = p1; eNew->p2 = (e1->firstSon)->p2;
    eNew->type = T_GREEN; eNew->class = t->class;
    t1New->e1 = e3; t1New->e3 = eNew; t1New->type = T_GREEN;
    t2New->e1 = e2; t2New->e2 = eNew; t2New->type = T_GREEN;
    if ((e1->p1)==p2)                 /* test for orientation */
      {
        t1New->e2 = e1->firstSon; t2New->e3 = (e1->firstSon)->next;
      }
    else
      {
        t1New->e2 = (e1->firstSon)->next; t2New->e3 = e1->firstSon;
      }
    DelTE(t, 3);
    SetTP(t1New,t); SetTP(t2New,t);
    SetTE(t1New);   SetTE(t2New);
    t->firstSon = t1New;
	t->type = T_GREEN;
/*
   Substitute the old triangle by the two new ones
*/    
    if ((t->last)==nil) actTriang->firstTriangle = t1New;
    else (t->last)->next = t1New;
    if ((t->next)==nil) actTriang->lastTriangle = t2New;
    else (t->next)->last = t2New;
    t1New->last = t->last;
    t2New->next = t->next;
 
    return true;
  }
 
/* *******************************************************************
   RelGreen()   remove all green triangles
   ----------
****************************************************************** */
static void RelGreen()
  {
    TR *t = actTriang->firstTriangle,*t1old,*t2old,*tFather;
    EDG *eMid = nil;
    int k;
  
    while (t!=nil)                /* Loop for all triangles   */
      {                           /* ApplyT could not be used */
        k = 0;
	    tFather = t->father;
        if (tFather==nil) { t = t->next; continue; }
        if ((tFather->type)!=T_GREEN) { t = t->next; continue; }
/*
   Remove two green triangles, print internal error message
   if the green edge is not found
*/
        t1old = t; t2old = t->next;
        if (((t->e1)->type)==T_GREEN) { eMid = t->e1; k++;}
        if (((t->e2)->type)==T_GREEN) { eMid = t->e2; k++;}
        if (((t->e3)->type)==T_GREEN) { eMid = t->e3; k++;}
        if (k!=1)
		  {
		    sprintf(globBuf, "Tri: impossible state (RelGreen), k=%d)\n", k);
			ZIBStdOut(globBuf);
		  }
        t = t2old->next;
/* 
   Substitute the old (father) triangle for the two green ones
   in the list of triangles; return the space       
*/     
        if ((t1old->last)==nil) actTriang->firstTriangle = tFather;
        else                    (t1old->last)->next = tFather;
        if ((t2old->next)==nil) actTriang->lastTriangle = tFather;
        else                    (t2old->next)->last = tFather;
        tFather->next = t2old->next;
        tFather->last = t1old->last;
        tFather->firstSon = nil;
		tFather->type = T_WHITE;
if (tFather->mark) PrintTr("RelGreen",tFather);
		tFather->mark = false;
        DelTE(t1old, 3);   DelTE(t2old, 3);
        SetTE(tFather);
        Return4T(t1old);
		(actTriang->noOfTriangles)--;
        ReturnE(eMid);
		(actTriang->noOfEdges)--;
      }
    return;
  }
 
/* *******************************************************************
   DelTE(t, check)  delete a triangle on its edges
   ----------
   TR *t        Adress of the triangle
****************************************************************** */
static void DelTE(t, check)
  TR *t;
  int check;
  {
    EDG *ed;
    int k = 0;
/*
   if trinagle on edge, remove it, if not generate internal
   error message
*/
    ed = t->e1;
    if ((ed->t1)==t)      { ed->t1 = nil; k++; }
    else if ((ed->t2)==t) { ed->t2 = nil; k++; }
    ed = t->e2;
    if ((ed->t1)==t)      { ed->t1 = nil; k++; }
    else if ((ed->t2)==t) { ed->t2 = nil; k++; }
    ed = t->e3;
    if ((ed->t1)==t)      { ed->t1 = nil; k++; }
    else if ((ed->t2)==t) { ed->t2 = nil; k++; }

    if (k!=check)
	  {
	    sprintf(globBuf, "Tri: impossible state (DelTE), k=%d)\n", k);
		ZIBStdOut(globBuf);
	  }
    return;
  }
 
/* *******************************************************************
   MkDelGreen(t)   generates two green triangles or red ones
   ----------
   TR *t           Adress of the triangle
   return int      true or false
****************************************************************** */
static int MkDelGreen(t)
  TR *t;
  {
    return (Refined2(t)==true)?DoRefTr(t):MkGreen(t);
  }
 
static int Refined2(t)
  TR *t;
  {
    int k = 0;
    if (NeighborRefined(t->e1,t)) k++;
    if (NeighborRefined(t->e2,t)) k++;
    if (NeighborRefined(t->e3,t)) k++;
    if (k>1) { t->mark = true; return true; }
    return false;
  }
 
static int NeighborRefined(ed,t)
  EDG *ed;
  TR *t;
  {
    TR *neighbor = ((ed->t1)==t)?(ed->t2):(ed->t1);
    if (neighbor==nil) return false;
    return ((neighbor->firstSon)==nil)?false:true;
  }
 
/* ******************************************************************
   OpenDel     prepares the triangulation modul to accept requests
               to delete a triangle. The green closure of the actual
               triangulation is removed.
   --------
   return int  true (or false)
***************************************************************** */
int OpenDel()
  {
    if (actTriang==nil)
      { ZIBStdOut("Tri: no triangulation (OpenDel)\n"); return false; }
    RelGreen();
    noDelTr = 0;
    return true;
  }
 
/* *******************************************************************
   DelTr(t)     request to delete the triangle and its brothers/sisters.
                The actual deletion is postphoned to the call
                of CloseDel
   ----------
   TR *t        Adress of the triangle
   return int   true (or false)
****************************************************************** */
int DelTr(t)
  TR *t;
  {
    TR *tFather = t->father,*son;
 
    if (tFather==nil) return true;
    if (tFather->mark) return true;
    son = tFather->firstSon;
    if ((son->father)!=tFather) son = son->father;
    if ((son->firstSon)!=nil) return true;
    son = son->next;
    if ((son->father)!=tFather) son = son->father;
    if ((son->firstSon)!=nil) return true;
    son = son->next;
    if ((son->father)!=tFather) son = son->father;
    if ((son->firstSon)!=nil) return true;
    son = son->next;
    if ((son->father)!=tFather) son = son->father;
    if ((son->firstSon)!=nil) return true;
    tFather->mark = true;
    return true;
  }
 
/* *****************************************************************
   CloseDel    deletes the marked triangles and completes the
               triangulation with the neccessary green ones
   ----------
   return int  true or false
***************************************************************** */
int CloseDel()
  { 
    int countE = actTriang->noOfEdges,
		countT = actTriang->noOfTriangles;
    

	if (DoDelTr()==false)
      {
        ZIBStdOut("Tri: failure deleting red triangles (CloseDel)\n");
        return false;
      }
 
    if (ApplyT(MkDelGreen,all)==false)
      {
        ZIBStdOut("Tri: failure deleting green triangles (CloseDel)\n");
        return false;
      }
    sprintf(globBuf,"Tri:             %4d edges, %4d triangles for green closure\n",
		(actTriang->noOfEdges)-countE,
		(actTriang->noOfTriangles)-countT);
    return true;
  }
 
/* ***************************************************************
   DoDelTr     substitute all triangles with a marked father by
               the father
   --------
   return int  true or false
*************************************************************** */
static int DoDelTr()
  {
    TR *t=actTriang->firstTriangle,*tFather,*tLastSon;
    int countP = actTriang->noOfPoints,
		countE = actTriang->noOfEdges,
		countT = actTriang->noOfTriangles;
 
    while (t!=nil)
      {
        tFather = t->father;
        if (tFather==nil) { t = t->next; continue; }
        if (tFather->mark)
          {
            tFather->mark = false;
            t = tFather->firstSon; /* to make shure: t is firstSon */
            tLastSon = t->next;
            tLastSon = tLastSon->next;
            tLastSon = tLastSon->next;
            DelTE(t, 3); DelTE(t->next, 3);
            DelTE(tLastSon->last, 3); DelTE(tLastSon, 3);
            if ((t->last)==nil) actTriang->firstTriangle = tFather;
            else                (t->last)->next = tFather;
            if ((tLastSon->next)==nil)
                                actTriang->lastTriangle = tFather;
            else                (tLastSon->next)->last = tFather;
            tFather->next = tLastSon->next;
            tFather->last = t->last;
            tFather->firstSon = nil;
			tFather->type = T_WHITE;
            DeleteTr(tFather,t,tLastSon);
            t = tFather->next;
          }
        else t = t->next;
      }
    sprintf(globBuf,"Tri: %4d points, %4d edges, %4d triangles deleted\n",
	    (actTriang->noOfPoints)-countP,
		(actTriang->noOfEdges)-countE,
		(actTriang->noOfTriangles)-countT);
	ZIBStdOut(globBuf);
    return true;
  }
 
static void DeleteTr(t,t1,t4)
  TR *t,*t1,*t4;
  {
 
    DeleteEdg(t->e1,t); DeleteEdg(t->e2,t); DeleteEdg(t->e3,t);
    ReturnE(t4->e1); ReturnE(t4->e2); ReturnE(t4->e3);
    (actTriang->noOfEdges) -= 3;
    Return4T(t1);
    (actTriang->noOfTriangles) -= 3;
    return;
  }
 
static void DeleteEdg(ed,t)
  EDG *ed;
  TR *t;
  {
    TR *neighbor = ((ed->t1)==t)?(ed->t2):(ed->t1);
    EDG *e1,*e2;
 
    if (neighbor!=nil)
      if ((neighbor->firstSon)!=nil) return;
    e1 = ed->firstSon; e2 = e1->next;
    ReturnE(e1); ReturnE(e2);
    (actTriang->noOfEdges) -= 1;
    ReturnP(e1->p2);
    (actTriang->noOfPoints)--;
    ed->firstSon = nil;
    return;
  }
