// Univariate Polynomials over the integer numbers.

#ifndef _CL_UNIVPOLY_INTEGER_H
#define _CL_UNIVPOLY_INTEGER_H

#include "cl_ring.h"
#include "cl_univpoly.h"
#include "cl_number.h"

// Normal univariate polynomials with stricter static typing:
// `cl_I' instead of `cl_ring_element'.

#ifdef notyet

typedef cl_UP_specialized<cl_I> cl_UP_I;
typedef cl_univpoly_specialized_ring<cl_I> cl_univpoly_integer_ring;
//typedef cl_heap_univpoly_specialized_ring<cl_I> cl_heap_univpoly_integer_ring;

#else

class cl_heap_univpoly_integer_ring;

class cl_univpoly_integer_ring : public cl_univpoly_ring {
public:
	// Default constructor.
	cl_univpoly_integer_ring () : cl_univpoly_ring () {}
	// Copy constructor.
	cl_univpoly_integer_ring (const cl_univpoly_integer_ring&);
	// Assignment operator.
	cl_univpoly_integer_ring& operator= (const cl_univpoly_integer_ring&);
	// Automatic dereferencing.
	cl_heap_univpoly_integer_ring* operator-> () const
	{ return (cl_heap_univpoly_integer_ring*)heappointer; }
};
// Copy constructor and assignment operator.
CL_DEFINE_COPY_CONSTRUCTOR2(cl_univpoly_integer_ring,cl_univpoly_ring)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_univpoly_integer_ring,cl_univpoly_integer_ring)

class cl_UP_I : public cl_UP {
public:
	const cl_univpoly_integer_ring& ring () const { return The(cl_univpoly_integer_ring)(_ring); }
	// Conversion.
	CL_DEFINE_CONVERTER(cl_ring_element)
	// Destructive modification.
	void set_coeff (uintL index, const cl_I& y);
	void finalize();
	// Evaluation.
	cl_I operator() (const cl_I& y) const;
public:	// Ability to place an object at a given address.
	void* operator new (size_t size) { return cl_malloc_hook(size); }
	void* operator new (size_t size, cl_UP_I* ptr) { (void)size; return ptr; }
	void operator delete (void* ptr) { cl_free_hook(ptr); }
};

class cl_heap_univpoly_integer_ring : public cl_heap_univpoly_ring {
	SUBCLASS_cl_heap_univpoly_ring()
	// High-level operations.
	void fprint (cl_ostream stream, const cl_UP_I& x)
	{
		cl_heap_univpoly_ring::fprint(stream,x);
	}
	cl_boolean equal (const cl_UP_I& x, const cl_UP_I& y)
	{
		return cl_heap_univpoly_ring::equal(x,y);
	}
	cl_UP_I zero ()
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::zero());
	}
	cl_boolean zerop (const cl_UP_I& x)
	{
		return cl_heap_univpoly_ring::zerop(x);
	}
	cl_UP_I plus (const cl_UP_I& x, const cl_UP_I& y)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::plus(x,y));
	}
	cl_UP_I minus (const cl_UP_I& x, const cl_UP_I& y)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::minus(x,y));
	}
	cl_UP_I uminus (const cl_UP_I& x)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::uminus(x));
	}
	cl_UP_I one ()
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::one());
	}
	cl_UP_I canonhom (const cl_I& x)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::canonhom(x));
	}
	cl_UP_I mul (const cl_UP_I& x, const cl_UP_I& y)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::mul(x,y));
	}
	cl_UP_I square (const cl_UP_I& x)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::square(x));
	}
	cl_UP_I expt_pos (const cl_UP_I& x, const cl_I& y)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::expt_pos(x,y));
	}
	sintL degree (const cl_UP_I& x)
	{
		return cl_heap_univpoly_ring::degree(x);
	}
	cl_UP_I monomial (const cl_I& x, uintL e)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::monomial(cl_ring_element(cl_I_ring,x),e));
	}
	cl_I coeff (const cl_UP_I& x, uintL index)
	{
		return The(cl_I)(cl_heap_univpoly_ring::coeff(x,index));
	}
	cl_UP_I create (sintL deg)
	{
		return The2(cl_UP_I)(cl_heap_univpoly_ring::create(deg));
	}
	void set_coeff (cl_UP_I& x, uintL index, const cl_I& y)
	{
		cl_heap_univpoly_ring::set_coeff(x,index,cl_ring_element(cl_I_ring,y));
	}
	void finalize (cl_UP_I& x)
	{
		cl_heap_univpoly_ring::finalize(x);
	}
	cl_I eval (const cl_UP_I& x, const cl_I& y)
	{
		return The(cl_I)(cl_heap_univpoly_ring::eval(x,cl_ring_element(cl_I_ring,y)));
	}
private:
	// No need for any constructors.
	cl_heap_univpoly_integer_ring ();
};

// Lookup of polynomial rings.
inline cl_univpoly_integer_ring cl_find_univpoly_ring (const cl_integer_ring& r)
{ return The(cl_univpoly_integer_ring) (cl_find_univpoly_ring((const cl_ring&)r)); }
inline cl_univpoly_integer_ring cl_find_univpoly_ring (const cl_integer_ring& r, const cl_symbol& varname)
{ return The(cl_univpoly_integer_ring) (cl_find_univpoly_ring((const cl_ring&)r,varname)); }

// Operations on polynomials.

// Add.
inline cl_UP_I operator+ (const cl_UP_I& x, const cl_UP_I& y)
	{ return x.ring()->plus(x,y); }

// Negate.
inline cl_UP_I operator- (const cl_UP_I& x)
	{ return x.ring()->uminus(x); }

// Subtract.
inline cl_UP_I operator- (const cl_UP_I& x, const cl_UP_I& y)
	{ return x.ring()->minus(x,y); }

// Multiply.
inline cl_UP_I operator* (const cl_UP_I& x, const cl_UP_I& y)
	{ return x.ring()->mul(x,y); }

// Squaring.
inline cl_UP_I square (const cl_UP_I& x)
	{ return x.ring()->square(x); }

// Exponentiation x^y, where y > 0.
inline cl_UP_I expt_pos (const cl_UP_I& x, const cl_I& y)
	{ return x.ring()->expt_pos(x,y); }

// Scalar multiplication.
inline cl_UP_I operator* (const cl_I& x, const cl_UP_I& y)
	{ return y.ring()->mul(y.ring()->canonhom(x),y); }
inline cl_UP_I operator* (const cl_UP_I& x, const cl_I& y)
	{ return x.ring()->mul(x.ring()->canonhom(y),x); }

// Coefficient.
inline cl_I coeff (const cl_UP_I& x, uintL index)
	{ return x.ring()->coeff(x,index); }

// Destructive modification.
inline void set_coeff (cl_UP_I& x, uintL index, const cl_I& y)
	{ x.ring()->set_coeff(x,index,y); }
inline void finalize (cl_UP_I& x)
	{ x.ring()->finalize(x); }
inline void cl_UP_I::set_coeff (uintL index, const cl_I& y)
	{ ring()->set_coeff(*this,index,y); }
inline void cl_UP_I::finalize ()
	{ ring()->finalize(*this); }

// Evaluation. (No extension of the base ring allowed here for now.)
inline cl_I cl_UP_I::operator() (const cl_I& y) const
{
	return ring()->eval(*this,y);
}

// Derivative.
inline cl_UP_I deriv (const cl_UP_I& x)
	{ return The2(cl_UP_I)(deriv((const cl_UP&)x)); }

#endif

CL_REQUIRE(cl_I_ring)


// Returns the n-th Tchebychev polynomial (n >= 0).
extern cl_UP_I cl_tschebychev (sintL n);

// Returns the n-th Hermite polynomial (n >= 0).
extern cl_UP_I cl_hermite (sintL n);

// Returns the n-th Laguerre polynomial (n >= 0).
extern cl_UP_I cl_laguerre (sintL n);

#endif /* _CL_UNIVPOLY_INTEGER_H */
