;;;
;;; This file contains the scheme code which implements
;;; the simultaneous substitution operator used in
;;; our model of substitution
;;; 
;;;
;;; Derek Lindner  buddha@theory.lcs.mit.edu
;;; Justin Liu     dondon@theory.lcs.mit.edu
;;; Brian So       brianso@theory.lcs.mit.edu
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; free variables                                                       ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (occurs-free? x e)
  (memq x (free-variables e)))

(define (free-variables exp)
  (cond ((variable? exp) (list exp))
	((lambda? exp) (difference-set (free-variables (lam-body exp))
				       (lam-formals exp)))
	((or (if? exp) (cond? exp)) (mapunion free-variables exp))
	((combination? exp) (union (free-variables (first exp))
				   (free-variables (rest exp))))
	((let? exp)
	 (union
	  (mapunion 
	   (lambda (binding)
	     (free-variables (binding-init-letc binding)))
	   (bindings-letc exp))
	  (difference-set
	   (free-variables (body-letc exp))
	   (binding-variable-list (bindings-letc exp)))))
	((letrec? exp)
	 (difference-set
	  (union
	   (mapunion
	    (lambda (binding)
	      (free-variables (binding-init-letc binding)))
	    (bindings-letc exp))
	   (free-variables (body-letc exp)))
	  (binding-variable-list (bindings-letc exp))))
	(else '() )))


;;;;;;;;;;;;;;;;;;;;
;;;              ;;;
;;; SUBSTITUTION ;;;
;;;              ;;;
;;;;;;;;;;;;;;;;;;;;

;;; For more information on these procedures, 
;;; please see the accompanying rules file.

(define (substitute substlist exp)
  (cond ((quotestable-value? exp) exp)
	((variable? exp)
	 (let ((val (assq exp substlist)))  ; (assq exp env)
	   (cond (val (cadr val))
		 (else exp))))
	((lambda? exp) (subst-lambda substlist exp))
        ((let? exp)    (subst-let substlist exp))
	((letrec? exp) (subst-letrec substlist exp))
	((if? exp)     (subst-if substlist exp))
	((cond? exp)   (subst-cond substlist exp))
	((apply? exp)  (subst-apply substlist exp))
	((quote? exp) exp)
	((sequence? exp) (subst-sequence substlist exp)) ;;use nonbinding keyword form
                                                         ;; for if begin, and or
        ((and? exp) (subst-and substlist exp))
        ((or? exp)  (subst-or substlist exp))
	((combination? exp)
	 (map (lambda (operexp) (substitute substlist operexp))
	      (cons (operator exp) (operands exp)))) ;abstraction barrier
	(else (sm-error "SUBSTITUTE can't handle expression" exp))))

(define (subst-lambda substlist exp)
  (let*  ((Y (lam-formals exp))
	  (Xn (variable-list substlist))
	  (Mn (source-exp-list substlist))
	  (FVN (free-variables (lam-body exp)))
	  (Int (intersect Xn Y))
	  (FVMn (mapunion free-variables Mn)))
    (if Int; (not (null? Int))
	(substitute (restrict substlist (difference-set Xn Y)) exp)
	(let* ((allfreevars (union FVN FVMn))
	       (freshvlist (fresh-no-bug allfreevars Y FVMn Xn FVN))
	       (exsblist (extend-sub substlist freshvlist Y)))
	  (make-lambda freshvlist (substitute exsblist (lam-body exp)))))))

(define (subst-let substlist letexp)
  (if (named-let? letexp)
      (substitute substlist (reduce-let letexp))
      (let* ((Xn (variable-list substlist))
	     (Varm (binding-variable-list (bindings-letc letexp)))
	     (Initm (binding-init-list (bindings-letc letexp)))
	     (body (body-letc letexp))
	     (FVbody (free-variables body))
	     (Xl (intersect Xn (difference-set FVbody Varm)))
	     (newsubstlist (restrict substlist Xl))
	     (Ml (source-exp-list newsubstlist))
	     (FVMl (mapunion free-variables Ml))
	     (Zn (freshvlist-letc Varm FVMl FVbody)))
	(make-let (make-bindings-letc
		   Zn
		   (substitute substlist Initm))
		  (substitute (extend-sub
			       newsubstlist
			       Zn
			       Varm)
			      body)))))
  
(define (subst-letrec substlist letrecexp)
  (let* ((Xn (variable-list substlist))
	 (Varm (binding-variable-list (bindings-letc letrecexp)))
	 (Initm (binding-init-list (bindings-letc letrecexp)))
	 (FVInitm (mapunion free-variables Initm))
	 (body (body-letc letrecexp))
	 (FVbody (free-variables body))
	 (Xl (intersect Xn (difference-set (union FVbody FVInitm) Varm)))
	 (newsubstlist (restrict substlist Xl))
	 (Ml (source-exp-list newsubstlist))
	 (FVMl (mapunion free-variables Ml))
	 (Zn (freshvlist-letc Varm FVMl FVbody)))
    (make-letrec (make-bindings-letc         
		  Zn
		  (substitute newsubstlist Initm))
		 (substitute (extend-sub
			     newsubstlist
			     Zn
			     Varm)
			    body))))

(define (subst-if substlist ifexp)
  (make-if (list (substitute substlist (test-if ifexp))
		 (substitute substlist (consequent-if ifexp))
		 (substitute substlist (alternative-if ifexp)))))

(define (subst-cond substlist condexp)
  (make-cond (substitute substlist (clauses-cond condexp))))

(define (subst-apply substlist appexp)
  (make-apply (substitute substlist (procedure-apply appexp))
	      (substitute substlist (arglist-apply appexp))))

(define (subst-sequence substlist seqexp)
  (make-sequence (substitute substlist (rest seqexp))))

(define (subst-and substlist andexp)
  (make-and (substitute substlist (rest andexp))))

(define (subst-or substlist orexp)
  (make-or (substitute substlist (rest orexp))))


;;;;;;;;;;;;;;;
;;;         ;;;
;;; GEN-SYM ;;;
;;;         ;;;
;;;;;;;;;;;;;;;

;;; This section contains procedures which generates
;;; unique symbols needed to handle renaming in substitution

(define (fresh-no-bug allfreevars Y FVMn Xn FVN)
  (define (new-z-if-needed a-y)
    (if (or (not (memq a-y FVMn))
	    (not (intersect Xn FVN)))
	a-y
	(fresh1 allfreevars a-y)))
  (map new-z-if-needed Y))

(define (freshvlist-letc Varm FVMl FVbody)
  (let ((allfreevars (accunion (list Varm FVMl FVbody))))
    (define (new-z-if-needed var)
      (if (memq var FVMl)
	  (fresh1 allfreevars var)
	  var))
    (map new-z-if-needed Varm)))

; FRESH1 generates a symbol which is not in varset

(define (fresh1 varset var)
  (let ((a (next-symbol var)))
    (if (memq a varset)
	(fresh1 varset a)
	a)))

; NEXT-SYMBOL generates a symbol

(define (next-symbol var)
  (let* ((varstring (symbol->string var))
         (varname-end-index (string-find-next-char varstring #\#)))
    (if varname-end-index
        (let ((varnamepart (string-head varstring varname-end-index)))
          (string->symbol
           (string-append varnamepart
                          "#"
                          (number->string (fresh-var-seed! (string->symbol varnamepart))))))
        (string->symbol
         (string-append varstring
                        "#"
                        (number->string (fresh-var-seed! var)))))))


; COMPLETELY-FRESH-Zs creates a list of N completely fresh symbols

(define (completely-fresh-zs n)
  (if (<= n 0)
      '()
      (cons (next-symbol fresh)
	    (completely-fresh-zs (-1+ n)))))

(define (fresh-var-seed! var)           ;REVISED
  (let ((var-num-binding (assq var numbered-variable-list)))
    (cond (var-num-binding
           (set-cdr! var-num-binding (1+ (cdr var-num-binding)))
           (cdr var-num-binding))
          (else (set! numbered-variable-list
                      (cons (cons var 1) numbered-variable-list))
                1))))

(define numbered-variable-list '())

;;; Abstractions for handling substitution lists

(define make-subst list)

(define (make-substlist variable-list source-exp-list)
  (map make-subst variable-list source-exp-list))

; SMART-MAKE-SUBSTLIST makes a substitution list omitting 
; the substitutionsm of a variable for itself

(define (smart-make-substlist variable-list source-exp-list)
  (cond ((null? variable-list)
	 '())
	((eq? (car variable-list) (car source-exp-list))
	 (smart-make-substlist (cdr variable-list) (cdr source-exp-list)))
	(else (cons (make-subst (car variable-list)
				(car source-exp-list))
		    (smart-make-substlist (cdr variable-list)
					  (cdr source-exp-list))))))

(define (variable-list substlist)
  (map car substlist))

(define (source-exp-list substlist)
  (map cadr substlist))

; RESTRICT returns an substitution list containing only those
; pairs whose first element is in KEYLIST.

(define (restrict substlist keylist)
  (cond ((null? substlist) substlist)
	((memq (caar substlist) keylist)
	 (cons (car substlist) (restrict (cdr substlist) keylist)))
	(else (restrict (cdr substlist) keylist))))

(define (extend-sub original-list new-source-exp new-params)
  (append original-list (smart-make-substlist new-params new-source-exp)))
