/*
  Solve a system of linear equations

  Written by Al Vermeulen 1 Sept 1989
*/

#include "stream.h"
#include "rw/DLUDecomp.h"

filebuf* open_file_for_input( filebuf& f, char* filename );
filebuf* open_file_for_output( filebuf& f, char* filename );
int      readFlags( int, char**, char&, char&, char& );

static int verboseOn = 0;

main( int argc, char** argv )
{
  DGEMatrix A;
  // Set up variables indicating command line options

  char    AFormat = 'p';	// pretty print, ascii, or binary
  char    xFormat = 'p';
  char    bFormat = 'p';
  int	  argCount= readFlags( argc, argv, AFormat, xFormat, bFormat );
  {				// New scope so filebuf gets destructed
    filebuf ABuf;		// Filebuf for A (of Ax=b fame)

    // Read in the matrix

    {	// Keep the streams in a different scope from the filebufs
      istream Astream( open_file_for_input(ABuf,argv[argCount++]) );
      switch (AFormat) {
        case 'p':
	    Astream >> A;
	    break;
        case 'a':
	    A.readFrom(Astream);
	    break;
        case 'b':
	    A.readFrom(ABuf.fd);
	    break;
        default:
	    cerr << argv[0] << ": bad A matrix format flag\n";
	    exit(-1);
      }
      if (verboseOn) cout << "A(" << argv[argCount-1] << "):\n" << A << "\n";
    }
  }

  // Build the decomposition

  DLUDecomp decomp(A);

  // Go through right hand side/output filename pairs

  while( (argCount+1) < argc ) {
    filebuf bBuf;
    filebuf xBuf;
    {
      istream bstream( open_file_for_input(bBuf,argv[argCount++]) );
      ostream xstream( open_file_for_output(xBuf,argv[argCount++]) );

      DoubleVec b;
      switch (bFormat) {			// Read in the b vector
        case 'p':
	  bstream >> b;
	  break;
        case 'a':
	  b.readFrom(bstream);
	  break;
        case 'b':
	  b.readFrom(bBuf.fd);
	  break;
        default:
	  cerr << argv[0] << ": bad b matrix format flag\n";
	  exit(-1);
      }
      if (verboseOn) cout << "b(" << argv[argCount-2] << "):\n" << b << "\n";

      DoubleVec x(solve(decomp,b));	// Create the solution vector

      switch (xFormat) {			// Output the x vector
        case 'p':
	  xstream << x << "\n";
	  break;
        case 'a':
	  x.storeOn(xstream);
	  break;
        case 'b':
	  x.storeOn(xBuf.fd);
	  break;
        default:
	  cerr << argv[0] << ": bad x matrix format flag\n";
	  exit(-1);
      }
      if (verboseOn) cout << "x(" << argv[argCount-1] << "):\n" << x << "\n";
    }
  }
}
      

// Read the arguments on the command line, everything into this
// routine is a reference,
// all are changed according to the command line (except argc/v)

int readFlags( int argc, char** argv,
	       char& AFormat, char& xFormat, char& bFormat )
{
  for( int argcount=1; argcount<argc; argcount++) {
    char* arg = argv[argcount];
    if (arg[0] == '-') {
      switch( arg[1] ) {
	case 'A':
	  if (arg[2] == '\0')			// Allow either -Ax or -A x
	    AFormat = argv[++argcount][0];
	  else
	    AFormat = arg[2];
	  break;
	case 'x':
	  if (arg[2] == '\0')
	    xFormat = argv[++argcount][0];
	  else
	    xFormat = arg[2];
	  break;
	case 'b':
	  if (arg[2] == '\0')
	    bFormat = argv[++argcount][0];
	  else
	    bFormat = arg[2];
	  break;
	case 'v':
	  verboseOn = 1;
	  break;
	default:
	  cerr << argv[0] << ": unknown flag " << arg << "\n";
	  exit(-1);
      }
    }
    else {
      return argcount;
    }
  }
  cerr << argv[0] << ": no matrix file given\n";
  exit(-1);
  return 0;
}

/* These two routines are useful for opening files */

static filebuf* open_file_for_input( filebuf& f, char* filename )
{
  if ( (f.opened) || (f.open(filename,input) == 0) )
    {
      perror( form("Could not open %s", filename) );
      exit(-1);
    }
  return &f;
}

static filebuf* open_file_for_output( filebuf& f, char* filename )
{
  if ( (f.opened) || (f.open(filename,output) == 0) )
    {
      perror( form("Could not open %s", filename) );
      exit(-1);
    }
  return &f;
}
