/*
 *  pixedit.c     edit pixels in a binary image         -Brian Tierney,  LBL
 *    part of 'segal'
 */

#include "segal.h"

/* these need to be global */
pixedit_win_objects *edit_win;
extern view_win_objects *view_win;
extern segal_win_objects *segal_win;

/************************************************************/
void
edit_win_init(owner)
    Xv_opaque owner;
{
    void      Edit_make_simple_colormap();
    void      create_cursors(), pixedit_resize_proc();

    edit_win = pixedit_win_objects_initialize(NULL, owner);

    /* use same colormap as the view window */
    (void) xv_set(edit_win->canvas, WIN_CMS_NAME, "segal", NULL);

    create_cursors();
}

/**************************************************************/
void
edit_setup()
{
    void      Edit_map_image_to_lut(), edit_repaint_proc(), zoom();
    void      zoom(), create_zoom_ximages(), image_repaint_proc();
    void      set_window_size_and_loc();

    if ((int) xv_get(view_win->win, XV_SHOW, NULL) == FALSE)
	return;			/* must load images first */

    /* set default display type for editting */
    if (himage.fp != NULL) {
	if (segal.mode == 2 && segal.display_type == 0) {
	    segal.blend_type = 0;
	    segal.mask_type = 0;
	    segal.display_type = 2;
	}
    } else {			/* mask only */
	segal.display_type = 1;
    }

    (void) xv_set(segal_win->blend_type, PANEL_VALUE, 0, NULL);
    (void) xv_set(segal_win->mask_type, PANEL_VALUE, 0, NULL);
    (void) xv_set(segal_win->display_type, PANEL_VALUE,
		  segal.display_type, NULL);

    /* intial zoom */
    zoom_info.mag = 2;
    (void) xv_set(edit_win->mag_item, PANEL_VALUE, 1, NULL);

    edit_info.changing = 1;	/* flag for chaning zoom window size (not
				 * working yet)  */

    set_window_size_and_loc(ZOOM_WIN_SIZE);

    create_zoom_ximages();

    zoom_info.cx = segal.cols / 2;
    zoom_info.cy = segal.rows / 2;
    zoom_info.x = zoom_info.cx - (zoom_info.size / 2 / zoom_info.mag);
    zoom_info.y = zoom_info.cy - (zoom_info.size / 2 / zoom_info.mag);

    zoom();

    image_repaint_proc();
}

/*************************************************************/
void
set_window_size_and_loc(win_size)
    int       win_size;
{
    int       min_dim;
    Rect      srect, erect;

    min_dim = MIN(segal.cols, segal.rows);

    if (min_dim * zoom_info.mag < win_size) {
	zoom_info.size = (min_dim * zoom_info.mag) - 20;
    } else {
	zoom_info.size = win_size;
    }

#ifdef DEBUG
    fprintf(stderr, "zoom window size set to %d \n", zoom_info.size);
#endif

    edit_info.canvas_width = edit_info.canvas_height = zoom_info.size;

    /* compute  window size */
    edit_info.win_height = edit_info.canvas_height + EDIT_CONTROL_HEIGHT + 10;
    if (edit_info.canvas_width >= EDIT_CONTROL_WIDTH)
	edit_info.win_width = edit_info.canvas_width + 5;
    else
	edit_info.win_width = EDIT_CONTROL_WIDTH;

    /* set size */
    (void) xv_set(edit_win->canvas, XV_WIDTH, edit_info.canvas_width, NULL);
    (void) xv_set(edit_win->canvas, XV_HEIGHT, edit_info.canvas_height, NULL);
    (void) xv_set(edit_win->win, XV_WIDTH, edit_info.win_width, NULL);
    (void) xv_set(edit_win->win, XV_HEIGHT, edit_info.win_height, NULL);

    XFlush(display);		/* sometimes window size doesnt change ??? */

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == TRUE)
	return;

    (void) xv_set(edit_win->win, XV_SHOW, TRUE, NULL);

    /*
     * set location: NOTE: to set the location of a frame using
     * 'frame_set_rect'  (at least under the twm window manager) XV_SHOW must
     * be TRUE first
     */

    /* set edit window location relative to segal window */
    frame_get_rect(segal_win->win, &srect);

    if (srect.r_top > 0 && srect.r_left > 0) {	/* segal window must be
						 * mapped */
	frame_get_rect(edit_win->win, &erect);
	erect.r_top = srect.r_top + srect.r_height;
	erect.r_left = srect.r_left;
	frame_set_rect(edit_win->win, &erect);
    }
#ifdef DEBUG
    fprintf(stderr, " segal frame at: %d,%d \n", srect.r_top, srect.r_left);
    fprintf(stderr, " setting edit frame to: %d,%d : size: %d, %d\n",
	    erect.r_top, erect.r_left, erect.r_width, erect.r_height);
#endif
}

/****************************************************************/
void
edit_repaint_proc()
{
    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == FALSE)
	return;

#ifdef DEBUG
    fputs("pixedit: edit_repaint_proc\n", stderr);
#endif

    if (segal.display_type == 0 && zoom_image != NULL)
	XPutImage(display, edit_xid, gc, zoom_image, 0, 0, 0, 0,
		  zoom_image->width, zoom_image->height);
    if (segal.display_type == 1 && zoom_mask_image != NULL)
	XPutImage(display, edit_xid, gc, zoom_mask_image, 0, 0, 0, 0,
		  zoom_mask_image->width, zoom_mask_image->height);
    if (segal.display_type == 2 && zoom_blend_image != NULL)
	XPutImage(display, edit_xid, gc, zoom_blend_image, 0, 0, 0, 0,
		  zoom_blend_image->width, zoom_blend_image->height);
}

/**************************************************************/
Notify_value
edit_event_proc(win, event, arg, type)
    Xv_Window win;
    Event    *event;
    Notify_arg arg;
    Notify_event_type type;
{
    void      draw_location(), paint();
    int       x, y;

/*
  mouse events:
    any button: paint
    shift and control, any button: show gray value
    meta, any button:   show object statistics
*/

#ifdef DEBUG
    fprintf(stderr, "event id is %d \n", event_id(event));
#endif

    /* meta events */
    if (event_meta_is_down(event)) {
	if (event_is_down(event)) {	/* down events only */
	    switch (event_id(event)) {
	    case MS_RIGHT:
	    case MS_LEFT:
	    case MS_MIDDLE:
		x = zoom_info.x + (event_x(event) / zoom_info.mag);
		y = zoom_info.y + (event_y(event) / zoom_info.mag);
		get_stats(x, y);
		break;
	    }
	}
	return notify_next_event_func(win, (Notify_event) event, arg, type);

    }
    /* cntrl and shift events */
    if (event_ctrl_is_down(event) || event_shift_is_down(event)) {
	switch (event_id(event)) {
	case MS_RIGHT:
	case MS_LEFT:
	case MS_MIDDLE:
	case LOC_DRAG:
	    x = zoom_info.x + (event_x(event) / zoom_info.mag);
	    y = zoom_info.y + (event_y(event) / zoom_info.mag);
	    if (event_is_down(event))
		draw_location(x, y, 1);	/* routine in view.c */
	    else
		draw_location(x, y, 0);
	    break;
	}
	return notify_next_event_func(win, (Notify_event) event, arg, type);
    }
    /* other events */
    switch (event_id(event)) {
    case MS_RIGHT:
    case MS_LEFT:
    case MS_MIDDLE:
    case LOC_DRAG:
	if (event_is_down(event))
	    paint(event_x(event), event_y(event));
	break;
    default:
	break;
    }

    return notify_next_event_func(win, (Notify_event) event, arg, type);
}

/******************************************************/
void
paint(xpos, ypos)		/* determine if ok to paint and what kind of
				 * painting */
    int       xpos, ypos;
{
    void      draw_points();

    if (segal.mode < 2 || segal.display_type < 1)
	return;

    if (xpos < 0)
	xpos = 0;
    if (ypos < 0)
	ypos = 0;
    if (xpos >= zoom_info.size)
	xpos = zoom_info.size - 1;
    if (ypos >= zoom_info.size)
	ypos = zoom_info.size - 1;

    if (((int) xv_get(edit_win->edit_mode, PANEL_VALUE, NULL)) == 0)
	draw_points(xpos, ypos, PVAL);
    else
	draw_points(xpos, ypos, 0);
}

/**************************************************************/
void
draw_points(x, y, val)
    int       x, y, val;
{
    void      paint_zoom_image(), paint_main_image();
    int       brush_size, paint_size, cnum, round_curs;

    segal.changed = 1;		/* mask modified flag */

#ifdef DEBUG
    fprintf(stderr, " painting from point: %d,%d \n", x, y);
#endif

    XSetFunction(display, gc, GXcopy);

    brush_size = ((int) xv_get(edit_win->brush_type, PANEL_VALUE, NULL)) + 1;
    round_curs = ((int) xv_get(edit_win->cursor_item, PANEL_VALUE, NULL));

    cnum = brush_size * zoom_info.mag;
    if (my_cursor[cnum].size <= 0) {
	fprintf(stderr, " cursor structure error \n");
	return;
    }
    if (brush_size == 7)	/* last brush size setting is 20 */
	brush_size = 20;
    paint_size = brush_size * zoom_info.mag;

    paint_zoom_image(x, y, val, paint_size, cnum, round_curs);

    paint_main_image(x, y, val, brush_size, brush_size, round_curs);
}

/**********************************************************/
void
paint_zoom_image(x, y, val,paint_size, cnum, rcurs)
    int       x, y, val;
    int       paint_size, cnum, rcurs;
{

    register int i, j, i2, j2;
    int       is = 0, js = 0, rx, ry;
    u_long    paint_pixel, blend_pixel, get_blend_pixel();
    u_char  **pmask;

    if (val == 0)
	paint_pixel = colors[0];
    else
	paint_pixel = colors[NGRAY - 1];
    XSetForeground(display, gc, paint_pixel);

    if (rcurs) {		/* round cursor, uses mask to determine paint
				 * locations */
	pmask = my_cursor[cnum].paint_mask;
	is = js = my_cursor[cnum].corner;
	paint_size++;
	x -= paint_size / 2. + .5;
	y -= paint_size / 2. + .5;
	if (y < 0)
	    y = 0;
	if (x < 0)
	    x = 0;
#ifdef DEBUG
	fprintf(stderr, " paint size: %d;  cursor size: %d \n",
		paint_size, my_cursor[cnum].size);
	fprintf(stderr, "  x: %d, y: %d, i2: %d, j2: %d \n", x, y, i2, j2);
	for (i = is; i < is + paint_size; i++) {
	    fprintf(stderr, " \n");
	    for (j = js; j < js + paint_size; j++) {
		fprintf(stderr, "%d ", pmask[i][j]);
	    }
	}
	fprintf(stderr, "\n");
#endif
    }
    i2 = is;			/* for handling round cursors */
    j2 = js;

    for (i = y; i < y + paint_size && i < zoom_image->height; i++) {
	for (j = x; j < x + paint_size && j < zoom_image->width; j++) {
	    if ((rcurs && pmask[i2][j2] > 0) || (!rcurs)) {
		rx = zoom_info.x + (j / zoom_info.mag);
		ry = zoom_info.y + (i / zoom_info.mag);
		XPutPixel(zoom_mask_image, j, i, paint_pixel);
                if (himage.fp != NULL) {
                    blend_pixel = get_blend_pixel(rx, ry, val);
                    XPutPixel(zoom_blend_image, j, i, blend_pixel);
                }
		if (segal.display_type == 2)
		    XSetForeground(display, gc, blend_pixel);
		else
		    XSetForeground(display, gc, paint_pixel);
                XDrawPoint(display, edit_xid, gc, j, i);
	    }
	    j2++;
	}
	j2 = js;
	i2++;
    }
}

/*******************************************************/
void
paint_main_image(x, y, val, paint_size, cnum, rcurs)
    int       x, y, val;
    int       paint_size, cnum, rcurs;

 /*
  * Implementation note: it would be faster to just do the blend once in the
  * view image, then replicate to the zoom image. However, magnify the
  * pixelization effects with the round cursors.  -BT
  */
{
    register int i, j, i2, j2;
    int       is = 0, js = 0, rx, ry;
    u_long    paint_pixel, blend_pixel, get_blend_pixel();
    u_char  **pmask;

    if (val == 0)
	paint_pixel = colors[0];
    else
	paint_pixel = colors[NGRAY - 1];

    if (rcurs) {		/* round cursor, uses mask to determine paint
				 * locations */
	pmask = my_cursor[cnum].paint_mask;
	is = js = my_cursor[cnum].corner;
	paint_size++;
	rx = zoom_info.x + (((float) x / zoom_info.mag) -
			    ((float) paint_size / 2.) + .5);
	ry = zoom_info.y + (((float) y / zoom_info.mag) -
			    ((float) paint_size / 2.) + .5);
	if (rx < 0)
	    rx = 0;
	if (ry < 0)
	    ry = 0;
#ifdef DEBUG
	fprintf(stderr, " cursor size: %d \n", my_cursor[cnum].size);
	fprintf(stderr, "  rx: %d, ry: %d, i2: %d, j2: %d \n", rx, ry, i2, j2);
	for (i = is; i < is + paint_size; i++) {
	    fprintf(stderr, " \n");
	    for (j = js; j < js + paint_size; j++) {
		fprintf(stderr, "%d ", pmask[i][j]);
	    }
	}
	fprintf(stderr, "\n");
#endif
    } else {
	rx = zoom_info.x + ((float) (x / zoom_info.mag) + .5);
	ry = zoom_info.y + ((float) (y / zoom_info.mag) + .5);
    }

    i2 = is;			/* for handling round cursors */
    j2 = js;
    for (i = ry; i < ry + paint_size && i < segal.rows; i++) {
	for (j = rx; j < rx + paint_size && j < segal.cols; j++) {
	    if ((rcurs && pmask[i2][j2] > 0) || (!rcurs)) {

		work_buf[i][j] = val;
		if (himage.fp != NULL) {
		    blend_pixel = get_blend_pixel(j, i, val);
		    XPutPixel(blend_image, j, i, blend_pixel);
		}
		if (segal.display_type == 2 && himage.fp != NULL) {
		    XSetForeground(display, gc, blend_pixel);
		    XDrawPoint(display, view_xid, gc, j, i);
		} else {
		    XSetForeground(display, gc, paint_pixel);
		    XDrawPoint(display, view_xid, gc, j, i);
		}
		XSetForeground(display, gc, paint_pixel);
		XPutPixel(mask_image, j, i, paint_pixel);
	    }
	    j2++;
	}
	j2 = js;
	i2++;
    }
}

/***************************************************************/
void
mask_restore_proc(item, event)
    Panel_item item;
    Event    *event;
{
    void      edit_repaint_proc(), blend(), image_repaint_proc();

    if (segal.changed == 0)
	return;

    set_watch_cursor();
    bcopy((char *) hmask.data[0], (char *) work_buf[0],
	  segal.rows * segal.cols);
    segal.changed = 0;

    if (himage.fp != NULL) {
	blend(himage.data[0], work_buf[0], (u_char *) blend_image->data,
	      segal.rows * segal.cols);
    }
    map_image_to_lut(work_buf[0], mask_image->data,
		     mask_image->width * mask_image->height);

    image_repaint_proc();

    zoom();

    edit_repaint_proc();

    unset_watch_cursor();
}

/****************************************************************/
void
mag_proc(item, event)		/* procedure called when change the
				 * magnification */
    Panel_item item;
    Event    *event;
{
    static int last_mag = -1;
    void      zoom(), image_repaint_proc(), edit_repaint_proc();

    zoom_info.mag = ((int) xv_get(edit_win->mag_item, PANEL_VALUE, 0)) + 1;
    if (zoom_info.mag == last_mag)
	return;

    set_watch_cursor();

    edit_info.changing = 1;
    set_window_size_and_loc(ZOOM_WIN_SIZE);
    create_zoom_ximages();

    zoom_info.x = zoom_info.cx - (zoom_info.size / 2 / zoom_info.mag);
    zoom_info.y = zoom_info.cy - (zoom_info.size / 2 / zoom_info.mag);

    /* redo image */
    image_repaint_proc();

    /* redo zoom images */
    zoom();

    edit_repaint_proc();

    last_mag = zoom_info.mag;
    unset_watch_cursor();
}

/*************************************************************/
Notify_value
paint_win_event_proc(win, event, arg, type)
    Xv_Window win;
    Event    *event;
    Notify_arg arg;
    Notify_event_type type;
{
    void      pixedit_resize_proc();

    switch (event_id(event)) {
    case LOC_WINENTER:
	/* slow down the mouse for painting */
	XChangePointerControl(display, True, True, 2, 3, 2);	/* 2/3 speed */
	break;
    case LOC_WINEXIT:
	XChangePointerControl(display, True, True, 1, 1, 1);
	break;
#ifdef RESIZE			/* not currently working */
    case WIN_RESIZE:
	pixedit_resize_proc();
#endif
    default:
	break;
    }
    return notify_next_event_func(win, event, arg, type);
}

/*********************************************************/
void
pixedit_resize_proc()
{				/* DOESNT WORK !! */
    /*
     * Note: this will get called many time when we dont want it to be
     * called, so all the 'ifs' at the beginning are to try to screen out
     * unwanted calls
     */

    /* This not yet working correctly: try to fix later */

    static int sw = 0, sh = 0;
    int       win_size;
    Rect      rect;

    if ((int) xv_get(edit_win->win, XV_SHOW, NULL) == FALSE)
	return;

    frame_get_rect(edit_win->win, &rect);

    if (sw == 0 && sh == 0) {
	sw = rect.r_width;
	sh = rect.r_height;
	return;
    }
    /* make sure changes at least 20 pixels */
    if ((sw > rect.r_width - 20 && sw < rect.r_width + 20) ||
	(sh > rect.r_height && sh < rect.r_height + 20))
	return;

    if (edit_info.changing == 1 || rect.r_width == 0 || rect.r_height == 0) {
	edit_info.changing = 0;
	return;			/* ignore resize event */
    }
    if (verbose)
	fprintf(stderr, "window resized \n");

    zoom_info.mag = ((int) xv_get(edit_win->mag_item, PANEL_VALUE, 0)) + 1;

    win_size = MIN(MAX(rect.r_width, EDIT_CONTROL_WIDTH),
		   (rect.r_height - EDIT_CONTROL_HEIGHT));
    if (verbose)
	fprintf(stderr, " win_size set to: %d \n", win_size);

    set_watch_cursor();
    set_window_size_and_loc(win_size);
    create_zoom_ximages();

    zoom_info.x = zoom_info.cx - (zoom_info.size / 2 / zoom_info.mag);
    zoom_info.y = zoom_info.cy - (zoom_info.size / 2 / zoom_info.mag);

    zoom();

    edit_repaint_proc();

    image_repaint_proc();
    unset_watch_cursor();

    sw = rect.r_width;
    sh = rect.r_height;

}
