/*****************************************************************************/
/* 				   inter.c				     */
/*===========================================================================*/
/* 									     */
/*	inter.c	holds all of the functions that convert the source	     */
/*	statements into the intermediate runtime structures		     */
/*									     */
/*   Copyright (C) 1990 by Ron Sass.					     */
/*									     */
/*   Permission to use, copy, modify, and distribute this software and	     */
/*   its documentation for any purpose is hereby granted, provided that	     */
/*   no fee beyond distribution costs is charged, the above copyright	     */
/*   notice appear in all copies, and that both that copyright notice	     */
/*   and this permission notice appear in the supporting documentation.	     */
/*   This software is provided "as is" without express or implied warranty.  */
/* 									     */
/*****************************************************************************/

#include <stdio.h>
#include "runtime.h"
#include "symbol.h"
#include "inter.h"
#include "calc.h"

/*__________________________  Global Variables  _____________________________*/
char	*procorder[512] ;		/* maximum number of procs */
int	procnumber = 0 ;		/* current number of procs */
proc_t	*proc_head = 0 ;		/* head of proc sources */

/*___________________________  Local Variables  _____________________________*/
static dblock_t	defdesc = { 0 , 1.0 , 30.0 } ;
static proc_t	newproc = {
	0,				/* function name */
	{ 0 , 1.0 , 30.0 },		/* descriptor block (def) */
	0,				/* trigger list */
	0,				/* variable list */
	0				/* next in link */
} ;

/*---------------------------------------------------------------------------*/
/* 				 create_proc				     */
/*	o  copies static proc_t newproc into a dynamic list		     */
/*	o  names the procedure						     */
/*	o  resets newproc's values					     */
/*	o  removes the symbol name					     */
/*---------------------------------------------------------------------------*/
void
  create_proc ( name )
sym_t	*name ;
{
	register proc_t	*tmp ;
	register trig_t	*t1, *t2 ;

	tmp = (proc_t *)malloc(sizeof(proc_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in add procedure") ;
	    exit(-1) ;
	}
	bcopy(&newproc,tmp,sizeof(proc_t)) ;
	tmp->procname = name->u.idval ;
	tmp->next = proc_head ;
	proc_head = tmp ;
	free(name) ;
	/*........................ re-init newproc ..........................*/
	bcopy(&defdesc,&newproc.desc,sizeof(dblock_t)) ;
	t1 = newproc.trig_head ;
	while( t1!=0 ) {
	    t2=t1->next ;
	    free(t1->trigname) ;
	    free(t1) ;
	    t1=t2 ;
	}
	newproc.trig_head = 0 ;
	newproc.var_head = 0 ;
}

/*---------------------------------------------------------------------------*/
/* 				  add_order				     */
/*	o  adds a procedure to the order list				     */
/*	o  NOTE: procedure may not be defined yet, but that's ok	     */
/*	o  removes symbol						     */
/*---------------------------------------------------------------------------*/
void
  add_order ( proc_name )
sym_t	*proc_name ;
{
	procorder[procnumber++] = proc_name->u.idval ;
	free(proc_name) ;
}

/*---------------------------------------------------------------------------*/
/* 				  add_desc				     */
/*	o  sets a value in the procedure's descriptor block		     */
/*	o  NOTE:  runtime, speed cannot be changed once a trigger has	     */
/*	   been calculated						     */
/*	o  removes both symbols						     */
/*---------------------------------------------------------------------------*/
void
  add_desc ( name , val )
sym_t	*name, *val ;
{
	FILE	*fp ;

#define	CASEOF(XX)	if( strcmp(name->u.idval,XX)==0 )
	CASEOF("source") {
	    if( val->type!=S_ID ) {
		yyerror("source must be an identifier") ;
		exit(1) ;
	    }
	    if( (fp=fopen(val->u.idval,"r"))==0 ) {
		yyerror("cannot open file") ;
		exit(1) ;
	    }
	    fclose(fp) ;
	    newproc.desc.source = val->u.idval ;
	} else CASEOF("runtime") {
	    if( val->type!=S_NUM ) {
		yyerror("runtime must be a number") ;
		exit(1) ;
	    }
	    if( newproc.trig_head!=0 )
		yyerror("runtime not changed, trigger already set") ;
	    else
		newproc.desc.runtime = val->u.numval ;
	} else CASEOF("speed") {
	    if( val->type!=S_NUM ) {
		yyerror("speed must be a number") ;
		exit(1) ;
	    }
	    if( newproc.trig_head!=0 )
		yyerror("speed not changed, trigger already set") ;
	    else
		newproc.desc.speed = val->u.numval ;
	} else {
	    yyerror("descriptor unknown; ignoring") ;
	    if( val->type==S_ID )	free(val->u.idval) ;
	}
#undef CASEOF
	free(name->u.idval) ;
	free(name) ;
	free(val) ;
}

/*---------------------------------------------------------------------------*/
/* 				  add_trig				     */
/*	o  calculates frame number from parameter, runtime, and speed	     */
/*	   (taking the last two from the procedure's descriptor block)	     */
/*	o  the name and number are stored at the head of the trigger	     */
/*	   list								     */
/*	o  NOTE:  obviously 0<val<runtime				     */
/*	o  removes both symbols						     */
/*---------------------------------------------------------------------------*/
void
  add_trig ( name , val )
sym_t	*name, *val ;
{
	register trig_t	*tmp ;

	tmp = (trig_t *)malloc(sizeof(trig_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in add trigger") ;
	    exit(-1) ;
	}
	if( 0<val->u.numval && val->u.numval<newproc.desc.runtime ) {
	    tmp->trigname = name->u.idval ;
	    tmp->trigtime = (int)newproc.desc.speed*val->u.numval ;
	    tmp->next = newproc.trig_head ;
	    newproc.trig_head = tmp ;
	    free(name) ;
	    free(val) ;
	} else {
	    yyerror("no trigger can be zero") ;
	    exit(1) ;
	}
}

/*---------------------------------------------------------------------------*/
/* 				   add_var				     */
/*	o  creates a variable called name and puts it at the head of	     */
/*	   the var list							     */
/*	o  puts interval first at beginning of the interval list in	     */
/*	   name								     */
/*	o  sets the frame start number in the interval to zero		     */
/*	o  adds the rest of the interval list (which may be empty) to	     */
/*	   the end of first						     */
/*	o  NOTE:  name must NOT already exist				     */
/*	o  removes all three symbols					     */
/*---------------------------------------------------------------------------*/
void
  add_var ( name , first , rest )
sym_t	*name, *first, *rest ;
{
	register var_t	*tmp ;
	register var_t	*iter ;
	register frm_t	diff ;

	iter = newproc.var_head ;
	while( iter!=0 && strcmp(iter->vname,name->u.idval)!=0 )
	    iter = iter->next ;
	if( iter!=0 ) {
	    yyerror("variable declared twice") ;
	    exit(1) ;
	}
	tmp = (var_t *)malloc(sizeof(var_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in add var") ;
	    exit(-1) ;
	}
	tmp->vname = name->u.idval ;
	tmp->intlist = first->u.intlist ;
	tmp->intlist->next = rest->u.intlist ;
	tmp->intlist->start = 0 ;
	tmp->next = newproc.var_head ;
	newproc.var_head = tmp ;
	if( tmp->intlist->next==0 )
	    diff = (int)(newproc.desc.runtime*newproc.desc.speed) ;
	else
	    diff = tmp->intlist->next->start ;
	tmp->intlist->u.params=tmp->intlist->init(diff,tmp->intlist->u.ptlist) ;
	free(rest) ;
	free(first) ;
	free(name) ;
}

/*---------------------------------------------------------------------------*/
/* 				 last_inter				     */
/*	o  creates a symbol of type S_INT (interval list) and sets the	     */
/*	   value to NULL to indicate end of list			     */
/*	o  RESULT:  the empty list symbol				     */
/*---------------------------------------------------------------------------*/
sym_t
  * last_inter ( )
{
	register sym_t	*tmp ;

	tmp = (sym_t *)malloc(sizeof(sym_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in last interval") ;
	    exit(-1) ;
	}
	tmp->type = S_INT ;
	tmp->u.intlist = (inter_t *)0 ;
	return(tmp) ;
}

/*---------------------------------------------------------------------------*/
/* 				  add_inter				     */
/*	o  takes the interval curr and adds it to the list rest		     */
/*	o  the trigger, trig, is used to set the frame start value	     */
/*	   for curr							     */
/*	o  once the trigger is found, it can be used to call init and set    */
/*	   the params							     */
/*	o  NOTE:  trig must exist to calculate frame start value!	     */
/*	o  NOTE:  triggers must ascend in order (or descend because parse    */
/*	   goes left to right)						     */
/*	o  RESULT: a symbol that contains a list of intervals		     */
/*	o  removes first two symbols, revamps last symbol for return	     */
/*---------------------------------------------------------------------------*/
sym_t
  * add_inter ( trig , curr , rest )
sym_t	*trig, *curr, *rest ;
{
	register trig_t	*iter ;
	register inter_t *tmp ;
	register frm_t	diff ;

	iter = newproc.trig_head ;
	while( iter!=0 && strcmp(iter->trigname,trig->u.idval)!=0 )
	    iter = iter->next ;
	if( iter==0 ) {
	    yyerror("not a trigger") ;
	    exit(1) ;
	}
	curr->u.intlist->start = iter->trigtime ;
	curr->u.intlist->next = rest->u.intlist ;
	rest->u.intlist = curr->u.intlist ;
	tmp = rest->u.intlist ;
	if( tmp->next==0 )
	    diff = (int)(newproc.desc.runtime*newproc.desc.speed)-tmp->start ;
	else
	    diff = tmp->next->start - tmp->start ;
	if( diff<=0 ) {
	    yyerror("triggers are not in ascending order") ;
	    exit(1) ;
	}
	tmp->u.params = tmp->init(diff,tmp->u.ptlist) ;
	free(trig->u.idval) ;
	free(trig) ;
	free(curr) ;
	return(rest) ;
}

/*---------------------------------------------------------------------------*/
/* 				create_inter				     */
/*	o  matches the ord argument to a pair of functions that handle	     */
/*	   that polynomial order					     */
/*	o  allocates an inter_t and sets the run-time function from above    */
/*	o  NOTE:  the rest of inter_t (next, start, params) are instantiated */
/*	   in add_var and add_inter					     */
/*	o  RESULT:  a symbol that points to the interval created	     */
/*	o  removes one symbols revamps other				     */
/*---------------------------------------------------------------------------*/
sym_t
  * create_inter ( forder , samps )
sym_t	*forder, *samps ;
{
	register inter_t	*tmp ;

	tmp = (inter_t *)malloc(sizeof(inter_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in create interval") ;
	    exit(-1) ;
	}
	if( forder->type == S_NUM ) {
	    switch( (int)forder->u.numval ) {
	      case 0:
		tmp->init = constant_i ;
		tmp->calc = constant_c ;
		break ;
	      case 1:
		tmp->init = linear_i ;
		tmp->calc = linear_c ;
		break ;
	      case 2:
		tmp->init = quadratic_i ;
		tmp->calc = quadratic_c ;
		break ;
	      default:
		if( forder->u.numval>0 )
		    yyerror("polynomial order is too big") ;
		else
		    yyerror("polynomial orders are always positive") ;
		exit(1) ;
	    }
	} else {
	    yyerror("only polynomials are supported now") ;
	    exit(1) ;
	    free(forder->u.idval) ;
	}
	free(forder) ;
	tmp->u.ptlist = samps->u.ptlist ;
	samps->u.intlist = tmp ;
	return(samps) ;
}

/*---------------------------------------------------------------------------*/
/* 			      create_inter_def				     */
/*	o  counts the number of sample points				     */
/*	o  generates a symbol for a polynomial function of that order	     */
/*	   and calls create_inter					     */
/*	o  RESULT:  the returned symbol from create_inter		     */
/*---------------------------------------------------------------------------*/
sym_t
  * create_inter_def ( samps )
sym_t	*samps ;
{
	register int	count ;
	register point_t *iter ;
	register sym_t	*tmp ;

	iter = samps->u.ptlist ;
	count = 0 ;
	while( iter!=0 ) {
	    count++ ;
	    iter = iter->next ;
	}
	tmp = (sym_t *)malloc(sizeof(sym_t)) ;
	if( tmp==0 ) {
	    yyerror("malloc failed in create def interval") ;
	    exit(-1) ;
	}
	tmp->type = S_NUM ;
	tmp->u.numval = count -1 ;   /* order is one less than num of points */
	return(create_inter(tmp,samps)) ;
}

/*---------------------------------------------------------------------------*/
/* 				 last_point				     */
/*	o  creates a one element point list that holds the value	     */
/*	   passed							     */
/*	o  RESULT: the point list in a symbol				     */
/*	o  revamps the symbol						     */
/*---------------------------------------------------------------------------*/
sym_t
  * last_point ( sample )
sym_t	*sample ;
{
	register point_t	*end ;

	end = (point_t *) malloc(sizeof(point_t)) ;
	if( end==0 ) {
	    yyerror("malloc failed in adding point") ;
	    exit(-1) ;
	}
	end->samp = sample->u.numval ;
	end->next = (point_t *)0 ;
	sample->type = S_PT ;
	sample->u.ptlist = end ;
	return(sample) ;
}

/*---------------------------------------------------------------------------*/
/* 				  add_point				     */
/*	o  adds the point_t currpt to the list pts			     */
/*	o  RESULT:  a new list						     */
/*	o  removes first symbol, revamps other				     */
/*---------------------------------------------------------------------------*/
sym_t
  * add_point ( currpt , pts )
sym_t	*currpt, *pts ;
{
	register point_t	*new ;

	new = (point_t *) malloc(sizeof(point_t)) ;
	if( new==0 ) {
	    yyerror("malloc failed in adding point") ;
	    exit(-1) ;
	}
	new->samp = currpt->u.numval ;
	new->next = pts->u.ptlist ;
	pts->u.ptlist = new ;
	free(currpt) ;
	return(pts) ;
}
