
/*  @(#)special.c 1.6 93/06/01
 *
 *  Special transformations used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

/*  Special transformations from chapter 6 of BP.
 *
 *  The way this is done is fairly nasty at the moment, but it does work.
 */

#include <sys/types.h>
#include <ctype.h>
#include <time.h>

#ifdef      BSD
#include <sys/timeb.h>
#endif   /* BSD */

#include "popi.h"
#include "expr.h"

#define  GAMMA     7.5    /* Default gamma for matte() */
#define  TILESIZE  25     /* Default tile size for tile() */
#define  N         3      /* Default blur size for oil() */

#define  New      (CurNew->pix)

static void CloseLog P((FILE **)) ;


void
oil(t)
Tree *t ;
{
  register int color, dyoff, x, y, yoff ;
  register int dx, dy ;
  pixel_t mfp ;
  static int *histo = 0 ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

  if (histo == (int *) 0 &&
      (histo = (int *)
       LINT_CAST(Emalloc((unsigned) Zsize * sizeof (int)))) == (int *) 0)
    return ;

  for (y = N; y < CurNew->height-N; y++)
    {
      yoff = y * Xsize ;
      for (x = N; x < CurNew->width-N; x++)
        {
	  for (color = 0; color < srcp->ncolors; ++color)
	    {
	      for (dx = 0; dx < Zsize; dx++) histo[dx] = 0 ;
	      for (dy = y-N; dy <= y+N; dy++)
		{
                  dyoff = dy * srcp->Xalloc ;
                  for (dx = x-N; dx <= x+N; dx++)
		    histo[img[color][dyoff+dx]]++ ;
                }
     
	      for (dx = dy = 0; dx < Zsize; dx++)
		if (histo[dx] > dy)
		  {
		    dy = histo[dx] ;
		    mfp = (pixel_t) dx ;
		  } 
	      New[color][yoff+x] = mfp ;
	    }
        }
      disp_percentdone((y+1) * 100 / CurNew->height) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
shear(t)
Tree *t ;
{
  register int color, dyoff, r, x, y, yoff ;
  int dx, dy ;
  static int *yshift = 0 ;
  static int YShift = 0 ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;    
  img = srcp->pix ;

  if (YShift != Xsize)
    {
      if (yshift != NULL) FREE((char *) yshift) ;
      if ((yshift = (int *)
        LINT_CAST(Emalloc((unsigned) Xsize * sizeof(int)))) == 0)
          return ;
      YShift = Xsize ;
    }

  for (x = r = 0; x < CurNew->width; x++)
    {
      if (RANDOM % Zsize < 128) r-- ;
      else                      r++ ;
      yshift[x] = r ;
    }

  for (y = 0; y < CurNew->height; y++)
    {
      if (RANDOM % Zsize < 128) r-- ;
      else                      r++ ;

      yoff = y * Xsize ;
      for (x = 0; x < CurNew->width; x++)
	{
	  dx = x + r ;
	  dy = y + yshift[x] ;
	  if (dx >= CurNew->width || dy >= CurNew->height || dx < 0 || dy < 0)
	    continue ;
	  dyoff = dy * srcp->Xalloc ;
          for (color = 0; color < srcp->ncolors; color++)
	    New[color][yoff+x] = img[color][dyoff+dx] ;
	}
      disp_percentdone((y+1) * 100 / CurNew->height) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
slice(t)
Tree *t ;
{
  register int color, dyoff, r, x, y, yoff ;
  int dx, dy ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;
  static int *xshift = 0 ;
  static int XShift = 0 ;
  static int *yshift = 0 ;
  static int YShift = 0 ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

  if (XShift != Ysize)
    {
      if (xshift != NULL) FREE((char *) xshift);
      if ((xshift = (int *)
        LINT_CAST(Emalloc((unsigned) Ysize * sizeof(int)))) == 0)
          return ;
      XShift = Ysize ;
    }
  if (YShift != Xsize)
    {
      if (yshift != NULL) FREE((char *) yshift);
      if ((yshift = (int *)
        LINT_CAST(Emalloc((unsigned) Xsize * sizeof(int)))) == 0)
          return ;
      YShift = Xsize ;
    }

  for (x = dx = 0 ; x < CurNew->width; x++)
    {
      if (dx == 0)
        {
          r = (RANDOM & 63) - 32 ;
          dx = 8 + RANDOM & 31 ;
        }
      else dx-- ;
      yshift[x] = r ;
    }

  for (y = dy = 0; y < CurNew->height; y++)
    {
      if (dy == 0)
        {
          r = (RANDOM & 63) - 32 ;
          dy = 8 + RANDOM & 31 ;
        }
      else dy-- ;
      xshift[y] = r ;
    }

  for (y = 0; y < CurNew->height; y++)
    {
      yoff = y * Xsize ;
      for (x = 0; x < CurNew->width; x++)
	{
	  dx = x + xshift[y] ;
	  dy = y + yshift[x] ;
	  dyoff = dy * srcp->Xalloc ;
          if (dx < CurNew->width && dy < CurNew->height && dx >= 0 && dy >= 0)
	    for (color = 0; color < srcp->ncolors; color++)
	      New[color][yoff+x] = img[color][dyoff+dx] ;
	}
      disp_percentdone((y+1) * 100 / CurNew->height) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
tile(t)
Tree *t ;
{
  register int color, dx, dy, dyoff, nyoff, x, y ;
  int ox, oy, nx, ny ;
  int TileSize = TILESIZE ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

  for (y = 0; y < CurNew->height-TileSize; y += TileSize)
    {
      for (x = 0; x < CurNew->width-TileSize; x += TileSize)
	{
	  ox = (RANDOM & 31) - 16 ;       /* Displacement. */
	  oy = (RANDOM & 31) - 16 ;

	  for (dy = y; dy < y+TileSize; dy++)
	    {
              dyoff = dy * srcp->Xalloc ;
              for (dx = x; dx < x+TileSize; dx++)
	        {
		  nx = dx + ox ;
		  ny = dy + oy ;
		  nyoff = ny * Xsize ;
                  if (nx >= srcp->width  || nx < 0 ||
		      ny >= srcp->height || ny < 0)
		    continue ;
		  for (color = 0; color < srcp->ncolors; color++)
		    New[color][nyoff+nx] = img[color][dyoff+dx] ;
	        }
            }
	}
      disp_percentdone((y+1) * 100 / CurNew->height) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


/* Note: affects source image in situ. Buffers not swapped after processing. */

void
melt(t)
Tree *t ;
{
  register int color, k, NumPixels, x, y, yoff, yoff1 ;
  pixel_t val ;
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

/* Copy New from img, then manipulate New */

  for (y = 0; y < CurNew->height; y++)
    {
      yoff = y * Xsize ;
      yoff1 = y * srcp->Xalloc ;
      for (x = 0; x < CurNew->width; x++)
        for (color = 0; color < srcp->ncolors; color++)
	  New[color][yoff+x] = img[color][yoff1+x] ;
    }

  NumPixels = CurNew->width * CurNew->height ;
  for (k = 0; k < NumPixels; k++)
    {
      x     = RANDOM % CurNew->width ;
      y     = RANDOM % (CurNew->height - 1) ;
      yoff  =  y    * Xsize ;
      yoff1 = (y+1) * Xsize ;

      for (color = 0; color < srcp->ncolors; color++)
        while (y < CurNew->height-1 &&
               New[color][yoff+x] <= New[color][yoff1+x])
          {
            val                 = New[color][yoff +x] ;
            New[color][yoff +x] = New[color][yoff1+x] ;
            New[color][yoff1+x] = val ;
            y++ ;
            yoff                =  y    * CurNew->width ;
            yoff1               = (y+1) * CurNew->width ;
          }
      disp_percentdone((k+1) * 100 / NumPixels) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


void
matte(t)
Tree *t ;
{
  struct SRC *srcp = (struct SRC *) 0 ;
  pixel_t **img ;
  double gamma ;
  register color, x, y, yoff, yoff1 ;
  static pixel_t *lookup = (pixel_t *) 0 ;
  static double lastgamma = 0.0 ;

  DEBUG((Debug, "matte()\n")) ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  img = srcp->pix ;

  if (t->kids[1])
    {
      if (((Tree *) (t->kids[1]))->t == T_Fnum)
        gamma = ((Tree *) (t->kids[1]))->d ;
      else gamma = (double) ((Tree *) (t->kids[1]))->i ;
    }
  else gamma = GAMMA ;

  if (lookup == 0 &&
      (lookup = (pixel_t *) Emalloc((unsigned) Zsize)) == (pixel_t *) 0)
    return ;

  if (lastgamma != gamma)
    {
      for (x = 0; x < Zsize; ++x)
        lookup[x] = ((double) Zmax * pow(x / (double) Zmax, gamma) < 3.0)
                    ? Zmax : 0 ;
      lastgamma = gamma ;
    }

  for (y = 0; y < CurNew->height; y++)
    {
      yoff = y * Xsize ;
      yoff1 = y * srcp->Xalloc ;
      for (color = 0; color < srcp->ncolors; color++)
        for (x = 0; x < CurNew->width; x++)
          New[color][yoff+x] = lookup[img[color][yoff1+x]] ;
      disp_percentdone((y+1) * 100 / CurNew->height) ;
    }

  disp_percentdone(100) ;
  CurNew->ncolors = srcp->ncolors ;
  if (disp_active) drawimg(CurNew) ;
  SwapOldNew() ;
}


/* Although this is set up to use one particular graphics mode
 * of a 24-pin printer, everything should be generic enough
 * that it can be changed to work with a different graphics
 * mode on an 8-pin printer, just by changing a few numbers.
 */

void
genepson(t)
Tree *t ;
{
  struct SRC *srcp ;
  char *PinVals ;
  pixel_t *PinThresh, ThreshStep ;
  FILE *ostr ;
  register int x ;
  register pixel_t **img ;
  int y, yn ;
  int pin ;
  int BytesPerChunk, PinsPerPixel, BytesPerColumn ;
  int yPixelsPerByte, xPinsPerPixel, yPinsPerPixel ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;
  else            srcp = CurOld ;
  if ((ostr = EfopenW(lastfname)) == NULL) return ;

  img = srcp->pix ;

/* Printer specific */

  xPinsPerPixel  = 2 ;
  yPinsPerPixel  = 2 ;
  BytesPerColumn = 24 / BITSINBYTE ;

  PinsPerPixel   = xPinsPerPixel * yPinsPerPixel ;
  BytesPerChunk  = xPinsPerPixel * BytesPerColumn ;
  yPixelsPerByte = BITSINBYTE / yPinsPerPixel ;

/* Must be whole number of pixels (y direction) per  byte. */

  assert(yPinsPerPixel * yPixelsPerByte == BITSINBYTE) ;

/* Reallocate these each time, as changing the print mode
 * may change the sizes of these arrays.
 */

  if ((PinVals = Emalloc((unsigned)BytesPerChunk)) == 0 ||
      (PinThresh = (pixel_t *)
      Emalloc((unsigned) PinsPerPixel * sizeof(pixel_t))) == 0)
    return ;

  ThreshStep = (pixel_t) (Zsize / (PinsPerPixel + 1)) ;
  for (pin = 0; pin <  PinsPerPixel; ++pin)
    PinThresh[pin] = (pixel_t) ((pin + 1) * ThreshStep) ;

  for (y = 0;  y < srcp->height; y = yn)
    {

/* Printer specific */

/* This print line is width srcp->width pixels, and
 * (srcp->width * xPinsPerPixel) * pin positions (dots on the page).
 * No. of dots vertical  is (BytesPerColumn * BITSINBYTE)
 * which is yPinsPerPixel times  the no.  of image scanlines.
 */

      FPRINTF(ostr, "\033*%c%c%c",
              39,                             /* 180 dpi in both directions */
              (srcp->width * xPinsPerPixel) % 256,
              (srcp->width * xPinsPerPixel) / 256) ;

      for (x = 0; x <  srcp->width; ++x)
        {
          register int ycur, yoff ;
          int ByteCount, xpin ;
          char *dp ;

/* Clear the PinVals array */

          for (ByteCount = 0, dp = PinVals; ByteCount < BytesPerChunk;
               ++ByteCount)
            *dp++ = 0 ;

          dp = PinVals ;

/* For each byte-sized row of the print head, collect 1 pixel width of data. */

          for (ByteCount = 0, dp = PinVals, ycur = y;
               ByteCount < BytesPerColumn; ++ByteCount, dp  += xPinsPerPixel)
            {
              register unsigned char  bit ;

              yn = ycur + yPixelsPerByte ;
              if (yn > srcp->height) yn = srcp->height ;

/*  For the current byte row of the print-head (ie. yPixelsPerByte image
 *  scanlines), collect a pixel width of data.
 */

              for (bit = 0x80; ycur <  yn; ++ycur)
                {
                  pixel_t val ;
                  int ypin ;

                  yoff = ycur * srcp->Xalloc ;
                  if (colors == 1) val = img[0][yoff+x] ;  /* Grey level */
                  else                             /* Calculate NTSC luma */
                    val = img[0][yoff+x] * 0.299 +         /* R */
                          img[1][yoff+x] * 0.587 +         /* G */
                          img[2][yoff+x] * 0.114 ;         /* B */

/* Now use an appropriate no. of pins to simulate the greyscale value. */

                  for (ypin = 0, pin =  0; ypin < yPinsPerPixel; ++ypin)
                    {
                      for (xpin = 0; xpin < xPinsPerPixel; ++xpin, ++pin)
                        if (val < PinThresh[pin]) dp[xpin] |= bit ;
                      bit >>=  1 ;
                    }
                }
            }

/*  We have set up enough columns for a single pixel in the x direction.
 *  Now print them in the correct order.
 */

          for (xpin = 0; xpin < xPinsPerPixel; ++xpin)
            {
              for (ByteCount = 0; ByteCount < BytesPerColumn; ++ByteCount)
                PUTC(PinVals[ByteCount * xPinsPerPixel + xpin], ostr) ;
            }
        }

/* Printer specific */

      FPRINTF(ostr, "\033J%c\r", 24);
    }

  Efclose(ostr) ;
  free(PinVals) ;
  free((char *) PinThresh) ;
}


void
readimg(t)      /* :read "filename" [imagename] */
Tree *t ;
{
  char *imgname, *p ;

  if (t->kids[1]) imgname = lastimage ;
  else
    {
      for (p = imgname = lastfname; *p; ++p)
        if (*p == '/' && p[1] != '\0') imgname = p + 1 ;
    }
  getpix(lastfname, imgname, TRUE) ;
}


void
stopict(t, whichcol, fromimg)	/* :store [imagename] */
Tree *t ;
int  whichcol ;
char *fromimg ;
{
  char		*imgname ;
  char		noname[80] ;
  char          rgbchars[4] ;
  struct SRC	*img ;
  struct SRC	*unused = (struct SRC *) 0 ;
  struct SRC	*srcp = (struct SRC *) 0 ;
  int		i, j, ui, noofcols ;

  STRCPY(rgbchars, "rgb") ;

/* Now some hacks: if whichcol is -1 then its a normal :store, if whichcol
 * is 0, 1 or 2 it's a store of one component done by :explode. */

  if (whichcol >= 0)
    {
      srcp = &Images[getImageNo(fromimg)] ;
      SPRINTF(noname, "%s%c", fromimg, rgbchars[whichcol]) ;
      imgname = noname ;
      noofcols = 1 ;
    }
  else
    {
      if (t->kids[0]) imgname = lastimage ;
      else
        {
          SPRINTF(noname, "store_%d", nimages - 1) ;
          imgname = noname ;
	}
      srcp = CurOld ;
      noofcols = CurOld->ncolors ;
    }

/* See if the named image exist, search for an unused image. */

  for (img = Images, i = ui = 0; img != &Images[nimages]; ++img, i++)
    {
      if (img->str && strcmp(img->str, imgname) == 0) break ;
      if (img->pix == (pixel_t **) NULL && unused == (struct SRC *) NULL)
        {
	  unused = img ;
	  ui = i - 1 ;
	}
    }

  if (img == &Images[nimages])
    {		                   /* Not found, copy to a new image. */
      if (ui)	                   /* Use an unused image. */
	{
	  img = unused ;
	  if (whichcol < 0) SPRINTF(imgname, "store_%d", ui) ;
	}
      else img = &Images[nimages++] ;	/* This is a really new one. */

      if ((img->pix = ImgAlloc(srcp->width,
                               srcp->height, noofcols)) == 0 ||
          (img->str = (char *)
                      Emalloc((unsigned int) (strlen(imgname)+1))) == 0)
        return ;
      STRCPY(img->str, imgname) ;
    }
  else if (img->width   != srcp->width || img->height != srcp->height ||
           img->ncolors != noofcols)
    {                          /* Existing image has changed size or depth. */
      ImgFree(img) ;
      if ((img->pix = ImgAlloc(srcp->width,
                               srcp->height, noofcols)) == 0 ||
          (img->str = (char *)
                      Emalloc((unsigned int) (strlen(imgname)+1))) == 0)
        return ;

      STRCPY(img->str, imgname) ;
    }

  img->width   = srcp->width ;	/* copy it */
  img->height  = srcp->height ;
  img->ncolors = noofcols ;
  img->Xalloc  = srcp->width ;
  for (i = 0; i < img->ncolors; ++i)
    {
    for (j = 0; j < img->height; ++j)
      COPYMEM((char *) &srcp->pix[(whichcol < 0 ? i : whichcol)][j*srcp->Xalloc],
              (char *) &img->pix[i][j*img->width],
              img->width) ;
    }
}


void
writeimg(t)     /* :write "filename" [imagename] */
Tree *t ;
{
  struct SRC *img ;

  if (t->kids[1]) img = &Images[((Tree *) (t->kids[1]))->i] ;
  else            img = CurOld ;
  putpix(img, lastfname) ;
}


void
freeimg(t)
Tree *t ;
{
  struct SRC *srcp = (struct SRC *) 0 ;

  if (t->kids[0]) srcp = &Images[((Tree *) (t->kids[0]))->i] ;

  if (srcp == (struct SRC *) 0) return ;
    
  if (srcp == CurOld || srcp == CurNew)
    {
      SPRINTF(ErrBuf, "Cannot free 'old' or 'new'") ;
      error(0) ;
      return ;
    }

  ImgFree(srcp) ;
}


void
displayimg(t)    /* :display [+|-image] */
Tree *t ;
{
  struct SRC *image ;
  int imgno ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : disp_active = 1 ;
                         return ;
          case T_Minus : disp_active = 0 ;
                         return ;
          case T_Inum  : imgno = ((Tree *) (t->kids[0]))->i ;
	                 if ((imgno >= nimages) || (imgno < 0))
	  		   {
			     STRCPY(ErrBuf, "no such image") ;
			     run_error(ERR_SNARK) ;
			   }
	  		 image = &Images[imgno] ;
			 break ;
	  default:       FPRINTF(stderr,"*** invalid tree in displayimg\n");
        }
    }
  else image = CurOld ;
  drawimg(image) ;
}


static void
CloseLog(filep)
FILE **filep ;
{
  if (*filep == NULL) return ;

  FPRINTF(*filep, "\n---\n") ;
  FCLOSE(*filep) ;
  *filep = NULL ;
}


void
OpenLog(filep)
FILE **filep ;
{
  time_t time() ;
  time_t clock ;

  CloseLog(filep) ;

  if ((*filep = fopen(LogFile, "a")) == NULL)
    {
      SPRINTF(ErrBuf,
              "Can't open log file '%s' - logging is off", LogFile) ;
      error(ERR_SYS) ;
      return ;
    }

  clock = time((time_t *) 0) ;
  FPRINTF(*filep, "==>> %s", ctime(&clock)) ;    /* Includes '\n' */
}


void
getrgb(t)
Tree *t ;
{
  getputrgb(t, TRUE) ;
}


void
putrgb(t)
Tree *t ;
{
  getputrgb(t, FALSE) ;
}


void
dolog(t)
Tree *t ;
{
  static char *buf = (char *) 0 ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : OpenLog(&LogStr) ;
                         break ;
          case T_Minus : CloseLog(&LogStr) ;
                         break ;
          case T_File  : if (buf == (char *) 0 &&
                             (buf = Emalloc((unsigned int) 512)) == (char *) 0)
                           return ;
                         STRCPY(buf, lastfname) ;
                         LogFile = buf ;
                         OpenLog(&LogStr) ;
			 break ;
	  default:       FPRINTF(stderr, "*** invalid tree in dolog\n") ;
        }
    }

  if (LogStr) PRINTF("Logging is active on file '%s'\n", LogFile) ;
  else        PRINTF("Logging is off\n") ;
}


void
dostore(t)
Tree *t ;
{
  stopict(t, -1, "") ;
}


void
debug(t)
Tree *t ;
{
  static char *buf = (char *) 0 ;

  if (t->kids[0])
    {
      switch (((Tree *) (t->kids[0]))->t)
        {
          case T_Plus  : OpenLog(&Debug) ;
                         break ;
          case T_Minus : CloseLog(&Debug) ;
                         break ;
          case T_File  : if (buf == (char *) 0 &&
                             (buf = Emalloc((unsigned int) 512)) == (char *) 0)
                           return ;
                         STRCPY(buf, lastfname) ;
                         LogFile = buf ;
                         OpenLog(&Debug) ;
			 break ;
          default:       FPRINTF(stderr, "*** invalid tree in debug\n") ;
        }
    }

  if (Debug) PRINTF("Logging is active on file '%s'\n", LogFile) ;
  else       PRINTF("Logging is off\n") ;
}


static char *HelpMsg[] = {
  "binary ops:   ** * /  % + - << >> < > >= <= == != & ^ | && ||",
  "funcs:        sin(deg) cos(deg) atan(y, x) hypot(x, y)",
  "              log(val) sqrt(val) abs(val) rand()",
  "values:       x y r a X Y R A Z",
  "special functions:",
  "\t?",
  "\tq",
  "",
  "\t:color\t\t\t\t:debug      [+|-|\"filename\"]",
  "\t:display    [+|-|image]\t\t:exit",
  "\t:free       image\t\t:genepson   \"filename\" [image]",
  "\t:genps      \"filename\" [image]\t:grayscale",
  "\t:help\t\t\t\t:list",
  "\t:logfile    [+|-|\"filename\"]\t:matte      [image [gamma]]",
  "\t:melt       [image]\t\t:monochrome",
  "\t:ofmt       [+|-]\t\t:oil        [image]",
  "\t:quit\t\t\t\t:range      wrap|cut|minmax",
  "\t:read       \"filename\" [image]\t:shear      [image]",
  "\t:signed     [+|-]\t\t:size       [x [y]]",
  "\t:slice      [image]\t\t:store      [image]",
  "\t:tile       [image]\t\t:truncate   [+|-]",
  "\t:undo\t\t\t\t:verbose    [+|-]",
  "\t:version\t\t\t:write      \"filename\" [image]",
  "\t:combine    [image]\t\t:explode    [image]",
  "\t:get        r|g|b image\t\t:put        r|g|b image",
  "\t:luma       [image]",
  (char *) 0
} ;


/*ARGSUSED*/
void
help(t)
Tree *t ;
{
  PrStrs(HelpMsg) ;
}


/*ARGSUSED*/
void
setcolor(t)
Tree *t ;
{
  if (scr_depth > 1) dtype = IS_COLOR ;
}


/*ARGSUSED*/
void
setgray(t)
Tree *t ;
{
  if (scr_depth > 1) dtype = IS_GRAY ;
}


/*ARGSUSED*/
void
setmono(t)
Tree *t ;
{
  dtype = IS_MONO ;
}


/*ARGSUSED*/
void
undo(t)
Tree *t ;
{
  FPRINTF(stderr, "undo.\n") ;
  SwapOldNew() ;
}


void
verbose(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  Verbose = 1 ;
      else if (op == T_Minus) Verbose = 0 ;
    }
  PRINTF("Verbose is %s\n", Verbose ? "on" : "off") ;
}


void
imtrunc(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  Truncate = 1 ;
      else if (op == T_Minus) Truncate = 0 ; 
    }
  PRINTF("Truncation is %s\n", Truncate ? "on" : "off") ;
}


void
dosigned(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  signed_io = 1 ;
      else if (op == T_Minus) signed_io = 0 ; 
    }
  PRINTF("Signed I/O is %s\n", signed_io ? "on" : "off") ;
}


void
ofmt(t)
Tree *t ;
{
  if (t->kids[0])
    {
      enum tree op = ((Tree *) (t->kids[0]))->t ;
           if (op == T_Plus)  oldfmt = 1 ;
      else if (op == T_Minus) oldfmt = 0 ; 
    } 
  PRINTF("Old output format is %s\n", oldfmt ? "on" : "off") ;
}


void dosize(t)     /* just set the size of "old" and "new" */
Tree *t ;
{
  int newx, newy ;
  int rmpolar ;

  newx = Xsize ;
  newy = Ysize ;
  rmpolar = 0 ;
  if (t->kids[0])
    {
      if (((Tree *) (t->kids[0]))->t == T_Fnum)
        newx = ((Tree *) (t->kids[0]))->d ;
      else
        newx = ((Tree *) (t->kids[0]))->i ;
    }
  if (t->kids[1])
    {
      if (((Tree *) (t->kids[1]))->t == T_Fnum)
        newy = ((Tree *) (t->kids[1]))->d ;
      else
        newy = ((Tree *) (t->kids[1]))->i ;
    }
  if ((newx > 0) && (newx <= Xsize))
    {
      if (CurOld->width != newx) ++rmpolar ;
      CurOld->width = newx ;
      CurNew->width = newx ;
    }
  if ((newy > 0) && (newy <= Ysize))
    {
      if (CurOld->height != newy) ++rmpolar ;
      CurOld->height = newy ;
      CurNew->height = newy ;
    }
  if ((rmpolar) && (avals))     /* current polar-table is invalid */
    {
      FREE((char *) avals) ;
      FREE((char *) rvals) ;
      avals = NULL ;
      rvals = NULL ;
    }
}


void dorange(t)   /* set the policy for handling range errors */
Tree *t ;
{
  enum tree op = ((Tree *) (t->kids[0]))->t ;
  if (op == T_Rcut)    rng_policy = CUT ;     /* outside value = 0 */
  if (op == T_Rminmax) rng_policy = MINMAX ;  /* outside value = edge value */
  if (op == T_Rwrap)   rng_policy = WRAP ;    /* outside value = modulo size */
}
