/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/vcrDub/RCS/vcrDub.c,v 0.24 92/09/08 14:33:38 drapeau Exp $ */
/* $Log:	vcrDub.c,v $
 * Revision 0.24  92/09/08  14:33:38  drapeau
 * Fixed errors in the way edit lists are opened and saved, especially when
 * dealing with the "label" field of an edit.  Proper care is now taken to
 * allocating the correct amount of space for the label, to avoid string
 * length errors.
 * 
 * Revision 0.23  92/09/04  17:10:51  drapeau
 * Modified ConvertToTime() and ConvertToAddress() to eliminate calculations
 * based on NTSC-specific frame rates.  The calculations now use the defined
 * constant "FrameRate", which can be set at compile time.
 * Also, minor reformatting of the code to comply with coding standards.
 * Also, removed offset of 25 frames in DubThisEdit(); this offset should
 * really be done in the specific device driver, as frame drift will vary from
 * device to device.
 * Also, commented DubThisEdit() to better explain the dubbing process between
 * source and edit decks, and replaced NTSC-specific references with the
 * defined constant "FrameRate".
 * In main(), used xv_set_rect() to set location and placement of the dub
 * setup window, instead of using the less reliable "XV_X" and "XV_Y"
 * attributes.
 * 
 * Revision 0.22  92/05/14  17:36:47  drapeau
 * Made several changes:
 * * Removed code that checks for tape identifiers in the header.  This
 *   feature turned out to be too cumbersome, time-consuming, and unreliable.
 * * Added code to support the new Info panel (showing, removing, etc.).
 * * Formatting changes to the code to conform to coding standards.
 * 
 * Revision 0.21  92/01/09  18:17:37  drapeau
 * Slight modifications to make code ANSI-compliant.
 * 
 * Revision 0.20  91/10/10  11:58:46  lim
 * Corrected tempLabel in OpenHandler(). 
 * Added 8 bytes more.
 * 
 * Revision 0.19  91/09/29  16:19:07  lim
 * Moved DisplayChoice to videoObjects.c
 * 
 * Revision 0.18  91/09/24  22:18:26  lim
 * Changed 'DubEdit' to 'vcrDub' everywhere.
 * 
 * Revision 0.17  91/09/02  14:40:35  lim
 * Fixed savehandler. Header is 20 char long!
 * 
 * Revision 0.16  91/08/28  12:44:22  lim
 * SSet canvas height to 1800 pixels (paint window).
 * 
 * Revision 0.15  91/08/27  10:49:25  lim
 * notSaved set when deleting edit too.
 * 
 * Revision 0.14  91/08/25  16:24:30  lim
 * 1. Implemented '-d' option in command line, to indicate whether
 * user wants to see diagnostic messages on stderr or not.
 * 2. File reading and writing has been modified to fit the new format
 * of VCR Edit documents.
 * 3. Generalized CopyMergeEdit by taking in second parameter denoting
 * the index to the srcEditMergeEdit array. Used to take the index to
 * be srcEditSel.
 * 4. Merge edits are not disposed when an edit in the canvas is deleted,
 * because otherwise the source edit list does not have a record of
 * what the merged edits contained once the merged edit on the canvas is
 * deleted.
 * 
 * Revision 0.13  91/08/20  14:55:02  lim
 * Corrected DubQuit()'s error message checking.
 * 
 * Revision 0.12  91/08/20  09:57:38  lim
 * Fixed EditAdd() for single edits.
 * 
 * Revision 0.11  91/08/19  17:10:32  lim
 * Fixed adding merged edit from edit list.
 * 
 * Revision 0.10  91/08/19  16:22:16  lim
 * Initial revision.
 * This version only supports NEC players.
 *  */
static char rcsid[] = "$Header: /Source/Media/collab/vcrDub/RCS/vcrDub.c,v 0.24 92/09/08 14:33:38 drapeau Exp $";

#include "vcrDub.h"

/* Video stuff */
VideoObject* playVideo;
VideoObject* recVideo;


vcrDub_window1_objects*		vcrDub_window1;			    /* Xview stuff */
vcrDub_dubSetupWindow_objects*	vcrDub_dubSetupWindow;
vcrDub_infoPopup_objects*	vcrDub_infoPopup;
enum Boolean			hitQuit;			    /* Flag to denote if 'Quit' button is hit */

/* Command line variables */
char* startFilename;				                    /* Name of file in command line */
Xv_font* font;							    /* Font for scroll list */
	
/* Edit list stuff */
int srcEditSel;							    /* Current edit list selection */
int origLines;							    /* Number of lines in source edit list */
int notSaved;							    /* Set when edit manipulations are not saved */
int repeat;							    /* To see if this is a repeat call - XView bug */
char srcEditLabel[MaxListSize][MaxLabelSize+1];			    /* Array of srcEdit labels */
int srcEditDur[MaxListSize];					    /* Array of srcEdit durations */
char srcEditStart[MaxListSize][12];				    /* Array of srcEdit start times  */
char srcEditEnd[MaxListSize][12];				    /* Array of srcEdit end times */
int srcEditSpeed[MaxListSize];					    /* Array of srcEdit speed settings */
int srcEditAudio[MaxListSize];					    /* Array of srcEdit audio settings */
MergeEdit* srcEditMergeEdit[MaxListSize];			    /* Array of subsequent edits in a merged edit */
enum Boolean srcEditAudioDub[MaxListSize];			    /* Array of srcEdit audio dub settings 
								       - initialized to Yes */

/* Canvas stuff */
CanvasData* editCanvas;						    /* Pointer to editCanvas data */
Pixmap audioImage;						    /* Image to draw when audio dubbing is on */
static Xv_singlecolor myColors[] = 
{
  {214, 214, 214},						    /* Grey84 */
  {235, 235, 235},						    /* Grey92 */
  {166, 166, 166},						    /* Grey65 */
  {255, 255, 255},						    /* White */
  {255, 0, 0},							    /* Red */
  {235, 235, 235},						    /* Unused1 */
  {211, 211, 211},						    /* Unused2 */
  {0, 0, 0},							    /* Black */
}
;

/* Video settings */
char tapeCode[MaxNameLength];					    /* Tape header found in edit list file */
int dubbing;							    /* Flag to indicate dubbing */

enum Boolean diagMode = No;					    /* Whether to print diagnostic messages */

/*
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  
 * In this case, it will be the filename to be loaded.  At present, only one filename will be handled.  So if
 * there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"diagnostics", 0, 0, 'd'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "d:",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
    {
      break;
    }
    switch (optionChar)						    /* Modified : No host or port options */
    {								    /* Dub is not used with PortManager/TimeLine */
     case 'd':
      diagMode = Yes;
      break;
     default:
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
  {
    startFilename = (char *) malloc (256);
    strcpy (startFilename, argv[optind]);
  }
}



/* Convert time to address in frames */
int
  ConvertToAddress(time)
char time[12];
{
  int address;
  int hr;
  int min;
  int sec;
  int frame;
  
  sscanf(time, "%d:%d:%d:%d", &hr, &min, &sec, &frame);
  address = FrameRate * (hr*3600 + min*60 + sec) + frame;
  return address;
}


/* Convert address in frames to time */
void
  ConvertToTime(hr, min, sec, frame, address)
int* hr;
int* min;
int* sec;
int* frame;
int  address;
{
  int	framesPerHour;
  int	framesPerMinute;
  int	framesPerSecond;
  
  framesPerHour = 3600 * FrameRate;				    /* 3600 = seconds per hour */
  framesPerMinute = 60 * FrameRate;
  framesPerSecond = FrameRate;
  
  *hr = address / framesPerHour;
  *min = (address % framesPerHour) / framesPerMinute;
  *sec = ((address % framesPerHour) % framesPerMinute) / framesPerSecond;
  *frame = ((address % framesPerHour) % framesPerMinute) % framesPerSecond;
  
}



/* Set buttons for edit operations active */
void
  EditOpActive(audioDubOn)
enum Boolean audioDubOn;
{
  xv_set(vcrDub_window1->dubDelButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_VALUE, (int) audioDubOn, NULL);
}  


/* Set buttons for edit operations inactive */
void
  EditOpInactive()
{
  xv_set(vcrDub_window1->dubDelButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_VALUE, 1, NULL);
}  




/* Displays edit list */
void
  DisplayEditList(oldlines)
int oldlines;
{
  int i;
  char listEntry[70];
  
  xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, TRUE, NULL);
  if (srcEditSel >= 0)
    xv_set(vcrDub_window1->dubScrollList, PANEL_LIST_SELECT, srcEditSel, FALSE, NULL);
  
  srcEditSel = -1;
  
  xv_set(vcrDub_window1->dubScrollList,			    /* To prevent blinking, set XV_SHOW false */
	 XV_SHOW, FALSE,
	 NULL);
  
  for (i=0; i<oldlines; i++)					    /* Replace the old string by the new one */
  {
    sprintf(listEntry, "%5d.  %-26.22s%3d", i+1, 
	    srcEditLabel[i], srcEditDur[i]);
    xv_set (vcrDub_window1->dubScrollList,
	    PANEL_LIST_STRING, i, listEntry,
	    PANEL_LIST_FONT, i, font,
	    NULL);
  }
  
  if (oldlines < origLines)					    /* Insert the additional new strings */
    for (i=oldlines; i<origLines; i++) 
    {
      sprintf(listEntry, "%5d.  %-26.22s%3d", i+1, 
	      srcEditLabel[i], srcEditDur[i]);
      xv_set (vcrDub_window1->dubScrollList,
	      PANEL_LIST_INSERT, i,
	      PANEL_LIST_STRING, i, listEntry,
	      PANEL_LIST_FONT, i, font,
	      NULL);
    }
  
  else for (i=oldlines-1; i> origLines-1; i--)			    /* delete all additional old entries in the edit list */
    xv_set(vcrDub_window1->dubScrollList,
	   PANEL_LIST_DELETE, i,
	   NULL);
  
  xv_set(vcrDub_window1->dubScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);  
}


/*
 * Menu handler for `DocMenu (Open)'.
 */
Menu_item
  DubFileOpen(item, op)
Menu_item	item;
Menu_generate	op;
{
  int result;
  
  if (op == MENU_NOTIFY)
  {
    if (notSaved)						    /* New edit manipulations made */
    {
      result = DisplayChoice("The edit list manipulations will be lost",
			     "Go ahead and load another file?", "No", "Yes");
      if (result == NOTICE_YES)
	return item;
    }
    
    Browse(NULL, BrowseOpen, NULL, "#VCR Edit Document#", "vcrDub");
  }
  
  return item;
}


EditLink*
  GetEdit(which)
int which;							    /* from 0 to n-1 */
{
  int i;
  EditLink* current;
  
  current = editCanvas->listHead;
  
  for (i = 0; ((i < which) && current); i++)
    current = current->nextEdit;
  
  if (i == which)
    return current;
  else
    return NULL;
}



/* Frees up the merged list */
void
  DisposeMergedList(head)
MergeEdit* head;
{
  MergeEdit* p;
  MergeEdit* q;
  
  for (p = head; p != NULL; p = q)
  {
    q = p->nextMergeEdit;
    free(p);
  }
}


/* Frees up the linked list */
void
  DisposeLinkedList(head)
EditLink* head;
{
  EditLink* p;
  EditLink* q;
  
  for (p = head; p != NULL; p = q)
  {
    q = p->nextEdit;
    if (p->mergeList)
      DisposeMergedList(p->mergeList);
    free(p);
  }
}


/* Sets up the linked list of edits in the canvas */
void
  SetUpLinkedList(head, tail)
EditLink** head;
EditLink** tail;
{
  int i;
  EditLink* tmp;
  EditLink* prev;
  
  if (*head)							    /* Dispose of old list */
    DisposeLinkedList(*head);
  
  (*head) = (EditLink*) malloc(sizeof(EditLink));		    /* Initialize head of list */
  (*head)->nextEdit = NULL;
  (*head)->prevEdit = NULL;
  (*head)->origNum = 0;
  (*head)->audioDub = Yes;
  (*head)->mergeList = NULL; 
  (*head)->sMode = Unselected; 
  prev = (*head);
  
  for (i=1; i < origLines; i++)
  {
    tmp = (EditLink*) malloc(sizeof(EditLink));
    tmp->nextEdit = NULL;
    tmp->prevEdit = prev;					    /* Points to previous edit */
    tmp->origNum = i;
    tmp->audioDub = Yes;
    tmp->mergeList = NULL; 
    prev->nextEdit = tmp;
    prev->sMode = Unselected; 
    prev = tmp;
  }  
  (*tail) = prev;
  editCanvas->numInLinkedList = origLines;
  return;
  
}


/* Draws outline of edit, depending on whether selected or not */
void
  DrawOutline(x, y, sel)
int x;
int y;
enum SelectMode sel;
{
  XPoint *points = (XPoint*) malloc (sizeof (XPoint) *3);	    /* Array of point coordinates to be passed to  XDrawLines() */
  
  if (sel == Unselected)					    /* Drawing top and left borders */
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[White]);
  else
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Black]);
  
  points[0].x = x;
  points[0].y = y+EditHeight;
  points[1].x = x;
  points[1].y = y;
  points[2].x = x+EditWidth;
  points[2].y = y;
  XDrawLines(editCanvas->display, editCanvas->drawable, editCanvas->gc, points, 3, CoordModeOrigin);
  
  
  if (sel == Unselected)					    /* Drawing bottom and right borders */
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Black]);
  else
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[White]);
  
  points[0].x = x+EditWidth;
  points[0].y = y;
  points[1].x = x+EditWidth;
  points[1].y = y+EditHeight;
  points[2].x = x;
  points[2].y = y+EditHeight;
  XDrawLines(editCanvas->display, editCanvas->drawable, editCanvas->gc, points, 3, CoordModeOrigin);
  
}



/* Draws each edit.
 * Fill rectangle accordingly.
 * Call function to draw outlines accordingly.
 * Draws audio icon if audio dub is on
 */
void
  DrawEdit(x, y, string, audio, sel)
int x;
int y;
char* string;
enum Boolean audio;
enum SelectMode sel;
{
  if (sel == Unselected)
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Grey92]);
  else 
    XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Grey65]);
  XFillRectangle(editCanvas->display, editCanvas->drawable, editCanvas->gc, x, y, EditWidth, EditHeight);
  
  XSetForeground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Black]);
  XSetBackground(editCanvas->display, editCanvas->gc, editCanvas->pixelTable[Grey92]);
  if (audio == Yes)
    XCopyPlane(editCanvas->display, audioImage, editCanvas->drawable, editCanvas->gc, 
	       0, 0, audioBitmap_width, audioBitmap_height, x+AudioXOffset, y+AudioYOffset, 1);
  
  XDrawString(editCanvas->display, editCanvas->drawable, editCanvas->gc, x+LabelXOffset, y+LabelYOffset,
	      string, strlen(string));
  DrawOutline(x, y, sel);
}



/* Draws the linked list onto the canvas */
void
  DrawLinkedList(head, renumber)
EditLink* head;
enum Boolean renumber;
{
  int i;
  int y;
  EditLink* current;
  char label[40];
  enum Boolean audioDubOn;
  
  XClearWindow(editCanvas->display, editCanvas->drawable);	    /* clear the whole display */
  current = head;
  y = 0;
  
  for (i = 0; current; i++)
  {
    y += (EditGap + EditHeight);
    audioDubOn = current->audioDub;
    if (renumber == Yes)
    {
      current->origNum = i;
      sprintf(label, "%3d. %-23.22s", i+1, srcEditLabel[current->origNum]);
    }
    else
      sprintf(label, "%3d. %-23.22s", current->origNum+1, srcEditLabel[current->origNum]);
    if (current->sMode == Selected)
      DrawEdit(LeftMargin, y, label, audioDubOn, Selected);
    else 
      DrawEdit(LeftMargin, y, label, audioDubOn, Unselected);
    
    current = current->nextEdit;
  }
  
}



/* Opens the edit file and loads into data structures.
 * Displays new canvas.
 * Returns -1 to OpenPanel if the file is of the wrong type.
 * Otherwise returns 0.
 */
int 
  OpenHandler(proposedPath, id)
char* proposedPath;
int id;
{
  int oldlines;
  int i;
  char tempLabel[MaxLabelSize+10];
  char statusMsg[MaxPathLength+20];
  char header[32];
  FILE* fp;
  
  fp = fopen(proposedPath, "r");				    /* Load file into array */
  fgets(header, 20, fp);
  if (strcmp(header, "#VCR Edit Document#") != 0) 
  {
    DisplayError("This is not a VCR edit file", " ");
    fclose(fp);
    return -1;							    /* Returns error to OpenPanel */
  }
  
  oldlines = origLines;
  
  fscanf(fp, "%*s%*s%s\n", tapeCode);
  
  sprintf(statusMsg, "Current document: %s", proposedPath);	    /* Set message on window */
  xv_set(vcrDub_window1->dubStatusMsg, PANEL_LABEL_STRING, 
	 statusMsg, NULL); 
  
  fscanf(fp, "%*s%*s%*s%d", &origLines);  
  if (!origLines)						    /* There are no edits, return */
  {
    DisplayError("There are no edits in this file", 
		 "Please choose another one");
    return -1;
  }
  
  for (i=0; i<origLines; i++) 
  {
    fscanf(fp, "%*s%*s%*d");
    fscanf(fp, "%*s%*s%11s\n", srcEditStart[i]);
    fscanf(fp, "%*s%*s%11s\n", srcEditEnd[i]);
    fscanf(fp, "%*s%*s%d\n", &srcEditAudio[i]);
    fscanf(fp, "%*s%*s%*s%d\n", &srcEditSpeed[i]);
    
    if (srcEditSpeed[i])
      srcEditDur[i] = (ConvertToAddress(srcEditEnd[i]) - ConvertToAddress(srcEditStart[i]))/srcEditSpeed[i];
    else
      srcEditDur[i] = 0;
    
    fgets(tempLabel, 32, fp);
    strncpy(srcEditLabel[i], &tempLabel[8], MaxLabelSize);
    if (strcmp(srcEditLabel[i], "\n") == 0)
      strcpy(srcEditLabel[i], "No label\n");
    srcEditAudioDub[i] = Yes;					    /* Defaults to Yes */
    srcEditMergeEdit[i] = NULL;					    
    
    if (strlen(srcEditLabel[i]) < MaxLabelSize)			    /* Remove newline character if necessary */
      srcEditLabel[i][strlen(srcEditLabel[i])-1] = '\0';
    if ((strlen(srcEditLabel[i]) == MaxLabelSize) 
	&& (srcEditLabel[i][MaxLabelSize-1] == '\n'))
      srcEditLabel[i][MaxLabelSize-1] = '\0';
  }
  fclose(fp);
  
  editCanvas->numMerged = 0;
  editCanvas->numInMergedLists = 0;
  editCanvas->firstSelected = -1;
  notSaved = 0;
  
  DisplayEditList(oldlines);
  EditOpInactive();
  SetUpLinkedList(&(editCanvas->listHead),			    /* Set up linked list of these edits */
		  &(editCanvas->listTail));			    
  DrawLinkedList(editCanvas->listHead, No);			    /* Draw linked list on canvas */
  return 0;
  
}								    /* end function OpenHandler */


/* Loads edit list from linked list.
 * Calls DisplayEditList().
 */
void
  LoadEditList(head)
EditLink* head;
{
  int i;
  int oldLines;
  EditLink* current;
  char tmpEditLabel[MaxListSize][MaxLabelSize+1];		    /* Array of tmpEdit labels */
  int tmpEditDur[MaxListSize];					    /* Array of tmpEdit durations */
  char tmpEditStart[MaxListSize][12];				    /* Array of tmpEdit start times  */
  char tmpEditEnd[MaxListSize][12];				    /* Array of tmpEdit end times */
  int tmpEditSpeed[MaxListSize];				    /* Array of tmpEdit speed settings */
  int tmpEditAudio[MaxListSize];				    /* Array of tmpEdit audio settings */
  MergeEdit* tmpEditMergeEdit[MaxListSize];			    /* Array of subseq edits for a merged edit */
  
  current = head;
  
  for (i = 0; i< origLines; i++)				    /* Place old values into temporary arrays */
  {
    strcpy(tmpEditLabel[i], srcEditLabel[i]);
    strcpy(tmpEditStart[i], srcEditStart[i]);
    strcpy(tmpEditEnd[i], srcEditEnd[i]);
    tmpEditDur[i] = srcEditDur[i];
    tmpEditSpeed[i] = srcEditSpeed[i];
    tmpEditAudio[i] = srcEditAudio[i];
    tmpEditMergeEdit[i] = srcEditMergeEdit[i];
  }
  
  for (i = 0; i<editCanvas->numInLinkedList; i++)		    /* Put linked list edits into source arrays */
  {
    strcpy(srcEditLabel[i], tmpEditLabel[current->origNum]);
    strcpy(srcEditStart[i], tmpEditStart[current->origNum]);
    strcpy(srcEditEnd[i], tmpEditEnd[current->origNum]);
    srcEditDur[i] = tmpEditDur[current->origNum];
    srcEditSpeed[i] = tmpEditSpeed[current->origNum];
    srcEditAudio[i] = tmpEditAudio[current->origNum];
    srcEditAudioDub[i] = current->audioDub;
    srcEditMergeEdit[i] = current->mergeList;			    
    current = current->nextEdit;
  }
  oldLines = origLines;						    
  origLines = editCanvas->numInLinkedList;			    /* Update origLines */
  
  DisplayEditList(oldLines);
  
}  



/*
 * Menu handler for `DocMenu (Save)'.
 */
Menu_item
  DubFileSave(item, op)
Menu_item	item;
Menu_generate	op;
{
  if (op == MENU_NOTIFY)
    Browse(NULL, BrowseSave, NULL, "#VCR Edit Document#", "vcrDub");
  
  return item;
}


/* Saves into a file from edit list. 
 * Called only after dubbing. 
 */
void
  SaveFromEditList(fp)
FILE* fp;
{
  int i;
  
  for (i = 0; i < origLines; i++)
  {
    fprintf(fp, "#Edit Number:\t%d\n", i+1);
    fprintf(fp, "#Start Address:\t%.11s\n", srcEditStart[i]);
    fprintf(fp, "#End Address:\t%.11s\n", srcEditEnd[i]);
    fprintf(fp, "#Audio Setting:\t%d\n", srcEditAudio[i]);
    fprintf(fp, "#Speed in frames/second:\t%d\n", srcEditSpeed[i]);
    fprintf(fp, "#Label:\t%s\n\n", srcEditLabel[i]);
  }
}    



/* Saves the file into proposed path.
 * Loads the new file into the edit list.
 * Returns error to OpenPanel if there is already a file of the same name that the user does
 * not want to overwrite.
 * Otherwise, returns 0.
 */
int
  SaveHandler(proposedPath, id)
char* proposedPath;
int id;
{
  FILE* fp;
  struct stat stbuf;
  EditLink* current;
  MergeEdit* currentMerge;
  char header[21];
  char buf[30];
  char diagMsg[30];
  int result;
  int editNum;
  int i;
  
  if (stat(proposedPath, &stbuf) == 0)				    /* check if file existed previously ...
								       ... and if it is a vcr edit file */
  {
    fp = fopen(proposedPath, "r");
    fgets(header, 20, fp);
    fclose(fp);
    
    if (strcmp(header, "#VCR Edit Document#") != 0) 
      sprintf(buf, "This file is not a vcr edit file.");
    else
      sprintf(buf, "This vcr edit file exists.");
    result = DisplayChoice(buf, "Do you wish to overwrite it?",
			   "No", "Yes");
    if (result == NOTICE_YES) 
      return 1;							    /* Returns 1 to openPanel */
  }
  
  fp = fopen(proposedPath, "w");
  fprintf(fp, "#VCR Edit Document#\n");
  fprintf(fp, "#Tape Code:\t%s\n", tapeCode);
  fprintf(fp, "#Number of Edits:\t%d\n\n", (editCanvas->numInLinkedList)+(editCanvas->numInMergedLists)-(editCanvas->numMerged));
  sprintf(diagMsg, "%d + %d - %d = %d\n", (editCanvas->numInLinkedList), (editCanvas->numInMergedLists), 
	  (editCanvas->numMerged), (editCanvas->numInLinkedList)+(editCanvas->numInMergedLists)-(editCanvas->numMerged));
  PrintDiagnostics(diagMsg);
  
  if (id)
    SaveFromEditList(fp);
  else 
  {
    current = editCanvas->listHead;
    i = 0;
    while (current)
    {
      editNum = current->origNum;
      currentMerge = current->mergeList;
      
      fprintf(fp, "#Edit Number:\t%d\n", i+1);
      
      if (!currentMerge)					    /* Not merged with other edits */
      {
	fprintf(fp, "#Start Address:\t%.11s\n", srcEditStart[editNum]);
	fprintf(fp, "#End Address:\t%.11s\n", srcEditEnd[editNum]);
	fprintf(fp, "#Audio Setting:\t%d\n", srcEditAudio[editNum]);
	fprintf(fp, "#Speed in frames/second:\t%d\n", srcEditSpeed[editNum]);
	fprintf(fp, "#Label:\t%s\n\n", srcEditLabel[editNum]);
	i++;
      }
      else							    /* Is merged with other edits */
      {
	fprintf(fp, "#Start Address:\t%.11s\n", currentMerge->startAddress);
	fprintf(fp, "#End Address:\t%.11s\n", currentMerge->endAddress);
	fprintf(fp, "#Audio Setting:\t0\n");
	fprintf(fp, "#Speed in frames/second:\t30\n");
	fprintf(fp, "#Label:\t%s\n\n", srcEditLabel[editNum]);
	currentMerge = currentMerge->nextMergeEdit;
	i++;
      }
      
      while (currentMerge)					   
      { 
	fprintf(fp, "#Edit Number:\t%d\n", i+1);
	fprintf(fp, "#Start Address:\t%.11s\n", currentMerge->startAddress);
	fprintf(fp, "#End Address:\t%.11s\n", currentMerge->endAddress);
	fprintf(fp, "#Audio Setting:\t0\n");
	fprintf(fp, "#Speed in frames/second:\t30\n");
	fprintf(fp, "#Label:\tTo merge with above\n\n");
	currentMerge = currentMerge->nextMergeEdit;
	i++;
      }
      current = current->nextEdit;
    }
    LoadEditList(editCanvas->listHead);				    /* Display new edit list */
    DrawLinkedList(editCanvas->listHead, Yes);
  }
  
  fclose (fp);
  
  notSaved = 0;							    /* File has been saved */
  
  return 0;
}



/*
 * Notify callback function for `dubStartButton'.
 * Dubbing requires the edit list be updated.
 */
void
  DubStart(item, event)
Panel_item	item;
Event		*event;
{
  if (!origLines)
  {
    DisplayError("There are no edits in the edit list",
		 "Dubbing aborted");
    return;
  }
  
  xv_set(vcrDub_dubSetupWindow->dubSetupWindow, XV_SHOW, TRUE, NULL);
  xv_set(vcrDub_dubSetupWindow->dubSetupWindow, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
}


/*
 * Notify callback function for `dubQuitButton'.
 */
void
  DubQuit(item, event)
Panel_item	item;
Event		*event;
{
  int result;
  
  if (playVideo)
    DevStop(playVideo);
  if (recVideo)
    DevStop(recVideo);
  
  if ((notSaved)&&(!hitQuit))					    /* check if unsaved changes exist in canvas */
  {
    result = DisplayChoice("New edit manipulations will be lost.", 
			   "Go ahead and quit?",
			   "No", "Yes");
    if (result == NOTICE_YES)
      return;
  }
  
  hitQuit = 1;
  
  DisposeLinkedList(editCanvas->listHead);
  xv_destroy_safe(vcrDub_window1->window1);
}


/* Returns number of items in merged list */
int
  NumInList(list)
MergeEdit* list;
{
  int result;
  
  result = 0;
  while (list)
  {
    result ++;
    list = list->nextMergeEdit;
  }
  return result;
}


void
  CopyMergeEdits(new, editnum)
MergeEdit** new;
int editnum;						    /* from 0 to n-1 */
{
  MergeEdit* current;
  MergeEdit* prev;
  
  if (srcEditMergeEdit[editnum] == NULL)
    return;
  
  (*new) = (MergeEdit*) malloc(sizeof(MergeEdit));
  strcpy((*new)->startAddress, srcEditMergeEdit[editnum]->startAddress);
  strcpy((*new)->endAddress, srcEditMergeEdit[editnum]->endAddress);
  (*new)->nextMergeEdit = NULL;
  current = srcEditMergeEdit[editnum]->nextMergeEdit;
  prev = (*new);
  
  while (current)
  {
    prev->nextMergeEdit = (MergeEdit*) malloc(sizeof(MergeEdit));
    strcpy(prev->nextMergeEdit->startAddress, current->startAddress);
    strcpy(prev->nextMergeEdit->endAddress, current->endAddress);
    prev->nextMergeEdit->nextMergeEdit = NULL;
    prev = prev->nextMergeEdit;
    current = current->nextMergeEdit;
  }
}


/*
 * Notify callback function for `dubAddButton'.
 * New edit is put at the tail of the linked list.
 * The list is displayed.
 */
void
  EditAdd(item, event)
Panel_item	item;
Event		*event;
{
  EditLink* newEdit;
  int mergedInThisEdit;
  char label[40];
  char diagMsg[30];
  
  if (srcEditSel < 0)
    return;
  
  newEdit = (EditLink*) malloc(sizeof(EditLink));
  if (editCanvas->listTail)
  {
    editCanvas->listTail->nextEdit = newEdit;
    newEdit->prevEdit = editCanvas->listTail;
  }
  else
  {
    editCanvas->listHead = newEdit;
    newEdit->prevEdit = NULL;
  }
  
  newEdit->nextEdit = NULL;
  newEdit->origNum = srcEditSel;
  newEdit->audioDub = Yes;					    /* Defaults to Yes */
  newEdit->mergeList = NULL;					    /* Initialized to NULL */
  CopyMergeEdits(&newEdit->mergeList, srcEditSel);
  editCanvas->listTail = newEdit;
  editCanvas->numInLinkedList ++;
  mergedInThisEdit = NumInList(newEdit->mergeList);
  if (mergedInThisEdit)
    editCanvas->numMerged ++;
  editCanvas->numInMergedLists += mergedInThisEdit;
  sprintf(diagMsg, "Num in merged list: %d\n", editCanvas->numInMergedLists);
  PrintDiagnostics(diagMsg);
  
  sprintf(label, "%3d. %-23.22s", srcEditSel+1, srcEditLabel[srcEditSel]);
  DrawEdit(LeftMargin, editCanvas->numInLinkedList*(EditGap+EditHeight), label, Yes, Unselected);
}


/*
 * Notify callback function for `dubScrollList'.
 */
int
  DubSelect(item, string, client_data, op, event)
Panel_item	item;
char		*string;
Xv_opaque	client_data;
Panel_list_op	op;
Event		*event;
{
  int selection;
  
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, TRUE, NULL);
    srcEditSel = -1;		  
    break;
    
   case PANEL_LIST_OP_SELECT:
    xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, FALSE, NULL);
    sscanf(string, "%d", &selection);
    srcEditSel = selection - 1;
    break;
  }
  return XV_OK;
}


/*
 * Notify callback function for `DubAudioStg'.
 */
void
  SetAudioDub(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  char string[30];
  EditLink* current;
  
  if (editCanvas->firstSelected)
    current = GetEdit(editCanvas->firstSelected-1);
  
  if (value)
    current->audioDub = Yes;
  else
    current->audioDub = No;
  
  sprintf(string, "%3d. %-23.22s", current->origNum+1, srcEditLabel[current->origNum]);
  DrawEdit(LeftMargin, (editCanvas->firstSelected)*(EditGap+EditHeight),
	   string, current->audioDub, Selected);
}



/* Deletes edit from linked list and decrements 
 * numInLinkedList
 */
void
  EditDelete(toDel)
EditLink* toDel;
{
  char diagMsg[30];
  
  if (toDel)
  {
    if (toDel->prevEdit)
      toDel->prevEdit->nextEdit = toDel->nextEdit;
    else
      editCanvas->listHead = toDel->nextEdit;
    
    if (toDel->nextEdit)
      toDel->nextEdit->prevEdit = toDel->prevEdit;
    else
      editCanvas->listTail = toDel->prevEdit;
    
    toDel->nextEdit = NULL;
    toDel->prevEdit = NULL;    
    if (toDel->mergeList)
    {
      editCanvas->numInMergedLists -= (NumInList(toDel->mergeList));
      sprintf(diagMsg, "number in merged list :%d\n", editCanvas->numInMergedLists);
      PrintDiagnostics(diagMsg);
      editCanvas->numMerged--;
    }
    free(toDel);    
    editCanvas->numInLinkedList--;
  }
}



/*
 * Notify callback function for `dubMergeButton'.
 * Creates a new entry in srcEdit arrays. 
 * Increments origLines.
 * Deletes selected edits and replaces them by one new edit.
 * Defaults to audio dub on.
 * Previously set audio and speed settings will be reset to defaults.
 * Label must be entered.
 */
void
  EditMerge(item, event)
Panel_item	item;
Event		*event;
{
  EditLink* mergedEdit;
  EditLink* current;
  EditLink* toDel;
  MergeEdit* newMerge;
  MergeEdit* tmp;
  char listEntry[30];
  char diagMsg[30];
  
  current = editCanvas->listHead;				    /* Start from the end */
  
  strcpy(srcEditLabel[origLines], 
	 (char*) xv_get(vcrDub_window1->dubMergeLabel, PANEL_VALUE));
  srcEditSpeed[origLines] = 30;					    /* Defaults to '30' */
  srcEditAudio[origLines] = 0;					    /* Defaults to 'Stereo' */
  srcEditAudioDub[origLines] = Yes;
  srcEditMergeEdit[origLines] = NULL;				    /* Initialized to NULL */
  strcpy(srcEditStart[origLines], "na");			    /* Initialized to 'na' */
  strcpy(srcEditEnd[origLines], "na");				    /* Initialized to 'na' */
  
  while (current)						    /* Go up linked list */
  {
    if (current->sMode == Selected)
    {
      srcEditDur[origLines] += srcEditDur[current->origNum];
      sprintf(diagMsg, "duration : %d\n", srcEditDur[origLines]);
      PrintDiagnostics(diagMsg);
      toDel = current;
      
      if (srcEditMergeEdit[origLines])
      {
	tmp = srcEditMergeEdit[origLines];
	while (tmp->nextMergeEdit)					   
	  tmp = tmp->nextMergeEdit;				    /* tmp points to the last merge edit currently */
      } 
      
      if (current->mergeList)					    /* This is a merged edit */
      { 
	if (srcEditMergeEdit[origLines] == NULL)		    
	  CopyMergeEdits(&srcEditMergeEdit[origLines], current->origNum);
	else							    
	  CopyMergeEdits(&tmp->nextMergeEdit, current->origNum);
      }    
      else							    /* This is a normal edit */
      {
	newMerge = (MergeEdit*) malloc(sizeof(MergeEdit));
	strcpy(newMerge->startAddress, srcEditStart[current->origNum]);
	strcpy(newMerge->endAddress, srcEditEnd[current->origNum]);
	newMerge->nextMergeEdit = NULL;
	if (srcEditMergeEdit[origLines])
	  tmp->nextMergeEdit = newMerge;
	else
	  srcEditMergeEdit[origLines] = newMerge;
      }
      
      current = current->nextEdit;
      EditDelete(toDel);					    /* Delete the edit that was just merged */
    }
    else
      current = current->nextEdit;
  }
  editCanvas->firstSelected = -1;				    /* Nothing is selected */
  
  mergedEdit = (EditLink*) malloc(sizeof(EditLink));		    /* Add new edit to end of the list */
  if (editCanvas->listTail)
    editCanvas->listTail->nextEdit = mergedEdit;
  else
    editCanvas->listHead = mergedEdit;
  mergedEdit->nextEdit = NULL;
  mergedEdit->prevEdit = editCanvas->listTail;
  mergedEdit->origNum = origLines;
  mergedEdit->audioDub = Yes;					    /* Defaults to Yes */
  mergedEdit->mergeList = srcEditMergeEdit[origLines];
  mergedEdit->sMode = Unselected;
  editCanvas->listTail = mergedEdit;
  editCanvas->numInLinkedList ++;
  editCanvas->numMerged ++;
  editCanvas->numInMergedLists += (NumInList(mergedEdit->mergeList));
  sprintf(diagMsg, "Number in merged lists: %d\n", editCanvas->numInMergedLists);
  PrintDiagnostics(diagMsg);
  
  sprintf(listEntry, "%5d.  %-26.22s%3d", origLines+1,		    /* Display new edit on source edit list */
	  srcEditLabel[origLines], srcEditDur[origLines]);
  xv_set (vcrDub_window1->dubScrollList,
	  PANEL_LIST_INSERT, origLines,
	  PANEL_LIST_STRING, origLines, listEntry,
	  PANEL_LIST_FONT, origLines, font,
	  NULL);
  
  origLines ++;      
  
  DrawLinkedList(editCanvas->listHead, No);
  xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubMergeLabel, PANEL_VALUE, "Merge - No label", NULL);
  xv_set(vcrDub_window1->dubMergeLabel, PANEL_INACTIVE, TRUE, NULL);
  
  notSaved = 1;
}


/*
 * Notify callback function for `dubDelButton'.
 */
void
  DubDelete(item, event)
Panel_item	item;
Event		*event;
{
  EditLink* toDel;
  
  toDel = GetEdit(editCanvas->firstSelected -1);
  EditDelete(toDel);
  
  notSaved = 1;
  editCanvas->firstSelected = -1;				    /* Deselect edit */
  EditOpInactive();
  DrawLinkedList(editCanvas->listHead, No);
}


/* Switches the position of 2 edits next to each other in linked list */
void
  SwitchConsec(top, bottom)
EditLink* top;
EditLink* bottom;
{
  if (top->prevEdit == NULL)
    editCanvas->listHead = bottom;
  else
    top->prevEdit->nextEdit = bottom;
  
  if (bottom->nextEdit)
    bottom->nextEdit->prevEdit = top;
  else
    editCanvas->listTail = top;
  
  bottom->prevEdit = top->prevEdit;
  top->nextEdit = bottom->nextEdit;
  bottom->nextEdit = top;
  top->prevEdit = bottom;
}


/* Switches the position of 2 edits not next to each other in linked list */
void
  SwitchNonConsec(top, bottom)
EditLink* top;
EditLink* bottom;
{
  EditLink* tmp1;
  EditLink* tmp2;
  
  if (top->prevEdit == NULL)
    editCanvas->listHead = bottom;
  else
    top->prevEdit->nextEdit = bottom;
  
  if (top->nextEdit)
    top->nextEdit->prevEdit = bottom;
  
  if (bottom->prevEdit)
    bottom->prevEdit->nextEdit = top;
  
  if (bottom->nextEdit)
    bottom->nextEdit->prevEdit = top;
  else
    editCanvas->listTail = top;
  
  tmp1 = bottom->prevEdit;
  tmp2 = bottom->nextEdit;
  bottom->prevEdit = top->prevEdit;
  top->prevEdit = tmp1;
  bottom->nextEdit = top->nextEdit;
  top->nextEdit = tmp2;
  
}


void RearrangeEdit(canvas, replacedEditNum)
     CanvasData* canvas;
     int replacedEditNum;					    /* from 1 to n */
{
  EditLink* movingEdit;
  EditLink* replacedEdit;
  
  if (replacedEditNum != canvas->firstSelected)			    /* Not moving to same place */
  {
    replacedEdit = GetEdit(replacedEditNum-1);			    /* Get the edits involved in shift */
    movingEdit = GetEdit(canvas->firstSelected - 1);
    
    if (replacedEdit->nextEdit == movingEdit)			    /* Two edits are right next to each other */
      SwitchConsec(replacedEdit, movingEdit);
    else if (movingEdit->nextEdit == replacedEdit)
      SwitchConsec(movingEdit, replacedEdit);
    else if (replacedEditNum > canvas->firstSelected)		    /* replaced is below original */
      SwitchNonConsec(movingEdit, replacedEdit);
    else if (replacedEditNum < canvas->firstSelected)		    /* replaced is above original */
      SwitchNonConsec(replacedEdit, movingEdit);
  }  
  canvas->firstSelected = replacedEditNum;
  EditOpActive(movingEdit->audioDub);
  DrawLinkedList(editCanvas->listHead, No);
  
}

/*
 * Event callback function for `dubEditCanvas'.
 */
Notify_value
  EditEventHandler(win, event, arg, type)
Xv_window	win;
Event		*event;
Notify_arg	arg;
Notify_event_type type;
{
  int tempChosenEdit;
  int replacedEdit;
  static int leftButtonWasDown;
  static int prevEditY;
  static int numRightSelected;
  char label[40];
  EditLink* current;
  EditLink* prev;
  
  numRightSelected = 0;
  if (event_is_down(event) && editCanvas->numInLinkedList >0)	    /* Something got selected */
  {
    tempChosenEdit = event_y(event) / (EditHeight + EditGap);	    /* Get the relative position of the mouse click, 
								       with respect to the height and spacing of the icons */
    if (tempChosenEdit && (tempChosenEdit <= editCanvas->numInLinkedList)) 
    {								    /* An application is now chosen */
      current = GetEdit(tempChosenEdit-1);
      if (event_id(event) == MS_LEFT)				    /* Left mouse button was clicked */
      {
	prevEditY = event_y(event) - (EditHeight)/2;		    /* Only left mouse dragging is permitted */
	
	leftButtonWasDown = 1;
	if (editCanvas->firstSelected != tempChosenEdit) 
	{
	  if (editCanvas->firstSelected > 0)			    /* Redraw the previously chosen edit so that 
								       it is unselected */
	  {
	    prev = GetEdit(editCanvas->firstSelected-1);
	    prev->sMode = Unselected;
	    sprintf(label, "%3d. %-23.22s", prev->origNum+1,
		    srcEditLabel[prev->origNum]);
	    DrawEdit(LeftMargin, editCanvas->firstSelected*(EditGap + EditHeight), 
		     label, prev->audioDub, Unselected);			    
	  }
	  sprintf(label, "%3d. %-23.22s", current->origNum+1,
		  srcEditLabel[current->origNum]);
	  DrawEdit(LeftMargin, tempChosenEdit*(EditGap + EditHeight), /* Redraw the current edit to first-selected */
		   label, current->audioDub, Selected);	
	  xv_set(vcrDub_window1->dubAudioStg, PANEL_VALUE, 
		 (int) current->audioDub, NULL);
	  
	  editCanvas->firstSelected = tempChosenEdit;		    
	  EditOpActive(current->audioDub);
	  if (current->sMode == Selected)			    /* Used to be second-selected */
	  {
	    xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, TRUE, NULL);
	    xv_set(vcrDub_window1->dubMergeLabel, PANEL_VALUE, "Merge - No label", NULL);
	    xv_set(vcrDub_window1->dubMergeLabel, PANEL_INACTIVE, TRUE, NULL);
	  }
	  else
	    current->sMode = Selected;
	}
      }
      else if (event_id(event) == MS_RIGHT)
      {
	if (editCanvas->firstSelected != tempChosenEdit)	    /* If selected with left button already, do nothing */
	{
	  if (current->sMode == Selected)			    /* Right mouse button allows deselection on 2nd click */
	  {
	    sprintf(label, "%3d. %-23.22s", current->origNum+1,
		    srcEditLabel[current->origNum]);
	    DrawEdit(LeftMargin, tempChosenEdit*(EditGap + EditHeight), 
		     label, current->audioDub, Unselected);			
	    current->sMode = Unselected;
	    if (numRightSelected)
	      numRightSelected--;
	  }
	  else							    /* Select it */
	  {
	    current->sMode = Selected;	    
	    sprintf(label, "%3d. %-23.22s", current->origNum+1,
		    srcEditLabel[current->origNum]);
	    DrawEdit(LeftMargin, tempChosenEdit*(EditGap + EditHeight), 
		     label, current->audioDub, Selected);	
	    numRightSelected--;
	  }
	  
	  if ((editCanvas->firstSelected >0) && (numRightSelected)) /* First edit is also selected */
	  {
	    xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, FALSE, NULL);
	    xv_set(vcrDub_window1->dubMergeLabel, PANEL_INACTIVE, FALSE, NULL);
	  }
	  else
	  {
	    xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, TRUE, NULL);
	    xv_set(vcrDub_window1->dubMergeLabel, PANEL_VALUE, "Merge - No label", NULL);
	    xv_set(vcrDub_window1->dubMergeLabel, PANEL_INACTIVE, TRUE, NULL);
	  }
	}
      }
      else if (event_id(event) == LOC_DRAG && leftButtonWasDown)    /* Dragging */
      {
	if (leftButtonWasDown > 1) 
	{
	  XDrawRectangle(editCanvas->display, editCanvas->drawable, editCanvas->gcXor, 
			 LeftMargin, prevEditY,
			 EditWidth, EditHeight);
	}
	leftButtonWasDown ++;
	prevEditY = event_y(event);
	if (prevEditY < 0)					    /* Set the lower and upper limits where the 
								       'Edit box' can move */
	  prevEditY = 0;
	else if (prevEditY > editCanvas->numInLinkedList * (EditHeight + EditGap)) 
	  prevEditY = editCanvas->numInLinkedList * (EditHeight + EditGap);
	
	prevEditY -= EditHeight/2;
	XDrawRectangle(editCanvas->display, editCanvas->drawable, editCanvas->gcXor, 
		       LeftMargin, prevEditY,
		       EditWidth, EditHeight);
      }
    }
  }
  else if (event_is_up(event))					    /* Mouse up */
  {
    if (leftButtonWasDown > 1)					    /* Perform the following only if there had been dragging */
    {
      XDrawRectangle(editCanvas->display, editCanvas->drawable, editCanvas->gcXor, 
		     LeftMargin, prevEditY,
		     EditWidth, EditHeight);
      leftButtonWasDown = 0;
      replacedEdit = event_y(event) / (EditHeight + EditGap);	   
      if (replacedEdit < 1) 
	replacedEdit = 1;
      else if (replacedEdit > editCanvas->numInLinkedList) 
	replacedEdit = editCanvas->numInLinkedList;
      
      if (replacedEdit != editCanvas->firstSelected) 
      {
	RearrangeEdit(editCanvas, replacedEdit);
	notSaved = 1;
      }
    }
  }
  return notify_next_event_func(win, (Notify_event) event, arg, type);
}


/*
 * Repaint callback function for `dubEditCanvas'.
 */
void
  EditRepaintHandler(canvas, paint_window, rects)
Canvas		canvas;
Xv_window	paint_window;
Rectlist	*rects;
{
  if (editCanvas->listHead)
    DrawLinkedList(editCanvas->listHead, No);
}


/*
 * Notify callback function for `playDeckDeviceStg'.
 * Set up play deck's video object.
 */
void
  SetPlayDevice(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  int playPort;
  char deviceName[MaxNameLength];
  char playDev[10];
  
  playPort = xv_get(vcrDub_dubSetupWindow->playDeckStg, PANEL_VALUE);
  if (playPort)
    strcpy(playDev, "/dev/ttyb");
  else
    strcpy(playDev, "/dev/ttya");
  
  strcpy(deviceName, allDevices[value]);
  playVideo = DevInit(deviceName, playDev);
  DevStop(playVideo);
  DevSetDefaults(playVideo, 3, 0, 0, 0);	
  
}

/*
 * Notify callback function for `recDeckDeviceStg'.
 * Set up record deck's video object.
 */
void
  SetRecDevice(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  int recPort;
  char deviceName[MaxNameLength];
  char recDev[10];
  
  recPort = xv_get(vcrDub_dubSetupWindow->recDeckStg, PANEL_VALUE);
  if (recPort)
    strcpy(recDev, "/dev/ttyb");
  else
    strcpy(recDev, "/dev/ttya"); 
  
  strcpy(deviceName, allDevices[value]);
  recVideo = DevInit(deviceName, recDev);
  DevStop(recVideo);
  DevSetDefaults(recVideo, 3, 0, 0, 0);	
}


/*
 * Notify callback function for `playDeckStg'.
 */
void
  SetPlayDeck(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  if (value == 0)
    xv_set(vcrDub_dubSetupWindow->recDeckStg, PANEL_VALUE, 1, NULL);
  else
    xv_set(vcrDub_dubSetupWindow->recDeckStg, PANEL_VALUE, 0, NULL);
}

/*
 * Notify callback function for `recDeckStg'.
 */
void
  SetRecDeck(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  if (value == 0)
    xv_set(vcrDub_dubSetupWindow->playDeckStg, PANEL_VALUE, 1, NULL);
  else
    xv_set(vcrDub_dubSetupWindow->playDeckStg, PANEL_VALUE, 0, NULL);
}


/* Set all functions inactive while dubbing */
void
  SetAppMute()
{
  xv_set(vcrDub_window1->dubDocButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubMergeLabel, PANEL_VALUE, "Merge - No label", NULL);
  xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubDelButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_INACTIVE, TRUE, NULL);
}


/* Set all functions active after dubbing */
void
  SetAppActive()
{
  xv_set(vcrDub_window1->dubDocButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubDelButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_INACTIVE, FALSE, NULL);
}



char*
  Modify(time, duration)
char time[12];
int duration;						    /* in frames per second */
{
  char newAddress[12];
  int n;
  int hr;
  int min;
  int sec;
  int frame;
  
  n = ConvertToAddress(time) + duration;
  ConvertToTime(&hr, &min, &sec, &frame, n);
  
  sprintf(newAddress, "%.2d:%.2d:%.2d:%.2d", hr, min, sec, frame);
  return(newAddress);
  
}


void
  DubThisEdit(recordStart, playStart, playEnd, toDubAudio, editNum)
char recordStart[12];
char playStart[12];
char playEnd[12];
enum Boolean toDubAudio;
int editNum;
{
  static int rStart;
  int rEnd;
  int pStart;
  int pEnd;
  char status[50];						    /* String to describe status of dub */
  
  if (recordStart)
    rStart = ConvertToAddress(recordStart);
  pStart = ConvertToAddress(playStart);
  pEnd = ConvertToAddress(playEnd);
  rEnd = rStart + (pEnd - pStart);
  
  sprintf(status, "Searching to edit %d on play deck", editNum);    /* Do dubbing */
  xv_set(vcrDub_window1->dubStatusMsg, PANEL_VALUE, status, NULL);
  XFlush((Display*)xv_get(vcrDub_window1->window1, XV_DISPLAY));
  DevPlayFromTo(playVideo, pStart, pStart, FrameRate);		    /* Search to beginning of segment on playback deck */
  
  if (toDubAudio == Yes)
    DevSetAudio(playVideo, 1);
  else
    DevSetAudio(playVideo, 0);
  
  if (recordStart)						    /* If NULL, continue from current position */
  {
    sprintf(status, "Searching on record deck (edit %d)", editNum);
    xv_set(vcrDub_window1->dubStatusMsg, PANEL_VALUE, status, NULL);
    XFlush((Display*)xv_get(vcrDub_window1->window1, XV_DISPLAY));
    DevPlayFromTo(recVideo, rStart, rStart, FrameRate);		    /* Search to beginning of record point on record deck */
    DevRecordFromTo(recVideo, 0, 0, 0);				    /* Put record deck into Insert edit mode */
  }
  DevPlayFromTo(playVideo, NULL, pEnd, FrameRate);		    /* Begin playing the segment on the playback deck */
  DevRecordFromTo(recVideo, 0, rEnd, 0);			    /* Begin recording on the record deck */
  
  rStart = rEnd;						    /* In case of merged edit, we want next start address ...
								       ... on record side to be current end address */
  sprintf(status, "Done with edit %d", editNum);
  xv_set(vcrDub_window1->dubStatusMsg, PANEL_VALUE, status, NULL);
  XFlush((Display*)xv_get(vcrDub_window1->window1, XV_DISPLAY));
}


/* Start dubbing.
 * Load linked list into edit list.
 * Dub edits using info in edit list.
 */
void
  StartDub()
{
  MergeEdit* current;						    /* Current merged edit portion */
  char playTape[MaxNameLength];					    /* Play tape header */
  char recTape[MaxNameLength];					    /* Play tape header */
  char newEditStart[MaxListSize][12];				    /* Array of edit start times for dubbed tape */
  char newEditEnd[MaxListSize][12];				    /* Array of edit end times for dubbed tape */
  int i;
  int tmphr; 
  int tmpmin; 
  int tmpsec;
  int tmpframe;
  int mergeEditDur;						    /* Duration for entire merged edit */
  int editInterval;						    /* Interval on new tape between edits, in frames per second */
  
  LoadEditList(editCanvas->listHead);				    /* Loads linked list info into edit list */
  xv_set(vcrDub_window1->dubEditListMsg, PANEL_VALUE, "Dubbed Edit List");
  XFlush((Display*)xv_get(vcrDub_window1->window1, XV_DISPLAY));
  
  sprintf(recTape, "NULL");
  sprintf(playTape, "NULL");
  
  editInterval = FrameRate *
    (int) xv_get(vcrDub_dubSetupWindow->intervalTxt, PANEL_VALUE) ;
  
  ConvertToTime(&tmphr, &tmpmin, &tmpsec, &tmpframe,
		FrameRate * xv_get(vcrDub_dubSetupWindow->startAddTxt, PANEL_VALUE));
  
  sprintf(newEditStart[0], "%.2d:%.2d:%.2d:%.2d",
	  tmphr, tmpmin, tmpsec, tmpframe);
  
  for (i = 0; i< origLines; i++)
  {
    if (srcEditMergeEdit[i])					    /* This is a merged edit */
    {
      current = srcEditMergeEdit[i];
      DubThisEdit(newEditStart[i], current->startAddress,
		  current->endAddress, Yes, i+1);
      mergeEditDur = (ConvertToAddress(current->endAddress) 
		      - ConvertToAddress(current->startAddress));   /* In frames per second */
      current = current->nextMergeEdit;
      while (current)
      {
	DubThisEdit(NULL, current->startAddress,
		    current->endAddress, srcEditAudioDub[i], i+1);  /* Continue from here */
	
	mergeEditDur += (ConvertToAddress(current->endAddress) 
			 - ConvertToAddress(current->startAddress)); /* In frames per second */
	current = current->nextMergeEdit;
      }
      DevStop(recVideo);
      
      strcpy(newEditEnd[i], Modify(newEditStart[i], mergeEditDur));
      strcpy(newEditStart[i+1], Modify(newEditEnd[i], editInterval));
    }
    else							    /* This is a regular edit */
    {
      DubThisEdit(newEditStart[i], srcEditStart[i], srcEditEnd[i], srcEditAudioDub[i], i+1);
      if (srcEditAudioDub[i] == No)
	srcEditAudio[i] = 3;					    /* Audio is muted because not dubbed onto tape */
      strcpy(newEditEnd[i],
	     Modify(newEditStart[i], 
		    (ConvertToAddress(srcEditEnd[i])
		     - ConvertToAddress(srcEditStart[i]))));
      strcpy(newEditStart[i+1], Modify(newEditEnd[i], editInterval));
    }
  }
  
  DevStop(recVideo);						    /* End of dubbing */
  DevStop(playVideo);
  for (i = 0; i< origLines; i++)				    /* Copy dub array times into source array */
  {
    strcpy(srcEditStart[i], newEditStart[i]);
    strcpy(srcEditEnd[i], newEditEnd[i]);
    srcEditMergeEdit[i] = NULL;					    /* Already merged, so no longer set */
  }
  editCanvas->numMerged = 0;
  editCanvas->numInMergedLists = 0;
  
  strncpy(tapeCode, recTape, 4);
  notSaved = 1;
  dubbing = 0;
  
  SetAppActive();
  SetUpLinkedList(&editCanvas->listHead, &editCanvas->listTail);    /* Set up new linked list */
  DrawLinkedList(editCanvas->listHead, No);
  
  Browse(NULL, BrowseSave, 1, "#VCR Edit Document#", "vcrDub");	    /* Pop up OpenPanel, with id = 1 */
}								    /* end function StartDubbing */


/*
 * Notify callback function for `dubSerialDoneButton'.
 */
void
  SerialDone(item, event)
Panel_item	item;
Event		*event;
{
  xv_set(vcrDub_dubSetupWindow->dubSetupWindow, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(vcrDub_dubSetupWindow->dubSetupWindow, XV_SHOW, FALSE, NULL);
  
  xv_set(vcrDub_window1->dubStatusMsg, PANEL_VALUE, "Initializing decks", NULL);
  dubbing = 1;
  SetAppMute();
  StartDub();
}


void
  InitGlobals()
{  
  srcEditSel = -1;
  origLines = 0;
  notSaved = 0;
  repeat = 0;
  dubbing = 0;
  hitQuit = 0;
  playVideo = NULL;
  recVideo = NULL;
  
}



void
  InitCanvas(dataPtr)
CanvasData* dataPtr;
{
  dataPtr->paintWindow = canvas_paint_window(vcrDub_window1->dubEditCanvas);
  dataPtr->drawable = (Window) xv_get(dataPtr->paintWindow, XV_XID);
  dataPtr->display = (Display*) xv_get(vcrDub_window1->dubEditCanvas, XV_DISPLAY);
  
  dataPtr->gc = XCreateGC(dataPtr->display, dataPtr->drawable, 0, 0); /* GC stuff */
  XSetFont(dataPtr->display, dataPtr->gc, XLoadFont(dataPtr->display, "6x13"));
  XSetLineAttributes(dataPtr->display, dataPtr->gc, 1, LineSolid, CapRound, JoinRound);
  
  dataPtr->gcXor = XCreateGC(dataPtr->display, dataPtr->drawable, 0, 0); 
  XSetFunction(dataPtr->display, dataPtr->gcXor, GXxor);
  
  dataPtr->cms = xv_create(NULL, CMS, 
			   CMS_SIZE, NumColors, 
			   CMS_TYPE, XV_STATIC_CMS, 
			   CMS_COLORS, myColors,
			   NULL);
  
  xv_set(vcrDub_window1->dubEditCanvas,
	 CANVAS_HEIGHT, 1800,
	 WIN_CMS, dataPtr->cms,
	 CANVAS_AUTO_SHRINK, FALSE,
	 CANVAS_AUTO_EXPAND, TRUE, 
	 NULL);
  
  dataPtr->pixelTable = (unsigned long*) xv_get(dataPtr->cms, CMS_INDEX_TABLE);
  
  audioImage = XCreateBitmapFromData(dataPtr->display, dataPtr->drawable, 
				     audioBitmap_bits, audioBitmap_width, audioBitmap_height);
  
  XSetForeground(dataPtr->display, dataPtr->gcXor, 
		 dataPtr->pixelTable[Grey92]^dataPtr->pixelTable[Black]);
  
  dataPtr->firstSelected = -1;					    /* Nothing is initially selected */
  dataPtr->listHead = NULL;
  dataPtr->listTail = NULL;
  dataPtr->numInLinkedList = 0;
  dataPtr->numInMergedLists = 0;
  dataPtr->numMerged = 0;
  
}


/*
 * Notify callback function for `removeInfoWindowButton'.
 */
void
  RemoveInfoWindow(Panel_item item, Event *event)
{
  xv_set(vcrDub_infoPopup->infoPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(vcrDub_infoPopup->infoPopup, XV_SHOW, FALSE, NULL);
  
  /* gxv_start_connections DO NOT EDIT THIS SECTION */
  
  /* gxv_end_connections */
}


/*
 * Menu handler for `DocMenu (About vcrDub...)'.
 */
Menu_item
  ShowInfoPanel(Menu_item item, Menu_generate op)
{
  switch (op)
  {
   case MENU_DISPLAY:
    break;
    
   case MENU_DISPLAY_DONE:
    break;
    
   case MENU_NOTIFY:
    xv_set(vcrDub_infoPopup->infoPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(vcrDub_infoPopup->infoPopup, XV_SHOW, TRUE, NULL);
    
    /* gxv_start_connections DO NOT EDIT THIS SECTION */
    
    /* gxv_end_connections */
    
    break;
    
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}


#ifdef MAIN

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;

void
  main(argc, argv)
int		argc;
char		**argv;
{
  
  Rect  tempRect;
  char msg[MaxPathLength+20];
  int i;
  
  startFilename = NULL;						    /* Command line parsing */
  CheckOptions(argc, argv);
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /* Initialize XView. */
  INSTANCE = xv_unique_key();
  

  vcrDub_window1 = vcrDub_window1_objects_initialize(NULL, NULL);   /* Initialize user interface components. */
  vcrDub_dubSetupWindow =
    vcrDub_dubSetupWindow_objects_initialize(NULL, vcrDub_window1->window1);
  vcrDub_infoPopup =
    vcrDub_infoPopup_objects_initialize(NULL, vcrDub_window1->window1);
  
  frame_get_rect(vcrDub_window1->window1,&tempRect);		    /* Get the size of the top-level window */
  tempRect.r_top = 250;
  tempRect.r_left = 200;
  frame_set_rect(vcrDub_window1->window1,&tempRect);		    /* Set the position of the top-level window */
  
  frame_get_rect(vcrDub_infoPopup->infoPopup,&tempRect);	    /* Get the size of the info popup window */
  tempRect.r_top = 100;
  tempRect.r_left = 330;
  frame_set_rect(vcrDub_infoPopup->infoPopup,&tempRect);	    /* Set the position of the info popup window */

  xv_set(vcrDub_window1->dubStatusMsg, PANEL_VALUE, " ", NULL);	    /* Initialize window elements */
  xv_set(vcrDub_dubSetupWindow->playDeckStg, PANEL_VALUE, 1, NULL);
  xv_set(vcrDub_dubSetupWindow->recDeckStg, PANEL_VALUE, 0, NULL);
  xv_set(vcrDub_dubSetupWindow->startAddTxt, PANEL_VALUE, 10, NULL);

  frame_get_rect(vcrDub_dubSetupWindow->dubSetupWindow,&tempRect);  /* Get the size of the dub setup window */
  tempRect.r_top = 250;
  tempRect.r_left = 665;
  frame_set_rect(vcrDub_dubSetupWindow->dubSetupWindow,&tempRect);  /* Set the position of the dub setup window */

  xv_set(vcrDub_window1->dubAddButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubMergeButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubMergeLabel, PANEL_VALUE, "Merge - No Label", NULL);
  xv_set(vcrDub_window1->dubMergeLabel, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubDelButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrDub_window1->dubAudioStg, PANEL_VALUE, 1, NULL);
  
  font = (Xv_font *)  xv_find(vcrDub_window1->window1, FONT,	    /* Set font for panel list */
			      FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			      FONT_STYLE,  FONT_STYLE_NORMAL,
			      FONT_SIZE, 12,
			      NULL);
  
  InitGlobals();						    /* Initialize global variables */
  editCanvas = (CanvasData*) malloc(sizeof(CanvasData));	    /* Pointer to editCanvas data */
  InitCanvas(editCanvas);					    /* Initialize canvas variables */
  
  xv_set(vcrDub_dubSetupWindow->playDeckDeviceStg,		    /* Initialize choices to devices supported */
	 PANEL_NCHOICES, numDevices, NULL);
  xv_set(vcrDub_dubSetupWindow->recDeckDeviceStg, 
	 PANEL_NCHOICES, numDevices, NULL);
  
  for (i=0; i < numDevices; i++)
  {
    xv_set(vcrDub_dubSetupWindow->playDeckDeviceStg,
	   PANEL_CHOICE_STRING, i, allDevices[i], 
	   NULL);
    xv_set(vcrDub_dubSetupWindow->recDeckDeviceStg,
	   PANEL_CHOICE_STRING, i, allDevices[i], 
	   NULL);
  }
  
  CreateBrowse(OpenHandler, SaveHandler, 
	       vcrDub_window1->window1);			    /* Set up Open Panel */
  
  if (startFilename)						    /* Set filename at start-up if included */
  {								    /* as part of command line */
    sprintf(msg, "Current document: %s", startFilename);
    xv_set(vcrDub_window1->dubStatusMsg,				    
	   PANEL_LABEL_STRING, msg, 
	   NULL);
    Browse(startFilename, BrowseCheckOpen, NULL, "#VCR Edit Document", "vcrDub");
  }
  else
    xv_set(vcrDub_window1->dubStatusMsg,				    
	   PANEL_LABEL_STRING, "No document loaded",
	   NULL);
  
  xv_main_loop(vcrDub_window1->window1);			    /* Turn control over to XView. */
  if (!hitQuit)
    DubQuit(NULL, NULL);
  exit(0);
}								    /* end function main */

#endif

