/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/cdEdit/RCS/edit.c,v 2.0 91/10/06 21:01:18 chua Exp $ */
/* $Log:	edit.c,v $
 * Revision 2.0  91/10/06  21:01:18  chua
 * Update to version 2.0
 * 
 * Revision 1.22  91/09/24  16:29:23  chua
 * The duration array stores time in milliseconds instead of seconds.  So in line
 * 122, do the appropriate division to obtain seconds for the duration display.
 * 
 * Revision 1.21  91/09/03  14:57:09  chua
 * Added the copyright header.
 * Added new arrays to store the volume, balance and duration for each edit.
 * 
 * Added a notify procedure for the label textfield, so that hitting return on it has
 * the same effect as hitting the add button.
 * 
 * Changed the name of the edit list from EditPopupList to EditList.
 * 
 * In the edit notify functions (add, delete, modify, delete), make a call to UpdateHeader
 * to indicate that the file has been modified.
 * 
 * Revision 1.2  91/07/11  18:30:38  chua
 * In the panel list notify procedure, if an item is deselected, set 
 * editnum to -1.  
 * 
 * Revision 1.1  91/07/09  16:46:48  chua
 * 
 * 
 * Revision 1.0  91/07/08  13:45:47  chua
 * Initial revision
 *  */

static char editrcsid[] = "$Header: /Source/Media/collab/cdEdit/RCS/edit.c,v 2.0 91/10/06 21:01:18 chua Exp $";

#include "main.h"

Xv_font* font;							    /* Font used in the panel list */
int starttrack[MaxListSize];					    /* variables to store the edit list information */
int startmin[MaxListSize];
int startsec[MaxListSize];
int startframe[MaxListSize];
int endtrack[MaxListSize];
int endmin[MaxListSize];
int endsec[MaxListSize];
int endframe[MaxListSize];
int volume[MaxListSize];
int balance[MaxListSize];
int duration[MaxListSize];
char label[MaxListSize][MaxLabelLength];			    /* array to store the label of each entry in the edit list */
								       
/*
 * Formats an edit entry for the panel list.
 * current indicates which entry is to be formatted.
 * buf will contain the formatted entry.
 */
void PrintOnList(current, buf)
     int current;
     char buf[70];
{
  sprintf (buf, " %5d.  %-29.27s%02d:%02d:%02d:%02d  %02d:%02d:%02d:%02d", current+1,
	   label[current], starttrack[current], startmin[current],
	   startsec[current], startframe[current], 
	   endtrack[current], endmin[current], 
	   endsec[current], endframe[current]); 
}

/*
 * Print the number of edit entries on the list.
 */
void PrintNumEdit()
{
  char edit[20];
  
  sprintf(edit, "No. of Edits : %d", lines);
  xv_set(cdEdit_EditPopup->EditListNumEditsMsg, PANEL_LABEL_STRING,
	 edit, NULL);
}

/*
 * Notify callback function for `EditPopupList'.
 * On a Deselect operation, this function will update the WhichEdit message to indicate "New Edit". It will also make the modify and delete buttons inactive.
 * On a Select operation,  it will load in the chosen selection onto the current selection textfields.
 */
int List(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  int selection;
  char buf[80];
  
  switch(op) {
   case PANEL_LIST_OP_DESELECT:
    xv_set(cdEdit_EditPopup->CurrentSelectionMsg, PANEL_LABEL_STRING, "Current Selection : None",
	   NULL);
    editnum = -1;
    xv_set(cdEdit_EditPopup->ModifyButton, PANEL_INACTIVE, TRUE, NULL);	/* Make the modify and delete buttons inactive */
    xv_set(cdEdit_EditPopup->DeleteButton, PANEL_INACTIVE, TRUE, NULL);
    break;
   case PANEL_LIST_OP_SELECT:
    sscanf(string, "%d", &selection);
    editnum = selection - 1;
    LoadStartEnd();						    /* Load the current selection onto the current selection textfields */
    sprintf(buf, "%d", (duration[editnum] + 500) / 1000);
    xv_set(cdEdit_EditPopup->EditPopupDurationText, PANEL_VALUE, buf, NULL); /* Display the duration in the duration textfield */
    break;
  }
  return XV_OK;
}

/*
 * Notify callback function for `EditPopupLabelText'.
 * Calls the EditAdd function to add a new edit on the edit list.
 */
Panel_setting LabelText(item, event)
     Panel_item	item;
     Event		*event;
{
  if (discInPlayer == 1)					    /* Add only if there is a disc in the player, since editing functions are */
  {								    /* disabled when there is no disc */
    EditAdd(item, NULL);
  }
  return panel_text_notify(item, event);
}

/*
 * Notify callback function for `AddButton'.
 * Adds the current selection to the edit list.
 */
void EditAdd(item, event)
     Panel_item		item;
     Event		*event;
{
  char buf[100];
  
  if (SetDuration(lines) == Error)				    /* Display the duration of the current selection */
  {
    return;
  }
  GetStartEnd(lines);						    /* Gets the start and end times from the selection textfields and set it */
								    /* to the next free element in the edit list arrays */
  PrintOnList(lines, buf);					    /* Format the new entry to be inserted on the edit list */
  xv_set(cdEdit_EditPopup->EditList,
	 PANEL_LIST_INSERT, lines,
	 PANEL_LIST_STRING, lines, buf,
	 PANEL_LIST_FONT, lines, font,
	 NULL);
  if (editnum != -1)						    /* Deselect any previously selected entry */
  {
    xv_set(cdEdit_EditPopup->EditList,
	   PANEL_LIST_SELECT, editnum, FALSE,
	   NULL);
  }
  xv_set(cdEdit_EditPopup->EditList,				    /* Select the new entry */
	 PANEL_LIST_SELECT, lines, TRUE,
	 NULL);
  editnum = lines;
  lines++;
  if (discInPlayer == 1) 
  {
    xv_set(cdEdit_EditPopup->ModifyButton, PANEL_INACTIVE, FALSE, NULL); /* Make the modify and delete buttons active */
    xv_set(cdEdit_EditPopup->DeleteButton, PANEL_INACTIVE, FALSE, NULL); /* only if there is a disc in the player */
  }
  sprintf(buf, "Current Selection : %d", lines);		    /* Update the edit status information */
  xv_set(cdEdit_EditPopup->CurrentSelectionMsg, PANEL_LABEL_STRING, buf, NULL);
  PrintNumEdit();
  change = 1;
  UpdateHeader(1);
}

/*
 * Notify callback function for `ModifyButton'.
 * Modifies the currently selected entry in the edit list with the values in the current selection textfields.
 */
void EditModify(item, event)
     Panel_item	item;
     Event		*event;
{
  char buf[100];
  
  if (SetDuration(editnum) == Error)				    /* Display the duration of the current selection */
  {
    return;
  }
  GetStartEnd(editnum);						    /* Gets the start and end times from the selection textfields */
  PrintOnList(editnum, buf);					    /* Format the modified entry to be replaced on the edit list */
  xv_set(cdEdit_EditPopup->EditList,
	 PANEL_LIST_STRING, editnum, buf,
	 PANEL_LIST_FONT, editnum, font,
	 NULL);
  change = 1;
  UpdateHeader(1);
}

/*
 * Notify callback function for `DeleteButton'.
 * Delete the currently selected entry from the edit list.
 */
void EditDelete(item, event)
     Panel_item	item;
     Event		*event;
{
  int i;
  char buf[100];
  
  xv_set(cdEdit_EditPopup->EditList,			    /* Deselect the current selection */
	 PANEL_LIST_SELECT, editnum, FALSE,
	 NULL);
  if (editnum != 0) 
  {
    xv_set(cdEdit_EditPopup->EditList,			    /* Due to some strange XView (?) bug, we have to select another entry first */
	   PANEL_LIST_SELECT, editnum - 1, TRUE,		    /* before doing the deletion. */
	   NULL);
  }
  xv_set(cdEdit_EditPopup->EditList,			    /* Delete the entry */
	 PANEL_LIST_DELETE, editnum,
	 NULL);
  lines --;	
  for (i=editnum; i < lines;i++) 				    /* Now, move all the entries following the deleted one up by one */
  {
    starttrack[i] = starttrack[i+1];
    startmin[i] = startmin[i+1];
    startsec[i] = startsec[i+1];
    startframe[i] = startframe[i+1];
    endtrack[i] = endtrack[i+1];
    endmin[i] = endmin[i+1];
    endsec[i] = endsec[i+1];
    endframe[i] = endframe[i+1];
    volume[i] = volume[i+1];
    balance[i] = balance[i+1];
    duration[i] = duration[i+1];
    strcpy(label[i], label[i+1]);
    PrintOnList(i, buf);
    xv_set(cdEdit_EditPopup->EditList,
	   PANEL_LIST_STRING, i, buf,
	   PANEL_LIST_FONT, i, font,
	   NULL);
  }
  if (editnum != 0)						    /* Deselect any selected entry.  After the delete operation, no entry should be */
  {								    /* selected. */
    xv_set(cdEdit_EditPopup->EditList,
	   PANEL_LIST_SELECT, editnum - 1, FALSE,
	   NULL);
  }
  xv_set(cdEdit_EditPopup->CurrentSelectionMsg, PANEL_LABEL_STRING, "Current Selection : None", NULL); /* Update the edit status information */
  PrintNumEdit();
  editnum = -1;

  xv_set(cdEdit_EditPopup->ModifyButton, PANEL_INACTIVE, TRUE, NULL); /* Make the delete and modify buttons inactive */
  xv_set(cdEdit_EditPopup->DeleteButton, PANEL_INACTIVE, TRUE, NULL);
  change = 1;
  UpdateHeader(1);
}

/*
 * Notify callback function for `DeleteAllButton'.
 * This function will clear the whole edit list.
 */
void EditDeleteAll(item, event)
     Panel_item	item;
     Event		*event;
{
  int i;
  char edit[20];

  if (CheckChanges (DeleteAll) == NOTICE_YES)			    /* Check if there are unsaved changes */
  {
    return;
  }
  for (i=0; i < lines; i++)					    /* Delete all the entries in the list */
  {
    xv_set(cdEdit_EditPopup->EditList,
	   PANEL_LIST_DELETE, 0,
	   NULL);
  }
  lines  = 0;
  editnum = -1;							    /* Update the edit status information */
  PrintNumEdit();
  change = 1;
  UpdateHeader(1);
  sprintf(edit, "No. of Edits : 0");
  xv_set(cdEdit_EditPopup->EditListNumEditsMsg, PANEL_LABEL_STRING,
	 edit, NULL);
  xv_set(cdEdit_EditPopup->CurrentSelectionMsg, PANEL_LABEL_STRING, "Current Selection : None", NULL);
  xv_set(cdEdit_EditPopup->ModifyButton, PANEL_INACTIVE, TRUE, NULL); /* Make the modify and delete buttons inactive */
  xv_set(cdEdit_EditPopup->DeleteButton, PANEL_INACTIVE, TRUE, NULL);
}

/*
 * Menu handler for `DocumentMenu (Close this window)'.
 */
Menu_item CloseEditPopupHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(cdEdit_PreviewPopup->PreviewPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(cdEdit_PreviewPopup->PreviewPopup, XV_SHOW, FALSE, NULL);
    xv_set(cdEdit_EditPopup->EditPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(cdEdit_EditPopup->EditPopup, XV_SHOW, FALSE, NULL);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Frame done proc for the edit popup window pushpin.
 */
void EditDone()
{
  CloseEditPopupHandler(NULL, MENU_NOTIFY);
}
