/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/Sony1550Driver.c,v 1.13 92/09/28 12:40:04 drapeau Exp $ */
/* $Log:	Sony1550Driver.c,v $
 * Revision 1.13  92/09/28  12:40:04  drapeau
 * PlayFromTo() was modified to reflect the new semantics of the VideoObject
 * library.
 * 
 * Revision 1.12  92/07/30  15:18:02  drapeau
 * Several changes:
 * * Re-formatted all function declarations to conform to ANSI function
 *   prototype standards.
 * * Added more complete diagnostic messages: diagnostics now list the
 *   serial port used for the command when possible.
 * * Removed all hard-coded references to 30 frames-per-second frame rate,
 *   replacing references with the new definition "FrameRate".  All math
 *   is now based on this definition, allowing the driver to be used in
 *   places with frame rates other than 30 fps.
 * 
 * Revision 1.11  92/06/17  01:01:34  drapeau
 * Changed termio usage, used more portable "termio" structure and
 * associated calls instead of "termios".
 * 
 * Revision 1.10  91/09/30  17:08:13  lim
 * Implemented Sony1550Ping.
 * 
 * Revision 1.9  91/09/27  14:31:57  lim
 * Replaced first parameter of ioctl in line 137 with
 * theObject->DevConfig->fd
 * 
 * Revision 1.8  91/08/24  17:50:22  lim
 * Implemented PrintDiagnostics.
 * 
 * Revision 1.7  91/08/24  13:39:03  lim
 * 1. Updated to use status codes in new PlayerStatus.h
 * 2. Clear Marker() removed as part of video object.
 * 
 * Revision 1.6  91/08/17  20:41:24  lim
 * 1. Stop and Pause implemented with interrupt capability.
 * 2. Clear Marker implemented to do as specs in videoObj.h require.
 * 
 * Revision 1.5  91/08/08  17:17:21  lim
 *  1. PlayAtSpeedDir() - try to input device-dependent speed rather than an arbitrary speed.
 *    ie use CalcSpeed before calling PlayAtSpeedDir()
 * 2. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 3. QueryMedium() - one parameter added, 'result'. Used to return non-integer results.
 * 
 * Revision 1.4  91/08/07  13:17:36  lim
 * 1. Made as part of VideoLib.a. Removed references to vEdit application.
 * 2. Added instance pointer, "theObject" to all public functions. 
 * 3. ErrorDecode() is now a private function. It calls DisplayError() 
 * (in 'videoObjects.c') to display the error message. 
 * 
 * Revision 1.3  91/08/02  12:52:46  lim
 * 1. SetDefaults function changed to take in 4 parameters"
 * audio, addressingMode, addressDisplayOnOff, and displayMode
 * 2. xv_set is no longer called within driver code, but from
 * vEdit application.
 * 3. Pause is removed.
 * 
 * Revision 1.2  91/07/30  10:10:24  lim
 * Omission: previousReturnLength was never set. This has been corrected.
 * 
 * Revision 1.1  91/07/29  22:21:08  lim
 * Initial revision
 *  */

static char sony1550rcsid[] = "$Header: /Source/Media/collab/VideoObject/RCS/Sony1550Driver.c,v 1.13 92/09/28 12:40:04 drapeau Exp $";

#include "Sony1550Driver.h"

static int	completion=0;
static char	diagMsg[64];


/*  Support routines.  These implement the videodisc functions  */

/* Convert an integer to an alphanumeric string */

IntToAlpha (int n, char s[6])
{
  int i;
  int k;
  int maxnum;
  char temp[6];
  
  i = 0;
  temp[i++] = n%10 + '0';
  while ((n /= 10) > 0)
    temp[i++] = n%10 + '0';
  temp[i] = '\0';
  maxnum = i--;
  for (k=(5-maxnum); k < 5; k++)
    s[k] = temp[i--];
  for (k=0; k < (5-maxnum); k++)
    s[k] = '0';
  s[5] = '\0';
}								    /* end function IntToAlpha */


/* Sets up default values for video options */

int
  Sony1550SetDefaults(VideoObject* theObject,
		      int audio,
		      int addressingMode,			    
		      int addressDisplayOnOff,
		      int displayMode)				    /* Unused */
{  
  Sony1550SetAudio(theObject, audio);				    /* set audio */
  Sony1550SetAddMode(theObject, addressingMode);		    /* set addressing mode */
  Sony1550SetAddressDisplay(theObject, addressDisplayOnOff, 
			    displayMode);			    /* set address display */
  
  Sony1550Play(theObject);					    /* puts player in 'ready' state */
  Sony1550Still(theObject);
  return PlayerOk;
  
}								    /* end function Sony1550SetDefaults */


int
  Sony1550SendCode(VideoObject* theObject,
		   char*	code,
		   int		returnLength,
		   int		mode)
{
  char		in[20];
  char		out[20];
  char		errorMsg[50];
  int		nr;
  int		num;
  static int	previousReturnLength = 0;
  struct termio	lineAttributes;
  
  if (previousReturnLength != returnLength) 
  {
    ioctl(theObject->DevConfig->fd,TCGETA,(char*)&lineAttributes);
    lineAttributes.c_cc[VMIN] = returnLength;
    previousReturnLength = returnLength;
    if (ioctl(theObject->DevConfig->fd,TCSETA,(char*)&lineAttributes) == -1)
      return PlayerCantInitDevice;
  }
  
  if (completion) 
  {
    nr = read(theObject->DevConfig->fd, out, 1);		    /* get rid of completion code */
    sprintf(diagMsg, "%s :\tRead completion code.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }    
  completion = 0;
  
  strcpy(in,code);
  write(theObject->DevConfig->fd,in,strlen(in));
  sprintf(diagMsg, "%s :\tSent %s %d\n",
	  theObject->DevConfig->serialPort,
	  in, strlen(in));
  PrintDiagnostics(diagMsg);
  
  bzero(out, 20);
  nr = read(theObject->DevConfig->fd,out,20);
  sprintf(diagMsg, "%s :\tRead %s %d\n",
	  theObject->DevConfig->serialPort,
	  out, strlen(out));
  PrintDiagnostics(diagMsg);
  out[nr] = '\0';
  
  if (out[0] == 0x01) 
  {
    strcpy (in, out);
    strcpy (out, &(in[1]));
    nr --;
  }
  
  if (nr) 
  {
    if (out[0] == 0x0a) 
      return PlayerOk;
    else if (out[0] == 0x0b)
    {
      Sony1550ErrorDecode(theObject, PlayerNakError, errorMsg);
      return PlayerNakError;
    }
    else if (out[0] == 0x02) 
    {
      Sony1550ErrorDecode(theObject, PlayerReturnError, errorMsg);
      return PlayerReturnError;
    }
    else 
    {
      if (mode == 1)
	switch(out[4])
	{
	 case 0x01:
	  return PlayerForwardPlay;
	 case 0x02:
	 case 0x10:
	  return PlayerForwardFast;
	 case 0x04:
	  return PlayerForwardSlow;
	 case 0x08:
	  return PlayerReverseStep;
	 case 0x20:
	  return PlayerStill;
	 case 0x40:
	  return PlayerPause;
	 default:
	  Sony1550ErrorDecode(theObject, PlayerUnknownReturnCode, errorMsg);
	  return PlayerUnknownReturnCode;
	}
      else if (mode == 2) return (out[0]);
      
      sscanf(out,"%d",&num);
      return (num);
    }
  }
}								    /* end function Sony1550SendCode */



/*  Player functions  */

int
  Sony1550Eject(VideoObject* theObject)
{
  int returnCode;
  
  returnCode = Sony1550SendCode(theObject, "*",1,0);
  completion = 1;
  return (returnCode);
  
}								    /* end function Sony1550Eject */


int
  Sony1550Stop(VideoObject* theObject)
{
  completion = 0;
  Sony1550SendCode(theObject, "V", 1, 0);			    /* Interrupts search/repeat mode */
  return(Sony1550SendCode(theObject, "?",1,0));			    /* Stops */
}								    /* end function Sony1550Stop */


int
  Sony1550Play(VideoObject* theObject)
{
  completion = 0;
  return(Sony1550SendCode(theObject, ":",1,0));
}								    /* end function Sony1550Play */


int
  Sony1550PlayFromTo(VideoObject*	theObject,
		     int		from,
		     int		to,
		     int		speed)
{
  int status;
  int deviceSpeed;
  
  if ((from == to) || ((from > 0) && (to == 0)))		    /* Case a & b: search to "from" and still (non-blocking... */
  {								    /* ...search doesn't apply here since search time is quick) */
    return(Sony1550Search(theObject, from));			    /* So both types of searches fall into the same search code */
  }
  if ((from == 0) && (to > 0))					    /* Case c: play from current position until address... */
  {								    /* ..."to" has been reached. */
    Sony1550SendCode(theObject, "D",1,0);			    /* Using repeat segment function with parameter 1 */
    Sony1550StopMarker(theObject, to);					    
    Sony1550SendCode(theObject, "@",1,0);
    Sony1550SendCode(theObject, "1",1,0);
    Sony1550SendCode(theObject, "@",1,0);
    completion = 1;
    deviceSpeed = Sony1550CalcSpeed(theObject, speed, 0);
    return(Sony1550PlayAtSpeedDir(theObject, deviceSpeed, Forward)); /* Play at designated speed */
  }  
  if ((from > 0) && (to > 0) && (from < to))			    /* Case d: play from "from" to "to" */
  {
    status = Sony1550Search(theObject, from);			    /* Search to address "from" */
    if (status != PlayerOk)
      return(status);
    Sony1550SendCode(theObject, "D",1,0);			    /* Using repeat segment function with parameter 1 */
    Sony1550StopMarker(theObject, to);					    
    Sony1550SendCode(theObject, "@",1,0);
    Sony1550SendCode(theObject, "1",1,0);
    Sony1550SendCode(theObject, "@",1,0);
    completion = 1;
    deviceSpeed = Sony1550CalcSpeed(theObject, speed, 0);
    return(Sony1550PlayAtSpeedDir(theObject, deviceSpeed, Forward)); /* Play at designated speed */
  }
  return(PlayerReturnError);					    /* If this code is reached, incorrect from & to were requested */
}								    /* end function Sony1550PlayFromTo */



int
  Sony1550Still(VideoObject* theObject)
{
  completion = 0;
  Sony1550SendCode(theObject, "V", 1, 0);			    /* Interrupts search/repeat mode */
  return(Sony1550SendCode(theObject, "O",1,0));
}								    /* end function Sony1550Still */


int
  Sony1550Step(VideoObject* theObject, enum Direction dir)
{
  completion = 0;
  if (dir == Forward)
    return(Sony1550SendCode(theObject, "+",1,0));
  else
    return(Sony1550SendCode(theObject, ",",1,0));
}								    /* end function Sony1550Step */


int
  Sony1550FastForward(VideoObject* theObject)
{
  int i;
  
  completion = 0;
  Sony1550SendCode(theObject, ">",1,0);
  for (i=0; i<200000; i++);
  return(Sony1550SendCode(theObject, ":",1,0));
}								    /* end function Sony1550FastForward */


int
  Sony1550Reverse(VideoObject* theObject)
{
  int i;
  
  completion = 0;
  Sony1550SendCode(theObject, "N",1,0);
  for (i=0; i<200000; i++);
  return(Sony1550SendCode(theObject, ":",1,0));
}								    /* end function Sony1550Reverse */


int 
  Sony1550CalcSpeed(VideoObject* theObject,
		    int framesPerSecond,
		    int playMode)				    /* unused */
{
  if (framesPerSecond > FrameRate)
    return 3 * FrameRate;
  else if (framesPerSecond == FrameRate)
    return FrameRate;
  else if (framesPerSecond == 0)
    return 0;
  
  return(FrameRate / MyRound(FrameRate, framesPerSecond));
}								    /* end function Sony1550CalcSpeed */



int
  Sony1550PlayAtSpeedDir(VideoObject* theObject,
			 int framesPerSecond,
			 enum Direction dir)
{
  char		s[6];
  char		t[2];
  int		speed;
  int		denom;
  static int	lastSpeed = 0;
  
  speed = Sony1550CalcSpeed(theObject, framesPerSecond, 0);
  if (framesPerSecond == lastSpeed)				    /* Was a new speed requested? */
    return(PlayerOk);						    /* No, don't bother sending a new command to the player */
  
  lastSpeed = framesPerSecond;					    /* Update last speed requested */
  completion = 0;
  
  if (speed == 3 * FrameRate)					    /* There is only one speed (3x) higher than normal */
    if (dir == Forward)
      return(Sony1550SendCode(theObject, ";",1,0));		    /* Fast forward */
    else
      return(Sony1550SendCode(theObject, "K",1,0));		    /* Reverse */
  
  if (speed == FrameRate)
    if (dir == Forward)
      return(Sony1550Play(theObject));				    /* Play forward */
    else
      return(Sony1550SendCode(theObject, "J",1,0));		    /* Play Reverse */
  
  if (speed == 0)						    /* Still */
    return (Sony1550Still(theObject));
  
  denom = MyRound(FrameRate, framesPerSecond);			    /* Slower than regular speed */
  
  IntToAlpha(denom, s);
  
  if (dir == Forward)
    Sony1550SendCode(theObject, "=",1,0);
  else
    Sony1550SendCode(theObject, "M",1,0);
  
  t[0] = s[0];
  t[1] = '\0';
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[1];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[2];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[3];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[4];
  Sony1550SendCode(theObject, t,1,0);
  return (Sony1550SendCode(theObject, "@",1,0));
}								    /* end function Sony1550PlayAtSpeedDir */


int
  Sony1550Search(VideoObject* theObject, int to)
{
  char s[6];
  char t[2];
  
  completion = 0;
  IntToAlpha(to, s);
  Sony1550SendCode(theObject, "C",1,0);
  t[0] = s[0];
  t[1] = '\0';
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[1];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[2];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[3];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[4];
  Sony1550SendCode(theObject, t,1,0);
  return (Sony1550SendCode(theObject, "@",2,0));
  
}								    /* end function Sony1550Search */


int
  Sony1550StopMarker(VideoObject* theObject, int stopM)
{
  char s[6];
  char t[2];
  
  completion = 0;
  IntToAlpha(stopM, s);
  t[0] = s[0];
  t[1] = '\0';
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[1];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[2];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[3];
  Sony1550SendCode(theObject, t,1,0);
  t[0] = s[4];
  return (Sony1550SendCode(theObject, t,1,0));
}								    /* end function Sony1550StopMarker */


int
  Sony1550SetAudio(VideoObject* theObject, int ac)
{
  completion = 0;
  
  switch (ac) 
  {
   case PlayerAudioMute:
    Sony1550SendCode(theObject, "G",1,0);
    Sony1550SendCode(theObject, "I",1,0);
    break;
   case PlayerAudioLeft:
    Sony1550SendCode(theObject, "F",1,0);
    Sony1550SendCode(theObject, "I",1,0);
    break;
   case PlayerAudioRight:
    Sony1550SendCode(theObject, "G",1,0);
    Sony1550SendCode(theObject, "H",1,0);
    break;
   case PlayerAudioStereo:
    Sony1550SendCode(theObject, "F",1,0);
    Sony1550SendCode(theObject, "H",1,0);
    break;
  }
  return PlayerOk;
}								    /* end function Sony1550SetAudio */


int
  Sony1550SetVideo(VideoObject* theObject, int vc)
{
  if (vc == FeatureOn)
    return(Sony1550SendCode(theObject, "'", 1, 0));
  else
    return(Sony1550SendCode(theObject, "&", 1, 0));
  
}								    /* end function Sony1550SetVideo */



int
  Sony1550SetAddressDisplay(VideoObject* theObject,
			    int onOff,
			    int mode)				    /* Unused */
{
  completion = 0;
  if (onOff == FeatureOn)
    return (Sony1550SendCode(theObject, "P", 1, 0));
  else
    return (Sony1550SendCode(theObject, "Q", 1, 0));
}								    /* end function Sony1550SetAddressDisplay */


int
  Sony1550SetAddMode(VideoObject* theObject, int mode)
{
  completion = 0;
  switch (mode) 
  {
   case PlayerFrameMode:
    return(Sony1550SendCode(theObject, "U",1,0));
   case PlayerChapterMode:
    return(Sony1550SendCode(theObject, "i",1,0));
   default:
    return PlayerUnknownCommand;
  }
}								    /* end function Sony1550SetAddMode */


int 
  Sony1550QueryFrame(VideoObject* theObject)
{
  return(Sony1550SendCode(theObject, "`",5,0));
}								    /* end function Sony1550QueryFrame */


int 
  Sony1550QueryChapter(VideoObject* theObject)
{
  return (Sony1550SendCode(theObject, "v",2,0));
}								    /* end function Sony1550QueryChapter */


int 
  Sony1550QueryMedium(VideoObject* theObject,			    /* Inquires disk ID */
		      char* result)				    /* unused */
{
  return (Sony1550SendCode(theObject, "y",4,0));
}								    /* end function Sony1550QueryMedium */


int 
  Sony1550QueryStatus(VideoObject* theObject)
{
  return(Sony1550SendCode(theObject, "g",5,1));
}								    /* end function Sony1550QueryStatus */


int 
  Sony1550Ping(VideoObject* theObject)
{
  char out[4];
  int numRead;
  
  write(theObject->DevConfig->fd, "y", 1);			    /* Query disk ID command */
  sprintf(diagMsg, "%s :\tSent ping.\n",
	  theObject->DevConfig->serialPort);
  PrintDiagnostics(diagMsg);
  bzero(out, 4);
  numRead = read(theObject->DevConfig->fd, out, 1);
  if (numRead != 0)						    /* Was the read successful? */
  {								    /* Yes */
    sprintf(diagMsg, "%s :\tSuccessful ping.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  else
  {
    sprintf(diagMsg, "%s :\tPing unsuccessful.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  return numRead;
}								    /* end function Sony1550Ping */


void
  Sony1550ErrorDecode(VideoObject* theObject,
		      int ecode,
		      char *string)
{
  switch (ecode) 
  {
   case PlayerCantInitDevice:
    strcpy(string,"Cannot initialize device properly");
    break;
   case PlayerInvalidBaudRate:
    strcpy(string,"Baud rate is invalid");
    break;
   case PlayerNakError:
    strcpy(string, "Command not accepted by player in this mode");
    break;
   case PlayerReturnError:
    strcpy(string, "An error has occurred in the player");
    break;
   case PlayerUnknownCommand:
    strcpy(string,"An unknown command has been issued");
    break;
   default:
    strcpy(string, "Unknown error code");
    break;
  }
  DisplayError(string, " ");
}								    /* end function Sony1550ErrorDecode */
