/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/region.c,v 1.0 91/09/30 17:02:37 chua Exp Locker: drapeau $ */
/* $Log:	region.c,v $
 * Revision 1.0  91/09/30  17:02:37  chua
 * Update to version 1.0
 * 
 * Revision 0.49  91/09/23  17:19:37  chua
 * The Show Region Info item is now in the Options menu button.
 * 
 * Revision 0.48  91/09/19  17:29:05  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.47  91/09/17  17:18:40  chua
 * Renamed Region to tlRegion.
 * 
 * Revision 0.46  91/09/04  15:13:32  chua
 * Use AlertMessage whenever a notice_prompt is to be displayed.
 * 
 * Revision 0.45  91/08/16  17:06:45  chua
 * Removed some variables which are not used.
 * 
 * Revision 0.44  91/08/05  16:53:32  chua
 * Deleted the RepaintCanvas routine, as it is no longer necessary.  In places where it
 * is called, just call the ScrollToFirstQuarter routine, which will do the necessary
 * repaint as well.
 * 
 * Revision 0.43  91/08/05  13:06:21  chua
 * In the InsertRegion routine, check first that a region has been selected, before attempting to
 * insert a new entry in the region list.  
 * 
 * Revision 0.42  91/07/30  17:33:13  chua
 * When a region is added or deleted, set the change flag to 1.
 * 
 * Revision 0.41  91/07/26  17:30:11  chua
 * 
 * 
 * Revision 0.40  91/07/26  17:14:26  chua
 * This file contains the routines that deals with inserting and deleting annotated
 * regions from the TimeLine document.
 *  */

static char regionrcsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/region.c,v 1.0 91/09/30 17:02:37 chua Exp Locker: drapeau $";

#include "main.h"

/*
 * Menu handler for `OptionsMenu (Show Region Info ...)'.
 * Opens the region info popup window.
 */
Menu_item ShowRegionInfoHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(tlFrame->RegionPopup->RegionPopup,
	   FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(tlFrame->RegionPopup->RegionPopup,
	   XV_SHOW, TRUE, NULL);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Function that will return a pointer to the appropriate region node given its relative position in the region list.
 */
tlRegion *FindRegion(tlFrame)
     TimeLineFramePtr tlFrame;
{
  int 		i;
  tlRegion *currentRegion;
  
  currentRegion = tlFrame->regionHead;					   
  for (i=0; i < tlFrame->regionEdit && currentRegion != NULL; i++)
    currentRegion = currentRegion->next;
  return currentRegion;
}

/*
 * Notify callback function for `RegionList'.
 * On a PANEL_LIST_OP_DESELECT operation, this function will reset the selected region message to zero, and also the regionEdit and selectedRegion
 * variables to indicate that no edit has been selected.
 * On a PANEL_LIST_OP_SELECT operation, this function will set the regionEdit and selectedRegion variables to point to the selected node and scroll
 * the canvas so that the selected region will be visible.
 */
int RegionListNotify(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  int selection;
  char selected[4];
  TimeLineFramePtr tlFrame;
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->RegionPopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    tlFrame->regionEdit = -1;
    tlFrame->selectedRegion = NULL;
    sprintf(selected, "%d", tlFrame->regionEdit + 1);
    xv_set(tlFrame->RegionPopup->RegionSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
    break;
   case PANEL_LIST_OP_SELECT:
    sscanf(string, "%d", &selection);				    /* Get the number of the edit which has been selected */
    tlFrame->regionEdit = selection - 1;
    tlFrame->selectedRegion = (tlRegion *) FindRegion(tlFrame);
    ScrollToFirstQuarter(tlFrame, tlFrame->selectedRegion->startX, 0);
    sprintf(selected, "%d", tlFrame->regionEdit + 1);
    xv_set(tlFrame->RegionPopup->RegionSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
    if (tlFrame->areaSelected)					    /* Clear the previously selected area */
      XFillRectangle(tlFrame->dpyDraw, tlFrame->xidDraw, tlFrame->gcLine, (tlFrame->startX  / tlFrame->zoomLevel) - tlFrame->canvasStart, 
		     tlFrame->startY, (tlFrame->endX - tlFrame->startX) / tlFrame->zoomLevel,
		     tlFrame->endY - tlFrame->startY);
    DrawPlaybackHead(-1, tlFrame);				    /* Remove the playback head from the canvas */
    tlFrame->startX = tlFrame->selectedRegion->startX;
    tlFrame->endX = tlFrame->selectedRegion->endX;
    tlFrame->startY = 0;
    tlFrame->endY = tlFrame->numberOfApps * (IconHeight + IconGap);
    DeselectNote(tlFrame);					    /* Deselect any currently selected note first */
    DrawSelectArea(tlFrame);
    xv_set(tlFrame->RegionPopup->RegionLabelText, PANEL_VALUE, 
	   tlFrame->selectedRegion->label, NULL);
    break;
   case PANEL_LIST_OP_VALIDATE:
    break;
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}

/*
 * This function will update the region panel list which indicates where all the annotated regions are.
 * It goes through the Region list for the frame and updates the panel list accordingly.
 */
void UpdateRegionList(tlFrame)
     TimeLineFramePtr tlFrame;
{
  tlRegion *currentRegion;
  char 	buf[100], numRegions[5], selected[4];
  int 	count = 0, i;
  int 	oldlines, replace;
  
  sprintf(numRegions, "%d", tlFrame->numRegion);		    /* Update the display for the number of regions */
  xv_set(tlFrame->RegionPopup->RegionCountMsg,
	 PANEL_LABEL_STRING, numRegions, NULL);
  oldlines = xv_get(tlFrame->RegionPopup->RegionList,		    /* Get the number of rows currently in the panel list */
		    PANEL_LIST_NROWS);
  replace = oldlines;
  if (tlFrame->numRegion <= oldlines) 
    replace = tlFrame->numRegion;
  currentRegion = tlFrame->regionHead;
  xv_set(tlFrame->RegionPopup->RegionList,			    /* Hide the panel list while updating is done */
	 XV_SHOW, FALSE, NULL);
  for (i=0; i < replace; i++)					    /* Replace the old strings by the new ones. */
  {
    sprintf(buf, "   %4d.    %-25.22s%4d:%02d  %4d:%02d",
	    count+1, currentRegion->label,
	    currentRegion->startMin, currentRegion->startSec,
	    currentRegion->endMin, currentRegion->endSec); 
    xv_set (tlFrame->RegionPopup->RegionList,
	    PANEL_LIST_STRING, i, buf,
	    PANEL_LIST_FONT, i, font,
	    NULL);
    count++;
    currentRegion = currentRegion->next;
  }
  if (tlFrame->numRegion > oldlines) 
  {
    for (i=oldlines; i < tlFrame->numRegion; i++)		    /* Insert the additional new strings */
    {
      sprintf(buf, "   %4d.    %-25.22s%4d:%02d  %4d:%02d",
      	      count+1, currentRegion->label,
	      currentRegion->startMin, currentRegion->startSec,
	      currentRegion->endMin, currentRegion->endSec); 
      xv_set (tlFrame->RegionPopup->RegionList,
	      PANEL_LIST_INSERT, i,
	      PANEL_LIST_STRING, i, buf,
	      PANEL_LIST_FONT, i, font,
	      NULL);
      count++;
      currentRegion = currentRegion->next;
    }
  }
  else								    /* Delete the excess old strings */
    for (i=tlFrame->numRegion; i < oldlines; i++)
      xv_set(tlFrame->RegionPopup->RegionList,
	     PANEL_LIST_DELETE, tlFrame->numRegion,
	     NULL);
  if (tlFrame->regionEdit >= 0) 
    xv_set(tlFrame->RegionPopup->RegionList,			    /* Select the newly added entry on the panel list */
	   PANEL_LIST_SELECT, tlFrame->regionEdit, 
	   TRUE, NULL);
  sprintf(selected, "%d", tlFrame->regionEdit + 1);		    /* Update the selected Region field */
  xv_set(tlFrame->RegionPopup->RegionSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
  xv_set(tlFrame->RegionPopup->RegionList,
	 XV_SHOW, TRUE, NULL);
}

/*
 * This procedure will insert a new region node in the Region list.  
 * It will return the position of the new node in the region list.
 */
int InsertNewRegion(tlFrame, newRegion)
     TimeLineFramePtr tlFrame;
     tlRegion *newRegion;
{
  int selected;
  int found;
  tlRegion *currentRegion;
  
  selected = 0;
  if (tlFrame->regionHead == NULL)				    /* Region list is empty.  Insert at the beginning. */
    tlFrame->regionHead = newRegion;
  else								    /* Go down the note list to find the appropriate startX to insert */
  {
    currentRegion = tlFrame->regionHead;
    found = 0;
    if (currentRegion->startX > newRegion->startX) 
    {
      found = 1;
      newRegion->next = currentRegion;
      tlFrame->regionHead = newRegion;
    }
    else while (currentRegion->next != NULL && !found)
    {
      selected ++;
      if (currentRegion->next->startX < newRegion->startX) 
	currentRegion = currentRegion->next;
      else 
      {
	newRegion->next = currentRegion->next;
	currentRegion->next = newRegion;
	found = 1;
      }
    }
    if (!found)							    /* Insert at the end of list */
    {
      selected ++;
      currentRegion->next = newRegion;
    }
  }
  return selected;
}

/*
 * Notify callback function for `InsertRegionButton'.
 * This function will insert a new region node in the region list and updates the region panel list.
 */
void InsertRegion(item, event)
     Panel_item	item;
     Event		*event;
{
  tlRegion *newRegion;
  int selected;
  TimeLineFramePtr tlFrame;
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->RegionPopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->startX == tlFrame->endX || tlFrame->noteSelected)    /* Check if a region has been selected */
  {
    AlertMessage(tlFrame, "Please select a region first.", NULL, NULL);
    return;
  }
  newRegion = (tlRegion *) malloc (sizeof(struct tlRegion));
  newRegion->next = NULL;
  newRegion->startX = tlFrame->startX;
  newRegion->endX = tlFrame->endX;
  newRegion->startMin = 0;
  newRegion->startSec = newRegion->startX / PixelsPerSecond;
  strcpy (newRegion->label, (char *) xv_get(tlFrame->RegionPopup->RegionLabelText, PANEL_VALUE));
  while (newRegion->startSec >= 60) 
  {
    newRegion->startMin ++;
    newRegion->startSec -= 60;
  }
  newRegion->endMin = 0;
  newRegion->endSec = newRegion->endX / PixelsPerSecond;
  while (newRegion->endSec >= 60) 
  {
    newRegion->endMin ++;
    newRegion->endSec -= 60;
  }
  selected = InsertNewRegion(tlFrame, newRegion);
  tlFrame->numRegion++;
  tlFrame->regionEdit = selected;
  tlFrame->selectedRegion = newRegion;
  ScrollToFirstQuarter(tlFrame, newRegion->startX, 0);		    /* Scroll to make the newly inserted Region marker visible */
  if (tlFrame->areaSelected)					    /* Clear the previously selected area */
    XFillRectangle(tlFrame->dpyDraw, tlFrame->xidDraw, tlFrame->gcLine, (tlFrame->startX  / tlFrame->zoomLevel) - tlFrame->canvasStart, 
		   tlFrame->startY, (tlFrame->endX - tlFrame->startX) / tlFrame->zoomLevel,
		   tlFrame->endY - tlFrame->startY);
  tlFrame->startY = 0;
  tlFrame->endY = tlFrame->numberOfApps * (IconHeight + IconGap);
  DrawSelectArea(tlFrame);
  UpdateRegionList(tlFrame);
  tlFrame->change = 1;						    /* Set the change flag to 1 */
  UpdateHeader(tlFrame, 1);
}

/*
 * Notify callback function for `DeleteRegionButton'.
 * Deletes the currently selected entry from the region info panel list and also the corresponding node from the region list.
 */
void DeleteRegion(item, event)
     Panel_item	item;
     Event		*event;
{
  TimeLineFramePtr tlFrame;
  tlRegion *currentRegion, *prevRegion;
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  int found;
  Window	owner = xv_get(ip->RegionPopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->regionHead == NULL || tlFrame->regionEdit == -1)	    /* Region list is empty, or there is no edit chosen */
    return;	
  xv_set(tlFrame->RegionPopup->RegionList,			    /* Deselect the selected entry on the panel list */
	 PANEL_LIST_SELECT, tlFrame->regionEdit, 
	 FALSE, NULL);
  currentRegion = tlFrame->regionHead;
  if (currentRegion == tlFrame->selectedRegion) 
    tlFrame->regionHead = currentRegion->next;
  else 
  {
    prevRegion = currentRegion;
    currentRegion = currentRegion->next;
    found = 0;
    while (currentRegion != NULL && !found) 
    {
      if (currentRegion == tlFrame->selectedRegion) 
      {
	prevRegion->next = currentRegion->next;
	found = 1;
      }
      prevRegion = currentRegion;
      if (!found) 
	currentRegion = currentRegion->next;
    }
  }
  tlFrame->numRegion --;
  tlFrame->regionEdit = -1;
  tlFrame->selectedRegion = NULL;
  free (currentRegion);
  UpdateRegionList(tlFrame);
  tlFrame->change = 1;						    /* Set the change flag to 1 */
  UpdateHeader(tlFrame, 1);
}

/*
 * Notify callback function for `ModifyRegionButton'.
 * Modifies the currently selected entry in the region info panel list.
 */
void ModifyRegion(item, event)
     Panel_item	item;
     Event		*event;
{
  char buf[100];
  TimeLineFramePtr tlFrame;
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->RegionPopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->selectedRegion != NULL) 
  {
    tlFrame->selectedRegion->startX = tlFrame->startX;
    tlFrame->selectedRegion->endX = tlFrame->endX;
    strcpy (tlFrame->selectedRegion->label, (char *) xv_get(tlFrame->RegionPopup->RegionLabelText, PANEL_VALUE));
    tlFrame->selectedRegion->startMin = 0;
    tlFrame->selectedRegion->startSec = tlFrame->selectedRegion->startX / PixelsPerSecond;
    while (tlFrame->selectedRegion->startSec >= 60) 
    {
      tlFrame->selectedRegion->startMin ++;
      tlFrame->selectedRegion->startSec -= 60;
    }
    tlFrame->selectedRegion->endMin = 0;
    tlFrame->selectedRegion->endSec = tlFrame->selectedRegion->endX / PixelsPerSecond;
    while (tlFrame->selectedRegion->endSec >= 60) 
    {
      tlFrame->selectedRegion->endMin ++;
      tlFrame->selectedRegion->endSec -= 60;
    }
    sprintf(buf, "   %4d.    %-25.22s%4d:%02d  %4d:%02d",
	    tlFrame->regionEdit + 1, tlFrame->selectedRegion->label,
	    tlFrame->selectedRegion->startMin, tlFrame->selectedRegion->startSec,
	    tlFrame->selectedRegion->endMin, tlFrame->selectedRegion->endSec); 
    xv_set(tlFrame->RegionPopup->RegionList,
	   PANEL_LIST_STRING, tlFrame->regionEdit, buf,
	   PANEL_LIST_FONT, tlFrame->regionEdit, font,
	   NULL);
    tlFrame->change = 1;					    /* Set the change flag to 1 */
    if (tlFrame->areaSelected)					    /* Clear the previously selected area */
      XFillRectangle(tlFrame->dpyDraw, tlFrame->xidDraw, tlFrame->gcLine, (tlFrame->startX  / tlFrame->zoomLevel) - tlFrame->canvasStart, 
		     tlFrame->startY, (tlFrame->endX - tlFrame->startX) / tlFrame->zoomLevel,
		     tlFrame->endY - tlFrame->startY);
    tlFrame->startY = 0;
    tlFrame->endY = tlFrame->numberOfApps * (IconHeight + IconGap);
    DrawSelectArea(tlFrame);
    UpdateHeader(tlFrame, 1);
  }
}

/*
 * Notify callback function for `ClearAllRegionButton'.
 * Clear the whole region list of all entries.
 */
void ClearAllRegion(item, event)
     Panel_item	item;
     Event		*event;
{
  TimeLineFramePtr tlFrame;
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->RegionPopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->numRegion > 0) 
  {
    tlFrame->change = 1;						
    UpdateHeader(tlFrame, 1);
    FreeRegion(tlFrame);
    UpdateRegionList(tlFrame);
  }
}

/*
 * Notify callback function for `RegionDoneButton'.
 * Close the region info popup window.
 */
void CloseRegionPopup(item, event)
     Panel_item	item;
     Event		*event;
{
  Region_RegionPopup_objects	*ip = (Region_RegionPopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  xv_set(ip->RegionPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(ip->RegionPopup, XV_SHOW, FALSE, NULL);
}

/*
 * This procedure will free all the Region nodes in a Region list.
 */
void FreeRegion(tlFrame)
     TimeLineFramePtr tlFrame;
{
  tlRegion *region, *freeRegion;
  
  region = tlFrame->regionHead;					    /* Free all the Region nodes */
  while (region != NULL) 
  {
    freeRegion = region;
    region = region->next;
    free (freeRegion);
  }
  tlFrame->regionHead = NULL;
  tlFrame->regionEdit = -1;
  tlFrame->selectedRegion = NULL;
  tlFrame->numRegion = 0;
}

/*
 * This function will set the start and end fields in the region info popup window.
 */
void SetStartEndRegion(tlFrame, startX, endX)
     TimeLineFramePtr tlFrame;
     int startX;
     int endX;
{
  int min, sec;
  char buf[7];

  sec = startX / PixelsPerSecond;
  min = 0;
  while (sec >= 60) 
  {
    min ++;
    sec -= 60;
  }
  sprintf (buf, "%4d:%02d", min, sec);
  xv_set(tlFrame->RegionPopup->StartRegionValueMsg, PANEL_LABEL_STRING, buf, NULL);
  sec = endX / PixelsPerSecond;
  min = 0;
  while (sec >= 60) 
  {
    min ++;
    sec -= 60;
  }
  sprintf (buf, "%4d:%02d", min, sec);
  xv_set(tlFrame->RegionPopup->EndRegionValueMsg, PANEL_LABEL_STRING, buf, NULL);
  
}



