/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/notesInfo.c,v 1.0 91/09/30 17:01:34 chua Exp Locker: drapeau $ */
/* $Log:	notesInfo.c,v $
 * Revision 1.0  91/09/30  17:01:34  chua
 * Update to version 1.0
 * 
 * Revision 0.72  91/09/25  14:04:30  chua
 * Changed InitInstInfoWindow to InitEditInfoWindow.
 * 
 * Revision 0.71  91/09/25  13:51:12  chua
 * Changed the instrument field, instInfo, to editInfo.
 * Changed InstrumentInfo to EditInfo.
 * 
 * Revision 0.70  91/09/23  17:13:26  chua
 * Change tlFrame->selectedInstrument to tlFrame->noteInstrument as the
 * latter is now used to indicate which instrument has the currently
 * selected note.
 * 
 * Revision 0.69  91/09/19  17:28:59  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.68  91/08/05  16:53:24  chua
 * Deleted the RepaintCanvas routine, as it is no longer necessary.  In places where it
 * is called, just call the ScrollToFirstQuarter routine, which will do the necessary
 * repaint as well.
 * 
 * Revision 0.67  91/08/05  13:04:32  chua
 * Changed the name of the function call, ScrollToMiddle to ScrollToFirstQuarter.
 * 
 * Revision 0.66  91/07/26  17:26:47  chua
 * In SelectNoteInfo, update the fields in the region info popup window so that 
 * the start and end positions of a selected region are zero.
 * 
 * Revision 0.65  91/07/22  15:21:20  chua
 * In the NotesInfoListNotify procedure, where we check to see if scrolling is necessary, the
 * code is replaced by a call to ScrollToMiddle instead.
 * 
 * Revision 0.64  91/07/17  16:33:16  chua
 * Modified some of the comments to the NotesInfoListNotify function.
 * 
 * Revision 0.62  91/07/17  10:30:06  chua
 * In the NotesInfoListNotify procedure (at line 512), when clearing the selected
 * area (line 512), take into the account the canvasStart position when specifying
 * the x coordinate.
 * 
 * Revision 0.61  91/07/09  18:25:02  chua
 * Made changes to the startX, endX variables so that they now store the position at the
 * largest zoom level (zoom level = 1).   Thus, the appropriate multiplication or
 * division by the zoom level has to be made when these variables are used.  This will
 * include lastX (position of the playback head) as well.
 * 
 * Revision 0.60  91/07/09  17:01:23  chua
 * In SelectNoteInfo, when setting the selected region, set the note selected
 * region (startnoteX, etc) to the note boundary as well.
 * 
 * Revision 0.59  91/06/25  17:45:48  chua
 * Added a new field, LabelText to the instrument info popup window.
 * Replaced all occurrences of the constant HalfSecondScale with the value 5 as the constant is not in
 * use anymore.
 * 
 * Revision 0.58  91/06/05  16:22:58  chua
 * In the InitNotesInfo routine, add code to hide a panel list when updating and reshow the
 * list again once updating is doen.
 * 
 * Revision 0.57  91/06/04  17:37:23  chua
 * Added the copyright comments in the beginning of the file.
 * 
 * Revision 0.56  91/06/04  17:29:53  chua
 * In the InitInstInfoWindow routine, add code to initialize the message field indicating
 * which TimeLine document the pop-up window belongs to.
 * 
 * Revision 0.55  91/06/03  11:12:00  chua
 * Make changes to accomodate multiple documents.  This involves identifying
 * which is the current active window, that is, the one where the last mouse
 * click was done.
 * 
 * Revision 0.54  91/05/30  12:07:59  chua
 * In ShowInfoWindow, make a call to InitNotesInfo to update the panel list.
 * 
 * In InitNotesInfo, an extra parameter is added (deselect).  This parameter is 
 * usually 1, except when called from ShowInfoWindow, when it has the value 0.
 * It tells the function whether the currently selected note is to be deselected.
 * Also, a check is made in the function to see if the pop-up window is currently
 * showing.  If not, the updating of the panel list is not done.  This is to
 * avoid the delays in updating the panel list.
 * 
 * Revision 0.53  91/05/29  18:31:13  chua
 * In the InitInstInfoWindow, delete the if-statement testing if instrument->instInfo == NULL, as
 * it is not necessary.
 * 
 * In the InitNotesInfo procedure, a minor change is made to the last part of the function, when
 * extra old entries are deleted.  
 * 
 * Revision 0.52  91/05/29  14:41:15  chua
 * Delete the ClearNoteInfoList and UpdateNotesCount procedures, which are made unnecessary by the
 * rewriting of the InitNotesInfo procedure.  This rewriting is an attempt to avoid the random
 * segmentation faults caused by the previous way of updating the info window panel lists.
 * Please refer to the function heading commnets of the InitNotesInfo procedure for more
 * information.
 * 
 * Revision 0.51  91/05/28  12:13:54  chua
 * Change in the DESELECT option in NotesInfoListNotify procedure.  Do not set selectedInstrument = NULL
 * then, as that might cause a segmentation fault since selectedInstrument may be accessed later on.
 * 
 * Revision 0.50  91/05/24  16:37:18  chua
 * *** empty log message ***
 * 
 * Revision 0.49  91/05/23  17:31:37  chua
 * In the ClearNotesInfoList routine,  call DeselectNote first to deselect any selected note before
 * deleting the whole list.
 * 
 * Revision 0.48  91/05/22  16:40:27  chua
 * In the SelectNotesInfo function, add the code that was in the CheckIfNoteSelected function
 * previously in edit.c  The code has the effect of setting the selected region coordinates
 * to the selected note.
 * 
 * Revision 0.47  91/05/22  13:56:31  chua
 * In SelectNoteInfo, set noteSelected to 1.
 * In DeselectNote, check if noteSelected = 1 before performing any of the procedures.  At the`
 * end of the procedures, set noteSelected = 0.
 * 
 * Revision 0.46  91/05/22  11:43:42  chua
 * In the InitNotesInfo and ClearNoteInfoList info, instead of making use of an attribute list
 * to perform updates, we now just use one xv_set call for each item to be updated, which 
 * results in quite a number of xv_set calls.  The reason the attribute list method is not 
 * used is because only 250 attributes can be passed in a xv_set call at one time, which would
 * limit the number of notes that can be added.  Also, when allocating space for the avlist,
 * it seems it requires more than what has been calculated, or else the program crashes at
 * some stage (not always well defined).
 * 
 * Revision 0.45  91/05/17  16:57:05  chua
 * *** empty log message ***
 * 
 * Revision 0.44  91/05/15  14:54:23  chua
 * Added a new function DeselectNote() which will deselect the previously selected note on both the notes
 * info panel list as well as on the timeline canvas display.
 * 
 * In the PANEL_LIST_OP_SELECT switch statement in the NoteInfoListNotify function, the section of code
 * deselecting the previously selected note is replaced by the above function call.
 * 
 * Revision 0.43  91/05/15  14:15:25  chua
 * Changes in the NotesInfoListNotify function:
 * 
 * In the PANEL_LIST_OP_DESELECT switch statement,  if the current instrument is the same as the previously
 * selected instrument, the selectedInstrument pointer is set to NULL, since we are deselecting a note.
 * 
 * In the PANEL_LIST_OP_SELECT swithc statement, if the previously selected instrument (as pointed to
 * by the selectedInstrument pointer) is different from the currently selected instrument, we deselect the
 * previously selected note in the instrument pointed to by selectedInstrument.  The selectedInstrument
 * pointer is then set to the currently chosen instrument.
 * 
 * 
 * Revision 0.42  1991/04/24  01:04:21  chua
 * A new function, ShowInfoWindow is created, which when called, will bring the instrument pop-up
 * window to the front.
 *
 * The function InitInstInfoWindow is now called each time an instrument is created.  The last
 * two lines (showing the window) are deleted and put in the ShowInfoWindow function.
 *
 * In the InitNotesInfo and ClearNoteInfoList procedures, a check is made to see if there are
 * any notes in the note list, since an error occurs if we try to do an xv_set on
 * the ATTR_LIST with no strings.  For example, we have PANEL_LIST_STRINGS, which is expected to
 * be followed by at least one string.  But if there are no notes, no string follows and an error
 * results.
 *
 * Revision 0.41  1991/04/08  22:00:26  chua
 * Correcting a change in the notesinforcsid line.
 *
 * Revision 0.40  1991/04/08  21:49:11  chua
 * This file contains functions to maintain the instrument info pop-up window.  Brief descriptions of the
 * functions in this file follows.
 *
 * InitInstInfoWindow - Creates the pop-up window for an instrument if necessary and moves it to the foreground.
 *  			(make it visible).
 * UpdateNotesCount - Update the notes count message in the info pop-up window.
 * InitNotesInfo - Updates the panel list in the info pop-up window with the notes information from the
 * 		   instrument's notes list.
 * FindNote - Returns a pointer to a note given its relative position in the notes list.
 * ClearNoteInfoList - Clears the panel list of its entries and also deselect any currently selected note.
 * SelectNoteInfo - Updates the info display fields in the pop-up window with values of the selected note,
 * 		    and draw the selected note in the 'Sunken' mode.
 * DeselectNoteInfo - Draw the deselected note back in the 'Raised' mode and clear all the info display fields.
 * NotesInfoListNotify - Called when the user selects or deselects an entry in the panel list in a info pop-up
 * 			 window.  First, determine which instrument this pop-up window belongs to by checking
 * 			 the PANEL_CLIENT_DATA value.  Then perform the appropriate selecting or deselecting
 * 			 procedures.
 * */

static char notesinforcsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/notesInfo.c,v 1.0 91/09/30 17:01:34 chua Exp Locker: drapeau $";

#include "main.h"
#include "EditInfo_ui.h"

Xv_font *font;							    /* Font used in the note info panel list */

/*
 * This function takes an instrument node pointer as an argument and creates a instrument info pop-up window for that instrument.
 * First, a check is made to see if a pop-up window has already been created for that window.
 * The objects initialize procedure is called to create the pop-up window if it does not already exist.  The first parameter in the call is the pointer
 * to the pop-up window.  This is checked to see if it is NULL, or if it already exist.  If it is the latter, then no new creation of a pop-up window
 * is done.  The second parameter is the owner of the pop-up window, which in this case is the parent frame (the TimeLine Editor window).
 * The first parameter is actually redundant in this case (we could have just set it to NULL), since we already checked if the pop-up window exists
 * earlier.
 * Set the instrument's relative position in the instrument list to be the value in the panel's PANEL_CLIENT_DATA.  This is necessary so that we can
 * differentiate between the various pop-up windows since they all call the same notify procedure.
 * Next, the application name and the hostname is set on the pop-up window textfields.
 * The pop-up window is made visible by pushing in the push-pin and setting XV_SHOW for that window to be TRUE.
 * Called by ShowNotesInfoHandler (openApps.c)
 */
void InitEditInfoWindow(instrument, tlFrame)
     Instrument *instrument;
     TimeLineFramePtr tlFrame;
{
  int whichFrame;
  char buf[40];
  
  whichFrame = xv_get(tlFrame->TimeLine_window->controls, PANEL_CLIENT_DATA);
  instrument->editInfo = EditInfo_EditInfoWindow_objects_initialize (instrument->editInfo, tlFrame->TimeLine_window->window);
  font = (Xv_font *) xv_find(instrument->editInfo->EditInfoWindow,  /* Set the font for the note info panel list */
			     FONT,
			     FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			     FONT_STYLE, FONT_STYLE_NORMAL,
			     FONT_SIZE, 12,
			     NULL);
  if (whichFrame == 0) 
    sprintf(buf, "Clipboard");
  else 
    sprintf (buf, "Document %d\n", whichFrame);
  xv_set(instrument->editInfo->TimeLineDocMsg,			    /* To indicate which TimeLine document the popup window belongs to */
	 PANEL_LABEL_STRING, buf, NULL);
  xv_set(instrument->editInfo->NoteInfoList,			    /* Attach the instrument's relative position as the PANEL_CLIENT_DATA. */
	 PANEL_CLIENT_DATA, instrument->relativePosition,	    /* Lets the program know which instrument the panel belongs to. */
	 NULL);
  xv_set(instrument->editInfo->AppNameText,			    /* Set the application name on the pop-up window */
	 PANEL_LABEL_STRING, instrument->port->appName, NULL);
  xv_set(instrument->editInfo->HostnameText,			    /* Set the hostname on the pop-up window */
	 PANEL_LABEL_STRING, instrument->port->hostName, NULL);
  xv_set(instrument->editInfo->MiddlePanel, XV_FONT, font, NULL);  /* Set the font for the panel list */
  InitNotesInfo(instrument, 1, tlFrame);					    /* Initialize the notes count and notes info */
  xv_set(instrument->editInfo->EditInfoWindow, XV_SHOW, FALSE, NULL); /* Do not show the pop-up window until the user request it */
}

/* 
 * This function will pop up the info window and bring it to the front.
 * It will also call the InitNotesInfo window to update the information on the info pop-up window.
 */
void ShowInfoWindow(instrument, tlFrame)
     Instrument *instrument;
     TimeLineFramePtr tlFrame;
{
  xv_set(instrument->editInfo->EditInfoWindow,		    /* Make the pop-up window visibile */
	 FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set(instrument->editInfo->EditInfoWindow, XV_SHOW, TRUE, NULL);
  InitNotesInfo(instrument, 0, tlFrame);
}

/*
 * This function will go through an instrument's note list and create the information on the info window panel list which contains information for
 * all the notes found in the instrument.
 * Note that the function will be performed only if the pop-up window is showing.  This is to avoid the delay when inserting or deleting notes due
 * to the updating of the panel list.  Thus, when the pop-up window is not showing, the panel list will not be updated.
 * The parameter deselect indicates if the selected note is to be deselected.  This is desired in most cases except when the info window is brought
 * forward from a closed position.  (when the function is called by ShowInfoWindow).
 * First, check to see if deselecting a selected note needs to be done.  This is done if the instrument is the same as the selected instrument and
 * the deselect parameter = 1.
 * Update the display for the number of notes in the info pop-up window.
 * Next, find out the current number of lines in the panel list.
 * Go through the panel list and update it with the new information.  Any old strings are replaced by new ones.  If the number of old strings exceed
 * the number of new strings, the excess old strings are deleted.  Conversely, if the number of new strings exceed the number of old strings, the
 * extra new strings will be inserted.
 * Lastly, highlight the selected note (if any) in the panel list.
 * Called by InitEditInfoWindow (notesInfo.c), Load (file.c), DeleteNote and AddandDisplayNewNote (note.c), ShowInfoWindow(notesInfo.c)
 */
void InitNotesInfo(instrument, deselect, tlFrame)
     Instrument *instrument;
     int deselect;
     TimeLineFramePtr tlFrame;
{
  Note 	*currentNote;
  char 	buf[100], numnotes[5];
  int 	count = 0, i;
  int 	oldlines, replace;

  if (instrument == tlFrame->noteInstrument && deselect)
    DeselectNote(tlFrame);
  if (xv_get(instrument->editInfo->EditInfoWindow, XV_SHOW) == TRUE) 
  {  
    sprintf(numnotes, "%d", instrument->numnotes);		    /* Update the display for the number of notes */
    xv_set(instrument->editInfo->NotesCountMessage,
	   PANEL_LABEL_STRING, numnotes, NULL);
    oldlines = xv_get(instrument->editInfo->NoteInfoList,	    /* Get the number of rows currently in the panel list */
		      PANEL_LIST_NROWS);
    replace = oldlines;
    if (instrument->numnotes <= oldlines) 
      replace = instrument->numnotes;
    currentNote = instrument->firstNote;
    xv_set(instrument->editInfo->NoteInfoList,			    /* Hide the panel list while updating is done */
	   XV_SHOW, FALSE, NULL);
    for (i=0; i < replace; i++)					    /* Replace the old strings by the new ones. */
    {
      sprintf(buf, " %4d.    %-25.22s%4d:%02d   %4d:%02d   %4d:%02d",
	      count+1, currentNote->ms->selection->label, 
	      currentNote->startMin, currentNote->startSec, 
	      currentNote->endMin, currentNote->endSec,
	      currentNote->durationMin, currentNote->durationSec);
      xv_set (instrument->editInfo->NoteInfoList,
	      PANEL_LIST_STRING, i, buf,
	      PANEL_LIST_FONT, i, font,
	      NULL);
      count++;
      currentNote = currentNote->next;
    }
    if (instrument->numnotes > oldlines) 
    {
      for (i=oldlines; i < instrument->numnotes; i++)		    /* Insert the additional new strings */
      {
	sprintf(buf, " %4d.    %-25.22s%4d:%02d   %4d:%02d   %4d:%02d",
		count+1, currentNote->ms->selection->label,
		currentNote->startMin, currentNote->startSec, 
		currentNote->endMin, currentNote->endSec,
		currentNote->durationMin, currentNote->durationSec);
	xv_set (instrument->editInfo->NoteInfoList,
		PANEL_LIST_INSERT, i,
		PANEL_LIST_STRING, i, buf,
		PANEL_LIST_FONT, i, font,
		NULL);
	count++;
	currentNote = currentNote->next;
      }
    }
    else for (i=instrument->numnotes; i < oldlines; i++)	    /* Delete the excess old strings */
      xv_set(instrument->editInfo->NoteInfoList,
	     PANEL_LIST_DELETE, instrument->numnotes,
	     NULL);
    if (tlFrame->noteInstrument != NULL)
      if (tlFrame->noteInstrument->infoNote != NULL)		    /* Deselect the note on the panel list */
	xv_set(tlFrame->noteInstrument->editInfo->NoteInfoList,
	       PANEL_LIST_SELECT, tlFrame->noteInstrument->selectedInfoNote,
	       TRUE, NULL);
    xv_set(instrument->editInfo->NoteInfoList,			    /* Show the panel list again */
	   XV_SHOW, TRUE, NULL);
  }
}

/*
 * Function that will return a pointer to the appropriate note node given its relative position in the notes list of a given instrument node.
 * Called by NotesInfoListNotify (notesInfo.c)
 */
Note *FindNote(instrument, relativePosition)
     Instrument *instrument;
     int 	relativePosition;
{
  int 		i;
  Note *currentNote;
  
  currentNote = instrument->firstNote;					    /* To find the appropriate instrument node */
  for (i=0; i < relativePosition && currentNote != NULL; i++)
    currentNote = currentNote->next;
  return currentNote;
  
}
  
/* 
 * This function will load in the document name and the selection structure data onto the Instrument Info pop-up window for a given note.
 * It will also update the message field indicating which note was selected.
 * The note is then redrawn so that it appears selected (Sunken).
 * Called by NotesInfoListNotify (notesInfo.c), AddandDisplayNewNote (note.c)
 */
void SelectNoteInfo(instrument, tlFrame)
     Instrument *instrument;
     TimeLineFramePtr tlFrame;
{
  char start[4], end[4], selected[4];
  long templastX;
  
  tlFrame->noteInstrument = instrument;
  sprintf(start, "%d", instrument->infoNote->ms->selection->start); /* Set all the display values in the info pop-up window */
  sprintf(end, "%d", instrument->infoNote->ms->selection->end);
  sprintf(selected, "%d", instrument->selectedInfoNote + 1);
  xv_set (instrument->editInfo->DocNameText, PANEL_VALUE, instrument->infoNote->ms->documentName, NULL);
  xv_set (instrument->editInfo->SelectedNoteValue, PANEL_LABEL_STRING, selected, NULL);
  xv_set (instrument->editInfo->SelectionStartValue, PANEL_LABEL_STRING, start, NULL);
  xv_set (instrument->editInfo->SelectionEndValue, PANEL_LABEL_STRING, end, NULL);

  templastX = tlFrame->lastX;						    /* Draw note so that it is sunken (appearing selected) */
  DrawPlaybackHead(-1, tlFrame);
  DrawNote(instrument->infoNote->start, instrument->cableStart - NoteHeight/2,
	   instrument->infoNote->ms->duration * 5, Sunken, tlFrame);
  DrawPlaybackHead(templastX, tlFrame);
  tlFrame->startX = tlFrame->startnoteX = tlFrame->noteInstrument->infoNote->start;
  tlFrame->endX = tlFrame->endnoteX = tlFrame->noteInstrument->infoNote->end;
  tlFrame->startY = tlFrame->startnoteY = tlFrame->noteInstrument->relativePosition * (IconHeight + IconGap);
  tlFrame->endY = tlFrame->endnoteY = (tlFrame->noteInstrument->relativePosition + 1) * (IconHeight + IconGap);
  tlFrame->noteSelected = 1;
  tlFrame->areaSelected = 0;
  SetStartEndRegion(tlFrame, 0, 0);
}

/* 
 * This function will deselect a selected info note and clear the infoNote pointer and relative position variables in the instrument note 
 * that owns the note. It will redraw the note so that it appears 'Raised'.
 * Called by DeselectNote (notesInfo.c)
 */
void DeselectNoteInfo(instrument, tlFrame)
     Instrument *instrument;
     TimeLineFramePtr tlFrame;
{
  long templastX;
  
  if (instrument->infoNote != NULL)				    /* Redraw a previously selected note so that it now appears Raised again */
  {
    templastX = tlFrame->lastX;
    DrawPlaybackHead(-1, tlFrame);
    DrawNote(instrument->infoNote->start, instrument->cableStart - NoteHeight/2,
	     instrument->infoNote->ms->duration * 5, Raised, tlFrame);
    DrawPlaybackHead(templastX, tlFrame);
  }
  instrument->infoNote = NULL;					    /* Reset the pointer to the selected info note */
  instrument->selectedInfoNote = -1;
  xv_set (instrument->editInfo->DocNameText, PANEL_VALUE, "", NULL); /* Clear the info fields as no note is selected */  
  xv_set (instrument->editInfo->SelectedNoteValue, PANEL_LABEL_STRING, "none", NULL);
  xv_set (instrument->editInfo->SelectionStartValue, PANEL_LABEL_STRING, "", NULL);
  xv_set (instrument->editInfo->SelectionEndValue, PANEL_LABEL_STRING, "", NULL);
}
    
/*
 * This function will deselect the currently selected note from both the canvas display and the notes info panel list.  It first deselects the note
 * on the panel list by setting the PANEL_LIST_SELECT attribute to FALSE, and then calls DeselectNoteInfo to deselect the note on the canvas and also
 * update the information on the instrument info pop-up window.
 * Called by NoteInfoListNotify (notesInfo.c), AddandDisplayNewNote (note.c)
 */
void DeselectNote(tlFrame)
     TimeLineFramePtr tlFrame;
{
  if (tlFrame->noteInstrument != NULL)
  {
    if (tlFrame->noteInstrument->infoNote != NULL)		    /* Deselect the note on the panel list */
      xv_set(tlFrame->noteInstrument->editInfo->NoteInfoList,
	     PANEL_LIST_SELECT, tlFrame->noteInstrument->selectedInfoNote,
	     FALSE, NULL);
    DeselectNoteInfo (tlFrame->noteInstrument, tlFrame);	    /* Deselect the note on the canvas display */
    tlFrame->noteInstrument = NULL;
    tlFrame->noteSelected = 0;
  }
}

/*
 * Notify callback function for `NoteInfoList'.
 * First, get which instrument is called by doing a xv_get on the PANEL_CLIENT_DATA.
 * On a DESELECT operation, the function will call the DeselectNoteInfo function (see above) to deselect a note.
 * If the instrument is the same as the previously selected instrument, the selected instrument pointer is set to NULL since it has been deselected.
 * On a SELECT operation, the function will first check to see if the previously selected instrument is the same one.  If not, it needs to be
 * deselected first.  The function will then scan the panel list entry that is selected to determine which entry has been selected.
 * It will then get the pointer to the appropriate note in the instrument notes list (since it now knows the relative position of the note by its
 * position in the panel list.
 * A check is made to determine if the start of the note currently falls in the canvas window.  If not, the canvas is remapped to the correct portion of
 * the TimeLine so that the selected note will be visible somewhere in the middle of the canvas window.
 * The SelectNoteInfo procedure is then called to perform the necessary selection procedures (see above).
 */
int NotesInfoListNotify(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  Instrument 	*instrument;
  int 		selection;
  TimeLineFramePtr tlFrame;
  EditInfo_EditInfoWindow_objects	*ip = (EditInfo_EditInfoWindow_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->EditInfoWindow, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  instrument = (Instrument*) FindInstrument((int)xv_get(item, PANEL_CLIENT_DATA), tlFrame); /* Find the instrument for this panel list */
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    DeselectNote(tlFrame);
    break;
   case PANEL_LIST_OP_SELECT:
    if (tlFrame->noteInstrument != instrument)			    /* If the previously selected instrument is a different one, deselect it. */
      DeselectNote(tlFrame);					    /* If it was the same one, it would have already been deselected in the */
								    /* PANEL_LIST_OP_DESELECT routine */
    tlFrame->noteInstrument = instrument;			    /* Set the selected instrument to this newly selected one */
    sscanf(string, "%d", &selection);				    /* Get which note is selected */
    instrument->selectedInfoNote = selection - 1;
    instrument->infoNote = (Note *) FindNote(instrument, instrument->selectedInfoNote);
    ScrollToFirstQuarter(tlFrame, instrument->infoNote->start, 0);
    if (tlFrame->areaSelected)					    /* Clear the previously selected area */
      XFillRectangle(tlFrame->dpyDraw, tlFrame->xidDraw, tlFrame->gcLine, (tlFrame->startX / tlFrame->zoomLevel) - tlFrame->canvasStart,
		     tlFrame->startY, (tlFrame->endX - tlFrame->startX) / tlFrame->zoomLevel,
		     tlFrame->endY - tlFrame->startY);
    SelectNoteInfo (instrument, tlFrame);
    break;
   case PANEL_LIST_OP_VALIDATE:
    break;
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}

