/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/blankTime.c,v 1.0 91/09/30 16:49:23 chua Exp Locker: drapeau $ */
/* $Log:	blankTime.c,v $
 * Revision 1.0  91/09/30  16:49:23  chua
 * Update to version 1.0
 * 
 * Revision 0.67  91/09/23  17:06:30  chua
 * Insert Blank Time has been moved from the Edit menu button to the Options
 * menu button
 * 
 * Revision 0.66  91/09/19  17:28:38  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.65  91/09/04  15:05:02  chua
 * Replaced the notice_prompt call by a call to AlertMessage.
 * 
 * Revision 0.64  91/07/30  17:32:20  chua
 * When blank time is inserted, set the change flag to 1.
 * 
 * Revision 0.63  91/07/24  17:45:54  chua
 * In the InsertBlankTime procedure, add a check to see if a note or area has been selected.
 * If so, replace this selected area by the amount of blank time specified.
 * 
 * Revision 0.62  91/07/22  15:15:02  chua
 * Changed the name of the pointer to the blank time popup window to InsertBlankTimePopup,
 * instead of a longer name previously.
 * 
 * Revision 0.61  91/07/17  16:30:13  chua
 * In the CancelBlankTime routine, just use the ip pointer to find the pointer to the
 * popup window, instead of using tlFrame etc.
 * 
 * Revision 0.60  91/07/17  10:14:03  chua
 * In line 169, there is a slight change, as the lastX variable now represents time
 * stored in the lowest zoom level (same as start), so there is no need to multiply
 * by the zoom level.
 * 
 * Revision 0.59  91/07/09  18:23:37  chua
 * Made changes to the startX, endX variables so that they now store the position at the
 * largest zoom level (zoom level = 1).   Thus, the appropriate multiplication or
 * division by the zoom level has to be made when these variables are used.  This will
 * include lastX (position of the playback head) as well.
 * 
 * Revision 0.58  91/07/09  16:52:46  chua
 * In the InsertBlankTime routine, check if the playback head has been positioned on the
 * canvas.  If not, print a message and exit the procedure.
 * 
 * Revision 0.57  91/06/24  16:52:43  chua
 * In the InsertBlankTime procedure (line 136), as the constant HalfSecondScale is no longer
 * used, it is replaced by its value, 5.
 * 
 * Revision 0.56  91/06/04  17:37:07  chua
 * Added the copyright comments in the beginning of the file.
 * 
 * Revision 0.55  91/06/04  10:41:25  chua
 * Added a call to UpdateHeader whenever there is a change in the status of
 * the current document (the change flag set to 1).
 * 
 * Revision 0.54  91/06/03  11:09:42  chua
 * Make changes to accomodate multiple documents.  This involves identifying
 * which window the blank time popup window belongs to.
 * 
 * Revision 0.53  91/05/30  12:07:07  chua
 * Added an extra parameter in the call to InitNotesInfo.  The second parameter,
 * deselect, indicates if the currently selected note is to be deselected.
 * 
 * Revision 0.52  91/05/29  14:28:37  chua
 * Delete the ClearNoteInfoList function call in line 129, since the InitNotesInfo has been 
 * rewritten such that the ClearNoteInfoList call is not necessary.
 * 
 * Revision 0.51  91/05/28  12:09:27  chua
 * *** empty log message ***
 * 
 * Revision 0.50  91/05/24  16:35:42  chua
 * *** empty log message ***
 * 
 * Revision 0.49  91/05/23  17:33:18  chua
 * This file contains the functions to deal with inserting blank time onto either a selected track or
 * all tracks.  The functions are:
 * 
 * InsertBlankTimeHandler - displays the Insert Blank Time popup window.
 * InsertBlankTime - Button notify procedure which performs the insertion of blank time onto either
 * 		     a selected track or all tracks, depending on what the user has chosen.
 * CancelBlankTime - Button notify procedure/Done procedure for the Insert Blank Time popup window.
 * 		     Closes the popup window when the user no longer requires it.  Called by eitther
 * 		     pressing the Cancel button or releasing the pushpin.
 *  */

static char blankTimercsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/blankTime.c,v 1.0 91/09/30 16:49:23 chua Exp Locker: drapeau $";

#include "main.h"

/*
 * Menu handler for `OptionsMenu (Insert Blank Time ...)'.
 * This function will pop up the Insert Blank Time pop-up window, so that the user can choose how much blank time and where to insert the
 * blank time.
 */
Menu_item InsertBlankTimeHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(tlFrame->InsertBlankTimePopup->InsertBlankTimePopup,
	   FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(tlFrame->InsertBlankTimePopup->InsertBlankTimePopup,
	   XV_SHOW, TRUE, NULL);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Notify callback function for `InsertBlankTimeButton'.
 * This function will insert blank time onto either a selected track or all tracks, depending on what the user chose.
 * First, it will check if an area or note has been selected. If so, it will delete the selected area/note and move the playback head to the start of
 * the deleted area.
 * It will then determine whether it is a single track or all track insert.  It then checks to see that the playback head has been positioned
 * at some place and that if a single track is chosen, make sure an application has been selected.
 * Next, the blank time to be inserted is calculated from the text fields in the pop-up window.
 * The function then loops through all the instruments where the blank time is to be inserted to find out at which spot in the note list the blank time
 * is to be inserted.
 * Insertion is then performed.  This is done basically by updating the times of all notes following the insertion point.  If the insertion point falls in
 * the middle of a note, the blank space is inserted after the note.
 */
void InsertBlankTime(item, event)
     Panel_item	item;
     Event		*event;
{
  int 		found, choice, i;
  int 		startApp, endApp;
  Instrument 	*instrument;
  long 		blankTime;
  int 		inserted;
  Note 		*prevNote, *currentNote, *nextNote;
  TimeLineFramePtr tlFrame;
  BlankTime_InsertBlankTimePopup_objects	*ip = (BlankTime_InsertBlankTimePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->InsertBlankTimePopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->areaSelected || tlFrame->noteSelected)		    /* Check if either a note or an area is selected */
  {
    DeleteNotesFromList(0, tlFrame);
    DrawPlaybackHead(tlFrame->startX, tlFrame);
  }
  if (tlFrame->lastX < 0)					    /* Check if the playback head is on the canvas */
  {
    AlertMessage(tlFrame, "You must position the insertion point (playback head)",
		 "on the canvas first.", NULL);
    return;
  }
  startApp = tlFrame->chosenApp;
  endApp = tlFrame->chosenApp + 1;
  choice = xv_get(tlFrame->InsertBlankTimePopup->InsertBlankTimeChoice, PANEL_VALUE);
  if (choice == 1)						    /* Insert for all tracks */
  {
    startApp = 0;
    endApp = tlFrame->numberOfApps;
  }
  inserted = 0;
  if (tlFrame->lastX >= 0 && startApp >= 0)			    /* Check that the playback head is positioned someplace */
  {
    blankTime = xv_get(tlFrame->InsertBlankTimePopup->MinTimeText, PANEL_VALUE) * 60 + /* Calculate the blank time in seconds */
      xv_get(tlFrame->InsertBlankTimePopup->SecTimeText, PANEL_VALUE);
    blankTime = blankTime * PixelsPerSecond;			    /* Set the blank time to 1/10 second scale (the scale for zoomLevel = 1) */
    instrument = (Instrument *) FindInstrument(startApp, tlFrame);
    if (instrument == NULL) 
      return;
    for (i=startApp; i < endApp; i++) 
    {
      if (instrument->firstNote != NULL)			    /* Check that there are notes in this instrument.  Else no point adding blank time */
      {
	currentNote = instrument->firstNote;			    /* Go through the instrument note list to determine at which point in the */
	prevNote = currentNote;					    /* note list should the insertion of blank time should occur */
	found = 0;
	while (currentNote != NULL && !found) 
	{
	  if (currentNote->start >= tlFrame->lastX)
	    found = 1;
	  else 
	  {
	    prevNote = currentNote;
	    currentNote = currentNote->next;
	  }
	}
	if (currentNote == instrument->firstNote)		    /* Paste before all the notes in the current note list */
	  nextNote = instrument->firstNote;
	else 
	  nextNote = prevNote->next;				    /* duplicated clipboard note list */
	while (nextNote != NULL)				    /* Update the start and end positions of the notes following the selected region */
	{
	  inserted = 1;
	  nextNote->start += blankTime;
	  nextNote->end += blankTime;
	  CalculateNoteTime(nextNote);
	  nextNote = nextNote->next;
	}
      }
      InitNotesInfo(instrument, 1, tlFrame);			    /* Update the info pop-up window */
      instrument = instrument->next;
    }
    if (inserted == 1)						    /* Set change flag on only if the document is affected by the inserted blank time */
    {
      tlFrame->change = 1;					    /* Set the change flag on */
      UpdateHeader(tlFrame, 1);
    }
    DrawCanvasRepaintHandler(tlFrame->TimeLine_window->DrawCanvas, tlFrame->paintWinDraw, 
			     tlFrame->dpyDraw, tlFrame->xidDraw, NULL);
  }
}

/*
 * Notify callback function for `CancelBlankTimeButton'.
 * This function hides the insert blank time popup window.
 */
void CancelBlankTime(item, event)
     Panel_item	item;
     Event		*event;
{
  BlankTime_InsertBlankTimePopup_objects	*ip = (BlankTime_InsertBlankTimePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  xv_set(ip->InsertBlankTimePopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(ip->InsertBlankTimePopup, XV_SHOW, FALSE, NULL);
}
