/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lcfft.c
 >>>>
 >>>>      Program Name: cfft
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:34:51 1991 
 >>>>
 >>>>          Routines: lcfft - the library call for cfft
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
struct complex
  {
    double r;
    double i;
  } ;
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lcfft - library call for cfft
*
* Purpose:
*    
*    Computes the two dimensional Fast Fourier Transform (FFT)
*    
*    
* Input:
*    
*    image1         pointer to xvimage structure of type FLOAT or COM-
*                   PLEX.
*    
*    image2         pointer  to  xvimage  structure  of   type   FLOAT
*                   (optional).
*    
*    dir            integer  specifying  the  direction  of  the   FFT
*                   transform. 0 is forward, 1 is inverse.
*    
*    
* Output:
*    
*    c_flg          is an integer requesting  that  a  complex  output
*                   image  be  produiced.   Non-zero means produce the
*                   image.
*    
*    o_image1       is a pointer to the image structure  pointer  that
*                   will  receive  the  complex output image, if it is
*                   desired (see above).
*    
*    r_flg          is an integer requesting that a real output  image
*                   be produced.  Non-zero means produce the image.
*    
*    o_image2       is a pointer to the image structure  pointer  that
*                   will  receive  the  real  output  image,  if it is
*                   desired (see above).
*    
*    i_flg          is an integer requesting that a  imaginary  output
*                   image  be  produced.   Non-zero  means produce the
*                   image.
*    
*    o_image3       is a pointer to the image structure  pointer  that
*                   will  receive the imaginary output image, if it is
*                   desired (see above).
*    
*    Return Value:  1 on success, 0 on failure.
*    
*    
*
* Written By: Scott Wilson
*    
*    1/16/91  Modified code to accept either a FLOAT or COMPLEX  image
*    and  output  a  COMPLEX  image,  or the REAL, or IMAGINARY parts.
*    Routine was also modified to  perform  FORWARD  or  INVERSE  FFT.
*    Added call to powtwo() to check for proper image size. C.Gage
*    
*    8-Mar-91 Scott Wilson - Pixel sizes were  not  being  transferred
*    from input to output.
*    
*    13-Mar-91 Scott Wilson - Converted to Krukar's inline C FFTs
*    
*    
****************************************************************/


/* -library_def */
int
lcfft(image1, image2, c_flg, o_image1, r_flg, o_image2, i_flg, o_image3, dir)
struct xvimage *image1, *image2, **o_image1, **o_image2, **o_image3;
int c_flg, r_flg, i_flg;
int dir;
/* -library_def_end */

/* -library_code */
{
    struct xvimage *c_image, *r_image, *i_image, *createimage();

    int i,j,k,nr,nc,ic,ir;      /* Indexes and sizes of rows and columns */

    struct complex *p,*z,*x,*t;

    float *fptr,                /* pointer to image1 data */
          *imptr;               /* pointer to image2 (imaginary) data */

    int m,m1;                    /* Stuff for Nyquist modulation */

    float         *cptr,        /* pointer to complex image data */
                  *rptr,        /* pointer to real image data */
                  *iptr;        /* pointer to imaginary image data */

    char  *malloc(),
          *program = "lcfft";

    nr = image1->col_size;            /* Number of rows */
    nc = image1->row_size;            /* Number of columns */

    /* Check to make sure input image is of correct type */
    if ( (!propertype(program, image1, VFF_TYP_COMPLEX, FALSE)) && (!propertype(program, image1, VFF_TYP_FLOAT, FALSE)) )
    {
      (void) fprintf(stderr, "lcfft: Input image must be of data type COMPLEX or FLOAT.\n");
      return(0);
    }

    /* Check for proper map enable */
    if (!proper_map_enable(program,image1,VFF_MAP_OPTIONAL,FALSE))
    {
      (void) fprintf(stderr, "lcfft: Can only work on map scheme NONE\n");
      return(0);
    }

    /* Check to see if optional second input image matches first input image */
    if (image2 != NULL)
    {
      if ( (!match_num_images(program, image1, image2, FALSE)) && (!match_num_bands(program, image1, image2, FALSE)) && (!match_map_enable(program, image1, image2, FALSE)) )
      {
        (void) fprintf(stderr,"lcfft: Both input images must be of same dimension.\n");
        return(0);
      }
      if ( (!matchsize(program, image1, image2, FALSE)) && (!matchtype(program, image1, image2, FALSE)) )
      {
        (void) fprintf(stderr,"lcfft: Both input images must be of same size and type.\n");
        return(0);
      }
    }

    /* Make sure size is legal for an FFT */
    if ( (powtwo(nc) == 0) || (powtwo(nr) == 0) )
    {
      (void)fprintf(stderr,"lcfft: Input image is %d rows by %d cols.\n", nr, nc);
      (void)fprintf(stderr,"Image size MUST be a power of 2!\n ");
      return(0);
    }
    if (nr > 4096 || nc > 4096)
    {
      (void)fprintf(stderr,"lcfft: Max of 4096 rows or columns for this FFT\n");
      return(0);
    }

    /* Get space for the intermediate complex arrays */
    x = (struct complex *)malloc(nc*nr*sizeof(struct complex));
    z = (struct complex *)malloc(nc*sizeof(struct complex));

    if (x == NULL || z == NULL)
      {
        fprintf(stderr,"lcfft: Could not allocate enough memory!\n");
        return(0);
      }

    /****   Create the COMPLEX output image ****/
  if (c_flg)
  {
    c_image = createimage(nr,                  /* number of rows in image */
                          nc,                  /* number of cols in image */
                          VFF_TYP_COMPLEX,     /* data storage type */
                          1,                   /* number of images */
                          1,                   /* number of data bands */
                          "created by cfft", /* comment */
                          0,                   /* map row size */
                          0,                   /* map col size */
                          VFF_MS_NONE,         /* map scheme */
                          VFF_MAPTYP_NONE,
                          VFF_LOC_IMPLICIT,
                          0);
    c_image->pixsizx = image1->pixsizx;
    c_image->pixsizy = image1->pixsizy;

    if (c_image == NULL)
      {
        (void) fprintf(stderr,"lcfft: Unable to allocate enough memory for output image!\n");
        return(0);
      }

                               /* assign pointer to image data area */
    cptr = (float *) c_image->imagedata;
  }

    /**** Create the FLOAT output image corresponding to REAL part ****/
  if (r_flg)
  {
    r_image = createimage(nr,                  /* number of rows in image */
                          nc,                  /* number of cols in image */
                          VFF_TYP_FLOAT,       /* data storage type */
                          1,                   /* number of images */
                          1,                   /* number of data bands */
                          "created by cfft", /* comment */
                          0,                   /* map row size */
                          0,                   /* map col size */
                          VFF_MS_NONE,         /* map scheme */
                          VFF_MAPTYP_NONE,
                          VFF_LOC_IMPLICIT,
                          0);
    r_image->pixsizx = image1->pixsizx;
    r_image->pixsizy = image1->pixsizy;

    if (r_image == NULL)
      {
        (void) fprintf(stderr,"lcfft: Unable to allocate enough memory for output image!\n");
        return(0);
      }

                               /* assign pointer to image data area */
    rptr = (float *) r_image->imagedata;
  }

    /**** Create the FLOAT output image corresponding to IMAG part ****/
  if (i_flg)
  {
    i_image = createimage(nr,                  /* number of rows in image */
                          nc,                  /* number of cols in image */
                          VFF_TYP_FLOAT,       /* data storage type */
                          1,                   /* number of images */
                          1,                   /* number of data bands */
                          "created by cfft", /* comment */
                          0,                   /* map row size */
                          0,                   /* map col size */
                          VFF_MS_NONE,         /* map scheme */
                          VFF_MAPTYP_NONE,
                          VFF_LOC_IMPLICIT,
                          0);
    i_image->pixsizx = image1->pixsizx;
    i_image->pixsizy = image1->pixsizy;

    if (i_image == NULL)
      {
        (void) fprintf(stderr,"lcfft: Unable to allocate enough memory for output image!\n");
        return(0);
      }

                               /* assign pointer to image data area */
    iptr = (float *) i_image->imagedata;
  }

             /* assign pointer to input image data */
  if (image2 == NULL)          /* optional second input image not present */
  {
    fptr = (float *)(image1->imagedata);
  }
  else if (image2 != NULL)
  {
    fptr = (float *)(image1->imagedata);
    imptr = (float *)(image2->imagedata);
  }

 if (!dir)   /*<<<<<<<<    Forward FFT, dir =  0 (FALSE)    >>>>>>>>>>>*/
 {
    m1 = 1;                      /* Initialize Nyquist modulator */
    for(i=0; i<nr; i++)          /* Do Nyquist modulate and FFT on each row */
      {
        m = m1; m1 = -m1;
        if ( (image1->data_storage_type == VFF_TYP_FLOAT) && (image2 == NULL) )
        {
          for (j=0; j<nc; j++)
          {
            z[j].r = m*(*fptr++);
            z[j].i = 0.0;         /* set imaginary part to 0 if a FLOAT image */
            m = -m;
          }
        }
        else if ( (image1->data_storage_type == VFF_TYP_FLOAT) && (image2 != NULL) )
        {
          for (j=0; j<nc; j++)
          {
             z[j].r = m*(*fptr++);   /* get real part from first input image */
             z[j].i = m*(*imptr++);  /* get imaginary part from second input image */
             m = -m;
          }
        }
        else
        {            
          for (j=0; j<nc; j++)
          {
             z[j].r = m*(*fptr++);   /* get real and imag parts from complex image */
             z[j].i = m*(*fptr++);
             m = -m;
          }
        }
        fft(z,nc);

        p = x+i*nc;                 /* Store into (complex) rows) */
        for(j=0; j<nc; j++) *p++ = z[j];
      }

    free(z);               /* Give back temp row space */
    free(image1);            /* Give back input image(s) */
    if (image2 != NULL)
       free(image2);

    /* Get space for temporary complex row */
    t = (struct complex *)malloc(nr*sizeof(struct complex));
    if (t == NULL)
      {
        fprintf(stderr,"lcfft: Couldn't allocate enough memory!\n");
        return(0);
      }

    for(i=0; i<nc; i++)           /* Do FFT for the columns */
      {
        /* Load a column into temporary row */
        p = x+i;
        for (j=0; j<nr; j++) 
          {
            t[j] = *p;
            p += nc;
          }

        fft(t,nr);

        /* Put back into the same column */
        p = x+i;
        for (j=0; j<nr; j++)
          {
            *p = t[j];
            p += nc;
          }
      }

    /* Load the data in corresponding output images */
    p = x;

    if (c_flg)           /* output COMPLEX image */
    {
      for (i=0; i<nr; i++)
      {
        for (j=0; j<nc; j++)
        {
          *cptr++ = (*p).r;
          *cptr++ = (*p++).i;
        }
      }
    }

    if (r_flg)      /* output FLOAT image corresponding to REAL part */
    {
      for (i=0; i<nr; i++)
        for (j=0; j<nc; j++)
          *rptr++ = (*p++).r;
    }

    if (i_flg)      /* output FLOAT image corresponding to IMAG part */
    {
      for (i=0; i<nr; i++)
        for (j=0; j<nc; j++)
          *iptr++ = (*p++).i;
    }
    free(x);
    free(t);
 }
 else if (dir)   /*<<<<<<<<<<<<    Inverse FFT, dir = TRUE    >>>>>>>>>>*/
 {
    /* Load input data from corresponding input images */
    p = x;
    for (i=0; i<nr; i++)
    {
      if ( (image1->data_storage_type == VFF_TYP_FLOAT) && (image2 == NULL) )
      {
        for (j=0; j<nc; j++)
        {
          (*p).r = *fptr++;
          (*p++).i = 0.0;        /* set imag part to zero */
        }
      }
      else if ( (image1->data_storage_type == VFF_TYP_FLOAT) && (image2 != NULL) )
      {
        for (j=0; j<nc; j++)
        {
          (*p).r = *fptr++;
          (*p++).i = *imptr++;        /* get imag part from second input image */
        }
      }
      else            /* input image is COMPLEX */
      {
        for (j=0; j<nc; j++)
        {
          (*p).r = *fptr++;
          (*p++).i = *fptr++;
        }
      }
    }

    /* Give back input image(s) and get space for the temporaries */
    free(image1);
    if (image2 != NULL)
       free(image2);
    t = (struct complex *)malloc(nr*sizeof(struct complex));

    if (t == NULL)
      {
        fprintf(stderr,"lcft: Couldn't allocate enough memory!\n");
        return(0);
      }

    for(i=0; i<nc; i++)    /* Do IFFT for the columns */
    {
        /* Load a column into temporary row */
        p = x+i;
        for (j=0; j<nr; j++)
          {
            t[j] = *p;
            p += nc;
          } 
        ifft(t,nr);

        /* Put back into the same column */
        p = x+i;
        for(j=0; j<nr; j++)
          {
            *p = t[j];
            p += nc;
          }
    }

    free(t);               /* free temporary arrays */

    m1 = 1;
    for(i=0; i<nr; i++)          /* Do FFT for each row in image */
    {
        p = x+i*nc;             /* Store into (complex) rows) */
        for (j=0; j<nc; j++) z[j] = *p++;
        ifft(z,nc);

        /* Store resulting image */
        m = m1; m1 = -m1;

        if (c_flg)              /* output COMPLEX image */
        {
          for (j=0; j<nc; j++)
          {
            *cptr++ = m*z[j].r;
            *cptr++ = m*z[j].i;
            m = -m;
          }
        }

        if (r_flg)              /* output FLOAT image for REAL part */
        {
          for (j=0; j<nc; j++)
          {
            *rptr++ = m*z[j].r;
            m = -m;
          }
        }

        if (i_flg)              /* output FLOAT image for IMAG part */
        {
          for (j=0; j<nc; j++)
          {
            *iptr++ = m*z[j].i;
            m = -m;
          }
        }
    }
    free(x);               /* free real and imag data arrays */
    free(t);
 }

    /* Assign data to new image pointers */
   if (c_flg) *o_image1 = c_image;

   if (r_flg) *o_image2 = r_image;

   if (i_flg) *o_image3 = i_image;

    return(1);

}  /*** End of main ***/
/* -library_code_end */
