/* copyright 1988,1989,1990 Phil Andrews, Pittsburgh Supercomputing Center */
/* all rights reserved */
/* module to take care of clear text command processing */
#include <stdio.h>
#include "defs.h"	/* type definitions */
#include "ccdefs.h"	/* clear text definitions */

/* global pointers that use storage from the main program */

static struct 	mf_d_struct 	*glbl1;	/* the class 1 elements */
static struct 	pic_d_struct 	*glbl2, *dflt2, *a2;	/* class 2 elements */
static struct 	control_struct	*glbl3, *dflt3,	*a3;	/* class 3 elements */
static struct 	attrib_struct	*glbl5, *dflt5, *a5;	/* class 5 elements */
/* g stands for global, d for default, and a for active, we normally use
   the a pointer, with an = glbln except in the middle of a metafile defaults
   replacement element, when an = dfltn */

/* now the device-specific function array pointers */
static int (**delim)();		/* delimiter functions */
static int (**mfdesc)();	/* metafile descriptor functions */
static int (**pdesc)();		/* page descriptor functions */
static int (**mfctrl)();	/* mf control functions */
static int (**gprim)();		/* graphical primitives */
static int (**attr)();		/* the attribute functions */
static int (**escfun)();	/* the escape functions */
static int (**extfun)();	/* the external functions */
static int (**ctrl)();		/* external controller functions */
/* the device info structure */
static struct info_struct *dev_info;

/* the command line options */
static struct one_opt *opt;	/* the command line options */

/* the font info for devices that can't have it all */
#define max_fonts 100
static struct font cfarray[max_fonts];

/* the external font functions */
static int (*font_check)() = NULL;
     static int (*font_text)() = NULL;
     
     static int list_cgm;
#define get_all (opt[(int) debug].set || list_cgm)
     extern double sin(), cos(), atan();
     extern void exit(), free();
     extern char *strcat(), *strcpy();
     
     
     /* basic stuff */
#define bytes_per_word sizeof(int)
#define byte_size 8	/* 8 bits to the byte */
#define float_size (sizeof(float))
     static char *version_str;	/* version string pointer */
#define max_digits 10
     static char digits[max_digits] = {'0', '1', '2', '3', '4', '5', '6', '7',
					 '8', '9'};
     /* macros to handle direct colours */
     
#define dcr(ival) dcind(0, ival, glbl1->c_v_extent)
#define dcg(ival) dcind(1, ival, glbl1->c_v_extent)
#define dcb(ival) dcind(2, ival, glbl1->c_v_extent)
     
     /* cgm specific functions */
#define e_size 2
     extern char *malloc();			/* for neatness */
#define intalloc (int *) malloc /* for convenience */
       static char *cc_str();	/* does the necessary translation */
     static float cc_real();	/* does the necessary translation */
     static double pxl_vdc;	/* conversion from VDC units to pixels */
     static double sx, sy;	/* allow individual scaling */
     static double cosr;	/* value of cos theta */
     static double sinr;	/* value of sin theta */
     
     /* error macro, mainly to keep lint happy */
#define burp (void) fprintf
       /* and a listing macro */
#define may_list if (list_cgm) (void) fprintf
       /* now my globals */
       static int pages_done = 0;
     static char *g_in_name;	/* the full input file_name */
     static char pic_name[max_str];		/* picture name */
     
     /* stuff to take care of individual pages */
     static int in_page_no = 0;		/* page no in the metafile */
     static int skipping = 0;		/* are we skipping this page ? */
     /* set up a couple of work arrays for polygons, etc. */
#define wk_ar_size 2024
     static int x_wk_ar[wk_ar_size], y_wk_ar[wk_ar_size];
     
     static float xp0;	/* internal xoffset in pixels */
     static float yp0;	/* internal yoffset in pixels */
     
     static int xoffset, yoffset;	/* external offsets in pixels */
     static int xsize, ysize;	/* requestd page size in pixels */
     
/* handle rotations, clipping, scaling here, use macros */
extern int uclip(); /* in utils.c */
#define UCLIP(xin, yin) \
  ((int) a3->clip_ind && !(dev_info->capability & can_clip)) ? \
   uclip(&xin, &yin, a3->clip_rect.i) : 0
#define NEWX(xin, yin) ((int) (sx * xp0 + (sx * cosr * xin - sy * sinr * yin)))
#define newx(xin, yin) (UCLIP(xin, yin), NEWX(xin, yin))

#define NEWY(xin, yin) ((int) (sy * yp0 + (sy * cosr * yin + sx * sinr * xin)))
#define newy(xin, yin) (UCLIP(xin, yin), NEWY(xin, yin))
     
     
     /* some local I/O */
     /* return the next integer */
     static int cc_int(in_ptr)
     char **in_ptr;
{
  char *my_ptr;
  int ret;
  
  my_ptr = *in_ptr;
  /* simple implementation first */
  if (!my_scan_int(in_ptr, &ret)) {
    fprintf(stderr, "couldn't get an integer out of %s\n", my_ptr);
    /* terminate this command */
    *my_ptr = term_char;
    return(0);
  }
  while ((**in_ptr) && (**in_ptr == ' ')) ++*in_ptr;
  return(ret);
}
/* return the next float */
static float cc_real(in_ptr)
     char **in_ptr;
{
  char *my_ptr;
  float ret;
  
  my_ptr = *in_ptr;
  /* skip spaces */
  while ((**in_ptr == ' ') && (**in_ptr)) ++*in_ptr;
  if (!my_scan_float(in_ptr, &ret)) {
    fprintf(stderr, "couldn't get a real out of %s\n", my_ptr);
    return(0);
  }
  while ((**in_ptr) && (**in_ptr == ' ')) ++*in_ptr;
  return(ret);
}
/* get the general vdc value */
static int cc_vdc(in_ptr)
     char **in_ptr;
{
  int xi;
  float xr;
  
  switch (glbl1->vdc_type) {
  case vdc_int:	xi = cc_int(in_ptr);
    return((int) (xi * pxl_vdc));
    
  case vdc_real:	xr = cc_real(in_ptr);
    return((int) (xr * pxl_vdc));
  }
  return(0);	/* trouble */
}

/* read in an integer point */
static int get_ipoint(in_ptr, xptr, yptr)
     char **in_ptr;
     int *xptr, *yptr;
{
  int use_paren;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if ((use_paren = (**in_ptr == '('))) ++*in_ptr;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  *xptr = cc_int(in_ptr);
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if (**in_ptr == ',') ++*in_ptr;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  *yptr = cc_int(in_ptr);
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if ((use_paren) && (**in_ptr == ')')) {
    ++*in_ptr;
    while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  }
  return(1);
}
/* read in a real point */
static int get_rpoint(in_ptr, xptr, yptr)
     char **in_ptr;
     float *xptr, *yptr;
{
  int use_paren;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if ((use_paren = (**in_ptr == '('))) ++*in_ptr;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  *xptr = cc_real(in_ptr);
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if (**in_ptr == ',') ++*in_ptr;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  *yptr = cc_real(in_ptr);
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  if ((use_paren) && (**in_ptr == ')')) {
    ++*in_ptr;
    while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  }
  return(1);
}
/* get a general point */
static int get_vpoint(in_ptr, xptr, yptr)
     char **in_ptr;
     int *xptr, *yptr;
{
  int xi, yi;
  float xr, yr;
  
  switch (glbl1->vdc_type) {
  case vdc_int:	get_ipoint(in_ptr, &xi, &yi);
    *xptr = xi * pxl_vdc;
    *yptr = yi * pxl_vdc;
    return(1);
  case vdc_real:	get_rpoint(in_ptr, &xr, &yr);
    *xptr = xr * pxl_vdc;
    *yptr = yr * pxl_vdc;
    return(1);
  }
  return(0);	/* trouble */
}

/* the setup procedure */
ccgm_setup(do_list, full_oname, this_version, new_info, new_opt,
	   gl1, gl2, df2, gl3, df3, gl5, df5,
	   pf0, pf1, pf2, pf3, pf4, pf5, pf6, pf7, pfct)
     char *full_oname, *this_version;
     int do_list;
     struct info_struct *new_info;
     struct one_opt *new_opt;
     struct 	mf_d_struct 	*gl1;	/* the class 1 elements */
     struct 	pic_d_struct 	*gl2, *df2;	/* the class 2 elements */
     struct 	control_struct	*gl3, *df3;	/* the class 3 elements */
     struct 	attrib_struct	*gl5, *df5;	/* the class 5 elements */
     int (**pf0)(), (**pf1)(), (**pf2)(), (**pf3)(), (**pf4)(), (**pf5)(), 
       (**pf6)(), (**pf7)(), (**pfct)();	/* the function pointer arrays */
{
  int p_len;	/* parameter length */
  
  /* store globals */
  list_cgm = do_list;
  g_in_name = full_oname;
  version_str = this_version;
  dev_info = new_info;
  opt = new_opt;
  glbl1 = gl1;
  glbl2 = gl2;
  dflt2 = df2;
  glbl3 = gl3;
  dflt3 = df3;
  glbl5 = gl5;
  dflt5 = df5;
  
  delim 	= pf0;
  mfdesc 	= pf1;
  pdesc	= pf2;
  mfctrl	= pf3;
  gprim	= pf4;
  attr	= pf5;
  escfun	= pf6;
  extfun	= pf7;
  ctrl 	= pfct;
  
  /* make the active pointers refer to the globals */
  a2 = glbl2;	a3 = glbl3;	a5 = glbl5;
  
  return(1);
}

/* parcel out the commands here */
do_clear(s_ptr, p_len, this_ad, out_class, out_element)
     unsigned char *s_ptr;
     int p_len;
     struct ad_struct *this_ad;	/* unused here */
     int *out_class, *out_element;
{
  unsigned char cmd_name[max_str];
  unsigned int class, element, p_flag;
  int i, j, k, done, cmd_len, df_call;
  /* are we replacing defaults ? */
#define no_defs(cl) if (a2 == dflt2)\
  {(void) fprintf(stderr, "illegal class cl while replacing defaults !\n");\
     return(2);}
  
  /* may be running over nulls */
  if (!p_len || !s_ptr || !*s_ptr) return(1);

  /* first find the command */
  i = 0;
  while ((i<p_len) && (s_ptr[i] == ' ')) ++i;
  j = 0;
  while ((i<p_len) && (j<(max_str-1)) && 
	 (((s_ptr[i] >= 'A') && (s_ptr[i] <= 'Z')) ||
	  ((s_ptr[i] >= '0') && (s_ptr[i] <= '9')))) 
    cmd_name[j++] = s_ptr[i++];
  cmd_name[j] = '\0';
  cmd_len = i;
  done = 0;
  for (i=0; (i<8) && (!done); ++i) {
    for (j=1; (j<cc_size[i]) && (!done); ++j) {
      k = 0;
      while ((CC_cptr[i][j][k]) && (cmd_name[k]) &&
	     (CC_cptr[i][j][k] == cmd_name[k]))
	++k;
      done = 	((!CC_cptr[i][j][k]) && (!cmd_name[k]) && (k));
    }
  }
  /* may be replacing defaults */
  if (!done) {
    if (cc_same(cmd_name, "BEGMFDEFAULTS")) {
      i = 2;
      j = 13;
      done = 1;
      df_call = 0;
    } else 
      if (cc_same(cmd_name, "ENDMFDEFAULTS")) {
	i = 2;
	j = 13;
	done = 1;
	df_call = 1;
      }
  }
  /* still didn't find it ? */
  if (!done) {
    fprintf(stderr, "couldn't find a match for [%s]\n", cmd_name);
    return(2);
  }
  *out_class = class = i - 1;
  *out_element = element = j - 1;
  /* are we starting a new page ? */
  if ( (class == 0) && ((enum cgmcls0) element == B_Pic) ) {
    ++in_page_no;	 /* new page */
    skipping = (!want_page(in_page_no, opt[(int) pages].val.str));
  }
  /* if skipping is on, turn it off at the end of the page */
  if ((skipping) && (class == 0) && ((enum cgmcls0) element == E_Pic)){
    skipping = 0;
    return(1);
  }
  
  /* now if we don't want this page, don't do anything until ended */
  if (skipping) return(1);
  /* note trouble if we were to get a B_Pic inside a metafile def rep */ 
  
  i = cmd_len;
  while ((i<p_len) && (s_ptr[i] == ' ')) ++i;
  /* now nudge forward */
  s_ptr += i;
  p_len -= i;
  
  switch (class) {
  case 0: no_defs(0); 	return(class0(element, p_len, s_ptr));
  case 1: 	 	return(class1(element, p_len, s_ptr, df_call));
  case 2: 		return(class2(element, s_ptr));
  case 3: 		return(class3(element, s_ptr));
  case 4: no_defs(4); 	return(class4(element, p_len, s_ptr));
  case 5: 		return(class5(element, p_len, s_ptr));
  case 6: no_defs(6); 	return(class6(element, s_ptr));
  case 7: no_defs(7);	return(class7(element, s_ptr));
  default: burp(stderr, "illegal class, cmd = (%d,%d,%d)\n",
		class, element, p_len);
  }
  
  return(get_all);	/* may or may not want to terminate on error */
}
/* class 0, the delimiter elements */
static class0(el, p_len, dat_ptr)
     char *dat_ptr;
     enum cgmcls0 el;
     int p_len;
{
  switch (el) {
  case No_Op: 	may_list(stderr, "\n"); return(1);
  case B_Mf:	return(f_b_mf	(dat_ptr, p_len, delim[(int) el], 
				 ctrl[(int) el]));
  case E_Mf:	return(f_e_mf	(delim[(int) el], ctrl[(int) el]));	
  case B_Pic:	return(f_b_p	(dat_ptr, p_len, delim[(int) el], 
				 ctrl[(int) el]));
  case B_Pic_Body: return(f_b_p_body	(delim[(int) el], ctrl[(int) el]));
  case E_Pic:	return(f_e_pic	(delim[(int) el], ctrl[(int) el]));
  default:	burp(stderr, "illegal class 0 command, %d\n", (int) el);
    
    
  }
  return(get_all);	/* may or may not want to terminate on error */
}
/* class 1, the metafile descriptor elements */
static class1(el, p_len, dat_ptr, df_call)
     char *dat_ptr;
     int p_len, df_call;
     enum cgmcls1 el;
{
  
  switch (el) {
  case  MfVersion:	return(rd_mf_version	(dat_ptr, p_len, 
						 mfdesc[(int)el]));
  case  MfDescrip:	return(rd_mf_descriptor	(dat_ptr, p_len, 
						 mfdesc[(int) el]));
  case  vdcType:		return(s_vdc_type	(dat_ptr, mfdesc[(int) el]));
  case  IntPrec:		return(s_int_prec	(dat_ptr, mfdesc[(int) el]));
  case  RealPrec:		return(s_real_prec	(dat_ptr, mfdesc[(int) el]));
  case  IndexPrec:	return(s_index_prec	(dat_ptr, mfdesc[(int) el]));
  case  ColPrec:		return(s_col_prec	(dat_ptr, mfdesc[(int) el]));
  case  CIndPrec:		return(s_cind_prec	(dat_ptr, mfdesc[(int) el]));
  case  MaxCInd:		return(s_mcind		(dat_ptr, mfdesc[(int) el]));
  case  CVExtent:		return(s_cvextent	(dat_ptr, mfdesc[(int) el]));
  case  MfElList:		return(rd_mf_list	(dat_ptr, mfdesc[(int) el]));
  case  MfDefRep:		return(s_mf_defs	(dat_ptr, p_len, 
							 mfdesc[(int) el], df_call));
  case  FontList:		return(do_font_list	(dat_ptr, p_len, 
							 mfdesc[(int) el]));
  case  CharList:		return(do_char_list	(dat_ptr, p_len, 
							 mfdesc[(int) el]));
  case  CharAnnounce:	return(do_cannounce	(dat_ptr, mfdesc[(int) el]));
  default:		burp(stderr, "illegal class 1 command, %d\n", 
			     (int) el);
    
  }
  return(get_all);	/* may or may not want to terminate on error */
}

/* class 2, the picture descriptor elements */
static class2(el, dat_ptr)
     char *dat_ptr;
     enum cgmcls2 el;
{
  /* the device may not want calls during the metafile defaults replacement */
  
#define maybe2 (((a2 == dflt2) && (dev_info->capability & no_def_calls)) \
		? NULL : pdesc[(int) el])
  
  switch (el) {
  case  ScalMode:			return(s_scalmode	(dat_ptr, maybe2));
  case  ColSelMode:		return(s_c_s_mode	(dat_ptr, maybe2));
  case  LWidSpecMode:		return(s_lws_mode	(dat_ptr, maybe2));
  case  MarkSizSpecMode:		return(s_ms_mode	(dat_ptr, maybe2));
  case  EdWidSpecMode:		return(s_ew_mode	(dat_ptr, maybe2));
  case  vdcExtent:		return(s_vdc_extent	(dat_ptr, maybe2));
  case  BackCol:			return(s_back_col	(dat_ptr, maybe2));
  default:			burp(stderr, "illegal class 2 command, %d\n", el);
  }
		return(get_all);
#undef maybe2
	      }
/* class 3, the control elements */
static class3(el, dat_ptr)
     char *dat_ptr;
     enum cgmcls3 el;
{
  /* the device may not want calls during the metafile defaults replacement */
  
#define maybe3 (((a3 == dflt3) && (dev_info->capability & no_def_calls)) \
		? NULL : mfctrl[(int) el])
  
  switch (el) {
  case  vdcIntPrec:	return(s_vdc_i_p	(dat_ptr, maybe3));
  case  vdcRPrec:		return(s_vdc_r_p	(dat_ptr, maybe3));
  case  AuxCol:		return(s_aux_col	(dat_ptr, maybe3));
  case  Transp:		return(s_transp		(dat_ptr, maybe3));
  case  ClipRect:		return(s_clip_rec	(dat_ptr, maybe3));
  case  ClipIndic:	return(s_clip_ind	(dat_ptr, maybe3));
  default:		burp(stderr, "illegal class 3 command, %d\n", el);
  }
		return(get_all);
#undef maybe3
	      }
/* class 4, the graphical primitive elements */
static class4(el, p_len, dat_ptr)
     char *dat_ptr;
     int p_len;
     enum cgmcls4 el;
{
  /* now go do it */
  
  switch (el) {
  case  PolyLine: 	return(do_polyline	(dat_ptr, p_len, 
						 gprim[(int) el]));
  case  Dis_Poly:		return(do_dis_polyline	(dat_ptr, p_len, 
							 gprim[(int) el]));
  case  PolyMarker:	return(do_polymarker	(dat_ptr, p_len, 
						 gprim[(int) el]));
  case  Text:		return(s_text		(dat_ptr, gprim[(int) el]));
  case  Rex_Text:		return(s_rex_text	(dat_ptr, gprim[(int) el]));
  case  App_Text:		return(s_app_text	(dat_ptr, gprim[(int) el]));
  case  Polygon:		return(do_polygon	(dat_ptr, p_len, 
							 gprim[(int) el]));
  case  Poly_Set:		return(do_polyset	(dat_ptr, p_len, 
							 gprim[(int) el]));
  case  Cell_Array:	return(do_cell_array	(dat_ptr, p_len, 
						 gprim[(int) el]));
  case  Gen_D_Prim:	return(do_g_d_p		(dat_ptr, p_len, 
						 gprim[(int) el]));
  case  Rectangle:	return(do_rectangle	(dat_ptr, gprim[(int) el]));
  case  Cgm_Circle:	return(do_circle	(dat_ptr, gprim[(int) el]));
  case  Circ_3:		return(do_c3		(dat_ptr, gprim[(int) el]));
  case  Circ_3_Close:	return(do_c3_close	(dat_ptr, gprim[(int) el]));
  case  Circ_Centre:	return(do_c_centre	(dat_ptr, gprim[(int) el]));
  case  Circ_C_Close:	return(do_c_c_close	(dat_ptr, gprim[(int) el]));
  case  Ellipse:		return(do_ellipse	(dat_ptr, gprim[(int) el]));
  case  Ellip_Arc:	return(do_ell_arc	(dat_ptr, gprim[(int) el]));
  case  El_Arc_Close:	return(do_e_a_close	(dat_ptr, gprim[(int) el]));
  default:		burp(stderr, "illegal class 4 command, %d\n",
			     (int) el); 
  }
  return(get_all);
}
/* class 5, the attribute elements */
static class5(el, p_len, dat_ptr)
     char *dat_ptr;
     int p_len;
     enum cgmcls5 el;
{
  /* the device may not want calls during the metafile defaults replacement */
  
#define maybe5 (((a5 == dflt5) && (dev_info->capability & no_def_calls)) \
		? NULL : attr[(int) el])
  
  switch (el) {
  case   LBIndex:		return(s_lbindex	(dat_ptr, maybe5));
  case   LType:		return(s_l_type		(dat_ptr, maybe5));
  case   LWidth:		return(s_l_width	(dat_ptr, maybe5));
  case   LColour:		return(s_l_colour	(dat_ptr, maybe5));
  case   MBIndex:		return(s_mbindex	(dat_ptr, maybe5));
  case   MType:		return(s_mk_type	(dat_ptr, maybe5));
  case   MSize:		return(s_mk_size	(dat_ptr, maybe5));
  case   MColour:		return(s_mk_colour	(dat_ptr, maybe5));
  case   TBIndex:		return(s_tbindex	(dat_ptr, maybe5));
  case   TFIndex:		return(s_t_index	(dat_ptr, maybe5));
  case   TPrec:		return(s_t_prec		(dat_ptr, maybe5));
  case   CExpFac:		return(s_c_exp		(dat_ptr, maybe5));
  case   CSpace:		return(s_c_space	(dat_ptr, maybe5));
  case   TColour:		return(s_t_colour	(dat_ptr, maybe5));
  case   CHeight:		return(s_c_height	(dat_ptr, maybe5));
  case   COrient:		return(s_c_orient	(dat_ptr, maybe5));
  case   TPath:		return(s_tpath		(dat_ptr, maybe5));
  case   TAlign:		return(s_t_align	(dat_ptr, maybe5));
  case   CSetIndex:	return(s_csindex	(dat_ptr, maybe5));
  case   AltCSetIndex: 	return(s_acsindex	(dat_ptr, maybe5));
  case   FillBIndex:	return(s_fbindex	(dat_ptr, maybe5));
  case   IntStyle:	return(s_interior_style	(dat_ptr, maybe5));
  case   FillColour:	return(s_fill_colour	(dat_ptr, maybe5));
  case   HatchIndex:	return(s_hindex		(dat_ptr, maybe5));
  case   PatIndex:	return(s_pindex		(dat_ptr, maybe5));
  case   EdBIndex:	return(s_e_b_index	(dat_ptr, maybe5));
  case   EType:		return(s_edge_t		(dat_ptr, maybe5));
  case   EdWidth:		return(s_edge_w		(dat_ptr, maybe5));
  case   EdColour:	return(s_edge_c		(dat_ptr, maybe5));
  case   EdVis:		return(s_edge_v		(dat_ptr, maybe5));
  case   FillRef:		return(s_fill_ref	(dat_ptr, maybe5));
  case   PatTab:		return(p_tab_entry	(dat_ptr, p_len, maybe5));
  case   PatSize:		return(s_pat_size	(dat_ptr, maybe5));
  case   ColTab:		return(c_tab_entry	(dat_ptr, p_len, maybe5));
  case   AspsFlags:	return(do_aspsflags	(dat_ptr, p_len, maybe5));
  default:		burp(stderr, "illegal class 5 command, %d\n", el);
  }
		return(get_all);
#undef maybe5
	      }
/* class 6, the escape element */
static class6(el, dat_ptr)
     char *dat_ptr;
     enum cgmcls6 el;
{
  
  switch (el) {
  case Escape:	return(do_escape	(dat_ptr, escfun[(int) el]));
  default:	burp(stderr, "illegal class 6 command, %d\n", (int) el);
  }
  return(get_all);
}
/* class 7, the external elements */
/* not really implemented yet */
static class7(el, dat_ptr)
     enum cgmcls7 el;
     char *dat_ptr;
{
  
  switch (el) {
  case Message:	return(do_message(dat_ptr, extfun[(int) el]));
  case Ap_Data:	return(do_apdata(dat_ptr, extfun[(int) el]));
  default:	burp(stderr, "illegal class 7 command, %d\n", (int) el);
  }
  return(get_all);
}
/* now the routines that do the work, class by class */
/* class 0 routines */
/* function to start a metafile */
static f_b_mf(dat_ptr, p_len, dev_func, ctrl_func)
     int (*ctrl_func)();
     int (*dev_func)();
     char *dat_ptr;
     int p_len;
{
  int ret = 1;
  char buffer[2 * max_str + 1], prog_name[max_str], *name_ptr = NULL;
  extern void rs_defaults(); /* in utils.c */
  
  /* first put in the defaults */
  cosr = 1.0;
  sinr = 0.0;
  xp0 = 0;
  yp0 = 0;
  pxl_vdc = 1.0;
  pages_done = 0;
  /* set the defaults in case anything happens before the first pic */
  rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);
  
  (void) strcpy(prog_name, "GPLOT ");
  buffer[0] = '\0';
  if (p_len > 0) cc_str(&dat_ptr, buffer);
  if (get_all) {
    burp(stderr, "Begin Metafile");
    burp(stderr, ", name: [%s]\n", buffer);
  }
  
  in_page_no = 0;
  if (dev_func) ret = (*dev_func) 
    (buffer, g_in_name, strcat(prog_name, version_str));
  
  if (ctrl_func) (*ctrl_func) ();
  return(ret);
}
/* function to end a metafile */
static f_e_mf(dev_func, ctrl_func)
     int (*ctrl_func)();
     int (*dev_func)();
{
  int ret = 1;
  
  if (get_all) burp(stderr, "End Metafile\n");
  if (dev_func) ret = (*dev_func) (pages_done);
  if (ctrl_func) (*ctrl_func) ();
  return(0);	/* time to stop */
}
/* function to reset all of the defaults before starting picture body */
static f_b_p(dat_ptr, p_len, dev_func, ctrl_func)
     int (*ctrl_func)();
     int (*dev_func)();
     char *dat_ptr;
     int p_len;
{
  int ret = 1;
  extern void rs_defaults(), get_smart(); /* in utils.c */
  
  if (p_len > 1)	{
    cc_str(&dat_ptr, pic_name);
    pic_name[max_str-1] = '\0';
  } else 	pic_name[0] = '\0';
  
  if (get_all) {
    burp(stderr, "Begin Picture");
    if (p_len > 1) 
      burp(stderr, ", name: [%s]\n", pic_name);
    else {
      burp(stderr, "\n");
    }
  }
  if (dev_func) ret = (*dev_func) (pic_name);
  if (ctrl_func) (*ctrl_func) ();

	/* go back to the defaults for get_smart, use dummy pxl_vdc */
	pxl_vdc = 1;
	rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);

	/* call get_smart and get real pxl_vdc */
    get_smart(0, glbl1, a2, opt, dev_info, &cosr, &sinr, &pxl_vdc,
	    &xoffset, &yoffset, &xp0, &yp0, &xsize, &ysize, &font_check,
	    &font_text, &sx, &sy);
  
  /* go back to the defaults for real */
  rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);
  
  return(ret);
}
/* function to get the printer ready to start drawing; implement defaults */
static f_b_p_body(dev_func, ctrl_func)
     int (*ctrl_func)();
     int (*dev_func)();
{
  int ret = 1, i;
  extern void get_smart();	/* utils.c */
  get_smart(1, glbl1, a2, opt, dev_info, &cosr, &sinr, &pxl_vdc,
	    &xoffset, &yoffset, &xp0, &yp0, &xsize, &ysize, &font_check,
	    &font_text, &sx, &sy);
  
  may_list(stderr, "Begin Picture Body\n");
  
  
  if (dev_func) 
    ret = (*dev_func) (pic_name, xoffset, yoffset, 
		opt[(int) degrees].val.r,
		*(a5->ctab), *(a5->ctab + 1), *(a5->ctab + 2), 
		in_page_no, xsize, ysize);
  
  if ((ctrl_func) && (!opt[(int) included].val.i)) (*ctrl_func) ();
  
  return(ret);
}
/* function to end a picture */
static f_e_pic(dev_func, ctrl_func)
     int (*ctrl_func)();
     int (*dev_func)();
{
  int ret = 1;
  ++pages_done;
  if (get_all) burp(stderr, "End Picture\n");
  if (dev_func) ret = (*dev_func) (opt[(int) copies].val.i);
  if ((ctrl_func) && (!opt[(int) included].val.i)) 
    (*ctrl_func) (opt[(int) copies].val.i);
  return(ret);
}
/* now the class1 functions */
/* read the metafile version number (if present) */
static rd_mf_version(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1, vers_no;
  if (p_len > 0) vers_no = cc_int(&dat_ptr);
  if (get_all) { 
    burp(stderr, "Metafile Version");
    if (p_len > 0) 
      burp(stderr, ", number: [%d]\n", vers_no);
    else burp(stderr, "\n");
  } 
  if ((p_len > 0) && dev_func) ret = (*dev_func) (vers_no);
  
  return(ret);
}

/* read the metafile descriptor */
static rd_mf_descriptor(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  char *char_ptr;
  if (p_len > 1) char_ptr = cc_str(&dat_ptr, NULL);
  else char_ptr = NULL;
  /* send off to logging line */
  log_line(char_ptr, 0, NULL, 0, 0, 2);
  if (get_all) {
    burp(stderr, "Metafile Description");
    if (p_len > 1) 
      burp(stderr, ", name: [%s]\n", char_ptr);
    else burp(stderr, "\n");
  } 
  if ((p_len > 1) && dev_func) ret = (*dev_func) (char_ptr);
  return(ret);
}

/* set the VDC type */
static s_vdc_type(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  enum vdc_enum new_type;
  int ret = 1;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "INTEGER")) new_type = vdc_int;
  else
    if (cc_same(buffer, "REAL")) new_type = vdc_real;
    else {
      fprintf(stderr, "unrecognised token in s_vdc_type %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing vdc type from %d to %d\n", 
	   (int) glbl1->vdc_type, (int) new_type);
  glbl1->vdc_type = (enum vdc_enum) new_type;
  if (dev_func) ret = (*dev_func) (new_type);
  return(ret);
}
/* set the integer precision */
static s_int_prec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_max, new_min, new_prec, ret = 1;
  
  new_min = cc_int(&dat_ptr);
  new_max = cc_int(&dat_ptr);
  new_prec = get_prec(new_min, new_max, 0);
  may_list(stderr, "changing integer precision from %d to %d\n",
	   glbl1->ind_prec, new_prec);
  glbl1->ind_prec = new_prec;
  
  if (dev_func) ret = (*dev_func) (new_prec);
  return(ret);
}

/* set the real  precision */
static s_real_prec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_fixed, new_exp, new_fract, ret = 1;
  float minreal, maxreal;
  int no_digits;
  
  minreal = cc_real(&dat_ptr);
  maxreal = cc_real(&dat_ptr);
  no_digits = cc_int(&dat_ptr);
  
  get_rprec(minreal, maxreal, no_digits, &new_fixed, &new_exp, &new_fract);
  
  may_list(stderr, 
	   "changing real precision from (%d, %d, %d) to (%d, %d, %d)\n",
	   glbl1->real_prec.fixed, glbl1->real_prec.exp, 
	   glbl1->real_prec.fract, new_fixed, new_exp, new_fract);
  
  glbl1->real_prec.fixed	= new_fixed;
  glbl1->real_prec.exp 	= new_exp;
  glbl1->real_prec.fract	= new_fract;
  
  if (dev_func) ret = (*dev_func) (new_fixed, new_exp, new_fract);
  return(ret);
}
/* set the index precision */
static s_index_prec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_max, new_min, new_prec, ret = 1;
  
  new_min = cc_int(&dat_ptr);
  new_max = cc_int(&dat_ptr);
  new_prec = get_prec(new_min, new_max, 0);
  may_list(stderr, "changing index precision from %d to %d\n",
	   glbl1->ind_prec, new_prec);
  glbl1->ind_prec = new_prec;
  if (dev_func) ret = (*dev_func) (new_prec);
  return(ret);
}
/* set the colour precision */
static s_col_prec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_max, new_prec, ret = 1;
  
  new_max = cc_int(&dat_ptr);
  new_prec = get_prec(0, new_max, 0);
  may_list(stderr, "changing colour precision from %d to %d\n",
	   glbl1->col_prec, new_prec);
  glbl1->col_prec = new_prec;
  if (dev_func) ret = (*dev_func) (new_prec);
  return(ret);
}
/* set the colour index precision */
static s_cind_prec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_max, new_prec, ret = 1;
  
  new_max = cc_int(&dat_ptr);
  new_prec = get_prec(0, new_max, 0);
  may_list(stderr, "changing colour index precision from %d to %d\n",
	   glbl1->col_i_prec, new_prec);
  glbl1->col_i_prec = new_prec;
  if (dev_func) ret = (*dev_func) (new_prec);
  return(ret);
}
/* set the colour value extent */
static s_cvextent(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int cvmin[3], cvmax[3], ret = 1, i;
  
  for (i=0; i<3; ++i) {
    cvmin[i] = cc_int(&dat_ptr);
  }
  for (i=0; i<3; ++i) {
    cvmax[i] = cc_int(&dat_ptr);
  }
  
  may_list(stderr, 
	   "changing col value extent from (%d,%d,%d,%d,%d,%d) to (%d,%d,%d,%d,%d,%d)\n",
	   glbl1->c_v_extent.min[0], glbl1->c_v_extent.min[1], 
	   glbl1->c_v_extent.min[2], 
	   glbl1->c_v_extent.max[0], glbl1->c_v_extent.max[1], 
	   glbl1->c_v_extent.max[2], 
	   cvmin[0], cvmin[1], cvmin[1], cvmax[0], cvmax[1], cvmax[2]);
  
  for (i=0; i<3; ++i) {
    glbl1->c_v_extent.min[i] = cvmin[i];
    glbl1->c_v_extent.max[i] = cvmax[i];
  }
  
  if (dev_func) ret = (*dev_func) (cvmin, cvmax);
  
  return(ret);
}
/* set the maximum colour index */
/* split into two functions as we may have to handle illegal 
   metafiles (DI3000) that add c table entries before increasing
   glbl1->max_c_index */
static s_mcind(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_index, ret = 1;
  int do_mcind();
  new_index = cc_int(&dat_ptr);
  if (!do_mcind(new_index)) burp(stderr, "trouble setting max colind\n");
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
static do_mcind(new_index)
     int new_index;
{
  int i;
  float *new_d_ptr, *new_g_ptr;
  char *allocate_mem();
  
  may_list(stderr, "changing max colour index from %d to %d\n",
	   glbl1->max_c_index, new_index);
  if (new_index > glbl1->max_c_index) { /* need to make some new memory */
    new_d_ptr = 
      (float *)allocate_mem(float_size * 3 * (new_index + 1), 0);
    new_g_ptr = 
      (float *)allocate_mem(float_size * 3 * (new_index + 1), 0);
    if ((!new_d_ptr) || (!new_g_ptr)) return(0);
    /* move over the old data */
    for (i=0; i<(glbl1->max_c_index + 1) * 3; ++i) {
      *(new_d_ptr + i) = *(a5->ctab + i);
      *(new_g_ptr + i) = *(a5->ctab + i);
    }
    /* free up memory */
    free(dflt5->ctab);
    free(glbl5->ctab);
    /* and reassign the pointers */
    dflt5->ctab = new_d_ptr;
    glbl5->ctab = new_g_ptr;
  }
  glbl1->max_c_index = new_index;
  return(1);
}

/* read the metafile element list (may be useful) */
static rd_mf_list(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
#define max_buffer 1024
  int no_pairs, i, j, k, done, class, el, ret = 1, out_array[max_buffer];
  char *cptr;
  char buffer[max_buffer], cmd_name[max_str];
  
  /* lets see if we have a string */
  while (*dat_ptr == ' ') ++dat_ptr;
  if (is_term(*dat_ptr)) {
    fprintf(stderr, "empty mf element list\n");
    return(2);
  } else if (is_quote(*dat_ptr)) {
    ++dat_ptr;
    i = 0;
    while ((!is_quote(*dat_ptr)) && (i < (max_buffer - 1)))
      buffer[i++] = *(dat_ptr++);
    buffer[i] = '\0';
  } else {
    fprintf(stderr, "not expecting %c in mf elem list\n", *dat_ptr);
    return(2);
  }
  
  /* now clean up the string */
  clean_string(buffer);
  
  /* and pick out the tokens */
  cptr = buffer;
  no_pairs = 0;
  while ((*cptr) && get_token(&cptr, cmd_name)) {
    if (cc_same(cmd_name, "DRAWINGSET")) {
      class = -1;
      el = 0;
    } else
      if (cc_same(cmd_name, "DRAWINGPLUS")) {
	class = -1;
	el = 1;
      } else 
	/* may be replacing defaults */
	if (cc_same(cmd_name, "BEGMFDEFAULTS")) {
	  class = 2;
	  el = 13;
	} else 
	  if (cc_same(cmd_name, "ENDMFDEFAULTS")) {
	    class = 2;
	    el = 13;
	  } else {
	    done = 0;
	    for (i=0; (i<8) && (!done); ++i) {
	      for (j=1; (j<cc_size[i]) && (!done); ++j) {
		k = 0;
		while ((CC_cptr[i][j][k]) && (cmd_name[k]) &&
		       (CC_cptr[i][j][k] == cmd_name[k]))
		  ++k;
		done =((!CC_cptr[i][j][k]) && (!cmd_name[k]) && (k));
	      }
	    }
	    if (!done) {
	      fprintf(stderr, "couldn't find a match for %s in defrep\n", 
		      cmd_name);
	      return(2);
	    }
	    class = i - 1;
	    el = j - 1;
	  }
    out_array[2 * no_pairs] = class;
    out_array[2 * no_pairs + 1] = el;
    ++no_pairs;
    while (*cptr == ' ') ++cptr;
  }
  
  if (dev_func) ret = (*dev_func) (no_pairs, out_array);
  return(ret);
}
#undef max_buffer
/* replace the metafile defaults */
static s_mf_defs(dat_ptr, p_len, dev_func, df_call)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
     int df_call;
{
  int ret = 1;
  
  if (df_call == 0) {	
    may_list(stderr, "replacing metafile defaults[\n");
    a2 = dflt2;	a3 = dflt3;	a5 = dflt5;
    if (dev_func) ret = (*dev_func) (0);	/* start */
    /* default values are active */
    return(ret);
  } else
    if (df_call == 1) {	
      a2 = glbl2;	a3 = glbl3;	a5 = glbl5;
      /* globals are now active again */
      may_list(stderr, "] finished replacing defaults\n");
      if (dev_func) ret = (*dev_func) (1);		/* end */
      return(ret);
    } else {
      fprintf(stderr, "illegal argument to s_mf_defs\n");
      return(2);
    }
}
/* read the font list (implement fully later) */
static do_font_list(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  int no_strings;
  char font_list[max_fonts][max_str + 1], *new_ptr;
  
  new_ptr = dat_ptr;
  no_strings = 0;
  may_list(stderr, "recording font list\n");
  while ((no_strings < max_fonts) && (new_ptr < dat_ptr + p_len - 3)) {
    cc_str(&new_ptr, font_list[no_strings]);
    font_list[no_strings][max_str] = '\0';	/* for safety */
    may_list(stderr, "[%s]", font_list[no_strings]);
    ++no_strings;
  }
  may_list(stderr, "\n");
  if (no_strings > max_fonts) no_strings = max_fonts;
  
  if (dev_func) ret = (*dev_func) (no_strings, font_list);
  return(ret);
}
/* read the character list (implement fully later) */
static do_char_list(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  int no_strings, type_array[max_fonts];
  char font_list[max_fonts][max_str + 1];
  char *new_ptr;
  char buffer[max_str];
  
  new_ptr = dat_ptr;
  no_strings = 0;
  may_list(stderr, "recording character set list\n");
  while (*new_ptr == ' ') ++new_ptr;	/* skip spaces */
  
  while ((no_strings < max_fonts) && (!is_term(*new_ptr))) {
    if (!get_token(&new_ptr, buffer)) return(2);
    
    if (cc_same(buffer, "STD94")) type_array[no_strings] = 0;
    else if (cc_same(buffer, "STD96")) type_array[no_strings] = 1;
    else if (cc_same(buffer, "STD94MULTIBYTE")) 
      type_array[no_strings] = 2;
    else if (cc_same(buffer, "STD96MULTIBYTE")) 
      type_array[no_strings] = 3;
    else if (cc_same(buffer, "COMPLETECODE")) 
      type_array[no_strings] = 4;
    else {
      (void) fprintf(stderr, "unknown character set %s\n", buffer);
      return(2);
    }
    
    may_list(stderr, "%s ", buffer);
    cc_str(&new_ptr, font_list[no_strings]);
    font_list[no_strings][max_str] = '\0';	/* for safety */
    may_list(stderr, "[%s]", font_list[no_strings]);
    ++no_strings;
  }
  may_list(stderr, "\n");
  if (no_strings > max_fonts) no_strings = max_fonts;
  
  if (dev_func) ret = (*dev_func) (no_strings, type_array, font_list);
  return(ret);
}
/* do the character announcer */
static do_cannounce(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  int new_announce;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "BASIC7BIT")) new_announce = 0;
  else if (cc_same(buffer, "BASIC8BIT")) new_announce = 1;
  else if (cc_same(buffer, "EXTD7BIT")) new_announce = 2;
  else if (cc_same(buffer, "EXTD8BIT")) new_announce = 3;
  else {
    (void) fprintf(stderr, "unknown character announcer %s\n", buffer);
    return(2);
  }
  may_list(stderr, "changing character announcer from %d to %d\n",
	   glbl1->char_c_an, new_announce);
  glbl1->char_c_an = new_announce;
  if (dev_func) ret = (*dev_func) (new_announce);
  return(ret);
}
/* now the class2 functions */
/* set the scaling mode */
static s_scalmode(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  char buffer[max_str];
  int ret = 1, new_mode;
  float my_scale = 1;
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ABSTRACT")) new_mode = 0;
  else
    if (cc_same(buffer, "METRIC")) {
      new_mode = 1;
      my_scale = cc_real(&dat_ptr);
    }
    else {
      fprintf(stderr, "unrecognised token in s_scalmode %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing scaling mode from %d, %.4f to %d, %.4f\n",
	   a2->scale_mode.s_mode, a2->scale_mode.m_scaling,
	   new_mode, my_scale);
  
  a2->scale_mode.s_mode = new_mode;
  a2->scale_mode.m_scaling = my_scale;
  
  if (dev_func) ret = (*dev_func) (new_mode, my_scale);
  
  return(ret);
}

/* set the colour selection mode */
static s_c_s_mode(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum cs_enum new_c_s_mode;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "INDEXED")) new_c_s_mode = i_c_mode;
  else
    if (cc_same(buffer, "DIRECT")) new_c_s_mode = d_c_mode;
    else {
      fprintf(stderr, "unrecognised token in s_c_s_mode %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing colour selection mode from %d to %d\n",
	   (int) a2->c_s_mode, (int)new_c_s_mode);
  
  a2->c_s_mode = new_c_s_mode;
  if (dev_func) ret = (*dev_func) (new_c_s_mode);
  return(ret);
}
/* set the line width specification mode */
static s_lws_mode(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum spec_enum  new_lws_mode;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ABSTRACT")) new_lws_mode = absolute;
  else
    if (cc_same(buffer, "SCALED")) new_lws_mode = scaled;
    else {
      fprintf(stderr, "unrecognised token in s_lws_mode %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing line width spec mode from %d to %d\n",
	   (int) a2->l_w_s_mode, (int) new_lws_mode);
  a2->l_w_s_mode = new_lws_mode;
  if (dev_func) ret = (*dev_func) (new_lws_mode);
  return(ret);
}
/* set the marker size specification mode */
static s_ms_mode(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum spec_enum new_mode;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ABSTRACT")) new_mode = absolute;
  else
    if (cc_same(buffer, "SCALED")) new_mode = scaled;
    else {
      fprintf(stderr, "unrecognised token in s_ms_mode %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing marker size spec mode from %d to %d\n",
	   (int) a2->m_s_s_mode, (int) new_mode);
  
  a2->m_s_s_mode = new_mode;
  if (dev_func) ret = (*dev_func) (new_mode);
  return(ret);
}
/* set the edge width specification mode */
static s_ew_mode(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum spec_enum new_mode;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ABSTRACT")) new_mode = absolute;
  else
    if (cc_same(buffer, "SCALED")) new_mode = scaled;
    else {
      fprintf(stderr, "unrecognised token in s_ew_mode %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing edge width spec mode from %d to %d\n",
	   (int) a2->e_w_s_mode, (int) new_mode);
  
  a2->e_w_s_mode = new_mode;
  if (dev_func) ret = (*dev_func) (new_mode);
  return(ret);
}
/* set the VDC extent */
static s_vdc_extent(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_coords[4], i, ret = 1; float new_real[4];
  
  switch (glbl1->vdc_type) {
  case vdc_int:	get_ipoint(&dat_ptr, new_coords, new_coords + 1);
    get_ipoint(&dat_ptr, new_coords + 2, new_coords + 3);
    may_list(stderr, 
	     "changing VDC extent from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
	     a2->vdc_extent.i[0], a2->vdc_extent.i[1], 
	     a2->vdc_extent.i[2], a2->vdc_extent.i[3], 	    	    
	     new_coords[0], new_coords[1], new_coords[2],new_coords[3]);
    for (i=0; i<4; ++i) a2->vdc_extent.i[i] = new_coords[i];
    break;
  case vdc_real:	get_rpoint(&dat_ptr, new_real, new_real + 1);
    get_rpoint(&dat_ptr, new_real + 2, new_real + 3);
    may_list(stderr, 
	     "changing VDC extent from (%f,%f,%f,%f) to (%f,%f,%f,%f)\n",
	     a2->vdc_extent.r[0], a2->vdc_extent.r[1], 
	     a2->vdc_extent.r[2], a2->vdc_extent.r[3],
	     new_real[0], new_real[1], new_real[2],new_real[3]);
    for (i=0; i<4; ++i) a2->vdc_extent.r[i] = new_real[i];
    break;
  default:	burp(stderr, "illegal vdc_type = %d\n", glbl1->vdc_type);
    
    
  }
  if (dev_func) ret = (*dev_func) (new_coords, new_real);
  return(ret);
}
/* set the background colour */
static s_back_col(dat_ptr,dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  unsigned int ir, ig, ib;
  float r, g, b;
  
  ir = cc_int(&dat_ptr);
  r = dcr(ir);
  ig = cc_int(&dat_ptr);
  g = dcg(ig);
  ib = cc_int(&dat_ptr);
  b = dcb(ib);
  may_list(stderr, 
	   "changing background colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	   a2->back_col.red, a2->back_col.green, a2->back_col.blue,
	   r, g, b);
  a2->back_col.red = r;
  a2->back_col.green = g;
  a2->back_col.blue = b;

        /* we have to override the index 0 colour by the background
            colour (sounds screwy, but I think correct) */

        *(a5->ctab)             = a2->back_col.red;
        *(a5->ctab + 1)         = a2->back_col.green;
        *(a5->ctab + 2)         = a2->back_col.blue;
                                                                                
  if (dev_func) ret = (*dev_func)(ir, ig, ib);
  return(ret);
}
/* now the class3 functions */
/* set the vdc integer precision */
static s_vdc_i_p(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_max, new_min, new_prec, ret = 1;
  
  new_min = cc_int(&dat_ptr);
  new_max = cc_int(&dat_ptr);
  new_prec = get_prec(new_min, new_max, 0);
  may_list(stderr, "changing vdc int precision from %d to %d",
	   a3->vdc_i_prec, new_prec);
  a3->vdc_i_prec = new_prec;
  if (dev_func) ret = (*dev_func) (new_prec);
  return(ret);
}
/* set the vdc real precision */
static s_vdc_r_p(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_fixed, new_exp, new_fract, ret = 1;
  float minreal, maxreal;
  int no_digits;
  
  minreal = cc_real(&dat_ptr);
  maxreal = cc_real(&dat_ptr);
  no_digits = cc_int(&dat_ptr);
  
  get_rprec(minreal, maxreal, no_digits, &new_fixed, &new_exp, &new_fract);
  
  may_list(stderr, 
	   "changing vdc real precision from (%d, %d, %d) to (%d, %d, %d)\n",
	   a3->vdc_r_prec.fixed, a3->vdc_r_prec.exp, 
	   a3->vdc_r_prec.fract, new_fixed, new_exp, new_fract);
  
  a3->vdc_r_prec.fixed	= new_fixed;
  a3->vdc_r_prec.exp 	= new_exp;
  a3->vdc_r_prec.fract	= new_fract;
  
  if (dev_func) ret = (*dev_func) (new_fixed, new_exp, new_fract);
  return(ret);
}
/* set the auxiliary colour */
static s_aux_col(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  int new_index = 0, ir, ig, ib;
  float r, g, b, *rptr;
  
  switch ((int) a2->c_s_mode) {
  case (int) i_c_mode: /* indexed mode */
    new_index = cc_int(&dat_ptr);
    if (list_cgm) 
      burp(stderr, "changing auxiliary index from %d to %d\n",
	   a3->aux_col.ind, new_index);
    a3->aux_col.ind = new_index;
    rptr = a5->ctab + a3->aux_col.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case (int) d_c_mode: /* direct mode */
    ir = cc_int(&dat_ptr);
    r = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing auxiliary colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a3->aux_col.red, a3->aux_col.green, 
	     a3->aux_col.blue, r, g, b);
    a3->aux_col.red = r;
    a3->aux_col.green = g;
    a3->aux_col.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", (int) a2->c_s_mode);
  }
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  return(ret);
}
/* set the transparency */
static s_transp(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum boolean new_trans;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ON")) new_trans = on;
  else
    if (cc_same(buffer, "OFF")) new_trans = off;
    else {
      fprintf(stderr, "unrecognised token in s_transp %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing transparency from %d to %d\n",
	   (int) a3->transparency, (int) new_trans);
  a3->transparency = new_trans;
  if (dev_func) ret = (*dev_func) (new_trans);
  return(ret);
}


/* set the clipping rectangle */
static s_clip_rec(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int new_coords[4], i; float new_real[4], ret = 1;
  char *keep_ptr;
  
  switch (glbl1->vdc_type) {
  case vdc_int:
/*    get_ipoint(&dat_ptr, new_coords, new_coords + 1); */
/*    get_ipoint(&dat_ptr, new_coords + 2, new_coords + 3); */
    get_vpoint(&dat_ptr, new_coords, new_coords + 1);
    get_vpoint(&dat_ptr, new_coords + 2, new_coords + 3);
    may_list(stderr, 
	     "changing clipping rectangle from (%d,%d,%d,%d)to (%d,%d,%d,%d)\n",
	     a3->clip_rect.i[0], a3->clip_rect.i[1], a3->clip_rect.i[2],
	     a3->clip_rect.i[3],
	     new_coords[0], new_coords[1], new_coords[2],new_coords[3]);
    for (i=0; i<4; ++i) a3->clip_rect.i[i] = new_coords[i];
    break;
  case vdc_real:
    keep_ptr = dat_ptr;
    get_vpoint(&dat_ptr, a3->clip_rect.i, a3->clip_rect.i + 1);
    get_vpoint(&dat_ptr, a3->clip_rect.i + 2, a3->clip_rect.i + 3);
    get_rpoint(&keep_ptr, new_real, new_real + 1);
    get_rpoint(&keep_ptr, new_real + 2, new_real + 3);
    may_list(stderr, 
	     "changing clipping rectangle from (%d,%d,%d,%d)to (%d,%d,%d,%d)\n",
	     a3->clip_rect.r[0], a3->clip_rect.r[1], a3->clip_rect.r[2],
	     a3->clip_rect.r[3], 
	     new_real[0], new_real[1], new_real[2],new_real[3]);
    for (i=0; i<4; ++i) a3->clip_rect.r[i] = new_real[i];
    break;
  default:	burp(stderr, "illegal vdc_type = %d\n", glbl1->vdc_type);
    
  }
/* get into right format for device */
	for (i=0; i<2; ++i) {
	  new_coords[2 * i] = NEWX(a3->clip_rect.i[2*i],
				   a3->clip_rect.i[2*i + 1]);
	  new_coords[2 * i + 1] = NEWY(a3->clip_rect.i[2*i],
				       a3->clip_rect.i[2*i + 1]);
	}

  if (dev_func) ret = (*dev_func) (new_coords, new_real);
  return(ret);
}
/* set the clipping indicator */
static s_clip_ind(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum boolean new_clip;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ON")) new_clip = on;
  else
    if (cc_same(buffer, "OFF")) new_clip = off;
    else {
      fprintf(stderr, "unrecognised token in s_clip_ind %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing clipping indicator from %d to %d\n",
	   (int) a3->clip_ind, (int) new_clip);
  a3->clip_ind = new_clip;
  if (dev_func) ret = (*dev_func) (new_clip);
  return(ret);
}
/* now the class 4 functions */
/* take care of a series of points that need a line between them */
static do_polyline(dat_ptr, p_len, dev_func)
     int (*dev_func)();
     char *dat_ptr;
     int p_len;
{
  int ret = 1;
  int no_pairs, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
  
  no_pairs = get_pairs(dat_ptr);
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x1_ptr = intalloc(no_pairs * bytes_per_word);
    y1_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x1_ptr = x_wk_ar;
    y1_ptr = y_wk_ar;
  }
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with polyline memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  
  /* get the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    *x_ptr++ = newx(x, y);
    *y_ptr++ = newy(x, y);
    ++no_pairs;
  }
  
  /* and take care of the output */
  if (list_cgm) {
    (void) fprintf(stderr, "polyline = [");
    for (i=0;i<no_pairs;++i) 
      fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
    (void) fprintf(stderr, "]\n");
  }
  
  if (dev_func) ret = (*dev_func) (no_pairs, x1_ptr, y1_ptr);
  
  if (x1_ptr != x_wk_ar) free(x1_ptr);
  if (y1_ptr != y_wk_ar) free(y1_ptr);
  
  return(ret);
}
/* take care of a series of points that need a line between alternate points*/
static do_dis_polyline(dat_ptr, p_len, dev_func)
     int (*dev_func)();
     char *dat_ptr;
     int p_len;
{
  int ret = 1;
  int no_pairs, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
  
  no_pairs = get_pairs(dat_ptr);
  /* some arrays have odd number of points ! */
  if (no_pairs % 2) --no_pairs;
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x1_ptr = intalloc(no_pairs * bytes_per_word);
    y1_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x1_ptr = x_wk_ar;
    y1_ptr = y_wk_ar;
  }
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with disjoint polyline memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  
  /* now grab the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    *x_ptr++ = newx(x, y);
    *y_ptr++ = newy(x, y);
    ++no_pairs;
  }
  
  /* and take care of the output */
  if (list_cgm) {
    (void) fprintf(stderr, "disjoint polyline = [");
    for (i=0;i<no_pairs;++i) 
      fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
    (void) fprintf(stderr, "]\n");
  }
  
  ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr):
    em_dpline(no_pairs, x1_ptr, y1_ptr);
  
  if (x1_ptr != x_wk_ar) free(x1_ptr);
  if (y1_ptr != y_wk_ar) free(y1_ptr);
  
  return(ret);
}
/* do a series of markers at the specified points */
static do_polymarker(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  int no_pairs, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
  extern int em_pmarker();	/* emul.c */
  
  no_pairs = get_pairs(dat_ptr);
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x1_ptr = intalloc(no_pairs * bytes_per_word);
    y1_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x1_ptr = x_wk_ar;
    y1_ptr = y_wk_ar;
  }
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with polymarker memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  
  /* now grab the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    *x_ptr++ = newx(x, y);
    *y_ptr++ = newy(x, y);
    ++no_pairs;
  }
  /* and take care of the output */
  if (list_cgm) {
    (void) fprintf(stderr, "polymarker = [");
    for (i=0;i<no_pairs;++i) 
      fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
    (void) fprintf(stderr, "]\n");
  }
  ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr) :
    em_pmarker(no_pairs, x1_ptr, y1_ptr);
  
  if (x1_ptr != x_wk_ar) free(x1_ptr);
  if (y1_ptr != y_wk_ar) free(y1_ptr);
  
  return(ret);
}
/* set actual text, also take care of raster character descriptions */
static s_text(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  int x, y, xin, yin, no_chars, device_ok, i, ind;
  enum boolean final;
  char txt_buffer[2 * max_str + 1];
  char buffer[max_str];
  static int em_type = 0, font_size;
  int old_c_index, new_l_width, old_l_width;
  float r0, g0, b0, l0;
  double mag;
  enum line_enum  old_l_type = solid_l;
  
  /* first see if the device can handle it */
  device_ok = 
    ((a5->t_prec == string) && (dev_info->capability & string_text)) ||
      ((a5->t_prec == character) && (dev_info->capability & char_text))||
	((a5->t_prec == stroke) && (dev_info->capability & stroke_text)) ||
	  (opt[(int) font_type].set && !opt[(int) font_type].val.i) ||
	    (font_check == NULL) || (font_text == NULL);
  
  /* may be overrriding */
  if (opt[(int) font_type].set && opt[(int) font_type].val.i &&
      font_check && font_text) device_ok = 0;
  
  /* if don't have device_ok, but can't emulate, we'll force device_ok */
  if (!device_ok) {
    if (!em_type) em_type = (*font_check)(&font_size);
    switch (em_type) {
    case CA_EMULATION:	/* emulation via Cell Arrays */
      if (!gprim[(int) Cell_Array]) device_ok = 1;
      break;
    case PL_EMULATION:	/* emulation via Polyline */
      if (!gprim[(int) PolyLine]) device_ok = 1;
      break;
    }
  }
  
  /* now ready to procede */
  
  get_vpoint(&dat_ptr, &xin, &yin);
  x = newx(xin, yin);
  y = newy(xin, yin);
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "FINAL")) final = on;
  else
    if (cc_same(buffer, "NOTFINAL")) final = off;
    else {
      fprintf(stderr, "unrecognised token in s_text %s\n", buffer);
      return(2);
    }
  
  cc_str(&dat_ptr, txt_buffer);
  no_chars = strlen(txt_buffer);
  
  may_list(stderr, "text[%d] at (%d,%d) = [%s]\n",
	   *(dat_ptr + 1), x, y, txt_buffer);
  
  /* may have to do the positioning adjustment */
  if (!(dev_info->capability & v_center) && (device_ok)) {
    
    switch (a5->text_align.ver) {
    case top_v:		y -= 1.1 * dev_info->c_height; break;
    case cap_v:		y -= dev_info->c_height; break;
    case half_v:		y -= 0.5 * dev_info->c_height; break;
    case bottom_v:		y -= 0.1 * dev_info->c_height; break;
    }
  }
  
  if (!(dev_info->capability & h_center) && (device_ok)) { 
    switch (a5->text_align.hor) {	/* fix later */
    case center_h:		x -= 0.5 * no_chars * dev_info->c_width; break;
    case right_h:		x -= no_chars * dev_info->c_width; break;
    }
  }
  
  
  /* set the text */
  if (device_ok) { 
    ret = (dev_func) ? (*dev_func) (x, y, final, txt_buffer) : 2;
  } else {	/* may do it with emulation */
    /* decide the relevant font magnification */
    mag = (double) a5->c_height / font_size;
    
    switch (em_type) {
    case CA_EMULATION:	/* emulation via Cell Arrays */
      if (gprim[(int) Cell_Array])
	ret = (*font_text)(x, y, txt_buffer, 
			   gprim[(int) Cell_Array], 
			   a2, a5, no_chars, dev_info->capability, NULL, mag);
      break;
    case PL_EMULATION:	/* emulation via Polylines */
      if (gprim[(int) PolyLine]) {
	/* need to set the colour, line size/type */
	/* first the colour */
	old_c_index = a5->line_colour.ind;
	r0 = a5->line_colour.red;
	g0 = a5->line_colour.green;
	b0 = a5->line_colour.blue;
	a5->line_colour.ind = a5->text_colour.ind;
	a5->line_colour.red = a5->text_colour.red;
	a5->line_colour.green = a5->text_colour.green;
	a5->line_colour.blue = a5->text_colour.blue;
	if (attr[(int) LColour]) (*attr[(int) LColour])
	  (a5->text_colour.red, a5->text_colour.green, 
	   a5->text_colour.blue, a5->text_colour.ind);
	/* now the line type */
	if (a5->line_type != solid_l) {
	  old_l_type = a5->line_type;
	  a5->line_type = solid_l;
	  if (attr[(int) LType]) (*attr[(int) LType])
	    (a5->line_type);
	}
	/* now the line width */
	new_l_width = 0.5 + mag;
	if (new_l_width < 1) new_l_width = 1;
	if (new_l_width != a5->line_width.i) {
	  old_l_width = a5->line_width.i;
	  l0 = a5->line_width.r;
	  a5->line_width.i = new_l_width;
	  if (!dev_info->d_l_width) dev_info->d_l_width = 1; /* safety */
	  a5->line_width.r = (float) a5->line_width.i /
	    dev_info->d_l_width;
	  if (attr[(int) LWidth]) (*attr[(int) LWidth])
	    (a5->line_width.i, a5->line_width.r);
	}
	/* actually set the text */
	ret = (*font_text)(x, y, txt_buffer, gprim[(int) PolyLine], 
			   a2, a5, no_chars, dev_info->capability, NULL, mag);
	
	/* reset attributes */
	a5->line_colour.ind = old_c_index;
	a5->line_colour.red = r0;
	a5->line_colour.green = g0;
	a5->line_colour.blue = b0;
	if (attr[(int) LColour]) (*attr[(int) LColour])
	  (a5->line_colour.red, a5->line_colour.green, 
	   a5->line_colour.blue, a5->line_colour.ind);
	if (old_l_type != solid_l) {
	  a5->line_type = old_l_type;
	  if (attr[(int) LType]) (*attr[(int) LType])
	    (a5->line_type);
	}
	if (old_l_width != 1) {
	  a5->line_width.i = old_l_width;
	  a5->line_width.r = l0;
	  if (attr[(int) LWidth]) (*attr[(int) LWidth])
	    (a5->line_width.i, a5->line_width.r);
	}
	/* all done */
	
      }
      break;
    default:	ret = 0;
      break;
    }
  }
  return(ret);
}
/* restricted text */
static s_rex_text(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  int width, height, xin, yin, x, y;
  enum boolean final;
  char txt_buffer[2 * max_str + 1];
  char buffer[max_str];
  
  width = cc_vdc(&dat_ptr);
  height = cc_vdc(&dat_ptr);
  get_vpoint(&dat_ptr, &xin, &yin);
  x = newx(xin, yin);
  y = newy(xin, yin);
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "FINAL")) final = on;
  else
    if (cc_same(buffer, "NOTFINAL")) final = off;
    else {
      fprintf(stderr, "unrecognised token in s_text %s\n", buffer);
      return(2);
    }
  
  cc_str(&dat_ptr, txt_buffer);
  
  if (dev_func) ret = (*dev_func) (width, height, x, y, 
				   (int) final, txt_buffer);
  
  return(ret);
}
/* appended text */
static s_app_text(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  enum boolean final;
  char txt_buffer[2 * max_str + 1];
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "FINAL")) final = on;
  else
    if (cc_same(buffer, "NOTFINAL")) final = off;
    else {
      fprintf(stderr, "unrecognised token in s_text %s\n", buffer);
      return(2);
    }
  
  cc_str(&dat_ptr, txt_buffer);
  
  if (dev_func) ret = (*dev_func) ((int) final, txt_buffer);
  return(ret);
}

/* handle a polygon */
static do_polygon(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  int no_pairs, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
  
  no_pairs = get_pairs(dat_ptr);
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x1_ptr = intalloc(no_pairs * bytes_per_word);
    y1_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x1_ptr = x_wk_ar;
    y1_ptr = y_wk_ar;
  }
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with polyline memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with polygon memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  
  /* now grab the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    *x_ptr++ = newx(x, y);
    *y_ptr++ = newy(x, y);
    ++no_pairs;
  }
  
  /* and take care of the output */
  if (list_cgm) {
    (void) fprintf(stderr, "polygon = [");
    for (i=0;i<no_pairs;++i) 
      fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
    (void) fprintf(stderr, "]\n");
  }
  
  if (dev_func) ret = (*dev_func) (no_pairs, x1_ptr, y1_ptr);
  
  if (x1_ptr != x_wk_ar) free(x1_ptr);
  if (y1_ptr != y_wk_ar) free(y1_ptr);
  
  return(ret);
}
/* do a polyset */
static do_polyset(dat_ptr, p_len, dev_func)
     char *dat_ptr;
     int p_len;
     int (*dev_func)();
{
  int ret = 1;
  int no_pairs, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
  char *edge_ptr, *allocate_mem(), buffer[max_str]; 
  
  /* figure out how many pairs */
  no_pairs = get_pairs(dat_ptr);
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x1_ptr = intalloc(no_pairs * bytes_per_word);
    y1_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x1_ptr = x_wk_ar;
    y1_ptr = y_wk_ar;
  }
  edge_ptr = allocate_mem(no_pairs, 0);
  if (!edge_ptr) return(2);
  if ( (!x1_ptr) || (!y1_ptr) ) 
    burp(stderr, "trouble with polyset memory !\n");
  x_ptr = x1_ptr;
  y_ptr = y1_ptr;
  
  /* now grab the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    *x_ptr++ = newx(x, y);
    *y_ptr++ = newy(x, y);
    if (!get_token(&dat_ptr, buffer)) return(2);
    if (cc_same(buffer, "INVIS")) edge_ptr[no_pairs] = 0;
    else
      if (cc_same(buffer, "VIS")) edge_ptr[no_pairs] = 1;
      else
	if (cc_same(buffer, "CLOSEINVIS")) edge_ptr[no_pairs] = 2;
	else
	  if (cc_same(buffer, "CLOSEVIS")) edge_ptr[no_pairs] = 3;
	  else {
	    fprintf(stderr, "unrecognised token in do_polyset %s\n",
		    buffer);
	    return(2);
	  }
    ++no_pairs;
  }
  
  /* and take care of the output */
  if (list_cgm) {
    (void) fprintf(stderr, "polyset = [");
    for (i=0;i<no_pairs;++i) 
      fprintf(stderr, "\n(%d,%d, %d)", x1_ptr[i], y1_ptr[i], 
	      edge_ptr[i]);
    (void) fprintf(stderr, "]\n");
  }
  
  if (dev_func) ret = (*dev_func) (no_pairs, x1_ptr, y1_ptr, edge_ptr);
  
  free(edge_ptr);
  if (x1_ptr != x_wk_ar) free(x1_ptr);
  if (y1_ptr != y_wk_ar) free(y1_ptr);
  return(ret);
}
/* now try to take care of a general cell array */
static do_cell_array(dat_ptr, plen, dev_func)
     char *dat_ptr;
     int plen, (*dev_func)();
{
  int ret = 1, new_max;
  int p0[2], cp[2];	/* corner p */
  int q0[2], cq[2];	/* corner q */
  int r0[2], cr[2];	/* corner r */
  /* this is a parallelogram, diagonal between p and q, first row is p-r */
  unsigned int nx;		/* columns of data */
  unsigned int ny;		/* rows of data */
  int l_col_prec;	/* local colour precision */
  int rep_mode;	/* cell representation mode */
  int i, j, row_size, true_size, c_size;
  char *new_ptr, *allocate_mem(), *my_ptr;
  long int no_bytes;	/* number of bytes of data */
  
  get_vpoint(&dat_ptr, p0, p0 + 1);
  get_vpoint(&dat_ptr, q0, q0 + 1);
  get_vpoint(&dat_ptr, r0, r0 + 1);
  
  nx = cc_int(&dat_ptr);
  ny = cc_int(&dat_ptr);
  
  new_max = cc_int(&dat_ptr);
  l_col_prec = (new_max) ? get_prec(0, new_max, 1) : 0;
  l_col_prec = (l_col_prec) ? l_col_prec : glbl1->col_prec;
  if (l_col_prec < 8) l_col_prec = 8;	/* fix later */
  rep_mode = 1;	/* only possibility for clear text */
  
  /* get the precision right */
  switch (a2->c_s_mode) {
  case d_c_mode: 	c_size = 3 * l_col_prec; break;
  case i_c_mode: 	c_size = l_col_prec; break;
  }
  cp[0] = newx(p0[0], p0[1]);
  cp[1] = newy(p0[0], p0[1]);
  cq[0] = newx(q0[0], q0[1]);
  cq[1] = newy(q0[0], q0[1]);
  cr[0] = newx(r0[0], r0[1]);
  cr[1] = newy(r0[0], r0[1]);
  
  /* we will encode the data in the binary cgm format */
  /* figure how much memory we need */
  true_size = row_size = (nx * c_size + byte_size - 1) / byte_size;	/* check */
  row_size = (row_size % 2) ? row_size + 1 : row_size; 	/* round up */
  
  no_bytes = ny * row_size;
  /* zero the memory in case of incomplete lists */
  if (!(new_ptr = allocate_mem(no_bytes, 1))) return(2);
  
  /* now read in the cell rows */
  my_ptr = new_ptr;
  for (i=0; (i<ny) && (*dat_ptr); ++i) {
    get_clist(&dat_ptr, my_ptr, nx, a2->c_s_mode, l_col_prec);
    my_ptr += row_size;
  }
  
  
  may_list(stderr, 
	   "cell array (%d,%d,%d,%d,%d,%d), size %d,%d, prec %d, mode %d\n",
	   cp[0], cp[1], cq[0], cq[1], cr[0], cr[1], nx, ny, l_col_prec,
	   rep_mode);
  
  if (dev_func) ret = (*dev_func) (cp, cq, cr, nx, ny, l_col_prec, 
				   new_ptr, rep_mode, no_bytes);
  if ((new_ptr) && (no_bytes > 0)) free(new_ptr);
  return(ret);
}
/* generalised drawing primitive */
/* format is identifier, no_pairs, list of points, set of strings */
static do_g_d_p(dat_ptr, plen, dev_func)
     char *dat_ptr;
     int plen, (*dev_func)();
{
  int ret = 1;
  int gdp_id, no_pairs, *x_ptr, *y_ptr, i, x, y;
  char *data_record;
  
  
  gdp_id = cc_int(&dat_ptr);
  
  no_pairs = cc_int(&dat_ptr);
  
  /* first get the memory */
  if (no_pairs > wk_ar_size) {
    x_ptr = intalloc(no_pairs * bytes_per_word);
    y_ptr = intalloc(no_pairs * bytes_per_word);
  }
  else {
    x_ptr = x_wk_ar;
    y_ptr = y_wk_ar;
  }
  /* now grab the data */
  no_pairs = 0;
  while (!is_term(*dat_ptr)){
    get_vpoint(&dat_ptr, &x, &y);
    x_ptr[no_pairs] = newx(x, y);
    y_ptr[no_pairs] = newy(x, y);
    ++no_pairs;
  }
  data_record = cc_str(&dat_ptr, NULL);
  
  if (list_cgm) {
    (void) fprintf(stderr, 
		   "generalised drawing primitive %d, %d points, data record = [%s]\n", 
		   gdp_id, no_pairs, data_record);
    for (i=0; i<no_pairs; ++i) 
      (void) fprintf(stderr, "(%d, %d)\n", x_ptr[i], y_ptr[i]);
  }
  if (dev_func) ret = (*dev_func) 
    (gdp_id, no_pairs, x_ptr, y_ptr, data_record);
  
  if (x_ptr != x_wk_ar) free(x_ptr);
  if (y_ptr != y_wk_ar) free(y_ptr);
  
  return(ret);
}
/* do a rectangle */
static do_rectangle(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1;
  int x1, y1, x2, y2;
  int x_1, y_1, x_2, y_2;
  
  get_vpoint(&dat_ptr, &x_1, &y_1);
  get_vpoint(&dat_ptr, &x_2, &y_2);
  x1 = newx(x_1, y_1);
  y1 = newy(x_1, y_1);
  x2 = newx(x_2, y_2);
  y2 = newy(x_2, y_2);
  
  may_list(stderr, "rectangle = (%d, %d, %d, %d)\n", x1, y1, x2, y2);
  
  ret = (dev_func)? (*dev_func) (x1, y1, x2, y2) :
    em_rectangle(x1, y1, x2, y2);
  return(ret);
}
/* set a circle */
static do_circle(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, x1, y1, r, x, y;
  
  
  get_vpoint(&dat_ptr, &x1, &y1);
  x = newx(x1, y1);
  y = newy(x1, y1);
  
  r = cc_vdc(&dat_ptr);
  may_list(stderr, "circle, center (%d, %d), radius %d\n", x, y, r);
  
  ret = (dev_func) ? (*dev_func) (x, y, r) : em_circle(x, y, r);
  
  return(ret);
}
/* set an arc, get the positions of 1st pt, intermdiate pt, end pt */
static do_c3(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, x_array[6], i, first_array[6];
  
  for (i=0; i<6; i += 2)
    get_vpoint(&dat_ptr, first_array + i, first_array + i + 1 );
  
  for (i=0; i<3; ++i) {
    x_array[2 * i] = newx(first_array[2 * i], first_array[2*i+1]);
    x_array[2 * i + 1] = newy(first_array[2 * i], first_array[2*i+1]);
  }
  
  may_list(stderr, "arc thru (%d, %d) (%d, %d) (%d, %d)\n",
	   x_array[0], x_array[1], x_array[2], x_array[3], 
	   x_array[4], x_array[5]);
  ret = (dev_func) ? (*dev_func) (x_array) : em_c3(x_array);
  return(ret);
}
/* set a closed arc, get the positions of 1st pt, intermdiate pt, end pt */
static do_c3_close(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, x_array[6], i, first_array[6];
  enum boolean chord;
  char buffer[max_str];
  for (i=0; i<6; i += 2)
    get_vpoint(&dat_ptr, first_array + i, first_array + i + 1 );
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "CHORD")) chord = on;
  else
    if (cc_same(buffer, "PIE")) chord = off;
    else {
      fprintf(stderr, "unrecognised token in do_c3_close %s\n", buffer);
      return(2);
    }
  
  for (i=0; i<3; ++i) {
    x_array[2 * i] = newx(first_array[2 * i], first_array[2*i+1]);
    x_array[2 * i + 1] = newy(first_array[2 * i], first_array[2*i+1]);
  }
  
  may_list(stderr, "closed arc thru (%d, %d) (%d, %d) (%d, %d), %d\n",
	   x_array[0], x_array[1], x_array[2], x_array[3], 
	   x_array[4], x_array[5], (int) chord);
  ret = (dev_func) ? (*dev_func) (x_array, chord) : 
    em_c3_close(x_array, chord);
  return(ret);
}
/* set an arc, ends specified by vectors */
static do_c_centre(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, vec_array[4], i, x, y, r, x1, y1;
  
  get_vpoint(&dat_ptr, &x1, &y1);
  for (i=0; i<4; i += 2) get_vpoint(&dat_ptr, vec_array + i, 
				    vec_array + i + 1);
  r = cc_vdc(&dat_ptr);
  x = newx(x1, y1);
  y = newy(x1, y1);
  
  may_list(stderr, "circle centre (%d, %d) (%d, %d) (%d, %d), %d\n",
	   x, y, vec_array[0], vec_array[1], 
	   vec_array[2], vec_array[3], r);
  ret = (dev_func) ? (*dev_func) (x, y, vec_array, r) : 
    em_c_centre(x, y, vec_array, r);
  return(ret);
}
/* set an arc, ends specified by vectors, close it */
static do_c_c_close(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, vec_array[4], i, x, y, r, x1, y1;
  enum boolean chord;
  char buffer[max_str];
  
  get_vpoint(&dat_ptr, &x1, &y1);
  x = newx(x1, y1);
  y = newy(x1, y1);
  
  for (i=0; i<4; i += 2) get_vpoint(&dat_ptr, vec_array + i, 
				    vec_array + i + 1);
  r = cc_vdc(&dat_ptr);
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "CHORD")) chord = on;
  else
    if (cc_same(buffer, "PIE")) chord = off;
    else {
      fprintf(stderr, "unrecognised token in do_c_c_close %s\n", buffer);
      return(2);
    }
  may_list(stderr, 
	   "closed circle centre (%d, %d) (%d, %d) (%d, %d), %d, %d\n",
	   x, y, vec_array[0], vec_array[1], 
	   vec_array[2], vec_array[3], r, (int) chord);
  ret = (dev_func) ? (*dev_func) (x, y, vec_array, r, chord) : 
    em_c_c_close(x, y, vec_array, r, chord);
  return(ret);
}
/* set an ellipse, specify centre, two conjugate diameters (see the book !) */
static do_ellipse(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, pt_array[6], i;
  
  for (i=0; i<6; i += 2)
    get_vpoint(&dat_ptr, pt_array + i, pt_array + i + 1);
  
  may_list(stderr, "ellipse (%d, %d), (%d, %d), (%d, %d)\n",
	   pt_array[0], pt_array[1], pt_array[2], 
	   pt_array[3], pt_array[4], pt_array[5]);
  if (dev_func) ret = (*dev_func) (pt_array);
  return(ret);
}
/* set an elliptical arc, specify centre two conjugate diameters end vectors */
static do_ell_arc(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, pt_array[6], vec_array[4], i;
  
  for (i=0; i<6; i += 2)
    get_vpoint(&dat_ptr, pt_array + i, pt_array + i + 1);
  
  for (i=0; i<4; i += 2)
    get_vpoint(&dat_ptr, vec_array + i, vec_array + i + 1);
  
  may_list(stderr,
	   "elliptical arc (%d, %d), (%d, %d), (%d, %d), (%d,%d), (%d, %d)\n",
	   pt_array[0], pt_array[1], pt_array[2], 
	   pt_array[3], pt_array[4], pt_array[5],
	   vec_array[0], vec_array[1], vec_array[2], vec_array[3]);
  if (dev_func) ret = (*dev_func) (pt_array, vec_array);
  return(ret);
}
/* set an elliptical arc, close it */
static do_e_a_close(dat_ptr, dev_func)
     char *dat_ptr;
     int(*dev_func)();
{
  int ret = 1, pt_array[6], vec_array[4], i;
  enum boolean chord;
  char buffer[max_str];
  
  for (i=0; i<6; i += 2)
    get_vpoint(&dat_ptr, pt_array + i, pt_array + i + 1);
  
  for (i=0; i<4; i += 2)
    get_vpoint(&dat_ptr, vec_array + i, vec_array + i + 1);
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "CHORD")) chord = on;
  else
    if (cc_same(buffer, "PIE")) chord = off;
    else {
      fprintf(stderr, "unrecognised token in do_e_a_close %s\n", buffer);
      return(2);
    }
  
  may_list(stderr,
	   "elliptical arc close (%d, %d), (%d, %d), (%d, %d), (%d,%d), (%d, %d), %d\n",
	   pt_array[0], pt_array[1], pt_array[2], 
	   pt_array[3], pt_array[4], pt_array[5],
	   vec_array[0], vec_array[1], vec_array[2], vec_array[3], 
	   (int) chord);
  if (dev_func) ret = (*dev_func) (pt_array, vec_array, chord);
  return(ret);
}

/* now the class5 functions */
/* set the line bundle index */
static s_lbindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_lbindex;
  new_lbindex = cc_int(&dat_ptr);
  may_list(stderr, "changing line bundle index from %d to %d\n",
	   (int) a5->l_b_index, new_lbindex);
  a5->l_b_index = new_lbindex;
  
  if (dev_func) ret = (*dev_func) (a5->l_b_index);
  
  return(ret);
}
/* set the line type */
static s_l_type(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_l_type;
  new_l_type = cc_int(&dat_ptr);
  may_list(stderr, "changing line type from %d to %d\n",
	   (int) a5->line_type, new_l_type);
  a5->line_type = (enum line_enum) new_l_type;
  
  if (dev_func) ret = (*dev_func) (a5->line_type);
  
  return(ret);
}
/* set the line width */
static s_l_width(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  float rmul;
  int new_l_width;
  
  if (!dev_info->d_l_width) dev_info->d_l_width = 1;	/* safety */
  
  if (a2->l_w_s_mode == absolute) {
    new_l_width = cc_vdc(&dat_ptr) + 0.5;
    may_list(stderr, "changing abs line width from %d to %d\n",
	     a5->line_width.i, new_l_width);
    a5->line_width.i = new_l_width;
  } else if (a2->l_w_s_mode == scaled) {
    rmul = cc_real(&dat_ptr);
    may_list(stderr, 
	     "changing rel line width from %.4f to %.4f\n",
	     a5->line_width.r, rmul);
    a5->line_width.r = rmul;
    a5->line_width.i = rmul * dev_info->d_l_width;
  } else burp(stderr, "illegal line spec mode = %d\n", a2->l_w_s_mode);
  
  if (dev_func) ret = (*dev_func) 
    (a5->line_width.i, (float) a5->line_width.i  / dev_info->d_l_width);
  
  return(ret);
}
/* set the line colour */
static s_l_colour(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index = -1, ir, ig, ib;
  float r, g, b, *rptr;
  switch ((int) a2->c_s_mode) {
  case (int) i_c_mode: /* indexed mode */
    new_index = cc_int(&dat_ptr);
    if (list_cgm) 
      burp(stderr, "changing line index from %d to %d\n",
	   a5->line_colour.ind, new_index);
    a5->line_colour.ind = new_index;
    rptr = a5->ctab + a5->line_colour.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case (int) d_c_mode: /* direct mode */
    ir = cc_int(&dat_ptr);
    r = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing line colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a5->line_colour.red, a5->line_colour.green, a5->line_colour.blue,
	     r, g, b);
    a5->line_colour.red = r;
    a5->line_colour.green = g;
    a5->line_colour.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
  }
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  return(ret);
}
/* set the marker bundle index */
static s_mbindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index;
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing marker bundle index from %d to %d\n",
	   (int) a5->mk_b_index, new_index);
  a5->mk_b_index = new_index;
  
  if (dev_func) ret = (*dev_func) (a5->mk_b_index);
  
  return(ret);
}
/* set the marker type */
static s_mk_type(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_mk_type;
  new_mk_type = cc_int(&dat_ptr);
  may_list(stderr, "changing marker type from %d to %d\n",
	   a5->mk_type, new_mk_type);
  a5->mk_type = new_mk_type;
  if (dev_func) ret = (*dev_func) (a5->mk_type);
  return(ret);
}
/* set the marker size */
static s_mk_size(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  float rmul;
  int new_mk_size;
  
  if (!dev_info->d_m_size) dev_info->d_m_size = 1;	/* safety */
  
  if (a2->m_s_s_mode == absolute) {
    new_mk_size = cc_vdc(&dat_ptr);
    may_list(stderr, "changing marker size from %d to %d\n",
	     a5->mk_size.i, new_mk_size);
    a5->mk_size.i = new_mk_size;
  } else if (a2->m_s_s_mode == scaled) {
    rmul = cc_real(&dat_ptr);
    may_list(stderr, 
	     "changing rel mkr size from %.4f to %.4f\n",
	     a5->mk_size.r, rmul);
    a5->mk_size.r = rmul;
    a5->mk_size.i = rmul * dev_info->d_m_size;
  } else burp(stderr, "illegal marker size mode = %d\n", a2->m_s_s_mode);
  
  if (dev_func) ret = (*dev_func) 
    (a5->mk_size.i, (float) a5->mk_size.i / dev_info->d_m_size);
  
  return(ret);
}
/* set the marker colour */
static s_mk_colour(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index = -1, ir, ig, ib;
  float r, g, b, *rptr;
  
  
  switch ((int) a2->c_s_mode) {
  case (int) i_c_mode : /* indexed mode */
    new_index = cc_int(&dat_ptr);
    if (list_cgm) 
      burp(stderr, "changing marker index from %d to %d\n",
	   new_index, a5->mk_colour.ind);
    a5->mk_colour.ind = new_index;
    rptr = a5->ctab + a5->mk_colour.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case (int) d_c_mode : /* direct mode */
    ir = cc_int(&dat_ptr);
    r = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing mk colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a5->mk_colour.red, a5->mk_colour.green, a5->mk_colour.blue,
	     r, g, b);
    a5->mk_colour.red = r;
    a5->mk_colour.green = g;
    a5->mk_colour.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
  }
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  return(ret);
}
/* set the text bundle index */
static s_tbindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index;
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing text bundle index from %d to %d\n",
	   (int) a5->t_b_index, new_index);
  a5->t_b_index = new_index;
  
  if (dev_func) ret = (*dev_func) (a5->t_b_index);
  
  return(ret);
}
/* set the text font index */
static s_t_index(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_font_index;
  new_font_index = cc_int(&dat_ptr);
  may_list(stderr, "changing font index from %d to %d\n",
	   a5->t_f_index, new_font_index);
  a5->t_f_index = new_font_index;
  if (dev_func) ret = (*dev_func) (new_font_index);	
  return(ret);
}
/* set the text precision */
static s_t_prec(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  enum txt_enum new_t_prec;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "STRING")) new_t_prec = string;
  else
    if (cc_same(buffer, "CHAR")) new_t_prec = character;
    else
      if (cc_same(buffer, "STROKE")) new_t_prec = stroke;
      else {
	fprintf(stderr, "unrecognised token in s_t_prec %s\n", buffer);
	return(2);
      }
  
  may_list(stderr, "changing text precision from %d to %d\n",
	   (int) a5->t_prec, (int) new_t_prec);
  a5->t_prec = new_t_prec;
  if (dev_func) ret = (*dev_func) (new_t_prec);
  return(ret);
}
/* set the character expansion factor */
static s_c_exp(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  float new_c_exp_fac;
  new_c_exp_fac = cc_real(&dat_ptr);
  may_list(stderr, 
	   "changing character expansion factor from %.4f to %.4f\n",
	   a5->c_exp_fac, new_c_exp_fac);
  a5->c_exp_fac = new_c_exp_fac;
  if (dev_func) ret = (*dev_func) (new_c_exp_fac);
  return(ret);
}
/* set the character space */
static s_c_space(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  float new_c_space;
  new_c_space = cc_real(&dat_ptr);
  may_list(stderr, "changing character space %f to %f\n",
	   a5->c_space, new_c_space);
  a5->c_space = new_c_space;
  if (dev_func) ret = (*dev_func) (a5->c_space);
  return(ret);
}
/* set the text colour */
static s_t_colour(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index = -1, ir, ig, ib;
  float r, g, b, *rptr;
  
  switch ((int) a2->c_s_mode) {
  case 0: /* indexed mode */
    new_index = cc_int(&dat_ptr);
    if (list_cgm) burp(stderr, "changing text colour index from %d to %d\n",
		       a5->text_colour.ind, new_index);
    a5->text_colour.ind = new_index;
    rptr = a5->ctab + a5->text_colour.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case 1: /* direct mode */
    ir = cc_int(&dat_ptr);
    r = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing text colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a5->text_colour.red, a5->text_colour.green, a5->text_colour.blue,
	     r, g, b);
    a5->text_colour.red = r;
    a5->text_colour.green = g;
    a5->text_colour.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
  }
  
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  
  return(ret);
}
/* set character height */
static s_c_height(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_height, dev_height;
  
  new_height = cc_vdc(&dat_ptr) + 0.5;
  if (opt[(int) text_mag].set) new_height *= opt[(int) text_mag].val.r;
  may_list(stderr, "changing character height from %d to %d\n",
	   a5->c_height, new_height);
  a5->c_height = new_height;
  if (dev_func) ret = (*dev_func) (a5->c_height);
  return(ret);
}
/* set the character orientation structure */
static s_c_orient(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  struct orient_struct new_orient;
  
  
  new_orient.x_up = cc_vdc(&dat_ptr);
  new_orient.y_up = cc_vdc(&dat_ptr);
  new_orient.x_base = cc_vdc(&dat_ptr);
  new_orient.y_base = cc_vdc(&dat_ptr);
  
  may_list(stderr, 
	   "changing orientation from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
	   a5->c_orient.x_up,a5->c_orient.y_up,a5->c_orient.x_base,
	   a5->c_orient.y_base,
	   new_orient.x_up,new_orient.y_up,new_orient.x_base,new_orient.y_base);
  
  a5->c_orient.x_up = new_orient.x_up * sy;
  a5->c_orient.y_up = new_orient.y_up * sy;
  a5->c_orient.x_base = new_orient.x_base * sx;
  a5->c_orient.y_base = new_orient.y_base * sx;
  if (dev_func) ret = (*dev_func) (new_orient.x_up, 
				   new_orient.y_up, new_orient.x_base, new_orient.y_base);
  return(ret);
}
/* set the text path */
static s_tpath(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  enum path_enum new_path;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "RIGHT")) new_path = right;
  else
    if (cc_same(buffer, "LEFT")) new_path = left;
    else
      if (cc_same(buffer, "UP")) new_path = up;
      else
	if (cc_same(buffer, "DOWN")) new_path = down;
	else {
	  fprintf(stderr, "unrecognised token in s_tpath %s\n", buffer);
	  return(2);
	}
  may_list(stderr, "changing text path from %d to %d\n",
	   (int) a5->text_path, (int) new_path);
  a5->text_path = new_path;
  if (dev_func) ret = (*dev_func) (new_path);
  return(ret);
}
/* set the text alignment */
static s_t_align(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  struct align_struct new_align;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "NORMHORIZ")) new_align.hor = normal_h;
  else
    if (cc_same(buffer, "LEFT")) new_align.hor = left_h;
    else
      if (cc_same(buffer, "CTR"))  new_align.hor = center_h;
      else
	if (cc_same(buffer, "RIGHT"))new_align.hor = right_h;
	else
	  if (cc_same(buffer, "CONTHORIZ"))new_align.hor = cont_h;
	  else {
	    fprintf(stderr, "unrecognised token in s_t_align %s\n", buffer);
	    return(2);
	  }
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "NORMVERT")) new_align.ver = normal_v;
  else
    if (cc_same(buffer, "TOP")) new_align.ver = top_v;
    else
      if (cc_same(buffer, "CAP"))  new_align.ver = cap_v;
      else
	if (cc_same(buffer, "HALF"))new_align.ver = half_v;
	else
	  if (cc_same(buffer, "BASE"))new_align.ver = base_v;
	  else
	    if (cc_same(buffer, "BOTTOM"))new_align.ver = bottom_v;
	    else
	      if (cc_same(buffer, "CONTVERT"))new_align.ver = cont_v;
	      else {
		fprintf(stderr, "unrecognised token in s_t_align %s\n", buffer);
		return(2);
	      }
  
  new_align.cont_hor = cc_real(&dat_ptr);
  new_align.cont_ver = cc_real(&dat_ptr);
  
  may_list(stderr, 
	   "changing text alignment from (%d,%d,%f,%f) to (%d,%d,%f,%f)\n",
	   a5->text_align.hor, a5->text_align.ver, a5->text_align.cont_hor,
	   a5->text_align.cont_ver, new_align.hor, new_align.ver,
	   new_align.cont_hor, new_align.cont_ver);
  a5->text_align.hor = new_align.hor;
  a5->text_align.ver = new_align.ver;
  a5->text_align.cont_hor = new_align.cont_hor;
  a5->text_align.cont_ver = new_align.cont_ver;
  
  if (dev_func) ret = (*dev_func) (a5->text_align.hor, 
				   a5->text_align.ver, a5->text_align.cont_hor, 
				   a5->text_align.cont_ver);
  
  return(ret);
}
/* set the character set index */
static s_csindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing character set index from %d to %d\n",
	   a5->c_set_index, new_index);
  a5->c_set_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
/* set the alternate character set index */
static s_acsindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, 
	   "changing alternate character set index from %d to %d\n",
	   a5->a_c_set_index, new_index);
  a5->a_c_set_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
/* set the fill bundle index */
static s_fbindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing fill bundle index from %d to %d\n",
	   a5->f_b_index, new_index);
  a5->f_b_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}

/* set the interior style */
static s_interior_style(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  enum is_enum new_style;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "HOLLOW")) new_style = hollow;
  else
    if (cc_same(buffer, "SOLID"))  new_style = solid_i;
    else
      if (cc_same(buffer, "PAT"))  new_style = pattern;
      else
	if (cc_same(buffer, "HATCH")) new_style = hatch;
	else
	  if (cc_same(buffer, "EMPTY"))  new_style = empty;
	  else {
	    fprintf(stderr, "unrecognised token in s_interior_style %s\n",
		    buffer);
	    return(2);
	  }
  may_list(stderr, "changing interior style from %d to %d\n",
	   (int) a5->int_style, new_style);
  a5->int_style = (enum is_enum) new_style;
  if (dev_func) ret = (*dev_func)(a5->int_style);
  return(ret);
}
/* set the fill colour */
static s_fill_colour(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index = -1, ir, ig, ib;
  float r, g, b, *rptr;
  
  
  switch ((int) a2->c_s_mode) {
  case (int) i_c_mode: /* indexed mode */
    new_index = cc_int(&dat_ptr);
    may_list(stderr, "changing fill index from %d to %d\n",
	     a5->fill_colour.ind, new_index);
    a5->fill_colour.ind = new_index;
    rptr = a5->ctab + a5->fill_colour.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case (int) d_c_mode: /* direct mode */
    ir = cc_int(&dat_ptr);
    r = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing fill colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a5->fill_colour.red, a5->fill_colour.green, a5->fill_colour.blue,
	     r, g, b);
    a5->fill_colour.red = r;
    a5->fill_colour.green = g;
    a5->fill_colour.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
  }
  
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  return(ret);
}
/* set the hatch index */
static s_hindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing hatch index from %d to %d\n",
	   a5->hatch_index, new_index);
  a5->hatch_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
/* set the pattern index */
static s_pindex(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing pattern index from %d to %d\n",
	   a5->pat_index, new_index);
  a5->pat_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
/* set the edge bundle index */
static s_e_b_index(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1, new_index;
  
  new_index = cc_int(&dat_ptr);
  may_list(stderr, "changing edge bundle index from %d to %d\n",
	   a5->e_b_index, new_index);
  a5->e_b_index = new_index;
  if (dev_func) ret = (*dev_func) (new_index);
  return(ret);
}
/* set the edge type */
static s_edge_t(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_l_type;
  new_l_type = cc_int(&dat_ptr);
  may_list(stderr, "changing edge type from %d to %d\n",
	   (int) a5->edge_type, new_l_type);
  a5->edge_type = (enum line_enum) new_l_type;
  
  if (dev_func) ret = (*dev_func) (a5->edge_type);
  
  return(ret);
}

/* set the edge width */
static s_edge_w(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_flag;
  float rmul;
  int new_e_width;
  
  if (!dev_info->d_e_width) dev_info->d_e_width = 1;	/* safety */
  
  if (a2->e_w_s_mode == absolute) {
    new_e_width = cc_vdc(&dat_ptr) + 0.5;
    may_list(stderr, "changing abs edge width from %d to %d\n",
	     a5->edge_width.i, new_e_width);
    a5->edge_width.i = new_e_width;
  } else if (a2->e_w_s_mode == scaled) {
    rmul = cc_real(&dat_ptr);
    may_list(stderr, 
	     "changing rel edge width from %.4f to %.4f\n",
	     a5->edge_width.r, rmul);
    a5->edge_width.r = rmul;
    a5->edge_width.i = rmul * dev_info->d_e_width;
  } else burp(stderr, "illegal edge spec mode = %d\n", a2->e_w_s_mode);
  
  if (dev_func) ret = (*dev_func) 
    (a5->edge_width.i, (float) a5->edge_width.i / dev_info->d_e_width);
  
  return(ret);
}
/* set the edge colour */
static s_edge_c(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_index = -1, ir, ig, ib;
  float r, g, b, *rptr;
  switch ((int) a2->c_s_mode) {
  case (int) i_c_mode : /* indexed mode */
    new_index = cc_int(&dat_ptr);
    if (list_cgm) 
      burp(stderr, "changing edge index from %d to %d\n",
	   a5->edge_colour.ind, new_index);
    a5->edge_colour.ind = new_index;
    rptr = a5->ctab + a5->line_colour.ind * 3;
    r = *rptr;
    g = *++rptr;
    b = *++rptr;
    break;
  case (int) d_c_mode : /* direct mode */
    ir = cc_int(&dat_ptr);
    ir = dcr(ir);
    ig = cc_int(&dat_ptr);
    g = dcg(ig);
    ib = cc_int(&dat_ptr);
    b = dcb(ib);
    may_list(stderr, 
	     "changing edge colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
	     a5->edge_colour.red, a5->edge_colour.green, a5->edge_colour.blue,
	     r, g, b);
    a5->edge_colour.red = r;
    a5->edge_colour.green = g;
    a5->edge_colour.blue = b;
    break;
  default:
    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
  }
  if (dev_func) ret = (*dev_func) (r, g, b, new_index);
  return(ret);
}
/* set the edge visibility */
static s_edge_v(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  enum boolean new_flag;
  char buffer[max_str];
  
  if (!get_token(&dat_ptr, buffer)) return(2);
  if (cc_same(buffer, "ON")) new_flag = on;
  else
    if (cc_same(buffer, "OFF")) new_flag = off;
    else {
      fprintf(stderr, "unrecognised token in s_edge_v %s\n", buffer);
      return(2);
    }
  may_list(stderr, "changing edge visibility from %d to %d\n",
	   (int) a5->edge_vis, (int) new_flag);
  a5->edge_vis = new_flag;
  if (dev_func) ret = (*dev_func)(new_flag);
  return(ret);
}

/* set the fill reference point */
static s_fill_ref(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_fill[2], i, x, y;
  
  get_vpoint(&dat_ptr, &x, &y);
  
  new_fill[0] = newx(x, y);
  new_fill[1] = newy(x, y);
  
  may_list(stderr, "setting fill ref pt to (%d,%d)\n",
	   new_fill[0], new_fill[1]);
  for (i=0;i<2;++i) a5->fill_ref.i[i] = new_fill[i];
  if (dev_func) ret = (*dev_func)(a5->fill_ref.i[0], a5->fill_ref.i[1]);
  return(ret);
}
/* make a pattern table entry */
/* not really implemented yet */
static p_tab_entry(dat_ptr, p_len, dev_func)
     int p_len;
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int index, nx, ny, col_prec, no_bytes, l_col_prec;
  
  index = cc_int(&dat_ptr);
  nx = cc_int(&dat_ptr);
  ny = cc_int(&dat_ptr);
  col_prec = cc_int(&dat_ptr);
  if (a2->c_s_mode == d_c_mode) { 	/* direct */
    l_col_prec = (col_prec) ? col_prec : glbl1->col_prec;
  } else {		/* indexed */
    l_col_prec = (col_prec) ? col_prec : glbl1->col_i_prec;
  }
  
  no_bytes = nx * ny * l_col_prec / byte_size;
  may_list(stderr, "pattern table entry (%d), (%d, %d), %d, %d\n",
	   index, nx, ny, col_prec, no_bytes);
  
  if (dev_func) ret = (*dev_func) (index, nx, ny, col_prec, dat_ptr,
				   no_bytes);
  return(ret);
}

/* set the pattern size */
static s_pat_size(dat_ptr, dev_func)
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int new_size[4], i;
  
  for (i=0;i<4;++i) {
    new_size[i] = cc_vdc(&dat_ptr) + 0.5;
  }
  may_list(stderr, 
	   "changing pattern size from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
	   a5->pat_size.i[0], a5->pat_size.i[1], a5->pat_size.i[2], 
	   a5->pat_size.i[3],
	   new_size[0], new_size[1], new_size[2], new_size[3]);
  for (i=0;i<4;++i) a5->pat_size.i[i] = new_size[i];
  if (dev_func) ret = (*dev_func)(a5->pat_size.i);
  return(ret);
}
/* make a colour table entry */
static c_tab_entry(dat_ptr, p_len, dev_func)
     int p_len;
     int (*dev_func)();
     char *dat_ptr;
{
  int ret = 1;
  int beg_index, i, no_entries, col[3], j,
  max_index, do_mcind();
  float rcol[3];
  
  may_list(stderr, "colour table entry ");
  beg_index = cc_int(&dat_ptr);
  
  may_list(stderr, "beginning index = %d,\n",  beg_index);
  
  no_entries = 0;
  for (i = beg_index; (!is_term(*dat_ptr)) && (i < glbl1->max_c_index);
       ++i) {
    for (j=0; j<3; ++j) {
      col[j] = cc_int(&dat_ptr);
      rcol[j] = dcind(j, col[j], glbl1->c_v_extent);
    }
    
    may_list(stderr, "(%d, %d, %d) = (%.4f, %.4f, %.4f)\n", 
	     col[0], col[1], col[2], rcol[0], rcol[1], rcol[2]);
    for (j=0; j<3; ++j) 
      *(a5->ctab + 3 * i + j) = rcol[j];
    ++no_entries;
    while (*dat_ptr == ' ') ++dat_ptr;
  }
  if (dev_func) ret = (*dev_func) (beg_index, no_entries, a5->ctab);
  return(ret);
}
/* take care of the asps flags */
static do_aspsflags(dat_ptr, p_len, dev_func)
     int p_len;
     int (*dev_func)();
     char *dat_ptr;
{
#define MAX_PAIRS 1024		/* should be enough ! */
  int ret = 1, bundled, no_pairs, flags[2 * MAX_PAIRS], i, j;
  char buffer[max_str], buffer1[max_str];
  
  may_list(stderr, "setting asps flags\n");
  no_pairs = 0;
  while (*dat_ptr == ' ') ++dat_ptr;
  while ((!is_term(*dat_ptr)) && (no_pairs < (MAX_PAIRS-NO_ASPS_FLAGS))){
    if (!get_token(&dat_ptr, buffer)) return(2);
    if (!get_token(&dat_ptr, buffer1)) return(2);
    if (cc_same(buffer1, "INDIV")) bundled = 0;
    else if (cc_same(buffer1, "BUNDLED")) bundled = 1;
    else {
      (void) fprintf(stderr, "unknown flag in cc_aspsflags [%s]\n",
		     buffer1);
      return(2);
    }
    /* now figure out what we have */
    for (i=0; (i<NO_ASPS_FLAGS) && (!cc_same(asps_flags[i], buffer));
	 ++i);
    if (i < NO_ASPS_FLAGS) { 	/* found one */
      flags[2 * no_pairs] = i;
      flags[2 * no_pairs + 1] = bundled;
    } else if (cc_same(buffer, "ALL")) {	/* do them all */
      for (j=0; j<NO_ASPS_FLAGS; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else if (cc_same(buffer, "ALLLINE")) { 	/* all line stuff */
      for (j=0; j<3; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else if (cc_same(buffer, "ALLMARKER")) { 	/* all marker stuff */
      for (j=3; j<6; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else if (cc_same(buffer, "ALLTEXT")) { 	/* all text stuff */
      for (j=6; j<11; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else if (cc_same(buffer, "ALLFILL")) { 	/* all fill stuff */
      for (j=11; j<15; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else if (cc_same(buffer, "ALLEDGE")) { 	/* all edge stuff */
      for (j=15; j<18; ++j) {
	flags[2 * no_pairs] = j;
	flags[2 * no_pairs + 1] = bundled;
	++no_pairs;
      }
    } else {
      (void) fprintf(stderr, "unknown ASPS flag [%s]\n", buffer);
      return(2);
    }
    while (*dat_ptr == ' ') ++dat_ptr;
    ++no_pairs;
  }
  if (dev_func) ret = (*dev_func) (no_pairs, flags);
  return(ret);
}
/* now the class6 functions */
/* do the special command */
static do_escape(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  int escape_id;
  char *str_ptr;
  
  escape_id = cc_int(&dat_ptr);
  str_ptr = cc_str(&dat_ptr, NULL);
  may_list(stderr, "escape id = %d, string = [%s]\n",
	   escape_id, str_ptr);
  
  if (dev_func) ret = (*dev_func) (escape_id, str_ptr);
  
  return(1);
}
/* now the class 7 functions */
/* message command */
static do_message(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  char *str_ptr;
  enum boolean action;
  
  action = (enum boolean) cc_int(&dat_ptr);
  str_ptr = cc_str(&dat_ptr, NULL);
  may_list(stderr, "message, action = %d, string = [%s]\n",
	   (int) action, str_ptr);
  
  if (dev_func) ret = (*dev_func) (action, str_ptr);
  
  return(1);
}
/* application data */
static do_apdata(dat_ptr, dev_func)
     char *dat_ptr;
     int (*dev_func)();
{
  int ret = 1;
  char *str_ptr;
  int id;
  
  id = cc_int(&dat_ptr);
  str_ptr = cc_str(&dat_ptr, NULL);
  may_list(stderr, "apdata, id = %d, string = [%s]\n",
	   id, str_ptr);
  
  if (dev_func) ret = (*dev_func) (id, str_ptr);
  
  return(1);
}
/* take care of getting the next clear text command and its data into memory */
/* need to be careful about recognising quotes, see CGM standard */
get_clear(mem_ptr, mem_size, this_ad)
     int *mem_size;			/* how much memory we have available */
     char *(*mem_ptr);	/* ptr to a ptr */
     struct ad_struct *this_ad;	/* general address structure */
{
static struct ad_struct last_ad;	/* last address */
  char *my_ptr, c, *realloc(), last_quote = '\"';
  int p_len, done, new_size;
  enum {normal, quoting, spacing, commenting} my_state;
  
  my_ptr = *mem_ptr;	/* beginning of allocated memory */
  /* save the present address */
  this_ad->b_ad = last_ad.b_ad;
  this_ad->r_ad = last_ad.r_ad;
  this_ad->offset = last_ad.offset;
  
  my_state = normal;
  done = 0;
  while (!done) {	/* loop until done */
    p_len = my_ptr - *mem_ptr;
    if (p_len >= *mem_size ) {	/* leave room for null */
      new_size = p_len + 1024;
      *mem_ptr = realloc(*mem_ptr, new_size);	/* get more memory */
      if (!*mem_ptr) {
	fprintf(stderr, 
		"couldn't allocate enough memory, aborting\n");
	exit(0);
      }
      else fprintf(stderr, "grabbing more memory, %d -> %d\n",
		   *mem_size, new_size);
      my_ptr = *mem_ptr + p_len;
      *mem_size = new_size;
    }
    /* now stick the next byte in, but may end up overriding it */
    if (!(*my_ptr = c = clear_byte(&last_ad))) return(-1);
    /* now the big decision tree, don't trust C RTL */
    if (my_state == commenting) {	/* middle of comment */
      if is_comment(*my_ptr) my_state = normal; /* end comment */
    } else 
      if is_comment(*my_ptr) {		/* start comment */
	my_state = commenting;
      } else
	if (my_state == quoting) {
	  if (*my_ptr == last_quote) my_state = normal;
	  ++my_ptr;
	} else
	  if is_quote(*my_ptr) {
	    last_quote = *my_ptr;
	    my_state = quoting;
	    ++my_ptr;
	  } else
	    if is_term(*my_ptr) {		/* end of command */
	      *(my_ptr + 1) = '\0';		/* for cleanliness */
	      done = 1;
	    } else
	      if (is_sep(*my_ptr) || (*my_ptr == ',')){
	        if (my_state != spacing) {
		  my_state = spacing;
		  *my_ptr = sep_char;
		  ++my_ptr;
		}
	      } else
		if is_null(*my_ptr) {
		  if (my_state == quoting) ++my_ptr;
		} else 
		  if ((*my_ptr >= 'A') && (*my_ptr <= 'Z')) {
		    if (my_state != normal) my_state = normal;
		    ++my_ptr;	/* everything fine */
		  } else
		    if ((*my_ptr >= 'a') && (*my_ptr <= 'z')) {
		      if (my_state != quoting) *my_ptr += ('A' - 'a');
		      if (my_state != normal) my_state = normal;
		      ++my_ptr;	/* now everything fine */
		    } else
		      if ((*my_ptr == '+') || (*my_ptr == '-') || (*my_ptr == '#')
			  || ((*my_ptr >= '0') && (*my_ptr <= '9'))
			  || (*my_ptr == '(') || (*my_ptr == ')') || (*my_ptr == ',')
			  || (*my_ptr == '.')) {
			if (my_state != normal) my_state = normal;
			++my_ptr;
		      } else
			{ 
			  fprintf(stderr, "illegal character in input (%d)\n", *my_ptr);
			}
  }
  return(my_ptr - *mem_ptr);
}

/* copy in the next string from the clear text file */
/* also step the in pointer forward */
static char *cc_str(in_ptr, out_ptr)
     char **in_ptr, *out_ptr;
{
  char quote_flag, *start_ptr;
  int done;
  static char out_buffer[max_str];
  
  if (!out_ptr) out_ptr = out_buffer;
  start_ptr = out_ptr;
  while (**in_ptr == ' ') ++(*in_ptr);	/* skip spaces */
  quote_flag = *((*in_ptr)++);
  if (!is_quote(quote_flag)) {
    fprintf(stderr, "expecting a quoted string, got %s\n", 
	    (*in_ptr - 1));
    *out_ptr = '\0';
    return(start_ptr);
  }
  done = 0;
  while ((!done) && (**in_ptr)) {
    if (**in_ptr == quote_flag) {
      if (*(*in_ptr + 1) == quote_flag) {
	*out_ptr++ = quote_flag;
	*in_ptr += 2;
      } else {
	*out_ptr = '\0';
	++(*in_ptr);
	done = 1;
      }
    } else {
      *out_ptr++ = *(*in_ptr)++;
    }
  }
  if (!done) {
    fprintf(stderr, "unterminated string constant: %s\n", *in_ptr);
    *out_ptr = '\0';
    return(NULL);
  }
  while ((**in_ptr) && (**in_ptr == ' ')) ++(*in_ptr);
  return(start_ptr);
}
/* scan an input string for a CGM clear text integer, steps the pointer fwd */
static my_scan_int(inptr, outval)
     char **inptr;
     int *outval;
{
  register int is_neg, first_int, ret;
#define NUMBER_SIGN '#'
#define MIN_BASE 2
#define MAX_BASE 16
  
  /* skip spaces */
  while ((**inptr == ' ') && (**inptr)) ++*inptr;
  if (!**inptr) return(0);	/* empty string */
  
  /* is it signed ? */
  if (**inptr == '-') {
    is_neg = 1;
    ++*inptr;
  } else 	if (**inptr == '+') {
    ++*inptr;
    is_neg = 0;
  } else is_neg = 0;
  
  if ((**inptr<'0') || (**inptr>'9')) return(0);	/* no digits */
  /* now get the first (only ?) integer */
  first_int = (is_neg) ? -get_decimal(inptr) : get_decimal(inptr);
  
  /* do we have a base ? */
  if ((**inptr == NUMBER_SIGN) && (first_int <= MAX_BASE) &&
      (first_int >= MIN_BASE)) {	/* based integer */
    ++*inptr;
    return(get_based(inptr, first_int, outval));
  } else {
    *outval = first_int;
    return(1);
  }
}
/* grab a decimal, optimised, assumed setup O.K., steps pointer */
static get_decimal(inptr)
     char **inptr;
{
  register int ret, i;
  
  while (**inptr == '0') ++*inptr;	/* skip zeros */
  ret = 0;
  while ((**inptr>='0') && (**inptr<='9')) {
    for (i=0; (digits[i] != **inptr); ++i);
    ret = 10 * ret + i;
    ++*inptr;
  }
  return(ret);
}
/* grab a based integer, steps pointer */
static get_based(inptr, inbase, outval)
     char **inptr;
     int inbase, *outval;
{
#define max_extend 16
  static char ext_digits[max_extend] = {'0', '1', '2', '3', '4', '5', '6', '7',
					  '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};
  register int ret, i, is_neg;
  
  /* is it signed ? */
  if (**inptr == '-') {
    is_neg = 1;
    ++*inptr;
  } else 	if (**inptr == '+') {
    ++*inptr;
    is_neg = 0;
  } else is_neg = 0;
  
  /* quick check that we have a legit number */
  for (i=0; (i<inbase) && (ext_digits[i] != **inptr); ++i);
  if (i>=inbase) return(0);
  
  ret = 0;
  while (1) {
    for (i=0; (i<inbase) && (ext_digits[i] != **inptr); ++i);
    if (i>=inbase) {
      *outval = (is_neg) ? -ret: ret;
      return(1);
    } else {
      ret = inbase * ret + i;
      ++*inptr;
    }
  }
}
/* get a floating point value */
static my_scan_float(inptr, outval)
     char **inptr;
     float *outval;
{
  register int i, is_neg, first_int, exponent, neg_exp;
  register float fract, pwr10;;
  
  /* is it signed ? */
  if (**inptr == '-') {
    is_neg = 1;
    ++*inptr;
  } else 	if (**inptr == '+') {
    ++*inptr;
    is_neg = 0;
  } else is_neg = 0;
  
  /* do we have a legit number ? */
  if (((**inptr > '9') || (**inptr < '0')) && (**inptr != '.'))
    return(0);
  
  if ((**inptr <='9') && (**inptr >= '0')) {	/* a number here */
    first_int = get_decimal(inptr);
  } else first_int = 0;
  
  
  if (**inptr == '.') {	/* explicit point number or scaled real */
    ++*inptr;
    pwr10 = 10.0;
    fract = 0.0;
    while ((**inptr>='0') && (**inptr<='9')) {
      for (i=0; (digits[i] != **inptr); ++i);
      fract += i / pwr10;
      pwr10 *= 10.0;
      ++*inptr;
    }
    if (**inptr == 'E') {	/* scaled real */
      ++*inptr;
      /* is exponent signed ? */
      if (**inptr == '-') {
	neg_exp = 1;
	++*inptr;
      } else 	if (**inptr == '+') {
	++*inptr;
	neg_exp = 0;
      } else neg_exp = 0;
      exponent = get_decimal(inptr);
      pwr10 = 1.0;
      if (neg_exp) {
	for (i=0; i<exponent; ++i) pwr10 /= 10.0;
      } else {
	for (i=0; i<exponent; ++i) pwr10 *= 10.0;
      }
      *outval = (is_neg) ? - (first_int + fract) * pwr10 : 
	(first_int + fract) * pwr10;
      return(1);
    } else {			/* explicit point */
      *outval = (is_neg) ? -(first_int + fract) : first_int + fract;
      return(1);
    }
  } else if (**inptr == 'E') {	/* scaled real number, no fract */
    ++*inptr;
    /* is exponent signed ? */
    if (**inptr == '-') {
      neg_exp = 1;
      ++*inptr;
    } else 	if (**inptr == '+') {
      ++*inptr;
      neg_exp = 0;
    } else neg_exp = 0;
    exponent = get_decimal(inptr);
    pwr10 = 1.0;
    if (neg_exp) {
      for (i=0; i<exponent; ++i) pwr10 /= 10.0;
    } else {
      for (i=0; i<exponent; ++i) pwr10 *= 10.0;
    }
    *outval = (is_neg) ? - first_int * pwr10 : first_int * pwr10;
    return(1);
  } else {	/* just an integer */
    *outval = (float) (is_neg) ? -first_int : first_int;
    return(1);
  }
  
}
/* get the next token, should be a string */
static int get_token(in_ptr, out_str)
     char **in_ptr, *out_str;
{
  int i;
  while (**in_ptr == ' ') ++*in_ptr;	/* skip spaces */
  for (i=0; (((**in_ptr >= 'A') && (**in_ptr <= 'Z')) ||
	     ((**in_ptr >= '0') && (**in_ptr <= '9'))) && (i < max_str); ++i)
    out_str[i] = *((*in_ptr)++);
  out_str[i] = '\0';
  return(i);
}


/* function to compare two strings, will assume that 1) both in same case
   2) at least one terminated correctly */
static int cc_same(str1, str2)
     char *str1, *str2;
{
  while ((*str1 == *str2) && (*str1) && (*str2)) {
    ++str1;
    ++str2;
  }
  return(*str1 == *str2);
}

/* figure out the cgm binary precision from clear text version */
static int get_prec(new_min, new_max, for_carray)
     int new_min, new_max, for_carray;
{
  int i, start_prec;
  start_prec =  (for_carray) ? 1 : 8;
  /* possible values are 8, 16, 24 and 32, and 1, 2, 4 for carrays */
  if (new_min > new_max) {
    fprintf(stderr, "illegal min, max %d, %d\n", new_min, new_max);
    return(8);
  }
  if (new_max < 0) new_max = -new_max;
  if (-new_min > new_max) new_max = -new_min;
  
  for (i=start_prec; i < 8; i *= 2)
    if (!((~0 << i) & new_max)) return(i);
  
  for (i = 8; i <= 32; i += 8)
    if (!((~0 << i) & new_max)) return(i);
  
  return(32);	/* best we can do (may not be an error) */
}
get_rprec(minreal, maxreal, no_digits, new_fixed, new_exp, new_fract)
     float minreal, maxreal;
     int no_digits, *new_fixed, *new_exp, *new_fract;
{
  int temp, i;
  
  /* for now */
/*
  *new_fixed = 0;
  *new_exp = 9;
  *new_fract = 23;
*/
  *new_fixed = 1;
  *new_exp = 16;
  *new_fract = 16;
  
  return(1);
}
/* clean up a string into standard format */
clean_string(out_ptr)
     char *out_ptr;
{
  char *in_ptr, c;
  
  /* just march thru doing conversions */
  
  in_ptr = out_ptr;
  
  while (c = *(in_ptr++)){
    if (((c >= 'A') && (c <= 'Z'))|| ((c >= '0') && (c <= '9')))
      *(out_ptr++) = c;
    else
      if ((c >= 'a') && (c <= 'z'))
	*(out_ptr++) = c + ('A' - 'a');
      else
	if ((c == ',') || is_sep(c)) *(out_ptr++) = ' ';
  }
  *out_ptr = '\0';
  
  return(1);
}
/* read in a cell list */
get_clist(in_ptr, out_ptr, nx, c_s_mode, col_prec)
     char **in_ptr, *out_ptr;
     enum cs_enum c_s_mode;	/* the colour selection mode, indexed or direct */
     int col_prec;	/* colour precision in bits */
     int nx;
{
  char paren_flag;
  int i, j, k, b_entry;
  unsigned int inval;
  
  b_entry = (col_prec / byte_size); 	/* bytes per entry */
  while (**in_ptr == ' ') ++(*in_ptr);	/* skip spaces */
  
  paren_flag = (**in_ptr == '(');
  if (paren_flag) ++(*in_ptr);
  
  switch (c_s_mode) {
  case d_c_mode:	/* direct colour */
    for (i=0; (i< 3 * nx) && (**in_ptr != ')') && (**in_ptr); ++i) {
      while (**in_ptr == ' ') ++(*in_ptr);	/* skip spaces */
      inval = cc_int(in_ptr);
      for (k=b_entry - 1; k>=0; --k) {
	*(out_ptr + k) = inval & 255;
	inval >>= byte_size;
      }
      out_ptr += b_entry;
      while ((**in_ptr == ' ') || (**in_ptr == ',')) ++(*in_ptr);	/* skip spaces */
    }
    break;
  case i_c_mode:	/* indexed colour */
    for (i=0; (i<nx) && (**in_ptr != ')') && (**in_ptr); ++i) {
      while (**in_ptr == ' ') ++(*in_ptr);	/* skip spaces */
      inval = cc_int(in_ptr);
      for (k=b_entry - 1; k>=0; --k) {
	*(out_ptr + k) = inval & 255;
	inval >>= byte_size;
      }
      out_ptr += b_entry;
      while ((**in_ptr == ' ') || (**in_ptr == ',')) ++(*in_ptr);	/* skip spaces */
    }
    break;
  }
  if (**in_ptr == ')') {
    if (!paren_flag) fprintf(stderr, "unmatched parens in clist !\n");
    ++(*in_ptr);
  }
  while (**in_ptr == ' ') ++(*in_ptr);	/* skip spaces */
  return(1);
}
/* function to return the number of points in a string */
get_pairs(in_ptr)
     char *in_ptr;
{
  int no_pts;
  enum {between, in_number} my_state;
  
  no_pts = 0;
  my_state = between;
  while ((*in_ptr) && (!is_term(*in_ptr))) {
    if (is_sep(*in_ptr) || (*in_ptr == ',') || (*in_ptr == '(')
	| (*in_ptr == ')')) {
      if (my_state == in_number) {
	my_state = between;
	++no_pts;
      }
    } else if (my_state == between) {
      my_state = in_number;
      ++no_pts;
    }
    ++in_ptr;
  }
  return((no_pts + 1) / 2);
}

