/****
   standard_interp.c:  standard interpolation routines for "omni_interp".
   Written by Toby Orloff and Jim Larson
   orloff@poincare.geom.umn.edu
   September 18, 1990
****/

/*
 * Copyright (c) 1990, Geometry Supercomputer Project
 *                     University of Minnesota
 *                     1200 Washington Ave. S
 *                     Minneapolis, MN  55415
 *
 * email address: software@geom.umn.edu
 *
 * This software is copyrighted as noted above.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the authors, who may or may not act on them as they desire.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */

/*	This module contains the built-in interpolation routines for
 *	linear and cubic spline interpolation.  These can be used as
 *	examples for user-written interpolation routines.
 */

#include "omni_interp.h"

/*	Linear interpolation depends only on the surrounding data points:
 *	those indexed by data_pos and data_pos + 1.  It can work with any
 *	data_chunk value by separating the chunk into its components.
 */

void
linear_interp (time, in, out, file_pos, data_pos)
     double time;			/* Time of this interpolation. */
     double **in;			/* Input data. */
     double *out;			/* Interpolated data. */
     int file_pos;			/* Position of this data chunk */
					/* in whole data file. */
     int data_pos;			/* Position of this interpolation */
					/* time among data indices. */
{
  int i;
  double lcontrib, rcontrib;		/* Weighting factors for endpoints. */

  lcontrib = (in_time [data_pos + 1] - time) /
    (in_time [data_pos + 1] - in_time [data_pos]);
  rcontrib = 1.0 - lcontrib;

  for (i = 0; i < data_chunk; i++)
    out [i] = in [data_pos][i] * lcontrib
      + in [data_pos + 1][i] * rcontrib;
}


/*	Cardinal cubic spline interpolation can also be used with an
 *	arbitrary data_chunk value.  For information on cubic spline
 *	interpolation, see Bartels, Beatty, and Barsky (1987),
 *	 _An_Introduction_to_Splines_For_Use_in_Computer_Graphics_
 *	&_Geometry_Modeling_, Morgan Kaufman, Los Angeles.
 */

void
cardinal_interp (time, in, out, file_pos, data_pos)
     double time;
     double **in;
     double *out;
     int file_pos;			/* Unused. */
     int data_pos;
{
  double lderiv, rderiv;		/* Left and right derivatives. */
  double len;				/* Length of this interval. */
  double x, x2, x3;
  int i;

  if (time == in_time [data_pos])
    {
      for (i = 0; i < data_chunk; i++)
	out [i] = in [data_pos][i];
      return;
    }

  len = in_time [data_pos + 1] - in_time [data_pos];
  x = (time - in_time [data_pos]) / len;
  x2 = x * x;
  x3 = x2 * x;

  for (i = 0; i < data_chunk; i++)
    {
      if (data_pos == 0)		/* If we're at the leftmost segment  */
	{
	  rderiv = (in [data_pos + 2][i] - in [data_pos][i])
	    / (in_time [data_pos + 2] - in_time [data_pos]);
	  lderiv = (3 * (in [data_pos + 1][i] - in [data_pos][i]) / len
		    - rderiv) / 2.0;
	}
      else if (data_pos == num_in - 2)	/* ... rightmost segment. */
	{
	  lderiv = (in [data_pos + 1][i] - in [data_pos - 1][i])
	    / (in_time [data_pos + 1] - in_time [data_pos - 1]);
	  rderiv = (3 * (in [data_pos + 1][i] - in [data_pos][i]) / len
		    - lderiv) / 2.0;
	}
      else
	{
	  lderiv = (in [data_pos + 1][i] - in [data_pos - 1][i])
	    / (in_time [data_pos + 1] - in_time [data_pos - 1]);
	  rderiv = (in [data_pos + 2][i] - in [data_pos][i])
	    / (in_time [data_pos + 2] - in_time [data_pos]);
	}

      out [i] = in [data_pos][i] * (2*x3 - 3*x2 + 1)
	+ in [data_pos + 1][i] * (-2*x3 + 3*x2)
	  + lderiv * (x3 - 2*x2 + x)
	    + rderiv * (x3 - x2);
    }
}


      
      
