/****
   mn.c:  The main module for "omni_interp" 
   Written by Toby Orloff
   orloff@poincare.geom.umn.edu
   July 10, 1990
****/

/*
 * Copyright (c) 1990, Geometry Supercomputer Project
 *                     University of Minnesota
 *                     1200 Washington Ave. S
 *                     Minneapolis, MN  55415
 *
 * email address: software@geom.umn.edu
 *
 * This software is copyrighted as noted above.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the authors, who may or may not act on them as they desire.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */

#include <stdio.h>
#include "types.h"
#include "globals.c"

double atof ();

int stdoutflag = 0;			/* Direct output to stdout instead */
					/* of output files. */


/*	The main routine calls other routines to initialize global
 *	variables, process the command line, and to process and interpolate
 *	all files.
 */

main (argc, argv)
     int argc;
     char *argv[];
{
  InitializeInterp ();
#ifdef USER_FUN
  global_init ();
#endif /* USER_FUN */
  ProcessInterpCommandLine (&argc, argv, 0);

  if (argc < 3)
    {
      usage ();
      exit (0);
    }

  /* Massage file_chunk so that it is a multiple of data_chunk. */
  if (file_chunk % data_chunk)
    file_chunk += data_chunk - (file_chunk % data_chunk);

  CreateFileArrays (argc, argv);

#ifdef USER_FUN
  global_begin ();
#endif /* USER_FUN */

  switch (type)			/* Type of file */
    {
    case ASCII:
      SkipAndPrintAsciiLines ();
      ProcessAsciiFiles ();
      break;
    case BINARY:
      SkipAndPrintBinaryBytes ();
      ProcessBinaryFiles ();
      break;
    }

#ifdef USER_FUN
  global_end ();
#endif /* USER_FUN */
}


/*	This routine creates arrays of descriptors for all files.
 *	A file record contains:
 *		the type of data (currently, only type DOUBLE),
 *		the time parameter associated with the file,
 *		its index relative to the input files [see below],
 *		the current offset into the file,
 *	and
 *		the name of the file.
 *	If the standard output option is used, output file descriptors
 *	are still initialized to record the time information.
 */

CreateFileArrays (argc, argv)
     int argc;
     char *argv[];
{
  struct file_entry *pf;
  int i, numdig;			/* (Number of digits in index #'s) */

  /* Set up in_file_array. */

  pf = ((struct file_entry *)
	malloc (sizeof (struct file_entry) * num_in));
  if (!pf)
    MemError ();

  for (i = 0; i < num_in; i++)
    {
      pf[i].type = DOUBLE;
      pf[i].time = in_time[i];
      pf[i].index = i;			/* Index for in_files is simple. */
      pf[i].offset = 0L;
      strcpy (pf[i].name, argv[i+1]);
    }
  in_file_array = pf;

  /* Standard output stream option. */

  if (!strcmp(argv[argc - 1], "-"))
    stdoutflag = 1;

  /* Set up out_file_array. */

  pf = (struct file_entry *)
    malloc (sizeof (struct file_entry) * num_out);
  if (!pf)
    MemError ();

  /* Yes, it's ugly.  But it's probably faster than using the math library. */

  if (num_out < 11)
    numdig = 1;
  else if (num_out < 101)
    numdig = 2;
  else if (num_out < 1001)
    numdig = 3;
  else if (num_out < 10001)
    numdig = 4;
  else					/* Ridiculous number of files. */
    numdig = 17;			/* 42?  69?  Insert your favorite! */

  for (i = 0; i < num_out; i++)
    {
      pf[i].type = DOUBLE;
      pf[i].time = out_time [i];
      pf[i].offset = 0L;
      if (strchr (argv[argc - 1], '%'))
	sprintf (pf[i].name, argv[argc - 1], i);
      else if (suffix [0] != '\0')
	sprintf (pf[i].name, "%s.%0*d.%s", argv [argc-1], numdig, i, suffix);
      else
	sprintf (pf[i].name, "%s.%0*d", argv [argc-1], numdig, i);

      /* The indices of the input file are just their chronological order.
	 The indices of the output files follow these rules:
	   1) if an output file has the same time parameter as one of
	      the input files, it gets the corresponding index,
	   2) otherwise, find the interval on the time line closest to
	      the output time.  The index will be the index of the input
	      file on the left end of this interval.
	 These rules cover assigning indices to output times outside the
	 range of input values. */
	      
      for (pf[i].index = 0;
	   (in_time[pf[i].index + 1] <= pf[i].time)
	   && (pf[i].index < num_in - 2);
	   pf[i].index++)
	if (in_time[pf[i].index + 1] == pf[i].time)
	  {
	    pf[i].index++;
	    break;
	  }
    }
  out_file_array = pf;
}


/*	This function opens all output files and copies the specified
 *	header, if any, into them.
 */

SkipAndPrintBinaryBytes ()
{
  char *buffer;
  FILE *infile, *outfile, *prototype;
  int i;

  if (skip <= 0)			/* Simply open outfiles and return. */
    {
      if (stdoutflag)			/* Do nothing for standard output. */
	return;
      for (i = 0; i < num_out; i++)
	{
	  outfile = fopen (out_file_array[i].name, "w");
	  if (!outfile)
	    OpenError (out_file_array[i].name);
	  fclose (outfile);
	}
      return;
    }

  /* Set appropriate offsets in in_files. */

  for (i = 0; i < num_in; i++)
    {
      infile = fopen (in_file_array[i].name, "r");
      if (!infile)
	OpenError (in_file_array[i].name);
      in_file_array [i].offset = skip;
      fclose (infile);
    }

  /* If stdout, have no headers.  [Is this the RIGHT THING?  Maybe.] */

  if (stdoutflag)
    return;

  /* Copy header of first file into all output files. */

  buffer = (char *) malloc (skip);	/* Buffer for header information. */
  if (!buffer)
    MemError ();

  prototype = fopen (in_file_array[0].name, "r");
  if (fread ((void *) buffer, 1, skip, prototype) != skip)
    ReadError (in_file_array[0].name, prototype);

  for (i = 0; i < num_out; i++)
    {
      outfile = fopen (out_file_array[i].name, "w");
      if (!outfile)
	OpenError (out_file_array[i].name);
      fwrite ((void *) buffer, 1, skip, outfile);
      fclose (outfile);
    }
  free (buffer);
}


/*	This function does the same thing for ascii files.
 */

SkipAndPrintAsciiLines ()
{
  int i, total;
  FILE *infile, *outfile, *prototype;
  char ch;

  /* Set appropriate offset in all input files. */

  for (i = 0; i < num_in; i++)
    {
      infile = fopen (in_file_array[i].name, "r");
      if (!infile)
	OpenError (in_file_array[i].name);
      total = 0;
      while (total < skip)
	{
	  if (fscanf (infile, "%c", &ch) == EOF) /* Change to getc() later. */
	    {
	      fprintf (stderr,
		       "%s:  End of file reached in %s before %d lines.\n",
		       progname, in_file_array[i].name, skip);
	      exit (4);
	    }
	  if (ch == '\r' || ch == '\n')	/* End of line character. */
	    total++;
	}
      in_file_array[i].offset = ftell (infile);
      fclose (infile);
    }

  /* For stdout option, print no headers. */

  if (stdoutflag)
    return;

  /* Print out skipped lines in each outfile. */

  prototype = fopen (in_file_array[0].name, "r");

  for (i = 0; i < num_out; i++)
    {
      rewind (prototype);

      outfile = fopen (out_file_array[i].name, "w");
      if (!outfile)
	OpenError (out_file_array[i].name);

      total = 0;
      while (total < skip)
	{
	  fscanf (prototype, "%c", &ch);
	  putc (ch, outfile);

	  if (ch == '\r' || ch == '\n')
	    total++;
	}
      fclose (outfile);
    }
  fclose (prototype);
}


/*	To process files, we read in data a file_chunk at a time and
 *	pass it, by data_chunks, to the interpolation routine.  The
 *	interpolated values are then stored in the output files and
 *	a new file_chunk of data is read in.
 */


ProcessBinaryFiles ()
{
  char *data;			/* Binary data buffer. */
  FILE *infile, *outfile;	/* Current open input and output files. */
  int in_values;		/* Actual number of values read in. */
  int position;			/* Current position in data files. */
  int done;			/* Completion flag. */
  int file_pos = 0;		/* Ordinal position in data file. */
  int times_around = 0;		/* Number of file_chunks processed. */
  int i, j, k;
  void (*interp) ();		/* Pointer to interpolation function. */
  void linear_interp (), cardinal_interp ();
#ifdef USER_FUN
  void user_interp ();
#endif /* USER_FUN */
  double **in_data_array, *out_data_array;
  double **in_data_ref, *out_data_ref;

  /* Allocate space for data. */

  data = (char *) malloc (sizeof (char) * file_chunk);
  if (!data)
    MemError ();

  in_data_array = (double **) malloc (sizeof (double *) * num_in);
  if (!in_data_array)
    MemError ();
  for (i = 0; i < num_in; i++)
    {
      in_data_array[i] = (double *) malloc (sizeof (double) * file_chunk);
      if (!in_data_array[i])
	MemError ();
    }

  out_data_array = (double *) malloc (sizeof (double) * file_chunk);
  if (!out_data_array)
    MemError ();

  /* Allocate data pointers. */

  in_data_ref = (double **) malloc (sizeof (double *) * num_in);
  if (!in_data_ref)
    MemError ();

  /* Select interpolation style. */

  switch (style)
    {
    case LINEAR:
      interp = linear_interp;
      break;
    case CARDINAL:
      interp = cardinal_interp;
      break;
    default:
#ifdef USER_FUN
      interp = user_interp;
#else /* not USER_FUN */
      fprintf (stderr, "%s:  interpolation type not supported!\n");
      exit (5);
#endif /* not USER_FUN */
      break;
    }

  /* Read and process data a file_chunk at a time. */

  done = 0;
  position = -file_chunk;

  while (!done)
    {
      position += file_chunk;

      /* Read next file_chunk from input files. */

      for (i = 0; i < num_in; i++)
	{
	  infile = fopen (in_file_array[i].name, "r");
	  fseek (infile, in_file_array[i].offset, 0);
	  in_values = fread (data, 1, file_chunk, infile);
				/* Generalize to other item sizes later. */

	  /* Translate raw data to floating-point. */
	  
	  for (j = 0; j < in_values; j++)
	    in_data_array[i][j] = (double) (unsigned char) data[j];

	  fprintf (stderr, "%s:  processing %d values in file %s.\n",
		   progname, in_values, in_file_array[i].name);

	  if (in_values < file_chunk) /* Must be finished reading. */
	    done = 1;
	  else
	    in_file_array[i].offset = ftell (infile);

	  fclose (infile);
	}

      /* Signal possible warning. */

      if (done && times_around > 1 && stdoutflag)
	fprintf (stderr, "%s: Warning!  Output is interleaved!\n", progname);

      /* Interpolate and write for each output file. */

      for (i = 0; i < num_out; i++)
	{
	  /* Reposition data pointers to beginning of arrays. */

	  file_pos = position;
	  for (j = 0; j < num_in; j++)
	    in_data_ref[j] = in_data_array[j];
	  out_data_ref = out_data_array;

	  /* Interpolate an output file. */

	  for (j = 0; j < in_values; j += data_chunk)
	    {
	      file_pos += data_chunk;
	      
	      /* Interpolate for these data. */	      
	      
	      (*interp) (out_file_array[i].time,
			 in_data_ref,
			 out_data_ref,
			 file_pos,
			 out_file_array[i].index);

	      /* Increment data pointers. */

	      for (k = 0; k < num_in; k++)
		in_data_ref[k] += data_chunk;
	      out_data_ref += data_chunk;
	    }

	  /* Write results in output files. */

	  if (!stdoutflag)
	    outfile = fopen (out_file_array[i].name, "a");
	  else
	    outfile = stdout;
	  for (j = 0; j < in_values; j++)
	    if (out_data_array[j] < 0.0)
	      data[j] = (unsigned char) 0;
	    else if (out_data_array[j] > 255.0)
	      data[j] = (unsigned char) 255;
	    else
	      data[j] = (unsigned char) (int) out_data_array[j];
	  fwrite (data, 1, in_values, outfile);
	  if (!stdoutflag)
	    fclose (outfile);
	}			/* Interpolate for next output file. */
      times_around++;
    }				/* Read next file_chunk of data. */
}


ProcessAsciiFiles ()
{
  char strbuffer[80];		/* Buffer for reading strings. */
  char ch;
  FILE *infile, *outfile, *prototype;
  int in_values;		/* Actual number of values read in. */
  int position;			/* Current position in data files. */
  int total;			/* Counter for dealing with ascii stuff. */
  int done;			/* Completion flag. */
  int file_pos = 0;		/* Ordinal position of data in file. */
  int times_around;		/* Number of file_chunks processed. */
  int i, j, k;
  long offset;			/* Offset marker for prototype. */
  void (*interp) ();		/* Pointer to interpolation function. */
  void linear_interp (), cardinal_interp ();
#ifdef USER_FUN
  void user_interp ();
#endif /* USER_FUN */
  double **in_data_array, *out_data_array;
  double **in_data_ref, *out_data_ref;

  /* Allocate space for data. */

  in_data_array = (double **) malloc (sizeof (double *) * num_in);
  if (!in_data_array)
    MemError ();
  for (i = 0; i < num_in; i++)
    {
      in_data_array[i] = (double *) malloc (sizeof (double) * file_chunk);
      if (!in_data_array[i])
	MemError ();
    }

  out_data_array = (double *) malloc (sizeof (double) * file_chunk);
  if (!out_data_array)
    MemError ();

  /* Allocate data pointers. */

  in_data_ref = (double **) malloc (sizeof (double *) * num_in);
  if (!in_data_ref)
    MemError ();

  prototype = fopen (in_file_array[0].name, "r");
  fseek (prototype, in_file_array[0].offset, 0);

  /* Select interpolation style. */

  switch (style)
    {
    case LINEAR:
      interp = linear_interp;
      break;
    case CARDINAL:
      interp = cardinal_interp;
      break;
    default:
#ifdef USER_FUN
      interp = user_interp;
#else /* not USER_FUN */
      fprintf (stderr, "%s:  interpolation type not supported!\n");
      exit (5);
#endif /* not USER_FUN */
      break;
    }

  /* Read and process data a file_chunk at a time. */

  done = 0;
  position = -file_chunk;

  while (!done)
    {
      position += file_chunk;
      offset = ftell (prototype);

      /* Read next file_chunk from input files. */

      for (i = 0; i < num_in; i++)
	{
	  infile = fopen (in_file_array[i].name, "r");
	  fseek (infile, in_file_array[i].offset, 0);

	  in_values = 0;
	  while (in_values < file_chunk)
	    {
	      if (fscanf (infile, "%s", strbuffer) == EOF)
		break;

	      if (is_a_float (strbuffer))
		{
		  in_data_array[i][in_values] = atof (strbuffer);
		  in_values++;
		}
	    }

	  fprintf (stderr, "%s:  processing %d values in file %s.\n",
		   progname, in_values, in_file_array[i].name);

	  if (in_values < file_chunk) /* Must have finished reading. */
	    done = 1;
	  else
	    in_file_array[i].offset = ftell (infile);

	  fclose (infile);
	}

      /* Signal possible warning. */

      if (done && times_around > 1 && stdoutflag)
	fprintf (stderr, "%s: Warning!  Output is interleaved!\n", progname);

      /* Interpolate and write for each output file. */

      for (i = 0; i < num_out; i++)
	{
	  /* Reposition data pointers to beginning of arrays. */

	  file_pos = position;
	  for (j= 0; j < num_in; j++)
	    in_data_ref[j] = in_data_array[j];
	  out_data_ref = out_data_array;

	  /* Interpolate an output file. */

	  for (j = 0; j < in_values; j += data_chunk)
	    {
	      file_pos += data_chunk;

	      (*interp) (out_file_array[i].time,
			 in_data_ref,
			 out_data_ref,
			 file_pos,
			 out_file_array[i].index);

	      /* Increment data pointers. */

	      for (k = 0; k < num_in; k++)
		in_data_ref[k] += data_chunk;
	      out_data_ref += data_chunk;
	    }

	  /* Write results in output files. */

	  fseek (prototype, offset, 0);
	  if (!stdoutflag)
	    outfile = fopen (out_file_array[i].name, "a");
	  else
	    outfile = stdout;

	  total = 0;
	  while (total < in_values)
	    {
	      /* Print whitespace. */

	      if (fscanf (prototype, "%[ \t\r\n]", strbuffer))
		fprintf (outfile, "%s", strbuffer);

	      fscanf (prototype, "%s", strbuffer);
	      if (!is_a_float (strbuffer)) /* Not a floating pt. value. */
		fprintf (outfile, "%s", strbuffer);
	      else		/* Print interpolated value. */
		{
		  fprintf (outfile, "%lf", out_data_array[total]);
		  total++;
		}
	    }

	  if (done)		/* Print remaining character from prototype. */
	    while (fscanf (prototype, "%c", &ch) == 1)
	      putc (ch, outfile);

	  if (!stdoutflag)
	    fclose (outfile);
	}			/* Go on to next output file. */
      times_around++;
    }				/* Read next file_chunk of data. */
}

