/*@(#)uniform.h	2.28 (rl) 92/08/03*/
/**********************************************************
* kaleido
*
*	Kaleidoscopic construction of uniform polyhedra
*
*	Author:
*		Dr. Zvi Har'El,
*		Deptartment of Mathematics,
*		Technion, Israel Institue of Technology,
*		Haifa 32000, Israel.
*		E-Mail: rl@gauss.technion.ac.il
**********************************************************/
/*******************************************************************************
*	List of Uniform Polyhedra and Their Kaleidoscopic Formulae
*	==========================================================
*
*	Each entry contains the following items:
*
*	1)	Polyhedron name.
*	2)	Wythoff symbol (a list of three numbers and a BAR).
*	3)	Coxeter &al. reference figure.
*	4)	Wenninger reference figure.
*	5)	Dual name.
*
*	References:
*
*	Coxeter, H.S.M., Longuet-Higgins, M.S. & Miller, J.C.P.,
*		Uniform polyhedra, Phil. Trans. Royal Soc. London, Ser. A,
*		246 (1953), 401-409.
*	Wenninger, M.J.,
*		"Polyhedron Models", Cambridge University Press, 1971.
*		"Dual Models", Cambridge University Press, 1984.
*******************************************************************************/
#define BAR 0

struct {
	char *name;
	double Wythoff[4];
	int Coxeter, Wenninger;
	char *dual;
} uniform[] = {

/******************************************************************************
*		Dihedral Schwarz Triangles (D5 only)
******************************************************************************/

/* (2 2 5) (D1/5) */

/*1*/	{"pentagonal prism",{2,5,BAR,2},0,0,
		"pentagonal dipyramid"},
/*2*/	{"pentagonal antiprism",{BAR,2,2,5},0,0,
		"pentagonal deltohedron"},

/* (2 2 5/2) (D2/5) */

/*3*/	{"pentagrammic prism",{2,5./2,BAR,2},0,0,
		"pentagrammic dipyramid"},
/*4*/	{"pentagrammic antiprism",{BAR,2,2,5./2},0,0,
		"pentagrammic deltohedron"},

/* (5/3 2 2) (D3/5) */

/*5*/	{"pentagrammic crossed antiprism",{BAR,2,2,5./3},0,0,
		"pentagrammic concave deltohedron"},

/*******************************************************************************
*		Tetrahedral Schwarz Triangles 
*******************************************************************************/

/* (2 3 3) (T1) */

/*6*/	{"tetrahedron",{3,BAR,2,3},15,1,
		"tetrahedron"},
/*7*/	{"truncated tetrahedron",{2,3,BAR,3},16,6,
		"triakistetrahedron"},

/* (3/2 3 3) (T2) */

/*8*/	{"octahemioctahedron",{3./2,3,BAR,3},37,68, /*orientable*/
		"octahemioctacron"},

/* (3/2 2 3) (T3) */

/*9*/	{"tetrahemihexahedron",{3./2,3,BAR,2},36,67,
		"tetrahemihexacron"},

/******************************************************************************
*		Octahedral Schwarz Triangles
******************************************************************************/

/* (2 3 4) (O1) */

/*10*/	{"octahedron",{4,BAR,2,3},17,2,
		"cube"},
/*11*/	{"cube",{3,BAR,2,4},18,3,
		"octahedron"},
/*12*/	{"cuboctahedron",{2,BAR,3,4},19,11,
		"rhombic dodecahedron"},
/*13*/	{"truncated octahedron",{2,4,BAR,3},20,7,
		"tetrakishexahedron"},
/*14*/	{"truncated cube",{2,3,BAR,4},21,8,
		"triakisoctahedron"},
/*15*/	{"rhombicuboctahedron",{3,4,BAR,2},22,13,
		"deltoidal icositetrahedron"},
/*16*/	{"truncated cuboctahedron",{2,3,4,BAR},23,15,
		"disdyakisdodecahedron"},
/*17*/	{"snub cube",{BAR,2,3,4},24,17,
		"pentagonal icositetrahedron"},

/* (3/2 4 4) (O2b) */

/*18*/	{"small cubicuboctahedron",{3./2,4,BAR,4},38,69,
		"small hexacronic icositetrahedron"},

/* (4/3 3 4) (O4) */

/*19*/	{"great cubicuboctahedron",{3,4,BAR,4./3},50,77,
		"great hexacronic icositetrahedron"},
/*20*/	{"cubohemioctahedron",{4./3,4,BAR,3},51,78,
		"hexahemioctacron"},
/*21*/	{"cubitruncated cuboctahedron",{4./3,3,4,BAR},52,79,
		"tetradyakishexahedron"},

/* (3/2 2 4) (O5) */

/*22*/	{"rhombicuboctahedron",{3./2,4,BAR,2},59,85,
		"great deltoidal icositetrahedron"},
/*23*/	{"small rhombihexahedron",{3./2,2,4,BAR},60,86, /*even*/
		"small rhombihexacron"},

/* (4/3 2 3) (O7) */

/*24*/	{"stellated truncated hexahedron",{2,3,BAR,4./3},66,92,
		"great triakisoctahedron"},
/*25*/	{"great truncated cuboctahedron",{4./3,2,3,BAR},67,93,
		"great disdyakisdodecahedron"},

/* (4/3 3/2 2) (O11) */

/*26*/	{"great rhombihexahedron",{4./3,3./2,2,BAR},82,103, /*even*/
		"great rhombihexacron"},

/******************************************************************************
*		Icosahedral Schwarz Triangles
******************************************************************************/

/* (2 3 5) (I1) */

/*27*/	{"icosahedron",{5,BAR,2,3},25,4,
		"dodecahedron"},
/*28*/	{"dodecahedron",{3,BAR,2,5},26,5,
		"icosahedron"},
/*29*/	{"icosidodecahedron",{2,BAR,3,5},28,12,
		"rhombic triacontahedron"},
/*30*/	{"truncated icosahedron",{2,5,BAR,3},27,9,
		"pentakisdodecahedron"},
/*31*/	{"truncated dodecahedron",{2,3,BAR,5},29,10,
		"triakisicosahedron"},
/*32*/	{"rhombicosidodecahedron",{3,5,BAR,2},30,14,
		"deltoidal hexecontahedron"},
/*33*/	{"truncated icosidodechedon",{2,3,5,BAR},31,16,
		"disdyakistriacontahedron"},
/*34*/	{"snub dodecahedron",{BAR,2,3,5},32,18,
		"pentagonal hexecontahedron"},

/* (5/2 3 3) (I2a) */

/*35*/	{"small ditrigonal icosidodecahedron",{3,BAR,5./2,3},39,70,
		"small triambic icosahedron"},
/*36*/	{"small icosicosidodecahedron",{5./2,3,BAR,3},40,71,
		"small icosacronic hexecontahedron"},
/*37*/	{"small snub icosicosidodecahedron",{BAR,5./2,3,3},41,110,
		"small hexagonal hexecontahedron"},

/* (3/2 5 5) (I2b) */

/*38*/	{"small dodecicosidodecahedron",{3./2,5,BAR,5},42,72,
		"small dodecacronic hexecontahedron"},

/* (2 5/2 5) (I3) */

/*39*/	{"small stellated dodecahedron",{5,BAR,2,5./2},43,20,
		"great dodecahedron"},
/*40*/	{"great dodecahedron",{5./2,BAR,2,5},44,21,
		"small stellated dodecahedron"},
/*41*/	{"great dodecadodecahedron",{2,BAR,5./2,5},45,73,
		"medial rhombic triacontahedron"},
/*42*/	{"truncated great dodecahedron",{2,5./2,BAR,5},47,75,
		"small stellapentakisdodecahedron"},
/*43*/	{"rhombidodecadodecahedron",{5./2,5,BAR,2},48,76,
		"medial deltoidal hexecontahedron"},
/*44*/	{"small rhombidodecahedron",{2,5./2,5,BAR},46,74, /*even*/
		"small rhombidodecacron"},
/*45*/	{"snub dodecadodecahedron",{BAR,2,5./2,5},49,111,
		"medial pentagonal hexecontahedron"},

/* (5/3 3 5) (I4) */

/*46*/	{"ditrigonal dodecadodecahedron",{3,BAR,5./3,5},53,80,
		"medial triambic icosahedron"},
/*47*/	{"great ditrigonal dodecicosidodecahedron",{3,5,BAR,5./3},54,81,
		"great ditrigonal dodecacronic hexecontahedron"},
/*48*/	{"small ditrigonal dodecicosidodecahedron",{5./3,3,BAR,5},55,82,
		"small ditrigonal dodecacronic hexecontahedron"},
/*49*/	{"icosidodecadodecahedron",{5./3,5,BAR,3},56,83,
		"medial icosacronic hexecontahedron"},
/*50*/	{"icositruncated dodecadodecahedron",{5./3,3,5,BAR},57,84,
		"tridyakisicosahedron"},
/*51*/	{"snub icosidodecadodecahedron",{BAR,5./3,3,5},58,112,
		"medial hexagonal hexecontahedron"},

/* (3/2 3 5) (I6b) */

/*52*/	{"great ditrigonal icosidodecahedron",{3./2,BAR,3,5},61,87,
		"great triambic icosahedron"},
/*53*/	{"great icosicosidodecahedron",{3./2,5,BAR,3},62,88,
		"great icosacronic hexecontahedron"},
/*54*/	{"small icosihemidodecahedron",{3./2,3,BAR,5},63,89,
		"small icosihemidodecacron"},
/*55*/	{"small dodecicosahedron",{3./2,3,5,BAR},64,90, /*even*/
		"small dodecicosacron"},

/* (5/4 5 5) (I6c) */

/*56*/	{"small dodecahemidodecahedron",{5./4,5,BAR,5},65,91,
		"small dodecahemidodecacron"},

/* (2 5/2 3) (I7) */

/*57*/	{"great stellated dodecahedron",{3,BAR,2,5./2},68,22,
		"great icosahedron"},
/*58*/	{"great icosahedron",{5./2,BAR,2,3},69,41,
		"great stellated dodecahedron"},
/*59*/	{"great icosidodecahedron",{2,BAR,5./2,3},70,94,
		"great rhombic triacontahedron"},
/*60*/	{"great truncated icosahedron",{2,5./2,BAR,3},71,95,
		"great stellapentakisdodecahedron"},
/*61*/	{"rhombicosahedron",{2,5./2,3,BAR},72,96, /*even*/
		"rhombicosacron"},
/*62*/	{"great snub icosidodecahedron",{BAR,2,5./2,3},73,113,
		"great pentagonal hexecontahedron"},

/* (5/3 2 5) (I9) */

/*63*/	{"small stellated truncated dodecahedron",{2,5,BAR,5./3},74,97,
		"great pentakisdodekahedron"},
/*64*/	{"truncated dodecadodecahedron",{5./3,2,5,BAR},75,98,
		"medial disdyakistriacontahedron"},
/*65*/	{"inverted snub dodecadodecahedron",{BAR,5./3,2,5},76,114,
		"medial inverted pentagonal hexecontahedron"},

/* (5/3 5/2 3) (I10a) */

/*66*/	{"great dodecicosidodecahedron",{5./2,3,BAR,5./3},77,99,
		"great dodecacronic hexecontahedron"},
/*67*/	{"small dodecahemicosahedron",{5./3,5./2,BAR,3},78,100,
		"small dodecahemicosacron"},
/*68*/	{"great dodecicosahedron",{5./2,5./3,3,BAR},79,101, /*even*/
		"great dodecicosacron"},
/*69*/	{"great snub dodecicosidodecahedron",{BAR,5./3,5./2,3},80,115,
		"great hexagonal hexecontahedron"},

/* (5/4 3 5) (I10b) */

/*70*/	{"great dodecahemicosahedron",{5./4,5,BAR,3},81,102,
		"great dodecahemicosacron"},

/* (5/3 2 3) (I13) */

/*71*/	{"great stellated truncated dodecahedron",{2,3,BAR,5./3},83,104,
		"great triakisicosahedron"},
/*72*/	{"great rhombicosidodecahedron",{5./3,3,BAR,2},84,105,
		"great deltoidal hexecontahedron"},
/*73*/	{"great truncated icosidodecahedron",{5./3,2,3,BAR},87,108,
		"great disdyakistriacontahedron"},
/*74*/	{"great inverted snub icosidodecahedron",{BAR,5./3,2,3},88,116,
		"great inverted pentagonal hexecontahedron"},

/* (5/3 5/3 5/2) (I18a) */

/*75*/	{"great dodecahemidodecahedron",{5./3,5./2,BAR,5./3},86,107,
		"great dodecahemidodecacron"},

/* (3/2 5/3 3) (I18b) */

/*76*/	{"great icosihemidodecahedron",{3./2,3,BAR,5./3},85,106,
		"great icosihemidodecacron"},

/* (3/2 3/2 5/3) (I22) */

/*77*/	{"small retrosnub icosicosidodecahedron",{BAR,3./2,3./2,5./2},91,118,
		"small hexagrammic hexecontahedron"},

/* (3/2 5/3 2) (I23) */

/*78*/	{"great rhombidodecahedron",{3./2,5./3,2,BAR},89,109, /*even*/
		"great rhombidodecacron"},
/*79*/	{"great retrosnub icosidodecahedron",{BAR,3./2,5./3,2},90,117,
		"great pentagrammic hexecontahedron"},

/******************************************************************************
*		Last But Not Least
******************************************************************************/

/*80*/	{"great dirhombicosidodecahedron",{3./2,5./3,3,5./2},92,119,
		"great dirhombicosidodecacron"}
};

int last_uniform = sizeof (uniform) / sizeof (uniform[0]);
