

/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*     file:     quotient.c
*
*     Purpose:  Illustration of how to do quotient spaces.
*               This example does a flat torus in R^3.
*               The user needs to define the integer representation
*               used for group elements and three functions to
*               do group operations.  WRAPTYPE is the type of
*               group element representation, currently long int.
*               You must write your own versions of group_wrap(),
*               group_compose(), and group_inverse().
*
*               You don't have the privilege
*               of defining your own datafile syntax, but you can access
*               values of parameters defined in the datafile as
*               web.params[n].value.  Order is order of appearance.
*/

#include "include.h"

/* change torus stuff in following lines to your own names */

char *symmetry_name = "torus";  /* name appearing in datafile. lowercase! */

/*
       Group elements are encoded in four-bit fields, with 3 bits
       representing a signed integer and one overflow bit.
       There is one field for each dimension, with low dimension
       in the low bits.  You are free to encode
       any way you want, but use wrap = 0 for the identity.
*/

/* specific case of torus symmetry representation */
#define TORWRAPBITS 4
#define TORWRAPMASK  7
#define NEGSIGN   4
#define ALLWRAPMASK 0x77777777

/* torus lattice periods */
#define DIMENSION 3
static double periods[DIMENSION][DIMENSION] = {
  { 1.0, 0.0, 0.0 },
  { 0.0, 1.0, 0.0 },
  { 0.0, 0.0, 1.0 }
};

/*******************************************************************
*
*  function: group_wrap
*
*  purpose:  Provide adjusted coordinates for vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*/

void group_wrap(x,y,wrap)
REAL *x;   /* original coordinates */
REAL *y;   /* wrapped coordinates  */
WRAPTYPE wrap;  /* encoded symmetry group element */
{
  int i;        /* which lattice vector */
  int j;        /* space dimension */
  int n;        /* number times wraps around */

  /* copy original coordinates */
  memcpy((char*)y,(char*)x,DIMENSION*sizeof(REAL));

  /* adjust for wrapping */
  for ( i = 0 ; wrap != 0 ; i++, wrap >>= TORWRAPBITS )
    { n = wrap & TORWRAPMASK;  /* extract field */
      if ( n == 0 ) continue;
      if ( n & NEGSIGN ) n = -(TORWRAPMASK + 1 - n);
      for ( j = 0 ; j < DIMENSION ; j++ )
	   y[j] += n*periods[i][j];
    }
}

/********************************************************************
*
*  function: group_compose()
*
*  purpose:  do composition of two group elements
*
*/

WRAPTYPE group_compose(wrap1,wrap2)
WRAPTYPE wrap1,wrap2;  /* the elements to compose */
{
  return (wrap1 + wrap2) & ALLWRAPMASK;
}


/********************************************************************
*
*  function: group_inverse()
*
*  purpose:  return inverse of group element.
*
*/

WRAPTYPE group_inverse(wrap)
WRAPTYPE wrap;  /* the element to invert */
{
  return ((~ALLWRAPMASK)-wrap) & ALLWRAPMASK;
}
  
/*******************************************************************
*
*  function: group_form_pullback
*
*  purpose:  Pull back differential forms at vertices that get
*            wrapped around torus.  Serves as example for user-written
*            symmetry function.
*
*/

void group_form_pullback(x,xform,yform,wrap)
REAL *x;   /* original coordinates */
REAL *xform;  /* result pullback */
REAL *yform;   /* original form input  */
WRAPTYPE wrap;  /* encoded symmetry group element */
{
  int i;

  /* for flat torus, pullback is just copy */
  for ( i = 0 ; i < web.sdim ; i++ )
    xform[i] = yform[i];

}

