
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/**********************************************************
*
*  File: query.c
*
*  Contents:  query interpreter for Surface Evolver
*/

#include "include.h"
#include "express.h"
#include "ytab.h"



int commandverb;
char *cmdptr;

#define MAXHISTORY 20
#define HISTORYSPACE 1000
char *history_space;
int  history_list[MAXHISTORY];

void newcommand(text)
char *text;
{ 
  if ( command(text) )
    /* insert in history list if successful */
    new_history(text);
}

void new_history(text)
char *text;
{ int length;
  int k;

  if ( history_space == NULL ) history_space = mycalloc(HISTORYSPACE,1);
  length = strlen(text)+1;
  if ( length >= HISTORYSPACE ) error("Command too long.\n",RECOVERABLE);
  memmove(history_space+length,history_space,HISTORYSPACE-length);
  strcpy(history_space,text);
  for ( k = MAXHISTORY-2 ; k >= 0 ; k-- )
   { /* adjust pointers in history list */
     history_list[k+1] = history_list[k] + length;
   }
  for ( k = 0 ; k < MAXHISTORY - 1 ; k++ )
     if ( history_list[k+1] >= HISTORYSPACE )
	 break;
  history_space[history_list[k]] = '\0'; /* new sentinel */
}

void old_history(text)
char *text;
{ char *h;
  int k;

  if (text[1] == '!' ) 
     { h = history_space+history_list[0];
       if ( h[0] ) command(h);
       else outstring("History list is empty.\n");
     }
  else for ( k = 0 ;; k++ )
    { h = history_space+history_list[k]; 
      if ( h[0] == '\0' ) error("Command not found in history.\n",RECOVERABLE);
      if ( strncmp(h,text+1,strlen(text+1)) == 0 )
        { command(h);
	  break;
	}
    }
}

int command(text)
char *text;
{
  int retval;

  /* a little kludge */
  if ( strncmp(text,"alice",5) == 0 )
    { alice();
      return 1;
    }
  /* another little kludge */
  if ( strncmp(text,"estimate",8) == 0 )
    { estimate_flag = !estimate_flag;
      outstring(estimate_flag ? "Estimation ON.\n" : "Estimation OFF\n");
      return 1;
    }
  /* still another little kludge */
  if ( strncmp(text,"deturck",7) == 0 )
    { unit_normal_flag = !unit_normal_flag;
      if ( unit_normal_flag )
	{ sprintf(msg,"Enter unit normal weight factor(%g): ",deturck_factor);
	  prompt(msg,msg);
	  sscanf(msg,"%lf",&deturck_factor);
	}
      else
	outstring("Unit normal motion OFF.\n");
      return 1;
    }
  /* yak */
  if ( strncmp(text,"kusner",6) == 0 )
    { kusner_flag = !kusner_flag;
      outstring(kusner_flag ? "Edge square curvature ON.\n" :
			      "Edge square curvature OFF.\n");
      recalc();
      return 1;
    }
  if ( strncmp(text,"sqgauss",7) == 0 )
    { sqgauss_flag = !sqgauss_flag;
      outstring(sqgauss_flag ? "Squared Gaussian curvature ON.\n" :
			      "Squared Gaussian curvature OFF.\n");
      recalc();
      return 1;
    }
  if ( strncmp(text,"stability_test",14) == 0 )
    { stability_test();
      return 1;
    }
  if ( strncmp(text,"autopop",7) == 0 )
    { autopop_flag = !autopop_flag;
      outstring(autopop_flag ? "Autopopping ON.\n" :
			      "Autopopping OFF.\n");
      if ( autopop_flag )
	{ int n;
          if ( web.dimension == STRING )
            sprintf(msg,"Number of vertices popped: %d\n", n = verpop_str());
          else
            sprintf(msg,"Number of vertices popped: %d\n", n = edgepop_film());
          outstring(msg);
          if ( n > 0 ) update_display();
	}
      return 1;
    }
  if ( strncmp(text,"autochop",8) == 0 )
    { autochop_flag = !autochop_flag;
      outstring(autochop_flag ? "Autochopping ON.\n" :
			      "Autochopping OFF.\n");
      if ( autochop_flag )
	{ sprintf(msg,"Enter length cutoff(%g): ",autochop_size);
	  prompt(msg,msg);
	  sscanf(msg,"%lf",&autochop_size);
	}
      return 1;
    }
  if ( strncmp(text,"utest",5) == 0 ) { simplex_delauney_test(); return 1; }
  if ( (strncmp(text,"effective area",14) == 0) 
    || (strncmp(text,"effective_area",14) == 0) )
    { effective_area_flag = !effective_area_flag;
      outstring(effective_area_flag ? "effective_area ON.\n" :
			      "effective_area OFF.\n");
      return 1;
    }
  if ( strncmp(text,"old_area",8) == 0 )
    { old_area_flag = !old_area_flag;
      outstring(old_area_flag ? "old_area ON.\n" :
			      "old_area OFF.\n");
      return 1;
    }

  if ( strncmp(text,"approx_curv",10) == 0 )
    { approx_curve_flag = !approx_curve_flag;
      outstring(approx_curve_flag ? "approx_curvature ON.\n" :
			      "approx_curvature OFF.\n");
      return 1;
    }

  if ((strncmp(text,"runge-kutta",11)==0)||(strncmp(text,"runge_kutta",11)==0))
    { runge_kutta_flag = !runge_kutta_flag;
      outstring(runge_kutta_flag ? "Runge-Kutta ON.\n" :
			      "Runge-Kutta OFF.\n");
      return 1;
    }
  if ( strncmp(text,"check_increase",14) == 0 )
    { check_increase_flag = !check_increase_flag;
      outstring(check_increase_flag?"Increase check ON.\n":
		   "Increase check OFF");
      return 1;
    }
  if ( strncmp(text,"system",6) == 0 )
    { system(text+6);
      return 1;
    }




  qnode.root = NULL;
  qnode.evallist = NULL;

  qnode.root=list=(struct treenode *)mycalloc(LISTMAX,sizeof(struct treenode));
  listtop = 1;
  parse_error_flag = 0;
  cmdptr = text;
  if ( setjmp(cmdbuf) )
    { cmdptr = NULL; free((char *)qnode.root); return 0; }
  retval = yybegin();  /* 0 for accept, 1 for error */
  cmdptr = NULL;

  if ( (retval == 1) || (parse_error_flag) )
    { free((char *)qnode.root); return 0; }

  list[0] = list[listtop-1];  /* root also in first spot */
  list[0].left += listtop - 1;
  list[0].right += listtop - 1;

  /* put DONE marker after root */
  list[listtop++].type = FINISHED;

  /* free excess list */
  qnode.root = (struct treenode *)kb_realloc((char *)list,
                                      listtop*sizeof(struct treenode));
  
  /* fold constants */
  expr_convert(&qnode);

  if ( logfd )
   switch ( commandverb )
    { 
       case REFINE_:
       case DELETE_:
       case SET_:
       case UNSET_:
	 fprintf(logfd,"%s\n",text);
	 break;
    }

  switch ( commandverb )
    { 
       case REFINE_:  refine_query(); break;
       case DELETE_:  delete_query(); break;
       case LIST_:    list_query(); break;
       case SHOW_:    free_expr(&show_expr);
		      if ( condition_flag )
		        show_expr = qnode;  /* set permanent show condition */
                      qnode.root = NULL;
                      qnode.evallist = NULL;
		      torshow(); break;
       case UNSET_:
       case SET_:     set_query(); break;
       case READ_:
             commandfd = fopen(filename,"r");
             if ( commandfd == NULL )
               { perror(filename);
                 break;
               }
             break;
       case SHELL_:
#ifdef TC
          spawnlp(P_WAIT,"command",NULL);
#else
          if ( fork() == 0 )
			{ execlp("sh",NULL);
			  perror("sh");
			}
		      else wait(NULL);
#endif
		      break;
       case DID_:  break;  /* done in parser */
       case RECALC_: 
	   recalc();
           break;
       default: error("Bad command verb.\n",RECOVERABLE);
    }

  free_expr(&qnode);
  return 1;
}


void refine_query()
{
  int count = 0;
  element_id sentinel;

  switch ( celement )
    { 
       case VERTICES_:
          error("Cannot refine vertices.\n",COMMAND_ERROR);
	  break;
       case EDGES_:
	 q_id = NULLEDGE;
	 sentinel = web.skel[EDGE].last;
	 while ( generate_all(EDGE,&q_id) )
	   { if ( !condition_flag || eval(&qnode,NULL) )
	       count += edge_refine(q_id);
	     if ( equal_id(q_id,sentinel) )
	       break;
	   }
	 sprintf(msg,"Edges subdivided: %d\n",count);
	 outstring(msg);
	 break;
       case FACETS_:
	 q_id = NULLFACET;
	 sentinel = web.skel[FACET].last;
	 while ( generate_all(FACET,&q_id) )
	   { 
	     if ( !condition_flag || eval(&qnode,NULL) )
	      { face_triangulate(q_id,FACET_EDGES);
		count++;
	      }
	     if ( equal_id(q_id,sentinel) )
	       break;
	   }
	 sprintf(msg,"Facets subdivided: %d\n",count);
	 outstring(msg);
	 break;
       case BODIES_:
          error("Cannot refine bodies.\n",COMMAND_ERROR);
	  break;
       default: error("Bad query element type.\n",RECOVERABLE);
    }
  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();
  if ( count ) update_display();
}

void delete_query()
{
  int count = 0;

  switch ( celement )
    { 
       case VERTICES_:
          error("Cannot delete vertices.\n",COMMAND_ERROR);
	  break;
       case EDGES_:
	 q_id = NULLEDGE;
	 while ( generate_all(EDGE,&q_id) )
	   if ( !condition_flag || eval(&qnode,NULL) )
	     { int elimcount = eliminate_edge(q_id);
	       if ( elimcount ) free_element(q_id);
	       count += elimcount;
               if ( elimcount == 0 )
	        { sprintf(errmsg,"Cannot delete edge %d.\n",ordinal(q_id)+1);
                  error(errmsg,WARNING);
                }
	     }
	 sprintf(msg,"Edges deleted: %d\n",count);
	 outstring(msg);
	 break;
       case FACETS_:
	 q_id = NULLFACET;
	 while ( generate_all(FACET,&q_id) )
	   if ( !condition_flag || eval(&qnode,NULL) )
	      { edge_id e_id[FACET_EDGES];
		facetedge_id fe;
		int i;
		int elimcount=0;

		fe = get_facet_fe(q_id);
		for ( i =  0 ; i < FACET_EDGES ; i++ )
		  { e_id[i] = get_fe_edge(fe);
		    fe = get_next_edge(fe);
		  }
		for ( i =  0 ; i < FACET_EDGES ; i++ )
		  if ( valid_element(e_id[i]) )
		    { elimcount = eliminate_edge(e_id[i]);
	              if ( elimcount )
			{ free_element(e_id[i]);
	                  count += elimcount;
			  break;
                        }
	            }
                if ( elimcount == 0 )
		 { sprintf(errmsg,"Cannot delete facet %d.\n",ordinal(q_id)+1);
                   error(errmsg,WARNING);
                 }

	      }
	 sprintf(msg,"Facets deleted: %d\n",count);
	 outstring(msg);
	 break;
       case BODIES_:
          error("Cannot delete bodies.\n",COMMAND_ERROR);
	  break;
    }
  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();
  if ( count ) update_display();
}

void list_query()
{
  switch ( celement )
    { 
       case TOPINFO_:
	 top_dump(outfd);
	 break;

       case VERTICES_:
	 outstring(" Id                X                 Y                Z\n");
	 FOR_ALL_VERTICES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     vertex_dump(q_id,outfd);
	 break;

       case EDGES_:
         FOR_ALL_EDGES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     edge_dump(q_id,outfd);
	 break;

       case FACETS_:
         FOR_ALL_FACETS(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     facet_dump(q_id,outfd);
	 break;

       case BODIES_:
         FOR_ALL_BODIES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     body_dump(q_id,outfd);
	 break;

       case FACETEDGES_:
  fputs("\n\nfacet-edge structures\n",outfd);
  fputs("  id       edge  facet  prevedge nextedge   prevfacet nextfacet\n",
    outfd);
         FOR_ALL_FACETEDGES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     facetedge_dump(q_id,outfd);
	 break;

       default: error("Bad query element type.\n",RECOVERABLE);
     }
}

/***********************************************************************
*
*  function: set_query()
*
*  purpose:  Handles queries that set attributes.
*
*/

void set_query()
{
  if ((set_query_type == SET_CONSTRAINT)||(set_query_type == UNSET_CONSTRAINT))
   {
     int cnum = abs(query_intval);
     if ( (cnum >= CONSTRMAX) || (web.constraints[cnum].formula == NULL) )
      {
       sprintf(errmsg,
          "Bad constraint number: %d.\n",cnum);
       error(errmsg,RECOVERABLE);
      }
   }

  if ( set_query_type == SET_TRANS )
    { if ( (query_realval < 0.0) || (query_realval > 1.0) )
	error("Opacity must be between 0 and 1.\n",COMMAND_ERROR);
      else facet_alpha = query_realval;
      update_display();
      return;
    }

  switch ( celement )
    { 
       case VERTICES_:
	 FOR_ALL_VERTICES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     switch ( set_query_type )
	       {
		 case SET_Q_ATTR:
		   if ( commandverb == SET_ )
		     set_attr(q_id,set_query_attr); 
		   else
		     unset_attr(q_id,set_query_attr);
		   break;

		 case SET_CONSTRAINT:
		   if ( commandverb == SET_ )
		     { set_attr(q_id,CONSTRAINT);
		       set_v_constraint_map(q_id,query_intval);
		     }
		   else
		     { unset_v_constraint_map(q_id,query_intval);
		       if ( !get_v_constraint_map(q_id) )
		         unset_attr(q_id,CONSTRAINT);
		     }
		   break;
		   
		 case SET_COORD:
		   get_coord(q_id)[query_coord] = query_realval;
		   break;
		   
		 case SET_PARAM:
		   get_param(q_id)[query_coord] = query_realval;
		   break;
		}
	 break;
       
       case EDGES_:
         FOR_ALL_EDGES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     switch ( set_query_type )
	       {
		 case SET_Q_ATTR:
		   if ( commandverb == SET_ )
		     set_attr(q_id,set_query_attr);
		   else
		     unset_attr(q_id,set_query_attr); 
		   break;

		 case SET_CONSTRAINT:
		   if ( commandverb == SET_ )
		    { set_attr(q_id,CONSTRAINT);
		      set_e_constraint_map(q_id,query_intval);
		    }
		   else
		    { unset_e_constraint_map(q_id,query_intval);
		      if ( !get_e_constraint_map(q_id) )
		        unset_attr(q_id,CONSTRAINT);
		    }
		   break;
		   
		 case SET_DENSITY:
		   set_edge_density(q_id,query_realval);
		   break;

		 case SET_COLOR:
		   set_edge_color(q_id,query_intval);
		   break;
	       }
	 break;

       case FACETS_:
         FOR_ALL_FACETS(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     switch ( set_query_type )
	       {
		 case SET_Q_ATTR:
		   if ( commandverb == SET_ )
		     set_attr(q_id,set_query_attr); 
		   else
		     unset_attr(q_id,set_query_attr); 
		   break;

		 case SET_CONSTRAINT:
		   if ( commandverb == SET_ )
		    { set_attr(q_id,CONSTRAINT);
		      set_f_constraint_map(q_id,query_intval);
		    }
		   else
		     { unset_f_constraint_map(q_id,query_intval);
		       if ( !get_f_constraint_map(q_id) )
		         unset_attr(q_id,CONSTRAINT);
		     }
		   break;

		 case SET_COLOR:
		   set_facet_color(q_id,query_intval);
		   break;
		   
		 case SET_DENSITY:
		   if ( commandverb == SET_ )
		    { set_facet_density(q_id,query_realval);
                      set_attr(q_id,DENSITY);
		    }
		   else
                      unset_attr(q_id,DENSITY);
		   break;

		 case SET_TAG:
		   if ( commandverb == SET_ )
		     set_tag(q_id,(tagtype)query_intval);
		   else
		     set_tag(q_id,0);
		   break;
	       }
          break;

       case BODIES_:
         FOR_ALL_BODIES(q_id)
	   if ( !condition_flag || eval(&qnode,NULL) )
	     switch ( set_query_type )
	       {
		 case SET_Q_ATTR:
		   if ( commandverb == SET_ )
		     set_attr(q_id,set_query_attr); 
		   else
		     unset_attr(q_id,set_query_attr); 
		   break;

		 case SET_DENSITY:
		   if ( commandverb == SET_ )
		     { set_facet_density(q_id,query_realval);
                       set_attr(q_id,DENSITY);
		     }
		   else
                     unset_attr(q_id,DENSITY);
		   break;
		
		 case SET_VOLUME:
		   if ( commandverb == SET_ )
		    { set_body_fixvol(q_id,query_realval);
		      if ( !(get_attr(q_id) & FIXEDVOL) )
			  fixed_volumes++;
                      set_attr(q_id,FIXEDVOL);
		    }
		   else
                    { if ( get_attr(q_id) & FIXEDVOL)
			 { unset_attr(q_id,FIXEDVOL);
			   fixed_volumes--;
                         }
		     }
		   break;
		
		 case SET_PRESSURE:
		   if ( commandverb == SET_ )
		    { set_body_pressure(q_id,query_realval);
                      set_attr(q_id,PRESSURE);
		    }
		   else
                      unset_attr(q_id,PRESSURE);
		   break;
	       }
	 break;
       default: error("Bad query element type.\n",RECOVERABLE);
    }
}


