
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: filgraph.c
*
*  Purpose:  Triangle list file output.
*            Format: One triangle per line,
*                    x1 y1 x2 y2 x3 y3 w1 w2 w3 d
*            vertex coordinates, edge widths, and face density.
*/

#include "include.h"

static FILE *fd;

/*****************************************************************
*
*  Function: fil_init()
*
*  Purpose:  Get file name from user.
*/

void fil_init()
{
  char filename[100];

  for (;;)
  {
    prompt("Enter file name: ",filename);
    if ( filename[0] == '\0' ) /* abort */
       error("Output aborted. ",RECOVERABLE);

    fd = fopen(filename,"w");
    if ( fd == NULL )
      { perror(filename);
      }
    else return;
  }

}

/************************************************************
*
*  Function: fil_edge()
*
*  Purpose: Graphs one edge, already transformed.
*/

void fil_edge(t)
struct tsort *t;
{
  fprintf(fd,"  %f %f  ",t->x[0][0],t->x[0][1]);
  fprintf(fd,"  %f %f  ",t->x[1][0],t->x[1][1]);
  fprintf(fd,"  %f %f  ",t->x[1][0],t->x[1][1]);
  fprintf(fd,"  0.03  0.03  0.03  0.0 \n");
}


/************************************************************
*
*  Function: fil_facet()
*
*  Purpose: Graphs one facet, already transformed.
*/

void fil_facet(t)
struct tsort *t;
{
  double cosine;
  edge_id e_id;
  facetedge_id fe_id;
  int type;
  int i;
  
  if ( get_facet_color(t->f_id) == CLEAR ) return;

  fprintf(fd,"%f %f",t->x[0][0],t->x[0][1]);
  fprintf(fd," %f %f",t->x[1][0],t->x[1][1]);
  fprintf(fd," %f %f",t->x[2][0],t->x[2][1]);
  fe_id = get_facet_fe(t->f_id);
  for ( i = 0 ; i < 3 ; i++ )
    { 
      e_id = get_fe_edge(fe_id);
      if ( get_eattr(e_id) & FIXED ) type = 3;
      else if ( equal_id(get_next_facet(fe_id),fe_id) )
	type = 1;  /* edge of some sort */
      else if ( !equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) )
	type = 4;  /* triple line at least */
      else type = 0;  /* ordinary internal grid line */

      fprintf(fd," %1d",type);
      fe_id = get_next_edge(fe_id);
    }
  
  if ( valid_id(t->f_id) )
    {
      cosine = t->normal[1]/sqrt(dotf(t->normal,t->normal,3));
      if ( t->normal[2] < 0.0 ) cosine = -cosine;
    }
  else cosine = 0.0;
  fprintf(fd," %f\n",cosine);
}

/*************************************************************
*
*  Function:  fil_finish()
*
*  Purpose:   End output.
*/

void fil_finish()
{
  fclose(fd);
}
