/* module: 4dd_wire.c	 iris 4d wireframe window code */

static char SccsId[] = "@(#)4dd_wire.c	1.2";

#include <gl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <malloc.h>
#include <math.h>
#include <device.h>

#include "4dd_echo.h"
#include "4dd_cyfile.h"
#include "4dd_gt.h"

extern int errno;
extern struct wireinfo wire;

char *STR_MOVEMOUSE = "Move mouse, left button to stop";

/* values returned by menu */

#define ROTATE		(long)10
#define ZOOM		(long)11
#define PAN			(long)12
#define PERSP		(long)13
#define TWIST		(long)14

#define RENDER		(long)20
#define METER		(long)21
#define ERASE		(long)22
#define OPTIONS		(long)23
#define EXIT		(long)24
#define CLIP		(long)25
#define SNAP		(long)26

#define LTMIN		(long)30
#define LTMAX		(long)31
#define LGMIN		(long)32
#define LGMAX		(long)33

#define HIRES		(long)41
#define LORES		(long)42
#define DRAWLT		(long)43
#define DRAWLG		(long)44
#define COLOR		(long)45
#define DOUBLEBUF	(long)46
#define SINGLEBUF	(long)47
#define DEPTHCUE	(long)48
#define DRAWVOID	(long)49
#define OSCILLATE   (long)50

/*
**	#define AZIM_DELTA 400
**	#define INC_DELTA 100
**	#define THETA_INCR .02
*/


wire_event(vtx, event, value)

struct Vertex *vtx;
Device event;
short value;
{
	Boolean first_time = TRUE;

	/* handle event */
	switch (event) {
	default:
		break;
	case CREATE:
		wire.vtx = vtx;
		prefposition(
			wire.position_x, wire.size_x + wire.position_x,
			wire.position_y, wire.size_y + wire.position_y
		);
		wire.gid = winopen(wire.name);
		wintitle(wire.name);
		keepaspect(wire.size_x, wire.size_y);
		winconstraints();
		if (wire.doublebuffer) {
			doublebuffer();
		} else {
			singlebuffer();
		}
		RGBmode();
		gconfig();
		frontbuffer(FALSE);
		wire_redraw(&wire);
		wire.menu = wire_defpup(&wire); 
		break;
	case REDRAW:
		frontbuffer(FALSE);
		wire_redraw(&wire);
		break;
	case DESTROY:
		winclose(wire.gid);
		break;
	case WRITECONFIG:
		wire_putdefaults(&wire);
		break;
	case MENUBUTTON:
		help_event("Hold button, select item, release button",
			(Device)REDRAW, 0);
		if (getbutton(MENUBUTTON)) {
			switch (dopup(wire.menu)) {
			default:
				break;
			case ROTATE:
				wire_rotate(&wire);
				break;
			case ZOOM:
				wire_zoom(&wire);
				break;
			case PAN:
				wire_pan(&wire);
				break;
			case PERSP:
				wire_persp(&wire);
				break;
			case TWIST:
				wire_twist(&wire);
				break;
			case OSCILLATE:
				wire_oscillate(&wire);
				break;
			case EXIT:
				wire.vtx->gs->ltmin = vtx->ltmin;
				wire.vtx->gs->ltmax = vtx->ltmax;
				wire.vtx->gs->lgmin = vtx->lgmin;
				wire.vtx->gs->lgmax = vtx->lgmax;
				edit_exit = TRUE;
				break;
			case HIRES:
				if (--wire.ltpitch < 1) wire.ltpitch = 1;
				if (--wire.lgpitch < 1) wire.lgpitch = 1;
				if (--wire.ltresolution < 1) wire.ltresolution = 1;
				if (--wire.lgresolution < 1) wire.lgresolution = 1;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case LORES:
				if (++wire.ltpitch > 4) wire.ltpitch = 4;
				if (++wire.lgpitch > 8) wire.lgpitch = 8;
				if (++wire.ltresolution > 4) wire.ltresolution = 4;
				if (++wire.lgresolution > 8) wire.lgresolution = 8;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case DRAWLT:
				wire.ltflag = !wire.ltflag;
				if (!wire.lgflag) wire.lgflag = TRUE;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case DRAWLG:
				wire.lgflag = !wire.lgflag;
				if (!wire.ltflag) wire.ltflag = TRUE;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case DEPTHCUE:
				wire.depthcue = !wire.depthcue;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case DRAWVOID:
				wire.drawvoid = !wire.drawvoid;
				wire.menu = wire_defpup(&wire);
				qenter((short)REDRAW, wire.gid);
				break;
			case DOUBLEBUF:
				wire.doublebuffer = TRUE;
				winset(wire.gid);
				doublebuffer();
				gconfig();
				wire.menu = wire_defpup(&wire);
				break;
			case SINGLEBUF:
				wire.doublebuffer = FALSE;
				winset(wire.gid);
				singlebuffer();
				gconfig();
				wire.menu = wire_defpup(&wire);
				break;
			} /* end switch */
		} /* end if */
		break;
	} /* end switch */
	return(wire.gid);
}


wire_defpup(wi)

struct wireinfo *wi;
{
	static long pup;
	long wire_pup;
	long option_pup;
	long limits_pup;
	char *string = malloc(1000);
	char *s;

	if (pup != -1) {
		freepup(pup);
		pup = -1;
	}

	s = string;
	s += sprintf(s, "Wireframe Options %%t");
	if (wi->ltresolution != 1) {
		s += sprintf(s, "| Raise Resolution %%x%d", HIRES);
	} else {
		s += sprintf(s, "| ");
	}
	if (wi->ltresolution < 4) {
		s += sprintf(s, "| Lower Resolution %%x%d", LORES);
	} else {
		s += sprintf(s, "| ");
	}
	if (wi->ltflag) {
		s += sprintf(s, "| Hide Latitudes%%x%d", DRAWLT);
	} else {
		s += sprintf(s, "| Show Latitudes%%x%d", DRAWLT);
	}
	if (wi->lgflag) {
		s += sprintf(s, "| Hide Longitudes %%x%d", DRAWLG);
	} else {
		s += sprintf(s, "| Show Longitudes %%x%d", DRAWLG);
	}
	if (wi->drawvoid) {
		s += sprintf(s, "| Skip Voids at 0 %%x%d", DRAWVOID);
	} else {
		s += sprintf(s, "| Draw Voids at 0 %%x%d", DRAWVOID);
	}
	if (wi->depthcue) {
		s += sprintf(s, "| Stop Depthcue%%x%d", DEPTHCUE);
	} else {
		s += sprintf(s, "| Use Depthcue%%x%d", DEPTHCUE);
	}
	if (wi->doublebuffer) {
		s += sprintf(s, "| Single Buffer %%x%d", SINGLEBUF);
	} else {
		s += sprintf(s, "| Double Buffer %%x%d", DOUBLEBUF);
	}
	wire_pup = defpup(string);

	s = string;
	s += sprintf(s, "Miscellaneous %%t");
	option_pup = defpup(string);

	s = string;
	s += sprintf(s, "Data Clipping %%t");
	s += sprintf(s, "| Lat Minimum %%x%d", LTMIN);
	s += sprintf(s, "| Lat Maximum %%x%d", LTMAX);
	s += sprintf(s, "| Long Minimum %%x%d", LGMIN);
	s += sprintf(s, "| Long Maximum %%x%d", LGMAX);
	limits_pup = defpup(string);

	s = string;
	s += sprintf(s, "Wireframe View %%t");
	s += sprintf(s, "| Rotate %%x%d", ROTATE);
	s += sprintf(s, "| Zoom %%x%d", ZOOM);
	s += sprintf(s, "| Pan %%x%d", PAN);
	s += sprintf(s, "| Perspec %%x%d", PERSP);
	s += sprintf(s, "| Twist %%x%d", TWIST);
	s += sprintf(s, "| Oscillate %%x%d", OSCILLATE);
	s += sprintf(s, "| Wireframe Options %%m");
/*	s += sprintf(s, "| Options %%m"); */
/*	s += sprintf(s, "| Data Clipping %%m"); */
	s += sprintf(s, "| Exit Edit %%x%d", EXIT);
	pup = defpup(string, wire_pup, option_pup, limits_pup);

	free(string);
	return(pup);
}


wire_redraw(wi)

struct wireinfo *wi;
{
	winset(wi->gid);
	getorigin(&wi->position_x, &wi->position_y);
	getsize(&wi->size_x, &wi->size_y);
	lsetdepth(0x0, 0x7fffff);
	lRGBrange(0, 0, 0, 255, 255, 255, 0x0, 0x7fffff);
	depthcue(wi->depthcue);
	zbuffer(wi->depthcue);
	zclear();
	cpack(wi->color_background);
	clear();
	viewport(0, wi->size_x - 1, 0, wi->size_y - 1);
	wire_transform(wi);
	if (wi->vtx->nlg == 1) {
		scale(1., 1., 2.);
	}
	wire_draw(wi);
	if (wi->vtx->nlg == 1) {
		/* draw grid */
		zbuffer(FALSE);
		depthcue(FALSE);
		cpack(wi->color_meterlines);
		/* horizontal lines */
		move(.0, -.2,  .0);
		draw(.0, -.2, -.2);
		move(.0, -.1,  .0);
		draw(.0, -.1, -.2);
		move(.0,  .0,  .0);
		draw(.0,  .0, -.2);
		move(.0,  .1,  .0);
		draw(.0,  .1, -.2);
		move(.0,  .2,  .0);
		draw(.0,  .2, -.2);
		/* vertical lines */
		move(.0, -.2,  .0);
		draw(.0,  .2,  .0);
		move(.0, -.2, -.05);
		draw(.0,  .2, -.05);
		move(.0, -.2, -.1);
		draw(.0,  .2, -.1);
		move(.0, -.2, -.15);
		draw(.0,  .2, -.15);
		move(.0, -.2, -.2);
		draw(.0,  .2, -.2);
		/* 1 mm lines */
		move(.0, -.001, -.099);
		draw(.0,  .001, -.099);
		draw(.0,  .001, -.101);
		draw(.0, -.001, -.101);
		draw(.0, -.001, -.099);
	}
	swapbuffers();
}



wire_rotate(wi)

struct wireinfo *wi;
{
	short val;					/* qread event value */
	long dev;
	short base_x, base_y;
	Angle azim;
	Angle inc;
	char string[80];

	base_x = getvaluator(MOUSEX);
	base_y = getvaluator(MOUSEY);
	azim = wi->azim;
	inc = wi->inc;
	qdevice((Device)MOUSEX);
	qdevice((Device)MOUSEY);
	help_event(STR_MOVEMOUSE, (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, wi->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEX);
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
		case MOUSEY:
			while ((dev = qtest()) == MOUSEX || dev == MOUSEY) {
				qread(&val);
			}
			wi->azim = azim + ((getvaluator(MOUSEX) - base_x) << 3);
			wi->inc = inc - ((getvaluator(MOUSEY) - base_y) << 3);
			wire_redraw(wi);
#define DDTOD(dd) ((double)(dd)/10.)
sprintf(string, "azim=%7f decl=%7f", DDTOD(wi->azim), DDTOD(wi->inc));
help_event(string, (Device)REDRAW, 0);
			break;
		}
	}
}


wire_zoom(wi)

struct wireinfo *wi;
{
	short val;					/* qread event value */
	short base_y;
	real zoom;

	base_y = getvaluator(MOUSEY);
	zoom = wi->zoom;
	qdevice((Device)MOUSEY);
	help_event(STR_MOVEMOUSE, (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, wi->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEY:
			while (qtest() == MOUSEY) {
				qread(&val);
			}
			wi->zoom = zoom + (getvaluator(MOUSEY) - base_y) / 100.;
			wi->zoom = MIN(MAX(wi->zoom, .1), 10.);
			wire_redraw(wi);
			break;
		}
	}
}


wire_pan(wi)

struct wireinfo *wi;
{
	short val;					/* qread event value */
	long dev;
	short base_x, base_y;
	real pan_x;
	real pan_y;

	base_x = getvaluator(MOUSEX);
	base_y = getvaluator(MOUSEY);
	pan_x = wi->panx;
	pan_y = wi->pany;
	qdevice((Device)MOUSEX);
	qdevice((Device)MOUSEY);
	help_event(STR_MOVEMOUSE, (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, wi->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEX);
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
		case MOUSEY:
			while ((dev = qtest()) == MOUSEX || dev == MOUSEY) {
				qread(&val);
			}
			wi->panx = pan_x + (getvaluator(MOUSEX) - base_x)/800.;
			wi->pany = pan_y + (getvaluator(MOUSEY) - base_y)/800.;
			wire_redraw(wi);
			break;
		}
	}
}


wire_persp(wi)

struct wireinfo *wi;
{
	short val;					/* qread event value */
	short base_y;
	real persp;

	base_y = getvaluator(MOUSEY);
	persp = wi->dist;
	qdevice((Device)MOUSEY);
	help_event(STR_MOVEMOUSE, (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, wi->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEY:
			while (qtest() == MOUSEY) {
				qread(&val);
			}
			wi->dist = persp + (getvaluator(MOUSEY) - base_y) / 100.;
			wi->dist = MAX(wi->dist, wi->clipz);
			wire_redraw(wi);
			break;
		}
	}
}


wire_twist(wi)

struct wireinfo *wi;
{
	short val;					/* qread event value */
	short base_x;
	real twist;

	base_x = getvaluator(MOUSEY);
	twist = wi->twist;
	qdevice((Device)MOUSEX);
	help_event(STR_MOVEMOUSE, (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, wi->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
			while (qtest() == MOUSEX) {
				qread(&val);
			}
			wi->twist = twist + ((getvaluator(MOUSEX) - base_x) << 3);
			wire_redraw(wi);
			break;
		}
	}
}



wire_oscillate(wi)

struct wireinfo *wi;
{
	long dev;
	int azim_base = wi->azim;
	int inc_base = wi->inc;
	struct itimerval timer;
	float time;
	float time_angle;
	short val;

	help_event("Left button stops", (Device)REDRAW, 0);
	/* set the timer to count down for a long time */
	timer.it_interval.tv_sec = 1000000;
	timer.it_interval.tv_usec = 0;
	timer.it_value.tv_sec = 1000000;
	timer.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &timer, NULL);

	while (1) {
		if (qtest()) {
			dev = qread(&val);
			if (dev == MENUBUTTON || dev == LEFTMOUSE) {
				break;
			}
		}
		/* get timer value and convert to float seconds */
		getitimer(ITIMER_REAL, &timer);
		time = (float)timer.it_value.tv_sec;
		time += (float)timer.it_value.tv_usec * 1e-6;
		time = fmod(time, 15.);		/* time = 0 to 14.999 seconds */
		/* time angle rotates thru 360 degrees (2 pi) each 15 seconds */
		time_angle = M_PI * 2. * (float)time / 15.;

		wi->azim = azim_base + (550 * fcos(time_angle));
		wi->inc = inc_base + (130 * fsin(2. * time_angle));

		wire_redraw(wi);
	}
	return;
}



wire_transform(wi)

struct wireinfo *wi;
{
	/* projection transform */
	window(
		wi->panx - wi->clipx/wi->zoom, wi->panx + wi->clipx/wi->zoom,
		wi->pany - wi->clipy/wi->zoom, wi->pany + wi->clipy/wi->zoom,
		wi->dist - wi->clipz, wi->dist + wi->clipz
	);
		
	/* viewing transforms */
	polarview(wi->dist, wi->azim, wi->inc, wi->twist);

	/* world transforms */
	rotate(900, 'x');	/* make inc and azim work nice by swapping x/y axes */
}


wire_draw(wi)

struct wireinfo *wi;
{
	struct Vertex *vtx = wi->vtx;
	int lg, lt;

	cpack(wi->color_lines);
	if (wi->drawvoid == TRUE) {
		if (wi->lgflag == TRUE) {
			for (lg = vtx->lgmin; lg <= vtx->lgmax; lg += wi->lgpitch) {
				bgnline();
				for (lt=vtx->ltmin; lt <= vtx->ltmax; lt += wi->lgresolution) {
					v3f(&vtx->pnt[lg][lt][LX]);
				}
				endline();
			}
		}
		if (wi->ltflag == TRUE) {
				for (lt = vtx->ltmin; lt <= vtx->ltmax; lt += wi->ltpitch) {
				bgnline();
				for (lg=vtx->lgmin; lg <= vtx->lgmax; lg += wi->ltresolution) {
					v3f(&vtx->pnt[lg][lt][LX]);
					}
				endline();
			}
		}
	} else {
#		define MOVE(p,i,j) move(p[i][j][LX],p[i][j][LY],p[i][j][LZ])
#		define DRAW(p,i,j) draw(p[i][j][LX],p[i][j][LY],p[i][j][LZ])
		if (wi->lgflag == TRUE) {
			for (lg = vtx->lgmin; lg <= vtx->lgmax; lg += wi->lgpitch) {
				lt = vtx->ltmin;
				while (lt <= vtx->ltmax) {
					if (vtx->data[lg][lt] != VTX_VOID) {
						MOVE(vtx->pnt, lg, lt);
						lt += wi->lgresolution;
						while (lt <= vtx->ltmax) {
							if (vtx->data[lg][lt] != VTX_VOID) {
								DRAW(vtx->pnt, lg, lt);
								lt += wi->lgresolution;
							} else {
								lt += wi->lgresolution;
								break;
							}
						}
					} else {
						lt += wi->lgresolution;
					}
				}
			}
		}
		if (wi->ltflag == TRUE) {
			for (lt = vtx->ltmin; lt <= vtx->ltmax; lt += wi->ltpitch) {
				lg = vtx->lgmin;
				while (lg <= vtx->lgmax) {
					if (vtx->data[lg][lt] != VTX_VOID) {
						MOVE(vtx->pnt, lg, lt);
						lg += wi->ltresolution;
						while (lg <= vtx->lgmax) {
							if (vtx->data[lg][lt] != VTX_VOID) {
								DRAW(vtx->pnt, lg, lt);
								lg += wi->ltresolution;
							} else {
								lg += wi->ltresolution;
								break;
							}
						}
					} else {
						lg += wi->ltresolution;
					}
				}
			}
		}
	}
}



wire_getdefaults(wi)

struct wireinfo *wi;
{
	strcpy(wi->name, "Echo Demo Wireframe Display");
	wi->color_background = 0xff000000;
	wi->color_lines = 0xffffffff;
	wi->color_meterlines = 0x00ffffff;
	wi->color_eraselines = 0x00004f00;
	wi->position_x = 28;
	wi->position_y = 489;
	wi->size_x = 357;
	wi->size_y = 357;
	wi->clipx = .22;
	wi->clipy = .22;
	wi->clipz = .22;
	wi->panx = 0.;
	wi->pany = 0.;
	wi->panz = 0.;
	wi->zoom = 1.;
	wi->dist = 1.;
	wi->azim = 0;
	wi->inc = 900;
	wi->twist = 0;
	wi->ltflag = FALSE;
	wi->lgflag = TRUE;
	wi->ltpitch = 3;
	wi->lgpitch = 3;
	wi->ltresolution = 3;
	wi->lgresolution = 3;
	wi->depthcue = TRUE;
	wi->drawvoid = FALSE;
	wi->doublebuffer = TRUE;
}


wire_putdefaults(wi)

struct wireinfo *wi;
{
}

