/* module: cyconvert.c
 * 
 * - A translator for image files produced by the Cyberware Echo 
 *   digitizer software to the following formats:
 *
 *		movie.byu
 *		wavefront 
 *		digital_arts
 *
 * output:
 *		
 *		movie.byu     -	file created if the "-m" option is chosen. Contains
 *						Cyberware image data in movie.byu format.
 *		wavefront.obj - file created if the "-w" option is chosen. Contains
 *						Cyberware image data in wavefront format.
 *		dig_arts.aob  - file created if the "-d" option is chosen. Contains
 *						Cyberware image data in digital_arts Ascii Object format.
 *
 * usage: cyconvert [-mwd] [-g res] [-t res] infile 
 *
 * Caution:
 *		The movie.byu mode creates some possibly huge intermediate
 *		files in the current directory.  These files are named with
 *		names like 'cyconvert.????'.
 *
 * Compiler:
 *		SGI Iris 4d series:
 *			cc -DIRISGT -o cyconvert cyconvert.c cyfile.c strings.c -lm
 *
 *		Sun products:
 *			cc -DSUN -o cyconvert cyconvert.c cyfile.c strings.c -lm
 */

static char SccsId[] = "@(#)cyconvert.c	1.7";


#include "fcntl.h"
#include "stdio.h"
#include "math.h"
#include "cyfile.h"

extern int errno;
void perror();
extern char *sys_errlist[];
extern int sys_nerr;

#define VTXNLG		512
#define VTXNLT		512
#define NVAR		6

/* subscripts for pnt[] below */
#define NX		0
#define NY		1
#define NZ		2
#define LX		3
#define LY		4
#define LZ		5

/* output formats */
#define MOVIE_BYU		1
#define WAVEFRONT		2
#define DIGITAL_ARTS	3

#define TRUE	1
#define FALSE	0

struct Vertex {
	GSPEC *gs;
	int nlg;
	int nlt;
	int lgmin, lgmax;
	int ltmin, ltmax;
	int lgresol;
	int ltresol;
	float pnt[VTXNLG][VTXNLT][NVAR];
};
static char *filename; /* input file name */

void gstovtx(GSPEC *gs, struct Vertex *vtx);
void makenormals(struct Vertex *vtx);
void vnormal(float *p, float *vtx0, float *vtx1,
			 float *vtx2, float *vtx3, unsigned int cart);
void make_wavefront(struct Vertex *vtx);
void make_moviebyu(struct Vertex *vtx);
void make_digital_arts_aob(struct Vertex *vtx);

main(int argc, char **argv)
{
	int c;
	extern char *optarg;
	extern int optind;
	int errflg;

	int fd;							/* target image file */
	GSPEC *gs;						/* database descriptor */
	struct Vertex *vtx;				/* intermediate database */
	int outputmode = MOVIE_BYU;		/* output format */

	/* foreground(); */	/* DEBUG */

	vtx = (struct Vertex *)malloc(sizeof(struct Vertex));
	vtx->ltresol = 1;
	vtx->lgresol = 1;

	/* entry point, get any arguments, input is from given filename or
	 * standard input */

	while ((c = getopt(argc, argv, "mwdg:t:")) != -1) {
		switch (c) {
		default:
		case '?':
			errflg++;
			break;
		case 'm':
			outputmode = MOVIE_BYU;
			break;
		case 'w':
			outputmode = WAVEFRONT;
			break;
		case 'd':
			outputmode = DIGITAL_ARTS;
			break;
		case 'g':
			vtx->lgresol = atoi(optarg);
			if (vtx->lgresol < 1 || vtx->lgresol > 16) {
				fprintf(stderr, "Resolution out of range, 1 <= lg_res <= 16\n");
				exit(2);
			}
			break;
		case 't':
			vtx->ltresol = atoi(optarg);
			if (vtx->ltresol < 1 || vtx->ltresol > 16) {
				fprintf(stderr, "Resolution out of range, 1 <= lt_res <= 16\n");
				exit(2);
			}
			break;
		}
	}

	/* on argument error or missing image filename, error message and exit */
	if (errflg || optind == argc) {
		fprintf(stderr,
			"Usage: cyconvert [-mwd] [-g res] [-t res] infile > outfile\n");
		exit(2);
	}

	/* open the image file */
	if ((fd = open(argv[optind], O_RDONLY)) == -1) {
		perror("cyconvert: error opening image file");
		exit(3);
	}

	filename = argv[optind];	/* get the image filename */

	/* read image file into gs structure */
	if ((gs = cyread(0, fd)) == NULL) {
		fprintf(stderr, "cyconvert: problem with image file format\n");
		exit(4);
	}

	/* convert range map image (gs) to vertex tables (vtx) */
	gstovtx(gs, vtx);

	/* depending on selected mode, translate and output */
	switch (outputmode) {
	case MOVIE_BYU:
		make_moviebyu(vtx);
		break;
	case WAVEFRONT:
		make_wavefront(vtx);
		break;
	case DIGITAL_ARTS:
		make_digital_arts_aob(vtx);
		break;
	}

	/* free resources */
	cyfree(gs);
	close(fd);
}

void gstovtx(GSPEC *gs, struct Vertex *vtx)	
/*
 * gs to vertex conversion
 */
{
	float theta, theta_incr;		/* angle of lg and increment */
	short lt, lg;					/* latitude/longitude counters */
	int radius;						/* radius from input image */
	float r, x, y;					/* rectangular coords */
	float y_incr;					/* latitudnal increment */
	float sin_theta, cos_theta;		/* time savers */

	vtx->gs = gs;
	vtx->nlt = gs->nlt;
	vtx->nlg = gs->nlg;
	vtx->ltmin = gs->ltmin;
	vtx->ltmax = gs->ltmax;
	vtx->lgmin = gs->lgmin;
	vtx->lgmax = gs->lgmax;

	if (! (gs->flags & FLAG_CARTESIAN)) {
		theta_incr = (gs->lgincr * 1.e-6);		/* to radians */
		theta = 0.;
		y_incr = gs->ltincr * 1.e-6;			/* to meters */
		for (lg = gs->lgmin; lg <= gs->lgmax; ++lg) {
			y = -(vtx->nlt/2) * y_incr;
			sin_theta = sin(theta);
			cos_theta = cos(theta);
			for (lt = gs->ltmin; lt <= gs->ltmax; ++lt) {
				radius = GETR(gs, lt, lg);		/* cyl radius */
				if (radius != VOID) {
					r = radius * 1.e-6;			/* to meters */
					vtx->pnt[lg][lt][LX] = r * sin_theta;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = r * -cos_theta;
				} else {
					vtx->pnt[lg][lt][LX] = 0.;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = 0.;
				}
				y += y_incr;
			}
			theta += theta_incr;
		}
	} else {
		for (lg = 0; lg < vtx->nlg; ++lg) {
			x = (lg - vtx->nlg/2) * gs->lgincr * 1.e-6;
			for (lt = 0; lt < vtx->nlt; ++lt) {
				if (gs->flags & FLAG_BILATERAL) {
					y = (lt % (gs->nlt/2) - vtx->nlt) * gs->ltincr * 1.e-6;
				} else {
					y = (lt - vtx->nlt) * gs->ltincr * 1.e-6;
				}
				radius = GETR(gs, lt, lg);
				if (radius != VOID) {
					vtx->pnt[lg][lt][LX] = x;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = radius * 1.e-6;;
				} else {
					vtx->pnt[lg][lt][LX] = x;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = -.17;
				}
			}
		}
	}
/*	makenormals(vtx); */
}

void makenormals(struct Vertex *vtx)
{
	int lg;
	int lt;
	int i, j;
	unsigned int cart = (vtx->gs->flags & FLAG_CARTESIAN) ? TRUE : FALSE;

	/* special case, low longitude */
	vnormal(vtx->pnt[0][0], vtx->pnt[0][0], vtx->pnt[1][0],
		vtx->pnt[0][0], vtx->pnt[0][1], cart);
	for (lt = 1; lt < vtx->nlt-1; ++lt) {
		vnormal(vtx->pnt[0][lt], vtx->pnt[0][lt], vtx->pnt[1][lt],
			vtx->pnt[0][lt-1], vtx->pnt[1][lt+1], cart);
	}
	vnormal(vtx->pnt[0][lt], vtx->pnt[0][lt], vtx->pnt[1][lt],
		vtx->pnt[0][lt-1], vtx->pnt[0][lt], cart);

	/* each inside longitude */
	for (lg = 1; lg < vtx->nlg-1; ++lg) {
		/* special case, low latitude */
		vnormal(vtx->pnt[lg][0], vtx->pnt[lg-1][0], vtx->pnt[lg+1][0],
				vtx->pnt[lg][0], vtx->pnt[lg][1], cart);
		/* each inside latitude */
		for (lt = 1; lt < vtx->nlt-1; ++lt) {
			vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg+1][lt],
					vtx->pnt[lg][lt-1], vtx->pnt[lg][lt+1], cart);
		}
		/* special case, high latitude */
		vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg+1][lt],
			vtx->pnt[lg][lt-1], vtx->pnt[lg][lt], cart);
	}
/*	putchar('\n'); */

	/* special case, high longitude */
	vnormal(vtx->pnt[lg][0], vtx->pnt[lg-1][0], vtx->pnt[lg][0],
		vtx->pnt[lg][0], vtx->pnt[lg][1], cart);
	for (lt = 1; lt < vtx->nlt-1; ++lt) {
		vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg][lt],
			vtx->pnt[lg][lt-1], vtx->pnt[lg][lt+1], cart);
	}
	vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg][lt],
		vtx->pnt[lg][lt-1], vtx->pnt[lg][lt], cart);
}

#define SQ(n) ((n)*(n))

void vnormal(float *p, float *vtx0, float *vtx1,
		     float *vtx2, float *vtx3, unsigned int cart)
{
	float v0[3], v1[3];			/* vectors */
#	define X 0					/* subscript mnemonics */
#	define Y 1
#	define Z 2
	float magnitude;

	/* vectors */
	v0[X] = vtx1[LX] - vtx0[LX];
	v0[Y] = vtx1[LY] - vtx0[LY];
	v0[Z] = vtx1[LZ] - vtx0[LZ];
	v1[X] = vtx2[LX] - vtx3[LX];
	v1[Y] = vtx2[LY] - vtx3[LY];
	v1[Z] = vtx2[LZ] - vtx3[LZ];

	/* normal of vector pair */
	p[NX] = v0[Y]*v1[Z] - v0[Z]*v1[Y];
	p[NY] = v0[Z]*v1[X] - v0[X]*v1[Z];
	p[NZ] = v0[X]*v1[Y] - v0[Y]*v1[X];

	/* magnitude */
	magnitude = sqrt(SQ(p[NX])+SQ(p[NY])+SQ(p[NZ]));

	/* make unit magnitude */
	if (magnitude != 0.) {
		if (cart) {
			p[NX] /= -magnitude;
			p[NY] /= -magnitude;
			p[NZ] /= -magnitude;
		} else {
			p[NX] /= magnitude;
			p[NY] /= magnitude;
			p[NZ] /= magnitude;
		}
	} else {
		p[NX] = 0.;
		p[NY] = 0.;
		p[NZ] = 0.;
	}
}



void make_moviebyu(struct Vertex *vtx)
/*
 * Creates three temporary files, a vertex list, a connectivity list, and
 * a header.  The header is written after the lists
 * so it may contain an image dependent item count.  The 'cat' at the end
 * puts the header at the beginning of the lists.
 * It is difficult to calculate the counts prior to listing because
 * the effects of clipping, resolution and void points.
 */
{
	int lg, lt;						/* image indicies */
	FILE *fp;						/* the output file */
	int nvertex, npolygon;			/* out count of items */
	int nlt, nlg;					/* number of lats and longs in image */
	int nj, np, npt, nedge;			/* Movie.BYU header quantities */

	/* remove any existing temporary files */
	system("rm -f cyconvert.head cyconvert.coor cyconvert.conn");

	/* make movie.byu coordinate list */
	fp = fopen("cyconvert.coor", "w");
	nvertex = 0;
	for (lg = vtx->lgmin; lg <= vtx->lgmax; lg += vtx->lgresol) {
		for (lt = vtx->ltmin; lt <= vtx->ltmax; lt += vtx->ltresol) {
			fprintf(fp, "%f %f %f\n", vtx->pnt[lg][lt][LX],
				vtx->pnt[lg][lt][LY], vtx->pnt[lg][lt][LZ]);
			++nvertex;
		}
	}
	fclose(fp);

	/* make movie.byu connectivity list */
	fp = fopen("cyconvert.conn", "w");
	nlt = (vtx->ltmax - vtx->ltmin + 1) / vtx->ltresol;	/* verticies in y */
	nlg = (vtx->lgmax - vtx->lgmin + 1) / vtx->lgresol;	/* verticies in x */
	npolygon = 0;
	for (lg = 0; lg < nlg-1; ++lg) {				/* for polys in x */
		for (lt = 0; lt < nlt-1; ++lt) {			/* for polys in y */
			fprintf(fp, "%d %d %d -%d\n",
				(lg  )*nlt + (lt  ) + 1,	/* counterclockwise */
				(lg+1)*nlt + (lt  ) + 1,	/* add 1 for counting 1...n */
				(lg+1)*nlt + (lt+1) + 1,
				(lg  )*nlt + (lt+1) + 1
			);
			++npolygon;
		}
	}
	if (!(vtx->gs->flags & FLAG_CARTESIAN) &&
		vtx->lgmin == 0 && (vtx->lgmax == vtx->nlg-1)) {
		/* data is a complete cylinder, make polygons at seam */
		for (lt = 0; lt < nlt-1; ++lt) {
			fprintf(fp, "%d %d %d -%d\n",
				(lg  )*nlt + (lt  ) + 1,
				(0   )*nlt + (lt  ) + 1,
				(0   )*nlt + (lt+1) + 1,
				(lg  )*nlt + (lt+1) + 1
			);
			++npolygon;
		}
	}
	fclose(fp);

	/* create the Movie.BYU header */
	np = 1;								/* number of parts */
	nj = nvertex;						/* number of joints (aka vertex) */
	npt = npolygon;						/* number of polygons */
	nedge = npolygon * 4;				/* number of edges (sortof) */

	fp = fopen("cyconvert.head", "w");
	fprintf(fp, "%d %d %d %d\n", np, nj, npt, nedge);
	fprintf(fp, "%d %d\n", 1, npolygon);
	fclose(fp);

	/* combine the three parts into the output file */
	system("cat cyconvert.head cyconvert.coor cyconvert.conn > movie.byu");
	system("rm -f cyconvert.head cyconvert.coor cyconvert.conn");
}


void make_wavefront(struct Vertex *vtx)
/*
 * Creates one file (wavefront.obj) which can be read into model
 */
{
	int lg, lt;
	FILE *fp;
	int count;
	int nlt, nlg;

	system("rm -f wavefront.obj"); /* remove existing file */

	/* make wavefront vertex list */
	fp = fopen("wavefront.obj", "w");
	count = 0;
	for (lg = vtx->lgmin; lg <= vtx->lgmax; lg += vtx->lgresol) {
		for (lt = vtx->ltmin; lt <= vtx->ltmax; lt += vtx->ltresol) {
			fprintf(fp, "v %f %f %f\n", vtx->pnt[lg][lt][LX],
				vtx->pnt[lg][lt][LY], vtx->pnt[lg][lt][LZ]);
			++count;
		}
	}
	fprintf(fp, "\n\n# of Vertices =  %d \n", count);
	fprintf(fp, "\n\ng d\n\n");

	/* make wavefront.obj connectivity list */
	nlt = (vtx->ltmax - vtx->ltmin) / vtx->ltresol + 1;	/* verticies in y */
	nlg = (vtx->lgmax - vtx->lgmin) / vtx->lgresol + 1;	/* verticies in x */
	count = 0;
	for (lg = 0; lg < nlg-1; ++lg) {				/* for polys in x */
		for (lt = 0; lt < nlt-1; ++lt) {			/* for polys in y */
			fprintf(fp, "fo %d %d %d %d\n",
				lg*nlt + lt + 1,				/* clockwise */
				lg*nlt + lt + 2,
				(lg+1)*nlt + lt + 2,
				(lg+1)*nlt + lt + 1
			);
			++count;
		}
	}
	if (!(vtx->gs->flags & FLAG_CARTESIAN) &&
		vtx->lgmin == 0 && (vtx->lgmax == vtx->nlg-1)) {
		/* data is a complete cylinder, make polygons at seam */
		for (lt = 1; lt < nlt; ++lt) {
			fprintf(fp, "fo %d %d %d %d\n",
				(lg  )*nlt + (lt+1),
				(0   )*nlt + (lt+1),
				(0   )*nlt + (lt  ),
				(lg  )*nlt + (lt  )
			);
			++count;
		}
	}
	fclose(fp);
}

void make_digital_arts_aob(struct Vertex *vtx)
/*
 * Creates file "dig_arts.aob" containing image data in Ascii
 * Object format.
 */
{
	int lg, lt;
	FILE *fp;
	int count;
	int nlt, nlg;

	if (system("rm -f dig_arts.aob")) {
		perror(sys_errlist[errno]);
		exit(2);
	}

	/* make digital_arts vertex list */
	fp = fopen("da_vtx.tmp", "w");
	count = 0;
	for (lg = vtx->lgmin; lg <= vtx->lgmax; lg += vtx->lgresol) {
		for (lt = vtx->ltmin; lt <= vtx->ltmax; lt += vtx->ltresol) {
			fprintf(fp, "v %f %f %f\n", vtx->pnt[lg][lt][LX],
				vtx->pnt[lg][lt][LY], vtx->pnt[lg][lt][LZ]);
			++count;
		}
	}
	fprintf(fp, "\n");
	fclose(fp);

	/* make digital_arts vertex header file */
	fp = fopen("da_vtx.hdr", "w");
	fprintf(fp, "# Ascii Object File: %s\n", filename);
	fprintf(fp, "# vertex list begins (there are %d vertices)\n",count);
	fprintf(fp, "\n");
	fclose(fp);

	/* make digital_arts face list */
	fp = fopen("da_face.tmp", "w");
	nlt = (vtx->ltmax - vtx->ltmin) / vtx->ltresol + 1;	/* verticies in y */
	nlg = (vtx->lgmax - vtx->lgmin) / vtx->lgresol + 1;	/* verticies in x */
	count = 0;
	for (lg = 0; lg < nlg-1; ++lg) {				/* for polys in x */
		for (lt = 0; lt < nlt-1; ++lt) {			/* for polys in y */
			fprintf(fp, "f %d %d %d %d\n",
				lg*nlt + lt + 1,				/* clockwise */
				lg*nlt + lt + 2,
				(lg+1)*nlt + lt + 2,
				(lg+1)*nlt + lt + 1
			);
			++count;
		}
	}
	if (!(vtx->gs->flags & FLAG_CARTESIAN) &&
		vtx->lgmin == 0 && (vtx->lgmax == vtx->nlg-1)) {
		/* data is a complete cylinder, make polygons at seam */
		for (lt = 1; lt < nlt; ++lt) {
			fprintf(fp, "f %d %d %d %d\n",
				(lg  )*nlt + (lt+1),
				(0   )*nlt + (lt+1),
				(0   )*nlt + (lt  ),
				(lg  )*nlt + (lt  )
			);
			++count;
		}
	}
	fclose(fp);

	/* make digital_arts face list header file */
	fp = fopen("da_face.hdr", "w");
	fprintf(fp, "# face list begins (there are %d faces)\n",count);
	fprintf(fp, "surface 0\n");
	fclose(fp);

	/* make digital_arts patch list - not implemented!!! */
	count = 0;

	/* make digital_arts patch list header file */
	fp = fopen("da_patch.hdr", "w");
	fprintf(fp, "# patch list begins (there are %d Patches)\n", count);
	fclose(fp);

	/* combine the parts into the output file */
	if (system("cat da_vtx.hdr da_vtx.tmp da_face.hdr da_face.tmp da_patch.hdr \
				> dig_arts.aob")
		< 0) {
		perror(sys_errlist[errno]);
		exit(2);
	}
	system("rm -f da_vtx.hdr da_vtx.tmp da_face.hdr da_face.tmp da_patch.hdr");
}
