/****************************************************************************
 * xview_mi.c
 * Author Brian Welcker; mods Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module joins with the mouse user interface module to provide a point-
and-click interface for workstations under Open Look
*/

/* Notes-
   -FRAME_DONE_PROC should probably have a different paramter list than
    that used by quit_callback
   -Need to add help files to appropriate directory
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <sys/param.h>
#include <sys/time.h>
#include "ge_error.h"
#include "mi.h"
#include <xview/xview.h>
#include <xview/frame.h>
#include <xview/panel.h>
#include <xview/font.h>
#include <xview/textsw.h>
#include <xview/scrollbar.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>

/* These should have been defined by the includes, but if not... */
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif
#ifndef NULL
#define NULL 0
#endif

/* These are used to communicate with the drawcgm X device drivers */
extern Window window_id;
extern Window top_window_id;
extern Display *display_ptr;

static Display *display;
static Frame frame, loadframe, distanceframe;
static Canvas canvas;
static Menu filemenu, controlmenu;
static MousePosition mouseup, mousedown;
static Panel_item distance_label, load_name, load_dir;
static Cursor trans_cursor, rotate_cursor, arrow_cursor, wait_cursor;

/* The following flag is true if the 'wait' cursor has been defined.
 * (Note that the renderer may also define it, without setting this
 * flag).
 */
static int wait_cursor_up= 0;

static void show_load_frame(menu,menu_item)
/* Called when base frame load button is pushed */
Menu menu;
Menu_item menu_item;
{
  char path_string[MAXPATHLEN];
  static int first_pass= 1;

  ger_debug("show_load_frame");

  /* If the frame has never been displayed, fill in the directory. */
  if (first_pass) {
    getwd(path_string);
    xv_set(load_dir, PANEL_VALUE, path_string, NULL);
    first_pass= 0;
  }
  xv_set(loadframe, XV_SHOW, TRUE, NULL);
}

static int handle_load(item, event)
Panel_item item;
Event *event;
{
  char file_string[MAXPATHLEN];

  ger_debug("handle_load");

  sprintf(file_string, "%s/%s", (char *)xv_get(load_dir, PANEL_VALUE),
	  (char *)xv_get(load_name, PANEL_VALUE));
  miu_load(file_string);

  /* Put the wait cursor up while load is in progress */
  if (!wait_cursor_up) {
    XDefineCursor(display, window_id, wait_cursor);
    wait_cursor_up= 1;
    XFlush(display);
  }

  if ((int)xv_get(loadframe, FRAME_CMD_PUSHPIN_IN) == FALSE)
    xv_set(loadframe, XV_SHOW, FALSE, NULL);

  return( XV_OK );
}

static int handle_load_cancel(item, event)
Panel_item item;
Event *event;
{
  ger_debug("handle_load_cancel");

  if ((int)xv_get(loadframe, FRAME_CMD_PUSHPIN_IN) == FALSE)
    xv_set(loadframe, XV_SHOW, FALSE, NULL);

  return( XV_OK );
}

static void show_distance(menu,menu_item)
Menu menu;
Menu_item menu_item;
{
  char dist_string[64];

  ger_debug("show_mouse");

  sprintf(dist_string, "Distance: %f", miu_viewing_distance()); 
  xv_set(distance_label, PANEL_LABEL_STRING, dist_string,
	 NULL);
  xv_set(distanceframe,
	 XV_X, xv_get(frame, XV_WIDTH) / 2,
	 XV_Y, xv_get(frame, XV_HEIGHT),
	 NULL);
  xv_set(distanceframe, XV_SHOW, TRUE,
	 FRAME_CMD_PUSHPIN_IN, TRUE,
	 NULL);
}

static void handle_distance(item, value, event)
Panel_item item;
int value;
Event *event;
{
  float scale;
  char dist_string[64];

  ger_debug("handle_distance");

  scale = (float)value/100;

  miu_approach_model(scale);
  xv_set(item,  PANEL_VALUE, 100, NULL);
  (void)sprintf(dist_string, "Distance: %f", miu_viewing_distance());
  xv_set(distance_label, PANEL_LABEL_STRING, dist_string,
	 NULL);
}

static void handle_mouse(window, event)
Xv_Window window;
Event *event;
{
  ger_debug("handle_mouse");

  if (event_is_down(event)) {
    switch (event_action(event)) {
    case ACTION_SELECT :
      mousedown.maxx = xv_get(window, XV_WIDTH);
      mousedown.maxy = xv_get(window, XV_HEIGHT);
      mousedown.x = event_x(event);
      mousedown.y = mousedown.maxy - event_y(event);
      XDefineCursor(display, window_id, rotate_cursor); 
      break;
    case ACTION_ADJUST:
      mousedown.maxx = xv_get(window, XV_WIDTH);
      mousedown.maxy = xv_get(window, XV_HEIGHT);
      mousedown.x = event_x(event);
      mousedown.y = mousedown.maxy - event_y(event);
      XDefineCursor(display, window_id, trans_cursor); 
      break;
    default:
      /* nothing */ ;
    }
  } else {
    switch (event_action(event)) {    
    case ACTION_SELECT :
      mouseup.maxx = xv_get(window, XV_WIDTH);
      mouseup.maxy = xv_get(window, XV_HEIGHT);
      mouseup.x = event_x(event);
      mouseup.y = mouseup.maxy - event_y(event);
      XDefineCursor(display, window_id, arrow_cursor); 
      miu_rotate_model(mousedown, mouseup);
      break;
    case ACTION_ADJUST:
      mouseup.maxx = xv_get(window, XV_WIDTH);
      mouseup.maxy = xv_get(window, XV_HEIGHT);
      mouseup.x = event_x(event);
      mouseup.y = mouseup.maxy - event_y(event);
      XDefineCursor(display, window_id, arrow_cursor); 
      miu_translate_model(mousedown, mouseup);
      break;
    default:
      /* nothing */ ;
    }
  }
}

static void quit_callback( menu, menu_item )
/* This routine handles the quit button */
Menu menu;
Menu_item menu_item;
{
  ger_debug("quit_callback");
  miu_exit(); /* Takes effect on exit from mil_event */
}

static void next_callback( menu, menu_item )
/* This routine handles the next button */
Menu menu;
Menu_item menu_item;
{
  ger_debug("next_callback");
  miu_done(); /* Takes effect on exit from mil_event */
}

static void canvas_resize_proc( canvas, width, height )
/* Redraw the model in the event of a resize */
Canvas canvas;
int width;
int height;
{
  ger_debug("canvas_resize_proc");
  miu_redraw(); /* Takes effect on exit from mil_event */
}

static void canvas_repaint_proc( canvas, paint_window, repaint_area )
/* Redraw the model */
Canvas canvas;
Xv_Window paint_window;
Rectlist *repaint_area;
{
  ger_debug("canvas_repaint_proc");
  miu_redraw(); /* Takes effect on exit from mil_event */
}

void mil_setup( id_string, argc, argv )
char *id_string;
int argc;
char *argv[];
/* Initializes this user interface module. Called only once. */
{
  Panel panel;
  Atom atomcolormapwindows;
  int captive= 0;
  int i;

  ger_debug("mil_setup");

  /* Check the command line for the -captive switch */
  for ( i= 1; i<(argc-1); i++ )
    if ( !(strcmp(argv[i],"-captive")) ) captive= 1;

  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);

  frame = (Frame)xv_create(NULL, FRAME, 
			   FRAME_CLOSED, FALSE,
			   FRAME_LABEL, id_string,
			   FRAME_DONE_PROC, quit_callback,
			   NULL);
  display_ptr = display = (Display *)xv_get(frame, XV_DISPLAY); 

  canvas = (Canvas)xv_create(frame, CANVAS,
			     XV_X, 0,
			     XV_Y, 40,
			     XV_HEIGHT, 256,
			     XV_WIDTH, 256,
			     CANVAS_REPAINT_PROC, canvas_repaint_proc,
			     CANVAS_RESIZE_PROC, canvas_resize_proc,
			     XV_HELP_DATA, "P3D:Canvas",
			     NULL);

  xv_set(canvas_paint_window(canvas),
	 WIN_EVENT_PROC, handle_mouse,
	 WIN_CONSUME_EVENTS, WIN_MOUSE_BUTTONS, NULL,
	 NULL); 

  panel = (Panel)xv_create(frame, PANEL,
			   XV_X, 0,
			   XV_Y, 0,
			   XV_WIDTH, 256,
			   XV_HEIGHT, 40,
			   NULL);
  filemenu = (Menu)xv_create(NULL, MENU,
			 MENU_TITLE_ITEM, "File",
			 MENU_ITEM,
			   MENU_STRING, "Load...",
			   MENU_NOTIFY_PROC, show_load_frame,
			   XV_HELP_DATA, "P3D:Load",
			   MENU_FEEDBACK, (captive ? False : True),
			   NULL,
			 MENU_ITEM,
			   MENU_STRING, "Next",
			   MENU_NOTIFY_PROC, next_callback,
			   XV_HELP_DATA, "P3D:Next",
			   NULL,
			 MENU_ITEM,
			   MENU_STRING, "Quit",
			   MENU_NOTIFY_PROC, quit_callback,
			   XV_HELP_DATA, "P3D:Quit",
			   NULL,
			 NULL);
  controlmenu= (Menu)xv_create(NULL, MENU,
			 MENU_ITEM,
			   MENU_STRING, "Distance..",
			   MENU_NOTIFY_PROC, show_distance,
			   XV_HELP_DATA, "P3D:Distance",
			   NULL,
			 NULL);
  (void)xv_create(panel, PANEL_BUTTON,
		  PANEL_LABEL_STRING, "File",
		  PANEL_ITEM_MENU, filemenu,
		  XV_HELP_DATA, "P3D:FileMenu",
		  NULL);
  (void)xv_create(panel, PANEL_BUTTON,
		  PANEL_LABEL_STRING, "Controls",
		  PANEL_ITEM_MENU, controlmenu,
		  XV_HELP_DATA, "P3D:ControlMenu",
		  NULL);
  window_fit(frame);

  arrow_cursor = XCreateFontCursor(display, XC_top_left_arrow);
  trans_cursor = XCreateFontCursor(display, XC_fleur);
  rotate_cursor = XCreateFontCursor(display, XC_exchange);  
  wait_cursor = XCreateFontCursor(display, XC_watch);

  loadframe = (Frame)xv_create(frame, FRAME_CMD,
			       FRAME_LABEL, "P3D:Load",
			       NULL);
  panel = (Panel)xv_get(loadframe, FRAME_CMD_PANEL);
  load_dir = xv_create(panel, PANEL_TEXT,
		       XV_HELP_DATA, "P3D:Directory",
		       PANEL_LABEL_STRING, "Directory:",
		       PANEL_VALUE_DISPLAY_LENGTH, 30,
		       NULL);
  load_name = xv_create(panel, PANEL_TEXT,
			XV_HELP_DATA, "P3D:File",
			PANEL_NEXT_ROW, -1,
			PANEL_LABEL_STRING, "       File:",
			PANEL_VALUE_DISPLAY_LENGTH, 30,
			NULL);
  (void)xv_create(panel, PANEL_BUTTON,
		  XV_X, 50,
		  XV_HELP_DATA, "P3D:LoadButton",
		  PANEL_NEXT_ROW, -1,
		  PANEL_LABEL_STRING, "Load",
		  PANEL_NOTIFY_PROC, handle_load,
		  NULL);
  (void)xv_create(panel, PANEL_BUTTON,
		  XV_HELP_DATA, "P3D:CancelButton",
		  PANEL_LABEL_STRING, "Cancel",
		  PANEL_NOTIFY_PROC, handle_load_cancel,
		  NULL);
  window_fit(panel);
  window_fit(loadframe);

  distanceframe = (Frame)xv_create(frame, FRAME_CMD,
				XV_WIDTH, 200,
				XV_HEIGHT, 60,
				XV_HELP_DATA, "P3D:Distance",
				FRAME_LABEL, "P3D:Distance",
				FRAME_CMD_PUSHPIN_IN, TRUE,
				NULL);
  panel = xv_get(distanceframe, FRAME_CMD_PANEL);
  xv_set(panel, PANEL_LAYOUT, PANEL_VERTICAL, NULL);
  (void)xv_create(panel, PANEL_SLIDER,
		  XV_HELP_DATA, "P3D:DistancePanel",
		  PANEL_LABEL_STRING, "Factor:",		  
		  PANEL_NOTIFY_PROC, handle_distance,
		  PANEL_SHOW_RANGE, FALSE,
		  PANEL_SHOW_VALUE, FALSE,
		  PANEL_VALUE, 100,
		  PANEL_MIN_VALUE, 50,
		  PANEL_MAX_VALUE, 150,
		  NULL);
  distance_label = (Panel_item)xv_create(panel, PANEL_MESSAGE,
					 XV_HELP_DATA, "P3D:DistanceLabel",
					 PANEL_LABEL_BOLD, TRUE,
					 NULL);
  window_fit(panel);
  window_fit(distanceframe);

  /* Make the main window appear */
  xv_set(frame, XV_SHOW, TRUE, NULL);
  xv_set(XV_SERVER_FROM_WINDOW(frame), SERVER_SYNC_AND_PROCESS_EVENTS, 0);

  /* The following uses the ICCCM conventions to get the window manager
   * to try to load both the drawing canvas' and the main window's colormaps.
   * Setting the two window ID's equal causes the drawcgm XWS driver to
   * attach its color map to the correct window.
   */
  top_window_id = window_id = 
    (Window)xv_get(canvas_paint_window(canvas), XV_XID); 
  atomcolormapwindows= XInternAtom(display, "WM_COLORMAP_WINDOWS", False);
  XChangeProperty( display, top_window_id, atomcolormapwindows, 
		  XA_WINDOW, 32, PropModeAppend, 
		  (unsigned char *)&top_window_id, 1 );

  /* Put the wait cursor up while load is in progress */
  if (!wait_cursor_up) {
    XDefineCursor(display, window_id, wait_cursor);
    wait_cursor_up= 1;
  }

  XFlush(display);
}

void mil_update()
/* This routine updates the controls */
{
  char dist_string[64];

  ger_debug("mil_update");
  sprintf(dist_string, "Distance: %f", miu_viewing_distance()); 
  xv_set(distance_label, PANEL_LABEL_STRING, dist_string,
	 NULL);
}

static void sleep_xv_safe()
/* This routine causes 10 microsecond's sleep in an xview-compatible
 * way.  It's from the Xview Programming Manual by Dan Heller, for
 * version 3, p. 490.
 */
{
  int oldmask, mask;
  struct timeval tv;
  tv.tv_sec= 0;
  tv.tv_usec= 10000;
  mask= sigmask(SIGIO);
  mask |= sigmask(SIGALRM);
  oldmask= sigblock(mask);
  if ((select(0,0,0,0,&tv)) == -1) {
    perror("select");
  }

  sigsetmask(oldmask);
}

void mil_event()
{
#ifdef XVIEW2  
  /* The XView version 2 notifier doesn't work very well;  we have
   * to use this appalling wait loop.
   */
  static int first_pass= 1;

  if (wait_cursor_up) {
    XUndefineCursor(display,window_id);
    wait_cursor_up= 0;
  }
  sleep_xv_safe();
  notify_dispatch();
  if (first_pass) {
    miu_redraw();
    first_pass= 0;
  }
  XFlush(display);
#else
  /* Everything seems fine under XView 3 */
  XEvent report;
  if (wait_cursor_up) {
    XUndefineCursor(display,window_id);
    wait_cursor_up= 0;
  }
  XPeekEvent(display,&report); /* block until there is something to do */
  notify_dispatch();
#endif
}

void mil_shutdown()
/* This routine shuts down this user interface module */
{
  ger_debug("mil_shutdown");

  xv_destroy_safe(frame);
}
