/****************************************************************************
 * sphigs_ren.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* This module is the Sun Phigs+ renderer.  It runs under Sun Phigs. */

#include <stdio.h>
#include <math.h>
#include <phigs/phigs.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "assist.h"
 
/* Notes-
   -some implementations might use phigs_ws_type_create to get double buffering
   -some implementations may offer other HLHSR modes worth trying
*/

/* Needed external definitions */
extern char *malloc(), *realloc();

/* 
Variables to provide external hooks to the phigs workstation.
See ren_setup(...) for usage.  phigs_conn_id is used only there;
phigs_wsid is the workstation id used throughout the renderer.
*/
Pconnid phigs_conn_id;
Pint phigs_wsid= 1;
Window window_id;
Window top_window_id;
Display *display_ptr;

/* Phigs identifier constants */
#define ROOT -1
#define HEAD -2
#define VIEWINDEX 1

/* Renderer state */
typedef enum { DEFINING, TRAVERSING } State;
static State current_state= DEFINING;

/* Light source maintenance information */
#define MAX_LIGHTS 16  /* probably more than phigs will use */
static int current_light= 1;
static Pint light_sources[MAX_LIGHTS]=
  { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 };

/* Symbols defined by setup and used in parsing attribute lists */
static Symbol depth_cue_symbol, color_symbol, backcull_symbol, 
	     text_height_symbol, material_symbol;

static void set_area_properties();

static Pvec3 *get_phigs_vector(avect)
Vector avect;
{
	Pvec3 *result;

	result = (Pvec3 *) malloc(sizeof(Pvec3));
	if (!result) 
		ger_fatal("get_phigs_vector: can't allocate a Pvec3!");
	result->delta_x= (Pfloat)vector_x( avect );
	result->delta_y= (Pfloat)vector_y( avect );
	result->delta_z= (Pfloat)vector_z( avect );

	return(result);
}

static Ppoint3 *get_phigs_point(apoint)
Point apoint;
{
	Ppoint3 *result;

	result = (Ppoint3 *) malloc(sizeof(Ppoint3));
	if (!result) 
		ger_fatal("get_phigs_point: can't allocate a Ppoint3!");
	result->x= (Pfloat)point_x( apoint );
	result->y= (Pfloat)point_y( apoint );
	result->z= (Pfloat)point_z( apoint );

	return(result);
}

static Pgcolr *get_phigs_color(clrstruct)
Color clrstruct;
{
	Pgcolr *result;

	ger_debug("get_phigs_color:  getting colors");

	result = (Pgcolr *) malloc(sizeof(Pgcolr));
	if (!result) ger_fatal("get_phigs_color: can't allocate a Pgcolr!");

        result->type= PMODEL_RGB;
	result->val.general.x= color_red(clrstruct);
	result->val.general.y= color_green(clrstruct);
	result->val.general.z= color_blue(clrstruct);

	return(result);
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* This routine dumps a Phigs object. */
void ren_dump(thisgob)
Gob thisgob;
{
	int thisid;
 
	thisid= gob_idnum( thisgob );
	fprintf(stderr,"ren_dump: Dumping gob number %d\n",thisid);
fprintf(stderr,"Dump routine not yet implemented\n");

}

static Pfacet_vdata_list3 *get_loc_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations to a vertex
data list.
*/
{
	Vertex vertex;
	Pfacet_vdata_list3 *result;
	Ppoint3 *vertexlist, *vtx_copy;
                              
	ger_debug("get_loc_vtx: getting vertex list, locs only.");
 
	if ( !(result= 
	       (Pfacet_vdata_list3 *)malloc(sizeof(Pfacet_vdata_list3))) )
	  ger_fatal( "get_loc_vtx: unable to allocate 1 Pfacet_vdata_list3!");

	if ( !(vertexlist= (Ppoint3 *)malloc( vertexcount*sizeof(Ppoint3) ) ))
	  ger_fatal( "get_loc_vtx: unable to allocate %d Ppoint3's!",
		    vertexcount );
	result->num_vertices/*WARNING: may be num_<something else>*/= vertexcount;
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
	  {
	    vertex= first_vertex( vlist );
	    vlist= rest_of_vertices( vlist );
	    vtx_copy->x= (Pfloat)vertex_x(vertex);
	    vtx_copy->y= (Pfloat)vertex_y(vertex);
	    vtx_copy->z= (Pfloat)vertex_z(vertex);
	    vtx_copy++;
	  }

	result->vertex_data.points= vertexlist;
	return( result );
}
 
static Pfacet_vdata_list3 *get_locnrm_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing locations and normals to a vertex
data list.
*/
{
	Vertex vertex;
	Vector normal;
	Pfacet_vdata_list3 *result;
	Pptnorm3 *vertexlist, *vtx_copy;
                              
	ger_debug("get_locnrm_vtx: getting vertex list, locs and normals.");
 
	if ( !(result= 
	       (Pfacet_vdata_list3 *)malloc(sizeof(Pfacet_vdata_list3))) )
	  ger_fatal( "get_loc_vtx: unable to allocate 1 Pfacet_vdata_list3!");

	if ( !(vertexlist= 
	       (Pptnorm3 *)malloc( vertexcount*sizeof(Pptnorm3) ) ))
	  ger_fatal( "get_loc_vtx: unable to allocate %d Pptnorm3's!",
		    vertexcount );
	result->num_vertices/*WARNING: may be num_<something else>*/= vertexcount;
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
	  {
	    vertex= first_vertex( vlist );
	    vlist= rest_of_vertices( vlist );
	    vtx_copy->point.x= (Pfloat)vertex_x(vertex);
	    vtx_copy->point.y= (Pfloat)vertex_y(vertex);
	    vtx_copy->point.z= (Pfloat)vertex_z(vertex);
	    if ( !( normal= vertex_normal(vertex) ) )
	      ger_fatal("get_locnrm_vtx: Vertex missing normal!");
	    vtx_copy->norm.delta_x= (Pfloat)vector_x( normal );
	    vtx_copy->norm.delta_y= (Pfloat)vector_y( normal );
	    vtx_copy->norm.delta_z= (Pfloat)vector_z( normal );
	    vtx_copy++;
	  }

	result->vertex_data.ptnorms= vertexlist;
	return( result );
}
 
static Pfacet_vdata_list3 *get_locclr_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing locations and colors to a vertex
data list.
*/
{
	Vertex vertex;
	Color color;
	Pfacet_vdata_list3 *result;
	Pptco3 *vertexlist, *vtx_copy;
                              
	ger_debug("get_locclr_vtx: getting vertex list, locs and colors.");
 
	if ( !(result= 
	       (Pfacet_vdata_list3 *)malloc(sizeof(Pfacet_vdata_list3))) )
	  ger_fatal( "get_loc_vtx: unable to allocate 1 Pfacet_vdata_list3!");

	if ( !(vertexlist= 
	       (Pptco3 *)malloc( vertexcount*sizeof(Pptco3) ) ))
	  ger_fatal( "get_loc_vtx: unable to allocate %d Pptco3's!",
		    vertexcount );
	result->num_vertices/*WARNING: may be num_<something else>*/= vertexcount;
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
	  {
	    vertex= first_vertex( vlist );
	    vlist= rest_of_vertices( vlist );
	    vtx_copy->point.x= (Pfloat)vertex_x(vertex);
	    vtx_copy->point.y= (Pfloat)vertex_y(vertex);
	    vtx_copy->point.z= (Pfloat)vertex_z(vertex);
	    if ( !( color= vertex_color(vertex) ) )
	      ger_fatal("get_locclr_vtx: Missing vertex color data!");
	    vtx_copy->colr.direct.rgb.red= color_red( color );
	    vtx_copy->colr.direct.rgb.green= color_green( color );
	    vtx_copy->colr.direct.rgb.blue= color_blue( color );
	    vtx_copy++;
	  }

	result->vertex_data.ptcolrs= vertexlist;
	return( result );
}
 
static Pfacet_vdata_list3 *get_locnrmclr_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing locs, normals, and colors to a 
vertex data list.
*/
{
	Vertex vertex;
	Vector normal;
	Color color;
	Pfacet_vdata_list3 *result;
	Pptconorm3 *vertexlist, *vtx_copy;
                              
	ger_debug(
	  "get_locnrmclr_vtx: getting vertex list, locs, norms, and colors.");
 
	if ( !(result= 
	       (Pfacet_vdata_list3 *)malloc(sizeof(Pfacet_vdata_list3))) )
	  ger_fatal( "get_loc_vtx: unable to allocate 1 Pfacet_vdata_list3!");

	if ( !(vertexlist= 
	       (Pptconorm3 *)malloc( vertexcount*sizeof(Pptconorm3) ) ))
	  ger_fatal( "get_loc_vtx: unable to allocate %d Pptconorm3's!",
		    vertexcount );
	result->num_vertices/*WARNING: may be num_<something else>*/= vertexcount;
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
	  {
	    vertex= first_vertex( vlist );
	    vlist= rest_of_vertices( vlist );
	    vtx_copy->point.x= (Pfloat)vertex_x(vertex);
	    vtx_copy->point.y= (Pfloat)vertex_y(vertex);
	    vtx_copy->point.z= (Pfloat)vertex_z(vertex);
	    if ( !( normal= vertex_normal(vertex) ) )
	      ger_fatal("get_locnrm_vtx: Vertex missing normal!");
	    vtx_copy->norm.delta_x= (Pfloat)vector_x( normal );
	    vtx_copy->norm.delta_y= (Pfloat)vector_y( normal );
	    vtx_copy->norm.delta_z= (Pfloat)vector_z( normal );
	    if ( !( color= vertex_color(vertex) ) )
	      ger_fatal("get_locclr_vtx: Missing vertex color data!");
	    vtx_copy->colr.direct.rgb.red= color_red( color );
	    vtx_copy->colr.direct.rgb.green= color_green( color );
	    vtx_copy->colr.direct.rgb.blue= color_blue( color );
	    vtx_copy++;
	  }

	result->vertex_data.ptconorms= vertexlist;
	return( result );
}

static Pcolr_rep *get_vtx_clrs( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine returns a list of colors extracted from the nodes of
a vertex list.
*/
{
	Vertex vertex;
	Pcolr_rep *colorlist, *clr_copy;
                              
	ger_debug("get_vtx_clrs: getting vertex list colors.");
 
	if ( null( vertex_color( first_vertex(vlist) ) ) )
	  ger_fatal("get_vtx_clrs: vertex list has no colors!\n");

	if ( !(colorlist= (Pcolr_rep *)malloc( vertexcount*sizeof(Pcolr_rep) ) ))
		ger_fatal( "get_vtx_clrs: unable to allocate %d Pcolr_rep's!",
			vertexcount );
 
	clr_copy= colorlist;
	while ( vertexcount-- )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		clr_copy->rgb.red= (Pfloat)color_red(vertex_color(vertex));
		clr_copy->rgb.green= (Pfloat)color_green(vertex_color(vertex));
		clr_copy->rgb.blue= (Pfloat)color_blue(vertex_color(vertex));
		clr_copy++;
		}
 
	return( colorlist );
} 

void ren_sphere()
/*
This routine constructs a sphere object.  The sphere is of radius 1,
and centered at the origin.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_sphere");

	ast_sphere(ren_mesh);
}

void ren_cylinder()
/*
This routine constructs a cylinder object.  The cylinder is of radius
1, and is aligned with the z axis.  The ends of the cylinder are at z=0.0
and z=1.0 .
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_cylinder");

	ast_cylinder(ren_mesh);
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine constructs a torus object from the args passed it.  The 
arguments should be the major and minor radii of the torus respectively.  
The torus is drawn so that the hole lies along the z axis.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_torus");

	ast_torus( bigradius, smallradius, ren_mesh );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine constructs a text object from the args passed it.  
The argument list should contain a point, two vectors to define the 
u and v directions of the text plane, and a text string.
*/
{
        Ppoint3 *text_pt;
	Pvec3 dir[2];

	if (current_state != DEFINING) return;

	ger_debug("ren_text");

	text_pt= get_phigs_point(txtpoint);
	dir[0].delta_x= vector_x(uvec);
	dir[0].delta_y= vector_y(uvec);
	dir[0].delta_z= vector_z(uvec);
	dir[1].delta_x= vector_x(vvec);
	dir[1].delta_y= vector_y(vvec);
	dir[1].delta_z= vector_z(vvec);
	ptext3( text_pt, dir, txtstring );

	free( (char *)text_pt );
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polyline object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polyline, which may be concave but 
should not be self-intersecting.  The attributes (color and normal) at 
all vertices are assumed to be the same as those of the first vertex.  
*/
{
  Pfacet_vdata_list3 *vdatalst;
  Pint vertextype;
  Vertex thisvertex;
  
  if (current_state != DEFINING) return;

  ger_debug("ren_polyline");
  
  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  This will determine the vertex type we are dealing with.
   *  Since Phigs+ can't handle normals on polylines, we ignore them.
   */
  thisvertex= first_vertex( vlist );
  if ( !null(vertex_color( thisvertex )) ) vertextype= PVERT_COORD_COLOUR;
  else vertextype= PVERT_COORD;
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  switch ( (int)vertextype ) {
  case (int)PVERT_COORD:
    vdatalst= get_loc_vtx( vlist, count ); break; 
  case (int)PVERT_COORD_COLOUR:
    vdatalst= get_locclr_vtx( vlist, count ); break;
  };
  
  /* 
    Create the polyline.  It goes into the already open structure.
    */
  ppolyline_set3_data( vertextype, PMODEL_RGB, 1, (Pline_vdata_list3 *)vdatalst );
  
  /* Free memory */
  free( vdatalst->vertex_data );
  free( (char *)vdatalst );
}
 
void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polymarker object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polymarker.  The attributes (color 
and normal) at all vertices are assumed to be the same as those of the 
first vertex.  
*/
{
  Vertex thisvertex;
  Pfacet_vdata_list3 *vdatalst;
  Ppoint3 *vertices;
  Ppoint_list3 foo;
  Pcolr_rep *colors;
  Pgcolr thiscolor;
  int i, has_colors= 0;
  
  if (current_state != DEFINING) return;
  
  ger_debug("ren_polymarker");
  
  /* Check for color information */
  if ( !null( vertex_color( first_vertex(vlist) ) ) ) has_colors= 1;
  
  /* Allocate and extract vertex information */
  vdatalst= get_loc_vtx( vlist, count );
  vertices= vdatalst->vertex_data.points;
  if (has_colors) colors= get_vtx_clrs( vlist, count );
  
  /* Create the points.  They go into the already open structure,
   * either all at once (if there are no vertex colors) or as a
   * series of single markers.
   */
  if (!has_colors) {
      foo.num_points=count;
      foo.points=vertices;
      ppolymarker3( &foo );
  }
  else {
    thiscolor.type= PMODEL_RGB;
    foo.num_points=1;
    for (i=0; i<count; i++) {
      thiscolor.val.general.x= colors[i].rgb.red;
      thiscolor.val.general.y= colors[i].rgb.green;
      thiscolor.val.general.z= colors[i].rgb.blue;
      pset_marker_colr( &thiscolor );
      foo.points=vertices+i;
      ppolymarker3( &foo );
    }
  }

  /* Free memory */
  free( (char *)vertices );
  free( (char *)vdatalst );
  if (has_colors) free( (char *)colors );
}
 
void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polygon object from the vertices
in the list passed it.  The parameter count contains the number of
vertices in the list.  The object constructed is a single
polygon, which may be concave but should not be self-intersecting.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.  
*/
{
  Vertex thisvertex;
  Pfacet_vdata_list3 *vdatalst;
  Pint vertextype;
  
  if (current_state != DEFINING) return;

  ger_debug("ren_polygon");

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, add them.  The presence or absence
   *  of colors will then determine the vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) ast_polygon_normals(vlist, count);
  if ( !null(vertex_color( thisvertex )) ) 
    vertextype= PVERT_COORD_COLOUR_NORMAL;
  else vertextype= PVERT_COORD_NORMAL;
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  switch ( (int)vertextype ) {
  case (int)PVERT_COORD_NORMAL:
    vdatalst= get_locnrm_vtx( vlist, count ); break;
  case (int)PVERT_COORD_COLOUR_NORMAL:
    vdatalst= get_locnrmclr_vtx( vlist, count );
  };

  /* 
   * Create the polygon.  It goes into the already open structure.
   */

  pfill_area_set3_data( PFACET_NONE, PEDGE_NONE, vertextype, PMODEL_RGB,
	         (Pfacet_data3 *)NULL, 1, (Pedge_data_list *)NULL,
		 vdatalst );
  
  /* Free memory */
  free( vdatalst->vertex_data );
  free( (char *)vdatalst );
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a triangle mesh object from 
the args (all vertices) in the list passed it.  The object constructed is a 
triangle strip, with each adjacent triple of vertices specifying one triangle.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.
*/
{
  Pfacet_vdata_list3 *vdatalst;
  Pint vertextype;
  Vertex thisvertex;

  if (current_state != DEFINING) return;

  ger_debug("ren_triangle");

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, add them.  The presence or absence
   *  of colors will then determine the vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) ast_triangle_normals(vlist, count);
  if ( !null(vertex_color( thisvertex )) ) 
    vertextype= PVERT_COORD_COLOUR_NORMAL;
  else vertextype= PVERT_COORD_NORMAL;
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  switch ( (int)vertextype ) {
  case (int)PVERT_COORD_NORMAL:
    vdatalst= get_locnrm_vtx( vlist, count ); break;
  case (int)PVERT_COORD_COLOUR_NORMAL:
    vdatalst= get_locnrmclr_vtx( vlist, count );
  };
 
  /* 
   *  Create the triangle strip.  It goes into the already open structure.
   */
  ptri_strip3_data( PFACET_NONE, vertextype, PMODEL_RGB, count, 
	    (Pfacet_data_arr3 *)NULL, vdatalst->vertex_data );
  
  /* Free memory */
  free( vdatalst->vertex_data );
  free( (char *)vdatalst );

}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/* 
This routine causes the renderer to construct a general mesh object from 
the args passed it.  The attributes (color and normal) at all vertices 
are assumed to be the same as those of the first vertex.
*/
{
  Pfacet_vdata_list3 *vdatalst;
  Pint_list_list *vlstlst;
  Pint *thisintlst, vertextype;
  Vertex_list thisfacet, vlist_copy;
  Vertex thisvertex;
  int iloop, facet_vertices;
 
  if (current_state != DEFINING) return;

  ger_debug("ren_mesh: %d vertices, %d facets", vcount, fcount);

  if (fcount<0) {
    ger_error("ren_mesh: invalid facet count!");
    return;
  }

  /* 
   *  Check the first vertex to see if it has color and/or normals.
   *  If there are no normals, add them.  The presence or absence
   *  of colors will then determine the vertex type we are dealing with.
   */
  thisvertex= first_vertex( vlist );
  if ( null(vertex_normal( thisvertex )) ) 
    ast_mesh_normals( vlist, vcount, flist, fcount );
  if ( !null(vertex_color( thisvertex )) ) 
    vertextype= PVERT_COORD_COLOUR_NORMAL;
  else vertextype= PVERT_COORD_NORMAL;
  
  /* 
   * The following calls allocate enough memory for the vertex list
   * (depending on vertex type), and copy the vertex data into the
   * list. 
   */
  switch ( (int)vertextype ) {
  case (int)PVERT_COORD_NORMAL:
    vdatalst= get_locnrm_vtx( vlist, vcount ); break;
  case (int)PVERT_COORD_COLOUR_NORMAL:
    vdatalst= get_locnrmclr_vtx( vlist, vcount );
  };
 
  /* Traverse the vertex list, assigning an index to each vertex. */
  vlist_copy= vlist;
  for (iloop=0; iloop<vcount; iloop++) {
    (void)set_vertex_index( first_vertex(vlist_copy), iloop );
    vlist_copy= rest_of_vertices( vlist_copy );
  }

  /* Create the vertex index data structure.  This involves walking
   * the facets of the facet list one at a time, and building appropriate
   * data structures for each as we go.
   */
  if ( !( vlstlst= (Pint_list_list *)malloc( fcount*sizeof(Pint_list_list) ) ) ) 
    ger_fatal("ren_mesh: cannot allocate %d Pint_list_lists!",fcount);
  for (iloop=0; iloop<fcount; iloop++) {
    thisfacet= first_facet( flist );
    flist= rest_of_facets( flist );
    facet_vertices= list_length( thisfacet );

    vlstlst[iloop].num_lists/*WARNING: may be num_<something else>*/= 1;
    if ( !( vlstlst[iloop].lists= (Pint_list *)malloc(sizeof(Pint_list)) ) )
      ger_fatal("ren_mesh: cannot allocate 1 Pint_list!");
    vlstlst[iloop].lists->num_ints/*WARNING: may be num_<something else>*/= facet_vertices;
    if ( !( thisintlst= (Pint *)malloc(facet_vertices*sizeof(Pint)) ) )
      ger_fatal("ren_mesh: cannot allocte %d Pints!",facet_vertices);
    vlstlst[iloop].lists->ints= thisintlst;

    while ( !null(thisfacet) ) {
      *thisintlst++= vertex_index( first_vertex( thisfacet ) );
      thisfacet= rest_of_vertices( thisfacet );
    }
  }

  pset_of_fill_area_set3_data( PFACET_NONE, PEDGE_NONE, vertextype, PMODEL_RGB,
		   fcount, (Pfacet_data_arr3 *)NULL, (Pedge_data_list_list *)NULL,
		   vlstlst, vdatalst );

  /* Free memory.  This gets a little complicated due to freeing vlstlst. */
  free( vdatalst->vertex_data );
  free( (char *)vdatalst );
  for (iloop=0; iloop<fcount; iloop++) {
    free( (char *)(vlstlst[iloop].lists->ints) );
    free( (char *)(vlstlst[iloop].lists) );
  }
  free( (char *)vlstlst );
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a Bezier patch object from 
the args (16 vertices) in the list passed it.  The object constructed is a 
4 by 4 bicubic Bezier patch.  The attributes (color and normal) at all 
vertices are assumed to be the same as those of the first vertex.
*/
{
	if (current_state != DEFINING) return;

	ger_debug("ren_bezier");
	ast_bezier( vlist, ren_mesh );
}

static void add_trans( trans )
Transformation trans;
/* Add the transformation in this gob to the current structure. */
{
	float *matrix;
 
	ger_debug("add_trans: adding transformation matrix");

	matrix= array2d_to_c( trans );
	pset_local_tran3( (Pmatrix3 *)matrix, PTYPE_POSTCONCAT );
	free( (char *)matrix );
}
 
static void add_attr( attrlist )
Attribute_list attrlist;
/* Add the attributes in this gob to the current structure. */
{
        Pgcolr *color;
	float tempfloat;
	Pair thispair;
	Material new_material;
 
	ger_debug("add_attr:");

	/* check for 'material attribute */
	if ( !null( thispair= symbol_attr( material_symbol, attrlist ) ) ) {
		new_material= pair_material(thispair);
		set_area_properties(new_material);
		};
	/* check for 'color attribute */
 	if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) ) {
		color= get_phigs_color( pair_color( thispair ) );
		ger_debug("           color RGB= (%f %f %f)",
			  color->val.general.x, color->val.general.y, 
			  color->val.general.z );
		pset_line_colr( color );
		pset_text_colr( color );
		pset_marker_colr( color );
		pset_int_colr( color );
		free( (char *)color );
		};
	/* check for 'backcull attribute */
 	if ( !null( thispair= symbol_attr( backcull_symbol, attrlist ) ) ) {
		if ( pair_boolean( thispair ) ) {
			ger_debug("               backface culling on");
			pset_face_cull_mode( PCULL_BACKFACE );
			}
		else {
			ger_debug("               backface culling off");
			pset_face_cull_mode( PCULL_NONE );
			}
		};
	/* check for 'text-height attribute */
 	if ( !null( thispair= symbol_attr( text_height_symbol, attrlist ) ) ) {
		tempfloat= pair_float( thispair );
		ger_debug("           Text height= %f", tempfloat);
		pset_char_ht( tempfloat );
		};
}
 
static void add_kids( kidlist )
Child_list kidlist;
/*  
This routine adds the gobs in the 'children' slot of the given gob
to the current structure.
*/
{
	int thisid;
	Gob thiskid;
 
	ger_debug("add_kids: adding children");
 
 	while ( !null(kidlist) )
		{
		thiskid= first_child( kidlist );
		kidlist= rest_of_children( kidlist );
	    	thisid= gob_idnum( thiskid );
		ger_debug("            adding gob %d as child.",thisid);
		pexec_struct( thisid );
		};
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It adds them to
the Phigs database, and constructs the gob dag as appropriate.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
 
	/*   Open the new structure.  The struct id is the gob id. */
	popen_struct( current_gob ); 

	/* If a transform is present, add it. */
	if ( !null(trans) ) add_trans(trans);
	
	/* If attribute modifiers are present, add them. */
	if ( !null(attr) ) add_attr(attr);

	/* 
	If this gob has primitive operators, add them.  Otherwise,
	add the children of this gob to the current structure.
	*/
	if ( !null(primitive) ) eval_function( primitive_op(primitive) );
	else add_kids( children );
 
	/* Close the new structure, adding it to the gob list. */
	pclose_struct();
}

static void create_light( lightrep )
Plight_src_bundle *lightrep;
/* This routine adds the given light to the light source rep table,
 * so that it is there to be activated by add_lights.
 */
{
  ger_debug("create_light");

  if (current_light <= MAX_LIGHTS)
    pset_light_src_rep( phigs_wsid, current_light++, lightrep );
  else ger_error(
       "create_light: implementation only supports %d simultaneous lights",
		MAX_LIGHTS);
}

static void clear_lights()
/* This routine resets the light source maintenance system */
{
  ger_debug("clear_lights");

  current_light= 1;
}

static void add_lights()
/* This routine takes the lighting information saved up by activate_
 * and deactivate_light, and adds an appropriate element to the
 * currently open structure.
 */
{
  Pint_list activate, deactivate;

  ger_debug("add_lights");

  activate.num_ints/*WARNING: may be num_<something else>*/= current_light - 1;
  activate.ints= light_sources;
  deactivate.num_ints/*WARNING: may be num_<something else>*/= 0;
  deactivate.ints= (Pint *)0;

  pset_light_src_state(&activate,&deactivate);
}

void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine constructs a light object, first adding a translation
and a color if necessary.  All new objects are added to the already
open structure.
*/
{
  Plight_src_bundle lightrep;
  float lightpos[4], *transformed_pos;

  if (current_state != TRAVERSING) return;

  ger_debug("ren_light");

  /* Set the location, and transform to world coordinates */
  lightpos[0]= point_x( location );
  lightpos[1]= point_y( location );
  lightpos[2]= point_z( location );
  lightpos[3]= 1.0;
  transformed_pos= matrix_vector_c( ast_get_matrix(), lightpos ); 

  lightrep.type= PLIGHT_DIRECTIONAL;
  lightrep.rec.directional.colr.type= PMODEL_RGB;
  lightrep.rec.directional.colr.val.general.x= color_red( lightcolor );
  lightrep.rec.directional.colr.val.general.y= color_green( lightcolor );
  lightrep.rec.directional.colr.val.general.z= color_blue( lightcolor );
  lightrep.rec.directional.dir.delta_x= -transformed_pos[0];
  lightrep.rec.directional.dir.delta_y= -transformed_pos[1];
  lightrep.rec.directional.dir.delta_z= -transformed_pos[2];
  create_light( &lightrep );

  free( (char *)transformed_pos );
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine constructs a ambient light object, first adding a 
color if necessary.  All new objects are added to the already
open structure.
*/
{
  Plight_src_bundle lightrep;

  if (current_state != TRAVERSING) return;

  ger_debug("ren_ambient");

  lightrep.type= PLIGHT_AMBIENT;
  lightrep.rec.ambient.colr.type= PMODEL_RGB;
  lightrep.rec.ambient.colr.val.general.x= color_red( lightcolor );
  lightrep.rec.ambient.colr.val.general.y= color_green( lightcolor );
  lightrep.rec.ambient.colr.val.general.z= color_blue( lightcolor );
  create_light( &lightrep );
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine defines a camera.  The camera goes into its own structure.
That structure is not a gob, so it doesn't go in the gob table.
*/
{
        Pview_map3 map;
	Pvec3 vpn;
	Pvec3 *vup;
	Ppoint3  *vrp;
	Pview_rep3 rep;
	Pint error= 0;
	float range_x, range_y, range_z, range;
	Pcolr_rep color_rep;
 
	ger_debug("ren_camera");

	range_x= point_x(lookat) - point_x(lookfrom);
	range_y= point_y(lookat) - point_y(lookfrom);
	range_z= point_z(lookat) - point_z(lookfrom);
	range= sqrt( range_x*range_x + range_y*range_y + range_z*range_z );

	/* Set up viewing parameters */
	map.proj_type= PTYPE_PERSPECT;
	map.proj_ref_point.x= 0.0;
	map.proj_ref_point.y= 0.0;
	map.proj_ref_point.z= range;
	map.win.x_min= -range*tan( DegtoRad * fov/2.0 );
	map.win.x_max= -map.win.x_min;
	map.win.y_min= map.win.x_min;
	map.win.y_max= map.win.x_max;
	map.front_plane= map.proj_ref_point.z + hither;
	map.back_plane= map.proj_ref_point.z + yon;
	map.view_plane= 0.0;

	rep.clip_limit.x_min= map.proj_vp.x_min= 0.0;
	rep.clip_limit.x_max= map.proj_vp.x_max= 1.0;
	rep.clip_limit.y_min= map.proj_vp.y_min= 0.0;
	rep.clip_limit.y_max= map.proj_vp.y_max= 1.0;
	rep.clip_limit.z_min= map.proj_vp.z_min= 0.0;
	rep.clip_limit.z_max= map.proj_vp.z_max= 1.0;
	rep.xy_clip= rep.back_clip= rep.front_clip= PIND_CLIP;

	vpn.delta_x= -range_x;
	vpn.delta_y= -range_y;
	vpn.delta_z= -range_z;
	vup= get_phigs_vector( lookup );
	vrp= get_phigs_point( lookat );

	/* Calculate view orientation matrix and view mapping matrix */
	peval_view_ori_matrix3( vrp, &vpn, vup, &error,
				    rep.ori_matrix );
	if (error) {
	  ger_error("ren_camera: peval_view_ori_matrix3 error %d", error);
	  error= 0;
	}
	peval_view_map_matrix3( &map, &error, rep.map_matrix );
	if (error) {
	  ger_error("ren_camera: peval_view_map_matrix3 error %d",error);
	  error= 0;
	}

	/* set display update state to PDEFER_WAIT, to prevent re-rendering
	 * until ren_render is called.
	 */
	pset_disp_upd_st(phigs_wsid, PDEFER_WAIT, PMODE_NIVE);

	/* Set the view representation */
	pset_view_rep3(phigs_wsid, VIEWINDEX, &rep);

	/* set background color to the desired background color */
	color_rep.rgb.red= color_red( background );
	color_rep.rgb.green= color_green( background );
	color_rep.rgb.blue= color_blue( background );
	pset_colr_rep( phigs_wsid, 0, &color_rep );

	/* Clean up */
	free( (char *)vup );
	free( (char *)vrp );
}

void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine renders the object given, in the current rendering environment.
 */
{
  static int old_height= 0, old_width= 0;
  int thisid;
  
  thisid= gob_idnum( thisgob );
  
  ger_debug("ren_render: rendering object given by gob %d", thisid);
  
  /* Tell Phigs to check for a resize of its window.  This is
   * relevant only if we are drawing into a UI's X window.
   */
  if (display_ptr) { /* we are drawing into a UI's window */
    Pescape_in_data indata;
    int x, y;
    unsigned int width, height, border_width, depth;
    Window root;
    Plimit ws_vp;
    
    if (!XGetGeometry( display_ptr, window_id, &root, &x, &y,
		      &width, &height, &border_width, &depth )) {
      fprintf(stderr,"ren_render: XGetGeometry failed!\n");
      exit(2);
    }

    if ((width != old_width) || (height != old_height)) { /* resize occurred */
      old_width= width;
      old_height= height;
      indata.escape_in_u8.ws_id= phigs_wsid;
      pescape( PUESC_RASTER_RESIZE, &indata, NULL, NULL );

      ws_vp.x_min= 0.0;
      ws_vp.y_min= 0.0;
      ws_vp.x_max= width - 1;
      ws_vp.y_max= height - 1;
      pset_ws_vp( phigs_wsid, &ws_vp );
    }
  }
  
  /* set display update state to PDEFER_WAIT */
  pset_disp_upd_st(phigs_wsid, PDEFER_WAIT, PMODE_NIVE);
  
  /* Open and empty HEAD structure, then refill and close it. */
  popen_struct( HEAD );
  pempty_struct( HEAD );
  if (!null(thistrans)) add_trans(thistrans);
  if (!null(thisattrlist)) add_attr(thisattrlist);
  add_lights();
  pexec_struct( thisid );
  pclose_struct();
  
  /* set display update state to PDEFER_ASAP, causing redraw */
  pset_disp_upd_st(phigs_wsid, PDEFER_ASAP, PMODE_NIVE);
}

static void fast_traverse(thisgob)
Gob thisgob;
/* This routine traverses a gob DAG, by handling its transformation 
 * and attributes and either triggering execution of its primitive 
 * or rendering its children.  It is used to traverse the lighting DAG.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  float *transmatrix;
  Primitive newprim;
  Child_list kidlist;

  thisid= gob_idnum( thisgob );
  ger_debug("fast-traverse: traversing object given by gob %d", thisid);

  /* If there is a transformation, add it to the transformation stack */
  if ( !null( newtrans=gob_trans(thisgob) ) ) {
    transmatrix= array2d_to_c( newtrans );
    ast_push_matrix(transmatrix);
  }
  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );

  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) {
    eval_function( primitive_op(newprim) );
  }
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_traverse( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );
    }
  }

  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );
  if ( !null(newtrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
    }
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine adds the given object to the lights environment. */
{
  int thisid;
  static int lastid= -1; /* impossible id number */
  float *transmatrix;

  thisid= gob_idnum( thisgob );
  ger_debug("ren_traverselights: setting lights to gob %d",thisid);
 
  if (thisid != lastid ) { /* Changed since last render */

    /* Set up for traversal */
    current_state= TRAVERSING;
    clear_lights();

    /* If there is a transformation, add it to the transformation stack */
    if ( !null( thistrans ) ) {
      transmatrix= array2d_to_c( thistrans );
      ast_push_matrix(transmatrix);
    }

    /* Check for and if necessary handle new attributes */
    if ( !null( thisattrlist ) ) ast_push_attributes( thisattrlist );

    fast_traverse( thisgob );

    /* Pop attributes and transformations, and clean up. */
    if ( !null(thisattrlist) ) ast_pop_attributes( thisattrlist );
    if ( !null(thistrans) ) {
      ast_pop_matrix();
      free( (char *)transmatrix );
    }

    lastid= thisid;
    current_state= DEFINING;
  }
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free:  freeing gob %d",thisid);

	/* set display update state to PDEFER_WAIT, to prevent re-rendering
	 * as the deletion occurs.
	 */
	pset_disp_upd_st(phigs_wsid, PDEFER_WAIT, PMODE_NIVE);

	pdel_struct( thisid );
}

static void set_area_properties(material)
Material material;
/* This routine sets up the 'usual' area properties. */
{
	Prefl_props area_props;
	Pgcolr white;

	ger_debug("set_area_properties:");

	white.type= PMODEL_RGB;
	white.val.general.x= 1.0;
	white.val.general.y= 1.0;
	white.val.general.z= 1.0;
	area_props.ambient_coef= material_ka(material);
	if (area_props.ambient_coef < 0.35) area_props.ambient_coef= 0.35;
	area_props.diffuse_coef= material_kd(material);
	if (area_props.diffuse_coef < 0.35) area_props.diffuse_coef= 0.35;
	area_props.specular_coef= material_ks(material);
	area_props.specular_colr= white;
	area_props.specular_exp= material_exp(material);
	area_props.transpar_coef= 1.0;
	pset_refl_props( &area_props );
}

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes, if it is not already initialized. */
{
  static int initialized=0; /* to hold initialization state */
  Pair thispair;
  Pint ws_type;
  Pconnid conn_id;
  
  ger_debug("ren_setup: initializing phigs; renderer %s, device %s",
	    renderer, device);
  
  /* If the renderer was already initialized, return */
  if (initialized) {
    ger_debug("ren_setup: called twice; this call ignored.");
    return;
  }
  else initialized= 1;
  
  /* Initialize needed assist modules */
  ast_init_matrix();
  
  /* 
    Generate some symbols to be used later in attribute list
    parsing.
    */
  depth_cue_symbol= create_symbol("depth-cue");
  color_symbol= create_symbol("color");
  backcull_symbol= create_symbol("backcull");
  text_height_symbol= create_symbol("text-height");
  material_symbol= create_symbol("material");
  
  /* 
    If open_device is false, someone else is initializing Phigs and
    we're done.  Otherwise, initialize Phigs.
    */
  if (open_device || display_ptr) {
    /* set up phigs and workstation */
    popen_phigs( (char*)NULL, PDEF_MEM_SIZE );
    if (display_ptr) {
      /* A user interface has established a window to draw into.  Ignore
       * everything else and open that window for drawing.  This
       * short-cuts the "sun_canvas" device which appears below.
       */
      Pconnid_x_drawable *xconnid;
      xconnid= 
	(Pconnid_x_drawable *)malloc( sizeof(Pconnid_x_drawable) );
      xconnid->display= display_ptr;
      xconnid->drawable_id= window_id;
      conn_id= (char *)xconnid;
      ws_type= phigs_ws_type_create(phigs_ws_type_x_drawable, 0);
      phigs_ws_type_set(ws_type,
			PHIGS_X_BUF_MODE, PHIGS_BUF_DOUBLE,
			PHIGS_DC_MODEL, PHIGS_DC_LIMITS_ADJUST_TO_WINDOW,
			(char *)0);
    }
    else {
      if ( !device || !strcmp(device,"sun_tool") || !strcmp(device,"xws") ) {
	ws_type= phigs_ws_type_x_tool;
	conn_id= 0;
      }
      else if ( !strcmp(device,"sun_canvas") ) {
	ws_type= phigs_ws_type_x_drawable;
	conn_id= phigs_conn_id; 
	/* global variable; must be externally set */
      }
      else if ( !strcmp(device,"cgm") ) {
	ws_type= phigs_ws_type_cgm_out;
	conn_id= outfile ? outfile : "phigs_ren.cgm";
      }
      else ger_fatal("ren_setup: unknown phigs device %s",device);
    }
    /* WARNING: conn_id is now a void pointer */
    popen_ws( phigs_wsid, conn_id, ws_type );
    pset_disp_upd_st( phigs_wsid, PDEFER_WAIT, PMODE_NIVE );
    pset_hlhsr_mode( phigs_wsid, PHIGS_HLHSR_MODE_ZBUFF );
  }
  
  /* create and post root structure */
  popen_struct( ROOT );
  pset_view_ind( VIEWINDEX );
  pset_hlhsr_id( PHIGS_HLHSR_ID_ON );
  pset_linetype(PLINE_SOLID);
  pset_linewidth(1.0);
  pset_marker_type(PMARKER_DOT);
  pset_int_style(PSTYLE_SOLID);
  pset_refl_eqn( PREFL_AMB_DIFF_SPEC );
  pset_int_shad_meth( PSD_COLOUR );
  pexec_struct( HEAD ); /* implicitly creates HEAD struct */
  pclose_struct();
  ppost_struct( phigs_wsid, ROOT, 1.0 );
  
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
  ger_debug("ren_reset: resetting renderer; hard= %d", hard);

  /* Delete everything in the central structure store */
  pdel_all_structs();
  
  /* create and post root structure */
  popen_struct( ROOT );
  pset_view_ind( VIEWINDEX );
  pset_hlhsr_id( PHIGS_HLHSR_ID_ON );
  pset_linetype(PLINE_SOLID);
  pset_linewidth(1.0);
  pset_marker_type(PMARKER_DOT);
  pset_int_style(PSTYLE_SOLID);
  pset_refl_eqn( PREFL_AMB_DIFF_SPEC );
  pset_int_shad_meth( PSD_COLOUR );
  pexec_struct( HEAD ); /* implicitly creates HEAD struct */
  pclose_struct();
  ppost_struct( phigs_wsid, ROOT, 1.0 );

  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    depth_cue_symbol= create_symbol("depth-cue");
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    material_symbol= create_symbol("material");
  }
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	ger_debug("ren_shutdown");

	/* Close workstation and phigs */
	pclose_ws( phigs_wsid );
	pclose_phigs();
}
