/****************************************************************************
 * rui.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides a 'rocket user interface' which lets the user fly around
and examine models with only a keyboard.  A trace file called rui_trace.dat
is maintained;  it contains needed data to continue a flight or construct
a series of P3D 'snap' commands from the points of the flight path.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "control.h"
#include "ren.h"
#include "ui.h"
#include "assist.h"

/* Notes
   -is length_scale needed?
   -is mach_1 needed?
*/
/* 
Default angular and linear accelerations
*/
#define default_alpha 0.1
#define default_accel 1.0

/*
Dynamical constants:
	initial_velocity_multiplier: initial vel= this * view distance
	delta_t: time step (a time)
	torque_scale: linearly increases torque
	mach_1: used to scale velocities (a velocity)
*/
#define initial_velocity_multiplier 0.020661157
#define delta_t 1.0
#define thrust_scale 1.0
#define torque_scale 1.0
#define mach_1 1.0

/* 
   Dynamical variables of 'rocket' - it will initially be placed at
   the lookfrom point heading toward the lookat point.
*/
static float x, y, z;     /* position */
static float px, py, pz;  /* nose pointing direction */
static float vx, vy, vz;  /* velocity components */
static float ax, ay, az;  /* acceleration components */
static float upx, upy, upz; /* components of up vector */
static float pitchx, pitchy, pitchz; /* components of the pitch axis */
static float wx, wy, wz;  /* components of angular velocity */
static float alphax, alphay, alphaz; /* components of angular acceleration */
static float length_scale; /* overall length scale of geometry (a distance) */
static float thrust;  /* magnitude of thrust */

/*
Name of file to hold flight path trace, file pointer for file which holds 
flight path trace, and flag to indicate that tracing is enabled. 
*/
#define TRACE_FILE_NAME "rui_trace.dat"
static FILE *tracefile;
static int trace_on= 0;

/* Help and entry messages */
static char entry_message[]= "Ready for commands (type 'h' for help)\n";
static char help_message[]= "\
The following character inputs are accepted:\n\
\n\
  u	nose up\n\
  d	nose down\n\
  l     nose left\n\
  r     nose right\n\
  c	roll right (clockwise)\n\
  a	roll left (anti-clockwise)\n\
  t	accelerate (apply thrust)\n\
  b     brake (apply negative thrust)\n\
  space continue along current path\n\
  T     start saving a trace file of the flight\n\
  L     load position information\n\
  q	quit flying and go on to next model\n\
  e	exit from the program\n\
  h,?	print this message\n\
\n\
Momentum and angular momentum are conserved and there is no drag, so\n\
if you apply thrust you will accelerate until you apply 'brakes' to\n\
turn off the thrust.  Likewise, if you start the nose rotating in a\n\
given direction, it will accelerate until you apply 'force' in the\n\
opposite direction with the appropriate command.\n\
\n\
Any letter can be prefixed by an integer, which then multiplies \n\
the amount moved or rotated.\n\
"; /* End of help message */

/* Current camera data*/
static Point lookfrom, lookat;
static Vector up;
static float fovea, hither, yon;
static Color background;

/* Current light data */
static Gob lightgob;
static Attribute_list lightattr;

/* Variable to hold initialization state */
static int initialized= 0;

static void align_up()
/* 
This routine removes components of the up vector which are parallel to
the viewing direction, and renormalizes the up vector.
*/
{
  float unorm, dot;

  ger_debug("align_up:");

  dot= upx*px + upy*py + upz*pz;

  upx= upx - dot * px;
  upy= upy - dot * py;
  upz= upz - dot * pz;
  unorm= upx*upx + upy*upy + upz*upz;
  if (unorm==0.0) {
    ger_error("align_up: up vector vanishes, using z direction");
    upx= 0.0; upy= 0.0; upz= 1.0; unorm= 1.0;
  }

  unorm= sqrt(unorm);
  upx= upx/unorm;
  upy= upy/unorm;
  upz= upz/unorm;

  set_vector_x( up, upx );
  set_vector_y( up, upy );
  set_vector_z( up, upz );

  return;
}

static void nose_up( magnitude )
float magnitude;
/* This routine rotates the plane's nose up by 'angle' degrees. */
{
  float radians_per_degree= 3.14159265/180.0;

  ger_debug("nose_up: magnitude= %f",magnitude);

  alphax += magnitude*pitchx;
  alphay += magnitude*pitchy;
  alphaz += magnitude*pitchz;
}

static void nose_left( magnitude )
float magnitude;
/* This routine rotates the plane's nose up by 'angle' degrees. */
{
  float radians_per_degree= 3.14159265/180.0;

  ger_debug("nose_left: magnitude= %f",magnitude);

  alphax += magnitude*upx;
  alphay += magnitude*upy;
  alphaz += magnitude*upz;
}

static void roll_right( magnitude )
float magnitude;
/* This routine rolls the plane right by 'angle' degrees. */
{
  float *matrix, *new_up, up_vec[3];

  ger_debug("roll_right: magnitude= %f",magnitude);

  alphax += magnitude*px;
  alphay += magnitude*py;
  alphaz += magnitude*pz;
}

static void accelerate( accel )
float accel;
/* This routine accelerates the plane. */
{
  ger_debug("accelerate: accel= %f",accel);

  thrust += accel*thrust_scale*length_scale;
}

static void setup_dynamics()
/* This routine sets dynamical variables appropriately */
{
  float vmag;

  upx= point_x(up);
  upy= point_y(up);
  upz= point_z(up);
  
  /* Set the position and velocity information for the plane appropriately */  
  x= point_x(lookfrom);
  y= point_y(lookfrom);
  z= point_z(lookfrom);
  vx= initial_velocity_multiplier * ( point_x(lookat) - x );
  vy= initial_velocity_multiplier * ( point_y(lookat) - y );
  vz= initial_velocity_multiplier * ( point_z(lookat) - z );
  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  px= vx/vmag;
  py= vy/vmag;
  pz= vz/vmag;
  length_scale= vmag*delta_t/mach_1;

  wx= 0.0;
  wy= 0.0;
  wz= 0.0;
  alphax= 0.0;
  alphay= 0.0;
  alphaz= 0.0;

  thrust= 0.0;

  /* Calculate the pitch axis */
  pitchx= py*upz - pz*upy;
  pitchy= pz*upx - px*upz;
  pitchz= px*upy - py*upx;

  /* remove parallel component of up vector */
  align_up();
  ger_debug("setup_dynamics: length_scale= %f",length_scale);
}

static void step_position()
/* 
This routine integrates the plane's position and velocity through 
one time step 
*/
{
  float vmag, pmag, umag;
  float tx, ty, tz; /* components of torque */

  ger_debug("step_position:");

  /* Do the force calculations */
  vmag= sqrt( vx*vx + vy*vy + vz*vz );

  /* Sum forces */
  ax= thrust*px;
  ay= thrust*py;
  az= thrust*pz;

  /* Integrate velocity and position */
  vx= vx + ax*delta_t;
  vy= vy + ay*delta_t;
  vz= vz + az*delta_t;
  x= x + vx*delta_t;
  y= y + vy*delta_t;
  z= z + vz*delta_t;

  /* Calculate torque due to difference between vel. and pointing vec. */
/*
  tx= torque_scale/mach_1 * (vmag/mach_1)
    * (py*(vz - vmag*pz) - pz*(vy - vmag*py));
  ty= torque_scale/mach_1 * (vmag/mach_1)
    * (pz*(vx - vmag*px) - px*(vz - vmag*pz));
  tz= torque_scale/mach_1 * (vmag/mach_1)
    * (px*(vy - vmag*py) - py*(vx - vmag*px));
*/
tx= ty= tz= 0.0;

  /* Integrate angular velocity and position */
  wx= wx + (alphax+tx)*delta_t;
  wy= wy + (alphay+ty)*delta_t;
  wz= wz + (alphaz+tz)*delta_t;
  px= px + (wy*pz - wz*py)*delta_t;
  py= py + (wz*px - wx*pz)*delta_t;
  pz= pz + (wx*py - wy*px)*delta_t;
  upx= upx + (wy*upz - wz*upy)*delta_t;
  upy= upy + (wz*upx - wx*upz)*delta_t;
  upz= upz + (wx*upy - wy*upx)*delta_t;
vx= vmag*px;
vy= vmag*py;
vz= vmag*pz;

  /* Turn off angular thrust */
  alphax= 0.0;
  alphay= 0.0;
  alphaz= 0.0;

  /* Load the new viewing info into the camera */
  set_point_x(lookfrom, x);
  set_point_y(lookfrom, y);
  set_point_z(lookfrom, z);
  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  if (vmag != 0.0) {
    set_point_x(lookat, x + vx/vmag);
    set_point_y(lookat, y + vy/vmag);
    set_point_z(lookat, z + vz/vmag);
  }

  /* Make sure the pointing vector remains normalized */
  pmag= sqrt( px*px + py*py + pz*pz );
  px= px/pmag;
  py= py/pmag;
  pz= pz/pmag;

  /* Make sure the up vector remains normalized */
  umag= sqrt( upx*upx + upy*upy + upz*upz );
  upx= upx/umag;
  upy= upy/umag;
  upz= upz/umag;

  /* Calculate the pitch axis */
  pitchx= py*upz - pz*upy;
  pitchy= pz*upx - px*upz;
  pitchz= px*upy - py*upx;
}

static void report()
/* 
This routine prints relevant data and enters the current info in
the trace file.
*/
{
  float vmag, rmag, roll;
  static float degrees_per_radian= 180.0/3.14159265;
  static float eps= 0.0001;
  static int stepcount= 0;

  vmag= sqrt( vx*vx + vy*vy + vz*vz );
  rmag= degrees_per_radian*sqrt( wx*wx + wy*wy + wz*wz );
  if (abs(pz-1.0)<eps) roll= 0.0;
  else roll= degrees_per_radian*acos( upz/sqrt(1.0-pz*pz) );
  if ( (px*upy-py*upx)>0.0 ) roll= -roll;
  fprintf(stderr,"thrust %f\n", thrust);
  fprintf(stderr,"vel= %f, rot. vel= %f, roll= %f\n", vmag, rmag, roll);
  fprintf(stderr,"position= %f %f %f\n",x,y,z);

  if (trace_on) {
    fprintf(tracefile,
	    "----------------------------------------------------\n");
    fprintf(tracefile,"%d\n",stepcount++);
    fprintf(tracefile,"lookfrom= %f %f %f; lookat= %f %f %f\n",
	    x,y,z,point_x(lookat),point_y(lookat),point_z(lookat));
    fprintf(tracefile,"point= %f %f %f; up= %f %f %f\n",px,py,pz,upx,upy,upz);
    fprintf(tracefile,"vel= %f %f %f; acc= %f %f %f\n",vx,vy,vz,ax,ay,az);
    fprintf(tracefile,"omega= %f %f %f; alpha= %f %f %f\n",
	    wx,wy,wz,alphax,alphay,alphaz);
    fprintf(tracefile,"length_scale= %f, thrust= %f\n",
	    length_scale, thrust);
  }
}

static void start_trace()
/* This routine initiates the recording of a trace file of the flight. */
{
  ger_debug("start_trace: flight tracing enabled");

  /* Open the flight path trace file */
  tracefile= fopen(TRACE_FILE_NAME,"w");
  if (!tracefile) ger_error("Unable to open trace file %s!",TRACE_FILE_NAME);

  trace_on= 1;

  fprintf(tracefile,"fovea= %f, hither= %f, yon= %f\n",fovea,hither, yon);
  fprintf(stderr,"Trace begun.\n");
}

static void ask_and_get(string,pval)
char *string;
float *pval;
/* This routine interactively requests an input value (a float). */
{
  int num_got= 0;
  char input_buf[81];

  while (num_got != 1) {
    fprintf(stderr,string);
    fgets(input_buf,81,stdin);
    num_got= sscanf(input_buf,"%f",pval);
    if (num_got != 1) {
      clearerr(stdin);
      fprintf(stderr,"Couldn't parse that;  please try again.\n");
    }
  }
}

static void load_position()
/* 
This routine interactively loads plane position and velocity information,
allowing the plane to be instantaneously moved.
*/
{
  char junk[81];

  ger_debug("load_position: loading position information");

  /* Get rid of possible junk record in stdin */
  fgets(junk,81,stdin);

  fprintf(stderr,
    "Ready to load position and motion parameters, as from a trace file\n");

  ask_and_get("length scale: ",&length_scale);

  ask_and_get("x lookfrom coordinate: ",&x);
  ask_and_get("y lookfrom coordinate: ",&y);
  ask_and_get("z lookfrom coordinate: ",&z);

  ask_and_get("x pointing component: ",&px);
  ask_and_get("y pointing component: ",&py);
  ask_and_get("z pointing component: ",&pz);

  ask_and_get("x up component: ",&upx);
  ask_and_get("y up component: ",&upy);
  ask_and_get("z up component: ",&upz);

  ask_and_get("x velocity component: ",&vx);
  ask_and_get("y velocity component: ",&vy);
  ask_and_get("z velocity component: ",&vz);

  ask_and_get("x acceleration component: ",&ax);
  ask_and_get("y acceleration component: ",&ay);
  ask_and_get("z acceleration component: ",&az);

  ask_and_get("x angular vel (omega) component: ",&wx);
  ask_and_get("y angular vel (omega) component: ",&wy);
  ask_and_get("z angular vel (omega) component: ",&wz);

  ask_and_get("x angular acceleration (alpha) component: ",&alphax);
  ask_and_get("y angular acceleration (alpha) component: ",&alphay);
  ask_and_get("z angular acceleration (alpha) component: ",&alphaz);

  ask_and_get("thrust: ",&thrust);

  fprintf(stderr,"Position and motion parameters loaded.\n");
}

main(argc, argv)
int argc;
char *argv[];
{
	alisp(argc, argv);
}

void ui_setup( renderer, device, outfile, hints, controller, startframe, 
	      framecopies )
char *renderer, *device, *controller, *outfile;
int startframe, framecopies;
Attribute_list hints;
/* This routine initializes everything. */
{
  ger_debug(
	    "ui_setup: initializing renderer %s, device %s and controller %s",
	    renderer, device, controller);
  
  /* Initialize the controller and renderer */
  if (!initialized) {
    ctrl_setup( controller, startframe, framecopies );
    ren_setup( renderer, device, 1, outfile, hints );
    lookat= create_point();
    lookfrom= create_point();
    up= create_vector();
    background= create_color();
    initialized= 1;
  }
  else {
    ger_error(
	 "ui_setup: called twice, this call interpreted as a hard reset.");
    ui_reset( 1, startframe, framecopies );
  }
}

void ui_reset( hard, startframe, framecopies )
int hard, startframe, framecopies;
/* This routine resets everything */
{
        ger_debug("ui_reset: resetting renderer and controller.");

	/* hard resets require recreation of lisp-side variables */
	if (hard) {
	  lookat= create_point();
	  lookfrom= create_point();
	  up= create_vector();	  
	  background= create_color();
	  lightgob= (Gob)0;
	  lightattr= (Attribute_list)0;;
	  initialized= 1;
	}

	ast_text_reset( hard );
	ast_prim_reset( hard );
	ast_spln_reset( hard );
	ctrl_reset( hard, startframe, framecopies );
	ren_reset( hard );
}

void ui_shutdown()
/* This routine shuts everything down */
{
      ger_debug("ui_shutdown: shutting everything down.\n");

      ren_shutdown();
      ctrl_end();
}

void ui_camera(cam_lookat, cam_lookfrom, cam_up, cam_fovea, cam_hither, 
	       cam_yon, cam_background)
Point cam_lookat, cam_lookfrom;
Vector cam_up;
float cam_fovea, cam_hither, cam_yon;
Color cam_background;
/* This routine handles camera setting */
{
  ger_debug("ui_camera: fovea= %f, hither= %f, yon= %f", cam_fovea,
	    cam_hither, cam_yon);

  /* Save camera values */
  set_point_x( lookat, point_x(cam_lookat) );
  set_point_y( lookat, point_y(cam_lookat) );
  set_point_z( lookat, point_z(cam_lookat) );
  set_point_x( lookfrom, point_x(cam_lookfrom) );
  set_point_y( lookfrom, point_y(cam_lookfrom) );
  set_point_z( lookfrom, point_z(cam_lookfrom) );
  set_vector_x( up, vector_x(cam_up) );
  set_vector_y( up, vector_y(cam_up) );
  set_vector_z( up, vector_z(cam_up) );
  fovea= cam_fovea;
  hither= cam_hither;
  yon= cam_yon;
  set_color_red( background, color_red(cam_background) );
  set_color_green( background, color_green(cam_background) );
  set_color_blue( background, color_blue(cam_background) );
  set_color_alpha( background, color_alpha(cam_background) );

  /* Set all dynamical variables appropriately */
  setup_dynamics();

  /* Set camera */
  ren_camera( cam_lookat, cam_lookfrom, cam_up, cam_fovea, 
	     cam_hither, cam_yon, cam_background );
}

void ui_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine handles actual rendering of the gob. It implements the loop
that watches for keyboard input.
*/
{
  int multiplier= 0, mult_set= 0, redraw= 0, thischar;

  ger_debug("ui_render:");

  ren_render( thisgob, thistrans, thisattrlist );

  /* The following loop handles user interaction */
  align_up(); /* align up vector */
  fputs(entry_message,stderr);
  while (!feof(stdin)) {
    thischar= getchar();
    if (isdigit(thischar)) {
      multiplier= 10*multiplier + (thischar-48);
      mult_set= 1;
    }
    else {
      if (!mult_set) multiplier= 1;
      switch (thischar) {
      case (int)'h': 
      case (int)'?': 
	fputs(help_message,stderr); break; /* 'help' */
      case (int)'u': /* 'up' */
	nose_up( multiplier*default_alpha ); redraw= 1; break;
      case (int)'d': /* 'down' */
	nose_up( -multiplier*default_alpha ); redraw= 1; break;
      case (int)'l': /* 'left' */
        nose_left( multiplier*default_alpha ); redraw= 1; break;
      case (int)'r': /* 'right' */
        nose_left( -multiplier*default_alpha ); redraw=1; break;
      case (int)'c': /* 'clockwise' */
	roll_right( multiplier*default_alpha ); redraw= 1; break; 
      case (int)'a': /* 'anticlockwise' */
	roll_right( -multiplier*default_alpha ); redraw= 1; break; 
      case (int)'t': /* 'thrust' */
	accelerate( multiplier*default_accel ); redraw= 1; break;
      case (int)'b': /* 'brake' */
	accelerate( -multiplier*default_accel ); redraw= 1; break; 
      case (int)' ': /* continue along existing path */
	redraw= 1; break;
      case (int)'T': start_trace(); break;
      case (int)'L': load_position(); redraw= 1; break;
      case (int)'q': return; /* 'quit' */
      case (int)'e': ui_shutdown(); exit(0); /* 'exit' */
      }

      if (redraw) {
        step_position();
        align_up(); /* align up vector */
        ren_camera( lookat, lookfrom, up, fovea, hither, yon, background );
        ren_render( thisgob, thistrans, thisattrlist );
	report();
      }
      multiplier= 0;
      mult_set= 0;
      redraw= 0;
    }
  };

  /* The controller's doframe never gets called in this user interface */
}

void ui_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine handles actual rendering of the gob. */
{
  ger_debug("ui_traverselights");

  if ( lightgob && !null(lightgob) ) ungrab(lightgob);
  grab( lightgob= thisgob );
  if ( lightattr && !null(lightattr) ) ungrab(lightattr);
  grab( lightattr= thisattrlist );
  ren_traverselights( thisgob, thistrans, thisattrlist );
}

void ui_free(thisgob)
Gob thisgob;
/* This routine causes the renderer to free storage associated with the gob. */
{
  ger_debug("ui_free:");

  ren_free(thisgob);
}
