/****************************************************************************
 * ray_ren.c
 * Authors Brian Welcker, Joel Welling, Chris Nuuja
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This renderer renders images using the rayshade ray tracer.
*/
#include <stdio.h>
#include <signal.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "assist.h"
#include "ren.h"

/* Rayshade include files */
#include "constants.h"
#include "typedefs.h"
#include "defaults.h"
#include "funcdefs.h"

#ifdef SYSV
#include <sys/types.h>
#include <sys/times.h>
#include <sys/param.h>
#ifdef CRAY
#include <sys/machd.h>
#endif
#else
#include <sys/time.h>
#include <sys/resource.h>
#endif

/* Needed external definitions */
#ifndef NOMALLOCDEF
extern char *malloc(), *realloc();
#endif

static char LastName[10];
/* Symbols defined by ren_setup and used in parsing attribute lists */
static Symbol color_symbol, backcull_symbol, text_height_symbol;
static Symbol material_symbol;
static Symbol x_res_symbol, y_res_symbol, framenum_symbol;
static Symbol grid_symbol, x_grid_symbol, y_grid_symbol, z_grid_symbol;
static Symbol point_diameter_symbol, line_diameter_symbol;
static Symbol text_stroke_width_symbol, text_thickness_symbol;
static Symbol ignore_text_symbol;

/* Type of current traversal */
static enum { LIGHTING_TRAVERSAL, RENDER_TRAVERSAL } traversal_type;

/* Ambient light level */
static ray_Color current_ambient= {0.0, 0.0, 0.0};

/* Number of file to dump and basename */
char outfile_basename[BUFSIZ];

/* Text emulation data */
#define DEFAULT_IGNORE_TEXT 0 /* text prims ignored if nonzero */
static int ignore_text_flag= DEFAULT_IGNORE_TEXT; 
static float current_text_normal[4];

static int Current_Surface=1;
static Color PrevColor=(Color)0;
static Material PrevMaterial=(Material)0;


/* Rayshade global variables */
int               Npoints=0, 
                  CurXSize, 
                  CurYSize, 
                  CurZSize;
ray_ObjList       *CurObj;
ray_TransInfo     *CurTrans = (ray_TransInfo *)0, 
                  CurITrans;
ray_Object        *LastObj;

extern int        WorldXSize, 
                  WorldYSize, 
                  WorldZSize, 
                  nlight,
                  Xres,
                  Yres,
                  maxlevel,
                  Cache,
                  Jittered,
                  ResolutionSet,
                  ContrastSet,
                  SamplesSet,
                  CutoffSet,
                  AdaptiveSet, 
                  JitteredSet;
extern double     hfov,
                  vfov,
                  RedContrast,
                  GreenContrast,
                  BlueContrast,
                  TreeCutoff,
                  aperture,
                  focaldist;
extern ray_Object *World;
extern ray_Vector eyep,
                  lookp,
                  up;
extern ray_Color  background;
extern ray_SurfaceList *Surfaces;
extern ray_Light  light[];
extern char       outfilename[BUFSIZ];
extern int        outfilenumber;

unsigned long   CacheWorked, 
                CacheFailed,
                ShadowHits,
                TotalRays,      /* # of rays */
                HitRays,        /* # of rays which hit something. */
                SuperSampled,   /* # of supersampled pixels. */
                EyeRays,        /* # of eye rays spawned */
                ShadowRays,     /* # of shadow rays spawned */
                ReflectRays,    /* # of reflected rays */
                RefractRays,    /* # of refracted rays */
                BVTests;        /* # of bounding volume tests. */
double          ftmp,           /* Used by fabs() macro. */
                Utime, Stime;   /* User, system CPU time */

static void dump_point( point,p )
Point point;
ray_Vector *p;
/* This routine dumps a point */
{

  ger_debug("dump_point: dumping point");

  p->x = (double) point_x( point );
  p->y = (double) point_y( point );
  p->z = (double) point_z( point );
}

static void dump_vector( vector,v )
Vector vector;
ray_Vector *v;
/* This routine dumps a vector */
{
  ger_debug("dump_vector: dumping vector");
  
  v->x = (double) vector_x( vector );
  v->y = (double) vector_y( vector );
  v->z = (double) vector_z( vector );
}

static void  dump_vertex( vertex,v )
Vertex vertex;
ray_Vector *v;
/* This routine dumps vertices as a rayshade vector pointer */
{
  ger_debug("dump_vertex: coordinates [%f, %f, %f]",
	    vertex_x( vertex ), vertex_y( vertex ), vertex_z( vertex ) );

  v->x = (double) vertex_x( vertex );
  v->y = (double) vertex_y( vertex );
  v->z = (double) vertex_z( vertex );
}

static void dump_color( color,c )
Color color;
ray_Color *c;
/* This routine dumps colors as rayshade color pointerx */
{
  ger_debug("dump_color: values [%f, %f, %f, %f]",
	    color_red(color), color_green(color), color_blue(color),
	    color_alpha(color));

  c->r= (double) color_red( color );
  c->g= (double) color_green( color );
  c->b= (double) color_blue( color );
}

static char *dump_surface()
/*
This routine defines a surface representation
*/
{
  Color color;
  Material material;
  ray_Color amb, diff, spec;
  ray_Surface *s;

  ger_debug("dump_surface: Dumping surface s%d", Current_Surface);

  color = ast_color_attribute( color_symbol );
  material = ast_material_attribute( material_symbol );

  if ( !symbol_equality(color, PrevColor) || ( PrevMaterial != material ) ) 
    /* need to create a new surface */
    {
    diff.r = (double) material_kd(material) * (double) color_red( color );
    diff.g = (double) material_kd(material) * (double) color_green( color );
    diff.b = (double) material_kd(material) * (double) color_blue( color );
    amb.r = (double) material_ka(material) * current_ambient.r * 
	    (double) color_red( color );
    amb.g = (double) material_ka(material) * current_ambient.g * 
	    (double) color_green( color );
    amb.b = (double) material_ka(material) * current_ambient.b * 
	    (double) color_blue( color );
    spec.r = (double) material_ks(material);
    spec.g = (double) material_ks(material);
    spec.b = (double) material_ks(material);
    sprintf(LastName, "s%d", Current_Surface++);
    s = make_surface(LastName, &amb, &diff, &spec, (double)material_exp(material),
                     (double)material_reflect(material),
		     (1.0 - color_alpha( color )), 1.0, 0.0, 0.0);
    Surfaces = add_surface(s, Surfaces);
    PrevColor= color;
    PrevMaterial= material;
  }

  return(LastName);
}

/* 
This routine should dump the renderer's internal definition
of an object. 
*/
void ren_dump( thisgob )
Gob thisgob;
{
  int thisid;

  thisid= gob_idnum( thisgob );
  ger_debug("ren_dump: dump requested for gob number %d", thisid);
}

static char *dump_poly_surface ( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a surface representation of a polygon, averaging
the color at each vertex.
*/
{
  int i = 0, j;
  Color color;
  Material material;
  ray_Color amb, diff, spec;
  ray_Surface *s;
  float r = 0.0, g = 0.0, b = 0.0, a = 0.0;

  ger_debug("dump_poly_surface:");

  for (j = 0; j < count; j++) {
    if ( !null( color = vertex_color( first_vertex( vlist ) ) ) ) {
      r += (float) color_red( color );
      g += (float) color_green( color );
      b += (float) color_blue( color );
      a += (float) color_alpha( color );
      i++;
    }
    vlist = rest_of_vertices( vlist );
  }
  if (i == 0) return( dump_surface() ); /* no vertex colors */
  else {
    ger_debug("                  creating polygon surface v%d", 
	      Current_Surface);
    
    /* calculate and use an average vertex color */
    
    material = ast_material_attribute( material_symbol );
    
    diff.r = (double)material_kd(material) * r / (double)i;
    diff.g = (double)material_kd(material) * g / (double)i;
    diff.b = (double)material_kd(material) * b / (double)i;
    amb.r = (double)material_ka(material) * current_ambient.r * r / (double)i;
    amb.g = (double)material_ka(material) * current_ambient.g * g / (double)i;
    amb.b = (double)material_ka(material) * current_ambient.b * b / (double)i;
    spec.r = material_ks(material);
    spec.g = material_ks(material);
    spec.b = material_ks(material);
    sprintf(LastName, "v%d", Current_Surface++);
    s = make_surface(LastName, &amb, &diff, &spec, material_exp(material),
                     material_reflect(material),
		     (1.0 - (a/(float)i)), 1.0, 0.0, 0.0);
    Surfaces = add_surface(s, Surfaces);
    
    return(LastName);
  }
}

static void dump_trans()
/* This routine sets the CurTrans transformation matrix */
{
  float *matrix;
  register float *m_copy;
  
  ger_debug("dump_trans: Transforming CurTrans matrix");
  
  matrix = ast_get_matrix();
  CurTrans = new_transinfo();
  explicit_trans(CurTrans, 
		 *(matrix+0), *(matrix+1), *(matrix+2),
		 *(matrix+4), *(matrix+5), *(matrix+6),
		 *(matrix+8), *(matrix+9), *(matrix+10),
		 *(matrix+3), *(matrix+7), *(matrix+11),
		 CurTrans);
}

void finish_prim()
/* This routine adds the last primitive to the World object list */
{
  ger_debug("finish_prim: Adding primitive");

  if (LastObj != (ray_Object *)0) {
    dump_trans();
    set_prim_bounds( LastObj );
    add_prim( LastObj, (ray_Object *)0 ); /* This adds it to World */
    transform_bounds( CurTrans, LastObj->bounds );
    invert_trans( &CurITrans, CurTrans );
    LastObj->trans = new_trans( CurTrans, &CurITrans );
    LastObj->texture = ( ray_Texture *)0;
    World->prims++;
  }
  /* Clear CurTrans and LastObj */
  if (CurTrans) {
    free( (char *)CurTrans );
    LastObj = (ray_Object *)0;
    CurTrans = (ray_TransInfo *)0;
  }
}

void ren_sphere()
/*
This routine dumps a sphere primitive object.
*/
{
  ray_Vector v;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_sphere");

  v.x = 0.0;
  v.y = 0.0;
  v.z = 0.0;
  LastObj = maksph(dump_surface(), 1.0, &v);
  finish_prim();
}

void ren_cylinder()
/*
This routine dumps a cylinder primitive object.
*/
{
  ray_Vector v1, v2;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_cylinder");

  v1.x = 0.0;
  v1.y = 0.0;
  v1.z = 0.0;
  v2.x = 0.0;
  v2.y = 0.0;
  v2.z = 1.0;
  LastObj = makcyl(dump_surface(), &v1, &v2, 1.0);
  finish_prim();
}

void ren_torus( bigradius, smallradius )
float bigradius, smallradius;
/*
This routine dumps a torus primitive object.
*/
{
  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_torus");
  ast_torus( bigradius, smallradius, ren_mesh );
}

static void make_located_cylinder( pt1, diameter, depth )
float *pt1;
float diameter, depth;
/* This routine draws a cylinder of the given size at the given point,
 * for purposes of starting or terminating a text stroke. */
{
  ray_Vector v1, v2;

  v1.x= pt1[0];
  v1.y= pt1[1];
  v1.z= pt1[2];
  v2.x= pt1[0] - depth*current_text_normal[0];
  v2.y= pt1[1] - depth*current_text_normal[1];
  v2.z= pt1[2] - depth*current_text_normal[2];
  LastObj = makcyl(dump_surface(), &v1, &v2, diameter/2.0);
  finish_prim();
}

static void make_aligned_block( pt1, pt2, height, depth )
float *pt1, *pt2;
float height, depth;
/* This routine draws a block of the given size between the given points,
 * for purposes of doing one stroke in a text string.
 */
{
  float norm;
  float corner1[4], corner2[4], sep[4], yvec[4];
  float *rot1, *rot2, *roty, *trans, *temp, *matrix;

  /* Calculate coordinates */
  sep[0]= pt2[0] - pt1[0];
  sep[1]= pt2[1] - pt1[1];
  sep[2]= pt2[2] - pt1[2];
  sep[3]= 1.0;
  norm= sqrt( sep[0]*sep[0] + sep[1]*sep[1] + sep[2]*sep[2] );

  /* Corners of axis-aligned box */
  corner1[0]= norm;
  corner1[1]= 0.0;
  corner1[2]= height/2.0;
  corner1[3]= 1.0;
  yvec[0]= 0.0;
  yvec[1]= 1.0;
  yvec[2]= 0.0;
  yvec[3]= 1.0;

  /* Rotate the x axis into the alignment direction of the block.  We
   * have to watch out for the degenerate case where the alignment and
   * the x vector are anti-parallel.  If they are we simply rotate
   * about the y axis.  In all other cases the aligning rotation
   * is unique and we can't get into trouble.
   */
  if ( (sep[0] < 0.0) && (sep[1] == 0.0) && (sep[2] == 0.0) )
    rot1= make_rotate_c( 180.0, 0.0, 1.0, 0.0 );
  else
    rot1= make_aligning_rotation( 1.0, 0.0, 0.0, sep[0], sep[1], sep[2] );

  /* Rotate the y axis into the text normal direction.  We have to
   * watch out for the degenerate case where the text normal and the
   * rotated y vector are anti-parallel.  If they are we simply rotate
   * about the alignment axis.  In all other cases the aligning rotation
   * is unique and we can't get into trouble.
   */
  roty= matrix_vector_c( rot1, yvec );
  if ( (roty[0] == -current_text_normal[0])
      && (roty[1] == -current_text_normal[1])
      && (roty[2] == -current_text_normal[2]) )
    rot2= make_rotate_c( 180.0, sep[0], sep[1], sep[2] );
  else rot2= make_aligning_rotation( roty[0], roty[1], roty[2],
				current_text_normal[0], current_text_normal[1],
				current_text_normal[2] );

  /* Translate the box to the appropriate position */
  trans= make_translate_c( pt1[0], pt1[1], pt1[2] );

  /* Final matrix is the product. */
  temp= matrix_mult_c( rot2, rot1 );
  matrix= matrix_mult_c( trans, temp );

  /* Generate the box */
  ast_push_matrix( matrix );
  LastObj = makbox(dump_surface(), norm/2.0, -depth/2.0, 0.0,
		   norm/2.0, depth/2.0, height/2.0);
  finish_prim();
  ast_pop_matrix();

  /* Clean up */
  free( (char *)rot1 );
  free( (char *)rot2 );
  free( (char *)roty );
  free( (char *)trans );
  free( (char *)temp );
  free( (char *)matrix );
}

static void text_pline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine is used by ast_text to draw line segments.
*/
{
  float *matrix;
  float height, depth, textheight;
  float pt1[4], pt2[4];
  Vertex vertex1, vertex2;

  ger_debug("text_pline:");

  textheight= ast_float_attribute(text_height_symbol);
  height= textheight * ast_float_attribute(text_stroke_width_symbol);
  depth= textheight * ast_float_attribute(text_thickness_symbol);
  
  /* Generate the polyline of text */
  while ( !null(rest_of_vertices(vlist)) ) {
    vertex1= first_vertex(vlist);
    vlist= rest_of_vertices( vlist );
    vertex2= first_vertex(vlist);

    pt1[0]= vertex_x(vertex1);
    pt1[1]= vertex_y(vertex1);
    pt1[2]= vertex_z(vertex1);
    pt1[3]= 1.0;
    pt2[0]= vertex_x(vertex2);
    pt2[1]= vertex_y(vertex2);
    pt2[2]= vertex_z(vertex2);
    pt2[3]= 1.0;

    /* Put in a cylinder to miter the joint */
    make_located_cylinder(pt1, height, depth);

    /* Put in a block to draw the line */
    make_aligned_block(pt1, pt2, height, depth);
  }

  /* We still need the final cylinder */
  make_located_cylinder(pt2, height, depth);

}

static void text_trans(trans)
Transformation trans;
/* This function is used by ast_text to do transformations between letters. */
{
  float *transmatrix, *product;

  /* Premultiply the current text transform by the given transform,
   * and reset the current transform to the result.
   */
  transmatrix= array2d_to_c( trans );
  product= matrix_mult_c( transmatrix, ast_get_matrix() );
  ast_load_matrix( product );
  free( (char *)transmatrix );
  free( (char *)product );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine dumps a text primitive object.
*/
{
  static float *identity_trans= (float *)0;
  float norm;

  ger_debug("ren_text");
  
  if (traversal_type != RENDER_TRAVERSAL) return;

  if (!ignore_text_flag) {
    /* Initialize the transformation matrix */
    if (!identity_trans) identity_trans= make_scale_c(1.0,1.0,1.0);
    
    /* Calculate the text normal direction */
    current_text_normal[0]= 
      vector_y(uvec) * vector_z(vvec) - vector_z(uvec) * vector_y(vvec);
    current_text_normal[1]= 
      vector_z(uvec) * vector_x(vvec) - vector_x(uvec) * vector_z(vvec);
    current_text_normal[2]= 
      vector_x(uvec) * vector_y(vvec) - vector_y(uvec) * vector_x(vvec);
    norm= sqrt( current_text_normal[0]*current_text_normal[0]
	       + current_text_normal[1]*current_text_normal[1]
	       + current_text_normal[2]*current_text_normal[2] );
    if (norm==0.0) {
      ger_error("ren_text: u and v vectors align; prim ignored.");
      return;
    }
    current_text_normal[0]= current_text_normal[0]/norm;
    current_text_normal[1]= current_text_normal[1]/norm;
    current_text_normal[2]= current_text_normal[2]/norm;
    current_text_normal[3]= 1.0;
    
    /* Generate the text string. */
    ast_push_matrix( identity_trans );
    ast_text(txtpoint, uvec, vvec, txtstring, text_trans, text_pline);
    ast_pop_matrix();
  }
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polyline primitive object.
*/
{
  float radius;
  ray_Vector v1, v2;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_polyline: creating polyline as a cylinder object");
	
  if ( count < 2 )
    ger_error("***Error***: not enough vertices\n"); 

  radius= 0.5*ast_float_attribute(line_diameter_symbol);  
  while ( !null( rest_of_vertices( vlist ) ) )
    {
      dump_vertex( first_vertex( vlist ),&v1 );
      dump_vertex( first_vertex(rest_of_vertices((vlist))),&v2);
      LastObj = makcyl( dump_surface(), &v1, &v2, radius );
      finish_prim();
      vlist = rest_of_vertices( vlist );
    }
}

void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polymarker primitive object.
*/
{
  float radius;
  ray_Vector v1;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_polymarker: creating polymarker as sphere object");

  radius= 0.5*ast_float_attribute(point_diameter_symbol);
	
  while ( !null( rest_of_vertices( vlist ) ) )
    {
      dump_vertex(first_vertex( vlist ), &v1 );
      LastObj = maksph(dump_surface(), radius, &v1);
      finish_prim();
      vlist = rest_of_vertices( vlist );
    }
}

void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polygon primitive object.
*/
{
  ray_PointList *polypoints, *point;
  Vertex_list vwalk;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_polygon: making a polygon with %d points", count);

  if ( count < 3 )
    ger_error("***Error***: not enough vertices\n"); 

  /* If it's a triangle, phong-shade it. Otherwise, draw a polygon. */
  if (count==3) ren_triangle( vlist, count );
  else {
    polypoints = (ray_PointList *)0;

    for (vwalk = vlist; !null( vwalk ); vwalk = rest_of_vertices(vwalk) )
      {
	point = (ray_PointList *)malloc( sizeof( ray_PointList ) );
	dump_vertex( first_vertex ( vwalk ), &(point->vec) );
	point->next = polypoints;
	polypoints = point;
      }
    LastObj = makpoly( dump_poly_surface(vlist, count), polypoints, count );
    finish_prim();
  }
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a Bezier patch primitive object.
*/
{
  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_bezier");
  ast_bezier( vlist, ren_mesh );
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/*
This routine dumps a general mesh object from the args passed it.
*/
{
  Vertex_list thisfacet;
  int facet_vertices;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_mesh: using ren_polygon");

  /* Add vertex normals if they are not present */
  if ( null( vertex_normal( first_vertex(vlist) ) ) )
    ast_mesh_normals( vlist, vcount, flist, fcount );

  while (fcount--) {
    thisfacet= first_facet( flist );
    flist= rest_of_facets( flist );
    facet_vertices= length_list( thisfacet );
    ren_polygon( thisfacet, facet_vertices );
  }
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a triangle strip primitive object.
*/
{
  int i, j, even = 0;
  ray_Vector v1, v2, v3;
  ray_Vector n1, n2, n3;
  Vertex vert1, vert2, vert3;

  if (traversal_type != RENDER_TRAVERSAL) return;
  ger_debug("ren_triangle: rendering triangle strip");

  if ( count < 3) {
    ger_error("***Error***: not enough vertices"); 
    return;
  }

  /* Add vertex normals if they are not present */
  if ( null( vertex_normal( first_vertex(vlist) ) ) )
    ast_triangle_normals( vlist, count );

  for (i = 0; i < (count-2); i++) {
    vert1 = first_vertex(vlist);
    dump_vertex(vert1,&v1);
    dump_vector(vertex_normal(vert1),&n1);

    vert2 = first_vertex(rest_of_vertices(vlist));
    dump_vertex(vert2,&v2);
    dump_vector(vertex_normal(vert2),&n2);

    vert3 = first_vertex(rest_of_vertices(rest_of_vertices(vlist)));
    dump_vertex(vert3,&v3);
    dump_vector(vertex_normal(vert3),&n3);
    if (even = (1-even)) {
      LastObj = maktri( PHONGTRI, dump_poly_surface( vlist, 3 ), 
		        &v1, &v2, &v3, &n1, &n2, &n3 );
    }
    else {
      LastObj = maktri(PHONGTRI, dump_poly_surface( vlist, 3 ),
		       &v1, &v3, &v2, &n1, &n3, &n2 );
    }
    finish_prim();
    vlist = rest_of_vertices(vlist);
  }
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It dumps information
about their internal structure.
*/
{
  ger_debug("ren_gob: Defining gob %d.", current_gob);
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine creates a rayshade point light source.
*/
{
  if (traversal_type != LIGHTING_TRAVERSAL) return;
  ger_debug("ren_light: adding light to light array");

  if (nlight >= LIGHTS) {
    ger_error(
     "ren_light: attempt to define %d lights; %d allowed- this light ignored.",
     nlight, LIGHTS);
    return;
  }
  dump_color( lightcolor, &(light[nlight].color) );
  dump_point( location, &(light[nlight].pos) );
  light[nlight].type = LOCAL;
  nlight++;
}

void ren_ambient( lightcolor )
Color lightcolor;
/*                                             
This routine sets the ambient light global variable.
*/
{
  if (traversal_type != LIGHTING_TRAVERSAL) return;
  ger_debug("ren_ambient: setting ambient light level");

  current_ambient.r += color_red ( lightcolor );
  current_ambient.g += color_green( lightcolor );
  current_ambient.b += color_blue( lightcolor );
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, cam_background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color cam_background;
/*
This routine dumps a camera.
*/
{
  ger_debug("ren_camera");

  dump_point( lookfrom, &eyep );
  if (CurTrans) {
    transform_point(&eyep, CurTrans);
    free((char *)CurTrans);
    CurTrans = (ray_TransInfo *)0;
  }	
  dump_point( lookat, &lookp );
  dump_vector( lookup, &up );
  hfov = fov;
  dump_color( cam_background, &background );
}

void internal_render( thisgob )
Gob thisgob;
{
  int thisid; 
  float *transmatrix;
  Attribute_list newattr;
  Transformation newtrans;
  Child_list kidlist;
  Primitive newprim;

  thisid = gob_idnum( thisgob );
  ger_debug("internal_render: rendering object given by gob %d", thisid);

  kidlist = gob_children( thisgob );
  newtrans = gob_trans( thisgob );
  newattr = gob_attr( thisgob );
  newprim = gob_primitive( thisgob );

  /* Push new attributes */
  if ( !null( newattr ) ) ast_push_attributes( newattr );

  /* Push new transformation */
  if ( !null( newtrans ) ) {
    transmatrix = array2d_to_c( newtrans );
    ast_push_matrix( transmatrix );
  }
  
  if ( !null( gob_children( thisgob ) ) )
      while ( !null( kidlist ) ) {
	  internal_render( first_child( kidlist ) );
	  kidlist= rest_of_children( kidlist );
	}
  else
      if ( !null( newprim ) ) {
	eval_function( primitive_op( newprim ) );
      }

  if ( !null( newattr ) ) ast_pop_attributes( newattr );

  if ( !null( newtrans ) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
  }
}

void ren_render(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/* 
This routine sets up and renders the given object.
*/
{
  float *transmatrix;
  ray_ObjList *ListTmp;
  NODE *FrameNum;

  fprintf(stderr,"\n");
  ger_debug("ren_render: starting render");

  /* Push default attributes */
  if ( !null( thisattr ) ) ast_push_attributes( thisattr );

  /* Push initial transformation */
  if ( !null( thistrans ) ) {
    transmatrix = array2d_to_c( thistrans );
    ast_push_matrix( transmatrix );
  }

  traversal_type= RENDER_TRAVERSAL;
  internal_render( thisgob );

  /* Instance initial object and transform */
  if ( !null( thistrans ) ) {
    dump_trans();
    transform_bounds( CurTrans, World->bounds );
    invert_trans( &CurITrans, CurTrans );
    World->trans = new_trans( CurTrans, &CurITrans );
    free( ( char *)CurTrans);
    CurTrans = ( ray_TransInfo *)NULL;
    ast_pop_matrix();
    free( (char *)transmatrix );
  }

  /* Clean up attribute stack */
  if ( !null( thisattr ) ) ast_pop_attributes( thisattr );

  FrameNum = lookup(framenum_symbol);
  if (null(FrameNum))
     {
     fprintf(stderr,"Error, FrameNum not defined in the lisp environment?!?\n");
     exit(1);
     }
  /*   <outfilenumber> and <outfilename> are global variables in outputp.c   */
  outfilenumber = (int) get_number(FrameNum);
  strcpy(outfilename, outfile_basename);

  cleanup();    /* Set variables which weren't set in object definition. */
  startpic();   /* Start new picture. */
  viewing();    /* Set up viewing parameters. */

  /* If a grid was requested in the hints list (see ren_setup), use it.
   * If not, things default to using a list.
   */
  if (WorldXSize) World->type= GRID;
  SetupWorld(); /* Setup the world */

  get_cpu_time(&Utime, &Stime);
  fprintf(stderr,"Preprocessing time:\t");
  fprintf(stderr,"%2.2lfu  %2.2lfs\n",Utime, Stime);
  fprintf(stderr,"Starting trace.\n");
  
  raytrace();   /* Trace the image. */
  endpic();     /* Close the image file. */

  get_cpu_time(&Utime, &Stime);
  TotalRays = EyeRays + ShadowRays + ReflectRays + RefractRays;
  ShadowHits += CacheWorked;
  HitRays += ShadowHits;
  fprintf(stderr,"Eye rays:\t\t\t%ld\n", EyeRays);
  fprintf(stderr,"Shadow rays:\t\t\t%ld\n",ShadowRays);
  fprintf(stderr,"Reflected rays:\t\t\t%ld\n",ReflectRays);
  fprintf(stderr,"Refracted rays:\t\t\t%ld\n",RefractRays);
  fprintf(stderr,"Total rays:\t\t\t%ld\n", TotalRays);
  if (TotalRays != 0.)
    fprintf(stderr,"Intersecting rays:\t\t%ld (%3.3f%%)\n",
	    HitRays, 100. * (float)HitRays / (float)TotalRays);
  if (ShadowRays != 0) {
    if (Cache)
      fprintf(stderr,"Shadow cache hits:\t\t%ld (%ld misses)\n",
	      CacheWorked, CacheFailed);
    fprintf(stderr,"Total shadow hits:\t\t%ld (%3.3f%%)\n",
	    ShadowHits, 100.*(float)ShadowHits / (float)ShadowRays);
  }
  if (!Jittered)
    fprintf(stderr,"Supersampled pixels:\t\t%ld\n",SuperSampled);
  fprintf(stderr,"B.V. intersection tests:\t%ld\n", BVTests);
  print_prim_stats();
  fprintf(stderr,"Total CPU time (sec):\t\t");
  fprintf(stderr,"%2.2lf (%2.2lfu + %2.2lfs)\n",Utime+Stime, Utime, Stime);
  if (TotalRays != 0.)
    fprintf(stderr,"Seconds / ray:\t\t\t%4.4lf\n",
				(Utime + Stime) / (double)TotalRays);
  if (HitRays != 0.)
    fprintf(stderr,"Seconds / intersecting ray:\t%4.4lf\n",
	    (Utime + Stime) / (double)HitRays);
  /*  
      We have to free the world to ensure this object database won't be
      included in the next frame rendered.  This also ensures no
      memory will be wasted.
  */
  Free_World();
  reset_surfaces();
}

void ren_traverselights(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/*
This routine reports a request to traverse the given object in search
of lights.
*/
{
  int thisid,index;
  Child_list kidlist;
  float *transmatrix;

  thisid = gob_idnum( thisgob );
  ger_debug("ren_traverselights: setting lights to gob %d",thisid);

  /* Push default attributes */
  if ( !null( thisattr ) ) ast_push_attributes( thisattr );

  /* Push transformation */
  if ( !null( thistrans ) ) {
    transmatrix = array2d_to_c( thistrans );
    ast_push_matrix( transmatrix );
    dump_trans();
  }

  /* Reset light number and total ambient illumination. */
  /* This memory is reallocated during the rayshade routine "cleanup()",    */
  /* which is called right before the tracing begins.  It would, of course, */
  /* make more sense to allocate this memory just once, but this works for  */
  /* now.                                                                   */
    if (nlight != 0)
       {
       for (index = 0; index < nlight; index++)
          {
	  free((char *)light[index].cache);
	  free((char *)light[index].trans);
	  }
       nlight=0;
       }
  current_ambient.r= 0.0;
  current_ambient.g= 0.0;
  current_ambient.b= 0.0;
  traversal_type= LIGHTING_TRAVERSAL;

  /* Traverse the lighting DAG */
  internal_render(thisgob);

  /* pop tranformation */
  if ( !null( thistrans ) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
  }
  /* Pop default attributes */
  if ( !null( thisattr ) ) ast_pop_attributes( thisattr );
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine does nothing, since all memory is freed after each
rendering anyway.
*/
{
  ger_debug("ren_free:  freeing all memory!\n");
}

void ren_setup(renderer, device, open_device, outfile, hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes the renderer, if it is not already initialized. */
{
  Attribute_list initial;
  Pair thispair;
  int ival;
  extern int Xres, Yres;
  extern char outdevname[32];

  static int initialized=0; /* to hold initialization state */
 
  ger_debug("ren_setup: initializing renderer; renderer %s, device %s",
	    renderer, device);
  
/* Generate some symbols to be used later in attribute list parsing. */
  if (!initialized) {
    nlight=0;
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    grid_symbol= create_symbol("grid-resolution");
    x_grid_symbol= create_symbol("grid-x-resolution");
    y_grid_symbol= create_symbol("grid-y-resolution");
    z_grid_symbol= create_symbol("grid-z-resolution");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    ignore_text_symbol= create_symbol("ignore-text");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");
    framenum_symbol = create_symbol("p3d-framenumber");

    if (*outfile) fprintf(stderr, "outfile: %s\n", outfile);
    else fprintf(stderr,"output to standard out\n");
    if (*outfile) {
      strcpy(outfile_basename, outfile);
    }

    /* If device is defined, set the string.  Otherwise it defaults to 
     * cgmb.
     */
    if (*device) {
      strncpy(outdevname, device, 31);
    }

    /* initialize matrix functions */
    ast_init_matrix();

    /* Find out if we are to ignore text */
    if ( !null( thispair=symbol_attr(ignore_text_symbol,hints) ) ) {
      if ( ignore_text_flag= !null(pair_symbol( thispair )) ) 
	ger_debug("Text primitives will be ignored");
    }

    /* Override default resolution if it is requested in the hints list */
    if ( !null( thispair=symbol_attr(x_res_symbol,hints) ) ) {
      if ( ival= pair_int( thispair ) ) {
	ger_debug("X resolution %d", ival);
	Xres= ival;
      }
    };
    if ( !null( thispair=symbol_attr(y_res_symbol,hints) ) ) {
      if ( ival= pair_int( thispair ) ) {
	ger_debug("Y resolution %d", ival);
	Yres= ival;
      }
    }

    /* Set up use of a grid (rather than a list) if it is requested in
     * the hints list.
     */
    if ( !null( thispair=symbol_attr(grid_symbol,hints) ) ) {
      if ( ival= pair_int( thispair ) ) {
	ger_debug("Grid resolution %d", ival);
	WorldXSize= WorldYSize= WorldZSize= ival;
      }
    }
    else if ( !null( thispair=symbol_attr(x_grid_symbol,hints) ) ) {
      if ( ival= pair_int( thispair ) ) {
	ger_debug("Grid x resolution %d", ival);
	WorldXSize= ival;
      }
      if ( !null( thispair= symbol_attr(y_grid_symbol,hints) ) ) {
	if ( ival= pair_int(thispair) ) {
	  ger_debug("Grid y resolution %d", ival);
	  WorldYSize= ival;
	}
	else WorldYSize= WorldXSize;
      }
      else WorldYSize= WorldXSize;
      if ( !null( thispair= symbol_attr(z_grid_symbol,hints) ) ) {
	if ( ival= pair_int(thispair) ) {
	  ger_debug("Grid z resolution %d", ival);
	  WorldZSize= ival;
	}
	else WorldZSize= WorldXSize;
      }
      else WorldZSize= WorldXSize;
    }
    else WorldXSize= WorldYSize= WorldZSize= 0; /* means don't use a grid */

    /* Initialize the ray tracer */
    ger_debug("ren_setup: initializing rayshade");
    ray_setup();
    init_bounds(World->bounds);
    
    initialized= 1;
  }
  else ger_error("ren_setup: called twice, this call ignored");
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
  ger_debug("ren_reset: resetting renderer; hard= %d", hard);
  
  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    /* 
       The next two lines shouldn't be necessary, since they are taken
       care of every frame
    Free_World();
    reset_surfaces();
    */
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    grid_symbol= create_symbol("grid-resolution");
    x_grid_symbol= create_symbol("grid-x-resolution");
    y_grid_symbol= create_symbol("grid-y-resolution");
    z_grid_symbol= create_symbol("grid-z-resolution");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    ignore_text_symbol= create_symbol("ignore-text");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");
    framenum_symbol = create_symbol("p3d-framenumber");
  }
}

reset_surfaces()
{
   Current_Surface= 1;
   PrevColor = (Color)0;
   PrevMaterial = (Material)0;
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
  ger_debug("ren_shutdown: shutting down output file generator");

  /* Shut down the output file generator (closing any files still open) */
  shutdownpic();
}

char *ren_def_material(thismat)
Material thismat;
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
{
  ger_debug("ren_free_material: doing nothing");
}

#ifdef SYSV
get_cpu_time(usertime, systime)
double *usertime, *systime;
{
	struct tms time;

	(void)times(&time);
	*usertime = (double)time.tms_utime / (double)HZ;
	*systime = (double)time.tms_stime / (double)HZ;
}
#else
get_cpu_time(usertime, systime)
double *usertime, *systime;
{
	struct rusage usage;

	getrusage(RUSAGE_SELF, &usage);

	*usertime = (double)usage.ru_utime.tv_sec +
			(double)usage.ru_utime.tv_usec / 1000000.;
	*systime = (double)usage.ru_stime.tv_sec +
			(double)usage.ru_stime.tv_usec / 1000000.;
}
#endif
