/*
 * outputp.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: outputp.c,v 1.10 1991/12/18 20:24:43 welling Exp $
 *
 * $Log: outputp.c,v $
 * Revision 1.10  1991/12/18  20:24:43  welling
 * Fixed initialization of two variables (offset and irow) in WRITECGM mode.
 *
 * Revision 1.9  1991/12/18  20:19:46  nuuja
 * Added Joel's change to cgm output initialization.
 *
 * Revision 1.8  1991/12/17  20:07:11  welling
 * Tweaked 'NORLE' mode to properly use frame numbers in its dump files;
 * added a 'shutdownpic' routine to the outputp interface that gets
 * called by ren_shutdown;  created a shutdownpic routine for WRITECGM
 * mode which causes all frames rendered to go into a single CGM file,
 * which gets closed at the end.
 *
 * Revision 1.7  1991/12/16  23:56:16  nuuja
 * Added global variable outfilenumber.  This should be set in ray_ren.c
 * Its necessary for multiple .rle frames.  Since a distributed raytracer
 * slave knows nothing about what other frames have been completed.
 * outfilenumber must be set from ray_ren, which will know what frame it is
 * currently doing.
 *
 * Revision 1.6  1991/12/12  20:13:08  nuuja
 * Added an fclose(imgfile) to the endpic routine for RLE files.  This
 * closes the last file outputed to.  Previously, there was no need, since
 * it only drew on picture anyway
 *
 * Revision 1.5  1991/12/09  23:37:50  welling
 * Added #ifdef so that if MAKING_DEPEND is defined, the CGM generator
 * include files are searched.  This matches the default use of the renderer
 * in P3D.
 *
 * Revision 1.4  1991/04/15  19:23:19  welling
 * Replaced old sv_ routines with rle_ equivalents in the part of the code
 * that uses the Utah Raster Library.  This brings it into keeping with
 * the current release of the Utah library.
 *
 * Revision 1.3  91/02/26  19:53:56  welling
 * No longer attempt to undefine unix, which Cray Unicos doesn't like.
 * 
 * Revision 1.2  91/01/25  18:25:38  welling
 * Corrected some missing 'extern int Xres, Yres' lines.
 * 
 * Revision 1.1  91/01/04  18:35:11  welling
 * Initial revision
 * 
 * Revision 3.0.1.3.1 90/09/24 15:00:00 Joel Welling, Pittsburgh Supercomputing
 * added calls to support use of DRAWCGM/CGMGEN for CGM metafile output
 *
 * Revision 3.0.1.3  90/04/04  14:53:04  craig
 * patch5: Lint removal.
 * 
 * Revision 3.0.1.2  90/02/12  16:09:05  craig
 * patch4: Fixed bug with duplicate headers in mtv files when Appending.
 * 
 * Revision 3.0.1.1  89/12/06  16:33:17  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0  89/10/27  02:05:58  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include <math.h>
#include "typedefs.h"
#include "constants.h"
#include "funcdefs.h"

#ifdef MAKING_DEPEND
#define WRITECGM
#endif

#ifdef WRITECGM

#if ( unix && ( !CRAY && !ardent ) )
#define USE_UNIX
#endif

#ifdef VMS
#include descrip
#else
#include <string.h>
#endif
#ifdef ardent
#include <string.h>
#endif

/* Include defs files that allow linkage to Fortran on various systems */
#ifdef USE_UNIX
#include "unix_defs.h"
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"  /* these are also appropriate on Ardent Titan */
#endif

static float *rimage, *gimage, *bimage;
static int irow, offset;

#else
#ifndef NORLE

#include "rle.h"
rle_hdr out_hdr;                      /* rle output file header */
rle_pixel **buffer= (rle_pixel **)0;  /* Output buffer */
extern int gargc;
extern char **gargv;

#endif
#endif

FILE *imgfile;			/* Raster output file */
char outfilename[BUFSIZ];	/* Name of output file */
int outfilenumber;              /* For multiple RLE files */
char outdevname[32];            /* Output device name */
int Appending;			/* Appending to output file? */

/*
 * Convert floating-point to unsigned char (0-255).
 * This could easily be a macro, but the old SGI compilers dump core
 * on it for some reason.
 */
unsigned char
correct(x)
double x;
{
	if (x < 0)
		return 0;
	if (x > 255)
		return 255;
	return x;
}

#ifdef WRITECGM
/**************** CGM support, based on DRAWCGM library  **************/

startpic()
/* This routine prepares the file for writing */
{
  extern int Xres, Yres;
  int ierr= 0, ione= 1;
  float fzero= 0.0;
  static int initialized= 0;

  if (!initialized) {
    if ( !(rimage= (float *)malloc( Xres*Yres*sizeof(float) )) ) {
      fprintf(stderr,
	      "startpic: unable to allocate %d floats for red buffer!\n",
	      Xres*Yres);
      exit(2);
    }
    if ( !(gimage= (float *)malloc( Xres*Yres*sizeof(float) )) ) {
      fprintf(stderr,
	      "startpic: unable to allocate %d floats for grn buffer!\n",
	      Xres*Yres);
      exit(2);
    }
    if ( !(bimage= (float *)malloc( Xres*Yres*sizeof(float) )) ) {
      fprintf(stderr,
	      "startpic: unable to allocate %d floats for blue buffer!\n",
	      Xres*Yres);
      exit(2);
    }
    
    if (*outdevname) csetdev( outdevname, &ierr ); /* defaults to cgmb */
    if (ierr) {
      fprintf(stderr,"startpic: error %d on csetdev; dev name <%s>",
	      ierr, outdevname);
      exit(2);
    }
    
    if (*outfilename) wrcopn( outfilename, &ierr );
    else wrcopn("-",&ierr);
    if (ierr) {
      fprintf(stderr,"startpic: error %d on wrcopn; file %s",
	      ierr, (*outfilename) ? outfilename : "stdout" );
      exit(2);
    }
    initialized=1;
  }

  offset= 0;
  irow= 0; 

  wrbegp(&ierr);
  if (ierr) {
    fprintf(stderr,"startpic: error %d on wrbegp",ierr);
    exit(2);
  }

  wrtcsm(&ione, &ierr);
  if (ierr) {
    fprintf(stderr,"startpic: error %d on wrtcsm",ierr);
    exit(2);
  }

  wrbgdc( &fzero, &fzero, &fzero, &ierr);
  if (ierr) {
    fprintf(stderr,"startpic: error %d on wrbgdc",ierr);
    exit(2);
  }

  wrbgpb( &ierr );
  if (ierr) {
    fprintf(stderr,"startpic: error %d on wrbgpb",ierr);
    exit(2);
  }

}

static float
fcorrect(x)
double x;
/* This routine bounds a float to the range 0.0 t0 1.0. */
{
	if (x < 0.0)
		return 0.0;
	if (x > 1.0)
		return 1.0;
	return x;
}

outline(buf)
ray_Color *buf;
/* This routine stores a scan line. */
{
  extern int Xres, Yres;
  int i;

  for (i=0; i<Xres; i++) {
    *(rimage + offset + i)= fcorrect( (float)buf[i].r );
    *(gimage + offset + i)= fcorrect( (float)buf[i].g );
    *(bimage + offset + i)= fcorrect( (float)buf[i].b );
  }
  irow++;
  offset += Xres;
}

endpic()
/* This routine ends the picture and closes the file. */
{
  extern int Xres, Yres;
  int ierr= 0;
  float fzero= 0.0, maxx, maxy;

  maxy= (float)Yres/(float)Xres;
  if (maxy>1.0) {
    maxx= 1.0/maxy;
    maxy= 1.0;
  }
  else maxx= 1.0;

  wcladc( rimage, gimage, bimage, &Xres, &Yres,
	  &fzero, &fzero, &maxx, &maxy, &maxx, &fzero, &ierr );
  if (ierr) {
    fprintf(stderr,"endpic: error %d on wcladc",ierr);
    exit(2);
  }

  wrendp(&ierr);
  if (ierr) {
    fprintf(stderr,"endpic: error %d on wrendp",ierr);
    exit(2);
  }
}

shutdownpic()
/* This routine closes the CGM file which has been accumulating. */
{
  int ierr= 0;

  wrtend(&ierr);
  if (ierr) {
    fprintf(stderr,"shutdownpic: error %d on wrtend",ierr);
    exit(2);
  }
}
/************* End of CGM support *******************/
#else  /* else clause of ifdef WRITECGM */

#ifndef NORLE
/*
 * Open image file and write RLE header.
 */
startpic()
{
	extern int Xres, Yres;
	extern ray_Color background;
	int rle_backgnd[3];
	char temp[128];

	if (*outfilename) {
	  sprintf(temp,"%s%d.rle",outfilename,outfilenumber);
	  strcpy(outfilename,temp);
	  if (Appending) {
	    /*
	     * If we're appending, we *still* have to write
	     * the Utah Raster header to the file.  This is
	     * due to strangeness in the Utah Raster toolkit,
	     * which does some vital initialization in sv_setup().
	     */
	    imgfile= rle_open_f( "rletocgm", outfilename, "a" );
	  } else
	    imgfile= rle_open_f( "rletocgm", outfilename, "w" );
	} else
	  imgfile = rle_open_f( "rletocgm", (char *)0, "w" ); /*opens stdout*/
	out_hdr.rle_file= imgfile;
	out_hdr.ncolors= 3;
	rle_backgnd[0]= (int)(background.r*255);
	rle_backgnd[1]= (int)(background.g*255);
	rle_backgnd[2]= (int)(background.b*255);
	out_hdr.bg_color= rle_backgnd;
	out_hdr.alpha= 0;
	out_hdr.background= 2;
	out_hdr.xmin= out_hdr.ymin= 0;
	out_hdr.xmax= Xres - 1;
	out_hdr.ymax= Yres - 1;
	out_hdr.ncmap= 0;
	out_hdr.cmaplen= 8;
	out_hdr.cmap= (rle_map *)0;
	RLE_SET_BIT( out_hdr, RLE_RED );
	RLE_SET_BIT( out_hdr, RLE_GREEN );
	RLE_SET_BIT( out_hdr, RLE_BLUE );
	RLE_CLR_BIT( out_hdr, RLE_ALPHA );
	rle_addhist( gargv, (rle_hdr *)0, &out_hdr);
	rle_put_setup(&out_hdr);
	/*
	 * Flush the header.  If we don't, and LINDA forks off
	 * a bunch of workers, strange things will happen (they'll
	 * all flush the buffer when they die, and you end up with
	 * lots of headers at the end of the file).
	 */
	fflush(out_hdr.rle_file);

	/* Allocate a row's worth of image output buffer. */
	if ( rle_row_alloc(&out_hdr, &buffer) < 0 ) {
	  fprintf(stderr,
		  "Unable to allocate image output buffer in startpic!\n");
	  exit(2);
	}
}


/*
 * Write a scanline of output.
 * "buf" is an array of Color structures of size Xres.  Each color
 * component is normalized to [0, 1.].
 */
outline(buf)
ray_Color *buf;
{
	register int i;
	extern int Xres;

	for(i = 0;i < Xres;i++) {
		/*
		 * Scale colors to fit unsigned char and check for
		 * over/underflow.
		 */
		buffer[0][i] = correct(255 * buf[i].r);
		buffer[1][i] = correct(255 * buf[i].g);
		buffer[2][i] = correct(255 * buf[i].b);
	}
	rle_putrow( buffer, Xres, &out_hdr );
}

/*
 * Close image file.
 */
endpic()
{
        rle_puteof(&out_hdr);
	rle_row_free( &out_hdr, buffer );
	fclose(imgfile);
}

shutdownpic()
{
  /* do nothing; no files need to be closed. */
}
#else /* NORLE */

startpic()
{
	extern int Xres, Yres;
	char temp[128];

	if (*outfilename) {
	  sprintf(temp,"%s%d.raw",outfilename,outfilenumber);
	  strcpy(outfilename,temp);
	imgfile = fopen(outfilename,"w");

	if (!Appending)
		fprintf(imgfile,"%d %d\n",Xres, Yres);

	fflush(imgfile);
}

outline(buf)
ray_Color *buf;
{
	register int i;
	extern int Xres;

	for (i = 0; i < Xres; i++) {
		(void)fputc(correct(255.*buf[i].r), imgfile);
		(void)fputc(correct(255.*buf[i].g), imgfile);
		(void)fputc(correct(255.*buf[i].b), imgfile);
	}
	fflush(imgfile);
}

endpic()
{
	(void)fclose(imgfile);
}

shutdownpic()
{
  /* do nothing; no files need to be closed. */
}
#endif /* NORLE */

#endif  /* WRITECGM */
