/****************************************************************************
 * prims.c
 * Author Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

 /*   This file defines the execution of lisp  primitives */

#include "alisp.h"
#include "parse.h"
extern int ENDSESSION;
extern char *getenv();

/* 
Maximum length of file names for files to be loaded, environment
variable to check for a load directory name, and backup load
directory name to search for load files when they are not found
in the current directory.  Note that the file name length includes
the search directory.
*/
#define MAX_FILE_NAME_LENGTH 255
#define ENV_LOADDIR_NAME "P3DLOADDIR"
#ifndef LOAD_PATH
#ifdef VMS
#define LOAD_PATH "p3d$root:[load]"
#else
#define LOAD_PATH "/usr/local/p3d/load/"
#endif
#endif

/*
	Preforms the simple eq function to the point of  eql
*/
NODE *check_eq(arg1,arg2)
NODE *arg1, *arg2;
{
	NODE *result;

	if (floatp(arg1))
		{
		if (getflonum(arg1) == getflonum(arg2))
			result = TRUE_NODE;
		else
			result = NIL;
		}
	else if (fixp(arg1))
		{
		if (getfixnum(arg1) == getfixnum(arg2))
			result = TRUE_NODE;
		else
			result = NIL;
		}
	else if (arg1 == arg2)
		result = TRUE_NODE;

	else if ( (null(arg1)) && (null(arg2)) )
		result = TRUE_NODE;
	else
		result = NIL;

	return(result);
}
	
/*
	Preforms the lisp primitive 'aref'.  <indlist> should be a list of
	integers.  If it's not, errors may occur.  There should be an additional
	check for that.
*/
NODE *reference_array(theArray,indlist)
NODE *theArray,*indlist;
{
	NODE *result,*rest;
	int ind;

	ind = getfixnum(car(indlist));
	if (ind >= array_max(theArray) )
		{
		fprintf(stderr,"ERROR, ARRAY REFERENCE OUT OF BOUNDS: %d \n",ind);
		return(NIL);
		}
	rest = cdr(indlist);

	if (!null(rest))
		result = reference_array(array_ref(theArray,ind),rest);
	else
		result = array_ref(theArray,ind);

	return(result);

}

/*   
     This preforms the lisp primitive 'make-array'.  Note that this is one
     of the few functions that actually allocates more memory
     Defstructs are created as arrays with index zero being the name of the
     structure and every field of the array being a seperate index.  This is
     done transparently in the parse ( defstruct name f1 f2 ...) turns into
     (defun make-name (&key f1 f2 ...) (let ((result (make-array x))) ...).
*/
NODE *make_array(dimlist,initial)
NODE *dimlist,*initial;
{
	NODE *result,*rest;
	int dim,i;

	if (!consp(dimlist))
	   {	
	   if (!fixp(dimlist))
		return( handle_error("Non integer array size\n") );
	   dim = (int) getfixnum(dimlist);
	   rest = NIL;
	   }
	else
	   {
	   if (!fixp(car(dimlist)))
		return( handle_error("Non integer array size\n") );
	   dim = (int) getfixnum(car(dimlist));
	   rest = cdr(dimlist);
	   }
	result = new_node(N_ARRAY);
	array_max(result) = dim;
	array_st(result) = (NODE **) malloc(dim*sizeof(NODE *));

	if (!null(rest))
		for (i=0;i<dim;i++)
			{
			array_ref(result,i) = make_array(rest,
			   top_of(initial));
			initial = rest_of(initial);
			incr_ref(array_ref(result,i));
			}
	else
		for (i=0;i<dim;i++)
			{
			array_ref(result,i) = top_of(initial);
			incr_ref(array_ref(result,i));
			initial = rest_of(initial);
			}
	return(result);

}

/*   
     This function applies <op>, which is '+', '-', '/' or '*' to the list
     of arguments <alist>. <alist> should not contain anything other than
     FIX and FLOAT nodes.  If it does, an error is signaled. If any one of the 
     arguments of <alist> is a FLOAT, the result is FLOAT, else the result is 
     INT.
*/
NODE *apply_op(alist,op)
NODE *alist;
char op;
{
	NODE *addlist,*add_elem,*result;
	int float_result=0;
	float total=0.0;

	if (!consp(alist))
	   {
	   fprintf(stderr,"Error:need at least one argument to %s\n",op);
	   return(NIL);
	   }

	add_elem = car(alist);

	if ( (!fixp(add_elem)) && (!floatp(add_elem)) )
		{
		fprintf(stderr,"Error:Can only %s numbers, not:\n",op);
		print_out(add_elem);
		fprintf(stderr,"\n");
		return(NIL);
		}
	if (floatp(add_elem))
		{
		float_result = 1;
		total = getflonum(add_elem);
		}
	else
		total = (float) getfixnum(add_elem);

	for (addlist=cdr(alist);!null(addlist); addlist=cdr(addlist))
		{
		add_elem = car(addlist);
		if ( (!fixp(add_elem)) && (!floatp(add_elem)) )
		   {
		   fprintf(stderr,"Error:Can only %s numbers, not:\n",op);
		   print_out(add_elem);
		   fprintf(stderr,"\n");
		   return(NIL);
		   }
		if (floatp(add_elem))
			{
			float_result = 1;
			switch(op)
				{
				case '+' : total += getflonum(add_elem); break;
				case '-' : total -= getflonum(add_elem); break;
				case '*' : total *= getflonum(add_elem); break;
				case '/' : if ( getflonum(add_elem) != 0.0)
					   	total /= getflonum(add_elem);
					   else
						return(
						  handle_error("Zero divisor"));
					   break;
						
				}
			}
		else
			switch(op)
				{
				case '+' : total += (float) getfixnum(add_elem);
					   break;
				case '-' : total -= (float) getfixnum(add_elem);
					   break;
				case '*' : total *= (float) getfixnum(add_elem);
					   break;
				case '/' : if ( getfixnum(add_elem) != 0)
					   	total /= (float) 
						   getfixnum(add_elem);
					   else
						return(
						  handle_error("Zero divisor"));
					   break;
				}
		}
	if (float_result)
		{
		result = get_floatrep(total);
		}
	else
		{
		result = get_integerrep((int) total);
		}
	return(result);
}

/*  
   This function applies the primitve 'load' to <file_name>.
   It alters <env> to include the new definitions made in the evaluation of
   the parse of <file_name>.  Note that there is no way to specify a load from
   stdio, it must be from a file.
*/
NODE *apply_load(file_name)
NODE *file_name;
{
	char fname[MAX_FILE_NAME_LENGTH+1];
	char *env_loaddir;

	if (!symbolp(file_name) && (!stringp(file_name)))
		{
		fprintf(stderr,"Error, invalid file name for load\n");
		exit(1);
		}
	/* Try loading from current directory */
	if ( !access(symbol_name(file_name), 4) ) {
		/*  Open new file and push it onto file stack */
   		push_filestack(symbol_name(file_name));
		parse();
		/* pop file off stack, restore old parse variables */
    		pop_filestack(); 
		}
	else { 
	  /* Try directory specified in environment, or default 
	   * load directory 
	   */
	  if ( env_loaddir= getenv(ENV_LOADDIR_NAME) ) {
	    (void)strncpy( fname, env_loaddir, MAX_FILE_NAME_LENGTH );
	    fname[MAX_FILE_NAME_LENGTH]= '\0';
#ifndef VMS
	    (void)strncat( fname, "/", MAX_FILE_NAME_LENGTH-strlen(fname) );
#endif
	  }
	  else (void)strncpy( fname, LOAD_PATH, MAX_FILE_NAME_LENGTH );
	  fname[MAX_FILE_NAME_LENGTH]= '\0';
	  (void)strncat( fname, symbol_name(file_name),
			MAX_FILE_NAME_LENGTH - strlen(fname) );
	  if ( !access(fname,4) ) {
	    /*  Open new file and push it onto file stack */
	    push_filestack(fname);  
	    parse();
	    /* pop file off stack, restore old parse variables */
	    pop_filestack(); 
	  }
	  else 
	    {
	      fprintf(stderr,"Warning - unable to load %s\n\n",
		      symbol_name(file_name));
	      exit(1);
	    }
	}
	return(NIL);
}
/*
    Preforms the lisp primitive 'list'.  <Arglist> can be any length.
*/
NODE *apply_list(Arglist)
NODE *Arglist;
{
	NODE *result;

	if (null(Arglist))
		return(Arglist);
	result = new_node(N_LIST);
	rplaca(result,car(Arglist));
	incr_ref(car(result));

	rplacd(result,apply_list(cdr(Arglist)));
	incr_ref(cdr(result));

	return(result);
}

/*
    preforms the lisp primitive mapcar.  <func> is the function to be
    applied to the argument lists in <arglist>.
*/
NODE *apply_mapcar(func,arglist)
NODE *func, *arglist;
{
	NODE *alist,*result_so_far,*result,*theArgs,*args_so_far,*new;

	if (null(arglist))
		{
		fprintf(stderr,
			"Error: Mapcar requires a non-nil argument list");
		return(NIL);
		}
	result = new_node(N_LIST);
	car(result) = NIL;
	cdr(result) = NIL;
	result_so_far = result;
	while (1)
		{
		theArgs = new_node(N_LIST);
		car(theArgs) = NIL;
		cdr(theArgs) = NIL;
		args_so_far = theArgs;
		for(alist=arglist;!null(alist);alist=cdr(alist))
			{
			if (!consp(car(alist)))
				{
				incr_ref(result_so_far);
				decr_elem(result_so_far);
				result_so_far = result;
				result = cdr(result);
				free_node(result_so_far);
				return(result);
				}
			/* add new argument off top of car of alist */
			new = new_node(N_LIST);	

			car(new) = car(car(alist));
			incr_ref(car(new));
			cdr(new) = NIL;
			cdr(args_so_far) = new;
			incr_ref(new);
			args_so_far = new;
			/* advance the list at car of alist to it's cdr */
			car(alist) = cdr(car(alist));
			}
		new = new_node(N_LIST);
		incr_ref(theArgs);
		car(new) = apply_ilisp(func,cdr(theArgs));
		incr_ref(car(new));
		decr_elem(theArgs);
		cdr(new) = NIL;
		cdr(result_so_far) = new;
		incr_ref(new);
		result_so_far = new;
		}
}


/*
    Preforms the lisp primitive 'append'.  Note that <Arglist> can be a list
    of any size, while append() only handles two elements at a time.
*/
NODE *apply_append(Arglist)
NODE *Arglist;
{
	NODE *result;

	if (null(cdr(cdr(Arglist))))
		result= append(car(Arglist), car(cdr(Arglist)));
	else
		result= append(car(Arglist), apply_append( cdr(Arglist)));
	return(result);
}
NODE *boundp(var)
NODE *var;
{
	NODE *list_check;

	for (list_check=EVAL_ENV;!null(list_check);list_check=cdr(list_check))
		{
		if (car(car(list_check)) == var)
		      if (consp(cdr(car(list_check))) && 
		          (Type_Of(car(cdr(car(list_check)))) == N_SYMBOL) &&
		          (symbol_type(car(cdr(car(list_check)))) == K_LAMBDA))
			 return(NIL);
		      else
			 return(TRUE_NODE);
		}
	return(NIL);
}

/*
	Given the primitive type <aprim>, the list of evaluated arguments to
	the primitive, <theArgs>, and an environment <env>, this function
	preforms that primitive.  Note that this is faster than apply_lambda
	because no variables need to be bound to values.
*/
NODE *apply_prim(aprim,theArgs)
KEY_TYPE aprim;
NODE *theArgs;
{
	NODE *result,*arg1,*arg2;

	arg1 = (consp(theArgs) ? car(theArgs) : NIL);
	switch(aprim)
		{
		case K_EVAL:	
				result = eval_ilisp(arg1);
				break;
		case K_BOUNDP:	
				if (Type_Of(arg1) == N_SYMBOL)
					result = boundp(arg1);
				else
					result = NIL;	
				break;
		case K_FBOUNDP:	/* is it a normal symbol? whats it bound to? */
				if (Type_Of(arg1) == N_SYMBOL)
				   if (symbol_type(arg1) == K_NORMAL)
				      {  /* lookup definition of symbol */
				      arg1 =assoc(arg1,EVAL_ENV);
				      if (null(arg1))  /* No def */
					  result = NIL;
				      else   /* strip var of (var val) pair */
					  arg1 = cdr(arg1);
				      if (consp(arg1) && 
				          (Type_Of(car(arg1)) == N_SYMBOL) &&
				          (symbol_type(car(arg1)) == K_LAMBDA))
				  	     result = TRUE_NODE;
				     else  /* bound to a non-function def */
				  	     result = NIL;
				      }
				   else   /* its a primitive or construct */
				      result = TRUE_NODE;
				else     /* its garbage */
					result = NIL;
				break;
		case K_APPLY:
				if (null(theArgs))
					result = handle_error(
					   "No arguments to APPLY \n");
				else if (not_list(cdr(theArgs)))
					result = handle_error(
					   "Need list for body of APPLY \n");
				else
					result = apply_ilisp(arg1,
				   	   car(cdr(theArgs)));
				break;
		case K_PPRINT:	
				result = arg1;
				fprintf(stderr,"\n");
				print_out(arg1);
				fprintf(stderr,"\n");
				break;
		case K_LOAD:
				result = apply_load(arg1);
				break;
		case K_ASSOC:
				if (null(theArgs))
					result = handle_error(
					   "No first argument of ASSOC \n");
				else if (not_list(cdr(theArgs)))
					result = handle_error(
					   "No second argument of ASSOC \n");
				result = assoc(arg1,car(cdr(theArgs)));
				break;
		case K_CAR:	if (null(arg1))
					result = NIL;
				else if (not_list(arg1))
					result = 
					   handle_error("car of nonlist\n");
				else
					result =  car(arg1);
				break;
		case K_CDR: 	if (null(arg1))
					result = NIL;
				else if (not_list(arg1))
					result = 
					   handle_error("car of nonlist\n");
				else 
					result =  cdr(arg1);
				break;

		case K_CAAR:	if ((null(arg1)) || (null(car(arg1))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ))
					{
					result = 
					   handle_error("caar of nonlist\n");
					}
				else
					result =  car(car(arg1)); 
				break;

		case K_CDDR:	if ((null(arg1)) || (null(cdr(arg1))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ))
					result = 
					   handle_error("cddr of nonlist\n");
				else
					result =  cdr(cdr(arg1)); 
				break;

		case K_CADR:	if ((null(arg1)) || (null(cdr(arg1))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ))
					result = 
					   handle_error("cadr of nonlist\n");
				else
					result =  car(cdr(arg1)); 
				break;

		case K_CDAR:    if ((null(arg1)) || (null(car(arg1))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ))
					result = 
					   handle_error("cdar of nonlist\n");
				else
					result =  cdr(car(arg1)); 
				break;
		case K_CAAAR:
				if ((null(arg1)) || (null(car(arg1))) ||
				   (null (car(car(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ) || 
				   (not_list(car(car(arg1))) ))
					result = 
					   handle_error("caaar of nonlist\n");
				else
					result =  car(car(car(arg1))); 
				break;	
		case K_CAADR:
				if ((null(arg1)) || (null(cdr(arg1))) ||
				   (null (car(cdr(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ) || 
				   (not_list(car(cdr(arg1))) ))
					result = 
					   handle_error("caadr of nonlist\n");
				else
					result =  car(car(cdr(arg1))); 
				break;	
		case K_CADAR:
				if ((null(arg1)) || (null(car(arg1))) ||
				   (null (cdr(car(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ) || 
				   (not_list(cdr(car(arg1))) ))
					result = 
					   handle_error("cadar of nonlist\n");
				else
					result =  car(cdr(car(arg1))); 
				break;	
		case K_CDAAR:
				if ((null(arg1)) || (null(car(arg1))) ||
				   (null (car(car(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ) || 
				   (not_list(car(car(arg1))) ))
					result = 
					   handle_error("cdaar of nonlist\n");
				else
					result =  cdr(car(car(arg1))); 
				break;	
		case K_CDADR:
				if ((null(arg1)) || (null(cdr(arg1))) ||
				   (null (car(cdr(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ) || 
				   (not_list(car(cdr(arg1))) ))
					result = 
					   handle_error("cdadr of nonlist\n");
				else
					result =  cdr(car(cdr(arg1))); 
				break;	
		case K_CADDR:
				if ((null(arg1)) || (null(cdr(arg1))) ||
				   (null (cdr(cdr(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ) || 
				   (not_list(cdr(cdr(arg1))) ))
					result = 
					   handle_error("caddr of nonlist\n");
				else
					result =  car(cdr(cdr(arg1))); 
				break;	
		case K_CDDAR:
				if ((null(arg1)) || (null(car(arg1))) ||
				   (null (cdr(car(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(car(arg1)) ) || 
				   (not_list(cdr(car(arg1))) ))
					result = 
					   handle_error("cddar of nonlist\n");
				else
					result =  cdr(cdr(car(arg1))); 
				break;	
		case K_CDDDR:
				if (null(arg1) || null(cdr(arg1)) ||
				   (null (cdr(cdr(arg1)))) )
					result = NIL;
				else if ( (not_list(arg1)) || 
				   (not_list(cdr(arg1)) ) || 
				   (not_list(cdr(cdr(arg1))) ))
					result = 
					   handle_error("cdddr of nonlist\n");
				else
					result =  cdr(cdr(cdr(arg1))); 
				break;	
		case K_CONS:
				if (null(theArgs) || not_list(cdr(theArgs)) )
					return( handle_error(
					   "Cons requires at least two arguments\n"));
				result = cons(arg1,car(cdr(theArgs)));
				break;
		case K_LIST:
				result = apply_list(theArgs);
				break;
		case K_APPEND:
				if (null(theArgs) || not_list(cdr(theArgs)))
					return( handle_error(
					   "Append requires at least two arguments \n"));
				result = apply_append(theArgs);
				break;
		case K_RPLCA:	 
				if (null(theArgs) || not_list(cdr(theArgs)))
					return( handle_error(
					   "rplaca requires at least two arguments \n"));
		                decr_elem(car(car(theArgs)));	  
				rplaca( car(theArgs), car(cdr(theArgs)) );
				/* ????? */
				incr_ref(car(car(theArgs)));
				result = car(theArgs);
				break;
		case K_RPLCD: 
				if (null(theArgs) || not_list(cdr(theArgs)))
					return( handle_error(
					   "rplacd requires at least two arguments \n"));
		                decr_elem(cdr(car(theArgs)));
				rplacd( car(theArgs), car(cdr(theArgs)) ); 
				/* ????? */
	   			incr_ref(cdr(car(theArgs)));
				result = car(theArgs);
				break;
/* HIGH USAGE  2twise*/
		case K_NULL:
				if (null(theArgs))
					return( handle_error(
					   "null requires at least one argument \n"));
				if null(arg1) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_ATOMP:
				if (null(theArgs))
					return( handle_error(
					   "atomp requires at least one argument \n"));
				if (atom(arg1)) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_LISTP:
				if (null(theArgs))
					return( handle_error(
					   "listp requires at least one argument \n"));
				if (listp(arg1)) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_CONSP:
				if (null(theArgs))
					return( handle_error(
					   "consp requires at least one argument \n"));
				if (consp(arg1)) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_FIXP:
				if (null(theArgs))
					return( handle_error(
					   "fixp requires at least one argument \n"));
				if (fixp(arg1)) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_FLOATP:
				if (null(theArgs))
					return( handle_error(
					   "floatp requires at least one argument \n"));
				if (floatp(arg1)) 
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
		case K_EQ:	
				if (null(theArgs) || not_list(cdr(theArgs)))
					return( handle_error(
					   "eq requires at least two arguments \n"));
				arg2 = car(cdr(theArgs));
				if ( Type_Of(arg1) != Type_Of(arg2) )
					result = NIL;
				else
					result = check_eq(arg1,arg2);
				break;
		case K_GREATER:	
			if (null(theArgs) || not_list(cdr(theArgs)))
				return( handle_error(
				   "eq requires at least two arguments \n"));
			arg2 = car(cdr(theArgs));
			switch(Type_Of(arg1))
			 {
			 case N_INT:
			    switch(Type_Of(arg2))
				{
			        case N_INT:
				   if (getfixnum(arg1) > getfixnum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        case N_REAL:
				   if ((float)getfixnum(arg1) > getflonum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        default:
				   result=NIL;
				}
			    break;
			 case N_REAL:
			    switch(Type_Of(arg2))
				{
			        case N_INT:
				   if (getflonum(arg1) > (float)getfixnum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        case N_REAL:
				   if (getflonum(arg1) > getflonum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        default:
				   result=NIL;
				}
			    break;
		         default:
			       result=NIL;
			  }
			break;
		case K_LESS:	
			if (null(theArgs) || not_list(cdr(theArgs)))
				return( handle_error(
				   "eq requires at least two arguments \n"));
			arg2 = car(cdr(theArgs));
			switch(Type_Of(arg1))
			 {
			 case N_INT:
			    switch(Type_Of(arg2))
				{
			        case N_INT:
				   if (getfixnum(arg1) < getfixnum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        case N_REAL:
				   if ((float)getfixnum(arg1) < getflonum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        default:
				   result=NIL;
				}
			    break;
			 case N_REAL:
			    switch(Type_Of(arg2))
				{
			        case N_INT:
				   if (getflonum(arg1) < (float)getfixnum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        case N_REAL:
				   if (getflonum(arg1) < getflonum(arg2))
					result = TRUE_NODE;
				   else
					result = NIL;
				   break;
			        default:
				   result=NIL;
				}
			    break;
		         default:
			       result=NIL;
			  }
			break;
		case K_ARRAYP:
				if (null(theArgs))
					return( handle_error(
					   "arrayp requires one argument \n"));
				if (arrayp(arg1))
					{
					result = TRUE_NODE;
					}
				else result = NIL;
				break;
				
		case K_MARRAY:
				if (!consp(theArgs))
				   return( handle_error(
				     "make-array requires at least one arguments\n"));
				if (consp(cdr(theArgs)) && 
				     symbolp(car(cdr(theArgs))))
				   {
				   if (car(cdr(theArgs)) != 
				       get_symbolrep(":initial-contents",K_KEY,
				       17))
				      return( handle_error(
				         "bad keyword for make array\n"));
				   else if(!consp(cdr(cdr(theArgs))))
				      return( handle_error(
					 "missing initial-contents values\n"));
				   else
				      result = make_array(car(theArgs),
				        car(cdr(cdr(theArgs))));
				   }
				else
				   result = make_array(car(theArgs),NIL);
				break;
		case K_AREF:
				if (null(theArgs) || not_list(cdr(theArgs)))
					return( handle_error(
					   "aref requires at least two arguments \n"));
				result = reference_array(arg1,cdr(theArgs));
				break;
		case K_ADD:
				result = apply_op(theArgs,'+');
				break;
		case K_SUB:
				result = apply_op(theArgs,'-');
				break;
		case K_MUL:
				result = apply_op(theArgs,'*');
				break;
		case K_DIV:
				result = apply_op(theArgs,'/');
				break;
		case K_MAPCAR:  result = apply_mapcar(car(theArgs),
						      cdr(theArgs));
				break;
		case K_QUIT:
				SESSION_END = 1;
				result = NIL;
				break;
		default:	result = NIL;
				break;
		}
		return(result);
}
