/****************************************************************************
 * p3d.c
 * Author Joel Welling and Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include "alisp.h"
#include "ge_error.h"
#include "p3d.h"

/* Needed external symbols */
extern char *malloc();

/*
This module contains operations for extracting elements from the lisp
structures used by P3D.  Note that it is very implementation-dependent;
any change in the structure definitions in p3d.lsp or the lisp used as 
a basis for P3D may require mods to this file.
*/

float *array2d_to_c(lisp_array)
Transformation lisp_array;
/* This routine extracts a 'C' language array from a transformation. */
{
	float *result;
	NODE *place;
	char *malloc();
	int i,j;

	ger_debug("array2d_to_c");

	if ( (!arrayp((NODE *)lisp_array)) 
		|| (!arrayp(safe_aref((NODE *)lisp_array,0))) )
		ger_error("array2d_to_c error: argument not a 2d array");

	result = (float *) malloc(16*sizeof(float));
	if (!result) ger_fatal("array2d_to_c: couldn't allocate 16 floats!");

	for (i=0;i<4;i++)
	   for (j=0;j<4;j++)	
		{
		place = safe_aref( safe_aref( (NODE *)lisp_array, i ), j);
		result[4*i+j] = (float)get_number(place);
		}

	return(result);
}

Transformation array2d_to_lisp(arg1)
float arg1[16];
/* This routine produces a transformation from a 'C' language array. */
{
	NODE *result,*dimlist;
	int i,j;

	ger_debug("array2d_to_lisp");

	dimlist = new_node(N_LIST);
	rplaca(dimlist,get_integerrep(4));
	incr_ref(car(dimlist));

	rplacd(dimlist,new_node(N_LIST));
	incr_ref(cdr(dimlist));
	rplaca(cdr(dimlist),get_integerrep(4));
	incr_ref(car(cdr(dimlist)));

	rplacd(cdr(dimlist),NIL);
	incr_ref(cdr(cdr(dimlist)));

	incr_ref(dimlist);
	result = make_array(dimlist,NIL);
	decr_elem(dimlist);

	for (i=0;i<4;i++)
		for (j=0;j<4;j++)
			{
			decr_elem(array_ref(array_ref(result,i),j));    
			array_ref(array_ref(result,i),j) = 
			   get_floatrep( arg1[4*i+j] );
			incr_ref(array_ref(array_ref(result,i),j));    
			}
	return((Transformation)result);
}

void free_transformation( trans )
Transformation trans;
/*
This routine frees the given transformation.  In the current (alisp-based)
implementation, what actually happens is that the transformation is
garbage collected (assuming that there are no other links to it).  This
operation is designed to work on transformations created from the C
side of P3D, rather than from the lisp side;  the operation will have
no effect on a transformation which has been passed in from lisp.
*/
{
	decr_elem( (NODE *)trans );
}

Attribute_list merge_attribute_lists( attrlist1, attrlist2 )
Attribute_list attrlist1, attrlist2;
/* 
This routine merges the given attribute lists, returning a new list.
Attributes in attrlist1 supercede those in attrlist2.  In this implementation
(based on alisp), both lists are a-lists of attribute-value pairs.  We
want to cons an (inverted) copy of the first list onto the second, and
return a pointer to the first element of the new construct.
*/
{
	NODE *list1, *list2;

	list1= (NODE *)attrlist1;
	list2= (NODE *)attrlist2;

	while (!null(list1)) {
		list2= cons( car(list1), list2 );
		list1= cdr(list1);
		}

	incr_ref( list2 ); /* to allow the list to be GC'd later */
	return( (Attribute_list)list2 );
}

void free_attribute_list( attrlist )
Attribute_list attrlist;
/*
This function frees the given attribute list.  In the current (alisp-based
implementation, what actually happens is that those parts of the list
not referenced by other lisp structures are garbage collected.
*/
{
	decr_elem( (NODE *)attrlist );
}

Symbol create_symbol( string )
char *string;
/* Function to create a Symbol from a string */
{
	NODE *symbol;

/*  Must be careful that <string> isn't the name of a primitive like list */
/*  This needs a check							  */

	symbol = get_symbolrep( string,K_NORMAL,strlen(string) );
	incr_ref(symbol);
	return( (Symbol)symbol );
	
}

Vertex_list create_vertex_list(length)
/*
This routine creates a vertex list of the given length.  All the
vertices are initially null.
*/
{
	/* 
	This implementation simply conses together the required
	list of cons cells, leaving the car of each cons cell nil.
	*/
	NODE *list;
	int i;

	list= NIL;
	for (i=0; i<length; i++) list= cons( NIL, list );
	incr_ref( list );

	return( (Vertex_list)list );
}

void free_vertex_list( thisvlist )
Vertex_list thisvlist;
/*
This routine frees a vertex list.  If this was the last reference to this
vertex list, it will be deleted, and if there are no other references
to the vertices it contains they will be deleted as well.
*/
{
	decr_elem( (NODE *)thisvlist );
}

/*	The following routines provide Facet_list service functions */

Vertex_list set_first_vertex( thislist, thisvertex )
Vertex_list thislist;
Vertex thisvertex;
/* 
This routine sets the first vertex of the given list to the given Vertex,
doing appropriate memory management.
*/
{
	decr_elem( (NODE *)first_vertex( thislist ) );
	car( (NODE *)(thislist) )= (NODE *)thisvertex; /* set first vertex */
	incr_ref( (NODE *)thisvertex );
	return( thislist );
}

Facet_list create_facet_list(length)
/*
This routine creates a facet list of the given length.  All the
vertex lists in the facet list are initially null.
*/
{
	/* 
	This implementation simply conses together the required
	list of cons cells, leaving the car of each cons cell nil.
	*/
	NODE *list;
	int i;

	list= NIL;
	for (i=0; i<length; i++) list= cons( NIL, list );
	incr_ref( list );

	return( (Facet_list)list );
}

void free_facet_list( thisflist )
Facet_list thisflist;
/*
This routine frees a facet list.  If this was the last reference to this
facet list, it will be deleted, and if there are no other references
to the vertex lists it contains they will be deleted as well.
*/
{
	decr_elem( (NODE *)thisflist );
}

Facet_list set_first_facet( thislist, thisfacet )
Facet_list thislist;
Vertex_list thisfacet;
/* 
This routine sets the first facet of the given list to the given Vertex_list,
doing appropriate memory management.
*/
{
	decr_elem( (NODE *)first_facet( thislist ) );
	car( (NODE *)(thislist) )= (NODE *)thisfacet;  /* set first facet */
	incr_ref( (NODE *)thisfacet );
	return( thislist );
}

Camera create_camera()
/* This routine creates a camera.  Default values are as given below. */
{
	Camera thiscamera;
	Point lookat, lookfrom;
	Vector up;
	float fovea= 30.0, hither= -0.01, yon= -100.0;
	static float lookat_coords[3]= { 0.0, 0.0, 0.0 },
		lookfrom_coords[3]= { 0.0, 0.0, 0.0 },
		up_components[3]= { 0.0, 1.0, 0.0 };
	NODE *dimlist,*tempint;

        fprintf(stderr,"About to make point NNN\n");
	lookat= create_point();
	(void)set_point_x( lookat, lookat_coords[0] );
	(void)set_point_y( lookat, lookat_coords[1] );
	(void)set_point_z( lookat, lookat_coords[2] );

        fprintf(stderr,"About to make point OOO\n");
	lookfrom= create_point();
	(void)set_point_x( lookfrom, lookfrom_coords[0] );
	(void)set_point_y( lookfrom, lookfrom_coords[1] );
	(void)set_point_z( lookfrom, lookfrom_coords[2] );

	up= create_vector();
	(void)set_vector_x( lookfrom, up_components[0] );
	(void)set_vector_y( lookfrom, up_components[1] );
	(void)set_vector_z( lookfrom, up_components[2] );

	tempint = get_integerrep(7);
	dimlist= cons(tempint,NIL);
	incr_ref( dimlist );

	/*
	As the camera gets assembled, stray links to the parts get
	released.  The new references will keep the parts from being
	garbage collected.
	*/
	thiscamera= (Camera)make_array(dimlist,NIL);
	decr_elem( dimlist );
	array_ref( (NODE *)thiscamera, 0 )= create_symbol("camera");
	(void)set_camera_lookfrom( thiscamera, lookfrom );
	free_point( lookfrom ); 
	(void)set_camera_lookat( thiscamera, lookat );
	free_point( lookat ); 
	(void)set_camera_up( thiscamera, up );
	free_vector( up );
	(void)set_camera_fovea( thiscamera, fovea );
	(void)set_camera_hither( thiscamera, hither );
	(void)set_camera_yon( thiscamera, yon );

	incr_ref( (NODE *)thiscamera );

	return( thiscamera );
}

void free_camera(thiscamera)
Camera thiscamera;
/*
This routine frees a camera.  If this was the last reference to this
camera, it will be deleted, and if there are no other references
to the lookat, lookfrom, and up components it contains they will be 
deleted as well.
*/
{
	decr_elem( (NODE *)thiscamera );
}

Camera set_camera_lookfrom(thiscamera, thispoint)
Camera thiscamera;
Point thispoint;
/* 
This routine sets the lookfrom component of the given camera to the given
point, doing appropriate memory management.
*/
{
	decr_elem( (NODE *)camera_lookfrom( thiscamera ) );
	array_ref((NODE *)thiscamera,1)= (NODE *)thispoint;
	incr_ref( (NODE *)thispoint );
	return( thiscamera );
}

Camera set_camera_lookat(thiscamera, thispoint)
Camera thiscamera;
Point thispoint;
/* 
This routine sets the lookatcomponent of the given camera to the given
point, doing appropriate memory management.
*/
{
	decr_elem( (NODE *)camera_lookat( thiscamera ) );
	array_ref((NODE *)thiscamera,2)= (NODE *)thispoint;
	incr_ref( (NODE *)thispoint );
	return( thiscamera );
}

Camera set_camera_up(thiscamera, thisvector)
Camera thiscamera;
Vector thisvector;
/* 
This routine sets the up component of the given camera to the given
vector, doing appropriate memory management.
*/
{
	decr_elem( (NODE *)camera_up( thiscamera ) );
	array_ref((NODE *)thiscamera,3)= (NODE *)thisvector;
	incr_ref( (NODE *)thisvector );
	return( thiscamera );
}

Camera set_camera_fovea(thiscamera, value)
Camera thiscamera;
float value;
/* This routine sets the fovea of a camera, creating a node as needed. */
{
	decr_elem( array_ref( (NODE *)thiscamera, 4 ) );
	array_ref( (NODE *)thiscamera, 4 )= get_floatrep(value);
	incr_ref( array_ref( (NODE *)thiscamera, 4 ) );
	return( thiscamera );
}

Camera set_camera_hither(thiscamera, value)
Camera thiscamera;
float value;
/* 
This routine sets the hither distance of a camera, creating a node as 
needed. 
*/
{
	decr_elem( array_ref( (NODE *)thiscamera, 5 ) );
	array_ref( (NODE *)thiscamera, 5 )= get_floatrep(value);
	incr_ref( array_ref( (NODE *)thiscamera, 5 ) );
	return( thiscamera );
}

Camera set_camera_yon(thiscamera, value)
Camera thiscamera;
float value;
/* This routine sets the yon distance of a camera, creating a node as needed. */
{
	decr_elem( array_ref( (NODE *)thiscamera, 6 ) );
	array_ref( (NODE *)thiscamera, 6 )= get_floatrep(value);
	incr_ref( array_ref( (NODE *)thiscamera, 6 ) );
	return( thiscamera );
}

Camera set_camera_background(thiscamera, thiscolor)
Camera thiscamera;
Color thiscolor;
/* 
This routine sets the background color component of the given camera 
to the given color, doing appropriate memory management.
*/
{
	decr_elem( (NODE *)camera_background( thiscamera ) );
	array_ref((NODE *)thiscamera, 7)= (NODE *)thiscolor;
	incr_ref( (NODE *)thiscolor );
	return( thiscamera );
}

/*
   This routine returns the number of elements in a list.
   Again, this is assuming that alist is null-terminating.  Here, however, it
   would be easy to check for a non-null ending before you got to it, though it
   would require more code.  
*/
int length_list(alist)
NODE *alist;
{
	NODE *walk_list;
	int result=0;

	for(walk_list=alist;consp(walk_list);walk_list=cdr(walk_list))
		result++;
	return(result);

}

void grab( thisobj )
NODE *thisobj;
/*
This routine prevents the object passed it from being garbage collected
until a call to 'ungrab' has been made for the object.
*/
{
	incr_ref( thisobj );
}

void ungrab( thisobj )
NODE *thisobj;
/*
This routine permits but does not insure garbage collection of the object
passed it.
*/
{
	decr_elem( thisobj );
}
