/****************************************************************************
 * flyby.c
 * Author Joel Welling
 * Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
  This module provides a facility for grabbing a series of viewpoints and
  producing spline-interpolated flybys from mouse-driven user interfaces.
*/

#include <stdio.h>
#include <math.h>
#include "ge_error.h"
#include "mi.h"
#include "flyby.h"
#include "flyby_preamble.h"

/* This controls the integration of total flyby length */
#define INTEGR_SUBDIVISIONS 30

/* This sets the maximum number of frames in the acceleration and
 * deceleration phases of a flyby.
 */
#define MAX_ACCEL_LENGTH 30

/* The list of valid keyframes, and pointer to end of list */
static Viewpoint *keyframes= (Viewpoint *)0;
static Viewpoint *last_keyframe= (Viewpoint *)0;

/* 
Total number of control points/keyframes in use.
*/
static int npoints= 0;

/* Actual frames in the flyby */
static Viewpoint *flyby_frames= (Viewpoint *)0;

/* Struct to hold spline control point table data */
typedef struct spline_table_struct {
  int npts;
  int ndists;
  float *frmx, *frmx2;
  float *frmy, *frmy2;
  float *frmz, *frmz2;
  float *atx, *atx2;
  float *aty, *aty2;
  float *atz, *atz2;
  float *upx, *upx2;
  float *upy, *upy2;
  float *upz, *upz2;
  float *fov, *fov2;
  float *hither, *hither2;
  float *yon, *yon2;
  float *clocktime;
  float *distance;
} Spline_Table;

static float *float_alloc( size )
int size;
/* This is a convencience routine for space allocation */
{
  float *result;

  ger_debug("float_alloc: size= %d",size);

  if ( !(result= (float *)malloc( size*sizeof(float) )) )
    ger_fatal("float_alloc: unable to allocate %d floats!",size);
  return(result);
}

/* 
The following two routines are based on 'spline' and 'splint' from 
Numerical Recipes in C, The Art of Scientific Computing, by Press, 
Flannery, Teukolsky, and Vetterling, Cambridge University Press 1988.
*/
static float *make_spline_table(x, y, n)
float *x, *y;
int n;
/* 
This routine generates and returns a natural bicubic spline second derivative
table for the given data.  It is the equivalent of the Numerical Recipes
routine 'spline'.
*/
{
  float *y2, *u;
  register int i;
  register float sig,p;

  y2= float_alloc(n);
  u= float_alloc(n);

  y2[0]= u[0]= 0.0;

  for (i=1; i<n-1; i++) {
    sig= (x[i]-x[i-1])/(x[i+1]-x[i]);
    p= sig*y2[i-1] + 2.0;
    y2[i]= (sig-1.0)/p;
    u[i]= (y[i+1]-y[i])/(x[i+1]-x[i]) - (y[i]-y[i-1])/(x[i]-x[i-1]);
    u[i]= (6.0*u[i]/(x[i+1]-x[i-1])-sig*u[i-1])/p;
  }

  y2[n-1]= 0.0;

  for (i=n-2; i>=0; i--) y2[i]= y2[i]*y2[i+1] + u[i];

  free((char *)u);

  return(y2);
}

static float splint( xa, ya, y2a, n, x)
float *xa, *ya, *y2a, x;
int n;
/*
This routine produces a natural spline interpolation at the point x, given
that the values of the functions at the xa points are given in ya, and
that y2a is a valid result from make_spline_table.  This is equivalent to
the Numerical Recipes function 'splint'.  We assume spatially correlated
calls to the routine, rather than random calls, and thus do not use
bisection to pick the appropriate spline segment.
*/
{
  static int low= 0, high= 0;
  float delta, a, b, result;

  /* Make sure the range of xa's includes the value of interest. */
  if (xa[0] > x) ger_fatal("splint: x value %f less than table range.\n",x);
  if (xa[n-1] < x) 
    ger_fatal("splint: x value %f greater than table range.\n",x);

  /* Find appropriate low and high values */
  while ( xa[low]>x ) low--;
  while ( (xa[low+1]<x) && (low<n-2) ) low++;
  high= low+1;

  delta= xa[high]-xa[low];
  if (delta==0.0) ger_fatal("splint: bad xa input\n");
  a= (xa[high]-x)/delta;
  b= (x-xa[low])/delta;
  result= a*ya[low] + b*ya[high] 
    + ((a*a*a - a)*y2a[low] + (b*b*b - b)*y2a[high])*(delta*delta)/6.0;

  return(result);
}

static void free_view_list( runner )
Viewpoint *runner;
/* This is a utility routine to free a list of viewpoints */
{
  Viewpoint *next;

  ger_debug("free_view_list:");

  while (runner) {
    next= runner->next;
    free( runner );
    runner= next;
  }
}

void fb_new_path()
/* This routine initializes the flyby module if necessary, and discards
 * any existing path information.
 */
{
  ger_debug("fb_new_path:");

  npoints= 0;
  free_view_list(keyframes);
  keyframes= (Viewpoint *)0;
  last_keyframe= (Viewpoint *)0;
  flyby_frames= (Viewpoint *)0;
}

void fb_add_keyframe()
/* This routine gets a viewpoint from the mouse user interface, and
 * adds it to the list of keyframes.
 */
{
  Viewpoint *newview;

  ger_debug("fb_add_keyframe:");

  newview= miu_viewpoint();
  if (last_keyframe) last_keyframe->next= newview;
  else keyframes= newview;
  newview->next= (Viewpoint *)0;
  last_keyframe= newview;
  npoints++;
}

static int duplicate_views( view1, view2 )
Viewpoint *view1, *view2;
/* This routine returns true if the views differ for splining purposes.
 * Note that some characteristics, like fovea or up vector, can differ
 * and still have the viewpoints be the same as far as this routine is
 * concerned.
 */
{
  ger_debug("duplicate_views:");

  return( (view1->from.x == view2->from.x) 
	 && (view1->from.y == view2->from.y) 
	 && (view1->from.z == view2->from.z) 
	 && (view1->at.y == view2->at.y) 
	 && (view1->at.z == view2->at.z) 
	 && (view1->at.y == view2->at.y) );
}

static void transcribe_list_to_table( keyframes, tbl, remove_dups )
Viewpoint *keyframes;
Spline_Table *tbl;
int remove_dups;
/* This routine transcribes the given viewpoint list into the tables,
 * checking the size against npts.
 */
{
  int i;
  int npts;
  Viewpoint *last= (Viewpoint *)0;

  ger_debug("transcribe_list_to_table:");

  npts= 0;
  for (i=0; i<npoints; i++) {
    if (!keyframes) ger_fatal(
	"transcribe_list_to_table: algorithm error, ran out of points!");
    if (!remove_dups || i==0 || !duplicate_views(last,keyframes)) {
      tbl->frmx[npts]= keyframes->from.x;
      tbl->frmy[npts]= keyframes->from.y;
      tbl->frmz[npts]= keyframes->from.z;
      tbl->atx[npts]= keyframes->at.x;
      tbl->aty[npts]= keyframes->at.y;
      tbl->atz[npts]= keyframes->at.z;
      tbl->upx[npts]= keyframes->up.x;
      tbl->upy[npts]= keyframes->up.y;
      tbl->upz[npts]= keyframes->up.z;
      tbl->fov[npts]= keyframes->fovea;
      tbl->hither[npts]= keyframes->hither;
      tbl->yon[npts]= keyframes->yon;
      tbl->clocktime[npts]= (float)npts;
      npts++;
    }
    last= keyframes;
    keyframes= keyframes->next;
  }
  tbl->npts= npts;
}

static Viewpoint *new_viewpoint( proto )
Viewpoint *proto;
/* This generates uninitialized viewpoints.  If proto is non-null,
 * the new viewpoint is initialized with the values of the old.
 */
{
  Viewpoint *result;

  ger_debug("new_viewpoint:");

  if ( !(result= (Viewpoint *)malloc(sizeof(Viewpoint))) )
    ger_fatal("new_viewpoint: unable to allocate %d bytes!",
	      sizeof(Viewpoint));

  if (proto) {
    result->from.x= proto->from.x;
    result->from.y= proto->from.y;
    result->from.z= proto->from.z;
    result->at.x= proto->at.x;
    result->at.y= proto->at.y;
    result->at.z= proto->at.z;
    result->up.x= proto->up.x;
    result->up.y= proto->up.y;
    result->up.z= proto->up.z;
    result->fovea= proto->fovea;
    result->hither= proto->hither;
    result->yon= proto->yon;
    result->next= proto->next;
  }

  return(result);
}

static float dist_to_time( tbl, dist )
Spline_Table *tbl;
float dist;
/* This routine finds the appropriate time for a frame by looking at
 * the right place in the distance table.
 */
{
  int imin, imax, i;
  float time, dist_fraction;

  ger_debug("dist_to_time:");

  /* bounds check */
  if (dist < tbl->distance[0] || dist > tbl->distance[tbl->ndists-1])
    ger_fatal("dist_to_time: distance %f out of range!",dist);

  /* Do a binary search for the right distance table entry */
  imin= 0;
  imax= tbl->ndists - 2;
  i= imin;
  while (1) {
    if (tbl->distance[i] > dist) { /* gone too far */
      imax= i;
      i= (i + imin)/2;
    }
    else if (tbl->distance[i+1] < dist) { /* not far enough */
      imin= i;
      i= (i + imax + 1)/2;
    }
    else break;
  }

  dist_fraction= (dist - tbl->distance[i])
    / (tbl->distance[i+1]-tbl->distance[i]);

  time= ((float)i + dist_fraction) *
    ((tbl->clocktime[tbl->npts-1]-tbl->clocktime[0])/(float)(tbl->ndists-1));

  return(time);
}

static float accel_function( val )
float val;
/* This function takes a value between 0.0 and 1.0, which represents the
 * part of the acceleration interval which has passed, and returns a fraction
 * between 0.0 and 1.0 representing what fraction of the total distance
 * to be travelled during the acceleration interval has been travelled
 * so far.
 */
{
  float result;

  ger_debug("accel_function: val= %f",val);

  /* This function is the integral of the function 
   * v(t)= (sin(pi*t/2))**2, which has the nice properties that
   * the initial velocity and the initial and final acceleration
   * are zero, and that the final velocity is 1.0.
   */

#define pi 3.14159265
  result= 0.5*( val - sin( pi*val )/pi );
#undef pi

  return(result);
}

static float calc_distance( tbl, flags, nframes, frame )
Spline_Table *tbl;
int flags;
int nframes;
int frame;
/* This routine determines an appropriate distance along the flight
 * path for a given frame, given the option setting flags.
 */
{
  float dist;
  float start_pos, end_pos;
  int accel_steps, decel_steps, coast_steps, accel_length, total_steps;
  int accel_start, decel_start, coast_start;
  float accel_pos, decel_pos, coast_pos;
  float accel_norm;

  ger_debug("calc_distance: flags= %d, frame %d of %d",flags,frame,nframes);

  start_pos= tbl->distance[0];
  end_pos= tbl->distance[tbl->ndists-1];
  total_steps= nframes - 1;
  accel_norm= accel_function(1.0);

  /* This frame falls into either the acceleration phase, the deceleration
   * phase, or the constant velocity phase.  It will be constant velocity
   * unless we are starting and/or ending at rest.  In the latter case,
   * the acceleration and deceleration phases last for MAX_ACCEL_LENGTH 
   * frames or 1/4 the total number of frames, whichever is less.
   */
  accel_length= nframes/4;
  if (accel_length>MAX_ACCEL_LENGTH) accel_length= MAX_ACCEL_LENGTH;
  accel_start= 0;
  accel_pos= start_pos;
  if ( flags & START_FROM_REST_FLAG ) { /* acceleration phase exists */
    coast_start= accel_length;
    coast_pos= accel_norm * (end_pos - start_pos) 
      * ((float)accel_length/(float)total_steps);
    accel_steps= accel_length;
  }
  else { /* don't accelerate */
    coast_start= 0;
    coast_pos= start_pos;
    accel_steps= 0;
  }
  if ( flags & END_AT_REST_FLAG ) { /* deceleration phase exists */
    decel_start= nframes - accel_length;
    decel_pos= end_pos - accel_norm * (end_pos - start_pos)
      * ((float)accel_length/(float)total_steps);
    decel_steps= accel_length;
  }
  else { /* don't decelerate */
    decel_start= nframes;
    decel_pos= end_pos;
    decel_steps= 0;
  }
  coast_steps= total_steps - accel_steps - decel_steps;
  if (frame >= decel_start) { /* deceleration phase */
    dist= decel_pos 
      + (accel_norm - accel_function((float)(total_steps - frame)
				     /((float)decel_steps)))
	* (end_pos - decel_pos) / accel_norm;
  }
  else if (frame < coast_start) { /* acceleration phase */
    dist= accel_pos 
      + accel_function((float)(frame-accel_start)/((float)accel_steps))
	* (coast_pos - accel_pos) / accel_norm;
  }
  else { /* coast phase */
    dist= coast_pos + ((float)(frame-coast_start)/((float)coast_steps))
      * (decel_pos - coast_pos);
  }

  return(dist);
}

static Viewpoint *spline_path( flags, nframes, tbl )
int flags;
int nframes;
Spline_Table *tbl;
/* This routine actually generates the new path */
{
  Viewpoint *result, *runner;
  int i;
  float dist, t;

  ger_debug("spline_path:");

  if (nframes<2) {
    ger_error("spline_path: %d is too few frames to interpolate!",nframes);
    return(keyframes);
  }

  /* first view is copy of first keyframe */
  runner= result= new_viewpoint(keyframes);

  /* Interpolate intermediate frames */
  for (i=1; i<nframes-1; i++) {
    dist= calc_distance( tbl, flags, nframes, i );
    t= dist_to_time(tbl,dist);
    runner->next= new_viewpoint((Viewpoint *)0);
    runner= runner->next;
    runner->from.x= splint(tbl->clocktime, tbl->frmx, tbl->frmx2, 
			   tbl->npts, t);
    runner->from.y= splint(tbl->clocktime, tbl->frmy, tbl->frmy2, 
			   tbl->npts, t);
    runner->from.z= splint(tbl->clocktime, tbl->frmz, tbl->frmz2, 
			   tbl->npts, t);
    runner->at.x= splint(tbl->clocktime, tbl->atx, tbl->atx2, tbl->npts, t);
    runner->at.y= splint(tbl->clocktime, tbl->aty, tbl->aty2, tbl->npts, t);
    runner->at.z= splint(tbl->clocktime, tbl->atz, tbl->atz2, tbl->npts, t);
    runner->up.x= splint(tbl->clocktime, tbl->upx, tbl->upx2, tbl->npts, t);
    runner->up.y= splint(tbl->clocktime, tbl->upy, tbl->upy2, tbl->npts, t);
    runner->up.z= splint(tbl->clocktime, tbl->upz, tbl->upz2, tbl->npts, t);
    runner->fovea= splint(tbl->clocktime, tbl->fov, tbl->fov2, tbl->npts, t);
    runner->hither= 
      splint(tbl->clocktime, tbl->hither, tbl->hither2, tbl->npts, t);
    runner->yon= splint(tbl->clocktime, tbl->yon, tbl->yon2, tbl->npts, t);
  }

  /* Last frame is a copy of the last keyframe */
  runner->next= new_viewpoint(last_keyframe);

  return(result);
}

static void integrate_distances( tbl )
Spline_Table *tbl;
/* This routine does one integration along the flyby path, generating
 * distances.
 */
{
  float x, y, z, dx, dy, dz, t, dt, s, s_shift, total_s_shift, last_s;
  int i,j;

  ger_debug("integrate_distances:");

  s= 0.0;
  dt= (tbl->clocktime[tbl->npts-1] - tbl->clocktime[0])
      / (float)(tbl->ndists - 1);
  t= tbl->clocktime[0];
  x= tbl->frmx[0] - tbl->atx[0];
  y= tbl->frmy[0] - tbl->aty[0];
  z= tbl->frmz[0] - tbl->atz[0];
  for (i=0; i<tbl->ndists-1; i++) {
    tbl->distance[i]= s;

    t += dt;
    if (i < tbl->ndists-2) {
      dx= splint(tbl->clocktime, tbl->frmx, tbl->frmx2, tbl->npts, t)
	- splint(tbl->clocktime, tbl->atx, tbl->atx2, tbl->npts, t)
	- x;
      dy= splint(tbl->clocktime, tbl->frmy, tbl->frmy2, tbl->npts, t)
	- splint(tbl->clocktime, tbl->aty, tbl->aty2, tbl->npts, t)
	- y;
      dz= splint(tbl->clocktime, tbl->frmz, tbl->frmz2, tbl->npts, t)
	- splint(tbl->clocktime, tbl->atz, tbl->atz2, tbl->npts, t)
	- z;
    }
    else {
      dx= tbl->frmx[ tbl->npts - 1 ] - tbl->atx[ tbl->npts - 1 ] - x;
      dy= tbl->frmy[ tbl->npts - 1 ] - tbl->aty[ tbl->npts - 1 ] - y;
      dz= tbl->frmz[ tbl->npts - 1 ] - tbl->atz[ tbl->npts - 1 ] - z;
    }
    s += sqrt( dx*dx + dy*dy + dz*dz );
    x += dx;
    y += dy;
    z += dz;
  }
  tbl->distance[tbl->ndists-1]= s;
}

static void uniform_distances( tbl )
Spline_Table *tbl;
/* This routine produces a table of distances in which the keyframes
 * are separated by equal intervals, whatever their real separation 
 * might be.
 */
{
  float x, y, z, dx, dy, dz, t, dt, s, s_shift, total_s_shift, last_s;
  int i,j;

  ger_debug("uniform_distances:");

  dt= (tbl->clocktime[tbl->npts-1] - tbl->clocktime[0])
      / (float)(tbl->ndists - 1);
  t= tbl->clocktime[0];
  for (i=0; i<tbl->ndists-1; i++) {
    tbl->distance[i]= t;
    t += dt;
  }
  tbl->distance[tbl->ndists-1]= tbl->clocktime[tbl->npts-1];
}

void fb_generate_path(flags,nframes)
int flags;
int nframes;
/* This routine generates a path from the current list of keyframes */
{
  Spline_Table tbl;
  int remove_dups= 0;

  ger_debug("fb_generate_path: nframes= %d",nframes);

  /* Set up tables */
  tbl.npts= npoints;
  tbl.ndists= npoints*INTEGR_SUBDIVISIONS;
  tbl.frmx= float_alloc( npoints );
  tbl.frmy= float_alloc( npoints );
  tbl.frmz= float_alloc( npoints );
  tbl.atx= float_alloc( npoints );
  tbl.aty= float_alloc( npoints );
  tbl.atz= float_alloc( npoints );
  tbl.upx= float_alloc( npoints );
  tbl.upy= float_alloc( npoints );
  tbl.upz= float_alloc( npoints );
  tbl.fov= float_alloc( npoints );
  tbl.hither= float_alloc( npoints );
  tbl.yon= float_alloc( npoints );
  tbl.clocktime= float_alloc( npoints );
  tbl.distance= float_alloc( tbl.ndists );

  /* Transcribe the keyframe list to the table, removing duplicate
   * frames if necessary.
   */
  remove_dups= (flags & STEADY_MOTION_FLAG);
  transcribe_list_to_table( keyframes, &tbl, remove_dups );

  /* Create second derivative table */
  tbl.frmx2= make_spline_table( tbl.clocktime, tbl.frmx, tbl.npts );
  tbl.frmy2= make_spline_table( tbl.clocktime, tbl.frmy, tbl.npts );
  tbl.frmz2= make_spline_table( tbl.clocktime, tbl.frmz, tbl.npts );
  tbl.atx2= make_spline_table( tbl.clocktime, tbl.atx, tbl.npts );
  tbl.aty2= make_spline_table( tbl.clocktime, tbl.aty, tbl.npts );
  tbl.atz2= make_spline_table( tbl.clocktime, tbl.atz, tbl.npts );
  tbl.upx2= make_spline_table( tbl.clocktime, tbl.upx, tbl.npts );
  tbl.upy2= make_spline_table( tbl.clocktime, tbl.upy, tbl.npts );
  tbl.upz2= make_spline_table( tbl.clocktime, tbl.upz, tbl.npts );
  tbl.fov2= make_spline_table( tbl.clocktime, tbl.fov, tbl.npts );
  tbl.hither2= make_spline_table( tbl.clocktime, tbl.hither, tbl.npts );
  tbl.yon2= make_spline_table( tbl.clocktime, tbl.yon, tbl.npts );

  /* Generate the table of integrated distances along the path */
  if (flags & STEADY_MOTION_FLAG) integrate_distances( &tbl );
  else uniform_distances( &tbl );

  /* Generate the new viewpoint list */
  if (flyby_frames) free_view_list(flyby_frames);
  flyby_frames= spline_path( flags, nframes, &tbl );

  /* Free everything */
  free( (char *)tbl.frmx ); free( (char *)tbl.frmx2 );
  free( (char *)tbl.frmy ); free( (char *)tbl.frmy2 );
  free( (char *)tbl.frmz ); free( (char *)tbl.frmz2 );
  free( (char *)tbl.atx ); free( (char *)tbl.atx2 );
  free( (char *)tbl.aty ); free( (char *)tbl.aty2 );
  free( (char *)tbl.atz ); free( (char *)tbl.atz2 );
  free( (char *)tbl.upx ); free( (char *)tbl.upx2 );
  free( (char *)tbl.upy ); free( (char *)tbl.upy2 );
  free( (char *)tbl.upz ); free( (char *)tbl.upz2 );
  free( (char *)tbl.fov ); free( (char *)tbl.fov2 );
  free( (char *)tbl.hither ); free( (char *)tbl.hither2 );
  free( (char *)tbl.yon ); free( (char *)tbl.yon2 );
  free( (char *)tbl.clocktime );
  free( (char *)tbl.distance );
}

void fb_preview( keyframes_only )
int keyframes_only;
/* This routine previews the current path.  fb_generate_path must be
 * called first.  Previewing is done by having the mouse user interface
 * render from the appropriate viewpoints.
 */
{
  Viewpoint *runner;
  Viewpoint *save;

  ger_debug("fb_preview: keyframes_only= %d",keyframes_only);

  /* Save current view */
  save= miu_viewpoint();

  /* Run through keyframes */
  if (keyframes_only) runner= keyframes;
  else runner= flyby_frames;
  while (runner) {
    miu_set_viewpoint(runner);
    miu_prompt_redraw();
    runner= runner->next;
  }

  /* Go back to the view we started with */
  miu_set_viewpoint(save);
  free(save);
}

void fb_save_flyby(pathfile)
char *pathfile;
{
  FILE *ofile;
  Viewpoint *runner;

  ger_debug("fb_save_flyby: pathfile= <%s>", pathfile);

  /* open the file */
  if ( !(ofile= fopen(pathfile,"w")) )
    ger_fatal("fb_save_flyby: cannot open the file <%s> for writing!",
	      pathfile);

  /* emit preamble */
  fprintf( ofile, preamble, miu_filename() );

  /* emit snaps */
  runner= flyby_frames;
  while (runner) {
    fprintf(ofile,
      "(flysnap object lights %f %f %f\n   %f %f %f %f %f %f\n  %f %f %f)\n",
	    runner->from.x, runner->from.y, runner->from.z,
	    runner->at.x, runner->at.y, runner->at.z,
	    runner->up.x, runner->up.y, runner->up.z,
	    runner->fovea, runner->hither, runner->yon);
    runner= runner->next;
  }

  /* emit close and read of model */
  fprintf(ofile,"))) ;close defun of snap and if\n\n");
  fprintf(ofile,"(load \"%s\")\n", miu_filename() );

  /* close the file */
  if ( fclose(ofile) == EOF )
    ger_error("fb_save_flyby: cannot close the file <%s>!",
	      pathfile);
}

void fb_save_view(pathfile)
char *pathfile;
{
  FILE *ofile;
  Viewpoint *view;

  ger_debug("fb_save_view: pathfile= <%s>", pathfile);

  /* open the file */
  if ( !(ofile= fopen(pathfile,"w")) )
    ger_fatal("fb_save_view: cannot open the file <%s> for writing!",
	      pathfile);

  /* emit preamble */
  fprintf( ofile, preamble, miu_filename() );

  /* emit snap */
  view= miu_viewpoint();
  fprintf(ofile,
      "(flysnap object lights %f %f %f\n   %f %f %f %f %f %f\n  %f %f %f)\n",
	  view->from.x, view->from.y, view->from.z,
	  view->at.x, view->at.y, view->at.z,
	  view->up.x, view->up.y, view->up.z,
	  view->fovea, view->hither, view->yon);
  free( view );

  /* emit close and read of model */
  fprintf(ofile,"))) ;close defun of snap and if\n\n");
  fprintf(ofile,"(load \"%s\")\n", miu_filename() );

  /* close the file */
  if ( fclose(ofile) == EOF )
    ger_error("fb_save_view: cannot close the file <%s>!",
	      pathfile);
}

