/****************************************************************************
 * structures.c
 * Author Julie Roskies and Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>
#include "struct_defs.h"
#include "err_defs.h"

Color *create_color()
/* This routine creates a color cell */
{
	Color *thiscolor;

	/* No debugging since this will happen too often */

	/* Create the cell */

	/* Memory allocation for color cell */
	if ( !(thiscolor= (Color *)malloc(sizeof(Color)) ) )
		ger_fatal(
		  "create_color: unable to allocate %d bytes",
		  sizeof(Color) );

	/* Set the color elements to appropriate values */
	thiscolor->r= 0.0;
	thiscolor->g= 0.0;
	thiscolor->b= 0.0;
	thiscolor->a= 1.0;

	/* Return the new color cell */
	return( thiscolor );
}

void destroy_color( thiscolor )
Color *thiscolor;
/* This routine deallocates a color cell */
{
	/* No debugging since this will happen too often */

	/* Destroy the cell */
	free( (char *)thiscolor );
}

Vector *create_vector()
/* This routine creates a vector cell */
{
	Vector *thisvector;

	/* No debugging since this will happen too often */

	/* Create the cell */

	/* Memory allocation for vector cell */
	if ( !(thisvector= (Vector *)malloc(sizeof(Vector)) ) )
		ger_fatal(
		  "create_vector: unable to allocate %d bytes",
		  sizeof(Vector) );

	/* Set the vector elements to appropriate values */
	thisvector->x= 0.0;
	thisvector->y= 0.0;
	thisvector->z= 0.0;

	/* Return the new vector cell */
	return( thisvector );
}

void destroy_vector( thisvector )
Vector *thisvector;
/* This routine deallocates a vector cell */
{
	/* No debugging since this will happen too often */

	/* Destroy the cell */
	free( (char *)thisvector );
}

Vnode *create_vnode()
/* This routine creates a vnode cell */
{
	Vnode *thisvnode;

	/* No debugging since this will happen too often */

	/* Create the cell */

	/* Memory allocation for the vnode cell */
	if ( !(thisvnode= (Vnode *)malloc(sizeof(Vnode)) ) )
		ger_fatal(
		    "create_vnode: unable to allocate %d bytes",
		    sizeof(Vnode) );

	/* Set the Vnode elements to appropriate values */
	thisvnode->next= (Vnode *)NULL;
	thisvnode->myvertex= -1;

	/* Return the new vnode cell */
	return (thisvnode);
}

void destroy_vnode ( thisvnode )
Vnode *thisvnode;
/* 
This routine deallocates an entire Facet, i.e. frees the pointers to Vnodes
in a linked list.
*/
{
	ger_debug("destroy_vnode:");
        
	/* Recursively marches down a list of Vnodes, destroying pointers */
	if ( thisvnode->next ) 
	    destroy_vnode (thisvnode->next);
 	free ( (void *)thisvnode);
}

Model *create_model()
/* This routine creates an 'empty' model */
{
	Model *thismodel;

	ger_debug("create_model:");

	/* memory allocation for model cell */
	if ( ! (thismodel= (Model *)malloc(sizeof(Model)) ) )
		ger_fatal(
		  "create_model: unable to allocate %d bytes",
		  sizeof(Model) );

	/* Fills the cells with appropriate data */
	thismodel->bound_box= (Boundary *)NULL;
	thismodel->vertex_list= (Vertex *)NULL;
	thismodel->vertex_count= 0;
	thismodel->facet_list= (Facet *)NULL; 
	thismodel->facet_count= 0;
	thismodel->part_list= (Part *)NULL;
	thismodel->part_count= 0;
	thismodel->displace_list= (Vertex *)NULL;
        thismodel->displace_scale= 0.0;
	thismodel->color_table= (Color *)NULL;	
	/* Return the new model */
	return (thismodel );
}

void destroy_model ( thismodel )
Model *thismodel;
/* This routine destroys and existing model */
{
	Vertex *thisvertex;
	Facet *thisfacet;
	Part *thispart;
	Color *thiscolor;
	int aloop;

	ger_debug ("destroy_model:");

	/* 
	March down the vertex list, destroying each vertex' color and
	normal cells if they exist.
	*/
	thisvertex= thismodel->vertex_list;
	for (aloop=0; aloop<thismodel->vertex_count; aloop++) {
		if (thisvertex->color) destroy_color ( thisvertex->color );
		if (thisvertex->normal) destroy_vector ( thisvertex->normal );
		thisvertex++; };

	/* March down the facet list, destroying each facet's vnodes. */
	thisfacet= thismodel->facet_list;
	for (aloop=0; aloop<thismodel->facet_count; aloop++) {
		destroy_vnode ( thisfacet );
		thisfacet++; }

	/* 
	March down the part list, destroying each part's color cell
	if it exists.
	*/
 	thispart= thismodel->part_list;
	for (aloop=0; aloop<thismodel->part_count; aloop++) {
		if (thispart->color) destroy_color (thispart->color );
		thispart++; };
	
	/* 
	March down the displacement list, destroying each vertex' color and
	normal cells if they exist.
	*/
	thisvertex= thismodel->displace_list;
	for (aloop=0; aloop<thismodel->vertex_count; aloop++) {
		if (thisvertex->color) destroy_color ( thisvertex->color );
		if (thisvertex->normal) destroy_vector ( thisvertex->normal );
		thisvertex++; };

	/* Deallocate the bound_box, vertex list, facet list, part list, 
	   displacement list, and color_table */

	free ( (void *)thismodel->bound_box );
	free ( (void *)thismodel->vertex_list );
	free ( (void *)thismodel->facet_list );
	free ( (void *)thismodel->part_list );   
	free ( (void *)thismodel->displace_list );
	free ( (void *)thismodel->color_table );

	/* Destroy the model itself */
	free ( (void *)thismodel );
}
