/*****************************************************************************/
/* module winlib.c							     */
/*									     */
/* Author: Rene W. Lutz							     */
/*	   Labo Image							     */
/*	   Computing Science Center					     */
/*	   University of Geneva, Switzerland				     */
/* Date:   March 1989							     */
/* Modifications:   March 25, 1989: some cleaning.			     */
/*		    Sept 24, 1990: histo for zone also.
/* Copyright (c) A. Jacot-Descombes, T. Pun, C. Pellegrini, Uni. of Geneva   */
/* (This copyright notice should appear).				     */
/*									     */
/*****************************************************************************/

/****************************************************************/
/*								*/
/*			    winlib.c  1.0			*/
/*								*/
/*    Ensemble de procedures de gestions d'affichage d'images	*/
/*	    (integre dans le logiciel LaboImage)		*/
/*								*/
/****************************************************************/


#include <stdio.h>
#include <math.h>
#include <strings.h>
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/textsw.h>
#include <suntool/panel.h>
#include <pixrect/pixrect_hs.h>

#include "define.h"
#include "structure.h"

/*
    definition des constantes
*/

#define	    ZONE    1
#define	    BORD    0
#define	    MASQUE  1
#define	    PIXWIN  0
      
#define	    XSUP    0
#define	    YSUP    11
#define	    XINF    0
#define	    YINF    254

/*
    importation et definition des curseurs
*/

static short curseur[] = {
#include "icon/graphics.cursor"
};
mpr_static(zone_curs, 16, 16, 1, curseur) ;

static short sab[] = {
#include "icon/sablier.cursor"
};
mpr_static(wait_curs, 16, 16, 1, sab) ;
        
static short arrcurs[] = {
#include "icon/arrow.cursor"
};
mpr_static(arrow_curs, 16, 16, 1, arrcurs) ;

static short mouse_left_icone[] = {
#include <images/confirm_left.pr>    
};
mpr_static(mouse_left, 16, 16, 1, mouse_left_icone);

static short mouse_middle_icone[] = {
#include <images/confirm_middle.pr>    
};
mpr_static(mouse_middle, 16, 16, 1, mouse_middle_icone);

static short mouse_right_icone[] = {
#include <images/confirm_right.pr>    
};
mpr_static(mouse_right, 16, 16, 1, mouse_right_icone);


/****************************************************************

	    toutes les procedures de ce module

****************************************************************/

/*
    procedures externes au module, lien avec LaboImage
*/

extern Menu     mcurs, mcoord ;
extern int	write_master();
extern int	write_erreur();
extern void	fromto();
extern int	statis() ;
extern Pixwin	*mappw ;
extern void	information() ;
extern void	draw_map_on_canvas() ;
extern void	entree_zone_proc();

/*
    procedures appelees dans le menu de l'image
*/

static void	creer_zone() ;

static void	original_proc() ;
static void	masque_proc() ;
static void	zone_sur_image_proc() ;
static void	zone_proc() ;
static void	bord_proc() ;

static void	stat_zone() ;
static void	lire_zone() ;
static void	sauver_image() ;
static void	set_menu() ;
static void	afficher_histo() ;

/*
    procedures de gestion de liste de points
*/

static void	free_liste_pts();
static void	reset_zone() ;
static void	dessine_zone();
static void	do_zone();

/*
    procedures de gestion de pixrect
*/

static void	show_pixrect() ;
static void     creer_masque();
static void     set_graph_cursor() ;
static void     set_win_cursor() ;
static void     set_wait_cursor() ;
static void	destroy_mem_pixrect() ;

/* 
    procedures d'utilite generale
*/

static int	test_new_pixwin() ;
static void	show_info() ;
static void	fin_message() ;
static void	set_win_curseur() ;
static void	set_wait_curseur() ;
static void	set_graph_curseur() ;
static void	write_coord() ;
static void	master_coord() ;
static void	ret_inform() ;

/*
    les variables globales du module
*/

Frame       message_frame ;
Panel	    message_panel ;


extern int flag_break ;


typedef	struct coords	*ptr_point;           

struct	    coords  {
			short	    x,y;
			ptr_point   next;
		    };

typedef	struct str_zone	*ptr_zone;             

struct	    str_zone	{
			    int		nbr_pts ;
			    ptr_point	pts ;		    
			};

ptr_zone    pt_zone = NULL ;

ptr_point   pt_debut, pt_courant;

int	    curs_val = 1 ;
int	    coord_val = 1 ;
int	    xcoord = XSUP ;
int	    ycoord = YSUP ;

int	    WLnbr_pts = 0;

int	    WLcontrainte = 0;		/* Si 1 on force les droites a etre horizontales */
					/* ou verticales*/

/**/int     index_image[8];	      


static Pixwin		    *pw ;
static struct pixrect	    *im_pixrect,*original_pic,*masque, *im_zone,
			    *im_bord ;	    

static int		    width, height, 
			    masque_width, masque_height ;

/****************************************************************

		procedure de lien avec LaboImage

****************************************************************/


/****************************************************************/
/*								*/
/*  nom	    :	entree_zone_proc				*/
/*								*/
/*  fonction:	appelee depuis win.c, elle selectionne la	*/
/*		fonction de zone a executer en fonction de	*/
/*		nombre passe en parametre			*/
/*								*/
/*  entrees :	int	sel					*/
/*								*/
/*  globales:	WL_contrainte					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	creer_zone, lire_zone, set_menu,		*/
/*		test_new_pixwin, sauver_image, stat_zone, 	*/
/*		afficher_histo, original_proc, masque_proc	*/
/*		zone_sur_image_proc, zone_proc, bord_proc	*/
/*								*/
/****************************************************************/

extern void entree_zone_proc(canvas,sel)
Canvas	canvas ;
int	sel;

{
    test_new_pixwin(canvas) ;
  
    switch (sel)
    {
	case 20:    WLcontrainte = 0 ;
		    creer_zone(canvas);
		    break;

	case 21:    WLcontrainte = 1 ;
		    creer_zone(canvas);
		    break;

	case 4 :    lire_zone(canvas) ;
		    break;

	case 31: case 32: case 33: case 34 : set_menu(mcurs, sel, &curs_val) ;
		    break;

	case 91: case 92 : case 93 : set_menu(mcoord, sel, &coord_val)   ;
		    break;

	case 50 :   sauver_image(canvas,MASQUE);
		    break;

	case 51 :   sauver_image(canvas,PIXWIN);
		    break;

	case 60:    stat_zone(canvas,ZONE);
		    break;

	case 61:    stat_zone(canvas,BORD);
		    break;

	case 71 :  case 72 : case 73 :    afficher_histo(canvas, sel);
		    break;

	case 80:    original_proc();
		    break;

	case 81:    masque_proc();
		    break;

	case 82:    zone_sur_image_proc(canvas);
		    break;

	case 83:    zone_proc();
		    break;

	case 84:    bord_proc();
		    break;
    }
    
    return;
}


/****************************************************************

	    procedure appelees dans le menu de l'image

****************************************************************/


/****************************************************************/
/*								*/
/*  nom	    :	creer_zone					*/
/*								*/
/*  fonction:	donne le controle a la procedure de creation	*/
/*		de zone : do_zone				*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	---						*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	set_graph_curseur, show_info, do_zone		*/
/*                                                              */
/****************************************************************/

static void
creer_zone(canvas)
Canvas canvas ;
{
    set_graph_curseur(canvas) ;

    window_set(canvas, WIN_EVENT_PROC, do_zone, 0) ;

    show_info(" --> Fix points"," --> Zone done"," --> Abort") ;
}


/****************************************************************/
/*  nom	    :	original_proc					*/
/*								*/
/*  fonction:	affiche l'image originale			*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	original_pic					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	show_pixrect					*/
/*								*/
/****************************************************************/

static void
original_proc()
{
    if (original_pic != NULL)
	show_pixrect(original_pic) ;
}


/****************************************************************/
/*  nom	    :	masque_proc					*/
/*								*/
/*  fonction:	affiche le masque				*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	masque, width, height,				*/ 
/*		masque_width, masque_height			*/ 
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	show_pixrect					*/
/*								*/
/****************************************************************/

static void
masque_proc()
{	
    if ((masque != NULL) && (masque_width == width) &&
       (masque_height == height))
	    show_pixrect(masque);
}


/****************************************************************/
/*  nom	    :	zone_sur_image_proc				*/
/*								*/
/*  fonction:	affiche la zone sur l'image originale		*/
/*							        */
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	pt_zone, masque_height, masque_width,	        */
/*		height, width					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	dessine_zone					*/
/*								*/
/****************************************************************/

static void
zone_sur_image_proc(canvas)
Canvas canvas ;
{
    if ((pt_zone != NULL) && (masque_width == width) &&
	(masque_height == height))
	    dessine_zone(pt_zone) ;
}


/****************************************************************/
/*								*/
/*  nom	    :	zone_proc					*/
/*								*/
/*  fonction:	affiche la partie delimitee par la zone		*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	masque, width, height,				*/
/*		masque_width, masque_height			*/ 
/*		im_zone, original_pic				*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	destroy_mem_pixrect, show_pixrect, write_erreur		*/
/*								*/
/****************************************************************/

static void
zone_proc()
{
    if ((masque != NULL) && (masque_width == width) &&
       (masque_height == height))
    {
	destroy_mem_pixrect(im_zone) ;

	if ((im_zone = mem_create (width,height,8)) == NULL)
	{
	    write_erreur(3) ;
	    return ;
	}

	show_pixrect(original_pic) ;

	pw_read(im_zone, 0, 0, width, height, PIX_SRC, pw, 0, 0) ;
	pr_rop(im_zone, 0, 0, width, height, PIX_SRC & PIX_DST, masque, 0, 0) ;
	show_pixrect(im_zone);
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	bord_proc					*/
/*								*/
/*  fonction:	affiche l'original avec un trou			*/
/*		correspondant a la zone				*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	masque, width, height,				*/
/*		masque_width, masque_height			*/ 
/*		im_bord, original_pic				*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	destroy_mem_pixrect, show_pixrect, write_erreur		*/
/*								*/
/****************************************************************/

static void
bord_proc()
{

    if ((masque != NULL) && (masque_width == width) &&
       (masque_height == height))
    { 
	destroy_mem_pixrect(im_bord) ;

	if ((im_bord = mem_create (width,height,8)) == NULL)
	{
	    write_erreur(3) ;
	    return ;
	}

	show_pixrect(original_pic) ;
	pw_read(im_bord, 0, 0, width, height, PIX_SRC, pw, 0, 0) ;
	pr_rop(im_bord, 0, 0, width, height, PIX_NOT(PIX_SRC) & PIX_DST, masque, 0, 0) ;

	show_pixrect(im_bord);
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	stat_zone					*/
/*								*/
/*  fonction:	calcule les statistiques de la zone et affiche	*/
/*		les resultats					*/
/*								*/
/*  entrees :	Canvas canvas : int pix				*/
/*								*/
/*  globales:	masque, original_pic, masque_height,		*/
/*		masque_width, height, width,			*/ 
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	set_wait_curseur, set_win_curseur		*/
/*              zone_proc, bord_proc, show_pixrect              */
/*                                                              */
/****************************************************************/

static void
stat_zone(canvas,pix)
Canvas canvas ;
int pix ;
{
    int		    size;
    float	    ecart ;

    register int    n, nb ;
    float  t,q,r,f, mu, mmin, mmax;
    register unsigned char *m, *b ;

    Frame   finfo,fowner ;
    Textsw  tinfo ; 
    char    *buf,*line,*titre,*type ;

    if (masque_height == height && masque_width == width)
    {
	set_wait_curseur(canvas) ;

	size = height*width ;
	b = (unsigned char *) mpr_d(original_pic) -> md_image ; 
	m = (unsigned char *) mpr_d(masque) -> md_image ;
	nb = 1 ;
	t = 0.0 ;
	mu = 0 ;

        mmin = 255 ;
        mmax = 0 ;

        if (pix == ZONE)
	{
            zone_proc() ;

	    type = "zone-image" ;

	    for (n = 1 ; n <= size ; n++)
	    {
		if (*m != 0)
		{
		    f = (float) *b ;

		    if (f > mmax) 
			mmax = f ;  
                    else
		    if (f < mmin)
			mmin = f ;

		    q = f - mu ;
		    r = q/nb ;
		    mu = mu + r ;
		    t = t + ( nb - 1 ) * q * r ;

		    nb++ ;
		}

		b++ ;
		m++ ;
	    }

            nb = nb - 1 ;

	    ecart = (float) sqrt( t / nb ) ;
	}
	else    
	{
	    bord_proc() ;

	    type = "bord-image" ;

	    for (n = 1 ; n <= size ; n++)
	    {
		if (*m == 0)
		{
		    f = (float) *b ;

		    if (f > mmax) 
			mmax = f ;  
		    else if (f < mmin)
			mmin  = f ;

		    q = f - mu ;
		    r = q/nb ;
		    mu = mu + r ;
		    t = t + ( nb - 1 ) * q * r ;

		    nb++ ;

		}

		b++ ;
		m++ ;
	    }
	    nb = nb - 1 ;
	    ecart = (float) sqrt( t /  nb ) ;
	}
	
	set_win_curseur(canvas) ;

	if (nb > 0)
	{
	    fowner = (Frame) window_get(canvas, WIN_OWNER, 0) ;
	    titre = (char *) window_get(fowner, FRAME_LABEL, 0) ;
	    buf = (char *) malloc (50) ;

	    strncpy(buf,titre,4) ;
	    strcat(buf," Statistiques : ") ;
	    strcat(buf,type) ;

	    finfo = window_create(NULL, FRAME,
				FRAME_LABEL, buf,
				FRAME_NO_CONFIRM, TRUE,
				0) ;

	    tinfo = window_create(finfo, TEXTSW,
				WIN_WIDTH, 270 ,
				WIN_HEIGHT, 230 ,
				TEXTSW_SCROLLBAR, NULL,
				TEXTSW_IGNORE_LIMIT,
				TEXTSW_INFINITY, 0) ;

	    window_fit(finfo) ;

	    buf = (char *) malloc (500) ;
	    line = (char *) malloc (80) ;

	    strcpy(buf,"\0") ;
	    strcat(buf,"\nDESCRIPTION DE LA ZONE : \n\n\0") ;
	    strcat(buf, sprintf (line, "   Nombre de pixels : %d\n\0",nb)) ;
	    strcat(buf,"\nSTATISTIQUES : \n\n\0") ;
	    strcat(buf, sprintf (line, "   Minimum    : %.4f\n\0",mmin)) ;
	    strcat(buf, sprintf (line, "   Maximum    : %.4f\n\0",mmax)) ;
	    strcat(buf, sprintf (line, "   Esperance  : %.4f\n\0",mu)) ;
	    strcat(buf, sprintf (line, "   Ecart type : %.4f\n\0",ecart)) ;

	    textsw_insert(tinfo, buf, strlen(buf)) ;
	    window_set(tinfo, TEXTSW_READ_ONLY, TRUE, 0) ;
	    free(buf) ;
	    free(line) ;

	    window_set(finfo, WIN_SHOW, TRUE, 0) ;
	}
    }
}


/****************************************************************/
/*  nom	    :	lire_zone					*/
/*								*/
/*  fonction:	lecture d'un masque binaire			*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	masque, masque_height, masque_width		*/
/*		flag_break, dir_desc[], dir_image[]		*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	from_to, interruption, destroy_mem_pixrect,		*/
/*		set_wait_curseur, set_win_curseur,		*/
/*		masque_proc, reset_zone, ret_inform		*/
/*                                                              */
/****************************************************************/

static void
lire_zone(canvas)
Canvas canvas ;
{
    int plan, x, y ;
    unsigned char *d, *temp;
    char buf[50] ;


    write_master("Lecture masque binaire : ") ;

    fromto(FROM,DEFAUT) ;
    if (flag_break)
    {
	interruption() ;
	return ;
    }

	plan = index_image[0] ;
	if ((dir_desc[plan].type == -1) && (dir_image[plan].image != NULL))								    
	{
	    set_wait_curseur(canvas) ;
	
	    masque_width = dir_desc[plan].ncolonne ;
	    masque_height = dir_desc[plan].nligne ;
 
	    destroy_mem_pixrect(masque) ;

	    if ((masque = mem_create (masque_width, masque_height,8)) == NULL)
	    {
		write_erreur(3) ;
		return ;
	    }

	    d = (unsigned char *) mpr_d(masque) -> md_image ;

	    temp = dir_image[plan].image ;

	    for (y = 0 ; y <= masque_height ; y++) 
		for (x = 0 ; x <= masque_width ; x++)
		{
		    *d = *temp;
		    temp++ ;
		    d++ ;
		}

	    set_win_curseur(canvas) ;
	    sprintf(buf,"--> image %2d \n",plan) ;
	    write_master(buf) ;
	    masque_proc() ;
	    reset_zone(pt_zone) ;
	}
	else
	{
	    sprintf(buf," --- image %2d non binaire ---\n",plan) ;
	    write_master(buf) ;
	}
 
    ret_inform(canvas) ;
}


/****************************************************************/
/*  nom	    :	set_menu					*/
/*								*/
/*  fonction:	change l'etiquette du menu "mmenu" et modifie	*/
/*		la vriable globale *choice (par reference)	*/
/*								*/
/*  entrees :	Menu mmenu ; int number, *choice		*/
/*								*/
/*  globales:	---						*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
set_menu(mmenu, number, choice)
Menu	mmenu ;
int	number, *choice ;
{
    int		nth ;
    char	*m, *buf1, *buf2 ;
    Menu_item	item1, item2 ;

    nth = number % 10 ;
    if (nth != *choice)
    {
	item1 = menu_get(mmenu, MENU_NTH_ITEM, *choice) ;
	m = (char *) menu_get(item1, MENU_STRING) ;
	m++ ;

	buf1 = (char *) calloc(20, sizeof *buf1) ;

	strcpy(buf1, " ") ;
	strcat(buf1, m) ;

	menu_set(item1, MENU_STRING, buf1, 0) ;

	item2 = menu_get(mmenu, MENU_NTH_ITEM, nth) ;
	m = (char *) menu_get(item2, MENU_STRING) ;
        m++ ;

	buf2 = (char *) calloc(20, sizeof *buf2) ;

	strcpy(buf2, "X") ;
	strcat(buf2, m) ;
	
	menu_set(item2, MENU_STRING, buf2, 0) ;

        *choice = nth ;
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	sauver_image					*/
/*								*/
/*  fonction:	ecrit le masque dans une plan memoire de win	*/
/*								*/
/*  entrees :	Canvas canvas  ; int pix			*/
/*								*/
/*  globales:   masque, dir_image[], dir_desc[], masque_width,  */
/*		masque_height, im_pixrect			*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	fromto, interruption, set_wait_curseur		*/
/*              set_graph_curseur, ret_inform, fin_message      */
/*              destroy_mem_pixrect, write_erreur                       */
/*                                                              */
/****************************************************************/

static void
sauver_image(canvas,pix)
Canvas canvas ;
int pix ;
{
    int plan, x, y;
    unsigned char *d, *image_temp ;
    char buf[100] ;

    if ((pix == MASQUE && masque != NULL) || pix == PIXWIN )
    {
	fromto(TO,DEFAUT) ;
	if (flag_break)
	{
	    interruption() ;
	    return ;
	}

	write_master("Sauvegarde plan-pixel : \0");

	set_wait_curseur(canvas) ;

	plan = index_image[1] ;

	if (dir_image[plan].image != NULL)
	free(dir_image[plan].image) ;

	if (pix == MASQUE)
	{
	    d = (unsigned char *) mpr_d(masque) -> md_image ;

	    dir_image[plan].image = (unsigned char *) malloc (masque_width * masque_height);
	    dir_desc[plan].nligne = masque_height ;
	    dir_desc[plan].ncolonne = masque_width ;
	    dir_desc[plan].type = -1 ;
	    
	    image_temp = dir_image[plan].image ;

	    for (y = 0 ; y <= masque_height ; y++) 
		for (x = 0 ; x <= masque_width ; x++)
		{
		    *image_temp = *d ;
		    image_temp++ ;
		    d++ ;
		}

	    sprintf(buf,"masque --> image %2d\n\0",plan) ;
	    write_master(buf) ;
	}

	else
	{
	    destroy_mem_pixrect(im_pixrect) ;

	    if ((im_pixrect = mem_create (width,height,8)) == NULL)
	    {
		write_erreur(3) ;
		return ;
	    }

	    pw_read(im_pixrect, 0, 0, width, height, PIX_SRC, pw, 0, 0) ;

	    d = (unsigned char *) mpr_d(im_pixrect) -> md_image ;

	    dir_image[plan].image = (unsigned char *) malloc (width * height);
	    dir_desc[plan].nligne = height ;
	    dir_desc[plan].ncolonne = width ;
	    dir_desc[plan].type = 0 ;	
	
	    image_temp = dir_image[plan].image ;

	    for (y = 0 ; y <= height ; y++) 
		for (x = 0 ; x <= width ; x++)
		{
		    *image_temp = *d ;
		    image_temp++ ;
		    d++ ;
		}

	    sprintf(buf,"plan-pixrect --> image %2d\n\0",plan) ;
	    write_master(buf) ;
	}

	statis(dir_image[plan].image,
		dir_desc[plan].type,
		dir_desc[plan].nligne,
		dir_desc[plan].ncolonne,
		&(dir_desc[plan].mmin),  
		&(dir_desc[plan].mmax),  
		&(dir_desc[plan].mu),  
		&(dir_desc[plan].ecart)) ; 

	set_win_curseur(canvas) ;
	fin_message() ;
	ret_inform(canvas) ;
    }
}


/****************************************************************/
/*								*/
/*  nom	:	afficher_histo					*/
/*								*/
/*  fonction:	dessine l'histogramme sur la table de couleur	*/
/*								*/
/*  entrees :	Canvas canvas		                        */
/*								*/
/*  globales:   orginal_pic, mappw                              */ 
/*                                                              */
/*  return  :	---						*/
/*								*/
/*  routines:	draw_map_on_canvas, set_win_curseur		*/
/*		set_map_curseur					*/ 
/*								*/ 
/****************************************************************/

static void
afficher_histo(canvas,number)
Canvas canvas ;
int number ;
{
    int		c[256], size ;

    unsigned char *m, *im ;

    register int    x,y,posx,
		    y1,y0,
		    i,maxcol ;

    register unsigned	char col ;
    register float	dec, coeff ;


    if (mappw != NULL)
    {
	set_wait_curseur(canvas) ;

	for(i=0; i <= 255 ; i++) 
	    c[i] = 0 ;

	im = (unsigned char *) mpr_d(original_pic) -> md_image ;
	size = width*height ;

	switch(number)
	{
		case 71 :	for (x=0 ; x < size ; x++)
				{
					c[*im]++ ;
					im++ ;
				}
				show_pixrect(original_pic) ;
				break ;

		case 72 :       if (masque_height == height && masque_width == width)
				{
				    m = (unsigned char *) mpr_d(masque) -> md_image ;
				    for (x=0 ; x < size ; x++)
				    {
					if (*m != 0)
						c[*im]++ ;
					im++ ; m++ ;
				    }
				    zone_proc() ;
				}
				break ;

		case 73 :       if (masque_height == height && masque_width == width)
				{
				    m = (unsigned char *) mpr_d(masque) -> md_image ;
				    for (x=0 ; x < size ; x++)
				    {
					if (*m == 0)
						c[*im]++ ;
					im++ ; m++ ;
				    }
				    bord_proc() ;
				}
				break ;

	}


	maxcol = c[0] ;

	for(i=1; i <= 255 ; i++) 
	{
	    if (c[i] > maxcol)
	    maxcol = c[i] ;
	}

	dec = 95 ;
	coeff = (float) 95/maxcol ;
	y0 = (int) rint((double) dec - (double) c[0]*coeff) ;

	draw_map_on_canvas() ;

	for(i=0 ; i < 255 ; i++) 
	{
	    posx = 64 + 2*i ;
	    y1 = (int) rint((double) dec - (double) c[i]*coeff) ;
	    pw_vector(mappw,posx , y0 , posx+2 , y1, PIX_SRC | PIX_COLOR(255-i), 1) ;
	    pw_vector(mappw,posx+1 , y0 , posx+3 , y1, PIX_SRC | PIX_COLOR(255-i), 1) ;
	    y0 = y1 ;
	}

	set_win_curseur(canvas) ;
    }
}


/****************************************************************

	    procedure de gestion de liste de points

****************************************************************/


/****************************************************************/
/*								*/
/*  nom	:	free_liste_pts					*/
/*								*/
/*  fonction:	libere la liste des points pointee par pptr .   */
/*								*/
/*  entrees :	ptr_point   pptr                                */
/*								*/
/*  globales:   ---                                             */ 
/*                                                              */
/*  return  :	---						*/
/*								*/
/*  routines:	---		 				*/
/*								*/ 
/****************************************************************/

static void 
free_liste_pts(pptr)
ptr_point   *pptr;
{
    ptr_point   p;

    while (*pptr != NULL)
    {
	p = *pptr;
	*pptr = (*pptr) -> next;
	cfree(p);
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	reset_zone					*/
/*								*/
/*  fonction:	libere la zone memoire alouee pour le polygone	*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	pt_zone						*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	free_liste_pts					*/
/*								*/
/****************************************************************/

static void
reset_zone() 
{
    if (pt_zone != NULL)
    {
	free_liste_pts(&(pt_zone -> pts));
	cfree(pt_zone) ;
	pt_zone = NULL ;
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	dessine_zone					*/
/*								*/
/*  fonction:	relie successivement les points stockes dans    */
/*		la liste par des segments de droite bicolores	*/
/*								*/
/*  entrees :	ptr_zone	zptr                            */
/*								*/
/*  globales:   im_pixrect, width, height, pw, original_pic	*/ 
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	destroy_mem_pixrect, write_erreur			*/
/*                                                              */
/****************************************************************/

static void 
dessine_zone(zptr)
ptr_zone zptr;
{
    ptr_point	temp, oldtemp;

    if (zptr != NULL)
    {
	destroy_mem_pixrect(im_pixrect) ;

	if ((im_pixrect = mem_create (width,height,8)) == NULL)
	{
	    write_erreur(3) ;
	    return ;
	}

	show_pixrect(original_pic) ;
	pw_read(im_pixrect, 0, 0, width, height, PIX_SRC, pw, 0, 0) ;
	oldtemp = zptr -> pts;
	temp = oldtemp -> next;

	while (temp != NULL)
	{         
	    pr_vector (im_pixrect, oldtemp -> x, (oldtemp -> y) + 1, 
			temp -> x, (temp -> y) + 1 , PIX_SRC, 0);
	    
	    pr_vector (im_pixrect, oldtemp -> x, oldtemp -> y, 
			temp -> x, temp -> y, PIX_SRC, 255);
	    
	    oldtemp = temp;
	    temp = temp -> next;
	}

	show_pixrect(im_pixrect) ;
    }
}


/****************************************************************/
/*								*/
/*  nom	    :	do_zone						*/
/*								*/
/*  fonction:	dessine une zone a l'ecran et garde la liste	*/
/*		des points en memoire				*/
/*								*/
/*  entrees :	Canvas canvas  ; Event *event			*/
/*								*/
/*  globales:   pt_debut, pt_courant, pt_zone, im_pixrect 	*/
/*		WLnbr_ptr, WLcontrainte				*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	afficher_dessin, reset_zone, dessine_zone,	*/
/*		creer_masque					*/
/*                                                              */
/****************************************************************/

static void
do_zone(canvas,event)
Canvas canvas ;
Event *event ;
{
    int	    x, y, pix ;
    
    if (coord_val != 1)
    {
	x = event_x(event) ;
	y = event_y(event) ;
        pix = pr_get(original_pic, x,y) ;
	write_coord(canvas, xcoord, ycoord, x, y, pix ) ;
    }

    if ((event_id(event) == MS_LEFT) && (event_is_down(event) == TRUE))
    {
	    if (pt_debut == NULL)
	    {
		pt_debut = (ptr_point) calloc (1,sizeof (struct coords));
		pt_courant = pt_debut ;

		pt_courant -> next = NULL;
		pt_courant -> x = event_x(event);
		pt_courant -> y = event_y(event);
	    }
	    else
	    switch (WLcontrainte) 
	    {
		case 0 : pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
			     pt_courant = pt_courant -> next;

			     pt_courant -> next = NULL;
			     pt_courant -> x = event_x(event);
			     pt_courant -> y = event_y(event);
			     break;

		case 1 : if (abs((pt_courant -> x) - (event_x(event))) < 
						abs((pt_courant -> y) - (event_y(event))))
			 {
			    x = pt_courant -> x;
			    y = event_y(event);
			 }
			 else
			 {
			    x = event_x(event);
			    y = pt_courant -> y;
			 }
			 pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
			 pt_courant = pt_courant -> next;
			 pt_courant -> next = NULL;
	
			 pt_courant -> x = x;
			 pt_courant -> y = y;
			 break;
		 }

	    pw_rop(pw, (pt_courant -> x) - 1 , (pt_courant -> y) - 1, 3, 3, PIX_SRC | PIX_COLOR(255), NULL, 0, 0) ;
	    pw_put(pw, pt_courant -> x, pt_courant -> y, 0);

	    WLnbr_pts++;
    }
    else     
    if ((event_id(event) == MS_MIDDLE) && (event_is_down(event) == TRUE))
    {
        show_pixrect(original_pic) ;

	if (pt_debut != NULL)
	{
	    /* On ferme le polygone */
	    switch (WLcontrainte)
	    {
		case 0 :
			    pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
			    pt_courant =  pt_courant -> next;
			    pt_courant -> next = NULL;
			    pt_courant -> x = pt_debut -> x;
			    pt_courant -> y = pt_debut -> y;
			    WLnbr_pts++;
			    break;

    /* Si on a la contrainte hv il faut rajouter un point (en plus de celui pour fermer	*/ 
    /* la figure) afin que le dernier segment soit bien vertical ou horizontal	        */

		case 1 :
			    if ((pt_courant -> x) - (pt_debut -> x) < 0)
			    {
				pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
				x = pt_courant -> x;
				y = pt_debut -> y;
				pt_courant =  pt_courant -> next;
				pt_courant -> next = NULL;
				pt_courant -> x = x;
				pt_courant -> y = y;
	
				WLnbr_pts++;
	
			    }
			    else
			    if ((pt_courant -> x) - (pt_debut -> x) > 0)
			    {
				pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
				x = pt_debut -> x;
				y = pt_courant -> y;
				pt_courant =  pt_courant -> next;
				pt_courant -> next = NULL;
				pt_courant -> x = x;
				pt_courant -> y = y;
				WLnbr_pts++;
			    }

			    pt_courant -> next = (ptr_point) calloc (1,sizeof (struct coords));
			    pt_courant =  pt_courant -> next;
			    pt_courant -> next = NULL;
			    pt_courant -> x = pt_debut -> x;
			    pt_courant -> y = pt_debut -> y;
			    WLnbr_pts++;
			    break;

		}	/* switch */

		reset_zone() ;		

		pt_zone = (ptr_zone) calloc (1, sizeof(struct str_zone)) ;

		pt_zone -> pts = pt_debut ;
		pt_zone -> nbr_pts = WLnbr_pts ;
		WLnbr_pts = 0 ;

		pt_debut = NULL ;

		dessine_zone(pt_zone) ;
		creer_masque(pt_zone) ;
		master_coord(pt_zone) ;
	    }
        

	fin_message() ;
	set_win_curseur(canvas) ;
	window_set(canvas, WIN_EVENT_PROC, information, 0) ;
        ret_inform(canvas) ;

    }

    else if ((event_id(event) == MS_RIGHT) && (event_is_down(event) == TRUE))
    {
	fin_message() ;
	reset_zone(pt_debut) ;
	set_win_curseur(canvas) ;
	show_pixrect(original_pic) ;
	ret_inform(canvas) ;
    }
}


/****************************************************************

	        procedure de gestion de pixrect

****************************************************************/


/****************************************************************/
/*								*/
/*  nom	    :	show_pixrect	          			*/
/*								*/
/*  fonction:	affiche le pixrect pointe par pix_ptr .		*/
/*								*/
/*  entrees :	Pixrect    *pix_ptr                             */
/*								*/
/*  globales:   pw, width, height	                        */ 
/*                                                              */
/*  return  :	---						*/
/*								*/
/*  routines:	---		 				*/
/*								*/ 
/****************************************************************/

static void 
show_pixrect(pix_ptr)
Pixrect *pix_ptr ;	
{
    pw_rop(pw, 0, 0, width, height, PIX_SRC, pix_ptr, 0, 0) ; 
}



/****************************************************************/
/*								*/
/*  nom	    :	creer_masque       				*/
/*								*/
/*  fonction:	creation du masque binaire a partir de la zone	*/
/*								*/
/*  entrees :	ptr_zone    zptr                                */
/*								*/
/*  globales:	masque, masque_height, masque_width		*/ 
/*              height, width                                   */
/*                                                              */
/*  return  :	---						*/
/*								*/
/*  routines:	write_erreur, destroy_mem_pixrect			*/
/*                                                              */
/****************************************************************/

static void 
creer_masque(zptr)
ptr_zone zptr;
{
    struct pr_pos    *tab;
    ptr_point	    pos;
    int		    n, i, nbds = 1, npts[1]; 

    n = zptr -> nbr_pts ;
    pos = zptr -> pts;

    /* n-1 car on ne prend pas le dernier point (qui sert a fermer le polygone): */
    /* pw_polygon_2 le fait tout seul						 */

    tab = (struct pr_pos *) calloc (( n-1) , sizeof (struct pr_pos));

   /* creation du masque */

	destroy_mem_pixrect(masque) ;

	if ((masque = mem_create (width,height,8)) == NULL)
	{
	    write_erreur(3) ;
	    return ;
	}

    masque_width = width ;
    masque_height = height ;

/* On initialise le tableau */

    for (i = 0; i < (n-1); i++)	/* n-1 car on par de 0 avec un tableau */
    {
	tab[i].x = pos -> x;
	tab[i].y = pos -> y;
	pos = pos -> next;
    }

    npts[0] = n-1;
    pr_polygon_2 (masque, 0, 0, nbds, npts, tab, (PIX_SRC | PIX_COLOR(255)), NULL, 0, 0);
    cfree(tab);

}


/****************************************************************/
/*  nom	    :	set_graph_curseur				*/
/*								*/
/*  fonction:	associe le curseur "mire" a la fentre en 	*/
/*		ajoutant des crosshairs selon la variable 	*/
/*		globale associee au menu : curs-val		*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	zone_curs, curs_val				*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
set_graph_curseur(canvas)
Canvas canvas ;
{
    Cursor cursor ;

    window_set(canvas, WIN_CURSOR, 
		cursor_create(CURSOR_IMAGE, &zone_curs,
		CURSOR_XHOT, 7,		    
		CURSOR_YHOT, 7,
		CURSOR_OP, PIX_SRC | PIX_DST,
		CURSOR_CROSSHAIR_BORDER_GRAVITY, TRUE,
		CURSOR_CROSSHAIR_GAP, 5, 
		0),
		0) ; 

    cursor = window_get(canvas, WIN_CURSOR) ;
    cursor_set(cursor, CURSOR_SHOW_CROSSHAIRS, FALSE, 0) ;

    switch (curs_val)
    {
	case 1: break ;
	
	case 2:	cursor_set(cursor, CURSOR_SHOW_HORIZ_HAIR, TRUE, 0) ;
		break ;

	case 3:	cursor_set(cursor, CURSOR_SHOW_VERT_HAIR, TRUE, 0) ;
		break ;

	case 4:	cursor_set(cursor, CURSOR_SHOW_CROSSHAIRS, TRUE, 0) ;
		break ;
    }

    window_set(canvas, WIN_CURSOR, cursor, 0) ;
}


/****************************************************************/
/*  nom	    :	set_win_curseur					*/
/*								*/
/*  fonction:	associe le curseur "fleche" a la fenetre	*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	arrow_curs					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
set_win_curseur(canvas)
Canvas	canvas ;
{
    window_set(canvas, WIN_CURSOR, 
		  cursor_create(CURSOR_IMAGE, &arrow_curs,
				CURSOR_XHOT, 1,		    
				CURSOR_YHOT, 1,
				CURSOR_OP, PIX_SRC | PIX_DST,
				CURSOR_SHOW_CROSSHAIRS, FALSE,
				0),
		0) ; 
}


/****************************************************************/
/*  nom	    :	set_wait_curseur				*/
/*								*/
/*  fonction:	associe le curseur "sablier" a la fenetre	*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	wait_curs					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
set_wait_curseur(canvas)
Canvas	canvas ;
{
    window_set(canvas, WIN_CURSOR, 
		  cursor_create(CURSOR_IMAGE, &wait_curs,
				CURSOR_OP, PIX_SRC ^ PIX_DST,
				CURSOR_SHOW_CROSSHAIRS, FALSE,
				0),
		0) ; 
}


/****************************************************************/
/*  nom	    :	destroy_mem_pixrect					*/
/*								*/
/*  fonction:	detruit le pixrect et sa place memoire		*/
/*								*/
/*  entrees :	struct pixrect *pt_mem				*/
/*								*/
/*  globales:	---						*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
destroy_mem_pixrect(pt_mem) 
struct pixrect *pt_mem ;
{
    if (pt_mem != NULL)
	pr_destroy(pt_mem) ;
}


/****************************************************************

	        procedure d'utilite generale

****************************************************************/


/****************************************************************/
/*								*/
/*  nom	    :	test_new_pixwin					*/
/*								*/
/*  fonction:	teste si le curseur se trouve dans une		*/
/*		nouvelle image et met, le cas echeant,		*/
/*		les variables globales a jour			*/
/*								*/
/*  entrees :	Canvas canvas					*/
/*								*/
/*  globales:	xcoord, ycoord, width, height			*/
/*		pw, original_pic				*/
/*								*/
/*  return  :	TRUE si nouvelle image, FALSE sinon		*/
/*								*/
/*  routines:	destroy_mem_pixrect					*/
/*								*/
/****************************************************************/

static int
test_new_pixwin(canvas)
Canvas canvas ;
{
    Pixwin *new_pw ;
    int	    new_width, new_height ;


    if (coord_val == 2)
    {   
	xcoord = XSUP ;
	ycoord = YSUP ;
    }
    else if (coord_val == 3)
    {
	xcoord = XINF ;
	ycoord = height - 2 ;
    }

    new_pw = canvas_pixwin(canvas) ;

    if (new_pw != pw)
    {
	pw = canvas_pixwin(canvas) ;

	new_width = (int) window_get(canvas, WIN_WIDTH) - 15 ;
	new_height = (int) window_get(canvas, WIN_HEIGHT) - 15 ;

	if ((new_width != width) || (new_height != height))
	{
	    width = new_width ;
	    height = new_height ;
	    ycoord = height - 2 ;
	}

	destroy_mem_pixrect(original_pic) ;

	if ((original_pic = mem_create (width,height,8)) == NULL)
	{
	    write_erreur(3) ;		    
	    return ;
	}

	pw_read(original_pic, 0, 0, width, height, PIX_SRC, pw, 0, 0) ;
	
        return(TRUE) ;
    }
    else
	return(FALSE) ;	    
}


/****************************************************************/
/*								*/
/*  nom	    :   show_info					*/
/*								*/
/*  fonction:	affiche des informations relatives a l'utilisa-	*/
/*		tion de la souris				*/
/*								*/
/*  entrees :	char *left, *middle, *right			*/
/*								*/
/*  globales:   message_frame, message_panel                    */ 
/*                                                              */
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/ 
/****************************************************************/

static void
show_info(left, middle, right) 
char *left, *middle, *right ;
{
    message_frame = window_create(NULL, FRAME,
					     0) ;

    message_panel = window_create(message_frame, PANEL ,
					    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 10,
				    PANEL_LABEL_Y, 3,
				    PANEL_LABEL_IMAGE, &mouse_left,
				    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 30,
				    PANEL_LABEL_Y, 5,
				    PANEL_LABEL_STRING, left,
				    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 10,
				    PANEL_LABEL_Y, 31,
				    PANEL_LABEL_IMAGE, &mouse_middle,
				    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 30,
				    PANEL_LABEL_Y, 33,
				    PANEL_LABEL_STRING, middle,
				    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 10,
				    PANEL_LABEL_Y, 59,
				    PANEL_LABEL_IMAGE, &mouse_right,
				    0);

    panel_create_item(message_panel, PANEL_MESSAGE,
				    PANEL_LABEL_X, 30,
				    PANEL_LABEL_Y, 61,
				    PANEL_LABEL_STRING, right,
				    0);

    window_fit(message_panel) ;
    window_fit(message_frame) ;

    window_set(message_frame, WIN_X, 476, 
				WIN_Y, 25, 
				WIN_SHOW, TRUE ,
				0) ;
}


/****************************************************************/
/*								*/
/*  nom	    :	fin_message					*/
/*								*/
/*  fonction:	detruit le panneau de messages			*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	message_frame					*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*                                                              */
/****************************************************************/

static void
fin_message()
{
    window_set(message_frame, FRAME_NO_CONFIRM, TRUE, 0); 
    window_destroy(message_frame) ;
}


/****************************************************************/
/*  nom	    :	write_coord					*/
/*								*/
/*  fonction:	inscrit les parametres x, y et pix sur la	*/
/*		fenetre, au point orgx, orgy			*/
/*								*/
/*  entrees :	Canvas	canvas ; int orgx, orgy, x, y, pix 	*/
/*								*/
/*  globales:	pw						*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	---						*/
/*								*/
/****************************************************************/

static void
write_coord(canvas, orgx, orgy, x, y, pix)
Canvas	canvas ;
int	orgx, orgy, x, y, pix ;	
{    
    char buf[30] ;

    pw = canvas_pixwin(canvas) ;

    if ((x >= 0 && x <= width) && (y >= 0 && y <= height))
    {
	sprintf(buf,"X: %3d ; Y: %3d ; pixel : %3d", x, y, pix) ;
	pw_text(pw, orgx, orgy, PIX_SRC | PIX_COLOR(255) , NULL, buf) ; 
    }
}


/****************************************************************/
/*  nom	    :	master_coord					*/
/*								*/
/*  fonction:	inscrit les coordonnes des points de la zone	*/
/*		nouvellement cree dans les TEXTSW de LaboImage	*/
/*								*/
/*  entrees :	ptr_zone    zptr				*/
/*								*/
/*  globales:	arrow_curs, im_pixrect				*/
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	write_master, show_pixrect			*/
/*								*/
/****************************************************************/

static void 
master_coord(zptr)
ptr_zone zptr;
{
    ptr_point	temp, oldtemp;
    char *buf ;

    if (zptr != NULL)
    {
	oldtemp = zptr -> pts;
	temp = oldtemp -> next;

	buf = (char *) malloc (50) ;
        write_master("Coordonnees des points selectionnes :\n\n") ;

	while (temp != NULL)
	{      
   
	    sprintf(buf,"X: %3d    Y: %3d \n", temp->x, temp->y) ;
	    write_master(buf) ;
	    oldtemp = temp;
	    temp = temp -> next;
	}

        write_master("\n") ;
    }
}


/*****************************************************************/
/*								*/
/*  nom	    :	ret_inform					*/
/*								*/
/*  fonction:	redonne le controle a la procedure information	*/
/*								*/
/*  entrees :	---						*/
/*								*/
/*  globales:	canvas						*/ 
/*								*/
/*  return  :	---						*/
/*								*/
/*  routines:	information					*/
/*                                                              */
/****************************************************************/

	
static void
ret_inform(canvas)
Canvas canvas ;
{
	window_set(canvas, WIN_EVENT_PROC, information, 0) ;
}


