#include  <stdio.h>
#include  <string.h>
#include  <time.h>

#include  <genlib.h>

#include   DUT_H
#include  "dplimits.h"
#include  "nand3.h"


/*      Alliance CAD System 2.0                                   */
/*        FITPATH Package 0.0                                     */
/*                                                                */
/*  Author  :      Jean-Paul CHAPUT                               */
/*  E-mail  :  cao-vlsi@masi.ibp.fr                               */
/* ************************************************************** */
/*  module  :  "nand3.c"                                          */


/* Module internal constants. */
#define        E_NAND3_WIDTH         2
#define        E_NAND3_SLICE         3
#define        E_NAND3_DRIVE         4
#define        E_NAND3_NOVIEW        5
#define        E_NAND3_OPENVBE       6
#define        E_NAND3_MODELNAME     7


/* Names of nand3 leaf cells. */
static char *TablLeafCellNand3[2] = {  "na3_fp",
									  "na3p_fp" };


/* Interface function of module.
   extern       char *Dgn_DispachNand3();
   */

/* Internal functions of module. */
static       void  Dgn_ErrorNand3();
static       void  Dgn_VhdlNand3();
static       void  Dgn_NetlistNand3();


/*  Function  :  'Dgn_DispatchNand3'                               */
/* -------------------------------------------------------------- */
/*  Interface of nand3 generator.                                  */

extern char *Dgn_DispatchNand3(nand3_modelname,
							  nand3_width    ,
							  nand3_slice    ,
							  nand3_drive    ,
							  nand3_msb0     ,
							  nand3_vhdl     ,
							  nand3_netlist  ,
							  nand3_icon     )
	char *nand3_modelname;  /* Model name.                    */
	long  nand3_width;      /* Bus wide.                      */
	long  nand3_slice;      /* LSB starting slice.            */
	long  nand3_drive;      /* Output drive capacity.         */
	long  nand3_msb0;       /* Zero as MSB index.             */
	long  nand3_vhdl;       /* Generation of behavioral view. */
	long  nand3_netlist;    /* Generation of netlist view.    */
	long  nand3_icon;       /* Generation of icon.            */
{
	static short  DispachNand3_FirstCall = FALSE;
	static  char  ModelName[C_SUPP_MBKSZNAME];

	/* The first time the generator is call : */
	/*     - Load specific errors.            */
	if (!DispachNand3_FirstCall)
	{
		Dgn_ErrorNand3();
		DispachNand3_FirstCall = TRUE;
	}

	/* Check if, at least, one view is asked. */
	if (   (nand3_netlist == FALSE)
		 &&(nand3_vhdl    == FALSE)
		 &&(nand3_icon    == FALSE) )
		Dut_PrintError( E_NAND3_ERROR, E_NAND3_NOVIEW);

	/* Check the bus wide. */
	if ((nand3_width < 1) || (nand3_width > DP_WIDTH))
		Dut_PrintError( E_NAND3_ERROR, E_NAND3_WIDTH, nand3_width);

	/* Check the output drive capacity. */
	if ((nand3_drive < 1) || (nand3_drive > C_NAND3_NBDRIVE))
		Dut_PrintError( E_NAND3_ERROR, E_NAND3_DRIVE, nand3_drive);

	/* Check the LSB starting slice. */
	if ((nand3_slice < 0) || ( nand3_slice
							 +nand3_width > DP_WIDTH))
		Dut_PrintError( E_NAND3_ERROR, E_NAND3_SLICE, nand3_slice);

	/* Check the model name. */
	if (nand3_modelname == (char*)NULL)
#ifndef  CRUNCH_FILENAME
		/* Build a more explicit model name. */
		(void*)sprintf( ModelName, "nand3_%ldx%ldx%ldx%ld%c_cl",
					                   DP_WIDTH              ,
					                 nand3_width              ,
					                 nand3_slice              ,
					                 nand3_drive              ,
					               ((nand3_msb0 ) ? 'm' : 'l'));
#else
		/* Build the crunched model name. */
		strcpy( ModelName, Dut_MakeCrunchModelName(
									    "od"       ,
								   8L,   DP_WIDTH-1,
								   8L, nand3_width-1,
								   8L, nand3_slice  ,
								   8L, nand3_drive  ,
								   0L, nand3_msb0   ));
#endif
	else
		strcpy( ModelName, nand3_modelname);


	/* Generation of behaviorial view. */
	if (nand3_vhdl)
		Dgn_VhdlNand3( ModelName,
			   		  nand3_width,
					  nand3_slice,
			   		  nand3_drive,
			   		  nand3_msb0 );

	/* Generation of netlist view. */
	if (nand3_netlist)
		Dgn_NetlistNand3( ModelName,
						 nand3_width,
						 nand3_slice,
					     nand3_drive,
						 nand3_msb0 );

	/* Generation of icone. */
	/* if (nand3_icon) */
	
	/* Return name of the generated model. */
	return((nand3_modelname == (char*)NULL) ?      ModelName 
		                                   : nand3_modelname);
}


/*  Function  :  'Dgn_ErrorNand3'                                  */
/* -------------------------------------------------------------- */
/*  Adds generator specific errors to the main errors data base.  */

static void  Dgn_ErrorNand3()
{
	Dut_AddError( E_NAND3_ERROR, E_NAND3_WIDTH,
				  "invalid bus width : %ld.");
	Dut_AddError( E_NAND3_ERROR, E_NAND3_SLICE,
				  "invalid LSB starting slice : %ld.");
	Dut_AddError( E_NAND3_ERROR, E_NAND3_DRIVE,
				  "invalid output drive : %ld.");
	Dut_AddError( E_NAND3_ERROR, E_NAND3_MODELNAME,
				  "Missing model name.");
	Dut_AddError( E_NAND3_ERROR, E_NAND3_NOVIEW,
				  "No view to generate.");
	Dut_AddError( E_NAND3_ERROR, E_NAND3_OPENVBE,
				  "Unable to create file : \"%s\".");
}


/*  Function  :  'Dgn_NetListNand3'                                */
/* -------------------------------------------------------------- */
/*  Generation of nand3 behavioral view.                           */

static void  Dgn_VhdlNand3(nand3_modelname,
					      nand3_width    ,
					      nand3_slice    ,
					      nand3_drive    ,
						  nand3_msb0     )
	char *nand3_modelname;  /* Nom du modele.           */
	long  nand3_width;      /* Largeur du bus.          */
	long  nand3_slice;      /* LSB starting slice.      */
	long  nand3_drive;      /* Output drive power.      */
	long  nand3_msb0;       /* Zero comme index du MSB. */
{
	FILE *Nand3File;
	long  BitIndex,     LenString,      TmpTime;
	char  BusWide[256], TmpString[256];

	/* *** Creating the file holding behavioral view. *** */
	if ((Nand3File=mbkfopen(nand3_modelname,
						            "vbe",
						       WRITE_TEXT)) == (FILE*)NULL)
		Dut_PrintError( E_NAND3_ERROR, E_NAND3_OPENVBE,
					                  nand3_modelname);

	/* *** Write an identification header *** */

	/* Standart begin of header. */
	fprintf( Nand3File, 
"-- ###-------------------------------------------------------------###\n"
			);
	fprintf( Nand3File, 
"-- #                                                                 #\n"
			);
	fprintf( Nand3File,
"-- #    Alliance CAD System 2.0                                      #\n"
			);
    fprintf( Nand3File,
"-- #      FITPATH Package 0.0                                        #\n"
			);
    fprintf( Nand3File,
"-- #                                                                 #\n"
			);
    fprintf( Nand3File,
"-- # Author :     Jean-Paul CHAPUT                                   #\n" 
			);
    fprintf( Nand3File,
"-- # E-mail : cao-vlsi@masi.ibp.fr                                   #\n"
			);
	fprintf( Nand3File, 
"-- # *************************************************************** #\n"
			);
	fprintf( Nand3File,
"-- #                                                                 #\n"
			);
    fprintf( Nand3File,
"-- # Generator : \"nand3g\" (nand3)                                      #\n"
			);
	fprintf( Nand3File, 
"-- # --------------------------------------------------------------- #\n"
			);

	/* Print name of the generated file. */
	sprintf( TmpString, "\"%s.vbe\"."           , nand3_modelname);
    fprintf( Nand3File , "-- # File : %-55s  #\n",     TmpString);
	fprintf( Nand3File ,
"-- #                                                                 #\n"
			);

	/* Print date of generation. */
	TmpTime                = time((time_t*)NULL);
	sprintf( TmpString, "%s", asctime(localtime(&TmpTime)) );
	LenString              = strlen( TmpString);
	TmpString[LenString-1] = (char)0;
    fprintf( Nand3File , "-- #           Date : %-45s  #\n", TmpString);

	/* Give the call used to generate the model. */
	sprintf(  TmpString, "\"nand3g %ld %ld %ld %ld%s -vhdl -o %s\"" ,
			                         DP_WIDTH                      ,
			                       nand3_width                      ,
			                       nand3_slice                      ,
			                       nand3_drive                      ,
			                      (nand3_msb0     ) ? " -msb0"  : "",
			                       nand3_modelname                  );
    fprintf( Nand3File , "-- # Generator Call : %-45s  #\n", TmpString);

	fprintf( Nand3File, 
"-- ###-------------------------------------------------------------###\n"
			);


	/* Select terminals (and signals) bus indexation. */
	/* (depending to the flag "nand3_msb0")            */
	if (!nand3_msb0)
		sprintf( BusWide, "(%ld downto %ld)", nand3_width-1
				                             +nand3_slice  ,
				                              nand3_slice  );
	else
		sprintf( BusWide,     "(%ld to %ld)", DP_WIDTH-nand3_slice
				                                      -nand3_width  ,
				                              DP_WIDTH-nand3_slice-1);


	/* *** Interface description. *** */
	fprintf( Nand3File, "\nENTITY %s IS\n\tPORT(\n", nand3_modelname);

	/* Terminals of data buses. */
	fprintf( Nand3File, "%14s :  in BIT_VECTOR%s;\n", "i0", BusWide);
	fprintf( Nand3File, "%14s :  in BIT_VECTOR%s;\n", "i1", BusWide);
	fprintf( Nand3File, "%14s :  in BIT_VECTOR%s;\n", "i2", BusWide);
	fprintf( Nand3File, "%14s : out BIT_VECTOR%s;\n",  "o", BusWide);

	/* Supplies terminals. */
	fprintf( Nand3File, "%14s :  in BIT;\n", "vdd");
	fprintf( Nand3File, "%14s :  in BIT\n" , "vss");

	/* *** End of interface description *** */
	fprintf( Nand3File, "\t);\nEND %s;\n\n\n", nand3_modelname);


	/* *** Behavioral view *** */
	fprintf( Nand3File, "ARCHITECTURE behaviour_data_flow OF %s IS\n\n",
			                                            nand3_modelname);
	fprintf( Nand3File, "BEGIN\n\n");
	fprintf( Nand3File, "\to%s <= not (i0%s and i1%s and i2%s);\n\n" , BusWide,
			                                                BusWide,
			                                                BusWide,
			                                                BusWide);

	/* Check supplies connections. */
	fprintf( Nand3File, "\t-- ****************>");
	fprintf( Nand3File, " Supplies Check "      );
	fprintf( Nand3File, "<****************\n\n" );

	fprintf( Nand3File, "\tASSERT((vdd = '1') and (vss = '0'))\n");
	fprintf( Nand3File, "\t\tREPORT \"Power supply is missing on %s.\"\n",
			                                              nand3_modelname);
	fprintf( Nand3File, "\t\tSEVERITY WARNING;\n\n");

	/* *** End of behavioral view *** */
	fprintf( Nand3File, "END behaviour_data_flow;\n");

	/* *** Close the file holding the behavioral view *** */
	fclose( Nand3File);
}


/*  Function  :  'Dgn_NetListNand3'                                */
/* -------------------------------------------------------------- */
/*  Generation of nand3erter/buffer netlist view.                  */

static void  Dgn_NetlistNand3(nand3_modelname,
						     nand3_width    ,
						     nand3_slice    ,
							 nand3_drive    ,
						     nand3_msb0     )
	char *nand3_modelname;  /* Model name.            */
	long  nand3_width;      /* Bus wide.              */
	long  nand3_slice;      /* LSB starting slice.    */
	long  nand3_drive;      /* Output drive capacity. */
	long  nand3_msb0;       /* Zero as MSB index.     */
{
	long       BusIndex;
	long  LeafCellIndex;

	char *LeafCellSliceName;
	char  LeafCellInstaName[C_SUPP_MBKSZNAME];

	char  BusNameI0[C_SUPP_MBKSZNAME];
	char  BusNameI1[C_SUPP_MBKSZNAME];
	char  BusNameI2[C_SUPP_MBKSZNAME];
	char  BusNameO[ C_SUPP_MBKSZNAME];
	char  PinNameI0[C_SUPP_MBKSZNAME];
	char  PinNameI1[C_SUPP_MBKSZNAME];
	char  PinNameI2[C_SUPP_MBKSZNAME];
	char  PinNameO[ C_SUPP_MBKSZNAME];


	/* Select name of the leaf cell model. */
	/* (according to output drive power)   */
	LeafCellSliceName = TablLeafCellNand3[nand3_drive-1];

	/* Creation of the new logical figure. */
	DEF_LOFIG( nand3_modelname);

	/* Name of data bus terminals. */
	if (!nand3_msb0)
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", nand3_width-1+nand3_slice, nand3_slice);
		sprintf( BusNameI1,"i1[%ld:%ld]", nand3_width-1+nand3_slice, nand3_slice);
		sprintf( BusNameI2,"i2[%ld:%ld]", nand3_width-1+nand3_slice, nand3_slice);
		sprintf( BusNameO , "o[%ld:%ld]", nand3_width-1+nand3_slice, nand3_slice);
	}
	else
	{
		sprintf( BusNameI0,"i0[%ld:%ld]", DP_WIDTH-nand3_slice
				                                  -nand3_width  ,
				                          DP_WIDTH-nand3_slice-1);
		sprintf( BusNameI1,"i1[%ld:%ld]", DP_WIDTH-nand3_slice
				                                  -nand3_width  ,
				                          DP_WIDTH-nand3_slice-1);
		sprintf( BusNameI2,"i2[%ld:%ld]", DP_WIDTH-nand3_slice
				                                  -nand3_width  ,
				                          DP_WIDTH-nand3_slice-1);
		sprintf( BusNameO , "o[%ld:%ld]", DP_WIDTH-nand3_slice
				                                  -nand3_width  ,
				                          DP_WIDTH-nand3_slice-1);
	}

 
	/* Creation of figure terminals. */
	/* **** Control terminals **** */
	LOCON( BusNameI0, IN   , BusNameI0);
	LOCON( BusNameI1, IN   , BusNameI1);
	LOCON( BusNameI2, IN   , BusNameI2);
	LOCON( BusNameO ,   OUT, BusNameO );

	/* **** Data bus terminals **** */
	LOCON(     "vdd", INOUT,     "vdd"); 
	LOCON(     "vss", INOUT,     "vss"); 
	
	
	/* Creation of leaf cells instances. */
	/* ("normals" cells of column)       */
	for( LeafCellIndex=nand3_slice;
		 LeafCellIndex<nand3_slice+nand3_width; LeafCellIndex++)
	{
		/* Build leaf cell name. */
		sprintf( LeafCellInstaName, "nand3_%ld", LeafCellIndex);
		
		/* Compute terminal index. */
		if (!nand3_msb0) BusIndex =              LeafCellIndex;
		else            BusIndex = DP_WIDTH -1 -LeafCellIndex;

		/* Build signals names. */
		sprintf( PinNameI0, "i0[%ld]", BusIndex);
		sprintf( PinNameI1, "i1[%ld]", BusIndex);
		sprintf( PinNameI2, "i2[%ld]", BusIndex);
		sprintf( PinNameO ,  "o[%ld]", BusIndex);

		/* Leaf cell instanciation. */
		LOINS( LeafCellSliceName,
			   LeafCellInstaName,
			          PinNameI0 ,
			          PinNameI1 ,
			          PinNameI2 ,
			          PinNameO  , "vdd", "vss", EOL);
	}

	/* Ends and write to disk the new figure. */
	SAVE_LOFIG();
}
