// query.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This file declares a set of classes used to store and transport parameters
// that are set by the user in the dialog process.  These QueryItems and their
// subclasses carry string, int, double, or state flags, from the user
// interface objects to the Requester sublass that has "queried" for them.
// QueryItem instances are stored in Request objects, which are created by the
// Requester subclasses.  They are passed to the dialog objects via the 
// DialogConstructor class (see dialog_ctor.h).

#ifndef QUERY_H
#ifdef __GNUG__
#pragma interface
#endif
#define QUERY_H

#include <ctype.h>
#include <String.h>
#include <InterViews/resource.h>
#include "range.h"

// CharCheck class contains the char test functions for checking user input and
// for scanning default strings for value type (int, double, string)

class CharCheck {
public:
	// control chars and ints only
	static boolean posIntsOnly(const char c) { 
		return (iscntrl(c) || isdigit(c));
	}
	static boolean intsOnly(const char c) {
		return (c == '-' || posIntsOnly(c));
	}
	// control chars, '.',  and ints only
	static boolean posNumsOnly(const char c) {
		return (c == '.' || posIntsOnly(c));
	}
	static boolean numsOnly(const char c) { return (c == '.' || intsOnly(c)); }
	// for entering strings
	static boolean anyChar(const char) { return true; }
	static boolean isIntegerString(const char*);
	static boolean isDoubleString(const char*);
private:
	virtual void dontAllowInstantiation()=0;
};

class String;

// base class for all query items, and also used for plain labels

class QueryItem : public Resource {
public:
	enum Type { Label = 0, Browser, Choice, Value, RangedValue, Button };
public:
	QueryItem(const char *);
	QueryItem(const char *, const char *);
	virtual ~QueryItem();
	virtual const char* label();
	virtual Type type() { return Label; }
protected:
	static const char* separator;	// token for separating labels
private:
	String *myLabel;
};

// class for file browser

class QueryValue;

class QueryFile : public QueryItem {
	typedef QueryItem Super;
public:
	QueryFile(const char *path, const char* suffixes);
	virtual ~QueryFile();
	redefined Type type() { return Browser; }
	QueryValue* path() { return myPath; }
	const char* suffixes() { return *mySuffixes; }
private:
	QueryValue *myPath;
	String *mySuffixes;
};

// class for response buttons (confirm, cancel, etc.)

class QueryButton : public QueryItem {
	typedef QueryItem Super;
public:
	QueryButton(const char *, Response resp=Yes);
	virtual ~QueryButton();
	redefined Type type() { return Button; }
	Response response() { return myResponse; }
	boolean set(Response r) { myResponse = r; return true; }
private:
	Response myResponse;
};

// abstract class for all returned value types

class QueryReturn : public QueryItem {
	typedef QueryItem Super;
public:
	QueryReturn(const char *lbl) : QueryItem(lbl) {}
	virtual ~QueryReturn() {}
	virtual const char* value() const = 0;
	virtual boolean set(const char *) = 0;
	virtual operator const char * () { return value(); }
	void operator = (const QueryReturn& rhs) { set(rhs.value()); }
protected:
	QueryReturn() : QueryItem("") {}
	char tmpbuf[128];
private:
	friend class ValueArray;
	virtual int size() { return sizeof(*this); }
};

// class for querying sets of boolean options

class QueryChoice : public QueryReturn {
	typedef QueryReturn Super;
public:
	QueryChoice(const char *, const char *list, unsigned s, boolean ex);
	QueryChoice();
	virtual ~QueryChoice();
	redefined Type type() { return Choice; }
	redefined const char* value() const;
	redefined boolean set(const char *);
	boolean areExclusive() { return exclusive; }
	void setState(unsigned state) { myState = state; }
	int numberOfValues() { return myList.freq(separator) - 1; }
	void start();
	const char *nextLabel();
	boolean nextValue(int *);
	int state() const { return int(myState); }
	operator int () const { return state(); }
	boolean operator == (int rhs) { return state() == rhs; }
	void operator = (const QueryReturn& rhs) { set(rhs.value()); }
private:
	friend class ValueArray;
	redefined int size() { return sizeof(*this); }
private:
	String myList;		// encoded list of button labels
	String iterator;	// for extracting individual labels
	String currentLabel;
	unsigned myState;	// hex representation of states
	unsigned stateIterator;
	unsigned currentValue;
	boolean exclusive;	// only one button selection at a time?
};

// class for querying string, int, or double values

class QueryValue : public QueryReturn {
	typedef QueryReturn Super;
public:
	QueryValue(const char *, const char *, CharCheckFun ccf);
	QueryValue(const char *s = "");
	QueryValue(const char *, int, CharCheckFun ccf);
	QueryValue(int);
	QueryValue(const char *, double, CharCheckFun ccf);
	QueryValue(double);
	virtual ~QueryValue();
	redefined Type type() { return Value; }
	redefined const char* value() const;
	redefined boolean set(const char *);
	boolean checkCharacter(char c) { return (*checkFun)(c); }
	virtual operator int () const { return intValue(); }
	virtual operator float () const { return float(doubleValue()); }
	virtual operator double () const { return doubleValue(); }
	boolean operator == (int rhs) { return intValue() == rhs; }
	boolean operator == (double rhs) { return doubleValue() == rhs; }
	void operator = (const QueryReturn& rhs) { set(rhs.value()); }
protected:
	CharCheckFun checkFun;
	int intValue() const;
	double doubleValue() const;
	boolean isGood(const char *s) { return (s && strlen(s)); }
private:
	String *string;
	friend class ValueArray;
	redefined int size() { return sizeof(*this); }
};

class RangedQueryValue : public QueryValue {
public:
	RangedQueryValue(const char *, int, const Range &);
	RangedQueryValue(const char *, double, const Range &);
	redefined Type type() { return RangedValue; }
	Range range() { return valueRange; }
	boolean integerRange() { return (checkFun == CharCheck::intsOnly); }
private:
	Range valueRange;
	friend class ValueArray;
	redefined int size() { return sizeof(*this); }
};

class ValueArray {
public:
	ValueArray(QueryReturn *qr, int len) : array(addr(qr)), myLength(len) {
		offset = qr->size();
	}
	QueryReturn & operator [] (int index) const {
		index = index < myLength ? index : myLength - 1;
		return *((QueryReturn *)(array + offset * index));
	}
	int length() const { return myLength; }
private:
	addr array;
	int myLength;
	int offset;
};

// these all exist without inheritance because compilers do not support
// statically declared arrays of derived classes

struct QueryLabelInfo {
	char *label;
};

struct QueryButtonInfo {
	char *label;				// label for button
	Response response;			// behavior for this button (yes, no, cancel)
};

struct QueryValueInfo {
	char *label;				// label for text input box
	char *defaultVal;			// starting value
	CharCheckFun check;			// ptr to static fun to check typed input
	char *valueRange;			// range of acceptable values ("|" - separated, optional)
};

struct QueryChoiceInfo {
	char *label;				// main label for set of choice buttons
	char *choiceLabels;			// "|" - separated list of labels
	unsigned initialStates;		// hex number showing states
	boolean exclusive;			// flag for shared or separate states
};

struct QueryInfo {
	QueryLabelInfo *labelInfo;		// labels
	char *valueListLabel;
	QueryValueInfo *valueInfo;		// text entry items
	QueryChoiceInfo *choiceInfo;	// choice buttons
	QueryButtonInfo *buttonInfo;	// response buttons
	Response defaultResponse;
};

#endif
