// adjtext.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/perspective.h>
#include <InterViews/shape.h>
#include "adjtext.h"
#include "query.h"
#include "range.h"

AdjustableText::AdjustableText(ButtonState *st, const char *samp,
		QueryValue *value) : TextInput(st, samp, value) {
	initialize();
}

AdjustableText::AdjustableText(const char* name, ButtonState *st,
	const char *samp, QueryValue *value) : TextInput(name, st, samp, value) {
	initialize();
}

void
AdjustableText::initialize() {
	perspective = new Perspective;
	perspective->Init(0, 0, 1, 1);
	perspective->curwidth = perspective->curheight = 1;
}

AdjustableText::~AdjustableText() {
	Resource::unref(perspective);
}

void
AdjustableText::Reconfig() {
	TextInput::Reconfig();
	shape->Rigid(0, 0, 0, 0);	// disallow horiz changes as well
}

void
AdjustableText::Adjust(Perspective& np) {
	constrain(np);
	doAdjust(np);
	perspective->Update();
	setText();
}

inline int
AdjustableText::lower() const { return perspective->x0; }

inline int
AdjustableText::upper() const {
	return perspective->x0 + perspective->width - 1;
}

inline int
AdjustableText::current() const { return perspective->curx; }

// only allows changes in cur{x,y}

void
AdjustableText::constrain(Perspective& np) {
	register Perspective* p = perspective;
	int newx = np.curx;
	int newy = np.cury;
	np = *p;
	p->curx = np.curx = max(lower(), min(upper(), newx));
	p->cury = np.cury = newy;
}

void
AdjustableText::setPerspectiveBounds(int lower, int upper) {
	perspective->x0 = lower;
	perspective->width = upper - lower + 1;
	// small incr.  is 1/100 total, or 1
	perspective->sx = max(1, int(perspective->width / 1000.0));
	// large incr.  is 1/10 total, or small incr.
	perspective->lx = max(perspective->sx, int(perspective->width / 10.0));
	perspective->Update();
}

void
AdjustableText::setPerspectiveCurrent(int current) {
	perspective->curx = max(lower(), min(upper(), current));
	perspective->Update();
}

// ********

AdjustableNumber::AdjustableNumber(
		ButtonState *s, const char *samp, QueryValue *v)
	: AdjustableText(s, samp, v) {
}

AdjustableNumber::AdjustableNumber(
		const char* name, ButtonState *s, const char *samp, QueryValue *v)
	:  AdjustableText(name, s, samp, v) {
}

// ********

AdjustableInteger::AdjustableInteger(
			ButtonState *s, QueryValue *v, const Range& range)
		: AdjustableNumber(s, "000000", v) {
	initialize(range);
}

void
AdjustableInteger::initialize(const Range& range) {
	setBounds(range);
	setCurrent(Text());
}
	
void
AdjustableInteger::setBounds(const Range &bounds) {
	setPerspectiveBounds(bounds.intMin(), bounds.intMax());
}

int
AdjustableInteger::setCurrent(const char* str) {
	setPerspectiveCurrent((str && strlen(str)) ? atoi(str) : 0);
	return true;
}

void
AdjustableInteger::doAdjust(Perspective &np) {
}

int 
AdjustableInteger:: integerValue() { return current(); }

double
AdjustableInteger::doubleValue() { return current(); }

const char *
AdjustableInteger::stringValue() {
	static char string[32];
	sprintf(string, "%d", current());
	return string;
}

// ******

AdjustableDouble::AdjustableDouble(
			ButtonState *s, QueryValue *v, const Range& range)
		: AdjustableNumber(s, "000000000000", v) {
	initialize(range);
}

void
AdjustableDouble::initialize(const Range& range) {
	setBounds(range);
	setCurrent(Text());
}
	
// setting the bounds this high allows arbitrary preciseness of input between
// the actual value boundries (i.e., avoids quantization of values)

void
AdjustableDouble::setBounds(const Range &bounds) {
	setPerspectiveBounds(0, 1000000);
	valueRange = bounds;
}

int
AdjustableDouble::setCurrent(const char* str) {
	double currentValue = (str && strlen(str)) ? atof(str) : 0.0;
	double pcnt = (currentValue - valueRange.min()) / valueRange.spread();
	int location = lower() + int(pcnt * (upper() - lower()));
	setPerspectiveCurrent(location);
	return true;
}

double
AdjustableDouble::fraction() {
	double low = lower();
	double total = double(upper()) - low;
	return (double(current()) - low) / total;
}

int
AdjustableDouble::integerValue() {
	return int(doubleValue());
}

double
AdjustableDouble::doubleValue() {
	return valueRange.min() + (fraction() * valueRange.spread());
}

void
AdjustableDouble::doAdjust(Perspective &np) {
}

const char *
AdjustableDouble::stringValue() {
	static char string[64];
	sprintf(string, "%f", doubleValue());
	return string;
}
