// soundeditor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <X11/keysym.h>
#include "application.h"
#include "combfilter.h"
#include "controller.h"
#include "ellipfilt.h"
#include "datamodifier.h"
#include "fftfun.h"
#include "filecommand.h"
#include "filename.h"
#include "iallpole.h"
#include "interpolater.h"
#include "lowpassfilter.h"
#include "lpcdata.h"
#include "formantfilter.h"
#include "formantsynth.h"
#include "optionsetter.h"
#include "pitchtracker.h"
#include "pvanalyzer.h"
#include "pvsynthesizer.h"
#include "reson.h"
#include "sound.h"
#include "soundeditor.h"

Data *
SoundEditor::newModel() { return new Sound(); }

boolean
SoundEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_asciitilde:
		removeDC_Component();
		break;
	case XK_q:
		resonFilter();
		break;
	case XK_K:
		combFilter();
		break;
	case XK_H:
		ellipticalFilter();
		break;
	case XK_F:
		lpcFilter();
		break;
	case XK_KP_1:
		resynthesizeFromLPC();
		break;
	case XK_KP_2:
		resynthesizeFromPvoc();
		break;
	case XK_p:
		controller->play();
		break;
	case XK_colon:
		controller->record();
		break;
	case XK_BackSpace:
	case XK_Delete:
		controller->stop();
		break;
	case XK_parenright:
		controller->configureConverter();
		break;
	case XK_dollar:
		rescanForPeak();
		break;
	case XK_ampersand:
		changeFormat();
		break;
	case XK_k:
		changeSampleRate();
		break;
	case XK_3:
		FFTAnalysis();
		break;
	case XK_4:
		LPCAnalysis();
		break;
	case XK_5:
		pitchTrackAnalysis();
		break;
	case XK_6:
		LPCAndPitchTrackAnalysis();
		break;
	case XK_7:
		PVocAnalysis();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

void
SoundEditor::newFile() {
	SoundEditor::new_File(controller);
}

void
SoundEditor::saveToFile() {
	SoundFileSaver sfs(defaultDir(), model());
	applyModifier(sfs);
}

// static constructor

Controller *
SoundEditor::new_File(Controller* controller) {
	SoundCreator sc;
	applyModifierUsing(sc, controller);
	return sc.getNewController();
}

// new or redefined public functions for sounds only

void
SoundEditor::removeDC_Component() {
	// run filter with cutoff starting at 20 hz, 60db down at 5hz
	EllipticalFilter e(currentSelection(), 20.0, 5.0);
	applyModifier(e);
}

void
SoundEditor::interpolate() {
	setCopyBuffer(currentSelection()->newData(1));
	Transposer t(currentSelection(), copyBuffer());
    applyModifierToNew(t);
}

void
SoundEditor::insertSpace() {
	TimeInsert t(currentSelection());
	applyModifier(t);
}

void
SoundEditor::changeLength() {
	DurChanger d(model());
	if(applyModifier(d))
		reselect();
}

void
SoundEditor::lowPassFilter() {
	LowPassFilter l(currentSelection());
	applyModifier(l);
}

void
SoundEditor::resonFilter() {
	Reson3 r(currentSelection());
	applyModifier(r);
}

void
SoundEditor::combFilter() {
	// output will be new floating point resized by comb object
	setCopyBuffer(
		new Sound(1, soundModel()->sRate(),
			currentSelection()->channels(), FloatData)
	);
	CombFilter c(currentSelection(), copyBuffer());
	applyModifierToNew(c);
}

void
SoundEditor::ellipticalFilter() {
	EllipticalFilter e(currentSelection());
	applyModifier(e);
}

void
SoundEditor::lpcFilter() {
	LPCData* lpc = (LPCData *) currentSource();
	if(lpc != nil) {
			FormantFilter f(currentSelection(), currentSelection(), lpc);
			applyModifier(f);
	}
	freeSource((Data *) lpc);
}

void
SoundEditor::resynthesizeFromLPC() {
	LPCData* lpc = (LPCData *) currentSource();
	if(lpc != nil) {
			FormantSynthesizer f(currentSelection(), lpc);
			applyModifier(f);
	}
	freeSource((Data *) lpc);	// avoid having to #include lpcdata.h
}

void
SoundEditor::rescanForPeak() {
	boolean old = Data::deferRescan(false);
	model()->Notify();	// this automatically does a scan
	Data::deferRescan(old);
}

void
SoundEditor::rescale() {
	Application::inform("Rescaling...");
	soundModel()->rescale();
	reselect();
}

void
SoundEditor::changeFormat() {
	FormatChanger f(soundModel());
	if(applyModifier(f)) {
		reselect();
		controller->setFileName(
			FileName::changeSuffix(
				controller->fileName(), soundModel()->fileSuffix()
			)
		);
	}
}

void
SoundEditor::LPCAnalysis() {
	InverseAllPole analyzer(currentSelection());
	if(applyModifier(analyzer)) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(controller->world());
	}
}

void
SoundEditor::pitchTrackAnalysis() {
	PitchTracker analyzer(currentSelection());
	if(applyModifier(analyzer)) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(controller->world());
	}
}

void
SoundEditor::LPCAndPitchTrackAnalysis() {
	LPCAnalysis();
	pitchTrackAnalysis();
}

void
SoundEditor::FFTAnalysis() {
	FFT_Function analyzer(currentSelection());
	if(applyModifier(analyzer)) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(controller->world());
	}
}

void
SoundEditor::PVocAnalysis() {
	PVAnalyzer pva(currentSelection());
	if(applyModifier(pva)) {
		Controller *newctlr = new Controller(pva.getAnalysis());
		newctlr->display(controller->world());
	}
}

void
SoundEditor::resynthesizeFromPvoc() {
	Data* pvdata = currentSource();
	if(pvdata != nil) {
		if(pvdata->isA(Pvoc_Data)) {
			PVSynthesizer pvs(currentSelection(), (PvocData *) pvdata);
			applyModifier(pvs);
		}
		else
			Application::alert("Selected source for pvoc resynthesis",
				"is not a Phase Vocoder data file.");
	}
	freeSource(pvdata);
}

void
SoundEditor::setDataOptions() {
	SoundOptionSetter options;
	applyModifier(options);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
SoundEditor::new_DataEditor1(Controller *c) {
	return new SoundEditor(c);
}

DataEditor *
SoundEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new SoundEditor(c, d);
}
