// dialog_ctor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/border.h>
#include <InterViews/box.h>
#include <InterViews/button.h>
#include <InterViews/frame.h>
#include <InterViews/glue.h>
#include <InterViews/message.h>
#include <InterViews/painter.h>
#include "application.h"
#include "buttonarray.h"
#include "eventforwarder.h"
#include "query.h"
#include "request.h"
#include "dialogbox.h"
#include "fileselector.h"
#include "textinput.h"
#include "dialog_ctor.h"

// ResponseButton is a private subclass of PushButton whose label is
// highlighted when the particular instance is the default response to a 
// query, i.e., the one that will be activated if the user presses <return>

class ResponseButton : public PushButton {
public:
	ResponseButton(const char* text, ButtonState* s, int v,
			boolean isDefault=false)
		: PushButton(text, s, v), is_default(isDefault) {}
	ResponseButton(const char* text, ButtonState* s, void* v,
			boolean isDefault=false)
		: PushButton(text, s, v), is_default(isDefault) {}
	virtual ~ResponseButton() {}
	redefined void Refresh();
protected:
	boolean is_default;	// is this button the default for <return>
};

void
ResponseButton::Refresh() {
	int oldStyle = output->GetStyle();
	// button that is pressed when <return> is hit has bold font
	if(is_default) output->SetStyle(Boldface);
	PushButton::Refresh();
	if(is_default) output->SetStyle(oldStyle);
}

//********

DialogConstructor* DialogConstructor::ctorInstance = nil;

DialogConstructor*
DialogConstructor::getInstance() {
	if (ctorInstance == nil)
		ctorInstance = new DialogConstructor;
	return ctorInstance;
}

void
DialogConstructor::destroyInstance() {
	delete ctorInstance;
}

DialogBox *
DialogConstructor::createDialog(Interactor* under, Request &request) {
	typedef void (DialogBox::*(DialogForwardFunction))(Event&);
	class DialogEventForwarder : public EventForwarder {
	public:
		DialogEventForwarder(DialogBox* db, DialogForwardFunction dff)
			: myDialog(db), myFunction(dff) {}
		redefined void handle(Event &e) {
			(myDialog->*myFunction)(e);
		}
	private:
		DialogBox* myDialog;
		DialogForwardFunction myFunction;
	};
	// all confirm, deny buttons are tied to this state
	ButtonState* state = new ButtonState(NullResponse);
	VBox* interior = new VBox;
	DialogBox *dialog = nil;
	Response response = request.buttonList->defaultResponse();
	switch(request.type()) {
	case AlertType:
		dialog = new Alert(state, under, response);
		break;
	case ConfirmerType:
		dialog = new Confirmer(state, under, response);
		break;
	case ChoiceType:
		dialog = new ChoiceDialog(state, under, response);
		break;
	case InputType:
		dialog = new InputDialog(state, under, response);
		break;
	default:
		Application::die("Unknown dialog type!");
		break;	// not reached
	}
	if(request.hasLabels()) {
		interior->Insert(messageBox(request.labelList));
	}
	if(request.hasBrowser()) {
		interior->Insert(new HBorder);
		Scene* selector = fileSelector(request.browserQuery, state);
		interior->Insert(selector);
		FileSelector* fs = (FileSelector *) selector;
		fs->setForwarder(
			new DialogEventForwarder(dialog, &DialogBox::eventLoop)
		);
		((InputDialog *) dialog)->addFileSelector(fs);
	}
	if(request.hasValues()) {
		interior->Insert(new HBorder);
		interior->Insert(textInputBox((InputDialog *) dialog,
			request.valueList, state));
	}
	if(request.hasChoices()) {
		interior->Insert(new HBorder);
		interior->Insert(choiceButtonBox(request.choiceList, state));
	}
	interior->Insert(new HBorder);
	interior->Insert(new VGlue(5, 0));	// spacer above buttons

	// all dialogs have a button box
	interior->Insert(buttonBox(request.buttonList, state));
	interior->Insert(new VGlue(2, 0));	// spacer below buttons
	dialog->Insert(interior);
	dialog->setBell(request.useBell());
	Resource::unref(state);
	return dialog;
}

Scene *
DialogConstructor::messageBox(QueryList *list) {
	VBox* mesgBox = new VBox;
	mesgBox->Align(Center);
	for(list->start(); list->more(); list->next()) {
		mesgBox->Insert(new VGlue);
		mesgBox->Insert(
			new HBox(
				new HGlue(5),
				new Message("Title", list->query()->label()),
				new HGlue(5)
			)
		);
	}
	mesgBox->Insert(new VGlue);
	return mesgBox;
}

Scene *
DialogConstructor::fileSelector(QueryFile *qfile, ButtonState* response) {
	return new FileSelector(
		response, qfile->path(), qfile->suffixes(), 15, 32
	);
}

Scene *
DialogConstructor::textInputBox(InputDialog *ipd, ValueList *list,
		ButtonState *bs) {
	VBox* textBox = new VBox;
	const char* inputSubtitle = list->label();
	// insert text entry subtitle before inserting text entry items
	if(inputSubtitle != nil && strlen(inputSubtitle) > 0) {
		textBox->Insert(new VGlue);
		textBox->Insert(
			new HBox(
				new HGlue(10, 0),
				new Message("Subtitle", inputSubtitle)
			)
		);
		textBox->Insert(new VGlue);
	}
	TextInput* lasttxt = nil;
	for(list->start(); list->more(); list->next()) {
		register QueryValue *qvalue = list->query();
		textBox->Insert(new VGlue);
		const char* samp = "MMMMMMMMMMMMMM";
		TextInput* txt = new TextInput(bs, samp, qvalue);
		// add current TextInput to the linked list of TextInputs
		if(lasttxt != nil)
			lasttxt->appendTo(txt);
		else
			ipd->addTextInput(txt);
		lasttxt = txt;
		textBox->Insert(
			new HBox(
				new HGlue(10, 0),
				new Message("TextEntryLabel", qvalue->label()),
				new HGlue(5, 0),
				new Frame(txt, 1)
			)
		);
	}
	textBox->Insert(new VGlue);
	return textBox;
}

Scene *
DialogConstructor::choiceButtonBox(ChoiceList *list, ButtonState* state) {
	VBox* outerBox = new VBox;
	boolean notFirst = false;
	for(list->start(); list->more(); list->next()) {
		if(notFirst) outerBox->Insert(new HBorder);
		outerBox->Insert(new ButtonArray(state, list->query()));
		notFirst = true;
	}
	outerBox->Insert(new VGlue);
	return outerBox;
}

// create set of buttons initialized with appropriate responses (Yes, No, etc.)

Scene *
DialogConstructor::buttonBox(ButtonList *list, ButtonState *s) {
	HBox* btnBox = new HBox;
	for(list->start(); list->more(); list->next()) {
		btnBox->Insert(new HGlue);
		Response resp = list->query()->response();
		Interactor *button = new ResponseButton(
			list->query()->label(),
			s,
			int(resp),
			resp == list->defaultResponse()	// true if is default
		);
		btnBox->Insert(button);
	}
	btnBox->Insert(new HGlue);
	return btnBox;
}

