/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * so.c
 ******************************************************************************/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>

#include "pygmalion.h"
#include "sysdef.h"
#include "soconfig.h"
#include "pgmrc.h"

#include "sodef.h"
#include "built_in_fn.h"
#include "util.h"
#include "fontdefs.h"

/* --------------- functions defined for this algorithm ------------------- */
int	connect ();
int	build_rules ();

float	learn ();
float	recall ();

/* --------------- globals ------------------------------------------------ */
int		c_net = 0;				/* current net */
int		I_LAYER, O_LAYER;			/* input, output layers */
char		system_filename[ FILESIZE ];

float		*input_vector;				/* to receive input pattern */
float		*target_vector;				/* to receive target pattern */

 /* input_pattern_control may be overridden from parameter NET_P_input_control */
int		input_pattern_control	= INPUT_PATTERN_CONTROL;
int		target_pattern_control	= TARGET_PATTERN_CONTROL;
int		in_size, out_size;		/* size of input, output layers */

int		x_dimension = X_DIMENSION;	/* used to determine neighbourhood */
int		y_dimension = Y_DIMENSION;
/* ------------------------------------------------------------------------ *\
	Definions of the SOM
\* ------------------------------------------------------------------------ */

int	state_update();
int	weight_update();
int	pick_winner();
int	decrement_distance();
int	increment_pattern();
int	read_input_rule();
int	test_distance();
int	calculate_score();
float	distance();

/* class_type structures		    function		fn_name			gen.	ext. */
/* NET level */
class_type learn_meta_class		= { sexec_r,		"sexec_r",		0,	2 };
class_type learn_pattern_meta_class	= { sexec_c,		"sexec_c",		0,	4 };
class_type recall_meta_class		= { sexec,		"sexec",		0,	2 };
class_type pick_winner_meta_class	= { sexec,		"sexec",		0,	1 };
class_type weight_update_meta_class	= { sexec,		"sexec",		0,	1 };
class_type state_update_meta_class	= { sexec,		"sexec",		0,	1 };
class_type pick_winner_class		= { pick_winner,	"pick_winner",		1,	1 };
class_type decrement_distance_class	= { decrement_distance,	"decrement_distance",	0,	0 };
class_type increment_pattern_class	= { increment_pattern,	"increment_pattern",	0,	0 };
class_type read_input_rule_class	= { read_input_rule,	"read_input_rule",	1,	0 };
/* NEURON level */
class_type weight_update_class		= { weight_update,	"weight_update",	4,	2 };
class_type state_update_class		= { state_update,	"state_update",		2,	2 };
class_type calculate_score_class	= { calculate_score,	"calculate_score",	1,	2 };

/* ------ specify maximum counts of rules and parameters at all levels ---- */

int	xc	[ 6 ][ 2 ] = {	
/* extra counts rules,	params	   level	*/
		0,	0,	/* system	*/
		10,	14,	/* net		*/
		0,	0,	/* layer	*/
		0,	0,	/* cluster	*/
		3,	0,	/* neuron	*/
		0,	0	/* synapse	*/
};

/* ------ names for parameters -------------------------------------------- */

char	*pname [ 6 ] [ 14 ] = {

		{"","","","","","","","","","","","","",""},
		{	"score",
			"winner",
			"gain",
			"gain_start",
			"gain_step",
			"gain_finish",
			"distance",
			"distance_start",
			"distance_step",
			"distance_finish",
			"distance_control",
			"input_control",
			"current_pattern",
			"x_dimension"
		},
		{"","","","","","","","","","","","","",""},
		{"","","","","","","","","","","","","",""},
		{"","","","","","","","","","","","","",""},
		{"","","","","","","","","","","","","",""}
};
/* ------------------------------------------------------------------------ */


/* ------------------------------------------------------------------------ */
/*	external functions and variables                                    */
/* ------------------------------------------------------------------------ */

/* -------- declarations from pgmrc.c ------------------------------------- */

extern	char		*rc [];

/* -------- declarations from pattern.c ----------------------------------- */

extern	int		init_patterns();
extern	int		count_patterns();
extern	int		read_pattern();
extern	pat_elem	*load_patterns();
extern	pat_elem	*indextop();
extern	void		add_to_pattern_list();
extern	int		read_input();
extern	int		read_target();
extern	int		read_xfile();

extern	pat_elem	*pattern_list;		/* patterns loaded for net */
extern	pat_elem	*current_pattern;	/* to pass widths to show_net */
extern	int		pattern_count;		/* how many */
extern	int		pattern_index;		/* last one read */
extern	char		pattern_filename[];

/* -------- declarations from jload.c ------------------------------------- */

extern	int		sys_load();

/* -------- declarations from jsave.c ------------------------------------- */

extern	int		sys_save();

/* -------- declarations from shownet.c ----------------------------------- */

void			shownet();

/* -------- declarations from util.c -------------------------------------- */

extern system_type	*sys;
extern int		int_mode;
extern int		cycles;
extern float		rand_scale;
extern	char		*strstr();

/* -------- declarations from built_in_fn.c ------------------------------- */

extern	char		*jalloc();
extern	int		sexec();
extern	int		sexec_r();
extern	int		sexec_c();
extern	int		pexec();
extern	int		pexec_r();
extern	int		pexec_c();

extern int		err_cnt;

/* -------- declarations from alloc.c ------------------------------------- */

extern	system_type	*sys_alloc();

/* ------------------------------------------------------------------------ */
/*  Function bodies for the back-propagation rule classes                   */
/* ------------------------------------------------------------------------ */

float learn()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_learn ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
float recall()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_recall ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
int read_input_rule (p)	/* read i'th input pattern */
TAGVAL 	**p;
{
	int	i = (int) p[ 0 ]->value.f;

	read_input(i);
	return( 0 );
}

/* ------------------------------------------------------------------------ */
float distance (i, j)	/* calculate distance between two neurons in output layer */
int 	i,j;
{
	float	x1, x2, y1, y2;

	x1 = (float) ( i%x_dimension );
	x2 = (float) ( j%x_dimension );
	y1 = (float) ( i/x_dimension );
	y2 = (float) ( j/x_dimension );

	switch	( (int) sys->net[c_net]->parameters[ NET_P_distance_control ].parameter.value.f ) {

	case	EUCLIDEAN:
 		return ( sqrt ( (x2 - x1)*(x2 - x1) + (y2 - y1)*(y2 - y1)));

	case	CARTESIAN:
		return ( MAX ( fabs(x2 - x1), fabs(y2 - y1)));

	case	CITY_BLOCK:
		return ( fabs(x2 - x1) + fabs(y2 - y1) );

	default:
		printf ( "invalid distance control\n" );
		exit ( 1 );
		break;
	}
}

/* ------------------------------------------------------------------------ */
decrement_distance()	/* reduce distance and gain */
{
	if ( sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f <= 0.0 ) {
		sys->net[c_net]->parameters[ NET_P_winner ].parameter.value.f = -1.0;
		return ( TERM );		
	}

	sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f -=
		sys->net[c_net]->parameters[ NET_P_gain_step ].parameter.value.f;

	if ( sys->net[c_net]->parameters[ NET_P_gain_step ].parameter.value.f > 0.0  ) {
		if ( sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f <
			  sys->net[c_net]->parameters[ NET_P_gain_finish ].parameter.value.f ) {
				sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f =
				sys->net[c_net]->parameters[ NET_P_gain_finish ].parameter.value.f;
			  }
	}
	else {	
		if ( sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f >
			sys->net[c_net]->parameters[ NET_P_gain_finish ].parameter.value.f ) {
				sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f =
				sys->net[c_net]->parameters[ NET_P_gain_finish ].parameter.value.f;
		  	}
	}
	sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f -=
	sys->net[c_net]->parameters[ NET_P_distance_step ].parameter.value.f;

	if ( sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f < 0.0 ) {
		sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f = 0.0;
	}

#ifdef	DEBUG_SOM
	printf ( "decrement_distance/ distance = %f, gain = %f\n", 
		sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f,
		sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f );
#endif
	return ( 0 );
}
/* ------------------------------------------------------------------------ */
increment_pattern()	/* add one to pattern parameter and test against pattern_count */
{
	sys->net[c_net]->parameters[ NET_P_current_pattern].parameter.value.f += 1.0 ;

	if ( sys->net[c_net]->parameters[ NET_P_current_pattern ].parameter.value.f >= (float) pattern_count ) {
		sys->net[c_net]->parameters[ NET_P_current_pattern].parameter.value.f = 0.0 ;
		return ( TERM );
	}
	return ( 0 );
}
/* ------------------------------------------------------------------------ */
/* calculate_score_class */

int calculate_score (p)
TAGVAL **p;        /* [*state[ N_ACC ], *SIZE, *state1, *weight1, ... ] */
{
	int	i, size = *(int *)p[ 1 ];
	float	f;

	p[ 0 ]->value.f = 0.0;

	PAR for (i = 2; i < ( 2 * size + 2); i += 2) {
		f = p[ i ]->value.f - p[ i + 1 ]->value.f;
		p[ 0 ]->value.f += f * f;
	}
#ifdef	DEBUG_SOMX
	printf ( "calculate_score/ %f\n", p[ 0 ] ->value.f );
#endif
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* pick_winner_class		select neuron with minimum score */

int pick_winner (p)
TAGVAL **p;        /* [*winner, *SIZE, *state[ N_ACC ]0, *state[ N_ACC ]1, ... ] */
{
	int	size		= *(int *)p[ 1 ];
	int	i, winner	= 0;	/* assume winner is number 0 */
	float	f		= p[ 2 ]->value.f;
	float	d,maxd		= 0.0;
	float	distance();
	
	for (i = 3; i < size+2 ; i++) {
		if ( p[ i ]->value.f < f ) {
			f 	= p[ i ]->value.f;
			winner	= i - 2;
		}
	}

	if ( p[ 0 ]->value.f < 0.0 ) {
		if ( (d = distance( winner, 0 ) ) > maxd ) {
			maxd = d;
		}
		if ( (d = distance( winner, ( x_dimension - 1 ) ) ) > maxd ) {
			maxd = d;
		}
		if ( (d = distance( winner, ( y_dimension - 1 ) * x_dimension  ) ) > maxd ) {
			maxd = d;
		}
		if ( (d = distance( winner, y_dimension * x_dimension - 1 ) ) > maxd ) {
			maxd = d;
		}
#ifdef	DEBUG_SOM
		printf ( "pick_winner/ neighbourhood recalculated : distance = %f\n", maxd );
#endif
		sys->net[c_net]->parameters[ NET_P_distance ].parameter.value.f = maxd;
		sys->net[c_net]->parameters[ NET_P_gain ].parameter.value.f =
			sys->net[c_net]->parameters[ NET_P_gain_start ].parameter.value.f;
	}

#ifdef	DEBUG_SOM
	printf ( "pick_winner/ winner =  %d\n", winner );
#endif
	p[ 0 ]->value.f = (float) winner;
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* state_update_class */

int state_update (p)
TAGVAL **p;        /* [*state, *acc, *SIZE, *state1, *weight1, ... ] */
{
	float	scale = sqrt( (float) out_size );

 	p[ 1 ]->value.f = scale * (-0.5 + dp( p + 2 ) * scale );
  	p[ 0 ]->value.f = 1.0 / (1.0 + exp(- p[ 1 ]->value.f));
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* weight_update_class */

int weight_update (p)
TAGVAL **p;        /* [*distance, *winner, *"index", *gain, *SIZE, *state1, *weight1, ... ] */
{
	float		d1, d2		= p[ 0 ]->value.f;
	int		winner		= ( int ) p[ 1 ]->value.f;
	neuron_type	*n		= ( neuron_type * ) p[ 2 ];	/* bit of a fiddle */
	float		gain		= p[ 3 ]->value.f;
	int		i, size		= *(int *)p[ 4 ];
	int		index 		= n->route[ 3 ];

	if ( ( d1 = distance ( winner, index ) ) < d2 ) {

		PAR for (i = 5; i < ( 2 * size + 5); i += 2) {
			p[ i + 1 ]->value.f += gain * ( p[ i ]->value.f - p[ i + 1 ]->value.f ) ;
		}
	}
	return ( 0 );
}

/* ------------------------------------------------------------------------ */

/* SOM connect */

int connect (conf)
int	conf[];
{
	int		cn, i, j, k, l, m, p, s, x, y, index;
	caddr_t		*free_pointer1, *free_pointer2;
	rule_type	*rp;
	para_type	*pp;
	synapse_type	*sp;

	sys = sys_alloc(&conf);		/** allocates memory for networks and neurons **/

	/* allocate memory for the rules, meta rules and parameters */

	for ( x = SYS; x < SYN; x++ ) {

		y = xc [x] [0];
		if ( y ) {	/* rules */

			s = sizeof (rule_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_rules = y;
					sys->rules = (rule_type *) jalloc (s, y);
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_rules = y;
						sys->net[cn]->rules = (rule_type *) jalloc (s, y);
					}
					break;

				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					rp = (rule_type *) jalloc ( s, y * sys->net[cn]->layers);
					for (i=0; i < sys->net[cn]->layers; i++, rp += y) {
						sys->net[cn]->layer[i]->n_rules = y;
						sys->net[cn]->layer[i]->rules = rp;
					}
					}

				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					rp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, rp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->rules = rp;
					}
					}
					}

				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					rp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, rp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules = rp;
					}
					}
					}
					}

				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid count\n");
					exit(1);
					break;
			}
		}

		y = xc [x] [1];
		if ( y ) {	/* Parameters */

			s = sizeof (para_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_parameters = y;
					sys->parameters = (para_type *) jalloc (s, y);
					for ( p = 0; p < y; p++ ) {
						sys->parameters[p].name = pname [x] [p];
 					}
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_parameters = y;
						sys->net[cn]->parameters = (para_type *) jalloc (s, y);
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->parameters[p].name = pname [x] [p];
	 					}
					}
					break;

				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layers);
					for (i=0; i < sys->net[cn]->layers; i++, pp += y) {
						sys->net[cn]->layer[i]->n_parameters = y;
						sys->net[cn]->layer[i]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					break;

				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					break;

				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					}
					break;
			
				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid count\n");
					exit(1);
					break;
			}
		}
	}

	for ( cn = 0; cn < sys->nets; cn++ ) {

	sys->net[cn]->parameters[ NET_P_winner ].parameter.value.f		= (float) WINNER;
	sys->net[cn]->parameters[ NET_P_distance ].parameter.value.f		= (float) 0.0;
	sys->net[cn]->parameters[ NET_P_distance_start ].parameter.value.f	= (float) DISTANCE_START;
	sys->net[cn]->parameters[ NET_P_distance_step ].parameter.value.f	= (float) DISTANCE_STEP;
	sys->net[cn]->parameters[ NET_P_distance_finish ].parameter.value.f	= (float) DISTANCE_FINISH;
	sys->net[cn]->parameters[ NET_P_gain ].parameter.value.f		= (float) GAIN_START;
	sys->net[cn]->parameters[ NET_P_gain_start ].parameter.value.f		= (float) GAIN_START;
	sys->net[cn]->parameters[ NET_P_gain_step ].parameter.value.f		= (float) GAIN_STEP;
	sys->net[cn]->parameters[ NET_P_gain_finish ].parameter.value.f		= (float) GAIN_FINISH;
	sys->net[cn]->parameters[ NET_P_distance_control ].parameter.value.f	= (float) DISTANCE_CONTROL;
	sys->net[cn]->parameters[ NET_P_input_control ].parameter.value.f	= (float) INPUT_PATTERN_CONTROL;
	sys->net[cn]->parameters[ NET_P_current_pattern ].parameter.value.f	= (float) 0.0;
	sys->net[cn]->parameters[ NET_P_x_dimension ].parameter.value.f		= (float) X_DIMENSION;

	/********* allocating the output, teaching and input pattern pipe pointers ********/

	I_LAYER = 0;
	O_LAYER = 1;
	out_size = 0;
	in_size = 0;

	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
		out_size += sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons;
	}
	sys->net[cn]->fanout = out_size;
	sys->net[cn]->output_port	= ( caddr_t * )jalloc (sizeof( caddr_t ), out_size);
	sys->net[cn]->target		= ( caddr_t * )jalloc (sizeof( caddr_t ), out_size);

	for (i=0; i<sys->net[cn]->layer[I_LAYER]->clusters; i++) {
		in_size += sys->net[cn]->layer[I_LAYER]->cluster[i]->neurons;
	}
	sys->net[cn]->fanin = in_size;
	sys->net[cn]->input_port	= ( caddr_t * )jalloc (sizeof( caddr_t ), in_size);

	input_vector	= (float *) jalloc( sizeof (float), sys->net[cn]->fanin );	/* to receive input pattern */
	target_vector	= (float *) jalloc( sizeof (float), sys->net[cn]->fanout );	/* to receive input pattern */

	/**** build up the input pipe pointers ****/

	free_pointer1 = sys->net[cn]->input_port;
	for (i=0; i<sys->net[cn]->layer[I_LAYER]->clusters; i++) {
		for (j=0; j<sys->net[cn]->layer[I_LAYER]->cluster[i]->neurons; j++) {
		        *free_pointer1++ = 
				(caddr_t) &sys->net[cn]->layer[I_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ].value.f;
		}
	}

	/**** build up the output pipe pointers ****/

	free_pointer1 = sys->net[cn]->output_port;
	free_pointer2 = sys->net[cn]->target;
	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
		for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {
		        *free_pointer1++ = 
				(caddr_t) &sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ].value.f;
		        *free_pointer2++ = 
				(caddr_t) &sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_TARGET ].value.f;
		}
	}

	/********* allocating synapses for the neurons *********/

/* ------------------------------------------------------------------------ *\
	THIS SECTION NOT USED NOW

	The intention is to create extra shared synapses in the output layer
	to hold as `weights' the distances between neurons - so that the
	distance() function is not called at run time to determine whether or
	not a neuron is within the neighbourhood of the current winner.

	This implies that the shape of the output layer - i.e. x_dimension and
	y_dimension must be known during the connect() phase - and cannot, 
	therefore be loaded as a parameter after connect() has taken place.

	shared synapses within output layer store distances ( or will) : Wij = Wji :

	s = out_size - 1;
        for (i = 0; i < out_size; i++) {
		sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[i]->fanin = s;
		sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[i]->synapses = s;
		sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[i]->synapse = (synapse_type **) jalloc (sizeof(synapse_type *), s);
		sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[i]->input_neuron = (neuron_type **) jalloc (sizeof(neuron_type *), s );
	}

	s  = (out_size * (out_size - 1)) / 2;
	sp = (synapse_type *) jalloc (sizeof(synapse_type), s);

	for (i=0; i<out_size; i++) {
		for (j = i + 1; j<out_size; j++) {
			sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[ i ]->synapse[ j - 1 ] = sp;
			sys->net[cn]->layer[O_LAYER]->cluster[0]->neuron[ j ]->synapse[ i ] = sp++;
		}
	}
\* ------------------------------------------------------------------------ */

	s = in_size;
	for (j=0; j<sys->net[cn]->layer[O_LAYER]->clusters; j++) {
        for (k=0; k<sys->net[cn]->layer[O_LAYER]->cluster[j]->neurons; k++) {
		sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->fanin = s;
		sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->synapses = s;
		sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->synapse = 
			(synapse_type **) jalloc (sizeof(synapse_type *), s);
		sp = (synapse_type *) jalloc (sizeof(synapse_type), s);
		for (l = 0; l < s; l++) {
			sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->synapse[l] = sp++;
		}
		sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->input_neuron = 
			(neuron_type **) jalloc (sizeof(neuron_type *), s );
	}
	}

	/****  connecting the neurons => putting the address in the ****/
	/****       neuron.input_neuron  for the forward path       ****/

	for (j=0; j<sys->net[cn]->layer[O_LAYER]->clusters; j++) {
	for (k=0; k<sys->net[cn]->layer[O_LAYER]->cluster[j]->neurons; k++) {
		index = 0;
		for (l=0; l<sys->net[cn]->layer[I_LAYER]->clusters; l++) {
		      for (m=0; m<sys->net[cn]->layer[I_LAYER]->cluster[l]->neurons; m++) {
				sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->input_neuron[index++] =
					sys->net[cn]->layer[I_LAYER]->cluster[l]->neuron[m];
		      }
		}
		if (index != sys->net[cn]->layer[O_LAYER]->cluster[j]->neuron[k]->fanin) {
			error("Network topology specification inconsistent!\n");
			exit(3);
		}
	}
	}


	/* write route[] data */

	for (i=0; i<sys->net[cn]->layers; i++) {
		for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
			for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[0] = cn;
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[1] = i;
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[2] = j;
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[3] = k;
			}
		}
	}

	}	/* end for cn ... */

	return( OK );
}

/* -------------------------------------------------------------- */
/*	build_rule.c                                              */
/* -------------------------------------------------------------- */


/*****	Function Declarations *****/

int build_rules()
{
	int cn, i, j, k;

	if (!sys || !sys->net || ((O_LAYER < 0)))
		return ( FAIL );

	for ( cn = 0; cn < sys->nets; cn++ ) {

	/****  Initialize rules in output layer	 ****/

	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
	for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {

		rule_init (
			"neuron.weight_update",
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules [NEU_R_weight_update ],
			&weight_update_class,
			in_size,
			&sys->net[cn]->parameters[ NET_P_distance ].parameter,
			&sys->net[cn]->parameters[ NET_P_winner ].parameter,
			sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j],	/* Don't like this */
			&sys->net[cn]->parameters[ NET_P_gain ].parameter,
			EOP );

		rule_init (
			"neuron.state_update",
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules [NEU_R_state_update ],
			&state_update_class,
			in_size,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ],
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_ACC ],
			EOP );

		rule_init (
			"neuron.calculate_score",
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules [NEU_R_calculate_score ],
			&calculate_score_class,
			in_size,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_ACC ],
			EOP );

	}		/* end for j ... */
	}		/* end for i ... */

	/*** extending rules -- to fill in the parameter pointers ***/

	/* NO policy for connecting clusters at present - assume only one */

	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
	for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {
	for (k=0; k < in_size; k++ ) {

		rule_extend (
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_weight_update ],
			k,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->input_neuron[k]->state[ N_STATE ],
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->synapse[k]->weight,
			EOP );

		rule_extend (
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_calculate_score ],
			k,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->input_neuron[k]->state[ N_STATE ],
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->synapse[k]->weight,
			EOP );

		rule_extend (
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_state_update ],
			k,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->input_neuron[k]->state[ N_STATE ],
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->synapse[k]->weight,
			EOP );

	}	/* end for k ... */
	}	/* end for j ... */
	}	/* end for i ... */


	/*** Initialize the rules at the net level ***/

	rule_init (
		"net.learn",
		&sys->net[cn]->rules[ NET_R_learn ],
		&learn_meta_class,
		1,
		EOP );

	rule_extend (
		&sys->net[cn]->rules[ NET_R_learn ],
		0,
		&sys->net[cn]->rules[ NET_R_learn_pattern ],
		&sys->net[cn]->rules[ NET_R_decrement_distance ],
		EOP );

	rule_init (
		"net.learn_pattern",
		&sys->net[cn]->rules[ NET_R_learn_pattern ],
		&learn_pattern_meta_class,
		1,
		EOP );

	rule_extend (
		&sys->net[cn]->rules[ NET_R_learn_pattern ],
		0,
		&sys->net[cn]->rules[ NET_R_read_input_rule ],
		&sys->net[cn]->rules[ NET_R_meta_pick_winner ],
		&sys->net[cn]->rules[ NET_R_weight_update ],
		&sys->net[cn]->rules[ NET_R_increment_pattern ],
		EOP );

	rule_init (
		"net.read_input_rule",
		&sys->net[cn]->rules[ NET_R_read_input_rule ],
		&read_input_rule_class,
		0,
		&sys->net[c_net]->parameters[ NET_P_current_pattern ].parameter,
		EOP );

	rule_init (
		"net.recall",
		&sys->net[cn]->rules[ NET_R_recall ],
		&recall_meta_class,
		1,
		EOP );

	rule_extend (
		&sys->net[cn]->rules[ NET_R_recall ],
		0,
		&sys->net[cn]->rules[ NET_R_meta_pick_winner ],
		&sys->net[cn]->rules[ NET_R_state_update ],
		EOP );

	rule_init (
		"net.meta_pick_winner",
		&sys->net[cn]->rules[ NET_R_meta_pick_winner ],
		&pick_winner_meta_class,
		out_size + 1,
		EOP );
	
	rule_init (
		"net.pick_winner",
		&sys->net[cn]->rules[ NET_R_pick_winner ],
		&pick_winner_class,
		out_size,
		&sys->net[cn]->parameters[ NET_P_winner ].parameter,
		EOP );
	
	rule_init (
		"net.decrement_distance",
		&sys->net[cn]->rules[ NET_R_decrement_distance ],
		&decrement_distance_class,
		0,
		EOP );
	
	rule_init (
		"net.increment_pattern",
		&sys->net[cn]->rules[ NET_R_increment_pattern ],
		&increment_pattern_class,
		0,
		EOP );
	
	rule_init (
		"net.weight_update",
		&sys->net[cn]->rules[ NET_R_weight_update ],
		&weight_update_meta_class,
		out_size,
		EOP );
	
	rule_init (
		"net.state_update",
		&sys->net[cn]->rules[ NET_R_state_update ],
		&state_update_meta_class,
		out_size,
		EOP );
	
	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
	for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {

		rule_extend (
			&sys->net[cn]->rules[ NET_R_meta_pick_winner ],
			j,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_calculate_score ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_pick_winner ],
			j,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_ACC ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_weight_update ],
			j,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_weight_update ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_state_update ],
			j,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->rules[ NEU_R_state_update ],
			EOP );

	}	/* end for j ... */
	}	/* end for i ... */

	rule_extend (
		&sys->net[cn]->rules[ NET_R_meta_pick_winner ],
		j,
		&sys->net[cn]->rules[ NET_R_pick_winner ],
		EOP );

	}	/* end for cn ... */

	return ( OK );
}

/**********************  Main program ******************************************/
#ifndef	OMIT_MAIN
main (argc, argv)
int	argc;
char	*argv[];
{
	int		i, somefail, cycle;
	int		wh[4];
	float		result;
	pat_elem	*p;
	FILE		*fp;

	srandom(0xd5a4793c);	/*  Initialize drand48  */
	srand48(random());

	/*  Construct the network */

	if ( rc_read () ) {
		printf ( "Problem with .pgmrc file\n");
		exit ( 1 );
	}
	printf ( "local_user %s\n", rc [ RC_local_user ] );
	printf ( "local_host %s\n", rc [ RC_local_host ] );

	if (argc == 1) {
		if (connect(&config))		/* use internal configuration defaults */
			exit(3);
		if (build_rules())
			exit(3);
		rand_weight();
		sys_save ( "test" );
		exit(0);
	}
	else {
		strcpy( system_filename, argv[ 1 ] );
		sys_load( system_filename );	/* (possibly) different configuration */
	}

	/*	Set pattern_control to loaded parameter */
	input_pattern_control	= (int) sys->net[c_net]->parameters[ NET_P_input_control ].parameter.value.f;

	/*	Recalculate dimensions - to allow loaded configuration to be different from defaults.
		This is only for testing stand-alone version - since this main() function does not
		execute when the algorithm is called from the graphic monitor. */
	x_dimension		= (int) sys->net[c_net]->parameters[ NET_P_x_dimension ].parameter.value.f;
	y_dimension		= out_size / x_dimension;

	/*  Initialise patterns specified on command line */

	if ( argc > 2 ) {
		i = init_patterns ( argv [ 2 ] );
 		printf ( "init_patterns returns %d from %s\n", i, argv [ 2 ] );
	}

	if ( argc < 4 ) {	/* may specify test mode by adding any third parameter */
		result = learn();
	}

	if ( ( fp = fopen ( "temp", "w+" ) ) == NULL ) {
		printf ( "unable to open temporary log file\n");
	}
	for ( i = 0; i < pattern_count; i++ ) {
		current_pattern = indextop( i );
 		read_input	( i );
		current_pattern->target_width	= x_dimension;
		current_pattern->target_height	= y_dimension;
		recall();
		shownet();
		printf ( "input pattern index %4d : [ %c ] winner %4d\n",
			i, current_pattern->input_character,
			(int) sys->net[c_net]->parameters[ NET_P_winner ].parameter.value.f );

		if ( fp ) {
			fprintf ( fp, "input pattern index %4d : [ %c ] winner %4d\n",
				i, current_pattern->input_character,
				(int) sys->net[c_net]->parameters[ NET_P_winner ].parameter.value.f );
		}

	}
	if ( fp )
		fclose ( fp );

	if (system_filename[0] == '\0') {
		sys_save ( "test" );
	}
	else {
		sys_save( system_filename );
	}

	rc_write ();

	exit (0);
}

#endif
/* --------------------------------------------------------------------------- */
