/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * pgmrc.c
 ******************************************************************************/

#include	<stdio.h>
#include	<strings.h>
#include	<pwd.h>
#include	<sys/param.h>
#include	<time.h>
#include	"pygmalion.h"
#include	"sysdef.h"
#include	"pgmrc.h"

/* -------------------------------------------------------------- *\
	internals
\* -------------------------------------------------------------- */

void	rc_set_defaults ();	/* preset values before file read   */
int	rc_validate ();		/* validate lines read from .pgmrc  */

int	cmd_srch ();		/* search through cmd table */
char	*rcmd_srch ();		/* lookup command using number */

char	*trim ();		/* take white space from ends of value */
char	*catdup ();		/* join two strings */
char	*strstr ();		/* embedded string search */
int	lexequ ();		/* lexical string compare */

/* -------------------------------------------------------------- *\
	externals
\* -------------------------------------------------------------- */

extern	char		*getenv ();
extern	char		*getwd ();
extern	char		*cuserid ();
extern	int		gethostname ();
extern	struct	passwd	*getpwnam ();
extern	char		*strdup ();
extern	long		time ();
extern	char		*ctime ();

/* -------------------------------------------------------------- *\
	structure definitions
\* -------------------------------------------------------------- */

static	CMD_TABLE rc_cmds[] = {
	"local_user",			RC_local_user,			VAL_string,
	"local_host",			RC_local_host,			VAL_string,
	"remote_user",			RC_remote_user,			VAL_string,
	"remote_host",			RC_remote_host,			VAL_string,
	"display",			RC_display,			VAL_string,

	"working_directory",		RC_working_directory,		VAL_ignore,
	"home_directory",		RC_home_directory,		VAL_ignore,

	"algorithm_directory",		RC_algorithm_directory,		VAL_directory,
	"application_directory",	RC_application_directory,	VAL_directory,
	"configuration_directory",	RC_configuration_directory,	VAL_directory,
	"error_message_directory",	RC_error_message_directory,	VAL_directory,
	"font_directory",		RC_font_directory,		VAL_directory,
	"nc_code_directory",		RC_nc_code_directory,		VAL_directory,
	"nn_save_directory",		RC_nn_save_directory,		VAL_directory,
	"pattern_directory",		RC_pattern_directory,		VAL_directory,
	"pygmalion_directory",		RC_pygmalion_directory,		VAL_directory,
	"rpc_server_directory",		RC_rpc_server_directory,	VAL_directory,

	"algorithm_file",		RC_algorithm_file,		VAL_filename,
	"application_file",		RC_application_file,		VAL_filename,
	"configuration_file",		RC_configuration_file,		VAL_filename,
	"nc_code_file",			RC_nc_code_file,		VAL_filename,
	"nn_save_file",			RC_nn_save_file,		VAL_filename,
	"pattern_descriptor_file",	RC_pattern_descriptor_file,	VAL_filename,

	"connectivity_display",		RC_connectivity_display,	VAL_on_off,
	"matrix_display",		RC_matrix_display,		VAL_on_off,
	"inverse_display",		RC_inverse_display,		VAL_on_off,

	"fixed_font",			RC_fixed_font,			VAL_string,
	"small_font",			RC_small_font,			VAL_string,
	"large_font",			RC_large_font,			VAL_string,

	"main_foreground_red",		RC_main_foreground_red,		VAL_0_65535,
	"main_foreground_green",	RC_main_foreground_green,	VAL_0_65535,
	"main_foregound_blue",		RC_main_foreground_blue,	VAL_0_65535,

	"main_background_red",		RC_main_background_red,		VAL_0_65535,
	"main_background_green",	RC_main_background_green,	VAL_0_65535,
	"main_backgound_blue",		RC_main_background_blue,	VAL_0_65535,

	"button_foreground_red",	RC_button_foreground_red,	VAL_0_65535,
	"button_foreground_green",	RC_button_foreground_green,	VAL_0_65535,
	"button_foregound_blue",	RC_button_foreground_blue,	VAL_0_65535,

	"button_background_red",	RC_button_background_red,	VAL_0_65535,
	"button_background_green",	RC_button_background_green,	VAL_0_65535,
	"button_backgound_blue",	RC_button_background_blue,	VAL_0_65535,

	"menu_foreground_red",		RC_menu_foreground_red,		VAL_0_65535,
	"menu_foreground_green",	RC_menu_foreground_green,	VAL_0_65535,
	"menu_foregound_blue",		RC_menu_foreground_blue,	VAL_0_65535,

	"menu_background_red",		RC_menu_background_red,		VAL_0_65535,
	"menu_background_green",	RC_menu_background_green,	VAL_0_65535,
	"menu_backgound_blue",		RC_menu_background_blue,	VAL_0_65535,

	0,				-1,				-1
};

/*  character conversion table : lower to upper case letters  */
static	char	chrcnv[] = {
	'\0', '\1', '\2', '\3', '\4', '\5', '\6', '\7', '\10', '\t', '\n', '\13', '\14', '\r', '\16', '\17',
	'\20', '\21', '\22', '\23', '\24', '\25', '\26', '\27', '\30', '\31', '\32', '\33', '\34', '\35', '\36', '\37',
	' ', '!', '"', '#', '$', '%', '&', '\47', '(', ')', '*', '+', ',', '-', '.', '/',
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?',
	'@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_',
	'`', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '{', '|', '}', '~', '\177',
	'\0', '\1', '\2', '\3', '\4', '\5', '\6', '\7', '\10', '\t', '\n', '\13', '\14', '\r', '\16', '\17',
	'\20', '\21', '\22', '\23', '\24', '\25', '\26', '\27', '\30', '\31', '\32', '\33', '\34', '\35', '\36', '\37',
	' ', '!', '"', '#', '$', '%', '&', '\47', '(', ')', '*', '+', ',', '-', '.', '/',
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?',
	'@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_',
	'`', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
	'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '{', '|', '}', '~', '\177'
};

/* -------------------------------------------------------------- *\
	globals
\* -------------------------------------------------------------- */

	char	*rc [ RC_MAX ];
	char	*pgmrc_filename = ".pgmrc";
	char	pgmrc_path[ LINESIZE ];		/* including filename */

	char	*home_directory;
	char	*local_user;
	char	*local_host;
	char	*working_directory;
	char	*pygmalion_directory;
	char	buf[ LINESIZE ];

/* -------------------------------------------------------------- */

int	rc_read()
{
	FILE		*fd;
	char		*c, *key, *value;
	int		which;
	int		envset = FALSE;
	static	int	rc_has_been_read_already = 0;	/* I love short name */
	struct	passwd	*passwd;

	if ( rc_has_been_read_already ) {
		return ( OK );
	}
	else {
		rc_has_been_read_already = 1;
	}

	if ( ( c = getwd ( buf ) ) == NULLCP ) {
		printf ( "rc_read/ unable to determine current working directory\n" );
		return ( NOTOK );
	}
	working_directory = strdup ( buf );

	if ( ( c = cuserid ( buf ) ) == NULLCP ) {
		printf ( "rc_read/ unable to determine user id\n" );
		return ( NOTOK );
	}
	local_user = strdup ( buf );

	if ( gethostname ( buf, LINESIZE ) == -1 ) {
		printf ( "rc_read/ unable to determine hostname\n" );
		return ( NOTOK );
	}
	local_host = strdup ( buf );

	if ( ( passwd = getpwnam ( local_user ) ) == NULL ) {
		printf ( "rc_read/ unable to find passwd entry\n" );
		return ( NOTOK );
	}
	home_directory = strdup ( passwd->pw_dir );


	if ( ( c = getenv ( PGMRC_ENV ) ) != NULLCP ) {
		strcpy ( pgmrc_path, c );
		if ( ! ( fd = fopen( pgmrc_path, "r") ) ) {
			printf ( "rc_read/ unable to open environment PGMRC rc file : %s\n", pgmrc_path);
		}
		else {
			envset = TRUE;
		}
	}
	if ( ! envset ) {
		sprintf ( pgmrc_path, "%s/%s", working_directory, pgmrc_filename );
		if ( ! ( fd = fopen( pgmrc_path, "r") ) ) {
			/* not in working_directory - try home_directory */
			sprintf ( pgmrc_path, "%s/%s", home_directory, pgmrc_filename );
			if ( ! ( fd = fopen( pgmrc_path, "r") ) ) {
				printf ( "rc_read/ unable to find rc file %s - creating in home directory %s\n", pgmrc_filename, home_directory);
				rc_set_defaults( 1 );
				if ( rc_write() == NOTOK ) {
					return ( NOTOK );
				}
				else {
					return ( OK );
				}
			}
		}
	}
 	rc_set_defaults( 0 );

	while ( ( ( fgets ( buf, LINESIZE, fd ) ) != NULL ) && ( ! feof ( fd ) ) ) {
		if ( (buf [ 0 ] == '\n') || (buf [ 0 ] == '#') ) {
			/* blank line or comment */
			continue;
		}
		if ( ( c = index ( buf, EQUCHAR ) ) == NULLCP ) {
			printf ( "rc_read/ invalid line - no '%c' in line : %s", EQUCHAR, buf );
			continue;
		}
		*c = '\0';
		value = c + 1;
		key = buf;

		key = trim ( key );
		value = trim ( value );

		if ( ( which = cmd_srch(key, rc_cmds) ) > 0 ) {
			if ( rc_validate ( value, key, rc_cmds [ which - 1 ].cmd_valid ) == OK ) {
				rc_update ( which, value );
			}
		}
		else {
			printf ( "rc_read/ invalid keyword : %s\n", key );
		}
	}

	close ( fd );

	return ( OK );
}

/* ------------------------------------------------------------------------ */

int	rc_write()
{
	FILE			*fd;
	register CMD_TABLE	*cmd;
	long			clock;
	int			valid;

	if ( ! ( fd = fopen( pgmrc_path, "w+" ) ) ) {
		printf ( "rc_write/ unable to open rc file %s\n", pgmrc_path);
		return ( NOTOK );
	}
	fprintf ( fd, "#\t%s\tDefaults for Pygmalion Graphic Monitor\n", pgmrc_filename );
	fprintf ( fd, "#\n" );
	fprintf ( fd, "#\t%s\n", local_user );
	fprintf ( fd, "#\t%s\n", local_host );
	fprintf ( fd, "#\n" );
	clock =  time ( (long *) 0 );
	fprintf ( fd, "#\t%s", ctime ( &clock ) );
	fprintf ( fd, "#\n" );

	for (valid = -1, cmd = rc_cmds; cmd->cmd_key != NULLCP; cmd++ ) {
		if ( valid != cmd->cmd_valid ) {	/* blank line on change of validation type */
			valid = cmd->cmd_valid;
			fprintf ( fd, "\n" );
		}
		fprintf ( fd, "%-32s= %s\n", cmd->cmd_key, rc [ cmd->cmd_value ] );
	}

	close ( fd );

	return ( OK );
}

/* -------------------------------------------------------------- */

void	rc_update ( which, string )
int	which;
char	*string;
{
	free ( rc [ which ] );
	rc [ which ] = strdup ( string );
}
/* -------------------------------------------------------------- */

int	 rc_validate ( value, key, type )
char	*value, *key;
int	type;
{
	int	i, result = OK;
	char	*c;

	switch ( type ) {

		case VAL_ignore:	/* force acceptance of default value */
			result = NOTOK;
			break;

		case VAL_string:	/* OK - has been trimmed already */
		case VAL_directory:	/* may expand these */
		case VAL_filename:
			break;

		case VAL_on_off:
			if ( lexequ ( value, "on" ) == 0 ) {
				strcpy ( value, "on" );
			}
			else if ( lexequ ( value, "off" ) == 0 ) {
				strcpy ( value, "off" );
			}
			else {
				printf ( "rc_validate/ value (%s) for %s must be 'on' or 'off'\n", value, key);
				result = NOTOK;
			}
			break;

		case VAL_0_65535:
			c = value;
			while ( *c ) {
				if ( index ( "0123456789", *c ) == NULLCP ) {
					printf ( "rc_validate/ invalid character (%c) in %s : %s must be in range 0 to 65535\n", *c, value, key );
					result = NOTOK;
					break;
				}
				c++;
			}
			if ( result == OK ) {
				i = atoi ( value );
				if ( ( i < 0 ) || ( i > 65535 ) ) {
					printf ( "rc_validate/ value (%s) for %s must be in range 0 to 65535\n", value, key );
					result = NOTOK;
				}
			}
			if ( result == OK ) {
				sprintf ( value, "%d", i );
			}
			break;

		default:
			printf ( "rc_validate/ invalid validation type : %d\n", type );
			result = NOTOK;
	}
	return ( result );
}
/* -------------------------------------------------------------- */

void	rc_set_defaults(flag)
int	flag;			/* flag = 1 indicates that pgmrc file must be created */
{
	char	*c, *d;

	/* every entry must be memory obtained by malloc - e.g. strdup()'e */
	rc [ RC_local_user ] 			= strdup ( local_user );
	rc [ RC_local_host ]			= strdup ( local_host );
	rc [ RC_remote_user ]			= strdup ( local_user );
	rc [ RC_remote_host ]			= strdup ( local_host );
	sprintf ( buf, "%s", local_host );
	if ( c = index ( buf, '.' ) ) {
		*c = '\0';
	}
	strcat ( buf, ":0.0" );
	rc [ RC_display ]			= strdup ( buf );

	/* Assume that if the string `pygmalion' appears in the working
	directory, that the last occurence of `pygmalion' represents the
	RC_pygmalion_directory, so extract that directory name.  If not -
	then must set defaults w.r.t.  working directory anyway. */

	pygmalion_directory = strdup ( working_directory );

	if ( c = strstr ( pygmalion_directory, "pygmalion" ) ) {
		d = c;
		while ( c = strstr ( ++d, "pygmalion" ) ) {	/* find LAST occurrence */
			d = c;
		}
		if ( c = index ( d, '/' ) ) {
			*c = '\0';
 		}
		if ( flag ) {
			printf ( "rc_set_defaults/ assuming `%s' for RC_pygmalion_directory\n", pygmalion_directory );
		}
	}
	else {
		if ( flag ) {
			printf ( "rc_set_defaults/ warning : cannot find `pygmalion' in working directory\n" );
		}
	}
	rc [ RC_working_directory ]		= strdup ( working_directory );
	rc [ RC_home_directory ]		= strdup ( home_directory );

	rc [ RC_pygmalion_directory ]		= strdup ( pygmalion_directory );

	rc [ RC_algorithm_directory ]		= catdup ( pygmalion_directory, "data/algorithms");
	rc [ RC_application_directory ]		= catdup ( pygmalion_directory, "data/applications" );
	rc [ RC_configuration_directory ]	= catdup ( pygmalion_directory, "data/configuration" );
	rc [ RC_error_message_directory ]	= catdup ( pygmalion_directory, "data/errmsg" );
	rc [ RC_font_directory ]		= catdup ( pygmalion_directory, "data/fonts" );
	rc [ RC_nc_code_directory ]		= catdup ( pygmalion_directory, "data/nc_code" );
	rc [ RC_nn_save_directory ]		= catdup ( pygmalion_directory, "data/save" );
	rc [ RC_pattern_directory ]		= catdup ( pygmalion_directory, "data/patterns" );
	rc [ RC_rpc_server_directory ]		= catdup ( pygmalion_directory, "src/pgm/rpc" );

	rc [ RC_algorithm_file ]		= strdup ( "" );
	rc [ RC_application_file ]		= strdup ( "" );
	rc [ RC_configuration_file ]		= strdup ( "" );
	rc [ RC_nc_code_file ]			= strdup ( "" );
	rc [ RC_nn_save_file ]			= strdup ( "" );
	rc [ RC_pattern_descriptor_file ]	= strdup ( "" );

	rc [ RC_connectivity_display ]		= strdup ( "off" );
	rc [ RC_matrix_display ]		= strdup ( "off" );
	rc [ RC_inverse_display ]		= strdup ( "off" );

	rc [ RC_fixed_font ]			= strdup ( "fixed" );
	rc [ RC_small_font ]			= strdup ( "fixed" );
	rc [ RC_large_font ]			= strdup ( "fixed" );

	rc [ RC_main_foreground_red ]		= strdup ( "53760" );
	rc [ RC_main_foreground_green ]		= strdup ( "53760" );
	rc [ RC_main_foreground_blue ]		= strdup ( "53760" );

	rc [ RC_main_background_red ]		= strdup ( "21504" );
	rc [ RC_main_background_green ]		= strdup ( "21504" );
	rc [ RC_main_background_blue ]		= strdup ( "21504" );

	rc [ RC_button_foreground_red ]		= strdup ( "52224" );
	rc [ RC_button_foreground_green ]	= strdup ( "32512" );
	rc [ RC_button_foreground_blue ]	= strdup ( "12800" );

	rc [ RC_button_background_red ]		= strdup ( "28416" );
	rc [ RC_button_background_green ]	= strdup ( "16896" );
	rc [ RC_button_background_blue ]	= strdup ( "16896" );

	rc [ RC_menu_foreground_red ]		= strdup ( "0" );
	rc [ RC_menu_foreground_green ]		= strdup ( "0" );
	rc [ RC_menu_foreground_blue ]		= strdup ( "0" );

	rc [ RC_menu_background_red ]		= strdup ( "49152" );
	rc [ RC_menu_background_green ]		= strdup ( "49152" );
	rc [ RC_menu_background_blue ]		= strdup ( "49152" );
}
/* -------------------------------------------------------------- */

int	cmd_srch(str, cmd)	/* cmd_srch - search a lookup table: return numeric value */
register char		*str;
register CMD_TABLE	*cmd;
{
	for(;cmd->cmd_key != NULLCP; cmd++)
		if(lexequ(str, cmd->cmd_key) == 0)
			return(cmd->cmd_value);
	return(cmd->cmd_value);		/* table ends with -1 */
}
/* -------------------------------------------------------------- */

char   *rcmd_srch(val, cmd)	/* rcmd_srch: search a lookup table: return string value */
register int	   val;
register CMD_TABLE *cmd;
{
	for(;cmd->cmd_key != NULLCP; cmd++)
		if(val == cmd->cmd_value)
			return(cmd->cmd_key);
	return(NULLCP);
}
/* -------------------------------------------------------------- */

char	*catdup( s1, s2 )	/* concatenate directories and return strdup() */
char	*s1, *s2;
{
	char	buf [ FILESIZE ];

	sprintf ( buf, "%s/%s", s1, s2 );
	return ( strdup( buf ) );
}
/* -------------------------------------------------------------- */

int	lexequ (str1, str2)	/* lexequ - Compare two strings ignoring case */
register char   *str1,
		*str2;
{
	if (str1 == NULL)
		if (str2 == NULL)
			return (0);
		else
			return (-1);

	if (str2 == NULL)
		return (1);

	while (chrcnv[*str1] == chrcnv[*str2]) {
		str2++;
		if (*str1++ == NULL)
			return (0);
	}

	if (chrcnv[*str1] > chrcnv[*str2])
		return (1);
	else
		return (-1);
}
/* -------------------------------------------------------------- */

char	*trim ( s )	/* trim new line, space and tabs from front and back */
char	*s;
{
	char	*c;
	int	i;

	if ( ( i = strlen ( s ) ) == 0 ) {
		return ( s );
	}
	c = s + i - 1;
	while ( i && ( *c == '\n' ) || ( *c == ' ' ) || ( *c == '\t' ) ) {
		i--;
		*c-- = '\0';
	}
	c = s;
	while ( ( *c == '\n' ) || ( *c == ' ' ) || ( *c == '\t' ) ) {
		*c++ = '\0';
	}
	return ( c );
}
/* ------------------------------------------------------------------------ */

char	*strstr ( str1, str2)	/* looks for first str2 in str1 */
char	*str1, *str2;
{
	char		*p = str1, *q = NULL;
	int		i, j;

	j = strlen ( str2 );

	while ( p = index ( p, str2[ 0 ] ) ) {	/* first char found */
		for ( i = 0, q = p ; i < j ; i++, q++ ) {
			if ( *q != str2 [ i ] ) {
				break;
			}
		}
		if ( i < j ) {
			q = NULL;
		}
		else {
			q = p;
			break;
		}
		p++;
	}
	return ( q );
}
/* --------------------------------------------------------------- */

#ifdef	INCLUDE_MAIN

main( argc, argv )	/* 	main() for stand alone test version - this is normally omitted */
int	argc;
char	**argv;
{
	rc_read();
	if ( ( argc > 1 ) && ( strcmp ( argv[ 1 ], "-d" ) == 0 ) ) {
		rc_set_defaults( 1 );
	}
	printf ( "pgmrc initialisation file : %s\n", pgmrc_path );
	rc_write();
}

#endif

/* -------------------------------------------------------------- */
