/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * pattern.c
 ******************************************************************************/


#include	<stdio.h>
#include	<math.h>
#include	"pygmalion.h"
#include	"sysdef.h"
#include	"util.h"
#include	"fontdefs.h"
#include	"pgmrc.h"

/* --------------- functions defined here - not algorithm specific -------- */

int		init_patterns();
int		count_patterns();
int		read_pattern();
pat_elem	*load_patterns();
void		add_to_pattern_list();
int		readpat();
int		read_input();
int		read_target();
int		read_xfile();
int		read_floatfile();
int		represent();
pat_elem	*indextop ();
void		free_pattern_list();
char		*path_test();

/* --------------- externs ------------------------------------------------ */
extern	system_type	*sys;
extern	int		c_net;
extern	float		*input_vector;
extern	float		*target_vector;

extern	char		*rc [];

extern	int		input_pattern_control;
extern	int		target_pattern_control;

/* --------------- globals ------------------------------------------------ */

pat_elem	*pattern_list 		= NULL;		/* patterns loaded for net */
pat_elem	*current_pattern 	= NULL;		/* to pass widths to show_net */
char		pattern_filename[ FILESIZE ];
int		pattern_count		= 0;
int		pattern_index		= 0;

static	char	buf [ LINESIZE ];
static	FILE	*progress = NULL;

/* ------------------------------------------------------------------------ */

int init_patterns ( reqfile )		/* initialise and count */
char		*reqfile;
{
	FILE		*fd;
	int		i=0;
	pat_elem	*p;
	char		*filename;

	rc_read();
	filename = path_test ( reqfile );
	open_progress ();
	
	if ( ! ( fd = fopen( filename, "r" ) ) ) {
		fprintf ( progress,"init_patterns/ cannot open pattern file %s\n", filename);
		fflush ( progress );
		return ( NOTOK );
	}
	if ( ( p = load_patterns (fd, buf) ) == NULL ) {
		close ( fd );
		return ( NOTOK );
	}
	close ( fd );
	free_pattern_list ( &pattern_list );
	pattern_list = p;
	strcpy ( pattern_filename, filename );

	pattern_count = count_patterns ( pattern_list );

#ifdef	DEBUG_PATTERN
	fprintf ( progress,"init_patterns/ %d patterns in %s\n", pattern_count, filename );
	fflush ( progress );
#endif

	return ( pattern_count );
}
/* ------------------------------------------------------------------------ */
int		count_patterns ( p )
pat_elem	*p;
{
	int		i = 0;
	pat_elem	*q = p;

	while ( q ) {
		i++;
		q = q->next_ptr;
	}
	return ( i );
}
/* ------------------------------------------------------------------------ */

int read_pattern ( reqfile, index , wh)
char		*reqfile;
int		index;
int		wh[];	/* widths & heights */
{
        short unsigned int      i,j,k;
	FILE			*fd;
	pat_elem		*p;
	char			*filename;

	/*	if wh[0] is OK then read input AND target, if NOTOK only input	*/

	rc_read();
	filename = path_test ( reqfile );
	open_progress ();

        if ( strcmp (filename, pattern_filename) != 0 ) {    /* New pattern file */
#ifdef	DEBUG_PATTERN
                if (pattern_filename[0] != '\0') {
			fprintf ( progress,"read_pattern/ new pattern descriptor file %s\n", filename);
			fflush ( progress );
		}
#endif
		if ( ! ( fd = fopen( filename, "r" ) ) ) {
			fprintf ( progress,"read_pattern/ cannot open pattern file %s\n", filename);
			fflush ( progress );
                        return ( NOTOK );
		}
                if ( ( p = load_patterns (fd, buf)) == NULL ) {
			close ( fd );
                        return ( NOTOK );
		}
                else {
			close ( fd );
			pattern_list = p;
			pattern_count = count_patterns ( pattern_list );
			strcpy( pattern_filename, filename );
		}
	}

	if ( read_input( index ) == NOTOK ) {
		return ( NOTOK );
	}
	if ( wh[0] != NOTOK ) {
		if ( read_target( index ) == NOTOK ) {
			return ( NOTOK );
		}
	}

	if ( ( p = indextop ( index ) ) == NULL ) {
		fprintf ( progress, "read_pattern/ pattern index %d not found\n", index);
		fflush ( progress );
		return ( NOTOK );
	}
/* 	add_to_pattern_list(p); 	*/

	wh[0]	= p->input_width;
	wh[1]	= p->input_height;
	wh[2]	= p->target_width;
	wh[3]	= p->target_height;

	return ( OK );
}

/* ------------------------------------------------------------------------ */
char	*path_test ( reqfile )		/* test if full path supplied */
char	*reqfile;
{
	if ( *reqfile == '/' ) {
		return ( reqfile );
	}
	sprintf ( buf, "%s/%s", rc [ RC_pattern_directory ], reqfile );
	return ( (char *) strdup ( buf ) );
}
/* ------------------------------------------------------------------------ */
open_progress()
{
#ifdef	DEBUG_PROGRESS
	rc_read();
	if ( progress == NULL ) {
		sprintf( buf, "%s/.pgm_pattern_progress", rc[ RC_home_directory ] );
		if ( ( progress = fopen ( buf, "w+") ) != NULL ) {
			return ( OK );
		}
		else {
			fprintf ( progress, "unable to open progress message file\n");
			fflush ( progress );
			exit ( -1 );
			return ( NOTOK );
		}

	}
#else
	progress = stdout;
#endif
	return ( OK );
}
/* ------------------------------------------------------------------------ */
void	add_to_pattern_list(p)	/* add to list in pattern_list if necessary */
pat_elem	*p;
{
	int		found;
	pat_elem	*q, *r;

	found = FALSE;
	q = pattern_list;
	r = NULL;

	while ( q ) {
		if (	q->input_file		== p->input_file &&
			q->target_file		== p->target_file &&
			q->input_character	== p->input_character &&
			q->target_character	== p->target_character ) {

			found = TRUE;
			break;
		}
		r = q;
		q = q->next_ptr;
	}
	if ( !found ) {
		q 			= (pat_elem *) jalloc(sizeof(pat_elem), 1);
		q->input_pattern_type	= p->input_pattern_type;
		q->input_file		= (char *) strdup ( p->input_file );
		q->input_character	= p->input_character;
		q->input_width		= p->input_width;
		q->input_height		= p->input_height;
		q->target_pattern_type	= p->target_pattern_type;
		q->target_file		= (char *) strdup ( p->target_file );
		q->target_character	= p->target_character;
		q->target_width		= p->target_width;
		q->target_height	= p->target_height;
		q->next_ptr		= NULL;

		if ( r == NULL )
			pattern_list	= q;
		else
			r->next_ptr	= q;
	}
}
/* ------------------------------------------------------------------------ */
pat_elem	*load_patterns (fd, str)
FILE		*fd;
char		str[];
{
	int		i, sep = TRUE;	/* sep TRUE iff last character was white space */
	pat_elem	ptr;
	pat_elem	*pat_list = &ptr;
	char		f[4][ FILESIZE ];
	char		*p, c;
	int		input_pattern_type, target_pattern_type;

	open_progress();

	while ( !feof(fd) ) {
		for ( i=0; i<4; i++ ) f [i] [0] = '\0';	/* clear fields */
		i = 0;
		p = f[i];
		while ( ( c = (char) fgetc(fd) ) && !feof(fd) ) {
			if ( c == '\n' )
				break;

			if ( index (" \t",  c ) == NULL ) {
				sep = FALSE;
				*p++ = c;
				*p = '\0';
			}
			else {
				if ( ! sep ) {
					sep = TRUE;
					if ( i < 3 ) {
						i++;
						p = f[i];
					}
				}
			}
		}

		if ( ( f[0] [0] == '\0' ) || ( f[0] [0] == '#' ) ) {	/* blank line or comment */
			continue;
		}
		else if ( strcmp("_EOF", f[0]) ) {

			for ( i=4; i>1; i--) {	/* count fields */
				if ( f [i-1] [0] != '\0' )
					break;
			}
#ifdef	DEBUG_PATTERN
			fprintf ( progress,"load_patterns/ pat_elem  %d  fields %s %s %s %s\n", i, f[0], f[1], f[2], f[3] );
			fflush ( progress );
#endif
			if ( (input_pattern_type = pattern_type ( f[ 0 ] ) ) != PATTERN_TYPE_FONT ) {
				strcpy ( f[3], f[2] );
				strcpy ( f[2], f[1] );
				f[ 1 ][ 0 ] = '\0';
				i++;
			}

			switch ( i ) {
			case 0:
				fprintf ( progress,"load_patterns/ pat_elem - too few fields\n" );
				fflush ( progress );
				return ( NULL );
			case 1:
				strcpy( f[2], f[0] );
				break;
			case 2:
				strcpy( f[2], f[0] );
				strcpy( f[3], f[1] );
				break;
			case 3:
				strcpy( f[3], f[1] );
				break;
			case 4:
				break;

			default:
				fprintf ( progress,"load_patterns/ pat_elem - too many fields\n" );
				fflush ( progress );
				return ( NULL );
			}

			target_pattern_type = pattern_type ( f[ 2 ] );

			pat_list->next_ptr = (pat_elem *) jalloc(sizeof(pat_elem), 1);

			if ( f [ 0 ] [ 0 ] == '/' ) {
				pat_list->next_ptr->input_file		= (char *) strdup( f [ 0 ] );
			}
			else {
				sprintf ( buf, "%s/%s", rc [ ( input_pattern_type == PATTERN_TYPE_FONT ) ? RC_font_directory : RC_pattern_directory ] , f [ 0 ] );
				pat_list->next_ptr->input_file		= (char *) strdup( buf );
			}
			pat_list->next_ptr->input_pattern_type	= input_pattern_type;
			pat_list->next_ptr->input_character	= f [ 1 ] [ 0 ];
			pat_list->next_ptr->input_width		= 0;
			pat_list->next_ptr->input_height	= 0;

			if ( f [ 2 ] [ 0 ] == '/' ) {
				pat_list->next_ptr->target_file		= (char *) strdup( f [ 2 ] );
			}
			else {
				sprintf ( buf, "%s/%s", rc [ ( target_pattern_type == PATTERN_TYPE_FONT ) ? RC_font_directory : RC_pattern_directory ] , f [ 2 ] );
				pat_list->next_ptr->target_file		= (char *) strdup( buf );
			}
			pat_list->next_ptr->target_pattern_type	= target_pattern_type;
			pat_list->next_ptr->target_character	= f [ 3 ] [ 0 ];
			pat_list->next_ptr->target_width	= 0;
			pat_list->next_ptr->target_height	= 0;
/* 			fscanf(fd, "%f", &pat_list->next_ptr->score); */
			pat_list = pat_list->next_ptr;
		}
		else {
 			strcpy(str, f[0] );
			break;
		}
	}
	pat_list->next_ptr = NULL;
	return (ptr.next_ptr);
}
/* ------------------------------------------------------------------------ */
int	pattern_type ( s )
char	*s;
{
	if ( strstr ( s, ".bdf" ) ) {
		return ( PATTERN_TYPE_FONT );
	}
	else if ( strstr ( s, ".f" ) ) {
		return ( PATTERN_TYPE_FLOAT );
	}
	else 	if ( strstr ( s, ".i" ) ) {
		return ( PATTERN_TYPE_INT );
	}
	else 	if ( strstr ( s, ".x" ) ) {
		return ( PATTERN_TYPE_X );
	}
	else {
		return ( PATTERN_TYPE_X );	/* default */
	}
}
/* ------------------------------------------------------------------------ */
void	free_pattern_list ( p )
pat_elem	**p;
{
	pat_elem	*n;

	while ( *p != NULL ) {
		n = (*p)->next_ptr;
		free ( *p );
		*p = n;
	}
	return;
}
/* ------------------------------------------------------------------------ */
int read_input ( index )
int	index;
{
	int			i, input_size;
	int			xwh[2];
	struct  character	*ci;
	pat_elem		*p;

	pattern_index = index;

	if ( ( p = indextop ( index ) ) == NULL ) {
		fprintf ( progress, "read_input/ pattern index %d not found\n", index);
		fflush ( progress );
		return ( NOTOK );
	}

	input_size	= sys->net[c_net]->fanin;

	/* Read input_vector */

	switch ( p->input_pattern_type ) {

	case PATTERN_TYPE_FONT:
		if ( readchar ( p->input_file, p->input_character, input_vector, &ci ) != OK ) {
#ifdef	DEBUG_PATTERN
			fprintf ( progress, "read_input/ pattern \'%c\' not found in %s\n", p->input_character, p->input_file);
			fflush ( progress );
#endif
			return ( NOTOK );
		}
		p -> input_width = ci -> bbx.w ;
		p -> input_height = ci -> bbx.h ;

#ifdef	DEBUG_PATTERN
		fprintf ( progress, "read_input/  input  ( bdf ) pattern loaded from %16s \'%c\' [%dx%d]\n", p->input_file, p->input_character, p->input_width, p->input_height);
		fflush ( progress );
#endif
		break;

	case PATTERN_TYPE_FLOAT:
	case PATTERN_TYPE_INT:
		if ( read_floatfile( p->input_file, input_size, input_vector, xwh ) == NOTOK ) {
			return ( NOTOK );
		}
		p -> input_width = xwh[0];
		p -> input_height =xwh[1];
#ifdef	DEBUG_PATTERN
		fprintf ( progress, "read_input/  input  (float) pattern loaded from %16s     [%dx%d]\n", p->input_file, p->input_width, p->input_height);
		fflush ( progress );
#endif
		break;

	case PATTERN_TYPE_X:
	default:
		if ( read_xfile( p->input_file, input_size, input_vector, xwh ) == NOTOK ) {
			return ( NOTOK );
		}
		p -> input_width = xwh[0];
		p -> input_height =xwh[1];
#ifdef	DEBUG_PATTERN
		fprintf ( progress, "read_input/  input  (  X  ) pattern loaded from %16s     [%dx%d]\n", p->input_file, p->input_width, p->input_height);
		fflush ( progress );
#endif
		break;
	}

	/* Clamp input_vector */

	for ( i=0; i < input_size; i++ ) {
		*(float *)sys->net[c_net]->input_port[i] = input_vector[i];
	}
	represent( sys->net[c_net]->input_port, input_size, input_pattern_control );
	return ( OK );
}

/* ------------------------------------------------------------------------ */
int read_target ( index )
int	index;
{
	int			i, target_size;
	int			xwh[2];
	struct  character	*ct;
	pat_elem		*p;

	pattern_index = index;

	if ( ( p = indextop ( index ) ) == NULL ) {
		fprintf ( progress, "read_target/ pattern index %d not found\n", index);
		fflush ( progress );
		return ( NOTOK );
	}

	target_size	= sys->net[c_net]->fanout;

	if ( ( strcmp ( p->input_file, p->target_file ) == 0 ) && ( p->input_character == p->target_character ) ) {

		/* assume that fanout and fanin match : clamp input_vector to target */

#ifdef	DEBUG_PATTERN
		fprintf ( progress, "read_target/ target pattern is the same as input_pattern\n");
		fflush ( progress );
#endif
		p -> target_width  = p -> input_width;
		p -> target_height = p -> input_height;

		for ( i=0; i < target_size; i++ ) {
			*(float *)sys->net[c_net]->target[i] = input_vector[i];
		}
		represent( sys->net[c_net]->target, target_size, target_pattern_control );
	}
	else {
		/* Read target_vector */

		switch ( p->target_pattern_type ) {

		case PATTERN_TYPE_FONT:
			if ( readchar ( p->target_file, p->target_character, target_vector, &ct ) != OK ) {
#ifdef	DEBUG_PATTERN
				fprintf ( progress, "read_target/ pattern \'%c\' not found in %s\n", p->target_character, p->target_file);
				fflush ( progress );
#endif
				return ( NOTOK );
			}
			p -> target_width = ct -> bbx.w ;
			p -> target_height = ct -> bbx.h ;

#ifdef	DEBUG_PATTERN
			fprintf ( progress, "read_target/ target ( bdf ) pattern loaded from %16s \'%c\' [%dx%d]\n", p->target_file, p->target_character, p->target_width, p->target_height);
			fflush ( progress );
#endif
			break;

		case PATTERN_TYPE_FLOAT:
		case PATTERN_TYPE_INT:
			if ( read_floatfile( p->target_file, target_size, target_vector, xwh ) == NOTOK ) {
				return ( NOTOK );
			}
			p -> target_width = xwh[0];
			p -> target_height =xwh[1];
#ifdef	DEBUG_PATTERN
			fprintf ( progress, "read_target/ target (float) pattern loaded from %16s     [%dx%d]\n", p->target_file, p->target_width, p->target_height);
			fflush ( progress );
#endif
			break;

		case PATTERN_TYPE_X:
		default:
			if ( read_xfile( p->target_file, target_size, target_vector, xwh ) == NOTOK ) {
				return ( NOTOK );
			}
			p -> target_width = xwh[0];
			p -> target_height =xwh[1];
#ifdef	DEBUG_PATTERN
			fprintf ( progress, "read_target/ target (  X  ) pattern loaded from %16s     [%dx%d]\n", p->target_file, p->target_width, p->target_height);
			fflush ( progress );
#endif
			break;
		}

		/* Clamp target_vector */
		for ( i=0; i < target_size; i++ ) {
			*(float *)sys->net[c_net]->target[i] = target_vector[i];
		}
		represent( sys->net[c_net]->target, target_size, target_pattern_control );
	}
	return ( OK );
}

/* ------------------------------------------------------------------------ */
pat_elem	*indextop ( index )	/* look for index'th entry in list */
int		index;
{
	int		i = 0;
	pat_elem	*p = pattern_list;

	while ( i < index && p != NULL) {
		i++;
		p = p->next_ptr;
	}
	return ( p );
}
/* ------------------------------------------------------------------------ */
int read_xfile ( filename, size, vector, xwh )
char	filename[];
int	size;
int	xwh[];
float	vector[];
{
	FILE			*fildes;
	int			i;
	char			c;

	if (!(fildes = fopen(filename, "r"))) {
		fprintf ( progress, "read_xfile/ can't open the pattern file %s\n", filename);
		fflush ( progress );
		return( NOTOK );
	}

	xwh[ 0 ]=0;
	xwh[ 1 ]=0;

	i = 0;
	while ( i < size ) {
		if ((c = fgetc(fildes)) == EOF) {
			c = ' ';
		}

		switch (c) {

		case '#':
			while ( ( c = (char) fgetc(fildes) ) && !feof(fildes) ) {
				if ( c == '\n' )
					break;
			}
			break;
	
		case '\n':
			if (xwh[ 0 ] == 0) {
				xwh[ 0 ] = i;
			}
			xwh[ 1 ]++;
			break;

		case '\t':	/* tab equivalent to space */
		case ' ':
			vector [ i ] = 0.0 ;
			i++;
			break;

		case 'x':
		case 'X':
			vector [ i ] = 1.0 ;
			i++;
			break;

		default:
			vector [ i ] = 0.5 ;
			i++;
			break;
		}
	}
	xwh[ 1 ]++;		/* loop terminates at size entries - misses last newline */
	if ( xwh[ 0 ] == 0 ) {
		xwh[ 0 ] = i;	/* if all data on a single line */
	}
	fclose(fildes);
	return( OK );
}

/* ------------------------------------------------------------------------ */
int read_floatfile ( filename, size, vector, xwh )
char	filename[];
int	size;
int	xwh[];
float	vector[];
{
	FILE			*fildes;
	int			i,found;
	float			f;
	char			c;

	if (!(fildes = fopen(filename, "r"))) {
		fprintf ( progress, "read_floatfile/ can't open the pattern file %s\n", filename);
		fflush ( progress );
		return( NOTOK );
	}

	xwh[ 0 ]=0;
	xwh[ 1 ]=0;

	i = 0;
	found = 0;	/* count of actual values */
	while ( i < size ) {
		if ((c = fgetc(fildes)) == EOF) {
			c = ' ';
			i++;
		}

		switch (c) {

		case '#':
			while ( ( c = (char) fgetc(fildes) ) && !feof(fildes) ) {
				if ( c == '\n' )
					break;
			}
			break;
	
		case '\n':
			if (xwh[ 0 ] == 0) {	/* xwh[0] gets count from first line */
				xwh[ 0 ] = i;
			}
			xwh[ 1 ]++;
			break;

		case '\t':	/* tab equivalent to space */
		case ' ':
			break;

		default:
			ungetc(c,fildes);
			fscanf( fildes, "%f", &vector [ i++ ] );
			found++;
		}
	}
	if ( found == size ) {
		xwh[ 1 ]++;	/* loop terminates at size entries - misses last newline */
	}
	if ( xwh[ 0 ] == 0 ) {
		xwh[ 0 ] = i;	/* if all data on a single line */
	}
	fclose(fildes);
	return( OK );
}

/* ------------------------------------------------------------------------ */
int	represent( port, size, pattern_control )
float	**port;
int	size, pattern_control;
{
	int	i;
	float	**f;
	float	sum;

	if ( pattern_control == REAL ) {
		return( 0 );
	}
	else if ( pattern_control == NORMALISE ) {
		/* rescale weights so that sum of squares = 1.0 */
		
		for ( sum = 0.0, f = port, i=0; i< size; i++, f++ ) {
			sum += **f * **f;
		}
		for ( f = port, i=0; i< size; i++, f++ ) {
			**f = **f/sum;
		}
	}
	else {
		for ( f = port, i=0; i< size; i++, f++ ) {

			switch ( pattern_control ) {

			case BINARY:			/* 1.0 or 0.0 */
				if ( **f > 0.5 )
					**f = 1.0;
				else
					**f = 0.0;
				break;

			case PLUSMINUS:			/* return 1.0 or -1.0 */
				if ( **f > 0.5 )
					**f = 1.0;
				else
					**f = -1.0;
				break;

			case SCALED_SIGMOID:	/* scale using sigmoid function */
				**f = 1.0 / ( 1.0 + exp ( -**f ) );
				break;

			default:
				break;
			}
		}
	}
	return ( 0 );
}
/* ------------------------------------------------------------------------ */
