/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * built_in_fn.c
 ******************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <sys/wait.h>
#include <varargs.h>
#include "pygmalion.h"
#include "sysdef.h"

/*  Externals  */
extern	int	error();
extern	double	drand48();
extern	long	lrand48();

/*  Global variables  */
int	err_cnt	= 0;	  	/* error count kept by error() */

char	*jalloc();	/* accumulating calloc call for statistics 	*/

/* ------------------------------------------------------------------------ *\
		rule_init
\* ------------------------------------------------------------------------ */

/* int rule_init (name, rule, ruleclass, n_extend, arg1, arg2...)*/
int rule_init (va_alist)
va_dcl
{
	va_list		ap;

	int		i, size, n_extend;
	char		*name;
	rule_type	*rule;
	class_type	*class;
	caddr_t		ptr;

	va_start( ap );
	name	= va_arg( ap, char * );
	rule	= va_arg( ap, rule_type * );
	class	= va_arg( ap, class_type * );
	n_extend= va_arg( ap, int );

	rule->name	= name;
	rule->class	= class;

	/***  allocate memory for generic parameter pointers and extended parameter pointers ***/

	size = class->n_generic_parameters + 1 + ( class->n_extend_parameters * n_extend );
	rule->para_list = (caddr_t *) jalloc( sizeof( caddr_t ), size );

	for( i = 0; i < class->n_generic_parameters; i++) {
		if ( ( rule->para_list[ i ] = va_arg( ap, caddr_t ) ) == ( caddr_t ) EOP ) {
			printf("rule_init - too few generic parameters : %d expected in %s\n",
				class->n_generic_parameters,  rule->name);
			return ( NOTOK );
		}
	}
	if ( ( ptr = va_arg( ap, caddr_t ) ) != ( caddr_t ) EOP ) {
		printf("rule_init - too many generic parameters : %d expected in %s\n",
			class->n_generic_parameters,  rule->name);
		return ( NOTOK );
	}

	/* reserve space for extend counter - and point to it */
	rule->para_list[ i ] = (caddr_t ) jalloc( sizeof( exec_type ), 1 );
	( *(exec_type *)rule->para_list[ i ] ).size = 0;		/* initialise to zero */
	( *(exec_type *)rule->para_list[ i ] ).rule = rule;		/* point to rule */

	va_end( ap );

	return( 0 );
}

/* ------------------------------------------------------------------------ *\
		rule_extend

		assign the actual data access pointers to the places 
		in the pointer list

\* ------------------------------------------------------------------------ */

/* int rule_extend (rule, order, extend1, extend2...) */
int rule_extend (va_alist)
va_dcl
{
	va_list		ap;
	int		order, i, j;
	rule_type	*rule;
	caddr_t		ptr;

	va_start( ap );

	rule	= va_arg( ap, rule_type * );
	order	= va_arg( ap, int);

	for(	i=0, j = rule->class->n_generic_parameters + 1 + ( rule->class->n_extend_parameters * order );
		i< rule->class->n_extend_parameters;
		i++, j++ ) {
		
		if ( ( rule->para_list[ j ] = va_arg( ap, caddr_t ) ) == ( caddr_t ) EOP ) {
			printf("rule_extend - too few extension parameters : %d expected in %s\n",
				rule->class->n_extend_parameters,  rule->name);
			return ( NOTOK );
		}
	}
	if ( ( ptr = va_arg( ap, caddr_t ) ) != ( caddr_t ) EOP ) {
		printf("rule_extend - too many extension parameters : %d expected in %s\n",
			rule->class->n_extend_parameters,  rule->name);
		return ( NOTOK );
	}

	( *(int *)rule->para_list[ rule->class->n_generic_parameters ])++;	/* increment extend counter */

	va_end( ap );
	return( 0 );
}


/* ------------------------------------------------------------------------ *\
		EXEC

		execute a rule

\* ------------------------------------------------------------------------ */

int EXEC (rule)
rule_type	*rule;
{
#ifdef	DEBUG_EXEC
	static	int		count = 1;
	static	char		last_name[ 200 ] = "";

	if (  strcmp (rule->name, last_name) == 0 ) {
		count++;
	}
	else {
		printf ( "%d\n", count );
		count = 1;
		strcpy (last_name, rule->name );
		printf ( "EXEC\t%s\t", rule->name );
	}
#endif
	return(rule->class->fn(rule->para_list));	/*** apply the function to the parameters ***/
}

/* ------------------------------------------------------------------------ *\
		sexec

		sequential/conditional execution of a list of rules

		n_generic_parameters = 0
\* ------------------------------------------------------------------------ */

int sexec (rule_list)
caddr_t		*rule_list;
{
	int		i=0, count, status;
	exec_type	*e = (exec_type *) (*rule_list++);

	count = e->size * e->rule->class->n_extend_parameters;

	while ((i<count) && ((status = EXEC (rule_list[i++]) ) != TERM)) {
		i += status;	/* skip a number of following ops */
	}
	return( status );	/* status == TERM would terminate the rule execution */
}

/* ------------------------------------------------------------------------ *\
		sexec_r

		repeating sequential/conditional execution of a list of rules

\* ------------------------------------------------------------------------ */

int sexec_r (rule_list)
caddr_t		*rule_list;
{
	int		i, count, status, loop=0;
	exec_type	*e = (exec_type *) (*rule_list++);

	count = e->size * e->rule->class->n_extend_parameters;

	for (;;) {	/* status == TERM terminates the rule execution */

		i = 0;
		loop++;
#ifdef	DEBUG_EXEC_R
		printf ( "sexec_r/ loop = %d\n", loop );
#endif
		while ((i<count) && ((status = EXEC (rule_list[i++]) ) != TERM)) {
			i += status;	/* skip a number of following ops */
		}
		if ( status == TERM ) {
			break;
		}
	}
	return( status );
}
/* ------------------------------------------------------------------------ *\
		sexec_c

		repeating execution of a list of rules - without propagating
		termination code up to calling procedure

\* ------------------------------------------------------------------------ */

int sexec_c (rule_list)
caddr_t		*rule_list;
{
	sexec_r (rule_list);
	return ( 0 );
}

/* ------------------------------------------------------------------------ *\
		pexec

		parallel execution of a list of rules

		n_generic_parameters = 0
\* ------------------------------------------------------------------------ */

int pexec (rule_list)
caddr_t		*rule_list;
{
	exec_type	*e = (exec_type *) (*rule_list++);
	int		count = e->size * e->rule->class->n_extend_parameters;

	return (p_control(count, rule_list));
}

/* ------------------------------------------------------------------------ *\
		pexec_r

		repeating parallel execution of a list of rules
\* ------------------------------------------------------------------------ */

int pexec_r (rule_list)
caddr_t		*rule_list;
{	
	exec_type	*e = (exec_type *) (*rule_list++);
	int		count = e->size * e->rule->class->n_extend_parameters;
	int		status, loop=0;

	while ( TRUE ) {
		loop++;
#ifdef	DEBUG_EXEC_R
		printf ( "pexec_r/ loop = %d\n", loop );
#endif
		 if ( ( status = p_control(count, rule_list)) == TERM ) {
			break;
		 }
	}
	return ( status );
}

/* ------------------------------------------------------------------------ *\
		pexec_c

		repeating parallel execution of a list of rules - without
		propagating termination code up to calling procedure

\* ------------------------------------------------------------------------ */

int pexec_c (rule_list)
caddr_t		*rule_list;
{
	pexec_r (rule_list);
	return ( 0 );
}

/* ------------------------------------------------------------------------ *\
		p_control

		simplified implementation of pexec() in C
\* ------------------------------------------------------------------------ */

int p_control (count, rule_list)
int		count;
caddr_t		*rule_list;
{
	while(count--) {
		if ( EXEC (*rule_list++) ) {
			return(TERM);
		}
	}
	return( 0 );
}

/* ------------------------------------------------------------------------ *\
		rand_exec

		select rules at random from list and EXEC

		if repeat <= 0	continue indefinitely
		else EXEC repeat times

\* ------------------------------------------------------------------------ */

int rand_exec (repeat, rule_list)
int		repeat;
caddr_t		*rule_list;
{
	int	status, i;

	if ( repeat <= 0 ) {
		while ( ( status = rand_exec_once( rule_list ) ) != TERM) {
		}
	}
	else {
		for ( i = 0; i < repeat; i++ ) {
			if ( ( status = rand_exec_once( rule_list ) ) != TERM) {
				break;
			}
		}
	}
	return ( status );
}

/* ------------------------------------------------------------------------ *\
		rand_exec_once
		select one rule at random from list and EXEC
\* ------------------------------------------------------------------------ */

int rand_exec_once (rule_list)
caddr_t		*rule_list;
{
	int	i, count = *(int *)(*rule_list);

 	i = lrand48() % count;

#ifdef	DEBUG_EXEC_R
		printf ( "rand_exec_once  i=%d count=%d\n", i, count );
#endif

	return ( EXEC( rule_list[ i + 1 ] ) );
}

/* ------------------------------------------------------------------------ *\
		rand_span

		EXEC all rules in list once each in random sequence

\* ------------------------------------------------------------------------ */

int rand_span (rule_list)
caddr_t		*rule_list;
{
	int	i, count = *(int *)(*rule_list);

 	i = lrand48() % count;	/* first one */

	/*	now call each once starting at ith rule */

	return ( 0 );
}

/* ------------------------------------------------------------------------ *\
		dp

		dot product
\* ------------------------------------------------------------------------ */
float dp (p)
TAGVAL	**p;	  /* [SIZE, x1, y1, ... ]	 */
{
	int	i, size;
	float	acc = 0.0;

	size = *(int *)p[ 0 ];

	for (i=1; i<2*size+1; i+=2) {
		acc += p[ i ]->value.f * p[ i + 1 ]->value.f;
	}
	return( acc );
}

/* ------------------------------------------------------------------------ *\
		norm_rand_syn
\* ------------------------------------------------------------------------ */
float norm_rand_syn (syn, size, scale)
synapse_type	**syn;
float		scale;
int		size;
{
	int i;
	float norm;

	extern void rand_syn();
	extern float Syn_Norm();

	if (scale <= 0.0) {
		for (i=0; i<size; i++) {
			syn[i]->weight.value.f = 0.0;
		}
		return(0.0);
 	}

	rand_syn(syn, size);

	while ((norm = Syn_Norm(syn, size)) <= 0.0); {
		norm /= scale;
	}
	for (i=0; i<size; i++) {
		syn[i]->weight.value.f /= norm;
	}
	return( norm );
}

/* ------------------------------------------------------------------------ *\
		rand_syn
\* ------------------------------------------------------------------------ */
void rand_syn (syn, size)
synapse_type	**syn;
int		size;
{
	register	float	r;

	while(size--) {
		r = (float) (drand48()/10000000.0);
		(*syn)->weight.value.f = r - (float)((int) r);
		syn++;
	}
}
/* ------------------------------------------------------------------------ *\
		Syn_Norm
\* ------------------------------------------------------------------------ */
float Syn_Norm (syn, size)
synapse_type	**syn;
int		size;
{
	register float norm = 0.0;

	while (size--) {
		norm += (*syn)->weight.value.f * (*syn)->weight.value.f;
		syn++;
	}
	return((float) sqrt((double) norm));
}

/* ------------------------------------------------------------------------ *\
		error
\* ------------------------------------------------------------------------ */
int error (msg)	      /* Print error message */
char	*msg;
{
	fprintf(stderr, "%s\n", msg);
	++err_cnt;
	return ( NOTOK );
}

/* ------------------------------------------------------------------------ *\
		max_err_cal
\* ------------------------------------------------------------------------ */
float max_err_cal (p)
TAGVAL **p;	/* [*count, *pattern1, *state1, ... ] */
{
	register float	err, max_err = 0.0;
	register int	count;

	count = *(int *)(*p++);

	while (count--) {

		err =  (*p++)->value.f;
		err -= (*p++)->value.f;
		if (err<0.0) {
        		err *= -1;
		}
		if (err > max_err) {
        		max_err = err;
		}
  	}
	return( max_err );
}


/* ------------------------------------------------------------------------ *\
		ham_dis_cal
\* ------------------------------------------------------------------------ */
float ham_dis_cal (p)
TAGVAL **p;	/* [*count, *pattern1, state1, ... ] */
{
	/* count how many state and pattern values are on opposite sides of 0.5 */

	register float	dif = 0.0;
	register int	count;

	count = *(int *)(*p++);

	while (count--) {
		if ( (*p++)->value.f < 0.5) {
			if ( (*p++)->value.f >= 0.5) {
				dif += 1.0;
			}
     		}
		else {
			if ( (*p++)->value.f < 0.5) {
				dif += 1.0;
			}
		}
	}
	return( dif );
}

/* ------------------------------------------------------------------------ *\
		eucl_dis_cal
\* ------------------------------------------------------------------------ */
float eucl_dis_cal (p)
TAGVAL **p;
{
	register float	eucl = 0.0, dif;
	register int	count;

	count = *(int *)(*p++);

	while (count--) {
		dif =  (*p++)->value.f;
		dif -= (*p++)->value.f;

		eucl += dif * dif;
 	}
	return((float)sqrt((double) eucl));
}

/* ------------------------------------------------------------------------ *\
		angl_cal
\* ------------------------------------------------------------------------ */
float angl_cal (p)
TAGVAL **p;
{
	register float	norm1 = 0.0, norm2 = 0.0, inp = 0.0, tf;
	float		cosineQ, alpha;
	register int	count;

	count = *(int *)(*p++);

	while (count--) {
		norm1	+=	(*p)->value.f * (*p)->value.f;
		tf	=	(*p++)->value.f;
		norm2	+=	(*p)->value.f * (*p)->value.f;
		inp	+=	tf * (*p++)->value.f;
  	}

	cosineQ = inp/(float) sqrt((double) (norm1 * norm2));
	alpha = (float) acos((double) cosineQ);
	if (alpha < 0) {
		printf("arc-cosine maybe negative\n");
		alpha *= -1.0;
  	}

	return(alpha);
}

/* ------------------------------------------------------------------------ */
char	*jalloc(e,n)	/* version of calloc() adding memory usage */
int	e,n;
{
	static	long	memtot = 0L;
	unsigned int	a;
	char	*m;

	a = e * n;
	memtot += a;

#ifdef	DEBUG_JALLOC
  	printf ( "%d * %d = %d, total %ld\n", e,n,a,memtot);
#endif	

	m = (char *) calloc(e,n);
	if ( m == NULL ) {
		printf ( "calloc returns NULL pointer\n");
		exit(1);
	}
	return m;
}

/* --------------------------------------------------------------------------- */
