/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * bp.c
 ******************************************************************************/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>

#include "pygmalion.h"
#include "sysdef.h"
#include "bpconfig.h"
#include "pgmrc.h"

#include "bpdef.h"
#include "built_in_fn.h"
#include "util.h"
#include "fontdefs.h"

/* --------------- functions defined for this algorithm ------------------- */
int	connect ();
int	build_rules ();

float	learn ();
float	step_learn ();
float	recall ();

/* --------------- globals ------------------------------------------------ */
int		c_net = 0;				/* current net */
int		I_LAYER, O_LAYER;			/* input, output layers */
char		system_filename[ FILESIZE ];

float		*input_vector;				/* to receive input pattern */
float		*target_vector;				/* to receive target pattern */

int		input_pattern_control	= INPUT_PATTERN_CONTROL; /* may be overridden from parameter NET_P_input_control */
int		target_pattern_control	= INPUT_PATTERN_CONTROL; /* may be overridden from parameter NET_P_target_control */

/* ------------------------------------------------------------------------ */
/*   Definions of the back-propagation rule classes etc.                    */
/* ------------------------------------------------------------------------ */

/*  low level functions for the back-propagation model  */

int	state_update();
int	err_cal_output();
int	err_cal_hidden();
int	weight_update();
int	tolerance();

/* class_type structures		    function		fn_name			gen.	ext. */

class_type learn_meta_class		= { sexec_r,		"sexec_r",		0,	1 };
class_type step_learn_meta_class	= { sexec,		"sexec",		0,	1 };
class_type recall_meta_class		= { sexec,		"sexec",		0,	1 };
class_type weight_update_meta_class	= { pexec,		"pexec",		0,	1 };
class_type state_upd_meta_class		= { pexec,		"pexec",		0,	1 };
class_type err_cal_meta_class		= { pexec,		"pexec",		0,	1 };
class_type err_cal_output_class		= { err_cal_output,	"err_cal_output",	3,	2 };
class_type state_upd_class		= { state_update,	"state_update",		2,	2 };
class_type err_cal_hidden_class		= { err_cal_hidden,	"err_cal_hidden",	2,	2 };
class_type weight_update_class		= { weight_update,	"weight_update",	2,	2 };
class_type tol_test_class		= { tolerance,		"tolerance",		3,	2 };

/* ------ specify maximum counts of rules and parameters at all levels ---- */

int	xc	[ 6 ][ 2 ] = {	
/* extra counts rules,	params	   level	*/
		0,	0,	/* system	*/
		5,	6,	/* net		*/
		3,	1,	/* layer	*/
		3,	0,	/* cluster	*/
		3,	0,	/* neuron	*/
		0,	0	/* synapse	*/
};

/* ------ names for parameters -------------------------------------------- */

char	*pname [ 6 ] [ 6 ] = {
		{"","","","","",""},
		{"score", "tolerance", "learn_rate", "measure", "input_control", "target_control"},
		{"neurons","","","","",""},
		{"","","","","",""},
		{"","","","","",""},
		{"","","","","",""}
};
/* ------------------------------------------------------------------------ */


/* ------------------------------------------------------------------------ */
/*	external functions and variables                                    */
/* ------------------------------------------------------------------------ */

/* -------- declarations from pgmrc.c ------------------------------------- */

extern	char		*rc [];

/* -------- declarations from pattern.c ----------------------------------- */

extern	int		init_patterns();
extern	int		count_patterns();
extern	int		read_pattern();
extern	pat_elem	*load_patterns();
extern	pat_elem	*indextop();
extern	void		add_to_pattern_list();
extern	int		read_input();
extern	int		read_target();
extern	int		read_xfile();

extern	pat_elem	*pattern_list;		/* patterns loaded for net */
extern	pat_elem	*current_pattern;	/* to pass widths to show_net */
extern	int		pattern_count;		/* how many */
extern	char		pattern_filename[];

/* -------- declarations from jload.c ------------------------------------- */

extern	int		sys_load();

/* -------- declarations from jsave.c ------------------------------------- */

extern	int		sys_save();

/* -------- declarations from shownet.c ----------------------------------- */

void			shownet();

/* -------- declarations from util.c -------------------------------------- */

extern system_type	*sys;
extern int		int_mode;
extern int		cycles;
extern float		rand_scale;
extern	char		*strstr();

/* -------- declarations from built_in_fn.c ------------------------------- */

extern	char		*jalloc();
extern	int		sexec();
extern	int		sexec_r();
extern	int		pexec();
extern	int		pexec_r();
extern	float		max_err_cal();
extern	float		ham_dis_cal();
extern	float		eucl_dis_cal();
extern	float		angl_cal();

extern int		err_cnt;

/* -------- declarations from alloc.c ------------------------------------- */

extern	system_type	*sys_alloc();

/* ------------------------------------------------------------------------ */
/*  Function bodies for the back-propagation rule classes                   */
/* ------------------------------------------------------------------------ */

float learn()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_learn ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
float step_learn()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_step_learn ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */

float recall()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_recall ]);
	EXEC(&sys->net[c_net]->rules[ NET_R_tol_test ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
/* state_upd_class */

int state_update (p)
TAGVAL **p;        /* [*state, *acc, *SIZE, *state1, *weight1, ... ] */
{
	p[ 1 ]->value.f = dp( p + 2 );
	p[ 0 ]->value.f = 1.0 / (1.0 + exp(- p[ 1 ]->value.f));
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* err_cal_output_class  */

int err_cal_output (p)
TAGVAL **p;        /*  [*err, *target, *state] */
{
	p[ 0 ]->value.f = (p[ 1 ]->value.f - p[ 2 ]->value.f) * (1 - p[ 2 ]->value.f) * (p[ 2 ]->value.f);
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* err_cal_hidden_class */

int err_cal_hidden (p)
TAGVAL **p;        /* [*err, *state, *SIZE, *error1, *weight1, ... ] */
{
	p[ 0 ]->value.f = dp( p + 2 ) * (1 - p[ 1 ]->value.f) * (p[ 1 ]->value.f);
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* weight_update_class */

int weight_update (p)
TAGVAL **p;        /* [*error, *learn_rate, *SIZE, *weight1, *state1, ... ] */
{
	int	i, size;

	size = *(int *)p[ 2 ];

	PAR for (i = 3; i < ( 2 * size + 3); i += 2) {
		p[ i ]->value.f += (p[ 1 ]->value.f) * (p[ 0 ]->value.f) * (p[ i + 1 ]->value.f);
	}
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* tolerance_class */

int tolerance (p)
TAGVAL **p;	/* [*tolerance, *net.score, *net.measure, *SIZE, ... ]        */
{
	int	size, control;
	float	tol, *score;

	tol	= (*p++)->value.f;
	score	= &(*p++)->value.f;
	control	= (int) (*p++)->value.f;


	switch (control) {

	case MERR:
		*score = max_err_cal(p);
#ifdef	DEBUG_TOLERANCE
		printf ( "Tolerance test MERR : score %f tolerance %f\n", *score, tol );
#endif
		if ( *score < tol ) {
			return( TERM );
		}
		break;

	case HAM:
		size = *(int *)*p;
		*score = ham_dis_cal(p) / (float) size;
#ifdef	DEBUG_TOLERANCE
		printf ( "Tolerance test HAM : score %f tolerance %f\n", *score, tol );
#endif
		if ( *score < tol ) {
			return( TERM );
		}
		break;

	case EUCL:
		size = *(int *)*p;
		*score = eucl_dis_cal(p) / (float) size;
#ifdef	DEBUG_TOLERANCE
		printf ( "Tolerance test EUCL : score %f tolerance %f\n", *score, tol );
#endif
		if ( *score < tol ) {
			return( TERM );
		}
		break;

	case ANGL:
		*score = angl_cal(p);
#ifdef	DEBUG_TOLERANCE
		printf ( "Tolerance test ANGL : score %f tolerance %f\n", *score, tol );
#endif
		if ( *score < tol ) {
			return( TERM );
		}
		break;

	default:
		error("Illegal tolerance test control [%d]\n", control);
		return( NOTOK );
	}

	return( OK );
}

/* ------------------------------------------------------------------------ */

/* back propagation connect */

int connect (conf)
int	conf[];
{
	int		cn, i, j, k, l, m, p, s, x, y, index;
	register	int io_size;
	caddr_t		*free_pointer1, *free_pointer2;
	rule_type	*rp;
	para_type	*pp;
	synapse_type	*sp;

	sys = sys_alloc(&conf);		/** allocates memory for networks and neurons **/

	/* allocate memory for the rules, meta rules and parameters */

	for ( x = SYS; x < SYN; x++ ) {

		y = xc [x] [0];
		if ( y ) {	/* rules */

			s = sizeof (rule_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_rules = y;
					sys->rules = (rule_type *) jalloc (s, y);
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_rules = y;
						sys->net[cn]->rules = (rule_type *) jalloc (s, y);
					}
					break;

				/***	Nota bene  - loop starts at layer [ 1 ]  ***/
				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					rp = (rule_type *) jalloc ( s, y * ( sys->net[cn]->layers - 1 ) );
					for (i=1; i < sys->net[cn]->layers; i++, rp += y) {
						sys->net[cn]->layer[i]->n_rules = y;
						sys->net[cn]->layer[i]->rules = rp;
					}
					}

				/***	Nota bene  - loop starts at layer [ 1 ]  ***/
				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=1; i < sys->net[cn]->layers; i++) {
					rp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, rp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->rules = rp;
					}
					}
					}

				/***	Nota bene  - loop starts at layer [ 1 ]  ***/
				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=1; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					rp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, rp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules = rp;
					}
					}
					}
					}

				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid count\n");
					exit(1);
					break;
			}
		}

		y = xc [x] [1];
		if ( y ) {	/* Parameters */

			s = sizeof (para_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_parameters = y;
					sys->parameters = (para_type *) jalloc (s, y);
					for ( p = 0; p < y; p++ ) {
						sys->parameters[p].name = pname [x] [p];
 					}
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_parameters = y;
						sys->net[cn]->parameters = (para_type *) jalloc (s, y);
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->parameters[p].name = pname [x] [p];
	 					}
					}
					break;

				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layers);
					for (i=0; i < sys->net[cn]->layers; i++, pp += y) {
						sys->net[cn]->layer[i]->n_parameters = y;
						sys->net[cn]->layer[i]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					break;

				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					break;

				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					}
					break;
			
				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid count\n");
					exit(1);
					break;
			}
		}
	}

	for ( cn = 0; cn < sys->nets; cn++ ) {

	sys->net[cn]->parameters[ NET_P_tolerance ].parameter.value.f		= (float) TOLERANCE;
	sys->net[cn]->parameters[ NET_P_learn_rate ].parameter.value.f		= (float) LEARN_RATE;
	sys->net[cn]->parameters[ NET_P_measure ].parameter.value.f		= (float) MEASURE;
	sys->net[cn]->parameters[ NET_P_input_control ].parameter.value.f	= (float) INPUT_PATTERN_CONTROL;
	sys->net[cn]->parameters[ NET_P_target_control ].parameter.value.f	= (float) TARGET_PATTERN_CONTROL;

	/********* allocating the output, teaching and input pattern pipe pointers ********/

	I_LAYER = 0;				/* Input layer is layer 0 */
	O_LAYER = sys->net[cn]->layers - 1;	/* Output layer is layers - 1 */

	io_size = 0;
	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
		io_size += sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons;
	}
	sys->net[cn]->fanout = io_size;
	sys->net[cn]->output_port	= ( caddr_t * )jalloc (sizeof( caddr_t ), io_size);
	sys->net[cn]->target		= ( caddr_t * )jalloc (sizeof( caddr_t ), io_size);

	io_size = 0;
	for (i=0; i<sys->net[cn]->layer[I_LAYER]->clusters; i++) {
		io_size += sys->net[cn]->layer[I_LAYER]->cluster[i]->neurons;
	}
	sys->net[cn]->fanin = io_size;
	sys->net[cn]->input_port		= ( caddr_t * )jalloc (sizeof( caddr_t ), io_size);

	input_vector	= (float *) jalloc( sizeof (float), sys->net[cn]->fanin );	/* to receive input pattern */
	target_vector	= (float *) jalloc( sizeof (float), sys->net[cn]->fanout );	/* to receive input pattern */


	/**** build up the I/O pipe pointers ****/

	free_pointer1 = sys->net[cn]->output_port;
	free_pointer2 = sys->net[cn]->target;
	for (i=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
		for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {
			*free_pointer1++ = 
				(caddr_t) &sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ].value.f;
			*free_pointer2++ = 
				(caddr_t) &sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_TARGET ].value.f;
		}
	}
	free_pointer1 = sys->net[cn]->input_port;
	for (i=0; i<sys->net[cn]->layer[I_LAYER]->clusters; i++) {
		for (j=0; j<sys->net[cn]->layer[I_LAYER]->cluster[i]->neurons; j++) {
		        *free_pointer1++ = 
				(caddr_t) &sys->net[cn]->layer[I_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ].value.f;
		}
	}
	/********* set up the number of neurons in each layer *********/

	for (i=0; i < sys->net[cn]->layers; i++) {
		io_size = 0;
		for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
			io_size += sys->net[cn]->layer[i]->cluster[j]->neurons;
		}
		sys->net[cn]->layer[i]->parameters[ LAY_P_neurons ].parameter.value.f = (float) io_size;
	}

	/********* allocate synapses for the neurons *********/

	for (i=0; i < sys->net[cn]->layers; i++) {
		if (i != I_LAYER) {
			s = (int) sys->net[cn]->layer[i-1]->parameters[ LAY_P_neurons].parameter.value.f;
			for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
		        for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->fanin = s;
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->synapses = s;
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->synapse = 
					(synapse_type **) jalloc (sizeof(synapse_type *), s);
				sp = (synapse_type *) jalloc (sizeof(synapse_type), s);
				for (l = 0; l < s; l++) {
					sys->net[cn]->layer[i]->cluster[j]->neuron[k]->synapse[l] = sp++;
				}
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->input_neuron = 
					(neuron_type **) jalloc (sizeof(neuron_type *), s );
			}
			}
		}
		if (i != O_LAYER) {
			s = (int) sys->net[cn]->layer[i+1]->parameters[ LAY_P_neurons ].parameter.value.f;
			for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
		        for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->fanout = s;
				if (i != I_LAYER)
					sys->net[cn]->layer[i]->cluster[j]->neuron[k]->output_neuron = 
						(neuron_type **) jalloc (sizeof(neuron_type *), s);
			}
			}
		}
	}

	/****  connecting the neurons => putting the address in the ****/
	/****       neuron.input_neuron  for the forward path       ****/

	for (i=1; i<sys->net[cn]->layers; i++) {		/* NOTE from 1 */
	for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
	for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
		index = 0;
		for (l=0; l<sys->net[cn]->layer[i-1]->clusters; l++) {
		      for (m=0; m<sys->net[cn]->layer[i-1]->cluster[l]->neurons; m++) {
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->input_neuron[index++] =
					sys->net[cn]->layer[i-1]->cluster[l]->neuron[m];
		      }
		}
		if (index != sys->net[cn]->layer[i]->cluster[j]->neuron[k]->fanin) {
			error("Network topology specification inconsistent!\n");
			exit(3);
		}
	}
	}
	}

	/****  connecting the neurons => putting the address in the ****/
	/****       neuron.output_neuron  for the backward path      ****/

	for (i=1; i<sys->net[cn]->layers-1; i++) {
	for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
	for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
		index = 0;
		for (l=0; l<sys->net[cn]->layer[i+1]->clusters; l++) {
			for (m=0; m<sys->net[cn]->layer[i+1]->cluster[l]->neurons; m++) {
				sys->net[cn]->layer[i]->cluster[j]->neuron[k]->output_neuron[index++] =
					sys->net[cn]->layer[i+1]->cluster[l]->neuron[m];
			}
		}
		if (index != sys->net[cn]->layer[i]->cluster[j]->neuron[k]->fanout) {
			error("Network topology specification inconsistent!\n");
			exit(3);
		}
	}
	}
	}
		/* write route[] data */

		for (i=0; i<sys->net[cn]->layers; i++) {
		for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
		for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {
			sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[0] = cn;
			sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[1] = i;
			sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[2] = j;
			sys->net[cn]->layer[i]->cluster[j]->neuron[k]->route[3] = k;
		}
		}
		}
	}	/* end for cn ... */

	return( OK );
}

/* -------------------------------------------------------------- */
/*	build_rule.c                                              */
/* -------------------------------------------------------------- */


/*****	Function Declarations *****/

int build_rules()
{
	int cn, i, j, k;

	if (!sys || !sys->net || ((O_LAYER < 0)))
		return ( FAIL );

	for ( cn = 0; cn < sys->nets; cn++ ) {

	/****  Initialize rules in the hidden and output layers	 ****/

	for (i=1; i<sys->net[cn]->layers; i++) {
	for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
	for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {

		rule_init (
			"neuron.state_upd",
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_state_upd ],
			&state_upd_class,
			sys->net[cn]->layer[i]->cluster[j]->neuron[k]->fanin,
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_STATE ],
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_ACC ],
			EOP );

		if (i == O_LAYER) {
			rule_init (
				"neuron.err_cal",
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_err_cal ],
				&err_cal_output_class,
				0,
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_ERR] ,
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_TARGET ],
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_STATE ],
				EOP );
		}
		else {
			rule_init (
				"neuron.err_cal",
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_err_cal ],
				&err_cal_hidden_class,
				sys->net[cn]->layer[i+1]->cluster[j]->neurons,
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_ERR ],
				&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_STATE ],
				EOP );
		}
		rule_init (
			"neuron.weight_upd",
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules [NEU_R_weight_upd ],
			&weight_update_class,
			sys->net[cn]->layer[i-1]->cluster[j]->neurons,
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->state[ N_ERR ],
			&sys->net[cn]->parameters[ NET_P_learn_rate ].parameter,
			EOP );

	}		/* end for k ... */
	}		/* end for j ... */
	}		/* end for i ... */

	/*** extending rules -- to fill in the parameter pointers ***/

	/* NO policy for connecting clusters at present - assume only one */

	for (i=1; i<= O_LAYER; i++) {

		for (j=0; j<sys->net[cn]->layer[i]->cluster[0]->neurons; j++) {
		for (k=0; k<sys->net[cn]->layer[i-1]->cluster[0]->neurons; k++) {

			rule_extend (
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->rules[ NEU_R_state_upd ],
				k,
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->synapse[k]->weight,
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->input_neuron[k]->state[ N_STATE ],
				EOP );

			rule_extend (
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->rules[ NEU_R_weight_upd ],
				k,
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->synapse[k]->weight,
				&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->input_neuron[k]->state[ N_STATE ],
				EOP );

		}	/* end for k ... */
		}	/* end for j ... */

		if (i != O_LAYER) {

			for (j = 0; j<sys->net[cn]->layer[i]->cluster[0]->neurons; j++) {
			for (k=0; k<sys->net[cn]->layer[i+1]->cluster[0]->neurons; k++) {

				rule_extend (
					&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->rules[ NEU_R_err_cal ],
					k,
					&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->output_neuron[k]->state[ N_ERR ],
					&sys->net[cn]->layer[i]->cluster[0]->neuron[j]->output_neuron[k]->synapse[j]->weight,
					EOP );
	   		}
    			}

		}	/* endif */
	}		/* end for i ... */

	/*** Initialize the rules at the cluster level ***/

	for (i=1; i<sys->net[cn]->layers; i++) {
	for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {

		rule_init (
			"cluster.state_upd",
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_state_upd ],
			&state_upd_meta_class,
			sys->net[cn]->layer[i]->cluster[j]->neurons,
			EOP );

		rule_init (
			"cluster.err_cal",
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_err_cal ],
			&err_cal_meta_class,
			sys->net[cn]->layer[i]->cluster[j]->neurons,
			EOP );

		rule_init (
			"cluster.weight_upd",
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_weight_upd ],
			&weight_update_meta_class,
			sys->net[cn]->layer[i]->cluster[j]->neurons,
			EOP );
	}
	}

	/*** Extending rules at the cluster level ***/

	for (i=1; i<sys->net[cn]->layers; i++) {
	for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {
	for (k=0; k<sys->net[cn]->layer[i]->cluster[j]->neurons; k++) {

		rule_extend (
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_state_upd ],
			k,
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_state_upd ],
			EOP );

		rule_extend (
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_err_cal ],
			k,
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_err_cal ],
			EOP );

		rule_extend (
			&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_weight_upd ],
			k,
			&sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules[ NEU_R_weight_upd ],
			EOP );
	}
	}
	}

/*** Initialize the rules at the layer level ***/

	for (i=1; i<sys->net[cn]->layers; i++) {

		rule_init (
			"layer.state_upd",
			&sys->net[cn]->layer[i]->rules[ LAY_R_state_upd ],
			&state_upd_meta_class,
			sys->net[cn]->layer[i]->clusters,
			EOP );

		rule_init (
			"layer.err_cal",
			&sys->net[cn]->layer[i]->rules[ LAY_R_err_cal ],
			&err_cal_meta_class,
			sys->net[cn]->layer[i]->clusters,
			EOP );

 		rule_init (
			"layer.weight_upd",
			&sys->net[cn]->layer[i]->rules[ LAY_R_weight_upd ],
			&weight_update_meta_class,
			sys->net[cn]->layer[i]->clusters,
			EOP );
	}

	/*** Extending rules at the layer level ***/

	for (i=1; i<sys->net[cn]->layers; i++) {

		for (j=0; j<sys->net[cn]->layer[i]->clusters; j++) {

			rule_extend (
				&sys->net[cn]->layer[i]->rules[ LAY_R_state_upd ],
				j,
				&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_state_upd ],
				EOP );

			rule_extend (
				&sys->net[cn]->layer[i]->rules[ LAY_R_err_cal ],
				j,
				&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_err_cal ],
				EOP );

			rule_extend (
				&sys->net[cn]->layer[i]->rules[ LAY_R_weight_upd ],
				j,
				&sys->net[cn]->layer[i]->cluster[j]->rules[ CLU_R_weight_upd ],
				EOP );
    		}
	}


	/*** Initialize the rules at the net level ***/

	/*** Initialize the recall rule ***/

	rule_init (
		"net.recall",					/* recall() takes layers-1 steps to  */
		&sys->net[cn]->rules[ NET_R_recall ],	/* update the states of all layers  */
		&recall_meta_class,				/* except the input layer */
		sys->net[cn]->layers-1,
		EOP );

	/*** Initialize the tolerance rule ***/

	/*  tolerance is a ground rule which 		*/
	/*  calculates the Euclidean distance, Hamming	*/
	/*  distance or the maximum error of the	*/
	/*  output pattern according to the control	*/

	rule_init (
		"net.tol_test",				/*  control switch stored NET_P_measure */
		&sys->net[cn]->rules[ NET_R_tol_test ],	/*  result placed in NET_P_score	*/
		&tol_test_class,
		sys->net[cn]->fanout,
		&sys->net[cn]->parameters[ NET_P_tolerance ].parameter,
		&sys->net[cn]->parameters[ NET_P_score ].parameter,
		&sys->net[cn]->parameters[ NET_P_measure ].parameter, 
		EOP );

	/* Initialize the weight update meta_rule at net level */

	rule_init (
		"net.weight_upd",
		&sys->net[cn]->rules[ NET_R_weight_upd ],
		&weight_update_meta_class,
		sys->net[cn]->layers - 1,			/* except the input layer */
		EOP );


	/*** Initialize the learn rule ***/

	rule_init (
		"net.learn",					/* learning takes four steps: recall,     */
		&sys->net[cn]->rules[ NET_R_learn ],		/* tolerance test, error calculation      */
		&learn_meta_class,				/* and weight update, where weight update */
		2 + sys->net[cn]->layers,			/* is made in a meta rule and error       */
		EOP );						/* calculation are err_cal rules in each  */
								/* layer except the input layer           */

	/*** Initialize the step_learn rule ***/

	rule_init (
		"net.step_learn",
		&sys->net[cn]->rules[ NET_R_step_learn ],
		&step_learn_meta_class,
		2 + sys->net[cn]->layers,
		EOP );


	/*** Extend the recall rule ***/

	for (i=1; i<sys->net[cn]->layers; i++) {

		rule_extend (
			&sys->net[cn]->rules[ NET_R_recall ],
			i - 1,
			&sys->net[cn]->layer[i]->rules[ LAY_R_state_upd ],
			EOP );
	}

	/*** Extend the tolerance rule ***/

	for (i=0, k=0; i<sys->net[cn]->layer[O_LAYER]->clusters; i++) {
	for (j=0; j<sys->net[cn]->layer[O_LAYER]->cluster[i]->neurons; j++) {

		rule_extend (
			&sys->net[cn]->rules[ NET_R_tol_test ],
			k++,
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_TARGET ],
			&sys->net[cn]->layer[O_LAYER]->cluster[i]->neuron[j]->state[ N_STATE ],
			EOP );
	}
	}

	/* Extend the weight update rule at the net level */

	for (i=1; i<sys->net[cn]->layers; i++) {

		rule_extend (
			&sys->net[cn]->rules[ NET_R_weight_upd ],
			i - 1,
			&sys->net[cn]->layer[i]->rules[ LAY_R_weight_upd ],
			EOP );
	}

/* ------------------------------------------------------------------------ */
	/*** Extend the rule learn ***/

	/* This is a bit wasteful - learn and step_learn are practically the same */
	/* - but it will do for now ... */

	k=0;
	rule_extend (
		&sys->net[cn]->rules[ NET_R_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_recall ],	/* recall to evaluate the network output */
		EOP );

	rule_extend (
		&sys->net[cn]->rules[ NET_R_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_tol_test ],	/* calculate the maximum error of the output units */
		EOP );

	for (i=sys->net[cn]->layers-1; i>0; i--) {
		rule_extend (
			&sys->net[cn]->rules[ NET_R_learn ],
			k++,
			&sys->net[cn]->layer[i]->rules[ LAY_R_err_cal ], /* calculate the errors */
			EOP );
	}

	rule_extend (
		&sys->net[cn]->rules[ NET_R_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_weight_upd ],
		EOP );

/* ------------------------------------------------------------------------ */
	/*** Extend the rule step_learn ***/

	k=0;
	rule_extend (
		&sys->net[cn]->rules[ NET_R_step_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_recall ],	/* recall to evaluate the network output */
		EOP );

	rule_extend (
		&sys->net[cn]->rules[ NET_R_step_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_tol_test ],	/* calculate the maximum error of the output units */
		EOP );

	for (i=sys->net[cn]->layers-1; i>0; i--) {
		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_learn ],
			k++,
			&sys->net[cn]->layer[i]->rules[ LAY_R_err_cal ], /* calculate the errors */
			EOP );
	}

	rule_extend (
		&sys->net[cn]->rules[ NET_R_step_learn ],
		k++,
		&sys->net[cn]->rules[ NET_R_weight_upd ],
		EOP );

	}	/* end for cn ... */

	return ( OK );
}

/**********************  Main program ******************************************/
#ifndef	OMIT_MAIN
main (argc, argv)
int	argc;
char	*argv[];
{
	int		i, somefail, cycle;
	int		wh[4];
	float		result;
	pat_elem	*p;
	int		load_flags;

	srandom(0xd5a4793c);	/*  Initialize drand48  */
	srand48(random());

	/*  Construct the network */

	if ( rc_read () ) {
		printf ( "Problem with .pgmrc file\n");
		exit ( 1 );
	}
	printf ( "local_user %s\n", rc [ RC_local_user ] );
	printf ( "local_host %s\n", rc [ RC_local_host ] );

	if (argc == 1) {
		if (connect(&config))		/* use internal configuration defaults */
			exit(FAIL);
		if (build_rules())
			exit(FAIL);
	}
	else {
		strcpy( system_filename, argv[ 1 ] );
		/* load system file - if configuration not specified, 
		   connect() and build_rules() must be called here  */
		if ( ! ( load_flags = sys_load( system_filename ) ) & LD_CONFIG ) {
			if (connect(&config))	/* use internal configuration defaults */
				exit(FAIL);
			if (build_rules())
				exit(FAIL);
		}
	}

	/*	Set pattern_controls to loaded parameters.  This is
		necessary because for this algorithm, pattern types other
		than BINARY may be specified - and the default may be
		altered by editing the system_file	*/
	input_pattern_control = (int) sys->net[c_net]->parameters[ NET_P_input_control ].parameter.value.f;
	target_pattern_control = (int) sys->net[c_net]->parameters[ NET_P_target_control ].parameter.value.f;

	/*  Initialise patterns specified on command line */

	if ( argc > 2 ) {
		i = init_patterns ( argv [ 2 ] );
 		printf ( "init_patterns returns %d from %s\n", i, argv [ 2 ] );
	}

 	if ( pattern_list == NULL )	/* new system - randomise the weights */
		rand_weight();

	cycle = 0;

	while ( TRUE ) {
		cycle++;

		for ( i = 0; i < pattern_count; i++ ) {

			current_pattern = indextop ( i );	/* to gain access to pattern details */
			read_input	( i );
			read_target	( i );

			result = learn();

			printf ( "pattern [ %s \'%c\' ]\tlearn [ %f ]\n", current_pattern->input_file, current_pattern->input_character, result );
		}

		somefail = FALSE ;
		for ( i = 0; i < pattern_count; i++ ) {

			current_pattern = indextop ( i );
			read_input	( i );
			read_target	( i );

			result = recall ();

			current_pattern->score = result;

			printf ( "cycle [%d] pattern [ %s \'%c\' ]\trecall [ %f ]\n", cycle, current_pattern->input_file, current_pattern->input_character, result );
			if ( result > sys->net[c_net]->parameters[ NET_P_tolerance ].parameter.value.f) {
				somefail = TRUE;
			}
		}
		if ( somefail == FALSE )
			break;
		printf ( "\nat least one pattern fails tolerance test - re-learning\n");
		sys_save( system_filename );	/* may interrupt - last cycle is saved */
	}

	if ( argc > 3 ) {	/* may specify second test set in argv[ 3 ] */
		i = init_patterns ( argv [ 3 ] );
 		printf ( "init_patterns returns %d from %s\n", i, argv [ 3 ] );
	}

	for ( i = 0; i < pattern_count; i++ ) {
		current_pattern = indextop ( i );
		read_input	( i );
		read_target	( i );	/* only necessary to get pattern width for shownet */
		recall();
		shownet();
	}

	if (system_filename[0] == '\0') {
		sys_save ( "test" );
	}
	else {
		sys_save( system_filename );
	}

	rc_write ();
}

#endif
/* --------------------------------------------------------------------------- */
