/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains the routines used to set up, access and change
 * values and structures associated with the display (or graphics)
 * panel, including processing all input events (mouse buttons, moves,
 * and keystrokes) that occur in the display panel..
 *
 * Kenton Lynne
 *   last update: 02/24/87
 *                04/01/87
 * Nigel Goddard
 *   substantially hacked for X: April 1989
 *   custom popup menus added:   May 1989
 *********************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "cmd_panel.h"
#include "mode_panel.h"
#include "control_panel.h"
#include "info_panel.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Shell.h>
#include <X11/Label.h>
#include <X11/Command.h>
#include <X11/Box.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/VPanedP.h>
#include <X11/Grip.h>
#include <X11/VPaned.h>
#include "Display.h"

extern Widget gi_display_popup;
extern Widget gi_layout_panel;
extern struct gricons marker_icon;
extern button_list * gi_mode_buttons,
  * gi_what_buttons,
  * gi_how_buttons,
  * gi_lhow_buttons,
  * gi_ldir_buttons;

typedef struct disp_globs
{
  Window gfx;
  Cursor last_cursor; /* last "permanent" cursor used */
  GC gc;				/* global graphics context       */
  int origin_x; /* left_most coordinate   */
  int origin_y; /* top_most coordinate    */
  int extent_x;     /* right_most coordinate  */
  int extent_y;     /* bottom_most coordinate */
  XImage **pixptr_start;
  XImage **last_pixptr;
  struct grobj marker,      /* the on display grobj header  */
         off_display; /* the "off display" header     */
  info_panel_t info_panel_head;	/* head of info panel chain for this display */
  struct txobj text_head;
  struct drobj  draw_head;
  int reshow_flag;
  char new_link_parms;       /* link parms have changed  */
  int link_direction;  /* current link direction   */
  int cur_link_target;        /* current target unit index*/
  char *cur_link_site;      /* current target site      */
  XImage **cur_link_image;   /* points to link icon array */
  int cur_lhow_value;		/* how links are displayed */
  struct grobj *cur_target_ptr; /* points to target grobj */
  int mode;			/* current mode                  */
  char off_display_ok;       /* off display values are ok     */
  int draw_type;       /* current draw parameter        */   
  struct grobj *last_tagged; /* points to last tagged unit */
  int cur_image_value ;     /* holds currently selected icon */
  int cur_unit_what;     /* contains current aspect       */
  char lrange_buf[11];
  char hrange_buf[11];
  char llrange_buf[11];
  char lhrange_buf[11];
  char who_buf[MAX_CMD_LENGTH];
  char howmany_buf[11];
  char spacex_buf[11];
  char spacey_buf[11];
  char wherex_buf[11];
  char wherey_buf[11];
} display_globals_rec;

typedef struct _display_panel {
  display_globals_rec globals; /* saved information */
  Widget popup; /* popup widget */
  Widget displaywid; /* display widget */
  char name[40];		/* icon name */
  Widget namewid;		/* name label widget */
  Widget namepopup;		/* popup for dialog box */
  Widget namedialog;		/* dialog box widget */
  Widget freeze_button;		/* button for freezing */
  Boolean frozen;		/* if true, then don't update display */
  Boolean free; /* if free, then popped down, and available for "next" */
  struct _display_panel *next; /* pointer to next in line */
} *display_panel_t;

display_panel_t gi_display_panel_head = NULL;
display_panel_t gi_cur_ipanel = NULL;

/* the following static variables are accessed "globally" by
 * routines in this file
 */
static XEvent ie;             /* the current input event */
static short blob_set;              /* indicates whether blob is shown */
static int old_blob_x, old_blob_y;  /* current location of blob */
static int old_font_x, old_font_y;  /* x/y spacing for current font */
static int reshow;                  /* reshow should be done */
static int first_move;              /* indicates first move of drawn object */
static int mouse_mode=NORMAL;       /* current state of the mouse */
static int layout_popped_up = FALSE;	/* is layout window popped up? */

static short vertex_set;            /* a vertex has been set */
static struct drobj 
  *deldraw_ptr=NULL,                /* to-be-deleted drawn object */
  *drobj_ptr,                       /* current drawn object */ 
  *cur_draw;                        /* current drawn object */

static struct txobj 
    *deltext_ptr=NULL,              /* to-be-deleted text object */
    *txobj_ptr;                     /* current text object */
static struct grobj *grobj_ptr;

static int last_disp_move_x=0, 
           last_disp_move_y=0;
static int old_vertex_x, old_vertex_y;
static int old_move_x, old_move_y;
static int cur_x, cur_y;
static char o_buf[20];
static int last_x, last_y;
static int base_x, base_y;  
static int cur_text_x, cur_text_y, cur_text_len=0;

Widget gi_display_popup;

int gi_display_select(), gi_display_redraw(), gi_display_expose();

static jump_move(), vertex_nearby(), draw_box(),
  draw_lines(), find_move_obj(), drop_move_obj(), do_middle(), 
  do_link_mode(), do_text_mode(), do_draw_mode(),
  cont_move(), process_key(), process_ASCII();


gi_set_display_mode_to_custom(flag)
     int flag;

{
  SetDisplayModeToCustom(gi_cur_ipanel->displaywid, flag);
}

/****************************************************************/
gi_popdown_popups()

{
  display_panel_t dp;
  info_panel_t ip;

  for (dp = gi_display_panel_head;
       dp != NULL;
       dp = dp->next)
    if (dp->free == False)
      {
	for (ip = dp->globals.info_panel_head;
	     ip != NULL;
	     ip = ip->next)
	  if (ip->free == False)
	    XtPopdown(ip->popup);
	XtPopdown(dp->popup);
      }
  XtPopdown(gi_layout_panel);
  XSync(XtDisplay(gi_tool), 0);
}
      
static void change_display_panel_name(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;

{
  (void) sprintf(ipanel->name, "%s", XtDialogGetValueString(ipanel->namedialog));
  SetValue(ipanel->namewid, XtNlabel, ipanel->name);
  XSetIconName(XtDisplay(gi_tool), XtWindow(ipanel->popup), ipanel->name);
  XtDestroyWidget(ipanel->namepopup);
}

int gi_rename_display_name(old, new)
     char * old;
     char * new;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(old, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    {
      (void) sprintf(ip->name, "%s", new);
      SetValue(ip->namewid, XtNlabel, ip->name);
      XSetIconName(XtDisplay(gi_tool), XtWindow(ip->popup), ip->name);
      return(OK);
    }
}

static void cancel_display_panel_name(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;

{
    XtDestroyWidget(ipanel->namepopup);
}

gi_close_all_display_panels()

{
  display_panel_t ip;

  for (ip = gi_display_panel_head;
       ip != NULL && ip->free == True;
       ip = ip->next);
  if (ip == NULL)
    return;
  if (ip != gi_cur_ipanel)
    gi_set_display_globals(ip, False);
  do
    gi_close_display_panel(NULL, gi_cur_ipanel, NULL);
  while (gi_cur_ipanel != gi_display_panel_head);
}
  
gi_close_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;
{
  int inpos;
  display_panel_t ip;
  char buf[128];

  if ((ip = gi_cur_ipanel) != ipanel)
    gi_set_display_globals(ipanel, False);
  if (ip->frozen == True)
    SetValue(ip->freeze_button,XtNlabel," FREEZE ");
  ip->frozen = False;
  call_button_func(gi_mode_buttons, 0); /* reset mode to main */
  ipanel->free = True;
  (void) sprintf(ipanel->name,"UNUSED DISPLAY");
  SetValue(ipanel->namewid, XtNlabel, ipanel->name);
  XSetIconName(XtDisplay(gi_tool), XtWindow(ipanel->popup), ipanel->name);
  XtPopdown(ipanel->popup);
  XtPopdown(gi_layout_panel);
  gi_do_restart(0, NULL);	/* restart this display panel */
  if (ip != gi_cur_ipanel)
    gi_set_display_globals(ip, False);
  else
    {
      for (ip = gi_display_panel_head;
	   ip != NULL && ip->free == True;
	   ip = ip->next);
      if (ip != NULL)
	gi_set_display_globals(ip, True);	/* find a new current display */
      else
	gi_set_display_globals(gi_display_panel_head, True); /* no displays popped up */
    }
}

int gi_close_display_name(name)
     char * name;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(name, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    {
      gi_close_display_panel(NULL, ip, NULL);
      return(OK);
    }
}

int gi_freeze_display_name(name,frozen)
     char * name;
     int frozen;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(name, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    {
      if (frozen == FALSE)
	SetValue(ip->freeze_button,XtNlabel," FREEZE ");
      else
	SetValue(ip->freeze_button,XtNlabel,"UNFREEZE");
      ip->frozen = frozen;
      return(OK);
    }
}

gi_rename_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;
{
  Widget temp;
  Position x,y;
  static Arg p[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg q[] =
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg d[] = 
    {
      { XtNvalue, (XtArgVal) NULL },
      { XtNlabel, (XtArgVal) "Display Panel Name" },
    };
  XWindowAttributes atts;

  q[0].value = (XtArgVal) &x;
  q[1].value = (XtArgVal) &y;
  XtGetValues(w,q,2);		/* get button location */
  p[0].value = x;		/* display dialog at same location */
  p[1].value = y;
  XGetWindowAttributes(XtDisplay(gi_tool), XtWindow(ipanel->popup), &atts);
  p[0].value += atts.x;		/* display dialog at same location */
  p[1].value += atts.y;

  ipanel->namepopup = XtCreatePopupShell("display_dialog",transientShellWidgetClass,
				     w, p, XtNumber(p));
  d[0].value = (XtArgVal) ipanel->name;
  ipanel->namedialog = XtCreateManagedWidget("dialog_box", dialogWidgetClass,
					     ipanel->namepopup,d, XtNumber(d));
  AddButton(ipanel->namedialog, "CHANGE", change_display_panel_name, ipanel, NULL);
  AddButton(ipanel->namedialog, "CANCEL", cancel_display_panel_name, ipanel, NULL);
  XtPopup(ipanel->namepopup, XtGrabNone);
}

static reset_control_typeins(g)
     struct disp_globs * g;
				/* resets ranges */

{
  gi_switch_button_bufs(g->lrange_buf,
			g->hrange_buf, 
			g->llrange_buf, 
			g->lhrange_buf, 
			g->who_buf, 
			g->howmany_buf,
			g->spacex_buf,
			g->spacey_buf,
			g->wherex_buf,
			g->wherey_buf);
}


static gi_save_display_globals(ipanel)
     display_panel_t ipanel;

{
  struct disp_globs * g;

  g = &(ipanel->globals);

  g->gfx = gi_gfx;
  g->gc = gi_gc;
  g->last_cursor = gi_last_cursor;
  g->origin_x = gi_origin_x;
  g->origin_y = gi_origin_y;
  g->extent_x = gi_extent_x;
  g->extent_y = gi_extent_y;
  g->pixptr_start = gi_pixptr_start;
  g->last_pixptr = gi_last_pixptr;
  g->marker = gi_marker;
  g->off_display = gi_off_display;
  g->info_panel_head = gi_info_panel_head;
  g->text_head = gi_text_head;
  g->draw_head = gi_draw_head;
  g->reshow_flag = gi_reshow_flag;
  g->new_link_parms = gi_new_link_parms;
  g->link_direction = gi_link_direction;
  g->cur_link_target = gi_cur_link_target;
  g->cur_link_site = gi_cur_link_site;
  g->cur_link_image = gi_cur_link_image;
  g->cur_lhow_value = gi_cur_lhow_value;
  g->cur_target_ptr = gi_cur_target_ptr;
  g->mode = gi_mode;
  g->off_display_ok = gi_off_display_ok;
  g->draw_type = gi_draw_type;
  g->last_tagged = gi_last_tagged;
  g->cur_image_value = gi_cur_image_value;
  g->cur_unit_what = gi_cur_unit_what;
  strcpy(g->lrange_buf, gi_lrange_buf);
  strcpy(g->hrange_buf, gi_hrange_buf);
  strcpy(g->llrange_buf, gi_llrange_buf);
  strcpy(g->lhrange_buf, gi_lhrange_buf);
  strcpy(g->who_buf, gi_who_buf);
  strcpy(g->howmany_buf, gi_howmany_buf);
  strcpy(g->spacex_buf, gi_spacex_buf);
  strcpy(g->spacey_buf, gi_spacey_buf);
  strcpy(g->wherex_buf, gi_wherex_buf);
  strcpy(g->wherey_buf, gi_wherey_buf);
}

gi_set_display_globals(ipanel, mode_switch)
     display_panel_t ipanel;
     int mode_switch;	/* change control panel mode */
				
{
  struct disp_globs * g;
  char buf[128];

  if (gi_cur_ipanel != NULL)
    gi_save_display_globals(gi_cur_ipanel); /* save current values */

  if (mode_switch && !(gi_marker.flag & NOSHOW))
    {				/* if marker showing */
      gi_marker.flag |= NOSHOW;	/* blank out current marker */
      gi_display_grobj(&gi_marker,GXxor);
    }

  gi_display_popup = ipanel->popup;
  gi_cur_ipanel = ipanel;

  (void) sprintf(buf, "gi ys %s %d", gi_cur_ipanel->name, mode_switch);
  gi_show_prev(buf);
  gi_log(buf);

  g = &(ipanel->globals);

  gi_gfx = g->gfx;
  gi_gc = g->gc;
  gi_last_cursor = g->last_cursor;

  gi_origin_x = g->origin_x;
  gi_origin_y = g->origin_y;
  gi_extent_x = g->extent_x;
  gi_extent_y = g->extent_y;
  gi_pixptr_start = g->pixptr_start;
  gi_last_pixptr = g->last_pixptr;
  gi_marker = g->marker;
  gi_off_display = g->off_display;
  gi_info_panel_head = g->info_panel_head;
  gi_text_head = g->text_head;
  gi_draw_head = g->draw_head;
  gi_reshow_flag = g->reshow_flag;
  gi_new_link_parms = g->new_link_parms;
  gi_cur_link_target = g->cur_link_target;
  gi_cur_link_site = g->cur_link_site;
  gi_cur_link_image = g->cur_link_image;
  gi_cur_target_ptr = g->cur_target_ptr;
  gi_off_display_ok = g->off_display_ok;
  gi_draw_type = g->draw_type;
  gi_last_tagged = g->last_tagged;
  if (mode_switch)
    {
      call_button_func(gi_mode_buttons, g->mode); /* reset mode */
      call_button_func(gi_how_buttons,g->cur_image_value-1); /* button func adds 1! */
      call_button_func(gi_what_buttons,g->cur_unit_what-1);	/* button func adds 1! */
      call_button_func(gi_ldir_buttons,g->link_direction);
      call_button_func(gi_lhow_buttons,g->cur_lhow_value);
      reset_control_typeins(g);
    }
  else
    {
      gi_mode = g->mode;
      gi_cur_image_value = g->cur_image_value;
      gi_cur_unit_what = g->cur_unit_what;
      gi_link_direction = g->link_direction;
      gi_cur_lhow_value = g->cur_lhow_value;
    }
}

extern void gi_radio_button_callback();

call_button_func(buttons, value)
     button_list * buttons;
     int value;
				/* what a hack */
{
  Widget * Wptr;

  Wptr = buttons->toggle_list;
  gi_radio_button_callback(*(Wptr + value), /* the widget */
			 buttons,NULL);	/* call its callback function, stipple, etc */
}

static int reshow_display(new_x, new_y)
   int new_x, new_y;

{
/* called when the RESHOW button is pressed
 */
   char buf[20];
   char logbuf[30];

   /* check that coordinates are within range */
   if (abs(new_x)>MAX_COORD || abs(new_y)>MAX_COORD)
     {
       gi_put_error("Origin specification is out of range");
       return(ERROR);
     }

   /* if the coordinates have changed, perform the update */
   if (new_x!=gi_origin_x || new_y!=gi_origin_y)
   {
      gi_change_origin(new_x-gi_origin_x,new_y-gi_origin_y); 
      gi_update_grobj_chain(TRUE, TRUE);
   }

   /* set flag to make sure gi_reshow clears screen and
      writes all objects
   */
   gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
   gi_reshow();

   (void) sprintf(logbuf,"%s %s %d %d",GI_CMD,RESHOW_CMD_SHORT,new_x, new_y);
   gi_show_prev(logbuf);
   gi_log(logbuf);
}

static void change_display_panel_origin(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;

{
  char buf[20];
  int new_x, new_y;
  display_panel_t tpanel = NULL;

  if (gi_cur_ipanel != ipanel)
    {
      tpanel = gi_cur_ipanel;
      gi_set_display_globals(ipanel, False);
    }
  (void) sprintf(buf, "%s", XtDialogGetValueString(ipanel->namedialog));
  gi_strip(buf);
  if (sscanf(buf,"%d %d", &new_x, &new_y)!=2)
    gi_put_error("Illegal Origin specification");
  else
    reshow_display(new_x, new_y);
  if (tpanel != NULL)
    gi_set_display_globals(tpanel, False);
  XtDestroyWidget(ipanel->namepopup);
}

gi_jump_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;

{
  Widget temp;
  char buf[30];
  Position x,y;
  static Arg p[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg q[] =
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg d[] = 
    {
      { XtNvalue, (XtArgVal) NULL },
      { XtNlabel, (XtArgVal) "New Origin" },
    };
  Window root;
  XWindowAttributes atts;

  int width,height,border,depth;

  q[0].value = (XtArgVal) &x;
  q[1].value = (XtArgVal) &y;
  XtGetValues(w,q,2);		/* get button location */
  p[0].value = x;		/* display dialog at same location */
  p[1].value = y;
  
  XGetWindowAttributes(XtDisplay(gi_tool), XtWindow(ipanel->popup), &atts);

  p[0].value += atts.x;		/* display dialog at same location */
  p[1].value += atts.y;

  ipanel->namepopup = XtCreatePopupShell("display_dialog",transientShellWidgetClass,
				     w, p, XtNumber(p));
  if (gi_cur_ipanel != ipanel)
    {
    (void) sprintf(buf,"%d %d",ipanel->globals.origin_x, ipanel->globals.origin_y);
    d[0].value = (XtArgVal) buf;
    }
  else
    {
    (void) sprintf(buf,"%d %d",gi_origin_x,gi_origin_y);
    d[0].value = (XtArgVal) buf;
    }

  ipanel->namedialog = XtCreateManagedWidget("dialog_box", dialogWidgetClass,
					     ipanel->namepopup,d, XtNumber(d));
  AddButton(ipanel->namedialog, "CHANGE", change_display_panel_origin, ipanel, NULL);
  AddButton(ipanel->namedialog, "CANCEL", cancel_display_panel_name, ipanel, NULL);
  XtPopup(ipanel->namepopup, XtGrabNone);
}

static reshow_display_panel(ipanel, reset)
     display_panel_t ipanel;
     Boolean reset;

{				/* doesn't change current panel unless reset is false */
  display_panel_t tpanel;

  tpanel = gi_cur_ipanel;
  if (gi_cur_ipanel != ipanel)
    gi_set_display_globals(ipanel, False); /* don't change control panel */
  reshow_display(gi_origin_x, gi_origin_y);
  if (reset == True && gi_cur_ipanel != tpanel)	/* reset globals if needed */
    gi_set_display_globals(tpanel, False); /* don't reset control panel */
}

gi_reshow_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;
{			
  ipanel->globals.reshow_flag |= RESHOW_ALL | CLEAR_NEEDED;
  reshow_display_panel(ipanel, True); /* do change current panel */
}

int gi_reshow_display_name(name)
     char * name;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(name, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    {
      gi_reshow_display_panel(NULL, ip, NULL);
      return(OK);
    }
}

gi_reshow_all()
				/* update all display panels after sim command */
{
  display_panel_t ipanel, cpanel;
  int reshow_flag;

  reshow_flag = gi_reshow_flag;
  for (ipanel = gi_display_panel_head, cpanel = gi_cur_ipanel;
       ipanel != NULL;
       ipanel = ipanel->next)
    if (ipanel != cpanel && ipanel->free == False)
	{
	  ipanel->globals.reshow_flag = reshow_flag;
	  if (ipanel->frozen == False)
	    {
	      if (gi_cur_ipanel != ipanel)
		gi_set_display_globals(ipanel, False);
	      gi_reshow();
	    }
	}
  cpanel->globals.reshow_flag = reshow_flag;
  if (cpanel->frozen == False)
    {
      if (gi_cur_ipanel != cpanel)
	gi_set_display_globals(cpanel, False);
      gi_reshow();
    }
  else
    gi_set_display_globals(cpanel, False); /* change globals to current panel */
  gi_update_clock();
}

gi_reshow_all_displays()

{				/* update all displays regardless of whether frozen */
  display_panel_t ipanel,cpanel;

  for (ipanel = gi_display_panel_head, cpanel = gi_cur_ipanel;
       ipanel != NULL;
       ipanel = ipanel->next)
    if (ipanel != cpanel && ipanel->free == False)
      {
	ipanel->globals.reshow_flag = CLEAR_NEEDED | RESHOW_ALL;
	reshow_display_panel(ipanel, False); /* change globals */
      }
  cpanel->globals.reshow_flag = CLEAR_NEEDED | RESHOW_ALL; 
  reshow_display_panel(cpanel, False); /* change globals */
}

gi_layout_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;
{
  if (gi_cur_ipanel != ipanel)
    gi_set_display_globals(ipanel, True);
  XtPopup(gi_layout_panel, XtGrabNone);
}

gi_freeze_display_panel(w, ipanel, call_data)
     Widget w;
     display_panel_t ipanel;
     caddr_t call_data;
{
  if (ipanel->frozen == TRUE)
    {
      SetValue(w,XtNlabel," FREEZE ");
      ipanel->frozen = FALSE;
      reshow_display_panel(ipanel, True); /* don't change globals */
    }
  else
    {
      SetValue(w,XtNlabel,"UNFREEZE");
      ipanel->frozen = TRUE;
    }
}

gi_init_display_globals(ipanel)
     display_panel_t ipanel;

/* initializes the saved globals record */
{
  struct disp_globs * g;

  ipanel->frozen = False;
  g = &(ipanel->globals);

  /* gfx, gc and last_cursor are set in new_display_widget() */

  g->origin_x = 0; /* left_most coordinate   */
  g->origin_y = 0; /* top_most coordinate    */
  g->extent_x = 484;     /* right_most coordinate  */
  g->extent_y = 484;     /* bottom_most coordinate */
  g->pixptr_start = NULL;
  g->last_pixptr = NULL;
  g->marker.next = g->marker.prev = &gi_marker;   /* the on display grobj header  */
  g->marker.x_pos = 5;
  g->marker.y_pos = 5;
  g->marker.u_index = MARKER;
  g->marker.gptr = &marker_icon;
  g->marker.val = 0;
  g->marker.flag = VALUE_OK + NOSHOW;
  g->marker.icon_index = 0;

  g->off_display.next = g->off_display.prev = &gi_off_display;
  g->off_display.flag = 0;
				/* the "off display" header     */
  g->info_panel_head = NULL;
  g->text_head.next = g->text_head.prev = &gi_text_head;
  g->text_head.flag = 0;
  g->draw_head.next = g->draw_head.prev = &gi_draw_head;
  g->reshow_flag = 0;
  g->new_link_parms = FALSE;       /* link parms have changed  */
  g->link_direction = TO_TARGET;  /* current link direction   */
  g->cur_link_target = -1;        /* current target unit index*/
  g->cur_link_site = NULL;      /* current target site      */
  g->cur_link_image = NULL;   /* points to link icon array */
  g->cur_lhow_value = 0;	/* how links are displayed */
  g->cur_target_ptr = NULL; /* points to target grobj */
  g->mode = MODE_MAIN;			/* current mode                  */
  g->off_display_ok = 0;       /* off display values are ok     */
  g->draw_type = LINES;       /* current draw parameter        */   
  g->last_tagged = NULL; /* points to last tagged unit */
  g->cur_image_value = 1;     /* holds currently selected icon */
  g->cur_unit_what = POT;     /* contains current aspect       */
  (void) sprintf(g->lrange_buf, "-1000");
  (void) sprintf(g->hrange_buf, "1000");
  (void) sprintf(g->llrange_buf, "-1000");
  (void) sprintf(g->lhrange_buf, "1000");
  (void) sprintf(g->who_buf, "0");
  (void) sprintf(g->howmany_buf, "all");
  (void) sprintf(g->spacex_buf, "5");
  (void) sprintf(g->spacey_buf, "5");
  (void) sprintf(g->wherex_buf, "5");
  (void) sprintf(g->wherey_buf, "5");
}

gi_popup_first_display()

{
  gi_make_display_panel(NULL,NULL,NULL); /* make first display panel */
  XtPopup(gi_layout_panel, XtGrabNone);	/* make layout panel appear */
}

static Widget DAddButton(w,s,f,d,p)
/* w is the parent,
   s is the name of the button,
   f is the function to call,
   and d is the data passed to the function.
   and p is the neighbour to the left */
     Widget w;
     char *s;
     XtCallbackProc f;
     caddr_t d;
     Widget p;
{
  Widget temp;
  static Arg a[] = {
    { XtNfromHoriz, (XtArgVal) NULL },
    { XtNhorizDistance, (XtArgVal) 1 },
    { XtNfromVert, (XtArgVal) NULL },
    { XtNvertDistance, (XtArgVal) 1 },
    { XtNborderWidth, (XtArgVal) 1 },
    { XtNfont, (XtArgVal) &gi_default_font },
  };

#ifdef DEBUG
	printf("Xsupport: creating widget named %s.\n",s);
#endif
  a[0].value = (XtArgVal) p;
  temp = XtCreateWidget(s, commandWidgetClass, w, a, XtNumber(a));
  XtAddCallback(temp,XtNcallback,f,d);

  XtManageChild(temp);
  return(temp);
}

static display_panel_t new_display_widget(ipanel, name, xorig, yorig, width, height)
     display_panel_t ipanel;
     char * name;
     int xorig, yorig;
     int width, height;

{
/* Make a new display_widget shell */
  Widget form, box, box1, dbox, temp, fbox, vpane;
  XGCValues gc_values;
  Dimension dim;

  static Arg a[] =
    {
      {XtNheight, (XtArgVal) 0 },
      {XtNsensitive, (XtArgVal) TRUE },
      {XtNresizable, (XtArgVal) TRUE },
      {XtNborderWidth, (XtArgVal) 0},
      {XtNdefaultDistance, (XtArgVal) 0},
      {XtNfromVert, (XtArgVal) NULL},
      {XtNvertDistance, (XtArgVal) 0},
      {XtNfromHoriz, (XtArgVal) NULL},
      {XtNhorizDistance, (XtArgVal) 0},
      {XtNtop, (XtArgVal) XtChainTop },
      {XtNbottom, (XtArgVal) XtChainBottom },
      {XtNleft, (XtArgVal) XtChainLeft },
      {XtNright, (XtArgVal) XtChainRight },
    };
  static Arg b[] =
    {
      { XtNdefaultDistance, (XtArgVal) 1},
      { XtNborderWidth, (XtArgVal) 0 },
    };
  static Arg c[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
      { XtNwidth, (XtArgVal) 0 },
      { XtNheight, (XtArgVal) 0 },
      { XtNdefaultDistance, (XtArgVal) 0},
      { XtNborderWidth, (XtArgVal) 2},
    };
  static Arg d[] = 
    {
      { XtNfromHoriz, (XtArgVal) NULL},
      { XtNhorizDistance, (XtArgVal) 4},
      { XtNwidth, (XtArgVal) 150 },
      { XtNborderWidth, (XtArgVal) 0 },
      { XtNfromVert, (XtArgVal) NULL },
      { XtNvertDistance, (XtArgVal) 1 },
      { XtNfont, (XtArgVal) &gi_default_font }
    };
  static Arg e[] =
    {
      { XtNdefaultDistance, (XtArgVal) 0},
      { XtNborderWidth, (XtArgVal) 0 },
    };
  static Arg f[] =
    {
      { XtNdefaultDistance, (XtArgVal) 0},
      { XtNborderWidth, (XtArgVal) 0 },
    };

  gi_init_display_globals(ipanel);
  ipanel->free = False;
  ipanel->namepopup = NULL;
  ipanel->namedialog = NULL;
  ipanel->freeze_button = NULL;
  c[0].value = (XtArgVal) xorig;
  c[1].value = (XtArgVal) yorig;
  c[2].value = (XtArgVal) width;
  c[3].value = (XtArgVal) height;
  ipanel->popup = XtCreatePopupShell("display_shell",transientShellWidgetClass,
				     gi_tool, c, XtNumber(c));
  vpane = XtCreateManagedWidget("vpane_box", vPanedWidgetClass, ipanel->popup,
				f, XtNumber(f));
  box = XtCreateManagedWidget("form_box", formWidgetClass, vpane, b, XtNumber(b));
				/* add buttons */

  temp = DAddButton(box, "QUIT", gi_close_display_panel, ipanel, NULL);
  temp = DAddButton(box, " FREEZE ", gi_freeze_display_panel, ipanel, temp);
  ipanel->freeze_button = temp;
  temp = DAddButton(box, "LAYOUT", gi_layout_display_panel, ipanel, temp);
  temp = DAddButton(box, "RESHOW", gi_reshow_display_panel, ipanel, temp);
  temp = DAddButton(box, "JUMP", gi_jump_display_panel, ipanel, temp);
  temp = DAddButton(box, "RENAME", gi_rename_display_panel,ipanel,temp);

  d[0].value = (XtArgVal) temp;
  (void) sprintf(ipanel->name,"%s",name);
  ipanel->namewid = XtCreateManagedWidget(ipanel->name, labelWidgetClass, box,
					  d, XtNumber(d));

				/* make display panel, add callbacks */
  fbox = XtCreateManagedWidget("display_form", formWidgetClass, vpane, e, XtNumber(e));
  a[0].value = (XtArgVal) 100;

  dbox = XtCreateManagedWidget("display_box", displayWidgetClass, fbox,
			       a, XtNumber(a));
  XtSetMappedWhenManaged(ipanel->popup, False);
  XtRealizeWidget(ipanel->popup,XtGrabNone);
  XSetIconName(XtDisplay(gi_tool), XtWindow(ipanel->popup), ipanel->name);

  ipanel->globals.gfx = XtWindow(dbox);
  ipanel->displaywid = dbox;

  gc_values.function = GXcopy;
  gc_values.line_width = 0;
  gc_values.line_style = LineSolid;
  gc_values.foreground = BlackPixelOfScreen(XtScreen(gi_tool));
  gc_values.background = WhitePixelOfScreen(XtScreen(gi_tool));

  ipanel->globals.gc = XCreateGC(XtDisplay(gi_tool), ipanel->globals.gfx,
		    GCFunction | GCForeground | GCBackground
		    | GCLineWidth | GCLineStyle, &gc_values);
  XSetFunction(XtDisplay(gi_tool),ipanel->globals.gc, GXcopy);
  XSetFont(XtDisplay(gi_tool), ipanel->globals.gc, gi_default_font.fid);

  /* the next two statements depend on the internals of VPaned.c */
  XtSetMappedWhenManaged(((Pane)(box->core.constraints))->grip, False);
  XtSetMappedWhenManaged(((Pane)(fbox->core.constraints))->grip, False);

    /* change the cursor on the graphics panel to a crosshair */
  ipanel->globals.last_cursor = gi_main_cursor;
  XDefineCursor(XtDisplay(gi_tool), ipanel->globals.gfx, gi_main_cursor);

  ipanel->globals.extent_x = width;
  ipanel->globals.extent_y = height;
  gi_set_display_globals(ipanel, True);
  XtPopup(ipanel->popup,XtGrabNone);
  XtSetMappedWhenManaged(ipanel->popup, True);

  return(ipanel);
}

gi_make_display(name, xorig, yorig, width, height)
     char * name;
     int xorig, yorig;
     int width, height;

{
/* initial set up and display of the display (graphics) panel
 */
  display_panel_t ip,last;
  int i = 0;
  char buf[32];
  static Arg a[] =
    {
      {XtNwidth, (XtArgVal) 0 },
      {XtNheight, (XtArgVal) 0 },
      {XtNx, (XtArgVal) 0 },
      {XtNy, (XtArgVal) 0 },
    };

  for(i = 0, last=ip=gi_display_panel_head;
      (ip != NULL) && (last->free == False);
      last=ip, ip=ip->next, i++); /* find end of chain or free entry */

  /* set defaults */
  if (last != NULL && last->free == True && i != 0) /* what a mess */
    i--;
  if (name == NULL)
    {
    (void) sprintf(buf,"DISPLAY #%d",i);
    name = buf;
    }
  if (xorig < 0)
    xorig = i*50;
  if (yorig < 0)
    yorig = i*50;
  if (width <= 0)
    width = 484;
  if (height <= 0)
    height = 484;

  /* make new widget or reuse old one */
  if (ip == NULL && (last == NULL || last->free == False)) /* no free ip's */
    {
      ip = new_display_widget(malloc(sizeof(*ip)),name,xorig,yorig,width,height);
      if (last == NULL)		/* first ip to be made */
	gi_display_panel_head = ip;
      else
	last->next = ip;
      ip->next = NULL;
      ip->free = False;
      gi_cur_ipanel = ip;
      XtAddCallback(ip->displaywid, XtNredrawCallback, gi_display_redraw, ip);
      XtAddCallback(ip->displaywid, XtNactionCallback, gi_display_select, ip);
      XtAddCallback(ip->displaywid, XtNexposeCallback, gi_display_expose, ip);
    }
  else
    {
      ip = last;
      a[0].value = (XtArgVal) width;
      a[1].value = (XtArgVal) height;
      a[2].value = (XtArgVal) xorig;
      a[3].value = (XtArgVal) yorig;
      XtSetValues(ip->popup, a, XtNumber(a));
      (void) sprintf(ip->name, "%s", name);
      SetValue(ip->namewid, XtNlabel, ip->name);
      XtPopup(ip->popup, XtGrabNone);
      XSetIconName(XtDisplay(gi_tool), XtWindow(ip->popup), ip->name);
      gi_init_display_globals(ip);
      ip->globals.extent_x = width;
      ip->globals.extent_y = height;
      gi_set_display_globals(ip, True);
      ip->free = False;
      gi_cur_ipanel = ip;
    }
}

int gi_select_display_panel(name, mode_switch)
     char * name;
     int mode_switch;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(name, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    {
      gi_set_display_globals(ip, mode_switch);
      return(OK);
    }
}
      
int gi_select_display_mode(name, mode_switch)
     char * name;
     int mode_switch;

{
  display_panel_t ip;
  
  for (ip = gi_display_panel_head;
       ip != NULL && strcmp(name, ip->name);
       ip = ip->next);
  if (ip == NULL)
    return(ERROR);
  else
    if (gi_cur_ipanel == ip)
      call_button_func(gi_mode_buttons, mode_switch);
    else
      ip->globals.mode = mode_switch;
  return(OK);
}
      
gi_make_display_panel(w, client_data, call_data)
     Widget w;
     caddr_t client_data, call_data;

{
  gi_show_prev("gi yc");
  gi_log("gi yc");
  gi_make_display(NULL, -1, -1, -1, -1);
}

/*************************************************************************/
gi_display_expose(w, client_data, call_data)
     Widget w;
     display_panel_t client_data;
     XEvent * call_data;
     
{
  /* the graphics (and info) screen have been resized
   * thus change the rectlist accordingly and
   * redraw all the objects in it
   */
  
  Window root;
  int x,y,width,height,border,depth;
  char buf[128];
  /* assume that the window has been resized,
     won't redisplay if window is of same size */
  /* if the new size is larger in either dimension, then
     we need to determine what new objects have to 
     be displayed 
     */
  
  if (gi_cur_ipanel != client_data)
    gi_set_display_globals(client_data, TRUE); /* get globals for this display window */
  gi_reshow_flag |= RESHOW_ALL;
  gi_reshow();
}

/*************************************************************************/
gi_display_redraw(w, client_data, call_data)
     Widget w;
     display_panel_t client_data;
     XEvent * call_data;
     
{
  /* the graphics (and info) screen have been resized
   * thus change the rectlist accordingly and
   * redraw all the objects in it
   */
  
  Window root;
  int x,y,width,height,border,depth;
  static int count = 0;
  /* assume that the window has been resized,
     won't redisplay if window is of same size */
  /* if the new size is larger in either dimension, then
     we need to determine what new objects have to 
     be displayed 
     */
  
  if (gi_gfx == NULL)		/* do nothing during realization */
    return;
  if (gi_cur_ipanel != client_data)
    gi_set_display_globals(client_data, TRUE); /* get globals for this display window */
  
  XGetGeometry(XtDisplay(gi_tool),gi_gfx,
	       &root, &x, &y, &width, &height, &border, &depth);
  
  /* update the extent values on resize */
  gi_extent_x = gi_origin_x + width;
  gi_extent_y = gi_origin_y + height;
  
  gi_update_grobj_chain(TRUE, TRUE);
  gi_reshow_flag |= RESHOW_ALL;
  gi_reshow();
}
    

/*************************************************************************/
gi_display_select(w, client_data, call_data)
     Widget w;
     display_panel_t client_data;
     XEvent * call_data;
     
{
  /* handles all the input events for the display panel 
   */
  
  XEvent sw;
  
  if (gi_cur_ipanel != client_data)
    gi_set_display_globals(client_data, TRUE); /* get globals for this display window */

  if (gi_mode == MODE_CUSTOM)
    return;			/* handled in display widget (Display.c) */

  ie = *(call_data);
  /* xgi:  yuk!  this should _not_ be a global.  
     Kept for historical reasons */
  
  /* set up the current display coordinates */
  if (ie.type==ButtonPress || ie.type==ButtonRelease)
    {
      cur_x = ie.xbutton.x + gi_origin_x;
      cur_y = ie.xbutton.y + gi_origin_y;
    }
  else
    if (ie.type==MotionNotify)
      {
	cur_x = ie.xmotion.x + gi_origin_x;
	cur_y = ie.xmotion.y + gi_origin_y;
      }
  reshow = FALSE;
  
  if ((deldraw_ptr || deltext_ptr)
      && ie.type==ButtonPress
      && ie.xbutton.button != MS_RIGHT)
    {
      gi_put_message("Delete cancelled");
      deldraw_ptr = (struct drobj *) deltext_ptr = NULL;
      XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_last_cursor);
      return;
    }
  
  /* based on current mode, call appropriate handler */
  switch (gi_mode)
    {
    case MODE_MAIN:
      do_main_mode();
      break;
    case MODE_LINK:
      do_link_mode();
      break;
    case MODE_TEXT:
      do_text_mode();
      break;
    case MODE_DRAW:
      do_draw_mode();
      break;
    } 
  
  /* update the screen if necessary */
  if (reshow)
    {
      if ((gi_reshow_flag & (RESHOW_NEEDED+RESHOW_ALL+CLEAR_NEEDED))
	  || (gi_mode==MODE_LINK 
	      && gi_reshow_flag & (SOME_LINKS_NEEDED+ALL_LINKS_NEEDED))
	  || (gi_mode!=MODE_LINK 
	      && gi_reshow_flag & (SOME_VALS_NEEDED+ALL_VALS_NEEDED)))
	gi_reshow();
    } 
}

/*************************************************************************/
do_main_mode()
{
/* process a main mode input event on
 * the display screen
 */
   static struct grobj *obj_ptr;
   char buf[MAX_ITEM_LENGTH+1];
   char *sptr;

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   if (ie.type==LeaveNotify) {
     if (mouse_mode!=NORMAL)
       {
	 drop_move_obj(cur_x, cur_y);
       }
     return;
   }

   /* mouse being moved while button pushed DOWN */
   if (ie.type==MotionNotify)
   {
     cont_move();
     return;
   }

   /* left or right mouse button down while in MOVE_DISPLAY mode */
   if ((ie.type==ButtonPress)&&(mouse_mode==MOVE_DISPLAY &&
				(ie.xbutton.button==MS_LEFT || 
				 ie.xbutton.button==MS_RIGHT)))
   {
     /* jump move the display */
     jump_move();
   }

   else
   /**************************/
   /* left mouse button DOWN */
   /**************************/
   if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_LEFT 
				&& mouse_mode==NORMAL ))
   {
       /* update info panel for this unit if it exists */
       if (gi_do_info(cur_x, cur_y, TRUE,-1,-1,gi_cur_ipanel)==ERROR) 
       {
         /* no unit is here : so move marker here instead */
         gi_move_marker(cur_x-HOTX,cur_y-HOTY);

         /* if marker currently not shown, show it */
         if (gi_marker.flag & NOSHOW)
         {
           gi_marker.flag &= ~NOSHOW;
           gi_display_grobj(&gi_marker, GXxor);
         }
       }
   }

   /**********************************/ 
   /* middle mouse button UP or DOWN */
   /**********************************/ 
   else if (ie.xbutton.button==MS_MIDDLE)
   {
     do_middle();
   }

   /***************************/
   /* right mouse button DOWN */
   /***************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_RIGHT 
				       && mouse_mode==NORMAL))
   {
     /* check if the marker is here and if so, blank it out */
     if (gi_marker_here(cur_x,cur_y) && !(gi_marker.flag & NOSHOW))
     {
       /* marker is underneath, set flag to NOSHOW */
       gi_marker.flag |= NOSHOW;
       gi_display_grobj(&gi_marker,GXxor);
     }

     /* check if a grobj is here and tag or untag it as appropriate */
     else if ((obj_ptr=gi_find_xy(cur_x,cur_y))!=NULL)
     {
       if ((obj_ptr->flag & TAGGED)==TAGGED)
       {
          /* unmark this item as current LINK target */ 
          gi_last_tagged = NULL;
          obj_ptr->flag = obj_ptr->flag & ~TAGGED;
          gi_display_grobj(obj_ptr, GXcopy);
       }
       else 
       {
          /* mark this node as the current LINK target */
          obj_ptr->flag = obj_ptr->flag | TAGGED;
          gi_display_grobj(obj_ptr,GXcopyInverted);

          /* update last_tagged to point to this object */
          if (gi_last_tagged!=NULL)
          {
            gi_last_tagged->flag &= ~TAGGED;
            gi_display_grobj(gi_last_tagged, GXcopy);
          }
          gi_last_tagged = obj_ptr;

          /* update the target prompt on the control panel */
          (void) sprintf(buf,"%1d",obj_ptr->u_index);
          
          /* make sure to keep the site prompt, if any */
          for (sptr=gi_target_buf; *sptr!='\0'; sptr++)
          {
            if (*sptr==SITE_SEP_CHAR)
            {
              strcat(buf,sptr);
              break;
            }
          }
    
          /* copy new target/site arg to panel prompt */
	  strcpy(gi_target_buf,buf);
	  SetValue(gi_mitem[TARGET_ITEM],XtNstring,gi_target_buf);
       }
     }
   }       
}


/**********************************************************************/
gi_set_new_target(ptr,ui)
  struct grobj *ptr;
  int ui;
{
/* Sets up the unit indicated by 
 * by the passed unit index
 * or the ptr (if not NULL)
 * as the new target in link mode.
 * Note: if passed ptr is NULL, then the grobj chains will
 *       be searched for non-auxiliary unit with ui.
 *       In this case, we assume that the caller is "target_proc"
 *       and thus there is no need to update the control panel
 *       prompt with the new ui/site string 
 *       
 *       
 */
  char buf[MAX_ITEM_LENGTH+1];
  char *site_string;
  int update_prompt;
  int inpos;
  XtTextBlock text;

  /* if ptr is null, then get the pointer, if any,
     to the grobj for that unit
  */
  if (ptr)
  {
    ui = ptr->u_index;
    update_prompt = TRUE;
  }
  else
  {
    ptr = gi_find_unit(ui,0);
    update_prompt = FALSE;
  }

  /* make sure the old link target (if any) gets redisplayed */
  if (gi_cur_target_ptr)
    gi_cur_target_ptr->flag &= ~DISPLAYED;

  /* make sure the new link target (if any) gets redisplayed */
  if (ptr)
    ptr->flag &= ~DISPLAYED;

  /* get the current link site */
  gi_cur_link_site = gi_get_site(gi_ltarget_buf);

  /* update globals for current target */
  gi_cur_link_target = ui;
  gi_cur_target_ptr = ptr;  /* note: could be NULL */

  /* update the prompt for the target field (if no ptr supplied) */
  if (update_prompt)
  {
    if (gi_cur_link_site)
      site_string = gi_cur_link_site;
    else
      site_string = ANY_SITE;

    text.firstPos = 0;
    text.format = FMT8BIT;
    
    (void) sprintf(buf,"%1d/%s",gi_cur_link_target,site_string);
    text.length = strlen(buf);
    text.ptr = buf;
    inpos =  XtTextGetInsertionPoint(gi_litem[LSITE_ITEM]);
    XtTextReplace(gi_litem[LSITE_ITEM], 0, inpos, &text);
  }

  /* set the reshow flags appropriately */
  gi_reshow_flag |= ALL_LINKS_NEEDED;
}


/*************************************************************************/
static move_cur_object(x, y)
  int x, y;
{
/* given the current display space coordinates of the cursor,
 * moves the currently "moving" object to this position after
 * erasing the old object
 * note: assumes old_move_x, old_move_y are correct
 */
    int offset_x, offset_y, erase_op;
    char buf[2*MAX_ITEM_LENGTH+1];
    struct drobj *dptr;

    /* calculate the new offset since last position */
    offset_x = x - old_move_x;
    offset_y = y - old_move_y;

    switch(mouse_mode)
    {
      case MOVE_DISPLAY:
           (void) sprintf(buf,"%1d %1d", 
                         gi_origin_x + (base_x - x),
                         gi_origin_y + (base_y - y));
#if 0
	   SetValue(gi_cur_ipanel->originwid, XtNlabel, buf);
	   SetValue(gi_cur_ipanel->originwid, XtNwidth, strlen(buf));
	   strcpy(gi_origin_buf ,buf);
	   SetValue(gi_origin_item,XtNlabel, gi_origin_buf);
	   SetValue(gi_origin_item,XtNlength, strlen(gi_origin_buf));
#endif
           break;

      case MOVE_GROBJ:
           gi_move_grobj(grobj_ptr, offset_x, offset_y, TRUE);
           break;

      case MOVE_TXOBJ:
           gi_move_txobj(txobj_ptr,offset_x, offset_y, TRUE);
           break;

      case MOVE_DROBJ:
           /* determine what pixop to use for the erase */
           if (first_move)
           {
             /* use GXclear so as not to leave garbage around */
             first_move = FALSE;
             erase_op = GXclear;
           }
           else  /* use a mask so as not to disturb other objects */
             erase_op = GXxor;

           /* check if this is a bounding box */
           if (drobj_ptr->flag & BOUND_BOX)
           {
             /* go through the draw chain looking for bounding boxes */
             for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
             {
               if (dptr->flag & MOVING && dptr->flag & BOUND_BOX)
                 gi_move_drobj(dptr,offset_x,offset_y,
                               erase_op,GXxor);
             }
           }

           else  /* just a normal object to move */
             gi_move_drobj(drobj_ptr,offset_x,offset_y,
                           erase_op,GXxor);
           
           break;
    }
}


/*************************************************************************/
static save_text()
{
/* saves the current text parameters by making
 * a permanent text object on the text chain and
 * then logging (if necessary) the text command
 * that would recreate this object
 */
  struct txobj *ptr;

  ptr = gi_remember_txobj(cur_text_x,cur_text_y,cur_text_len,gi_cur_font);
  cur_text_len = 0;
  if (gi_log_on)
  {
    gi_log_text(ptr);
  }
}


/*************************************************************************/
static clear_blob()
{
/* clears out the old blob 
 * (using the old blob and old font values)
 * and sets blob_set FALSE
 */
  
   XClearArea(XtDisplay(gi_tool),gi_gfx,old_blob_x,old_blob_y-old_font_y+4,
			old_font_x, old_font_y, False);
   blob_set = FALSE;
}


/*************************************************************************/
static write_blob(xpos, ypos)
  int xpos, ypos;
{
/* write the text "blob" at indicated cursor "hot spot"
 * (using current font) and updates the global
 * blob and font variables
 */

   /* update the fonts and position for blob and write it out */
   old_font_x = gi_cur_font->max_bounds.width;
   old_font_y = gi_cur_font->max_bounds.ascent + gi_cur_font->max_bounds.descent;

   old_blob_x = xpos;
   old_blob_y = ypos;

   XFillRectangle(XtDisplay(gi_tool),gi_gfx,gi_gc,
			old_blob_x,old_blob_y-old_font_y+4,
			old_font_x, old_font_y);
   blob_set = TRUE;
}


/*************************************************************************/
static cont_move()
{
/* check if an object is currently being moved
 * and if so, continue to have it track 
 * the cursor
 */
   if (mouse_mode!=NORMAL)
   {
      move_cur_object(cur_x, cur_y);
      old_move_x = cur_x;
      old_move_y = cur_y;
   }
}


/*************************************************************************/
static process_key()
{
/* process a KeyPressed event */
  int i,j;
  char buf[MAX_CMD_LENGTH];

  i = XLookupString(&(ie.xkey),buf,MAX_CMD_LENGTH,NULL,NULL);
  for (j=0; j<i; j++)
    process_ASCII(buf[j]);
}

/*************************************************************************/
static process_ASCII(this_char)
     char this_char;
{
/* processes a key event
 * assumes blob_set and blob and font variables are current
 */

     switch (this_char)
     {
       case BACKSPACE:
       case DELETE:
            /* if the string has a positive length
             * backup and write a blank at new position
             */

            clear_blob();
	    write_blob(old_blob_x-old_font_x, old_blob_y);

            if (cur_text_len > 0)
            {  
              cur_text_len--;
            }
            break;

        case CR:
            /* if string has a positive length, save it
             * in a text object and link it onto the chain
             * and move the text blob to the next line
             */

	    clear_blob();
            if (cur_text_len > 0)
            {  
              save_text();
	      write_blob(cur_text_x-gi_origin_x, old_blob_y+old_font_y);
            }
            else
            {
	      write_blob(old_blob_x, old_blob_y+old_font_y);
            }
            break;
             
       default: /* (printable) character */
               
           if (cur_text_len==0)
           {
             /* save starting position of the string */
             /* using "display space" coordinates    */
             cur_text_x = old_blob_x + gi_origin_x;
             cur_text_y = old_blob_y + gi_origin_y;
           }
 
           /* save character in string space and increment length */
	   gi_save_char(this_char,cur_text_len);
           cur_text_len++;

           /* if string is at maximum length, 
              make a text object for it now and
              start a new object to contain future
              characters
           */

           if (cur_text_len >= MAX_TEXT_LENGTH)
             save_text();
             
	    clear_blob();
          /* output character at current blob position */
	    XSetFont(XtDisplay(gi_tool), gi_gc, gi_cur_font->fid);
	    XDrawString(XtDisplay(gi_tool),gi_gfx,gi_gc,
                  old_blob_x, old_blob_y,&this_char,1);
 
           /* move the blob right one character */
	   write_blob(old_blob_x+old_font_x, old_blob_y);

           break;
      } /* end of switch */
}


/*************************************************************************/
static build_cmd(numargs, args, ui, new_cmd)
  int numargs, ui;
  char **args, *new_cmd;
{
/* assumes that args contains pointers to the arg strings
 * and tries to substitute for the unit index and
 * coordinates and then builds the new command string
 * in the buffer new_cmd
 * and then returns the number of arguments still 
 * left in cmd_args (in case of multiple commands)
 * the substituted command is put in the static array "new_cmd"
 */

  char arg[MAX_CMD_LENGTH+1];
  static next_index=0;
  int i;

  /* blank out the destination buffer */
  new_cmd[0] = '\0';

  /* check each argument for substitution possibilities and
     and make the appropriate substitution */
  for (i=next_index; i<numargs; i++)
  {
    if (strcmp(args[i],UI_SUB_STRING)==0)
    {
      /* if ui is not valid, indicate an error */
      if (ui < 0)
      {
        gi_put_error("No valid substitution for $u");   
	return(NULL);
      }
      (void) sprintf(arg,"%1d ",ui);
    }

    /* check for "$x" and substitute x coordinate if necessary */
    else if (strcmp(args[i],X_SUB_STRING)==0)
      (void) sprintf(arg,"%1d ",cur_x);

    /* check for "$y" and substitute y coordinate if necessary */
    else if (strcmp(args[i],Y_SUB_STRING)==0)
      (void) sprintf(arg,"%1d ",cur_y);

    /* check for "--" making sure NOT to add it to the buffer */
    else if (strcmp(args[i],CONT_STRING)==0)
      (void) sprintf(arg," ");

    /* check for ";" indicating end of this command and exit early */
    else if (strcmp(args[i],TERM_STRING)==0)
    {
      next_index = i + 1;
      /* return the number of arguments left unprocessed in args array */
      return(numargs-i-1);
    }
    else
      (void) sprintf(arg,"%s ",args[i]);

    /* append the argument just found to the new command buffer */
    strcat(new_cmd,arg);
  }

  /* normal exit, reset next_index to start fresh on next call */
  next_index = 0;
  return(0);
}


/*************************************************************************/
gi_process_button(button_buf, buttonx, buttony)
     char * button_buf;
     int buttonx, buttony;

{
/* given
 * the custom string for a particular button action,
 * this does the appropriate
 * substitution and executes the command
 * (if command is not null)
 */

  static int cmd_waiting=FALSE;
  static char command[MAX_CMD_LENGTH+1];
  static char new_cmd[MAX_CMD_LENGTH+1];

  char buf[MAX_CMD_LENGTH+1];
  char *args[MAX_ARGS]; 
  char arg_buf[MAX_CMD_LENGTH+1];
  int args_left, num_args, ui;
  struct grobj *ptr;

  strcpy(buf,button_buf);
  cur_x = buttonx+gi_origin_x;
  cur_y = buttony+gi_origin_y;

  /* parse the panel string into args array and arg_buf */
  if ((num_args = gi_parse_cmd(buf,args,arg_buf)) == 0)
    return;

  /* find any grobj at this cursor location */
  if ((ptr=gi_find_xy(cur_x, cur_y))!=NULL)
    ui = ptr->u_index;
  else
    ui = -1;
  
  /* call build_cmd to process the next whole command
     in the cmd_args array, looping until no more
     arguments are left             
     */
  do
    {
      /* build new command substituting ui, and x and y location */
      args_left = build_cmd(num_args,args,ui,new_cmd);
      
      /* if there is no "continued" command already in the buffer
         initialize our command buffer to NULL
      */
      if (!cmd_waiting)
        command[0] = '\0';

      /* check that the maximum command length is not exceeded */
      if (strlen(command)+strlen(new_cmd) > MAX_CMD_LENGTH)
      {
        gi_put_error("Flushing command line : too long");
        cmd_waiting = FALSE;
      }

      /* check that new command has at least one character in it */
      if (strlen(new_cmd) > 0);
      {

        /* append the new command to the command buffer */
        strcat(command,new_cmd);
    
        /* if this command is "continued" delay execution */
        if (strcmp(args[num_args-args_left-1],CONT_STRING)==0)
        {
          cmd_waiting = TRUE;
          gi_put_message("Waiting for rest of command");
        }
        else /* execute it immediately */
        {
      
          /* execute this command */
          gi_command(command);
    
          /* reset cmd_waiting flag */
          cmd_waiting = FALSE;
        }
      }
    } while (args_left > 0);
}


/*************************************************************************/
static jump_move()
{
/* executes a "jump move" of the display window
 * by the indicated offsets
 * called when the user presses either the right or
 * left mouse button while in DISPLAY_MOVE mode
 */

  /* only do something if displacement non-zero */
  if (last_disp_move_x || last_disp_move_y)
  {
    /* jump in opposite directions depending on button pushed */
    if (ie.xbutton.button==MS_RIGHT)
      gi_change_origin(last_disp_move_x, last_disp_move_y);
    else
      gi_change_origin(-last_disp_move_x, -last_disp_move_y);

    /* update the grobj chain */
    gi_update_grobj_chain(TRUE,TRUE);

    /* set flags to clear the display and reshow everything */

    gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
    reshow = TRUE; 

    /* reinitialize base variables */
    base_x = ie.xbutton.x + gi_origin_x;
    base_y = ie.xbutton.y + gi_origin_y;
   }
}


/*************************************************************************/
static vertex_nearby(x,y)
  int x, y;
{
/* returns true if the display coordinates are
 * "near" (within 3 pixels) the last vertex set coordinates
 */

  if (abs(cur_x-old_vertex_x) < 4 
  && abs(cur_y-old_vertex_y) < 4)
     return(TRUE);
  else
    return(FALSE);
}


/*************************************************************************/
static draw_box(flag)
  int flag;
{
/* saves the box parameters from the global variables
 * on the drobj chain and then logs the command
 * to recreate that object (if logging enabled)
 * and writes is out to the display
 * note: flag indicates whether the box is a bounding box
 */
  
  struct drobj *box_ptr;

  box_ptr = gi_remember_drobj(old_vertex_x, old_vertex_y,
				 old_vertex_x, cur_y, flag);
  gi_save_vertex(box_ptr, cur_x, cur_y); 
  gi_save_vertex(box_ptr, cur_x, old_vertex_y);
  gi_save_vertex(box_ptr, old_vertex_x, old_vertex_y);
  gi_display_drobj(box_ptr,GXcopy);
  vertex_set = FALSE;

  if (gi_log_on) gi_log_draw(box_ptr);
}


/*************************************************************************/
static draw_lines()
{
/* finish up drawing the 
 * current line object
 * and log the command that
 * would recreate it if
 * logging is enabled
 */
    /* finish partially drawn object */
  if (cur_draw!=NULL)
  {
     gi_display_drobj(cur_draw,GXcopy);
     if (gi_log_on) gi_log_draw(cur_draw);
     cur_draw = NULL;
  }
  vertex_set = FALSE;
}


/*************************************************************************/
static find_move_obj(x,y)
  int x, y;
{
/* finds what object is at (or near) the 
 * display coordinates given and
 * sets mouse_mode and a pointer to that
 * object 
 */
   int done=FALSE;

   /* set the hand cursor -- but don't set gi_last_cursor! */
   XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_grab_cursor);

   /* check if cursor is over a drawn object */
   if ((drobj_ptr=gi_find_drobj(x,y))!=NULL)
   {
     mouse_mode = MOVE_DROBJ;

     /* further check if this is a bounding box */
     if (drobj_ptr->flag & BOUND_BOX)
     {
       /* blank out and mark as MOVING all objects contained within it */
       gi_mark_bound(drobj_ptr, TRUE);
     }
   }

   /* check if cursor is over a text object */
   else if ((txobj_ptr=gi_find_txobj(x,y))!=NULL)
   {
     mouse_mode = MOVE_TXOBJ;
   }
   
   /* check if a unit object is at the cursor */
   else if ((grobj_ptr=gi_find_xy(x,y))!=NULL)
   {
     mouse_mode = MOVE_GROBJ;
   }

   /* otherwise assume the intent is to move entire display */
   else
   {
     mouse_mode = MOVE_DISPLAY;
     XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_graball_cursor);
   }
}


/*************************************************************************/
static drop_move_obj(x, y)
  int x, y;
{
/* given display coordinates of the current
 * cursor, "drops" the object currently
 * being carried at that location
 * note: assumes base_x and base_y contain the
 *       original relative coordinates that
 *       the object was picked up from
 */

  switch(mouse_mode)
  {
  case MOVE_DISPLAY:
    /* only do something if the mouse has actually moved */
    if (base_x!=x || base_y!=y)
      {
	/* if at least 10 pixels of change
	 * save the last move displacement  
	 * for future "jump moves" 
	 */
	if (abs(base_x-x) > 9 || abs(base_y-y) > 9)
	  {
	    last_disp_move_x = base_x - x;
	    last_disp_move_y = base_y - y;
	  }
	
	/* officially change the display window parameters */
	gi_change_origin(base_x-x, base_y-y);
	
	/* update the display/off_display chains */
	gi_update_grobj_chain(TRUE, TRUE);
	
	/* set flags to clear screen and redisplay all objects */
	gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
	reshow = TRUE;
      }
    break;
  case MOVE_GROBJ:
    /* if not in link mode, display object normally */ 
    
    if (gi_mode!=MODE_LINK)
      gi_display_grobj(grobj_ptr,GXcopy);
    
    else  /* link mode */
      {
	/* if not the current target, display unit in link mode */
	if (grobj_ptr!=gi_cur_target_ptr)
	  gi_display_link(grobj_ptr,GXcopy);
	
	else  
	  /* unit is current target, reverse image it */
	  gi_display_link(grobj_ptr,GXcopyInverted);
      }
    break;
  case MOVE_TXOBJ:
    gi_display_txobj(txobj_ptr,GXor);
    break;
  case MOVE_DROBJ:
    /* check if this is a bounding box */
    if (drobj_ptr->flag & BOUND_BOX)
      {
	/* move all MOVING "bound" objects to new positions */
	gi_move_bound(cur_x-base_x, cur_y-base_y, TRUE);
      }
    else
      {
	/* move single object */
	gi_display_drobj(drobj_ptr,GXcopy);
      }
    
    break;
  }
    
    /* if logging is enabled and an object has just been
       moved, log the appropriate "move" command
       */
  
  if (mouse_mode!=MOVE_DISPLAY && gi_log_on)
    gi_log_move(base_x, base_y, cur_x, cur_y);
  
  mouse_mode = NORMAL;
  /* restore old cursor */
  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_last_cursor);
}


/*************************************************************************/
static do_middle()
{
/* processes the UP or DOWN action of the 
 * middle (move) mouse button.
 */
  if (ie.type==ButtonPress && mouse_mode==NORMAL)
  {
    /* see what object is below 
     * note: find_obj sets mouse_mode and either   
     * grobj_ptr, drobj_ptr or txobj_ptr depending
     * on what is below 
     */
    find_move_obj(cur_x,cur_y);
    base_x = old_move_x = cur_x;
    base_y = old_move_y = cur_y;

    /* if moving a drawn object, set first_move flag to TRUE
       (so that we can correctly set the first erase operation */
    if (mouse_mode==MOVE_DROBJ)
      first_move = TRUE;
  }   
  else if (ie.type==ButtonRelease && mouse_mode!=NORMAL)
  {
    drop_move_obj(cur_x, cur_y);
  }
}


/*************************************************************************/
static do_link_mode()
{
/* process an input event on the display panel while
 * in "link"  mode
 */
  struct grobj *ptr;

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   if (ie.type==LeaveNotify)
   {
     if (mouse_mode!=NORMAL)
       drop_move_obj(cur_x, cur_y);
     return;
   }

   /**********************************************/
   /* mouse being moved while button pushed DOWN */
   /**********************************************/
   if (ie.type==MotionNotify)
   {
     cont_move();
     return;
   }

   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   if (ie.xbutton.button==MS_MIDDLE)
   {
      do_middle(); 
   }

   /**************************/
   /* left mouse button DOWN */
   /**************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_LEFT))
   {
     gi_do_info(cur_x, cur_y, TRUE, -1, -1, gi_cur_ipanel);
   }

   /***********************************/    
   /* right  mouse button pushed DOWN */
   /***********************************/    
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_RIGHT))
   {
     /* if there is a unit at this location make it
        the target and set up gi_reshow so that
        new values are gotten for all the objects
     */
     if ((ptr=gi_find_xy(cur_x, cur_y))!=NULL)
     {
       gi_set_new_target(ptr,0); 
       reshow = TRUE;
     }
   }
}


/*************************************************************************/
static do_text_mode()
{
/* process an input event in text mode on
 * the display screen
 */

   int i;
   struct txobj *ptr;
   char buf[MAX_FONT_LENGTH+1];

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   if (ie.type==LeaveNotify)
   {
     if (mouse_mode!=NORMAL)
     {
       /* drop object currently being moved */
       drop_move_obj(cur_x, cur_y);
     }
     return;
   }

   /****************************************/
   /* mouse moved while button pushed DOWN */
   /****************************************/
   if (ie.type==MotionNotify)
   {
     cont_move();
     return;
   }

   /******************/
   /* keyboard input */
   /******************/
   if (ie.type==KeyPress && blob_set)
   {
     process_key();
     return;
   }
       
   /* if text is currently being entered any other significant
      mouse action will cause the current text object to be
      saved immediately
   */

   if (blob_set && ie.type==ButtonPress)
   {
     /* erase text blob */
     clear_blob();

     /* if any current text, save it */
     if (cur_text_len > 0)
       save_text();
   }

   /**************************************************************/
   /* left or right mouse button down while in MOVE_DISPLAY mode */
   /**************************************************************/
   else if ((ie.type==ButtonPress)&&(mouse_mode==MOVE_DISPLAY &&
				     (ie.xbutton.button==MS_LEFT ||
				      ie.xbutton.button==MS_RIGHT)))
   {
     /* jump move the display */
     jump_move();
   }

   /**************************/
   /* left mouse button DOWN */
   /**************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_LEFT))
   {

     /* gi_cur_font should be set, but if not, this should save us */
     if (gi_cur_font==NULL) gi_cur_font = &gi_default_font;

     /* write blob out */
     write_blob(ie.xbutton.x+1,ie.xbutton.y+5);
   }

   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   else if (ie.xbutton.button==MS_MIDDLE)
   {
     do_middle();
   }

   /***************************/
   /* right mouse button DOWN */
   /***************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_RIGHT))
   {
      if (deltext_ptr==NULL)
      {
        if ((deltext_ptr=gi_find_txobj(cur_x,cur_y))!=NULL)
        {
          gi_put_message("Confirm delete with right button");
	  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_delete_cursor);
        }
      }
      else
      {
        gi_put_message("Delete confirmed");
	XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_last_cursor);

	/* blank out the the object on the screen */
        gi_display_txobj(deltext_ptr,GXxor);

	/* delete text object from chain */
        gi_erase_txobj(deltext_ptr);
        deltext_ptr = NULL;

        /* if logging enabled, log appropriate delete command */

        if (gi_log_on) gi_log_delete(cur_x, cur_y);
      }
   }
}


/*************************************************************************/
static do_draw_mode()
{
/* process an input event in draw mode on
 * the display screen
 */
   int i;
   struct drobj *ptr;

   /*************************************/
   /* mouse has left the display window */
   /*************************************/
   if (ie.type==LeaveNotify)
   {
     if (mouse_mode!=NORMAL)
       drop_move_obj(cur_x, cur_y);
     else if (vertex_set)
     {
       if (gi_draw_type==LINES)
         draw_lines();
       else if (gi_draw_type==BOXES)
         draw_box(0);
       else if (gi_draw_type==BOUND)
         draw_box(BOUND_BOX);
     }
     return;
   }

   /************************/
   /* mouse is being moved */
   /************************/
   if (ie.type==MotionNotify && vertex_set)
   {
     if (gi_draw_type==LINES)
     {
       XSetFunction(XtDisplay(gi_tool), gi_gc, GXxor);
       for (i=0; i<2; i++)
       {
         /* on first pass, erase the previous line */
         /* on second pass, write new line */
	 XDrawLine(XtDisplay(gi_tool), gi_gfx, gi_gc,
	           old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y,
	           last_x-gi_origin_x, last_y-gi_origin_y);
	  /* update last vertex values */
          last_x = cur_x;
          last_y = cur_y;
       }
     }
     else /* if (gi_draw_type==BOXES) */
     {
       XSetFunction(XtDisplay(gi_tool), gi_gc, GXxor);
       /* erase the old box and draw a new one */
       for (i=0; i<2; i++)
       {
	 /* on first pass, erase old box */
	 /* on next pass, write out new box */
	 XDrawLine(XtDisplay(gi_tool), gi_gfx, gi_gc,
		 old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 old_vertex_x-gi_origin_x, last_y-gi_origin_y);
	 XDrawLine(XtDisplay(gi_tool), gi_gfx, gi_gc,
		 old_vertex_x-gi_origin_x, last_y-gi_origin_y, 
                 last_x-gi_origin_x, last_y-gi_origin_y);
	 XDrawLine(XtDisplay(gi_tool), gi_gfx, gi_gc,
		 last_x-gi_origin_x, last_y-gi_origin_y, 
                 last_x-gi_origin_x, old_vertex_y-gi_origin_y);
	 XDrawLine(XtDisplay(gi_tool), gi_gfx, gi_gc,
		 last_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y);

         /* set up new corner coordinates */		
         last_x = cur_x;
         last_y = cur_y;
       }
     }
   }

   /**********************************************/
   /* mouse being moved while button pushed DOWN */
   /**********************************************/
   if (ie.type==MotionNotify)
   {
     cont_move();
     return;
   }

   /* left or right mouse button down while in MOVE_DISPLAY mode */
   if ((ie.type==ButtonPress)&&(mouse_mode==MOVE_DISPLAY &&
				(ie.xbutton.button==MS_LEFT ||
				 ie.xbutton.button==MS_RIGHT)))
   {
     /* jump move the display */
     jump_move();
   }

   /*********************************/
   /* left mouse button pushed DOWN */
   /*********************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_LEFT 
				     && mouse_mode==NORMAL))
   {
     /* see if a vertex has already been set up */
     if (!vertex_set)
     {
       /* no vertex yet: save this position in relevant variables */
       old_vertex_x = last_x = cur_x;
       old_vertex_y = last_y = cur_y;
       vertex_set = TRUE;

       /* write out a single pixel at cursor "hot spot" */
       XDrawPoint(XtDisplay(gi_tool), gi_gfx, gi_gc,
		ie.xbutton.x, ie.xbutton.y);
     }
     else /* a vertex has already been set */
     {
       if (gi_draw_type==LINES)
       {
         /* check if this is near the last vertex set */
         if (cur_draw==NULL) /* 2nd vertex */
         {
	   /* set up a new graphic object for this drawing */
	   cur_draw = gi_remember_drobj(old_vertex_x, old_vertex_y,
			             cur_x, cur_y, 0);
	   old_vertex_x = cur_x;
	   old_vertex_y = cur_y;
         }
         else if (vertex_nearby(cur_x, cur_y))
         {
           /* finish drawing the object */
           draw_lines();
         }
         else /* add another vertex to this object */
         {
           /* limit the size of each object to 10 vertices */
           if (cur_draw->num_vertices >= MAX_VERTICES)
           {
             /* over vertex limit: finish up old object and
              * then create a new object and continue
              */
             draw_lines();
             vertex_set = TRUE;
	     cur_draw = gi_remember_drobj(old_vertex_x, old_vertex_y,
			             cur_x, cur_y, 0);
           }

           else
           {
             /* save this vertex with the object */
	     gi_save_vertex(cur_draw, cur_x, cur_y);
           }

           /* update the old vertex values to the current vertex */
	   old_vertex_x = cur_x;
	   old_vertex_y = cur_y;
         }
       }

       else if (gi_draw_type==BOXES)
       {
         draw_box(0);
       }

       else if (gi_draw_type==BOUND)
       {
         draw_box(BOUND_BOX);
       }
     }
   }


   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   else if (ie.xbutton.button==MS_MIDDLE)
   {
      do_middle();
   }

   /***************************/
   /* right mouse button DOWN */ 
   /***************************/
   else if ((ie.type==ButtonPress)&&(ie.xbutton.button==MS_RIGHT))
   {
      if (deldraw_ptr==NULL)
      {
        if ((deldraw_ptr=gi_find_drobj(cur_x,cur_y))!=NULL)
        {
          gi_put_message("Confirm delete with right button");
	  XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_delete_cursor);
        }
      }
      else
      {
        gi_put_message("Delete confirmed");
	XDefineCursor(XtDisplay(gi_tool), gi_gfx, gi_last_cursor);

	/* blank out the the object on the screen */
        gi_display_drobj(deldraw_ptr,GXclear);
	/* delete text object from chain */
        gi_erase_drobj(deldraw_ptr);
        deldraw_ptr = NULL;
        
        /* if logging enabled, log appropriate delete command */

        if (gi_log_on) gi_log_delete(cur_x, cur_y);
      }
   }
}


      
