/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*--------------------------------------------------------------------------
  Author: Nigel Goddard
  Date: May 1 1987
----------------------------------------------------------------------------*/

/*  Functions for debugging during network construction */

#ifdef BSIM
#  include "bflysim.h"
#else
#  include "uniproc.h"
#endif
#include "lex.h"
#include <ctype.h>				  /* token predicates */

extern long random();
extern char *strcat();

extern char* NameToType();

static show_errors(), read_line(), do_line();

typedef union m_type
{
  int    unit;
  Site * site;
  Link * link;
}  make_type;

/** set by DebugLex, the command parser **/

static char YYarg[MAX_COMMAND_LEN];		  /* text matched */
static char command_line[MAX_COMMAND_LEN];        /* command line chars */
static int current;				  /* command_line index */

static int saved = 0;				  /* if not 0, UnDebugLex value */

#define GETACHAR (command_line[current++])	  /* gets one command char */
#define PUTACHAR (current -= 1)			  /* "unget" one character */

static int debugtype;
static pot_type Gipot,Gpot;
static data_type Gdata;
static int Gistate,Gstate,Gunit,Gfrom;
static Output Gout;
static weight_type Gweight;
static func_ptr Gfunc;
static char Gtype[80];
	  
static shortval(x)
     int x;

/* returns TRUE if x is a short , FALSE otherwise */

{
  if (x < 32737 && x > -32767)
    return TRUE;
  else
    {
      LOGfprintf(stderr,"Value not in range of a short ingeter: -/+32767\n");
      return FALSE;
    }
}

/*---------------------------------------------------------------------------
  Resets values for call that caused debugging to occur.  Called by Cmd_set.
   type is field to be reset, value is integer to reset to.
----------------------------------------------------------------------------*/

si_Set_Debug_Value(type,val)
     int type, val;

{
  NameDesc nte;
  Site * sp;

BFLYCATCH
 if ((type == STATE || type == ISTATE) && shortval(val)) /* check short */
  switch (type)
    {
    case UNIT:
    case TO:
    case FROM:
      if (!LegalUnit(val))
	LOGfprintf(stderr,
		"Unit number %d out of range 0 to %d\n",val,NoUnits-1);
      else
	switch (type)
	  {
	  case UNIT:
	    if (debugtype != ADD_SITE)
	      {
		LOGfprintf (stderr,"Can only set the site `unit' in AddSite\n");
		return;
	      }
	    Gunit = val;
	    Errors &= (0xffffffff ^ SITE_UNIT_ERROR);
	    break;
	  case FROM:
	    if (debugtype != MAKE_LINK)
	      {
		LOGfprintf (stderr,"Can only set the `from unit' in MakeLink\n");
		return;
	      }
	    Gfrom = val;
	    Errors &= (0xffffffff ^ FROM_UNIT_ERROR);
	    break;
	  case TO:
	    if (debugtype != MAKE_LINK)
	      {
		LOGfprintf (stderr,"Can only set the `to unit' in MakeLink\n");
		return;
	      }
	    Gunit = val;
	    Errors &= (0xffffffff ^ TO_UNIT_ERROR);
	    if (FindName(Gtype,&nte) != NULL) /* check if site now ok */
	      {
		for (sp = UnitList[Gunit].sites;
		     sp != NULL && sp->name != nte.name;
		     sp = sp->next);
		if (sp != NULL)			  /* site name now valid */
		  Errors &= (0xffffffff ^ BAD_SITE_NAME);
		else
		  Errors |= BAD_SITE_NAME;
	      }
	    break;
	  }
      break;
    case STATE:
      Gstate = val;
      Errors &= (0xffffffff ^ STATE_VAL_ERROR);
      break;
    case ISTATE:
      Gistate = val;
      Errors &= (0xffffffff ^ ISTATE_VAL_ERROR);
      break;
    default:
      LOGfprintf(stderr,
		 "Internal error: invalid type passed to Set_Debug_Value: %d\n",
		 type);
    }
BFLYTHROW("Set_Debug_Value",MySimNumber)
}

/*---------------------------------------------------------------------------
  Version of preceding function to deal with float/int compatiblity.  All
  the values set in here may be floats or ints.  FLINT is defined in common.h
----------------------------------------------------------------------------*/

si_Set_Debug_Value_FLINT(type,val)
     int type;
     FLINT val;

{
BFLYCATCH
  switch (type)
    {
    case POT:
      Gpot = val;
      Errors &= (0xffffffff ^ POT_VAL_ERROR);
      break;
    case OUT:
      Gout = val;
      Errors &= (0xffffffff ^ OUT_VAL_ERROR);
      break;
    case IPOT:
      Gipot = val;
      Errors &= (0xffffffff ^ IPOT_VAL_ERROR);
      break;
    case WEIGHT:
      if (debugtype != MAKE_LINK)
	{
	  LOGfprintf (stderr,"Can only set the weight in MakeLink\n");
	  return;
	}
      Gweight = val;
      Errors &= (0xffffffff ^ WEIGHT_VAL_ERROR);
      break;
    case DATA:
      Gdata = val;
      break;
    default:
      LOGfprintf(stderr,
		 "Internal error: invalid type passed to Set_Debug_Value_FLINT: %d\n",
		 type);
    }
BFLYTHROW("Set_Debug_Value_FLINT",MySimNumber)
}

/*---------------------------------------------------------------------------
  Version of preceding functions to deal with strings.
----------------------------------------------------------------------------*/

si_Set_Debug_String(type,str)
     int type;
     char * str;

{
  NameDesc  nte;
  func_ptr func;
  Site * sp;

BFLYCATCH
  switch (type)
    {
    case FUNC:
      if ((func = NameToFunc(str)) != NULL)
	{
	  Gfunc = func;
	  Errors &= (0xffffffff ^ NO_FUNC_NAME);
	}
      else
	LOGfprintf(stderr,"%s is not a known function\n",str);
      break;
    case TYPE:					  /* in MakeUnit */
      if (FindName(str,&nte) != NULL && nte.type != TYPE_SYM)
	LOGfprintf(stderr,
		"name %s is already in use for another purpose, not a unit type\n",
		str); 
      else
	{
	  (void) sprintf(Gtype,"%s",str);
	  Errors &= (0xffffffff ^ NO_TYPE_NAME);
	}
      break;
    case SITE:
      switch (debugtype)
	{
	case ADD_SITE:
	  if (FindName(str,&nte) != NULL && nte.type != SITE_SYM)
	    LOGfprintf(stderr,
		    "name %s is already in use for another purpose, not a site name\n",
		    str);
	  else
	    {
	      (void) sprintf(Gtype,"%s",str);
	      Errors &= (0xffffffff ^ NO_SITE_NAME);
	    }
	  break;
	
	case MAKE_LINK:
	  if (FindName(str,&nte) == NULL)
	    LOGfprintf(stderr,"%s is not a known name\n",str);
	  else
	    if (nte.type != SITE_SYM)
	      LOGfprintf(stderr,"%s is not a site name\n",str);
	    else
	      if (!(Errors & TO_UNIT_ERROR))
		{
		  for (sp = UnitList[Gunit].sites; /*search sites for match*/
		       sp != NULL && sp->name != nte.name;
		       sp = sp->next);
		  if (sp == NULL)
		    LOGfprintf(stderr,
			    "No site named %s attached to unit %d\n",
			    str,Gunit);
		  else
		    if (sp->no_inputs > 32766)
		      LOGfprintf(stderr,
			      "Already too many inputs to site %s on unit %d\n",
			      str,Gunit);
		    else
		      {
			(void) sprintf(Gtype,"%s",str);
		        Errors &= (0xffffffff ^ (BAD_SITE_NAME|NO_SITE_NAME));
		      }
		}
	      else				  /* to unit not correct */
		{
		  (void) sprintf(Gtype,"%s",str);
		  LOGfprintf(stderr,"To unit number %d is incorrect, cannot check site %s validity\n",Gunit,str);
		}
	  break;

	default:
	  LOGfprintf(stderr,"? Cannot set site name at this time\n");
	} /*switch*/	
      break;
      
    default:
	LOGfprintf(stderr,
		"Internal error: invalid type passed to Set_Debug_String: %d\n",
		type);
      }
BFLYTHROW("Set_Debug_String",MySimNumber)
}

/*---------------------------------------------------------------------------
  Corrects all remaining errors to the simulator default values
----------------------------------------------------------------------------*/

si_Set_Debug_Default()

{
  int s_Debug;

BFLYCATCH
  if (debugtype == MAKE_LINK && !AutoFix)
    {
      LOGfprintf(stderr,"No defaults for links, links not necessary for building\n");
      return;
    }
  if (Errors & NO_TYPE_NAME)
    (void) sprintf(Gtype,"%s","NullTypeName");
  if (Errors & NO_FUNC_NAME)
    Gfunc = (func_ptr)NullFunc;
  if (Errors & IPOT_VAL_ERROR)
    Gipot = 0;
  if (Errors & POT_VAL_ERROR)
    Gpot = 0;
  if (Errors & OUT_VAL_ERROR)
    Gout = 0;
  if (Errors & WEIGHT_VAL_ERROR)
    Gweight = 0;
  if (Errors & ISTATE_VAL_ERROR)
    Gistate = 0;
  if (Errors & STATE_VAL_ERROR)
    Gstate = 0;
  if (Errors & SITE_UNIT_ERROR)
    Gunit = random() % NoUnits;
  if (Errors & NO_SITE_NAME)
    if (AutoFix)
      {
	LOGfprintf(stderr,
		   "Setting name table entry for %s to be a site name\n",
		   Gtype);
	AlterName(Gtype,SITE_SYM,0,0,0);
      }
    else
      (void) sprintf(Gtype,"%s","NullSiteName");
  if (Errors & FROM_UNIT_ERROR)
    Gfrom = 0;			/* links from unit 0 */
  if (Errors & BAD_SITE_NAME && !(Errors & TO_UNIT_ERROR))
    if (AutoFix)
      {
	LOGfprintf(stderr,"Adding site %s to unit %d\n",Gtype,Gunit);
	s_Debug = Debug;
	Debug = FALSE;
	AddSite(Gunit,Gtype,Gfunc,Gdata);
	Debug = s_Debug;
      }  

  Errors &= (NO_MORE_UNITS | TO_UNIT_ERROR); /* can't fix NO_MORE_UNITS */
BFLYTHROW("Set_Debug_Default",MySimNumber)
}

/*---------------------------------------------------------------------------
  Prints out to stderr the current settings for the call that caused
  debugging to happen.  Calls show_debug_error to print remaining errors
----------------------------------------------------------------------------*/

si_Show_Debug_Values()

{
BFLYCATCH
  switch (debugtype)
    {
    case MAKE_UNIT:
      LOGfprintf(stderr,"    Current values for MakeUnit are:\n");
      LOGfprintf(stderr,"\t- type = %s\n\t- function = %s\n\t- state = %d\n\t- init state = %d\n\t- potential = ",Gtype,FuncToName(Gfunc),Gstate,Gistate); 
      LOGfprintf(stderr,POT_S_TYPE,Gpot);
      LOGfprintf(stderr,"\n\t- init potential = ");
      LOGfprintf(stderr,POT_S_TYPE,Gipot);
      LOGfprintf(stderr,"\n\t- output = ");
      LOGfprintf(stderr,OUT_S_TYPE,Gout);
      LOGfprintf(stderr,"\n\t- data = ");
      LOGfprintf(stderr,DATA_S_TYPE,Gdata);
      LOGfprintf(stderr,"\n");
      break;

    case ADD_SITE:
      LOGfprintf(stderr,"    Current values for AddSite are:\n");
      LOGfprintf(stderr,"\t- unit = %d\n\t- sitename = %s\n\t- function = %s\n\t- data = ",Gunit,Gtype,FuncToName(Gfunc));
      LOGfprintf(stderr,DATA_S_TYPE,Gdata);
      LOGfprintf(stderr,"\n");
      break;

    case MAKE_LINK:
      LOGfprintf(stderr,"    Current values for MakeLink are:\n");
      LOGfprintf(stderr,"\t- from = %d\n\t- to = %d\n\t- sitename = %s\n\t- function = %s\n\t- weight = ",Gfrom,Gunit,Gtype,FuncToName(Gfunc));
      LOGfprintf(stderr,WEIGHT_S_TYPE,Gweight);
      LOGfprintf(stderr,"\n\t- data = ");
      LOGfprintf(stderr,DATA_S_TYPE,Gdata);
      LOGfprintf(stderr,"\n");

      break;

    default:
      LOGfprintf(stderr,
	      "Internal error: invalid debugtype in Show_Debug_Values\n");
      return;
    } /*case*/

  if (Errors)
    {
      LOGfprintf(stderr,"    remaining errors are;\n");
      show_errors();
      LOGfprintf(stderr,"    fix and quit, or type ignore\n");
    }
  else
    LOGfprintf(stderr,"    all values are ok, quit to continue building.\n");
BFLYTHROW("Show_Debug_Values",MySimNumber)
}

/*---------------------------------------------------------------------------
  Prints remaining errors.
----------------------------------------------------------------------------*/

static show_errors()

{
  char * str;

BFLYCATCH
      if (Errors & NO_MORE_UNITS)
	LOGfprintf(stderr,"\t- out of unit space (unrecoverable)\n");
      if (Errors & NO_TYPE_NAME)
	LOGfprintf(stderr,"\t- %s is already in use, not a type name\n",Gtype);
      if (Errors & NO_FUNC_NAME)
	{
	  if (debugtype == MAKE_UNIT)
	    str = "unit";
	  else
	    if (debugtype == ADD_SITE)
	      str = "site";
	    else
	      str = "link";
	  LOGfprintf(stderr,"\t- Unknown %s function\n",str);
	}
      if (Errors & IPOT_VAL_ERROR)
	LOGfprintf(stderr,
		   "\t- initial potential value %d out of range -/+32767\n",
		   Gipot);
      if (Errors & POT_VAL_ERROR)
	LOGfprintf(stderr,
		   "\t- potential value %d out of range -/+32767\n",Gpot);
      if (Errors & OUT_VAL_ERROR)
	LOGfprintf(stderr,"\t- output value %d out of range -/+32767\n",Gout);
      if (Errors & WEIGHT_VAL_ERROR)
	LOGfprintf(stderr,
		   "\t- weight value %d out of range -/+32767\n",Gweight);
      if (Errors & ISTATE_VAL_ERROR)
	LOGfprintf(stderr,
		   "\t- initial state value %d out of range -/+32767\n",
		   Gistate);
      if (Errors & STATE_VAL_ERROR)
	LOGfprintf(stderr,"\t- state value %d out of range -/+32767\n",Gstate);
      if (Errors & SITE_UNIT_ERROR)
	LOGfprintf(stderr,
		   "\t- unit %d for site out of range 0 to %d\n",
		   Gunit,NoUnits-1);
      if (Errors & NO_SITE_NAME)
	LOGfprintf(stderr,
		   "\t- %s is not a site name, is a %s\n",
		   Gtype,NameToType(Gtype));
      if (Errors & FROM_UNIT_ERROR)
	LOGfprintf(stderr,
		   "\t- `from' unit number %d is out of range 0 to %d\n",
		   Gfrom,NoUnits-1);
      if (Errors & TO_UNIT_ERROR)
	LOGfprintf(stderr,"\t- `to' unit number %d is out of range 0 to %d\n",
		   Gunit,NoUnits-1);
      if (Errors & BAD_SITE_NAME)
	LOGfprintf(stderr,
		"\t- site with name %s does not exist on `to' unit %d\n",
		Gtype,Gunit);
BFLYTHROW("show_errors",MySimNumber)
}

/*---------------------------------------------------------------------------
  Major function called by all debug enhanced simulator routines.  debugtype
  gives the type of call (make link, site, unit).  Returns either unit index,
  site pointer or link pointer.  Calls debug_command_reader to allow the
  user to fix the errors.
----------------------------------------------------------------------------*/

static make_type debug_errors ()

{
  make_type result;
  int s_Debug;

BFLYCATCH
  switch (debugtype)
    {
    case MAKE_UNIT:
      LOGfprintf(stderr,"    The following errors were encountered while trying to\n    make unit %d of type %s and function %s:\n",NoUnits,Gtype,FuncToName(Gfunc));
      show_errors();

      if (Errors & NO_MORE_UNITS)
	{
	  debug_command_reader("debug");
	  exit(0);
	}
      else
	if (AutoFix)
	  {
	    LOGfprintf(stderr,"    Continuing with default corrections\n");
	    si_Set_Debug_Default();
	    Debug = FALSE;
	    result.unit = MakeUnit(Gtype,Gfunc,Gipot,Gpot,
				   Gdata,Gout,Gistate,Gstate);
	    Debug = TRUE;
	  }
	else
	  {
	    do
	      {
		if (debug_command_reader("debug") == IGNORE)
		  {
		    LOGfprintf(stderr,"Ignoring error, not constructing unit\n");
		    result.unit = -1;
		    return result;
		  }
		if (Errors)
		  {
		    LOGfprintf(stderr,"    There are still errors in the unit definition:\n");
		    show_errors();
		  }
	      }
	    while (Errors);

	    s_Debug = Debug;
	    Debug = FALSE;
	    result.unit = MakeUnit(Gtype,Gfunc,Gipot,Gpot,
				   Gdata,Gout,Gistate,Gstate);
	    Debug = s_Debug;
	  }
      break;
      
    case ADD_SITE:
      LOGfprintf(stderr,"    The following errors were encountered while trying to\n    add site %s with function %s to unit %d:\n",Gtype,FuncToName(Gfunc),Gunit);
      show_errors();

      if (AutoFix)
	{
	  LOGfprintf(stderr,"    Continuing with default corrections\n");
	  si_Set_Debug_Default();
	  Debug = FALSE;
	  result.site = AddSite(Gunit,Gtype,Gfunc,Gdata);
	  Debug = TRUE;
	}
      else
	{
	  do
	    {
	      if (debug_command_reader("debug") == IGNORE)
		{	
		  LOGfprintf(stderr,"Ignoring error, not constructing site\n");
		  result.site = NULL;
		  return result;
		}
	      if (Errors)
		{
		  LOGfprintf(stderr,"    There are still errors in the site definition:\n");
		  show_errors();
		}
	    }
	  while (Errors);

	  s_Debug = Debug;
	  Debug = FALSE;
	  result.site = AddSite(Gunit,Gtype,Gfunc,Gdata);
	  Debug = s_Debug;
	}
      break;

    case MAKE_LINK:
      LOGfprintf(stderr,"    The following errors were encountered while trying to\n    make a link with function %s from unit %d to unit %d:\n",FuncToName(Gfunc),Gfrom,Gunit);
      show_errors();

      if (AutoFix)
	if (Errors & TO_UNIT_ERROR) /* can't fix */
	  LOGfprintf(stderr,"Can't fix error in to unit: ignoring link\n");
	else
	  {
	    LOGfprintf(stderr,"    Continuing with default corrections\n");
	    si_Set_Debug_Default();
	    Debug = FALSE;
	    result.link = MakeLink(Gfrom,Gunit,Gtype,Gweight,
				   Gdata,Gfunc);
	    Debug = TRUE;
	  }
      else
	{
	  do
	    {
	      if (debug_command_reader("debug") == IGNORE)
		{
		  LOGfprintf(stderr,"Ignoring error, not constructing link\n");
		  result.link = NULL;
		  return result;
		}
	      if (Errors)
		{
		  LOGfprintf(stderr,"    There are still errors in the link definition:\n");
		  show_errors();
		}
	    }
	  while (Errors);
	  
	  s_Debug = Debug;
	  Debug = FALSE;
	  result.link = MakeLink(Gfrom,Gunit,Gtype,Gweight,
				 Gdata,Gfunc);
	  Debug = s_Debug;
	}
      break;

    default:
      LOGfprintf(stderr,"internal error: debug called with invalid type\n");
    }
  return result;
BFLYTHROW("debug_errors",MySimNumber)
}

/*---------------------------------------------------------------------------
  For nesting debug situations.  Input parameters are all pointers to local
  variables in the function that called this.  Simply saves globals in these
  locals, increments Debug and exits.
----------------------------------------------------------------------------*/

static pushdebuglevel(s_debugtype,s_Errors,s_Gipot,s_Gpot,s_Gdata,
		      s_Gistate,s_Gstate,s_Gout,s_Gunit,s_Gfrom,s_Gweight,
		      s_Gfunc,s_Gtype)
     int * s_debugtype;
     unsigned int * s_Errors;
     FLINT * s_Gipot, * s_Gpot;
     data_type * s_Gdata;
     int * s_Gistate, * s_Gstate, * s_Gunit, * s_Gfrom;
     FLINT * s_Gout;
     FLINT * s_Gweight;
     func_ptr * s_Gfunc;
     char * s_Gtype;

{
BFLYCATCH
  *s_debugtype = debugtype;
  *s_Errors = Errors;
  *s_Gipot = Gipot;
  *s_Gpot = Gpot;
  *s_Gdata = Gdata;
  *s_Gistate = Gistate;
  *s_Gstate = Gstate;
  *s_Gout = Gout;
  *s_Gunit = Gunit;
  *s_Gfrom = Gfrom;
  *s_Gweight = Gweight;
  *s_Gfunc = Gfunc;
  (void) sprintf(s_Gtype,"%s",Gtype);
  Debug++;			/* increment debug level */
BFLYTHROW("pushdebuglevel",MySimNumber)
}

/*---------------------------------------------------------------------------
  For nesting debug situations.  Input parameters are all pointers to local
  variables in the function that called this.  Simply restores globals from
  these locals, deccrements Debug and exits
----------------------------------------------------------------------------*/

static popdebuglevel(s_debugtype,s_Errors,s_Gipot,s_Gpot,s_Gdata,
		      s_Gistate,s_Gstate,s_Gout,s_Gunit,s_Gfrom,s_Gweight,
		      s_Gfunc,s_Gtype)
     int * s_debugtype;
     unsigned int * s_Errors;
     FLINT * s_Gipot, * s_Gpot;
     data_type * s_Gdata;
     int * s_Gistate, * s_Gstate, * s_Gunit, * s_Gfrom;
     FLINT * s_Gout;
     FLINT * s_Gweight;
     func_ptr * s_Gfunc;
     char * s_Gtype;

{
BFLYCATCH
  debugtype = *s_debugtype;
  Errors = *s_Errors;
  Gipot = *s_Gipot;
  Gpot = *s_Gpot;
  Gdata = *s_Gdata;
  Gistate = *s_Gistate;
  Gstate = *s_Gstate;
  Gout = *s_Gout;
  Gunit = *s_Gunit;
  Gfrom = *s_Gfrom;
  Gweight = *s_Gweight;
  Gfunc = *s_Gfunc;
  (void) sprintf(Gtype,"%s",s_Gtype);
  if (Debug > 0)
    Debug--;			/* decrement debug level */
BFLYTHROW("popdebuglevel",MySimNumber)
}

/*---------------------------------------------------------------------------
  The function called by the command function that makes a unit,
  Cmd_MakeUnit.  Simply pushes debug level, calls function to make unit,
  pops debug level and exits.
----------------------------------------------------------------------------*/

int si_DebugMakeUnit(type,func,ipot,pot,data,out,istate,state)
     char *type;
     func_ptr func;
     FLINT ipot,pot,data,out;
     int istate,state;

{
  int s_debugtype;
  unsigned int s_Errors;
  FLINT s_Gipot,s_Gpot,s_Gdata,s_Gweight;
  int s_Gistate,s_Gstate;
  int s_Gout,s_Gunit,s_Gfrom;
  func_ptr s_Gfunc;
  char s_Gtype[80];
  int result;

BFLYCATCH
  pushdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  result = MakeUnit(type,func,ipot,pot,data,out,
		    istate,state);

  popdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  return result;
BFLYTHROW("DebugMakeUnit",MySimNumber)
}

/*---------------------------------------------------------------------------
  Checks the unit specification, and makes it if ok.  If not, accumulates
  errors in Error bit vector, and calls debug_errors to have them fixed.l
  Returns the index of the unit made.
----------------------------------------------------------------------------*/
  
int si_debug_makeunit(type,func,ipot,pot,data,out,istate,state)
     char *type;
     func_ptr func;
     FLINT ipot,pot,data,out;
     int istate,state;

{
  int i;
  NameDesc nte;
  make_type result;
  int s_Debug;

BFLYCATCH
  Errors = 0;					  /* expect no errors */
  if (NoUnits >= LastUnit)
    si_AutoAllocateUnits(/*200*/);	          /* default allocation */
  if (FindName(type,&nte) != NULL && nte.type != TYPE_SYM)
    Errors |= NO_TYPE_NAME;
  if (func != NULL && !strcmp(FuncToName(func),"**NO NAME**"))
    Errors |= NO_FUNC_NAME;
#ifndef FSIM
  if (ipot > 32767 || ipot < -32767)
    Errors |= IPOT_VAL_ERROR;
  if (pot > 32767 || pot < -32767)
    Errors |= POT_VAL_ERROR;
  if (out > 32767 || out < -32767)
    Errors |= OUT_VAL_ERROR;
#endif
  if (istate > 32767 || istate < -32767)
    Errors |= ISTATE_VAL_ERROR;
  if (state > 32767 || state < -32767)
    Errors |= STATE_VAL_ERROR;
  if (!Errors)
    {
      s_Debug = Debug;
      Debug = FALSE;
      i = MakeUnit(type,func,ipot,pot,data,out,
		   istate,state);
      Debug = s_Debug;
      return i;
    }
  else
    {
      debugtype = MAKE_UNIT;
      (void) sprintf(Gtype,"%s",type);
      Gfunc = func;
      Gipot = ipot;
      Gpot  = pot;
      Gdata = data;
      Gout  = out;
      Gistate = istate;
      Gstate  = state;
#ifdef BFLY
      LockStdin();
#endif
      result = debug_errors();
#ifdef BFLY
      UnlockStdin();
#endif
      return (result.unit);
    }
BFLYTHROW("debug_makeunit",MySimNumber)
}

/*---------------------------------------------------------------------------
  The function called by the command function that adds a site,
  Cmd_AddSite.  Simply pushes debug level, calls function to add site,
  pops debug level and exits.
----------------------------------------------------------------------------*/
Site * si_DebugAddSite(unit,name,func,data)
     int unit;
     FLINT data;
     char *name;
     func_ptr func;

{
  int s_debugtype;
  unsigned int s_Errors;
  FLINT s_Gipot,s_Gpot,s_Gdata,s_Gweight;
  int s_Gistate,s_Gstate;
  int s_Gout,s_Gunit,s_Gfrom;
  func_ptr s_Gfunc;
  char s_Gtype[80];
  Site * result;

BFLYCATCH
  pushdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  result = AddSite(unit,name,func,data);

  popdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  return result;
BFLYTHROW("DebugAddSite",MySimNumber)
}

/*---------------------------------------------------------------------------
  Checks the site specification, and makes it if ok.  If not, accumulates
  errors in Error bit vector, and calls debug_errors to have them fixed.l
  Returns the pointer to the site made.
----------------------------------------------------------------------------*/
  
Site * si_debug_addsite(unit,name,func,data)
     int unit;
     FLINT data;
     char *name;
     func_ptr func;
{
    Site *sp;
    NameDesc nte;
    make_type result;
    int s_Debug;

BFLYCATCH
    Errors = 0;
    if (!LegalUnit(unit))
      Errors |= SITE_UNIT_ERROR;
    if (FindName(name,&nte) != NULL && nte.type != SITE_SYM)
      Errors |= NO_SITE_NAME;
    if (func != NULL && !strcmp(FuncToName(func),"**NO NAME**"))
      Errors |= NO_FUNC_NAME;
    if (!Errors)
    {
      s_Debug = Debug;
      Debug = FALSE;
      sp = AddSite(unit,name,func,data);
      Debug = s_Debug;
      return sp;
    }
  else
    {
      debugtype = ADD_SITE;
      (void) sprintf(Gtype,"%s",name);
      Gfunc = func;
      Gdata = data;
      Gunit = unit;
#ifdef BFLY
      LockStdin();
#endif
      result = debug_errors();
#ifdef BFLY
      UnlockStdin();
#endif
      return (result.site);
    }
BFLYTHROW("debug_addsite",MySimNumber)
}

/*---------------------------------------------------------------------------
  The function called by the command function that makes a link,
  Cmd_MakeLink.  Simply pushes debug level, calls function to make link,
  pops debug level and exits.
----------------------------------------------------------------------------*/
Link * si_DebugMakeLink(from,to,site,weight,data,func)
     int from,to;
     FLINT weight,data;
     char *site;
     func_ptr func;

{
  int s_debugtype;
  unsigned int s_Errors;
  FLINT s_Gipot,s_Gpot,s_Gdata,s_Gweight;
  int s_Gistate,s_Gstate;
  int s_Gout,s_Gunit,s_Gfrom;
  func_ptr s_Gfunc;
  char s_Gtype[80];
  Link * result;

BFLYCATCH
  pushdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  result = MakeLink(from,to,site,weight,data,func);

  popdebuglevel(&s_debugtype, &s_Errors, &s_Gipot, &s_Gpot,
		 &s_Gdata, &s_Gistate, &s_Gstate, &s_Gout,
		 &s_Gunit, &s_Gfrom, &s_Gweight, &s_Gfunc, s_Gtype);

  return result;
BFLYTHROW("DebugMakeLink",MySimNumber)
}

/*---------------------------------------------------------------------------
  Checks the link specification, and makes it if ok.  If not, accumulates
  errors in Error bit vector, and calls debug_errors to have them fixed.
  Returns the pointer to the link made.
----------------------------------------------------------------------------*/

Link * si_debug_makelink(from,to,site,weight,data,func)
     int from,to;
     FLINT weight,data;
     char *site;
     func_ptr func;
{
   Link * ip;
   Site *sp;
   char *sitename;
   make_type result;
   int s_Debug;
   NameDesc  nte;

BFLYCATCH
   Errors = 0;					  /* expect no errors */
   if (!LegalUnit(from))
     Errors |= FROM_UNIT_ERROR;
   if (!LegalUnit(to))
     Errors |= TO_UNIT_ERROR;
#ifndef FSIM
   if (weight > 32767 || weight < -32767)
     Errors |= WEIGHT_VAL_ERROR;
#endif
   if (FindName(site,&nte) == NULL || nte.type != SITE_SYM)
     Errors |= NO_SITE_NAME;
   else
     if (!(Errors & TO_UNIT_ERROR))
       {
	 sitename = nte.name;
	 for (sp = UnitList[to].sites;		  /* search sites for match */
	      sp != NULL && sp->name != sitename;
	      sp = sp->next);
	 if (sp == NULL)
	   Errors |= BAD_SITE_NAME;
	 else
	   if (sp->no_inputs > 32766)
	     Errors |= TOO_MANY_LINKS;
       }
   if (func != NULL && !strcmp(FuncToName(func),"**NO NAME**"))
     Errors |= NO_FUNC_NAME;
   if (!Errors)
     {
       s_Debug = Debug;
       Debug = FALSE;
       ip = MakeLink(from,to,site,weight,data,func);
       Debug = s_Debug;
       return ip;
     }
   else
     {
       debugtype = MAKE_LINK;
       (void) sprintf(Gtype,"%s",site);
       Gfunc = func;
       Gdata = data;
       Gunit = to;
       Gfrom = from;
       Gweight = weight;
#ifdef BFLY
      LockStdin();
#endif
      result = debug_errors();
#ifdef BFLY
      UnlockStdin();
#endif
       return (result.link);
     }   
BFLYTHROW("debug_makelink",MySimNumber)
}

/*---------------------------------------------------------------------------
  For Butterfly.  Grabs and locks stdin, so that only one debug process can
  be talking to the user at a time.  Debugging runs on the sims, not the
  control node.
----------------------------------------------------------------------------*/

#ifdef BFLY
LockStdin()

{				/* InLock mapped in in main.c */
BFLYCATCH
  while(Atomic_add(InLock,1))
    {
      Atomic_add(InLock,-1);
      Sleep(1);
    }
BFLYTHROW("LockStdin",MySimNumber)
}

/*---------------------------------------------------------------------------
  Complimentary to preceding function
----------------------------------------------------------------------------*/

UnlockStdin()

{
BFLYCATCH
  Atomic_add(InLock,-1);		  /* unlock table */
BFLYTHROW("UnlockStdin",MySimNumber)
}
#endif

/*---------------------------------------------------------------------------
  For reading in command lines to debug interface.  Local to this file.
----------------------------------------------------------------------------*/

/***** DebugLex *****/
/*
  DebugLex puts the next token in YYarg ( a character string )
  and returns type of token found.  Command arrives in char string.
*/

static int DebugLex()
{
  int i,c;
#ifdef mips
  /* this is disgusting but we'll just do this to get it to compile for now */
  static char myeof = (char) EOF;
#else
#define myeof EOF
#endif

BFLYCATCH
  if(saved) {					  /* token is in saved */
    i = saved;					  /* note saved token  */
    saved = 0;					  /* reset saved to false */
    return i;					  /* return token */
  }
  
  while((c = GETACHAR) == ' ' || c == '\t');	  /* ignore whitespace */
  if(c == '\n' || c == '\0') return END_LINE;	  /* return if newline */
  if(c == myeof) return EOF;			  /* or end-of-file */
  if(c == '?')
    {
      YYarg[0] = '?';				  /* fill in YYarg */
      YYarg[1] = '\0';				  /* terminate string */
      return QUESTION_MARK;			  /* give help */
    }
  if(c == '"'){					  /* get quoted string */
    for(i = 1,YYarg[0] = c;			  /* save opening quote */
	((YYarg[i] = GETACHAR) != '"') &&	  /* read to closing quote */
	YYarg[i] != '\n';i++);			  /* or newline */
    YYarg[i+1] = '\0';				  /* add end of string char */
    return STRING;				  /* return type */
  }
  else {					  /* read to whitespace */
    for(i = 1,YYarg[0] = c;			  /* save first char */
	(c = GETACHAR) != ' ' && c != '\n' &&	  /* read to space or nl */
	c != '\0' && c != '\t' && c != myeof;	  /* or eos or tab or eof */
	i++)
      YYarg[i] = c;				  /* and save */
    YYarg[i] = '\0';				  /* add end of string char */
    PUTACHAR;					  /* read one too many */
    return TOKEN;				  /* return type */
  }
BFLYTHROW("DebugLex",MySimNumber)
}

/**** UnDebugLex ****/
/*
    Next call to DebugLex will return tok.
*/

static UnDebugLex(tok)
{
    saved = tok;
}

/*---------------------------------------------------------------------------
  This function runs the debug command interface.  The parameter is a
  string to be used in the prompt.   Debug  should probably be
  incremented before calling this function (the value of  Debug 
  is printed in the prompt) so that it is clear from the prompt how many
  layers of interfaces are running.  Returns when the quit command is
  issued, or any command whose command function returns non-zero.
----------------------------------------------------------------------------*/

debug_command_reader(str)
     char * str;

{
  int quit = FALSE;

BFLYCATCH
  while (!quit)
    {
      LOGfprintf(stdout,"%s[%d]> ",str,Debug);
      read_line(str);
      quit = do_line();
    }
  return quit;
BFLYTHROW("debug_command_reader",MySimNumber)
}

/*---------------------------------------------------------------------------
  Read a command line into the buffer.
----------------------------------------------------------------------------*/

static read_line(str)
     char * str;
     
{
  int i,c;
  
BFLYCATCH
  for(i = 0;i < MAX_COMMAND_LEN - 1;i++)
    {
      command_line[i] = c = getc(stdin);
      if (Logging)
	putc(c,LogFile);
      if (LogCmd)
	putc(c,CmdFile);
      if(i == 0 && c == ' ')
	i--;
      if(c == '\n')
	if(i == 0)
	  {
	    LOGfprintf(stdout,"%s[%d]> ",str,Debug);
	    i--;
	  }
	else
	  break;
      if(c == EOF) break;
    }
  if(i == MAX_COMMAND_LEN - 1)
    {
      LOGfprintf(stdout,"line too long (max %d characters): input ignored\n",
	     MAX_COMMAND_LEN);
      command_line[0] = '\0';
    }
  else
    command_line[i+1] = '\0';
  current = 0;
BFLYTHROW("read_line (debug)",MySimNumber)
}

/*---------------------------------------------------------------------------
  Call the function whose pointer is the argument, with an argc-argv
  structure as parameters, built with calls to DebugLex.
----------------------------------------------------------------------------*/

static CallFunc(func)
  int (* func)();

{
  char *argv[NUMARGS];
  int   argc = 0;
  int result;

BFLYCATCH
  do
    {
      argv[argc] = (char *) si_malloc(strlen(YYarg) + 1);
      strcpy(argv[argc++], YYarg);
    }
  while(DebugLex() != END_LINE);
  result = func(argc, argv);
  while (argc > 0)
    free(argv[--argc]);				  /* free up arg storage */
  return result;
BFLYTHROW("CallFunc (debug)",MySimNumber)
}  

/*---------------------------------------------------------------------------
  Using the command line read in by read_line, call DebugLex to get the first
  token.  Expect a command name, but others are possible.  Returns result
  of command function.
----------------------------------------------------------------------------*/

static do_line()

{
    int token, i, j;
    func_ptr func;
    char * cmd;
    char cmdbuff[MAX_COMMAND_LEN];
    MappingTable mte;

BFLYCATCH
    (void) sprintf(cmdbuff,"Debug_Cmd_");
    switch(token= DebugLex())
      {
      case TOKEN:				  /* should be command */
	(void) strcat(cmdbuff,YYarg);		  /* all commands prefix */
	if ((func = NameToFunc(cmdbuff)) == NULL) /* is it a debug command ? */
	  {					  /* no */
	    if ((func = NameToFunc(cmdbuff+6)) == NULL)	/* normal command ? */
	      {					  /* no */
		LOGfprintf(stdout,"%s is not a debug command! : input ignored\n", YYarg);
		if (NameToFunc(YYarg) != NULL)	  /* not command function */
		  LOGfprintf(stdout,"The function `%s' exists; use `call' to access it\n",
			 YYarg);
	      }
	    else				  /* it's a normal command */
	      return (CallFunc(func));		  /* call normal command */
	  }
	else					  /* it's a debug command */
	  return (CallFunc(func));		  /* call debug command */
	break;
      case STRING:				  /* unexpected */
	LOGfprintf(stdout,"? Command expected, quoted string found: %s\n", YYarg);
	break;
      case END_LINE : break;			  /* ignore blank lines */
      case EOF :
	clearerr(stdin);			  /* more reading to do */
	Cmd_quit(1,(char**)NULL);		  /* save and exit no error  */
      case QUESTION_MARK:
	PipeBegin();
	LOGfprintf(Dispf,"Debug Commands are:\n");
	for (i = j = 0; IndexToItem(i,&mte) != NULL; i++)
	  if ((cmd = mte.name) != NULL && strlen(cmd) > 11) /* not abbrev. */
	    {
	      (void) sprintf(cmdbuff,"%s",cmd);
	      cmdbuff[10] = '\0';
	      if (!strcmp(cmdbuff,"Debug_Cmd_"))
		if (j++%5 == 0)
		  {
		    si_termitem(Dispf);
		    si_additem(Dispf,"",cmd+10,(((j-1)%5)*15)+4,4);
		  }
		else
		  si_additem(Dispf,"",cmd+10,(((j-1)%5)*15)+4,4);
	    }
	si_termitem(Dispf);
	LOGfprintf(Dispf,"\nSome normal interface commands are available:\n");
	for (i = j = 0; IndexToItem(i,&mte) != NULL; i++)
	  if ((cmd = mte.name) != NULL && strlen(cmd) > 5) /* not abbrev. */
	    {
	      (void) sprintf(cmdbuff,"%s",cmd);
	      cmdbuff[4] ='\0';
	      if (!strcmp(cmdbuff,"Cmd_"))
		{
		  (void) sprintf(cmdbuff,"Debug_%s",cmd);
		  if (!NameToFunc(cmdbuff))	  /* no debug cmd same name */
		    if (j++%5 == 0)
		      {
			si_termitem(Dispf);
			si_additem(Dispf,"",cmd+4,(((j-1)%5)*15)+4,4);
		      }
		    else
		      si_additem(Dispf,"",cmd+4,(((j-1)%5)*15)+4,4);
		}
	    }
	si_termitem(Dispf);
	LOGfprintf(Dispf,"\nType `help <command>' or `command ?' for more information\n");
	PipeEnd();
	break;
      default :					  /* unknown */
	LOGfprintf(stdout,"Unknown command: %s(%d) ('?' for help)\n",YYarg,token);
	while(DebugLex() != END_LINE);		  /* ignore to end of line */
      }
    return FALSE;
BFLYTHROW("do_line (debug)",MySimNumber)
}











