/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*----------------------------------------------------------------------------
  Author: Nigel Goddard & Toby Mintz
  Date: May 1 1987
----------------------------------------------------------------------------*/

/* "makebind" takes one argument which will be the name of the output file.
 * ".bind.c" is concatonated onto the end of the filename.  The final
 * name must not be more than FN_LEN characters.  "makebind" reads
 * the procedures names from stdin.  The procedure names must be
 * separated by blanks of new lines.  The procedure names must not be
 * longer than PN_LEN characters.  "makebind" then outputs the 
 * resulting program to <filename>.bind.c .
 */

#include	<stdio.h>

extern char *malloc();
extern char *strcat();

/* file name length */
#define	FN_LEN	50
/* procedure name length */
#define	PN_LEN	50

#define TRUE 1
#define FALSE 0

static int UUD = FALSE;
static int USD = FALSE;
static int ULD = FALSE;
static int UUC = FALSE;
static int USC = FALSE;
static int ULC = FALSE;
static int UUR = FALSE;
static int USR = FALSE;
static int ULR = FALSE;
static int UUS = FALSE;
static int USS = FALSE;
static int ULS = FALSE;
static int UUL = FALSE;
static int USL = FALSE;
static int ULL = FALSE;
static int ULI = FALSE;   
static int UHI = FALSE;

struct proc {
	char	p_name[PN_LEN+1];	/* Procedure Name */
	struct proc	*p_next;	/* Pointer to next */
} *procs = 0;

/* add_proc Creates a new proc cell and stores "name."  Returns a pointer
 * to the new proc cell.
 */

static strcmpi(s1,s2)
     char * s1;
     char * s2;

{
  for ( ; *s1 != '\0' && *s1 == *s2; s1++,s2++);
  return (*s1 == '\0');
}

/*---------------------------------------------------------------------------
  checks if the current function is one of the ones for handling user
  defined data structures, and if so sets appropriate boolean to TRUE
----------------------------------------------------------------------------*/

static int check_user_fct(name)
     char * name;

{
	if (strcmpi("User_",name))
	  switch(name[5])
	    {
	    case 'U':
	      if (strcmpi("nit_",name+6))
		switch (name[10])
		  {
		  case 'D':
		    if (strcmpi("isplay",name+11))
		      UUD = TRUE;
		    break;
		  case 'C':
		    if (strcmpi("heckpoint",name+11))
		      UUC = TRUE;
		    break;
		  case 'R':
		    if (strcmpi("estore",name+11))
		      UUR = TRUE;
		    break;
		  case 'S':
		    if (strcmpi("ave",name+11))
		      UUS = TRUE;
		    break;
		  case 'L':
		    if (strcmpi("oad",name+11))
		      UUL = TRUE;
		    break;
		  default:
		    return FALSE;
		  }
	      break;
	    case 'S':
	      if (strcmpi("ite_",name+6))
		switch (name[10])
		  {
		  case 'D':
		    if (strcmpi("isplay",name+11))
		      USD = TRUE;
		    break;
		  case 'C':
		    if (strcmpi("heckpoint",name+11))
		      USC = TRUE;
		    break;
		  case 'R':
		    if (strcmpi("estore",name+11))
		      USR = TRUE;
		    break;
		  case 'S':
		    if (strcmpi("ave",name+11))
		      USS = TRUE;
		    break;
		  case 'L':
		    if (strcmpi("oad",name+11))
		      USL = TRUE;
		    break;
		  default:
		    return FALSE;
		  }
	      break;
	    case 'L':
	      if (strcmpi("ink_",name+6))
		switch (name[10])
		  {
		  case 'D':
		    if (strcmpi("isplay",name+11))
		      ULD = TRUE;
		    break;
		  case 'C':
		    if (strcmpi("heckpoint",name+11))
		      ULC = TRUE;
		    break;
		  case 'R':
		    if (strcmpi("estore",name+11))
		      ULR = TRUE;
		    break;
		  case 'S':
		    if (strcmpi("ave",name+11))
		      ULS = TRUE;
		    break;
		  case 'L':
		    if (strcmpi("oad",name+11))
		      ULL = TRUE;
		    if (strcmpi("ist",name+11))
		      ULI = TRUE;
		    break;
		  default:
		    return FALSE;
		  }
	      break;
	    case 'H':
	      if (strcmpi("elp_Info",name+6))
		UHI = TRUE;
	      else
		return TRUE;
	      break;
	    default:
	      return FALSE;
	    }
	return TRUE;
}

/*---------------------------------------------------------------------------
  Writes out code to program file to initialise simulator pointers to
  functions for user defined structures to NULL or the function if it exists.
----------------------------------------------------------------------------*/

static fill_user_fct(bind_fp)
  FILE * bind_fp;

{
  if (UUD == 0)
    fprintf(bind_fp,"   si_User_Unit_Display = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Unit_Display = User_Unit_Display;\n");
  if (USD == 0)
    fprintf(bind_fp,"   si_User_Site_Display = NULL;\n"); 
  else
    fprintf(bind_fp,"   si_User_Site_Display = User_Site_Display;\n"); 
  if (ULD == 0)
    fprintf(bind_fp,"   si_User_Link_Display = NULL;\n"); 
  else
    fprintf(bind_fp,"   si_User_Link_Display = User_Link_Display;\n"); 
  if (ULI == 0)
    fprintf(bind_fp,"   si_User_Link_List = NULL;\n");    
  else
    fprintf(bind_fp,"   si_User_Link_List = User_Link_List;\n");    
  if (UUC == 0)
    fprintf(bind_fp,"   si_User_Unit_Checkpoint = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Unit_Checkpoint = User_Unit_Checkpoint;\n");
  if (USC == 0)
    fprintf(bind_fp,"   si_User_Site_Checkpoint = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Site_Checkpoint = User_Site_Checkpoint;\n");
  if (ULC == 0)
    fprintf(bind_fp,"   si_User_Link_Checkpoint = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Link_Checkpoint = User_Link_Checkpoint;\n");
  if (UUR == 0)
    fprintf(bind_fp,"   si_User_Unit_Restore = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Unit_Restore = User_Unit_Restore;\n");
  if (USR == 0)
    fprintf(bind_fp,"   si_User_Site_Restore = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Site_Restore = User_Site_Restore;\n");
  if (ULR == 0)
    fprintf(bind_fp,"   si_User_Link_Restore = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Link_Restore = User_Link_Restore;\n");
  if (UUS == 0)
    fprintf(bind_fp,"   si_User_Unit_Save = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Unit_Save = User_Unit_Save;\n");
  if (USS == 0)
    fprintf(bind_fp,"   si_User_Site_Save = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Site_Save = User_Site_Save;\n");
  if (ULS == 0)
    fprintf(bind_fp,"   si_User_Link_Save = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Link_Save = User_Link_Save;\n");
  if (UUL == 0)
    fprintf(bind_fp,"   si_User_Unit_Load = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Unit_Load = User_Unit_Load;\n");
  if (USL == 0)
    fprintf(bind_fp,"   si_User_Site_Load = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Site_Load = User_Site_Load;\n");
  if (ULL == 0)
    fprintf(bind_fp,"   si_User_Link_Load = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Link_Load = User_Link_Load;\n");
  if (UHI == 0)
    fprintf(bind_fp,"   si_User_Help_Info = NULL;\n");
  else
    fprintf(bind_fp,"   si_User_Help_Info = User_Help_Info;\n");
}

/*---------------------------------------------------------------------------
   Adds a procedure to the linked list, calling check_user_fct to see
   if it is one of the special ones.
----------------------------------------------------------------------------*/

static struct proc * add_proc(name)

char	name[];
{
	struct proc	*p_tmp;

	p_tmp = (struct proc *)malloc(sizeof(struct proc));
	if (!p_tmp) {
		perror("makebind->add_proc");
		exit(-1);
	}

	(void) check_user_fct(name);
	p_tmp->p_next = procs;
	strcpy(p_tmp->p_name, name);
	procs = p_tmp;
	return p_tmp;
}

main(argc, argv)
int	argc;
char	*argv[];
{
	char	bind_file[FN_LEN+1],	/* Output file name */
		tmp_proc[PN_LEN+1];	/* For reading in procs */
	FILE	*bind_fp;		/* Output file pointer */
	int	size = 0;		/* For the #define in the output */
	struct proc	*p_tmp;

/* |||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
 *   The following are format strings for output.  The different sections
 *   have nothing special about them, they are sectioned the way they
 *   are for programming convenience
 * \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\///////////////////////////////////// */
        char    *sec_0 = "si_get_user_no(length)\n    int *length;\n{\n   *length = SIZE;\n}\n";
	char	*sec_1 = "#include <stdio.h>\n#ifdef BFLY\n#include \"bflycontrol.h\"\n#else\n#include \"uniproc.h\"\n#endif\n#define\tSIZE\t%d\n\n";
	char	*sec_3 = "si_user_bind_func()\n{\n";
	char	*sec_4 = "    AddItemToTable\(\"%s\",%s,FUNC_SYM,-1,NULL,-1);\n";
	char	*sec_5 = "    \n}\n";

/* //////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ */

/******  Get the output file name and open the file for writing  ******/

	if (argc != 2) {
		printf("Usage: makebind <filename> < functions\n");
		exit(0);
	} else strcpy(bind_file, argv[1]);
	(void) strcat (bind_file, ".bind.c");
	if ((bind_fp = fopen(bind_file, "w")) == 0) {
		perror("makebind");
		exit(-1);
	}
	
	while((scanf("%s", tmp_proc)) != EOF) 	/* Read a proc */
	  {
		add_proc(tmp_proc);		/* and add it to the list */
		++size;				/* Incr. size */
	      }

	if (!size)
	  {
	    printf("There are no procedures!!!!!\n");
	    exit(0);
	  }

	fprintf(bind_fp, sec_1, size);
	for (p_tmp = procs; p_tmp; p_tmp = p_tmp->p_next)
	  fprintf(bind_fp, "int %s();\n", p_tmp->p_name);
	fprintf(bind_fp, sec_0);
	fprintf(bind_fp, sec_3);
	fill_user_fct(bind_fp);
	for (p_tmp = procs; p_tmp; p_tmp = p_tmp->p_next)
	  fprintf(bind_fp, sec_4, p_tmp->p_name, p_tmp->p_name);
	fprintf(bind_fp, sec_5);
	fclose(bind_fp);
        return 0;
}
