/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/**********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains the routines that relate to reshowing the
 * screen after or during simulation steps and is responsible for
 * making sure all information on the screen and info panel is
 * correct and up-to-date.
 *
 * Kenton Lynne
 *********************************************************************/

#include "macros.h"
#include "externs.h"
#include "control_panel.h"

/*******************************************************************/
static get_index(value, num_ranges, lrange, hrange)
   FLINT value, lrange, hrange;
   int num_ranges;
{
/* returns an index from 1 to num_ranges depending
 * on where value lies in relation to lrange and hrange
 * (note: lrange may be greater than hrange)
 */
  int opposite=FALSE, index;
  FLINT save_value, range_size, range_factor;

  /* 0 values ALWAYS return an index of 0 */
  if (value==0) return(0);

  /* if user has reversed the high and low range, reverse ourselves */
  if (lrange > hrange)
  {
    opposite = TRUE;
    save_value = lrange;
    lrange = hrange;
    hrange = save_value;
  }

  /* normalize the values to range from 0 to hrange - lrange */
  value -= lrange;
  hrange -= lrange;
  lrange -= lrange;
 
  /* find the appropriate index for the required range */
  if (value <= lrange) index = 1;
  else if (value >= hrange) index = num_ranges;
  else
#ifdef FSIM
  {
    range_size = hrange / num_ranges;
    index = (value / range_size) + 1;
  }
#else
  {  
    range_size = hrange / num_ranges;
    if (range_size)
    {
      /* in case the ranges don't evenly divide
       * bump value down to compensate
       */
      value -= (value * (hrange % num_ranges)) / hrange;
      index = ((value-1) / range_size) + 1;
      if (index>num_ranges) index = num_ranges;
    }
    else  /* num_ranges > hrange */
    {
      /* there are more ranges than values:
       * so use upper and lower indexes only
       * (i.e. the larger positive and negative icons)
       */  
      range_factor = num_ranges / hrange;
      if (value <= hrange/2)
        index = (value * range_factor) + 1;
      else
        index = num_ranges - ((hrange - value) * range_factor);
    }
  }  
#endif

  /* if user has reversed high and low then reverse index as well */
  if (opposite)
    index = num_ranges - index + 1;
  return(index);
}  

/*******************************************************************/
static display_grobj_values()
{
/* goes through the entire graphics object chain and
 * displays those units it needs to based on the 
 * DISPLAY and VALUE_CHG bits and whether RESHOW_ALL flag is on
 * Assumes: all values are current
 *          all flags are correct
 *          mode is correct
 *          all objects are within the display window
 */

  struct grobj *ptr;
  int new_index;

  /* for each object on the display chain */
  for (ptr=gi_marker.next; ptr!=&gi_marker; ptr=ptr->next)
  {
    /* if value has changed, get new index */
    if (ptr->flag & VALUE_CHG)
    {
      new_index = get_index(ptr->val, ptr->gptr->num, 
			    ptr->lrange, ptr->hrange);
					
     /* update DISPLAYED flag and icon index as necessary */
     if (new_index!=ptr->icon_index)
     {
       ptr->icon_index = new_index;
       ptr->flag &= ~DISPLAYED;
     }

     /* indicate that value and icon index are now in sync */
     ptr->flag &= ~VALUE_CHG;
    }
    /* if the item is not displayed (or new icon index) display now */
    if (!(ptr->flag & DISPLAYED) || gi_reshow_flag & RESHOW_ALL)
    {
      gi_display_grobj(ptr,PIX_SRC);
      ptr->flag |= DISPLAYED;
    }
  }
}

/*******************************************************************/
gi_display_link(ptr,rastop)
  struct grobj *ptr;
  int rastop;
{
/* given a pointer to a graphic object,
 * attempts to display the link value of that object
 * to the current target unit
 * Assumes link value stored in object is valid
 */

#ifdef DEBUG
  char ebuf[80];
#endif
  int size_x, size_y;
  struct pixrect *link_image;

  /* if no link image specified, use the units image */
  if (gi_cur_link_image==NULL)
  {
    /* use unit image itself */
    size_x = ptr->gptr->size_x;
    size_y = ptr->gptr->size_y;
    link_image = ptr->gptr->pix_ptr[ptr->link_index];
#ifdef DEBUG
    /* check for possible invalid index */
    if (ptr->link_index > ptr->gptr->num)
    {       
      (void) sprintf(ebuf,"Invalid Link Index: unit=%1d value=%1d",
              ptr->u_index,ptr->link_index);
      gi_put_error(ebuf);
      link_image = gi_cur_link_image[0];
    }
#endif
  }

  /* otherwise use the selected default image */
  else
  {
    size_x = UNITSIZE;
    size_y = UNITSIZE;
    link_image = gi_cur_link_image[ptr->link_index];
#ifdef DEBUG
    /* check for possible invalid index */
    if (ptr->link_index > DFT_NRANGES) 
    {       
      (void) sprintf(ebuf,"Invalid Link Index: unit=%1d value=%1d",
              ptr->u_index,ptr->link_index);
      gi_put_error(ebuf);
      link_image = gi_cur_link_image[0];
    }
#endif
  }
  /* write out the link image */
  pw_write(gi_gfx->gfx_pixwin,
           ptr->x_pos-gi_origin_x,
           ptr->y_pos-gi_origin_y,
           size_x,
           size_y,
           rastop,
           link_image,
           0,
           0);
}

/*******************************************************************/
static display_grobj_links()
{
/* goes through the grobj chain displaying those
 * units as requested (but only if necessary)
 */ 

  FLINT lrange, hrange;
  int  num_ranges;
  static FLINT old_hrange=DFT_LRANGE_LINKIN, 
               old_lrange=DFT_HRANGE_LINKIN;

  struct grobj *ptr;
  int op, new_index;
  char *lr_ptr, *hr_ptr;

  /* get ranges from the panel prompts */
  lr_ptr = (char *) panel_get_value(gi_litem[LLRANGE_ITEM]);
  hr_ptr = (char *) panel_get_value(gi_litem[LHRANGE_ITEM]);

#ifdef FSIM
  /* make sure ranges are OK  (floating point version) */
  if (sscanf(hr_ptr,"%f",&hrange) == 0
  || sscanf(lr_ptr,"%f",&lrange) == 0
  || lrange == hrange)
#else
  /* make sure ranges are OK  (integer version) */
  if (gi_check_num(lr_ptr)!=OK 
  || gi_check_num(hr_ptr)!=OK
  || abs(lrange=atoi(lr_ptr)) > MAX_RANGE
  || abs(hrange=atoi(hr_ptr)) > MAX_RANGE)
#endif
  {
    gi_put_error("Invalid range: using [-1000 1000]");
    lrange = DFT_LRANGE_LINKIN;
    hrange = DFT_HRANGE_LINKIN;
  }

  /* check if this range is different than previous range */
  if (lrange!=old_lrange 
  || hrange!=old_hrange)
  {
    /* different : set up new values and set global flag */
    old_lrange = lrange;
    old_hrange = hrange;
    gi_new_link_parms = TRUE;
  }
   
  /* now look at each item on the chain and decide what to do */
  for (ptr=gi_marker.next; ptr!=&gi_marker; ptr=ptr->next)
  {
    /* if value has changed, or new icon or range: get new index */
    if ((ptr->flag & LINK_CHG) || gi_new_link_parms)
    {
      /* use either the default ranges or unit ranges */
      if (gi_cur_link_image==NULL)
        num_ranges = ptr->gptr->num;
      else
        num_ranges = DFT_NRANGES;

      new_index = get_index(ptr->link_val, num_ranges, lrange, hrange);
					
      /* update DISPLAYED flag and icon index as necessary */
      if (new_index!=ptr->link_index)
      {
	ptr->link_index = new_index;
	ptr->flag &= ~DISPLAYED;
      }

      /* indicate that value and link index are now in sync */
      ptr->flag &= ~LINK_CHG;
    }

    /* if the item is not displayed (or new icon index) display now */
    if (!(ptr->flag & DISPLAYED) 
    || gi_reshow_flag & RESHOW_ALL
    || gi_new_link_parms)
    {
      /* if current target unit then display reverse imaged */
      if (ptr==gi_cur_target_ptr)
        op = PIX_NOT(PIX_SRC);
      else
        op = PIX_SRC;

      /* display the appropriate link icon */
      gi_display_link(ptr,op);
     
      /* mark unit as displayed */
      ptr->flag |= DISPLAYED;
     }
  }

  /* reset flag indicating new link parameters */
  gi_new_link_parms = FALSE;
}

/******************************************************************/
gi_reshow()
{
/* update the display screen appropriately by looking
 * at the various flags to determine what kinds of 
 * updates to do so that only screen operations absolutely
 * necessary to make things consistent are done
 */

  int i;
  struct txobj *tptr;
  struct drobj *dptr;

  /* if link mode, then get all link values */
  if (gi_mode==MODE_LINK)
  {
    if (gi_reshow_flag & (ALL_LINKS_NEEDED+SOME_LINKS_NEEDED))
    {
      gi_get_links();
      gi_reshow_flag &= ~(ALL_LINKS_NEEDED+SOME_LINKS_NEEDED);
    }
  }

  else
  {
    /* if the only flags that are set are for LINK mode but we 
       are not in LINK mode, then just exit
    */
    if (!(gi_reshow_flag & ~(ALL_LINKS_NEEDED+SOME_LINKS_NEEDED))) 
      return;

    /* if some or all unit values need to be updated, do it now */
    if (gi_reshow_flag & (ALL_VALS_NEEDED+SOME_VALS_NEEDED))
    {
      gi_get_values();
      
      /* reset flags indicating values need to be updated */
      gi_reshow_flag &= ~(ALL_VALS_NEEDED+SOME_VALS_NEEDED);
    }
  }

  /* lock the graphics panel */
  pw_lock(gi_gfx->gfx_pixwin, &gi_gfx->gfx_rect);

  /* if CLEAR_NEEDED, erase the entire screen first */
  if (gi_reshow_flag & CLEAR_NEEDED)
  {
    pw_writebackground(gi_gfx->gfx_pixwin,
                      0,0,
                      gi_rects[GRAPH].r_width,
                      gi_rects[GRAPH].r_height,
                      PIX_CLR);
  }

  /* display objects on the chain as necessary */
  if (gi_mode==MODE_LINK)
    display_grobj_links();
  else
    display_grobj_values();

  /* if screen has been blanked, must redisplay all other
     graphic objects as well
  */

  if ((gi_reshow_flag & CLEAR_NEEDED) 
     || (gi_reshow_flag & RESHOW_ALL)) 
  {
    /* display text objects */
    for (tptr=gi_text_head.next; tptr!=&gi_text_head; tptr=tptr->next)
    {
      if (gi_overlap_txobj(tptr))
        gi_display_txobj(tptr,PIX_SRC | PIX_DST);
    }

    /* display drawn objects */
    for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
    {
      if (gi_overlap_drobj(dptr))
        gi_display_drobj(dptr,PIX_SRC);
    }

    /* redisplay the marker if appropriate */
    if (!(gi_marker.flag & NOSHOW) 
    && gi_overlap_grobj(&gi_marker))
     gi_display_grobj(&gi_marker,PIX_SRC ^ PIX_DST);
  }

  /* unlock the graphics panel */
  pw_unlock(gi_gfx->gfx_pixwin);
 
  /* now update the info panel as necessary */
  for (i=0; i<MAX_INFO_COLS; i++)
  {
    if (gi_info_item[i].ui >=0 && gi_info_item[i].pos < gi_rects[INFO].r_width)
      gi_show_info(i, FALSE);
  }
  
  /* now update the clock value */
  gi_update_clock();

  /* reset reshow flags */
  gi_reshow_flag &= ~(RESHOW_ALL+RESHOW_NEEDED+CLEAR_NEEDED);
}

/*******************************************************************/
static struct grobj *get_displayed(ptr, end_ptr, flag, and_flag, or_flag)
  struct grobj *ptr, *end_ptr;
  int flag, and_flag;
{
/* returns a pointer to the first grobj,
 * starting at ptr and ending before end_ptr, 
 * that: (flag) is within the current display window or
 *       (!flag) is not within the current display window
 * returning end_ptr if none is found
 *
 * NOTE: ptr may be same as end_ptr in which case
 *       end_ptr must be returned
 * NOTE: the "and_flag" is anded with the grobj flag field
 *       and the "or_flag" orred with the grobj flag field
 *       of all objects it "skips" in order to turn off (and_flag)
 *       of on (or_flag) caller specified flags
 */ 

  while (ptr!=end_ptr)
  {  
    if (!(gi_overlap_grobj(ptr) ^ flag)) return(ptr);
    ptr->flag &= and_flag;  /* first turn off indicated bits */
    ptr->flag |= or_flag;   /* then turn on indicated bits */
    ptr = ptr->next;
  }
  return(ptr);
}
  
/*******************************************************************/
gi_move_grobj_chain(where, first, last)
  struct grobj *where, *first, *last;
{
/* detaches the subchain first-last from the
 * doubly linked circular list they are on, and appends it   
 * to the doubly linked list after the where element
 * NOTE: subchain may be one element long      
 *       (i.e. first==last)
 * assumes chain it is removing the chain from will still
 * have at least one element (ie. the chain header)
 */

  /* remove this from the chain its on */

  first->prev->next = last->next;
  last->next->prev = first->prev;

  /* now insert it on the where chain */

  first->prev = where;
  last->next = where->next;

  last->next->prev = last;
  where->next = first;

}

/*******************************************************************/
gi_update_grobj_chain(off_flag, on_flag)
  int off_flag, on_flag;
{
/* updates the grobj chain making sure that 
 * all grobjs currently on the display are
 * on the display (marker) chain and all
 * grobjs not on the display are on the
 * off_display chain
 * off_flag: on means we must look at the off_display
 *           chain for grobjs that may now be on the display
 * on_flag: on means we must look at the on_display (marker)
 *          chain for grobjs that may now be off the display
 *
 * Note: in addition to updating the chains, this routine
 *       also sets the reshow_flag appropriately so that
 *       a subsequent call to gi_reshow should properly
 *       reshow the correct and updated display panel
 * Assumes: extent and origin globals are current
 *       all grobjs on the display chain are current
 *       (i.e. have correct values and indexes)
 */


  struct grobj temp1, temp2;
  struct grobj *ptr1, *ptr2;
  int and_flag, or_flag, done; 

  /* initialize the temporary chain headers */
  temp1.next = temp1.prev = &temp1;
  temp2.next = temp2.prev = &temp2;
  
  
  /******************/
  /* off_flag is on */
  /******************/
  /* look at off_display chain for grobjs that should be on the display */

  if (off_flag)
  {
    /* check if off_display values still OK */
    if (gi_off_display_ok)
    {
      /* only turn off display flag */
      and_flag = ~DISPLAYED;
    }
    else
    {
      /* set off display and val/link OK flags */
      and_flag = ~(VALUE_OK+LINK_OK+DISPLAYED);
    }
    or_flag = ON_DISPLAY;

    ptr1 = gi_off_display.next;
    done = FALSE;
    /* go through the off_display array accumulating
       grobjs that are now currently on the display
    */
    while (!done)
    {
      /* find first object on the display while turning off
         the others' value flags if appropriate
      */
      if ((ptr2=get_displayed(ptr1,&gi_off_display,TRUE,and_flag,0))
         ==&gi_off_display)
      {      
        done = TRUE;
      }
      else
      {
        /* find first object still off the display, while turning 
           on the others' ON_DISPLAY flag
        */
           
        ptr1 = get_displayed(ptr2,&gi_off_display,FALSE,and_flag,ON_DISPLAY);
      
        /* move this subchain to temp1 */
        gi_move_grobj_chain(&temp1, ptr2, ptr1->prev);
      }
    }
  }

  /******************/
  /* on_flag is on  */ 
  /******************/
  /* look at on_display array for grobjs that are now off the display */

  if (on_flag)
  {
    /* go through the on_display array accumulating
       grobjs that are now currently off the display
    */
    ptr1 = gi_marker.next;
    done = FALSE;
    and_flag = ~ON_DISPLAY; /* set off the "ON_DISPLAY" flag */
    while (!done)
    {
      /* get next object not on display, making sure
         not to affect any flags of object that are
         staying on the display chain
      */
      if ((ptr2=get_displayed(ptr1,&gi_marker,FALSE,0xFFFF,0))==&gi_marker)
      {      
        done = TRUE;
      }
      else
      {
        /* get next object still on the display 
           while setting all others' ON_DISPLAY flag off
        */
        ptr1 = get_displayed(ptr2,&gi_marker,TRUE,and_flag,0);
      
        /* append this subchain to the temp2 header */
        gi_move_grobj_chain(&temp2, ptr2, ptr1->prev);
      }
    }
    /* move these to the off_display chain */
    if (temp2.next != &temp2) /* anything on the chain ? */
    {
      gi_move_grobj_chain(gi_off_display.next,temp2.next,temp2.prev);
    }
  }

  /* if there are any objects to add to the on_display chain,
     append them now and set reshow_flags appropriately 
  */
   
  if (temp1.next != &temp1) /* anything on the chain ? */
  {
    /* definitely need to display these objects */
    gi_reshow_flag |= RESHOW_NEEDED;

    /* check if any values needed */
    for (ptr1=temp1.next; ptr1!=&temp1; ptr1=ptr1->next)
    {
      if (gi_mode==MODE_LINK && !(ptr1->flag & LINK_OK))
      {
        gi_reshow_flag |= SOME_LINKS_NEEDED;
        break;
      }
      else if (gi_mode!=MODE_LINK && !(ptr1->flag & VALUE_OK))
      {
        gi_reshow_flag |= SOME_VALS_NEEDED;
        break;
      }
    }

    /* finally, move these objects to the display chain */
    gi_move_grobj_chain(gi_marker.next,temp1.next,temp1.prev);
  }

  /* indicate that, at this point, objects on the off_display chain
     that have VALUE_OK or LINK_OK flags, are valid until
     the next time we go to the simulator
   */

   gi_off_display_ok = TRUE;
}
