/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

#include <stdio.h>
#include "../../include/sim.h"		/* simulator functions */
#include "../../include/lib.h"		/* library functions */
#include <math.h>				/* math functions */

#define max(r,s) (((r) > (s)) ? (r) : (s))	/* finds max of r and s */

#define CONTROL_WAIT 12				/* unit states,potentials... */
#define CONTROL_PLAY 2

#define DELAY_WAIT 3
#define DELAY_PRIME 4

#define DELAY_FIRE 5

#define MOVE_BLOCK 6
#define MOVE_PLAY 7
#define MOVE_WAIT 8

#define XPLAYING 9
#define OPLAYING 10
#define NOWINNER 11
#define DRAW 13

#define ON 1
#define OFF 0
#define DATA 0

#define BOARD_RESET_POT 1
#define BOARD_INIT_POT 1
#define BOARD_INIT_OUT 1

#define RESET_POT 0
#define INIT_POT 0
#define INIT_OUT 0

int  board_func  ();				/* unit and site functions */
int  line_func   ();
int  move_func   ();
int  delay_func  ();
int  control_func();
int  detect_func  ();
int  iotrigger_func ();
int  boardshow_func ();
int  moveexcite  ();
int  moveexciteb ();
int  movereset   ();

int TURN = 0;					/* user playing 1st or 2nd */


void MakeStateNames()				/* declare state names */

{DeclareState("c_wait",CONTROL_WAIT);
 DeclareState("c_play",CONTROL_PLAY);
 DeclareState("d_wait",DELAY_WAIT);
 DeclareState("d_prime",DELAY_PRIME);
 DeclareState("d_fire",DELAY_FIRE);
 DeclareState("m_block",MOVE_BLOCK);
 DeclareState("m_play",MOVE_PLAY);
 DeclareState("m_wait",MOVE_WAIT);
 DeclareState("x to go",XPLAYING);
 DeclareState("o to go",OPLAYING);
 DeclareState("on",ON);
 DeclareState("off",OFF);
}

void MakeBoard(name)				/* make board units */
   char *name;					/* for X or O */

{   int i,u;

    for(i = 1; i < 10; i++)
       {u = MakeUnit(	"board",
			board_func,
			BOARD_RESET_POT,
			BOARD_INIT_POT,
			DATA,
			BOARD_INIT_OUT,
			OFF,
			OFF);
	if (i == 1) NameUnit(name,1,u,9);
	AddSite(u,"excite",SFsum);
       }
}


void MakeLines(name)				/* make line units */
   char *name;					/* for X or O */

{   int i, u;

    for(i = 1; i < 9; i++)
       {u = MakeUnit(	"line",
			line_func,
			RESET_POT,
			INIT_POT,
			DATA,
			INIT_OUT,
			OFF,
			OFF);
	if (i == 1) NameUnit(name,1,u,9);
	AddSite(u,"excite",SFprod);
	AddSite(u,"inhibit",SFmax);
       }
}


void MakeMoves(name)				/* make move units */
   char *name;					/* for X or O */

{   int i, u;

    for(i = 1; i < 10; i++)
       {u = MakeUnit(	"move",
			move_func,
			RESET_POT,
			INIT_POT,
			DATA,
			INIT_OUT,
		        MOVE_WAIT,
		        MOVE_WAIT);
	if (i == 1) NameUnit(name,1,u,9);
	AddSite(u,"excite",moveexcite);
	AddSite(u,"exciteb",moveexciteb);
	AddSite(u,"inhibit",SFmax);
	AddSite(u,"block",SFmax);
	AddSite(u,"reset",movereset);
       }
}

void MakeGates(name)				/* make gate units */
   char *name;					/* for X or O */

{   int i, u;

    for(i = 1; i < 10; i++)
       {u = MakeUnit(	"gate",
			UFsum,
			RESET_POT,
			INIT_POT,
			DATA,
			INIT_OUT,
		        OFF,
		        OFF);
	if (i == 1) NameUnit(name,1,u,9);
	AddSite(u,"excite",SFand);
       }
}

void MakeControl()				/* make control, delay and */
						/* win/draw detector units */
{   int u;

    u = MakeUnit("delay",			/* X delay unit */
		 delay_func,
		 RESET_POT,
		 INIT_POT,
		 DATA,
		 INIT_OUT,
		 DELAY_PRIME,
		 DELAY_PRIME);
    NameUnit("xdelay",0,u,1);
    AddSite(u,"excite",SFsum);
    u = MakeUnit("delay",			/* O delay unit */
		 delay_func,
		 -1,
		 -1,
		 DATA,
		 INIT_OUT,
		 DELAY_WAIT,
		 DELAY_WAIT);
    NameUnit("odelay",0,u,1);
    AddSite(u,"excite",SFsum);
    u = MakeUnit("control",			/* X control unit */
		 control_func,
		 -1,
		 -1,
		 DATA,
		 1,
		 CONTROL_WAIT,
		 CONTROL_WAIT);
    NameUnit("xcontrol",0,u,1);
    AddSite(u,"excite",SFxor);
    AddSite(u,"inhibit",SFsum);
    AddSite(u,"play",SFsum);
    u = MakeUnit("control",			/* O control unit */
		 control_func,
		 -1,
		 -1,
		 DATA,
		 INIT_OUT,
		 CONTROL_WAIT,
		 CONTROL_WAIT);
    NameUnit("ocontrol",0,u,1);
    AddSite(u,"excite",SFxor);
    AddSite(u,"inhibit",SFsum);
    AddSite(u,"play",SFsum);
    u = MakeUnit("detect",			/* win/draw detector unit */
		 detect_func,
		 RESET_POT,
		 INIT_POT,
		 DATA,
		 INIT_OUT,
		 OFF,
		 OFF);
    NameUnit("detector",0,u,1);
    AddSite(u,"xlines",SFmax);
    AddSite(u,"olines",SFmax);
    AddSite(u,"occupied",SFsum);
}

void MakeIO()					/* make inpuy/ouyput units */
						/* and links */
{   int i,j,k,iotrig,brdsh;

    iotrig = MakeUnit("delay",			/* trigger board display */
		      iotrigger_func,
		      2,
		      2,
		      DATA,
		      INIT_OUT,
		      OPLAYING,
		      OPLAYING);
    NameUnit("iotrigger",0,iotrig,1);
    brdsh = MakeUnit("ioshow",			/* display board */
		     boardshow_func,
		     RESET_POT,
		     INIT_POT,
		     DATA,
		     INIT_OUT,
		     OFF,
		     OFF);
    NameUnit("boardshow",0,brdsh,1);
    AddSite(iotrig,"xplay",SFsum);
    AddSite(iotrig,"oplay",SFsum);
    MakeLink(NameToInd("xcontrol",0),iotrig,"xplay",1,0,NULL);
    MakeLink(NameToInd("ocontrol",0),iotrig,"oplay",1,0,NULL);
					/* display is triggered by a move */
    AddSite(brdsh,"show",SFsum);
    MakeLink(iotrig,brdsh,"show",1,0,NULL);
}

void BoardToLine(from,to,site)			/* links from board to */
    char *from,*to,*site;			/* line units, for X or O */

{   int i;
    
    for (i = 0; i < 9; i++)
       {int u;
	u = NameToInd(from,i);
	MakeLink(u,NameToInd(to,i / 3),site,1,0,NULL);
	MakeLink(u,NameToInd(to,3 + i % 3),site,1,0,NULL);
	if (i % 4 == 0)
	   MakeLink(u,NameToInd(to,6),site,1,0,NULL);
	if ((i % 8 != 0) && (i % 2 == 0))
	   MakeLink(u,NameToInd(to,7),site,1,0,NULL);
       }
}

void LineToMove(from,to,site)			/* links from line to move */
    char *from,*to,*site;			/* units, for X or O */

{   int i;
    
    for (i = 0; i < 9; i++)
       {int u;
	u = NameToInd(to,i);
	MakeLink(NameToInd(from,i / 3),u,site,1,0,NULL);
	MakeLink(NameToInd(from,3 + i % 3),u,site,1,0,NULL);
	if (i % 4 == 0)
	   MakeLink(NameToInd(from,6),u,site,1,0,NULL);
	if ((i % 8 != 0) && (i % 2 == 0))
	   MakeLink(NameToInd(from,7),u,site,1,0,NULL);
       }
}

void WTA(name,control)				/* make a vector of 9 units */
    char *name,*control;			/* into a winner takes all */
						/* network, with controller */
{   int i,j,u,c;

    for (i = 0; i < 9; i++)
      {u = NameToInd(name,i);
       c = NameToInd(control,0);
	for (j = 0; j < 9; j++)
	  if (i != j)
	    MakeLink(u,NameToInd(name,j),"inhibit",1,0,NULL);
	MakeLink(u,c,"excite",1,0,NULL);
       }
}

void ArrayMatches()				/* matching 3x3 arrays of */
						/* units are linked */
{   int i,bx,bo,mx,mo,gx,go;

    for (i = 0; i < 9; i++)
       {bx = NameToInd("xboard",i);
	bo = NameToInd("oboard",i);
	mx = NameToInd("xmove",i);
	mo = NameToInd("omove",i);
	gx = NameToInd("xgate",i);
	go = NameToInd("ogate",i);
	MakeLink(bx,mx,"block",1,0,NULL);
	MakeLink(bx,mo,"block",1,0,NULL);
	MakeLink(bo,mo,"block",1,0,NULL);
	MakeLink(bo,mx,"block",1,0,NULL);
	MakeLink(mo,go,"excite",1,0,NULL);
	MakeLink(mx,gx,"excite",1,0,NULL);
	MakeLink(gx,bx,"excite",1,0,NULL);
	MakeLink(go,bo,"excite",1,0,NULL);
	MakeLink(NameToInd("xcontrol",0),gx,"excite",1,0,NULL);
	MakeLink(NameToInd("ocontrol",0),go,"excite",1,0,NULL);
       }
}

void ControlToMove(control,move)		/* links control unit to */
    char *control,*move;			/* array of move units */

{   int c,i,u;

    c = NameToInd(control,0);
    for (i = 0; i < 9; i++)
       {u = NameToInd(move,i);
	MakeLink(c,u,"reset",1,0,NULL);
       }
}

void UnitsToDetect()				/* link line and board units */
						/* to detector for win/draw */
{   int xl,ol,det,i;

    xl = NameToInd("xline",0);
    ol = NameToInd("oline",0);
    det = NameToInd("detector",0);
    for (i = 0; i < 8; i++)
      {MakeLink(xl+i,det,"xlines",1,0,NULL);	/* line units indicate a win */
       MakeLink(ol+i,det,"olines",1,0,NULL);
      }
    xl = NameToInd("xboard",0);
    ol = NameToInd("oboard",0);
    det = NameToInd("detector",0);
    for (i = 0; i < 9; i++)			/* board units indicated a */
      {MakeLink(xl+i,det,"occupied",1,0,NULL);	/* draw - if 9 are on */
       MakeLink(ol+i,det,"occupied",1,0,NULL);
      }
}

void ControlCircuit()				/* links to ensure two */
						/* halves play sequentially */
{   int xc,oc,xd,od;

    xc = NameToInd("xcontrol",0);
    oc = NameToInd("ocontrol",0);
    xd = NameToInd("xdelay",0);
    od = NameToInd("odelay",0);
    MakeLink(xc,xc,"inhibit",1,0,NULL);
    MakeLink(oc,oc,"inhibit",1,0,NULL);
    MakeLink(xc,xd,"excite",1,0,NULL);
    MakeLink(oc,od,"excite",1,0,NULL);
    MakeLink(xd,oc,"play",1,0,NULL);
    MakeLink(od,xc,"play",1,0,NULL);
}


void build()					/* master network building */
						/* routine */
{
    AllocateUnits(100);				/* actually use 75 */

    /*Now make the units*/

    MakeBoard("xboard");			/* make units for board, */
    MakeBoard("oboard");			/* lines, moves and gates */
    MakeLines("xline");
    MakeLines("oline");
    MakeMoves("xmove");
    MakeMoves("omove");
    MakeGates("xgate");
    MakeGates("ogate");
    MakeControl();				/* make control units */

    /*Now make the links*/

    BoardToLine("xboard","xline","excite");	/* link boards to lines */
    BoardToLine("oboard","oline","excite");
    BoardToLine("xboard","oline","inhibit");
    BoardToLine("oboard","xline","inhibit");

    LineToMove("xline","xmove","excite");	/* link lines to moves */
    LineToMove("oline","omove","excite");
    LineToMove("xline","omove","exciteb");
    LineToMove("oline","xmove","exciteb");

    WTA("xmove","xcontrol");			/* move units are formed */
    WTA("omove","ocontrol");			/* into winner-takes-all */

    ArrayMatches();				/* link matching arrays */
    ControlToMove("xcontrol","omove");		/* feedback links from */
    ControlToMove("ocontrol","xmove");		/* control units to move */
    ControlToMove("xcontrol","xmove");		/* units, both X and O */
    ControlToMove("ocontrol","omove");
    ControlCircuit();				/* sequence players */
    UnitsToDetect();				/* link in win/draw detect */

    /*Now attach I/O */

    MakeIO();					/* attach board display */

    /*Now make the state names*/

    MakeStateNames();				/* declare the state names */

}

/*-------------------------------------------------------------------------*/
/*                         UNIT, SITE & LINK FUNCTIONS                     */
/*-------------------------------------------------------------------------*/
	
/*** site functions ***/

/* board_func returns 3 if it is excited, 1 otherwise */

 board_func (up)
   Unit *up;

{  Site *sp;
   int tot = 0;

   if (SiteValue("excite",up->sites) > 0)
	up->state = ON;				/* any excitement turns on! */
   up->output = up->potential = (up->state == ON) ? 3 : 1;
}

/* line_func returns 0 if the unit is inhibited, the excited value else*/

 line_func (up)
   Unit *up;

 { if (SiteValue("inhibit",up->sites) > 1)
      up->potential = up->output = 0;
   else
      up->potential = up->output = SiteValue("excite",up->sites);
 }

/* movepot sets the potential to be the difference between
   excites and inhibits, unless block is active
*/

move_func (up)
   Unit *up;

{switch(up->state)
   {case MOVE_PLAY: if (SiteValue ("block",up->sites) > 2) /*already occupied*/
                      {up->state = MOVE_BLOCK;
		       up->potential = 0;
		      }
		    else			/* not occupied */
		      if (SiteValue ("reset",up->sites) > 0)
			{up->state = MOVE_WAIT; /*reset for next*/
			 up->potential = 0;
		        }
		      else			/* winner-takes-all */
			{int myexc,othexc,pot,inh;

			 myexc = SiteValue("excite",up->sites);
						/* good offensive moves */
			 othexc = SiteValue("exciteb",up->sites);
						/* good blocking moves */
			 inh = SiteValue("inhibit",up->sites);
						/* maximum of competitors */
			 pot = up->potential;	/* previous activation */
			 myexc += max(0,othexc-1);
						/* can't be negative */
			 myexc = max(myexc,pot);
						/* not less than last time */
			 if (myexc == inh)	/* if two equal winners */
			   myexc = (((random() %128) / 128.0) > 0.5) ? 
			     (1+myexc) : myexc;	/* randomly add 1 to potent. */
			 else			/* definite winner or loser */
			   myexc = (myexc > inh) ? myexc : 0;
						/* set to zero if loser */
			 up->potential = myexc;
		        }
		    up->output = up->potential;
		    break;

    case MOVE_BLOCK: up->potential = 0;
		     up->output = 0;
		     break;

    case MOVE_WAIT: if (up->potential < 6)	/* wait for effects of */
                      up->potential++;		/* previous move to propogate*/
		    else			/* waited long enough */
		      {up->state = MOVE_PLAY;
		       up->potential = 0;
		      }
		    up->output = 0;
		    break;
   }
}
		   
 control_func (up)				/* determines when WTA */
   Unit *up;					/* network has settled */

{switch (up->state)
   {case CONTROL_WAIT:				/* other players turn */
     if (SiteValue ("play",up->sites) != 0)	/* now my turn */
       up->state = CONTROL_PLAY;
     up->potential = 0;
     break;

    case CONTROL_PLAY:				/* waiting for settling */
     if (SiteValue ("excite",up->sites) != 0)	/* if only one move active */
       {up->state = CONTROL_WAIT;
	up->potential = 1;			/* send signal */
       }
     else
       up->potential = 0;
   }
 up->output = up->potential;
}

 delay_func (up)				/* wait after move for */
   Unit *up;					/* effects to propogate */
						/* before signalling other */
{switch (up->state)
   {
   case DELAY_WAIT:				/* waiting for signal */
     if (SiteValue ("excite",up->sites) > 0)
       up->state = DELAY_PRIME;
     up->output = up->potential = 0;
     break;

   case DELAY_PRIME:				/* delay 6 timesteps */
     if (up->potential > 5)
       up->state = DELAY_FIRE;
     else
       up->potential++;
     up->output = 0;
     break;

   case DELAY_FIRE:				/* signal other player */
     up->state = DELAY_WAIT;
     up->potential = 0;
     up->output = 1;
     break;
   }
}

/* site functions */

/* moveexcite returns the sum of inputs, except that any
   input of 9, indicating 2 tokens in row, is doubled
*/

 moveexcite (up,sp)
   Unit *up;
   Site *sp;

   {Link *ip;
    int i,j, tot = 0;

    for (ip = sp->inputs; ip != NULL; ip = ip->next)
	{j = *(ip->value);
	 if (j == 9)
	    tot += (j + j);
	 else
	    tot += j;
	}
    sp->value = tot;
   }

/* moveexciteb returns the sum of the inputs, ignoring inputs of 1 */

 moveexciteb (up,sp)
   Unit *up;
   Site *sp;

   { int tot = 0;
     int temp;
     Link *ip;

    for (ip = sp->inputs; ip != NULL; ip = ip->next)
     {temp = *(ip->value);
      if (temp > 1) tot += temp;
     }
    sp->value = tot;
   }

/* movereset returns the sum of the positive inputs */

 movereset (up,sp)
   Unit *up;
   Site *sp;

{   int tot = 0;
    int i, j;
    Link *ip;

    for(ip = sp->inputs; ip != NULL; ip = ip->next)
     { j =  *(ip->value);
       if (j > 0) tot += j;
     }
    sp->value = tot;

}

/*-------------------------------------------------------------------------*/
/*                         USER INTERFACE FUNCTIONS                        */
/*-------------------------------------------------------------------------*/



showboard ()					/* display the board */

	    { /*show board*/
		int i, xb, ob, maxopt = 0;
		int brd[9],opt[9],state;
		char user[30];

		xb = NameToInd("xboard",0);
		ob = NameToInd("oboard",0);
		printf ("Board is now:\n");
		for (i = 0; i < 9; i++)		/* check each square */
		   {if (i % 3 == 0) printf("\n-------\n|");
		    if (UnitList[xb+i].state == ON)
			{brd[i] = -1;
			 printf("X|");		/* occupied by X */
			}
		    else
			if (UnitList[ob+i].state == ON)
			   {printf("O|");
			    brd[i] = -1;	/* occupied by O */
			   }
			else
			   {printf(" |");	/* not occupied */
			    brd[i] = maxopt;
			    opt[maxopt] = i;
			    maxopt += 1;
			   }
		  }
		printf("\n-------\n");
	      }

detect_func (up)				/* detect win or draw */
   Unit *up;

{int winner;

 if (SiteValue ("xlines",up->sites) > 26)	/* 3 in a row give 27 */
   {winner = XPLAYING;
    showboard();
    printf ("X has won!... Do you want another game ?\n");
  }
 else
   if (SiteValue ("olines",up->sites) > 26)
     {winner = OPLAYING;
      showboard();
      printf ("O has won!... Do you want another game ?\n");
    }
   else
     if (SiteValue ("occupied",up->sites) > 35)	/* all occupied give 36 */
       {winner = DRAW;
	showboard();
	printf ("The game is a draw!... Do you want to play again ?\n");
       }
     else
       winner = NOWINNER;			/* game unfinished */
 if (winner != NOWINNER)
   {char user[30];				/* ask user if wants to */
    user[0] = 'x';				/* play again */
    while ((user[0] != 'n') && (user[0] != 'y'))
      {printf("[yes,no] > ");
       scanf("%s",user);
      }
    if (user[0] == 'y')				/* another game, so... */
      {Reset();					/* reset the network */
       TURN = 0;				/* reset who goes first */
       Step(120);				/* run for another 120 steps */
      }
    else					/* no more games, so... */
      exit(0);					/* exit to UNIX shell */
   }
}

iotrigger_func(up)				/* fire when a move is made */
   Unit *up;

{  up->output = 0;
   if (up->potential == 0)
     if ((SiteValue("xplay",up->sites) > 0) ||
	 (SiteValue("oplay",up->sites) > 0))
	    up->potential = 1; else ;
   else
     if (up->potential > 1)
	{up->potential = 0;
	 up->output = 1;
	}
     else
	up->potential += 1;
    if (SiteValue("xplay",up->sites) > 0)
	up->state = OPLAYING;
    if (SiteValue("oplay",up->sites) > 0)
	up->state = XPLAYING;
}

boardshow_func(up)				/* display board and if it*/
   Unit *up;					/* is the users turn then */
						/* request a move */
{  up->output = up->potential = 0;
   if (up != NULL)
	if (up->sites->value != 0)
	  { /*show board*/
	    int i, xb, ob, maxopt = 0;
	    int brd[9],opt[9],state;
	    char user[30];

	    xb = NameToInd("xboard",0);		/* code as in showboard */
	    ob = NameToInd("oboard",0);
	    printf ("Board is now:\n");
	    for (i = 0; i < 9; i++)
	      {if (i % 3 == 0) printf("\n-------\n|");
	       if (UnitList[xb+i].state == ON)
		 {brd[i] = -1;
		  printf("X|");
		}
	       else
		 if (UnitList[ob+i].state == ON)
		   {printf("O|");
		    brd[i] = -1;
		  }
		 else
		   {printf(" |");
		    brd[i] = maxopt;
		    opt[maxopt] = i;
		    maxopt += 1;
		  }
	     }
	    printf("\n-------\n");
	    state = UnitList[NameToInd("iotrigger",0)].state;
	    if (state == XPLAYING)
	      printf("with X to play\n");
	    else
	      printf("with O to play\n");
	    if (TURN == 0)			/* first move to make */
	      {printf("Do you wish to play first or second ?\n");
	       user[0] = 'x';
	       while ((user[0] != '1') && (user[0] != '2'))
		 {printf("[1 or 2] > ");
		  scanf("%s",user);
		}
	       if (user[0] == '1')
		 TURN = 1;			/* user playing first */
	       else
		 TURN = 2;			/* user playing second */
	     }
	    if (((state == OPLAYING) && (TURN == 1)) ||
		((state == XPLAYING) && (TURN == 2)))
						/* users turn to play */
	      {int usq,uact,uslp;		/* show unoccupied squares */
	       printf("Your turn... here are the choices...\n");
	       for (i = 0; i < 9; i++)
		 {if (i % 3 == 0) printf("\n-------\n|");
		  if (brd[i] < 0)
		    printf(" |");
		  else
		    printf("%d|",brd[i]);
		}
	       printf("\n-------\n\n");
	       printf("Choose an option\n");
	       i = -1;
	       while ((i < 0) || (i > maxopt-1))
		 {printf("[0..%d] > ",maxopt-1);
		  scanf("%s",user);		/* get users move */
		  i = user[0] - '0';
		}
	       i = opt[i];
	       if (state == XPLAYING)		/* find units to adjust */
		 {usq = NameToInd("xboard",i);
		  uact = NameToInd("odelay",0);
		  uslp = NameToInd("xcontrol",0);
		}
	       else
		 {usq = NameToInd("oboard",i);
		  uact = NameToInd("xdelay",0);
		  uslp = NameToInd("ocontrol",0);
		}
	       UnitList[usq].potential = 1;	/* change network features */
	       UnitList[usq].output = 3;	/* to reflect move made */
	       UnitList[usq].state = ON;
	       UnitList[uact].potential = 0;
	       UnitList[uact].state = DELAY_WAIT;
	       UnitList[uslp].output = 1;
	       UnitList[uslp].state = CONTROL_WAIT;
	     }
	  }
        else ;
   else
     printf("Error in proc boardshow_func\n");
 }
