/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/* Context-free parsing network.  Written by Mark Fanty, April 1987.
 * Computer Science Department, University of Rochester   
 */

/* This file contains some functions to be called from the user
 * interface.  The first is 'parse', which turns on terminal units
 * corresponding to its arguments, so 'call parse noun verb noun'
 * turns on the noun unit in position one, the verb unit in position
 * two and the noun unit in position three.  The grammar must have
 * defined terminal units with the given names.
 *
 * The second is 'instantiate', which should be called from the
 * graphics interface (it is most effective when bound to a custom
 * button).  It cycles through all the instantiations of a production,
 * darkening them on the display an putting the unit information in
 * the data panel.
 * 
 * The third is 'drawit' which must be called from the graphics
 * interface.  It displays all the nonterminal units in a table with
 * the rows corresponding to the length of the input string that unit
 * is activated by and the column representing the starting position.
 * In addition,  the terminal (input) units are displayed by position
 * and labeled.
 */

#include <stdio.h>
#include <math.h>
#include "sim.h"
#include "parsing.h"

static char drawbuf[80];
extern int ntct,tct,Length;
extern char *words[2][MAXWORDS];

parse(argc,argv)
     int argc;
     char *argv[];
{
  int i,ind;
  char buf[30];

  Reset();
  for(i = 1;i < argc;i++)
    {
      (void) sprintf(buf,"%s.%1d",argv[i],i);
      ind = NameToInd(buf,1,1);
      if(ind < 0)
	{
	  printf("not a terminal: %s\n",argv[i]);
	  return;
	}
      SetPotential(ind,500);
      SetOutput(ind,500);
      SetState(ind,primed);
    }
}

/* instantiate(argc,argv)
 *
 * The argument (in argv[1]) should be the index of a nonterminal
 * unit.  For each match unit of the argument, the units which satisfy
 * it are darkened on the display.  Subsequent calls for the same unit
 * cycle through all the instantiations.
 */

instantiate(argc,argv)
     int argc;
     char *argv[];
{
  Site *sp;
  Link *lp;
  int ind,unitx,unity,coded;
  
  if(argc != 2)
    {
      printf("usage: instantiate <nt index>\n");
      return;
    }
  
  ind = atoi(argv[1]);
  if(ind < 0 || ind > NoUnits)
    {
      printf("instantiate: illegal unit %d\n",ind);
      return;
    }

  /* look through the links from match units until the last one to be
     instantiated is found (it is marked with data = 33).  If there
     was no last, start with the first. */
  for(sp = UnitList[ind].sites;sp != NULL;sp = sp->next)
    if(!strcmp(sp->name,"bottom")) break;
  if(sp == NULL)
    {
      printf("instantiate: unit %d has no bottom site \n");
      return;
    }

  for(lp = sp->inputs;lp != NULL && lp->data != 33;lp = lp->next);

  if(lp == NULL)
    if((lp = sp->inputs) == NULL)
      {
	printf("instantiate: unit %d has no bottom-up links\n",ind);
	return;
      }
  if(lp->data == 33)
    {
      unmark(lp->from_unit);	/* undarken instantiation */
      lp->data = 0;
      if(lp->next == NULL) return;
      else lp = lp->next;
    }

  /* unpack kludgy encoding of location of this unit for "gi i"
     command */
  coded = GetData(ind);
  unitx = coded >> 16;
  unity = coded & 0xffff;
  (void) sprintf(drawbuf,"gi i %d %d 1",unitx,unity);
  gi_command(drawbuf);
  
  mark(lp->from_unit);		/* darken instantiation of match unit */
  lp->data = 33;
}

/* mark(ind)
 *
 * Darken all units with bottom input to the match unit 'ind'.  Put
 * them in the info panel as well.
 */

mark(ind)
     int ind;
{
  Site *sp;
  Link *lp;
  int whichinfo = 2,unitx,unity,coded;
  
  for(sp = UnitList[ind].sites;sp != NULL;sp = sp->next)
    if(!strcmp(sp->name,"bottom")) break;
  if(sp == NULL)
    {
      printf("mark: unit %d has no bottom site \n");
      return;
    }

  for(lp = sp->inputs;lp != NULL;lp = lp->next)
    {
      /* A potential of 0 is a special case and will not show up.
	 Add one just in case. */
      SetPotential(lp->from_unit,GetPotential(lp->from_unit)+1);
      (void) sprintf(drawbuf,"gi c %d 1 P -1000 0 0 6",lp->from_unit);
      gi_command(drawbuf);
      coded = GetData(lp->from_unit);
      unitx = coded >> 16;
      unity = coded & 0xffff;
      (void) sprintf(drawbuf,"gi i %d %d %d",unitx,unity,whichinfo++);
      gi_command(drawbuf);
    }
}

/* unmark(ind)
 *
 * Undarken all units with bottom input to the match unit 'ind'.  
 * Don't bother taking them out of the info panel.
 */

unmark(ind)
     int ind;
{
  Site *sp;
  Link *lp;

  for(sp = UnitList[ind].sites;sp != NULL;sp = sp->next)
    if(!strcmp(sp->name,"bottom")) break;
  if(sp == NULL)
    {
      printf("unmark: unit %d has no bottom site \n");
      return;
    }

  for(lp = sp->inputs;lp != NULL;lp = lp->next)
    {
      /* undoes addition in mark */
      SetPotential(lp->from_unit,GetPotential(lp->from_unit)-1);
      (void) sprintf(drawbuf,"gi c %d 1 P -1000 1000 0 1",lp->from_unit);
      gi_command(drawbuf);
    }
}


/*
 * do the layout
 */

#define ICONSIDE 17		/* icons are 16X16; this gives small spacing */
#define TEXTHEIGHT 20		/* just a guess */

/* drawit()
 *
 * Should be called from within the graphics interface.  It will draw
 * a display consisting of a table containing icons for all the
 * nonterminal units.  The table will be triangular:  because the row
 * represents the length of the input string being parsed,
 * high-numbered rows must only exist for low numbered columns, which
 * represent the length of the input.
 */

drawit()
{
  int row,col,nt,t,icons_across,icons_down,basex,basey,ind,drawx,drawy;
  int boxwidth, boxheight;
  char buf[30];
  
  /* within a table box, the number of icons across and down */
  icons_across = (int) (sqrt((double)ntct) + 0.9999);
  icons_down = (icons_across % ntct ? (ntct / icons_across) + 1 :
		(ntct / icons_across));
  
  /* pixel width and height of box */
  boxwidth = icons_across * ICONSIDE + 16;
  boxheight = icons_down * ICONSIDE + 16;

  /* draw lines defining the table */
  
  draw_line(0,0,0,Length*boxheight);
  for(col = Length;col > 0;col--)
    draw_line(boxwidth*(Length+1-col),boxheight*(Length-col),
	      boxwidth*(Length+1-col),Length*boxheight);
  for(row = 1;row <= Length;row++)
    draw_line(0,(row-1)*boxheight,row*boxwidth,(row-1)*boxheight);
  draw_line(0,Length*boxheight,Length*boxwidth,Length*boxheight);

  /* draw nonterminal nodes */
  
  for(row = 1;row <= Length;row++)
    for(col = 1;col <= Length;col++)
      {
	basex = (col - 1) * boxwidth + 8;
	basey = (Length - row) * boxheight + 8;
	for(nt = 0;nt < ntct;nt++)
	  {
	    (void) sprintf(buf,"%s.%1d.%1d",words[NT][nt],row,col);
	    if((ind = NameToInd(buf,1,1)) >= 0)
	      {
		drawx = basex + ((nt % icons_across) * ICONSIDE);
		drawy = basey + ((nt / icons_across) * ICONSIDE);
		show_unit(ind, drawx, drawy);

		/* this nasty business is necessary if I am to be able
		 * to get from a unit index to its location.  I could
		 * have used a separate table, but the data field is not
		 * being used for anything else.
		 */
		   
		drawx = (drawx << 16) + drawy; /* pack'm in */
		SetData(ind,drawx);
	      }
	  }
      }

  /* draw terminal nodes */

  basey = Length * boxheight + 20;
  for(col = 1;col <= Length;col++)
    for(t = 0;t < tct;t++)
      {
	(void) sprintf(buf,"%s.%1d",words[T][t],col);
	if((ind = NameToInd(buf,1,1)) >= 0)
	  show_unit(ind, (col-1) * boxwidth + (boxwidth/2 - ICONSIDE/2),
		    basey + (t * TEXTHEIGHT));
      }
  for(t = 0;t < tct;t++)
    {
      put_text(-30,basey + ((t+1) * TEXTHEIGHT) - 5,words[T][t]);
    }
}

/* draw_line(x1,y1,x2,y2)
 *
 * Draws a line on the graphics display from x1,y1 to x2,y2
 */

static draw_line(x1,y1,x2,y2)
     int x1,y1,x2,y2;
{
  (void) sprintf(drawbuf,"gi d 2 %d %d %d %d",x1,y1,x2,y2);
  gi_command(drawbuf);
}

/* show_unit(ind,x,y)
 *
 * Puts a potential icon for the unit 'ind' at location x,y
 */

static show_unit(ind,x,y)
     int ind,x,y;
{
  (void) sprintf(drawbuf,"gi s %d 1 P -1000 1000 0 1 %d %d 5 5 max",ind,x,y);
  gi_command(drawbuf);
}

/* put_text(x,y,text)
 *
 * Puts the string 'text' at x,y.
 */

static put_text(x,y,text)
     int x,y;
     char * text;
{
  (void) sprintf(drawbuf,"gi t \"%s\" %d %d *default",text,x,y);
  gi_command(drawbuf);
}
